package org.mule.weave.v2.metadata.api

import org.mule.metadata.api.annotation.DescriptionAnnotation
import org.mule.metadata.api.annotation.EnumAnnotation
import org.mule.metadata.api.annotation.ExampleAnnotation
import org.mule.metadata.api.annotation.LabelAnnotation
import org.mule.metadata.api.annotation.MetadataFormatPropertiesAnnotation
import org.mule.metadata.api.annotation.TypeAliasAnnotation
import org.mule.metadata.api.annotation.TypeIdAnnotation
import org.mule.metadata.api.annotation.UniquesItemsAnnotation
import org.mule.metadata.api.model.AttributeKeyType
import org.mule.metadata.api.model.MetadataType
import org.mule.metadata.api.model.ObjectKeyType
import org.mule.metadata.api.{ model => mmodel }
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation
import org.mule.weave.v2.parser.ast
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.BinaryType
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.DateTimeType
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.FunctionTypeParameter
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.LocalDateTimeType
import org.mule.weave.v2.ts.LocalDateType
import org.mule.weave.v2.ts.LocalTimeType
import org.mule.weave.v2.ts.MetadataConstraint
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NameValuePairType
import org.mule.weave.v2.ts.NothingType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.PeriodType
import org.mule.weave.v2.ts.RegexType
import org.mule.weave.v2.ts.SimpleReferenceType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TimeType
import org.mule.weave.v2.ts.TimeZoneType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.IdentityHashMap

import java.util.Optional
import java.util.concurrent.atomic.AtomicLong
import javax.xml.namespace.QName
import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

class WeaveTypesConverter {

  private val secuenser = new AtomicLong(0)

  def toWeaveType(metadataType: MetadataType): WeaveType = {
    val weaveType = transformToWeaveType(metadataType, mutable.ArrayBuffer[MetadataType](), mutable.Map[String, WeaveType](), expandedMode = false, ids = IdentityHashMap())
    weaveType.withMetadataConstraint(MetadataConstraint(WeaveTypesConverter.METADATA_FORMAT_ANNOTATION, metadataType.getMetadataFormat.getId))
    weaveType
  }

  def toWeaveCatalog(muleCatalog: Map[String, MetadataType]): Map[String, WeaveType] = {
    val catalog: mutable.Map[String, WeaveType] = mutable.Map()
    val typesCache = mutable.Map[String, WeaveType]()
    val ids = IdentityHashMap[MetadataType, String]()
    muleCatalog.foreach((entry) => {
      ids.put(entry._2, toValidWeaveId(entry._1))
    })

    muleCatalog.foreach((entry) => {
      val id = toValidWeaveId(entry._1)
      catalog.+=(id -> toWeaveType(id, entry._2, mutable.ArrayBuffer[MetadataType](), typesCache, expandedMode = true, ids))
    })
    catalog.toMap
  }

  def toValidWeaveId(id: String): String = {
    id.map({
      case '{' | '}' | '.' | '-' | '#' => '_'
      case c                           => c
    })
  }

  def getIdFor(metadataType: MetadataType, ids: IdentityHashMap[MetadataType, String]): String = {
    ids.getOrElseUpdate(
      metadataType, {
      val typeId = getTypeIdAnnotation(metadataType)
      if (typeId.isPresent) {
        val typeIdValue = toValidWeaveId(typeId.get().getValue)
        if (ids.values.toSeq.contains(typeIdValue)) {
          typeIdValue + "_" + secuenser.getAndIncrement()
        } else {
          typeIdValue
        }
      } else {
        WeaveTypesConverter.ANONYMOUS_TYPE_PREFIX + secuenser.getAndIncrement()
      }
    })
  }

  def getAliasFor(metadataType: MetadataType, ids: IdentityHashMap[MetadataType, String]): String = {
    ids.getOrElseUpdate(
      metadataType, {
      val typeId = getTypeAliasAnnotation(metadataType)
      if (typeId.isPresent) {
        val typeIdValue = toValidWeaveId(typeId.get().getValue)
        if (ids.values.toSeq.contains(typeIdValue)) {
          typeIdValue + "_" + secuenser.getAndIncrement()
        } else {
          typeIdValue
        }
      } else {
        WeaveTypesConverter.ANONYMOUS_TYPE_PREFIX + secuenser.getAndIncrement()
      }
    })
  }

  //Inline this method to avoid big stacks
  //When big data structures are created this method is executed too many times
  //And by inlining we could reduce the stack
  @inline
  private def transformToWeaveType(metadataType: MetadataType, stack: mutable.ArrayBuffer[MetadataType], context: mutable.Map[String, WeaveType], expandedMode: Boolean, ids: IdentityHashMap[MetadataType, String]): WeaveType = {
    val id = getIdFor(metadataType, ids)
    if (expandedMode && isContainerType(metadataType) && getTypeIdAnnotation(metadataType).isPresent && !contains(stack, metadataType)) {
      val weaveType = toWeaveType(id, metadataType, mutable.ArrayBuffer[MetadataType](), context, expandedMode, ids)
      weaveType match {
        case rt: SimpleReferenceType => rt
        case _ =>
          context.+=(id -> weaveType)
          SimpleReferenceType(NameIdentifier(id), None, () => {
            context.getOrElse(id, throw new RuntimeException(s"Invalid reference ${id}"))
          })
      }
    } else {
      toWeaveType(id, metadataType, stack, context, expandedMode, ids)
    }
  }

  private def getTypeIdAnnotation(metadataType: MetadataType) = {
    metadataType.getAnnotation(classOf[TypeIdAnnotation])
  }

  private def getTypeAliasAnnotation(metadataType: MetadataType) = {
    metadataType.getAnnotation(classOf[TypeAliasAnnotation])
  }

  private def isContainerType(metadataType: MetadataType) = {
    metadataType.isInstanceOf[mmodel.ObjectType]
  }

  def contains(stack: ArrayBuffer[MetadataType], metadataType: MetadataType): Boolean = {
    stack.exists((actual) => actual eq metadataType)
  }

  private def toWeaveType(id: String, metadataType: MetadataType, stack: mutable.ArrayBuffer[MetadataType], typesReference: mutable.Map[String, WeaveType], expandedMode: Boolean, ids: IdentityHashMap[MetadataType, String]) = {
    if (contains(stack, metadataType)) {
      SimpleReferenceType(
        NameIdentifier(id),
        None,
        () => {
          typesReference.getOrElse(id, throw new RuntimeException(s"Invalid reference to recursive type ${id} this is likely to be a bug."))
        })
    } else if (typesReference.contains(id) && isContainerType(metadataType)) {
      SimpleReferenceType(NameIdentifier(id), None, () => {
        typesReference(id)
      })
    } else {
      stack.+=(metadataType)
      val result = metadataType match {
        case ot: mmodel.ObjectType => {
          var fields = ot.getFields.asScala
            .map((field) => {
              val key: ObjectKeyType = field.getKey
              val name = if (key.isName) {
                val name: QName = key.getName
                NameType(Some(toWeaveQName(name)))
              } else {
                NameType().withMetadataConstraint(MetadataConstraint(WeaveTypesConverter.PATTERN_ANNOTATION, key.getPattern.toString))
              }
              val attrs: Iterable[NameValuePairType] = toWeaveAttributes(key)
              KeyValuePairType(KeyType(name, attrs.toSeq), transformToWeaveType(field.getValue, stack, typesReference, expandedMode, ids), !field.isRequired, field.isRepeated)
            })
            .toSeq
          if (ot.isOpen) {
            if (ot.getOpenRestriction.isPresent) {
              val restrictionType = ot.getOpenRestriction.get()
              fields = fields.:+(KeyValuePairType(KeyType(NameType()), transformToWeaveType(restrictionType, stack, typesReference, expandedMode, ids), optional = true))
            }
          }
          ObjectType(fields, !ot.isOpen, ot.isOrdered)
        }
        case at: mmodel.ArrayType => {
          val arrayType: MetadataType = at.getType
          ArrayType(transformToWeaveType(arrayType, stack, typesReference, expandedMode, ids))
        }
        case ft: mmodel.FunctionType => {
          val rt = if (ft.getReturnType.isPresent) transformToWeaveType(ft.getReturnType.get(), stack, typesReference, expandedMode, ids) else NullType()
          FunctionType(
            Seq(),
            ft.getParameters.asScala.map((ftp) => {
              FunctionTypeParameter(ftp.getName, transformToWeaveType(ftp.getType, stack, typesReference, expandedMode, ids), ftp.isOptional)
            }),
            rt)
        }
        case ot: mmodel.AnyType          => AnyType()
        case tuple: mmodel.TupleType     => ArrayType(TypeHelper.unify(tuple.getTypes.asScala.map((t) => transformToWeaveType(t, stack, typesReference, expandedMode, ids))))
        case ut: mmodel.UnionType        => UnionType(ut.getTypes.asScala.map((t) => transformToWeaveType(t, stack, typesReference, expandedMode, ids)))
        case it: mmodel.IntersectionType => IntersectionType(it.getTypes.asScala.map(transformToWeaveType(_, stack, typesReference, expandedMode, ids)))
        case nt: mmodel.NullType         => NullType()
        case nt: mmodel.StringType => {
          val enums = nt.getAnnotation(classOf[EnumAnnotation[_]])
          if (enums.isPresent) {
            val values = enums.get().getValues.iterator.toStream.map((sv) => {
              StringType(Some(sv.toString))
            })
            UnionType(values)
          } else {
            StringType()
          }
        }
        case nt: mmodel.BooleanType => {
          val enums = nt.getAnnotation(classOf[EnumAnnotation[_]])
          if (enums.isPresent) {
            val values = enums.get().getValues.iterator.toStream.map((sv) => {
              BooleanType(Some(sv.asInstanceOf[Boolean]))
            })
            UnionType(values)
          } else {
            BooleanType()
          }
        }
        case nt: mmodel.DateTimeType      => DateTimeType()
        case nt: mmodel.RegexType         => RegexType()
        case nt: mmodel.PeriodType        => PeriodType()
        case nt: mmodel.LocalDateTimeType => LocalDateTimeType()
        case nt: mmodel.BinaryType        => BinaryType()
        case nt: mmodel.DateType          => LocalDateType()
        case nt: mmodel.NumberType => {
          val enums = nt.getAnnotation(classOf[EnumAnnotation[_]])
          if (enums.isPresent) {
            val values = enums.get().getValues.iterator.toStream.map((sv) => {
              NumberType(Some(sv.toString))
            })
            UnionType(values)
          } else {
            NumberType()
          }
        }
        case nt: mmodel.TimeType      => TimeType()
        case nt: mmodel.TimeZoneType  => TimeZoneType()
        case lr: mmodel.LocalTimeType => LocalTimeType()
        case nt: mmodel.NothingType   => NothingType()
        case nt: mmodel.VoidType      => NullType()
        //Types that are not supported in DW
        case tp: mmodel.TypeParameterType => {
          typesReference.getOrElseUpdate(tp.getName, TypeParameter(tp.getName))
        }
      }

      metadataType.getAnnotations.asScala
        .flatMap {
          case alias: TypeAliasAnnotation => Some(MetadataConstraint(alias.getName, alias.getValue))
          case typeId: TypeIdAnnotation => Some(MetadataConstraint(typeId.getName, typeId.getValue))
          case label: LabelAnnotation => Some(MetadataConstraint(label.getName, label.getValue))
          case description: DescriptionAnnotation => Some(MetadataConstraint(description.getName, description.getValue))
          case exampleAnnotation: ExampleAnnotation => Some(MetadataConstraint(exampleAnnotation.getName, exampleAnnotation.getValue))
          case uniqueItems: UniquesItemsAnnotation => Some(MetadataConstraint(uniqueItems.getName, true))
          case ci: ClassInformationAnnotation if (!ci.isMap && ci.isInstantiable) => Some(MetadataConstraint(MetadataConstraint.CLASS_PROPERTY_NAME, ci.getClassname))
          case readerProperties: MetadataFormatPropertiesAnnotation => {
            readerProperties.getValue.asScala.map((entry) => {
              MetadataConstraint(entry._1, entry._2)
            })
          }
          case _ => None
        }
        .foreach(result.withMetadataConstraint)
      stack.remove(stack.size - 1)
      typesReference.put(id, result)
      result
    }
  }

  def toWeaveAttributes(key: ObjectKeyType): Iterable[NameValuePairType] = {
    val attrs = key.getAttributes.asScala.map((attr) => {
      val attrKey: AttributeKeyType = attr.getKey
      val attrName = if (attrKey.isName) {
        val name: QName = attrKey.getName
        NameType(Some(toWeaveQName(name)))
      } else {
        NameType()
      }
      NameValuePairType(attrName, toWeaveType(attr.getValue), !attr.isRequired)
    })
    attrs
  }

  def toWeaveQName(name: QName): ast.QName = {
    val uri: String = name.getNamespaceURI
    ast.QName(name.getLocalPart, if (uri == null || uri.isEmpty) None else Option(uri))
  }

}

object WeaveTypesConverter {

  val PATTERN_ANNOTATION = "pattern"

  val METADATA_FORMAT_ANNOTATION = "MetadataFormat"

  val ANONYMOUS_TYPE_PREFIX = "anonymousType_"

  def toWeaveCatalog(muleCatalog: Map[String, MetadataType]): Map[String, WeaveType] = {
    new WeaveTypesConverter().toWeaveCatalog(muleCatalog)
  }

  def toWeaveType(metadataType: MetadataType): WeaveType = {
    new WeaveTypesConverter().toWeaveType(metadataType)
  }

  def toValidWeaveId(id: String): String = {
    new WeaveTypesConverter().toValidWeaveId(id)
  }

  def getTypeIdAnnotation(metadataType: MetadataType): Optional[TypeIdAnnotation] = {
    new WeaveTypesConverter().getTypeIdAnnotation(metadataType)
  }

  def getTypeAliasAnnotation(metadataType: MetadataType): Optional[TypeAliasAnnotation] = {
    new WeaveTypesConverter().getTypeAliasAnnotation(metadataType)
  }

}
