package org.mule.weave.v2.codegen

import scala.collection.mutable

class StringCodeWriter(indentSpaces: Int) extends CodeWriter {
  private var _indent: Int = 0
  private val _content = new StringBuilder()
  private var _newLineCallBacks = mutable.ArrayBuffer[(CodeWriter) => Unit]()
  private var _ignoreNewLine = false

  def this() = {
    this(2)
  }

  def startIgnoringNewLines(): Unit = {
    _ignoreNewLine = true
  }

  def endIgnoringNewLines(): Unit = {
    _ignoreNewLine = false
  }

  override def println(code: String): CodeWriter = {
    _content.append(code)
    println()
    this
  }

  override def println(): CodeWriter = {
    callNewLineCallbacks()
    if (!_ignoreNewLine) {
      _content.append(System.lineSeparator())
    }
    this
  }

  private def callNewLineCallbacks(): Unit = {
    val buffer = _newLineCallBacks.clone()
    _newLineCallBacks.clear()
    buffer.foreach((callback) => callback.apply(this))
  }

  override def indent(): CodeWriter = {
    _indent += 1
    this
  }

  override def printSpace(code: String): CodeWriter = {
    append(code).append(" ")
    this
  }

  override def printSpace(): CodeWriter = {
    append(" ")
    this
  }

  override def printIndent(): CodeWriter = {
    _content.append(" " * _indent * indentSpaces)
    this
  }

  override def print(code: String): CodeWriter = {
    append(code)
    this
  }

  override def printQuoted(text: String): CodeWriter = {
    append("\"").append(text).append("\"")
    this
  }

  private def append(code: String): StringBuilder = {
    if (_content.nonEmpty && _content.last == '\n') {
      printIndent()
    }
    _content.append(code)
  }

  override def printForeachWithSeparator[A](separator: String, elements: Seq[A], code: (A) => Unit): CodeWriter = {
    elements.zipWithIndex.foreach((elementWithIndex) => {
      if (elementWithIndex._2 >= 1) {
        print(separator)
      }
      code(elementWithIndex._1)
    })
    this
  }

  override def dedent(): CodeWriter = {
    _indent -= 1
    this
  }

  override def onNewLine(callback: (CodeWriter) => Unit): CodeWriter = {
    _newLineCallBacks.+=(callback)
    this
  }

  override def hasNewLineListener(): Boolean = _newLineCallBacks.nonEmpty

  def codeContent(): String = _content.toString()

  override def toString: String = codeContent()

  override def print(code: Char): CodeWriter = {
    print(code + "")
  }
  override def nonEmpty(): Boolean = {
    _content.nonEmpty
  }
}
