package org.mule.weave.v2.grammar

import org.mule.weave.v2.grammar.literals.Literals
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.annotation.BooleanNotTypeAnnotation
import org.mule.weave.v2.parser.annotation.DeprecatedAnnotation
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.annotation.InfixNotationFunctionCallAnnotation
import org.mule.weave.v2.parser.annotation.NotType
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.UndefinedExpressionNode
import org.mule.weave.v2.parser.ast.conditional.DefaultNode
import org.mule.weave.v2.parser.ast.conditional.IfNode
import org.mule.weave.v2.parser.ast.conditional.UnlessNode
import org.mule.weave.v2.parser.ast.functions._
import org.mule.weave.v2.parser.ast.header.HeaderNode
import org.mule.weave.v2.parser.ast.logical.AndNode
import org.mule.weave.v2.parser.ast.logical.OrNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.operators.UnaryOpNode
import org.mule.weave.v2.parser.ast.patterns.PatternExpressionsNode
import org.mule.weave.v2.parser.ast.patterns.PatternMatcherNode
import org.mule.weave.v2.parser.ast.types.TypeParametersApplicationListNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionsNode
import org.mule.weave.v2.parser.ast.updates.UpdateNode
import org.mule.weave.v2.parser.location.ParserPosition
import org.parboiled2._
import org.parboiled2.support.hlist.HNil

trait Expressions extends PositionTracking with Tokens with Literals with Functions with Values with Patterns with Identifiers with Updaters with ErrorRecovery {
  this: Expressions with Grammar with Directives =>

  val createPatternsMatchNode = (lhs: AstNode, rhs: PatternExpressionsNode) => {
    resolveStartPosition(lhs) :: PatternMatcherNode(lhs, rhs) :: HNil
  }

  val createUpdateNode = (lhs: AstNode, rhs: UpdateExpressionsNode) => {
    resolveStartPosition(lhs) :: UpdateNode(lhs, rhs) :: HNil
  }

  val createAndNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: AndNode(lhs, rhs) :: HNil
  }
  val createOrNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: OrNode(lhs, rhs) :: HNil
  }
  val createEqNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(EqOpId, lhs, rhs) :: HNil
  }
  val createSimilarNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(SimilarOpId, lhs, rhs) :: HNil
  }
  val createIsNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(IsOpId, lhs, rhs) :: HNil
  }
  val createNotNode = (expr: AstNode) => {
    val node = UnaryOpNode(NotOpId, expr)
    node.annotate(BooleanNotTypeAnnotation(NotType.Word))
    node
  }
  val createExclamationNotNode = (expr: AstNode) => {
    val node = UnaryOpNode(NotOpId, expr)
    node.annotate(BooleanNotTypeAnnotation(NotType.Exclamation))
    node
  }

  val createNotEqNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(NotEqOpId, lhs, rhs) :: HNil
  }
  val createGreaterThanNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(GreaterThanOpId, lhs, rhs) :: HNil
  }
  val createLessThanNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(LessThanOpId, lhs, rhs) :: HNil
  }
  val createLessOrEqualThanNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(LessOrEqualThanOpId, lhs, rhs) :: HNil
  }
  val createGreaterOrEqualThanNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(GreaterOrEqualThanOpId, lhs, rhs) :: HNil
  }
  val createAdditionNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(AdditionOpId, lhs, rhs) :: HNil
  }

  val createSubtractionNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(SubtractionOpId, lhs, rhs) :: HNil
  }
  val createRightShiftNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(RightShiftOpId, lhs, rhs) :: HNil
  }

  val createLeftShiftNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(LeftShiftOpId, lhs, rhs) :: HNil
  }
  val createDivisionNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(DivisionOpId, lhs, rhs) :: HNil
  }
  val createMultiplicationNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(MultiplicationOpId, lhs, rhs) :: HNil
  }
  val createAsNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(AsOpId, lhs, rhs) :: HNil
  }
  val createMetadataInjectorNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: BinaryOpNode(MetadataInjectorOpId, lhs, rhs) :: HNil
  }
  val createDefaultNode = (lhs: AstNode, rhs: AstNode) => {
    resolveStartPosition(lhs) :: DefaultNode(lhs, rhs) :: HNil
  }
  val createUnlessNode = (condition: AstNode, expr: AstNode, elseExpr: AstNode) => {
    UnlessNode(expr, condition, elseExpr)
  }
  val createIfNode = (condition: AstNode, expr: AstNode, elseExpr: AstNode) => {
    IfNode(expr, condition, elseExpr)
  }

  val createBinaryNode = (left: AstNode, fn: AstNode, typeParameters: Option[TypeParametersApplicationListNode], right: AstNode) => {
    val node: FunctionCallNode = FunctionCallNode(fn, FunctionCallParametersNode(Seq(left, right)), typeParameters)
    node.annotate(InfixNotationFunctionCallAnnotation())
    val startPosition = left.semanticStartPosition()
    startPosition :: node :: HNil
  }

  val createMinusNode = (expr: AstNode) => {
    UnaryOpNode(MinusOpId, expr)
  }
  val createUsingNode = (vars: Seq[UsingVariableAssignment], expr: AstNode) => {
    UsingNode(UsingVariableAssignments(vars), expr)
  }

  val createDoBlockNode = (header: HeaderNode, body: AstNode) => {
    DoBlockNode(header, body)
  }

  val createUnaryNode = (startPos: ParserPosition, fn: AstNode, right: AstNode) => {
    startPos :: FunctionCallNode(fn, FunctionCallParametersNode(Seq(right))) :: HNil
  }
  val createUndefinedExpression = () => UndefinedExpressionNode()

  val markEnclosedNode = (startPosition: ParserPosition, node: AstNode) => {
    node.annotate(EnclosedMarkAnnotation(createWeaveLocation(startPosition)))
    node
  }

  val markDeprecatedNode = (node: AstNode, version: String) => {
    node.annotate(DeprecatedAnnotation(version))
    node
  }

  def expr: Rule1[AstNode] = namedRule("Expression") {
    highLevelExprWithAnnotations | highLevelExpr
  }

  def highLevelExprWithAnnotations: Rule1[AstNode] = namedRule("Expression") {
    pushPosition ~ (oneOrMoreAnnotations ~ ws ~ highLevelExpr ~> injectAnnotationsToExpression) ~ injectPosition
  }

  def undefinedExpression: Rule1[AstNode] = namedRule("???") {
    pushPosition ~ (atomic(str("???")) ~> createUndefinedExpression) ~ injectPosition
  }

  def enclosedExpr: Rule1[AstNode] = namedRule("Enclosed Expression") {
    (pushPosition ~ atomic(parenStart ~ (expr | missingExpression("Missing Expression.")) ~!~ (parenEnd | fail("')' for the enclosed expression.")))) ~> markEnclosedNode
  }

  def ifExpr: Rule1[AstNode] = namedRule("if") {
    pushPosition ~ (ifKeyword ~!~ ws ~ (parenStart | fail("'(' for the if expression.")) ~ (expr | missingExpression(MISSING_IF_CONDITION_EXPRESSION_MESSAGE)) ~ (parenEnd | fail("')' for the if expression.")) ~ ws ~ (expr | missingExpression("Missing If Body Expression")) ~!~ (elseSubExpr | missingToken("Missing Else", "else")) ~> createIfNode) ~ injectPosition
  }

  def unlessExpr: Rule1[AstNode] = namedRule("unless") {
    pushPosition ~ (unlessKeyword ~!~ ws ~ parenStart ~!~ (expr | missingExpression("Missing Unless Condition Expression")) ~ parenEnd ~!~ ws ~ (expr | missingExpression("Missing Unless Body Expression")) ~!~ (elseSubExpr | missingToken("Missing Else expression", "else")) ~> createUnlessNode) ~ injectPosition
  }

  def elseSubExpr = namedRule("else") {
    ws ~ elseKeyword ~!~ (expr | missingExpression("Missing else Expression"))
  }

  def highLevelExpr: Rule1[AstNode] = rule {
    booleanOrExpr ~ atomic(optional(ws ~ oneOrMore(updateGroupsSubExpr | matchGroupsSubExpr | defaultSubExpr | binaryFunction | metadataInjectorSubExpr).separatedBy(ws)))
  }

  def binaryFunction = namedRule("Infix Function Call") {
    (variable ~ ws ~ optional(since("2.5") ~ typeParameterApplicationList) ~ ws ~ !ch('`') ~!~ (booleanOrExpr | missingExpression("Missing Expression")) ~> createBinaryNode) ~ injectPosition
  }

  def matchGroupsSubExpr = namedRule("match") {
    (matchKeyword ~ (patterns ~> createPatternsMatchNode)) ~ injectPosition
  }

  def updateGroupsSubExpr = namedRule("update") {
    (updateKeyword ~ (updateExpressions ~> createUpdateNode)) ~ injectPosition
  }

  def booleanAndExpr: Rule1[AstNode] = rule {
    equalityExpr ~ atomic(optional(ws ~ oneOrMore(andSubExpr).separatedBy(ws)))
  }

  def andSubExpr = namedRule("and") {
    (andKeyword ~!~ (equalityExpr | missingExpression("Missing Boolean Expression")) ~> createAndNode) ~ injectPosition
  }

  def booleanOrExpr: Rule1[AstNode] = namedRule("Expression") {
    booleanAndExpr ~ atomic(optional(ws ~ oneOrMore(orSubExpr).separatedBy(ws)))
  }

  def orSubExpr = namedRule("or") {
    (orKeyword ~!~ (booleanAndExpr | missingExpression("Missing Boolean Expression")) ~> createOrNode) ~ injectPosition
  }

  def negatedExpr: Rule1[AstNode] = namedRule("not") {
    pushPosition ~ (notKeyword ~!~ (booleanOrExpr | missingExpression("Missing Boolean Expression")) ~> createNotNode ~ push("2.2.0") ~> markDeprecatedNode) ~ injectPosition
  }

  def equalityExpr: Rule1[AstNode] = namedRule("Equality Operator") {
    relationalExpr ~ optional(ws ~ oneOrMore(equalSubExpr | notEqualSubExpr | similarSubExpr).separatedBy(ws))
  }

  def relationalExpr: Rule1[AstNode] = namedRule("Relational Operator") {
    additiveExpr ~ optional(ws ~ oneOrMore(lessOrEqualThanSubExpr | greaterOrEqualThanSubExpr | greaterThanSubExpr | lessThanSubExpr | isSubExpr).separatedBy(ws))
  }

  def isSubExpr = namedRule("is") {
    (isKeyword ~!~ (typeExpression | missingTypeExpression()) ~> createIsNode) ~ injectPosition
  }

  def equalSubExpr = namedRule("==") {
    (quiet(equals) ~!~ (relationalExpr | missingExpression("Missing Expression")) ~> createEqNode) ~ injectPosition
  }

  def similarSubExpr = namedRule("~=") {
    (quiet(similar) ~!~ (relationalExpr | missingExpression("Missing Expression")) ~> createSimilarNode) ~ injectPosition
  }

  def notEqualSubExpr = namedRule("!=") {
    (quiet(notEquals) ~!~ (relationalExpr | missingExpression("Missing Expression")) ~> createNotEqNode) ~ injectPosition
  }

  def greaterThanSubExpr = namedRule(">") {
    (quiet(greaterThan) ~!~ (additiveExpr | missingExpression("Missing Expression")) ~> createGreaterThanNode) ~ injectPosition
  }

  def lessThanSubExpr = namedRule("<") {
    (quiet(lessThan) ~!~ (additiveExpr | missingExpression("Missing Expression")) ~> createLessThanNode) ~ injectPosition
  }

  def lessOrEqualThanSubExpr = namedRule("<=") {
    (quiet(lessOrEqualThan) ~!~ (additiveExpr | missingExpression("Missing Expression")) ~> createLessOrEqualThanNode) ~ injectPosition
  }

  def greaterOrEqualThanSubExpr = namedRule(">=") {
    (quiet(greaterOrEqualThan) ~!~ (additiveExpr | missingExpression("Missing Expression")) ~> createGreaterOrEqualThanNode) ~ injectPosition
  }

  def plusSubExpr = namedRule("+") {
    (quiet(plus) ~!~ (multiplicativeExpr | missingExpression("Missing addition expression.")) ~> createAdditionNode) ~ injectPosition
  }

  def minusSubExpr = namedRule("-") {
    (quiet(minus) ~!~ (multiplicativeExpr | missingExpression("Missing minus expression.")) ~> createSubtractionNode) ~ injectPosition
  }

  def rightShiftSubExpr = namedRule(">>") {
    (quiet(rightShift) ~!~ (multiplicativeExpr | missingExpression("Missing Shift Expression.")) ~> createRightShiftNode) ~ injectPosition
  }

  def leftShiftSubExpr = namedRule("<<") {
    (quiet(leftShift) ~!~ (multiplicativeExpr | missingExpression("Missing Shift Expression.")) ~> createLeftShiftNode) ~ injectPosition
  }

  def additiveExpr: Rule1[AstNode] = namedRule("Math Operator") {
    multiplicativeExpr ~ optional(ws ~ oneOrMore(plusSubExpr | minusSubExpr | rightShiftSubExpr | leftShiftSubExpr).separatedBy(ws))
  }

  def multiplicationSubExpr = namedRule("*") {
    (quiet(multiply) ~!~ (coerceExpr | missingExpression("Missing Division Expression.")) ~> createMultiplicationNode) ~ injectPosition
  }

  def divisionSubExpr = namedRule("/") {
    (quiet(divide ~ !divide) ~!~ (coerceExpr | missingExpression("Missing Division Expression.")) ~> createDivisionNode) ~ injectPosition
  }

  def multiplicativeExpr: Rule1[AstNode] = namedRule("Multiplicative Operator") {
    coerceExpr ~ optional(ws ~ oneOrMore(multiplicationSubExpr | divisionSubExpr).separatedBy(ws))
  }

  def unaryMinusExpr: Rule1[AstNode] = rule {
    quiet(pushPosition ~ (minus ~ !number ~ booleanOrExpr ~> createMinusNode) ~ injectPosition)
  }

  def asSubExpr = namedRule("as") {
    (asKeyword ~!~ (typeExpression | missingTypeExpression()) ~> createAsNode) ~ injectPosition
  }

  def metadataInjectorSubExpr = namedRule("<~") {
    (metadataInjector ~!~ (unaryExpr | missingExpression("Missing metadata expression.")) ~> createMetadataInjectorNode) ~ injectPosition
  }

  def defaultSubExpr = namedRule("default") {
    (defaultKeyword ~!~ (booleanOrExpr | missingExpression("Missing default expression.")) ~> createDefaultNode) ~ injectPosition
  }

  def usingExpr: Rule1[UsingNode] = namedRule("using") {
    pushPosition ~ (usingKeyword ~!~ parenStart ~ oneOrMore(variableAssignment).separatedBy(commaSep) ~ parenEnd ~ ws ~ expr ~> createUsingNode) ~ injectPosition
  }

  def doBlockExpr: Rule1[DoBlockNode] = namedRule("do block") {
    pushPosition ~ (doBlockKeyword ~!~ curlyBracketsStart ~ ((fullHeader ~ ((headerSeparator ~ ws ~ content) | missingExpression("Missing Do Block Expression ie. do {\n var a = 1 \n --- \n a }"))) | (push(HeaderNode.emptyHeader()) ~ ws ~ content)) ~ (curlyBracketsEnd | fail("'}' for the do block")) ~> createDoBlockNode) ~ injectPosition
  }

  def coerceExpr: Rule1[AstNode] = rule {
    unaryExpr ~ atomic(optional(ws ~ oneOrMore(asSubExpr).separatedBy(ws)))
  }

  /**
    * This was added because precedence of not operator was wrong. <br/>
    * E.g. (not a and b) equals (not (a and b))
    *
    * @since 2.2.0 (Mule 4.2.0)
    */
  def booleanNotExpr = rule {
    pushPosition ~ exclamationMark ~!~ ws ~ value ~> createExclamationNotNode ~ injectPosition
  }

  def unaryExpr: Rule1[AstNode] = namedRule("unary expr") {
    (undefinedExpression | unaryMinusExpr | booleanNotExpr | negatedExpr | usingExpr | ifExpr | unlessExpr | doBlockExpr) | value
  }

}
