package org.mule.weave.v2.hover

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.completion.DataFormatDescriptor
import org.mule.weave.v2.completion.DataFormatDescriptorProvider
import org.mule.weave.v2.completion.EmptyDataFormatDescriptorProvider
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.header.directives.DirectiveOptionName
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.InputDirective
import org.mule.weave.v2.parser.ast.header.directives.InputOutputDirective
import org.mule.weave.v2.parser.ast.header.directives.OutputDirective
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.utils.AsciiDocMigrator
import org.mule.weave.v2.utils.WeaveTypeParser

import scala.util.Try

class HoverService(editorSupport: WeaveEditorSupport, provider: DataFormatDescriptorProvider = EmptyDataFormatDescriptorProvider) {

  /**
    * Returns the Documentation Information of the element at the specified offset.
    *
    * @param location The offset location
    * @return MayBe HoverMessage
    */
  def hover(location: Int): Option[HoverMessage] = {
    editorSupport.astNavigator().flatMap((navigator) => {
      collectHoverMessages(navigator, navigator.nodeAt(location), editorSupport.typeGraph(), location)
    })
  }

  def isCompatible(df: DataFormatDescriptor, outputDirective: InputOutputDirective): Boolean = {
    if (outputDirective.mime.isDefined) {
      df.mimeType.equals(outputDirective.mime.get.mime)
    } else {
      df.id.exists(id => id.equals(outputDirective.dataFormat.get.id))
    }
  }

  private def collectHoverMessages(navigator: AstNavigator, maybeNode: Option[AstNode], maybeTypeGraph: Option[TypeGraph], location: Int): Option[HoverMessage] = {
    maybeNode match {
      case Some(ni: NameIdentifier) => {
        val isModuleNameIdentifier = navigator.granParentOf(ni).exists(_.isInstanceOf[ImportDirective])
        if (isModuleNameIdentifier) {
          moduleHoverInfo(ni)
        } else {
          val mayBeParent = ni.parent()
          if (mayBeParent.isDefined) {
            val moduleNameIdentifier = mayBeParent.get
            //Should we verify if the NameIdentifier is a module and check if the cursor is over the module name then show module information
            val moduleName = moduleNameIdentifier.name
            val moduleEndOffset = moduleName.length + ni.location().startPosition.index
            if (location < moduleEndOffset) {
              //Retrieve module documentation
              val result = editorSupport.tryToParseModule(mayBeParent.get)
              if (result.exists(_.hasResult())) {
                result.flatMap((f) => moduleHoverInfo(f))
              } else {
                val matchedModule = navigator.importDirectives().find((id) => {
                  val importedModule = id.importedModule
                  val importedModuleName: NameIdentifier = if (importedModule.alias.isDefined) {
                    importedModule.alias.get
                  } else {
                    importedModule.elementName.localName()
                  }
                  importedModuleName.equals(moduleNameIdentifier)
                })
                matchedModule.flatMap((importedModule) => {
                  moduleHoverInfo(importedModule.importedModule.elementName)
                })
              }
            } else {
              collectHoverMessages(navigator, navigator.parentOf(ni), maybeTypeGraph, location)
            }
          } else {
            collectHoverMessages(navigator, navigator.parentOf(ni), maybeTypeGraph, location)
          }
        }
      }
      case Some(st: StringNode) if navigator.parentOf(st).exists(_.isInstanceOf[NameNode]) => {
        collectHoverMessages(navigator, navigator.parentOf(st), maybeTypeGraph, location)
      }
      case Some(vr: VariableReferenceNode) if navigator.parentOf(vr).exists(_.isInstanceOf[FunctionCallNode]) => {
        val parent = navigator.parentOf(vr)
        val functionCallNode = parent.get.asInstanceOf[FunctionCallNode]
        val maybeParentHover = collectHoverMessages(navigator, parent, maybeTypeGraph, location)
        val localHover = maybeTypeGraph.flatMap((typeGraph) => {
          val maybeNode = typeGraph.findNode(vr)
          val inferredType = maybeNode.flatMap(_.resultType())
          inferredType.map {
            case wtype @ (ft: FunctionType) if ft.overloads.nonEmpty => {
              val overloads = ft.overloads
              //Filter based on the arg types
              val argTypes = functionCallNode.args.args.map((arg) => typeGraph.findNode(arg).flatMap(_.resultType()))

              val functions = overloads.filter((functionType) => {
                if (argTypes.size <= functionType.params.size) {
                  argTypes.isEmpty || !argTypes.zipWithIndex.exists((argWithIndex) => {
                    val argType = argWithIndex._1
                    val argIndex = argWithIndex._2
                    val parameter = functionType.params.apply(argIndex)
                    if (argType.isDefined) {
                      !Try(TypeHelper.canBeSubstituted(argType.get, parameter.wtype, new WeaveTypeResolutionContext(maybeTypeGraph.get))).getOrElse(true)
                    } else {
                      false
                    }
                  })
                } else {
                  false
                }
              })
              val documentation = functions.map((wtype) => {
                s"#### ${wtype.toString(prettyPrint = false, namesOnly = true)}\n\n${wtype.getDocumentation().getOrElse("")}"
              }).mkString("\n______________________________\n")
              HoverMessage(wtype, Some(documentation), vr.location())
            }
            case wtype => HoverMessage(wtype, wtype.getDocumentation(), vr.location())
          }
        })
        if (localHover.isDefined) {
          maybeParentHover match {
            case Some(parentHover) => {

              Some(parentHover.copy(documentation = parentHover.documentation.orElse(localHover.get.documentation)))
            }
            case _ => localHover
          }
        } else {
          maybeParentHover
        }
      }
      case Some(fd: FunctionDirectiveNode) => {
        getHoverMessagesOf(maybeTypeGraph, fd.variable)
      }
      case Some(fd: VarDirective) => {
        getHoverMessagesOf(maybeTypeGraph, fd.variable)
      }
      case Some(fd: TypeDirective) => {
        getHoverMessagesOf(maybeTypeGraph, fd.variable)
      }
      case Some(nn: NameNode) => {
        collectHoverMessages(navigator, navigator.parentOf(nn), maybeTypeGraph, location)
      }
      case Some(don: DirectiveOptionName) => {
        val maybeDirective = navigator.parentWithType(don, classOf[InputOutputDirective])
        maybeDirective match {
          case Some(inputOutputDirective) => {
            val maybeFormatDescriptor: Option[DataFormatDescriptor] = provider.dataFormats().find((df) => isCompatible(df, inputOutputDirective))
            maybeFormatDescriptor.flatMap((dataFormat) => {
              inputOutputDirective match {
                case _: InputDirective => {
                  dataFormat.readerProperties.find(p => {
                    p.name.equals(don.name)
                  })
                }
                case _: OutputDirective => {
                  dataFormat.writerProperties.find(p => {
                    p.name.equals(don.name)
                  })
                }
                case _ => None
              }

            }).map((prop) => {
              val wtype: Option[WeaveType] = WeaveTypeParser.parse(prop.wtype, editorSupport.buildParsingContext())
              HoverMessage(wtype.getOrElse(AnyType()), Some(prop.description), don.location())
            })
          }
          case _ => None
        }
      }
      case Some(node) => {
        getHoverMessagesOf(maybeTypeGraph, node)
      }
      case _ => None
    }
  }

  private def moduleHoverInfo(ni: NameIdentifier): Option[HoverMessage] = {
    val mayBeModule: Option[PhaseResult[ParsingResult[ModuleNode]]] = editorSupport.tryToParseModule(ni)
    mayBeModule.flatMap((phaseResult) => {
      if (phaseResult.hasResult()) {
        moduleHoverInfo(phaseResult)
      } else {
        None
      }
    })
  }

  private def moduleHoverInfo(phaseResult: PhaseResult[ParsingResult[ModuleNode]]): Option[HoverMessage] = {
    val moduleNode = phaseResult.getResult().astNode
    moduleNode.weaveDoc.map((doc) => {
      HoverMessage(AnyType(), Some(doc.literalValue), moduleNode.location())
    })
  }

  private def getHoverMessagesOf(maybeTypeGraph: Option[TypeGraph], node: AstNode): Option[HoverMessage] = {
    maybeTypeGraph.flatMap((typeGraph) => {
      val maybeNode = typeGraph.findNode(node)
      val inferredType = maybeNode.flatMap(_.resultType())
      inferredType.map((wtype) => {
        HoverMessage(wtype, wtype.getDocumentation(), node.location())
      })
    })
  }
}

//TODO we need to do resultType optional
case class HoverMessage(resultType: WeaveType, documentation: Option[String], weaveLocation: WeaveLocation) {
  def markdownDocs: Option[String] = documentation.map(AsciiDocMigrator.toMarkDown)
}
