package org.mule.weave.v2.grammar

import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.ErrorAstNode
import org.mule.weave.v2.parser.InvalidFieldNameIdentifierError
import org.mule.weave.v2.parser.InvalidVariableNameIdentifierError
import org.mule.weave.v2.parser.MissingDataFormatDefinition
import org.mule.weave.v2.parser.MissingExpressionErrorAstNode
import org.mule.weave.v2.parser.MissingFormatDefinition
import org.mule.weave.v2.parser.MissingFunctionDefinitionErrorAstNode
import org.mule.weave.v2.parser.MissingFunctionParameter
import org.mule.weave.v2.parser.MissingKeySelector
import org.mule.weave.v2.parser.MissingNameIdentifierError
import org.mule.weave.v2.parser.MissingSchemaNode
import org.mule.weave.v2.parser.MissingSelectorExpressionErrorAstNode
import org.mule.weave.v2.parser.MissingTokenErrorAstNode
import org.mule.weave.v2.parser.MissingUri
import org.mule.weave.v2.parser.MissingVersionMajor
import org.mule.weave.v2.parser.MissingVersionMinor
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper.markInjectedNode
import org.mule.weave.v2.parser.ast.WeaveLocationCapable
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.header.directives.DataFormatId
import org.mule.weave.v2.parser.ast.header.directives.FormatExpression
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.ParserPosition
import org.mule.weave.v2.parser.location.WeaveLocation
import org.parboiled2.Parser
import org.parboiled2.Rule0
import org.parboiled2.Rule1

trait ErrorRecovery extends PositionTracking {
  this: ErrorRecovery with Parser with WhiteSpaceHandling =>

  val MISSING_IF_CONDITION_EXPRESSION_MESSAGE = "Missing If Condition Expression"

  def missingExpression(message: String): Rule1[ErrorAstNode] = rule {
    push(markInjectedNode(MissingExpressionErrorAstNode(message))) ~ injectErrorPosition
  }

  def missingFunctionExpression(message: String): Rule1[ErrorAstNode] = rule {
    push(markInjectedNode(MissingFunctionDefinitionErrorAstNode(message))) ~ injectErrorPosition
  }

  def missingSelectorExpression(message: String): Rule1[ErrorAstNode] = rule {
    push(markInjectedNode(MissingSelectorExpressionErrorAstNode(message))) ~ injectErrorPosition
  }

  def missingToken(message: String, token: String): Rule1[MissingTokenErrorAstNode] = rule {
    push(markInjectedNode(MissingTokenErrorAstNode(message, token))) ~ injectErrorPosition
  }

  def missingSelectorName(message: String): Rule1[NameNode] = rule {
    push(markInjectedNode(new MissingKeySelector(message))) ~ injectErrorPosition
  }

  def missingIdentifier(message: String): Rule1[NameIdentifier] = rule {
    push(markInjectedNode(new MissingNameIdentifierError(message))) ~ injectErrorPosition
  }

  def missingFunctionParameter(): Rule1[FunctionParameter] = rule {
    push(markInjectedNode(new MissingFunctionParameter())) ~ injectErrorPosition
  }

  def missingTypeExpression(): Rule1[WeaveTypeNode] = rule {
    (missingIdentifier("Missing Type Expression e.g String.") ~> ((nameIdentifier) => TypeReferenceNode(nameIdentifier))) ~ injectErrorPosition
  }

  def missingSchema(): Rule1[SchemaNode] = rule {
    push(markInjectedNode(new MissingSchemaNode())) ~ injectErrorPosition
  }

  val createInvalidVariableNameIdentifier = (identifier: String) => {
    new InvalidVariableNameIdentifierError(identifier)
  }

  val createInvalidFieldNameIdentifier = (identifier: String) => {
    new InvalidFieldNameIdentifierError(identifier)
  }

  def injectErrorPosition[A] = rule {
    run { (node: A) =>
      {
        val startPosition: ParserPosition = if (cursor == 0) {
          ParserPosition(cursor, stringInput)
        } else {
          var i = cursor
          var done = false
          while (i > 0 && !done) {
            val c = input.charAt(i - 1)
            if (Character.isWhitespace(c)) {
              i = i - 1
            } else {
              done = true
            }
          }
          ParserPosition(i, stringInput)
        }
        val endPosition: ParserPosition = ParserPosition(cursor, stringInput)
        node match {
          case pn: WeaveLocationCapable =>
            pn._location = Some(WeaveLocation(startPosition, endPosition, resourceName))
          case Some(pn: WeaveLocationCapable) =>
            pn._location = Some(WeaveLocation(startPosition, endPosition, resourceName))
        }
        node
      }
    }
  }

  def missingFormatDefinition(message: String): Rule1[FormatExpression] = rule {
    push(new MissingFormatDefinition(message)) ~ injectErrorPosition
  }

  def missingDataFormatDefinition(message: String): Rule1[DataFormatId] = rule {
    push(new MissingDataFormatDefinition(message)) ~ injectErrorPosition
  }

  def missingVersionMajor(): Rule1[MissingVersionMajor] = rule {
    push(new MissingVersionMajor()) ~ injectErrorPosition
  }

  def missingVersionMinor(): Rule1[MissingVersionMinor] = rule {
    push(new MissingVersionMinor()) ~ injectErrorPosition
  }

  def missingUriNode(): Rule1[MissingUri] = rule {
    push(new MissingUri()) ~ injectErrorPosition
  }

  def guardedByToken[T <: AstNode](token: () => Rule0, expr: () => Rule1[T], tokenString: String, errorMessage: Option[String] = None): Rule1[AstNode] =
    rule {
      (token() ~!~ ws ~
        expr()) |
        missingToken(errorMessage.getOrElse(s"Missing ${tokenString} token"), tokenString)
    }
}
