package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.DuplicatedVariableMessage
import org.mule.weave.v2.parser.FunctionNameClashedWithVariableMessage
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.UsingNode
import org.mule.weave.v2.parser.ast.functions.UsingVariableAssignment
import org.mule.weave.v2.parser.ast.header.HeaderNode
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.module.ModuleNode

/**
  * Validates two variables doesn't have the same name
  */
class UniqueNameVariablesDeclarationValidation extends AstNodeVerifier {

  def checkDirectives(directives: Seq[DirectiveNode], context: ParsingContext): Unit = {
    val variablesByName: Map[String, Seq[AstNode]] = directives.collect({
      case vd: VarDirective       => vd
      case vd: NamespaceDirective => vd
      case vd: TypeDirective      => vd
      case vd: InputDirective     => vd
    }).groupBy({
      case vd: VarDirective       => vd.variable.name
      case vd: NamespaceDirective => vd.prefix.name
      case vd: TypeDirective      => vd.variable.name
      case vd: InputDirective     => vd.variable.name
    })
    variablesByName.foreach({
      case (x, ys) if ys.size > 1 =>
        ys.foreach(variable => {
          context.messageCollector.error(DuplicatedVariableMessage(x), variable.location())
        })
      case _ =>
    })

    directives.collect({ case fd: FunctionDirectiveNode => fd }).foreach(fd => {
      val funName: String = fd.variable.name
      if (variablesByName.contains(funName)) {
        context.messageCollector.error(FunctionNameClashedWithVariableMessage(funName), fd.location())
        context.messageCollector.error(FunctionNameClashedWithVariableMessage(funName), variablesByName(funName).head.location())
      }
    })
  }

  override def verify(node: AstNode, context: ParsingContext): Unit = {
    node match {
      case HeaderNode(directives) =>
        checkDirectives(directives, context)
      case ModuleNode(_, nodes) =>
        checkDirectives(nodes, context)
      case UsingNode(assignments, _, _) =>
        val assignmentsByName: Map[String, Seq[UsingVariableAssignment]] = assignments.assignmentSeq.groupBy(_.name.name)
        assignmentsByName.foreach({
          case (x, ys) if ys.size > 1 =>
            ys.foreach(variable => {
              context.messageCollector.error(DuplicatedVariableMessage(x), variable.location())
            })
          case _ =>
        })
      case _ =>
      // Nothing to do
    }
  }

}
