package org.mule.weave.v2.signature

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.AsciiDocMigrator

/**
  * This service provides Signature Information of the function being called
  *
  * @param editor The editor
  */
class FunctionSignatureHelpService(val editor: WeaveEditorSupport) {

  /**
    *
    * @param offset
    * @return
    */
  def functionSignatureAt(offset: Int): Option[FunctionSignatureResult] = {

    val maybeNavigator: Option[AstNavigator] = editor.astNavigator()
    maybeNavigator match {
      case Some(navigator) => {
        navigator.nodeAt(offset) match {
          case Some(argumentNode) => {
            val mayBeFunctionCallParameters: Option[FunctionCallParametersNode] = navigator.nodeWith(argumentNode, classOf[FunctionCallParametersNode])
            mayBeFunctionCallParameters match {
              case Some(functionCallParametersNode) => {
                val maybeFunctionCallNode = navigator.parentWithType(functionCallParametersNode, classOf[FunctionCallNode])
                maybeFunctionCallNode match {
                  case Some(value) => {
                    val maybeWeaveType: Option[WeaveType] = editor.typeGraph().flatMap((tg) => {
                      tg.findNode(value.function)
                        .flatMap((node) => node.resultType())
                    })
                    maybeWeaveType match {
                      case Some(wt: FunctionType) => {
                        var index: Int = 0
                        if (!argumentNode.isInstanceOf[FunctionCallParametersNode]) {
                          val args: Seq[AstNode] = functionCallParametersNode.args
                          var found: Boolean = false
                          while (index < args.length && !found) {
                            if (navigator.isDescendantOf(args(index), argumentNode)) {
                              found = true
                            } else {
                              index = index + 1
                            }
                          }
                        }
                        val argsLength = functionCallParametersNode.args.length
                        val signatures = if (wt.overloads.nonEmpty) {
                          wt.overloads.map((ft) => {
                            FunctionSignatureData(toParameterSignatureData(ft, index), active = ft.params.length >= argsLength, ft.getDocumentation())
                          }).toArray
                        } else {
                          Array(FunctionSignatureData(toParameterSignatureData(wt, index), active = wt.params.length >= argsLength, wt.getDocumentation()))
                        }
                        Some(FunctionSignatureResult(wt.name.getOrElse(""), signatures, index))
                      }
                      case _ => None
                    }
                  }
                  case _ => None
                }
              }
              case _ => None
            }
          }
          case _ => None
        }
      }
      case _ => None
    }
  }

  private def toParameterSignatureData(wt: FunctionType, index: Int): Array[FunctionParameterData] = {
    val arguments = wt.params.zipWithIndex.map((paramWithIndex) => {
      val param = paramWithIndex._1
      val paramIndex = paramWithIndex._2
      FunctionParameterData(param.name, param.wtype, active = paramIndex == index)
    }).toArray
    arguments
  }
}

case class FunctionSignatureResult(name: String, signatures: Array[FunctionSignatureData], currentArgIndex: Int)

case class FunctionSignatureData(parameters: Array[FunctionParameterData], active: Boolean, documentation: Option[String]) {
  def docAsMarkdown(): Option[String] = documentation.map((d) => AsciiDocMigrator.toMarkDown(d))
}

case class FunctionParameterData(name: String, wtype: WeaveType, active: Boolean)
