package org.mule.weave.v2.utils

import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.parser.ast.operators.OpNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.ts.Edge
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.resolvers.RetyperResolver

object DataGraphDotEmitter {

  val MAX_LABEL_LENGTH = 40

  def print(graph: TypeGraph, code: String = "", name: String = "WeaveType"): String = {
    val writer: StringCodeWriter = new StringCodeWriter()
    writer.println(s"digraph ${name.replaceAll("\"", "\'").replaceAll(" ", "_")} {")
    writer.indent()
    writer.println("  node [fixedsize=shape fontsize=10]")
    printNodes(writer, graph)

    val subgraphs: Seq[TypeGraph] = graph
      .subGraphsWithLabel()
      .zipWithIndex
      .map((graph: ((TypeGraph, String), Int)) => {
        val subgraph = graph._1._1

        writer.printIndent()
        writer.println(s"subgraph cluster${graph._2.toString.replaceAll("\"", "\'")} {")
        writer.indent()
        writer.printIndent()
        writer.println("node [style=filled];")
        printNodes(writer, subgraph)
        writer.dedent()
        writer.printIndent()
        writer.println("label = \"" + graph._1._2.replaceAll("\"", "\'") + "\";")
        writer.println("}")
        subgraph
      })

    subgraphs.foreach(printEdges(writer, _))
    printEdges(writer, graph)
    writer.printIndent()
    writer.println("label=\"" + code.replaceAll("\"", "\'") + "\";")
    writer.dedent()
    writer.println("}")
    writer.codeContent()
  }

  def printEdges(writer: StringCodeWriter, graph: TypeGraph): Unit = {
    graph.nodes.foreach((node) => {
      node
        .outgoingEdges()
        .foreach((edge) => {
          writer.printIndent()
          writer.print(s"${id(edge.source)} -> ${id(edge.target)}")
          writer.println("[taillabel=" + "\"" + label(edge).replaceAll("\"", "\'") + "\"" + " labeldistance=\"1\" fontname=\"times  italic\" fontsize = 10 " + (if (edge.error()) "color=\"red\"" else "") + " ];")
        })
    })
  }

  def printNodes(writer: StringCodeWriter, graph: TypeGraph): Unit = {
    graph.nodes.foreach((node) => {
      writer.printIndent()
      writer.println(s"${id(node)} [label=" + "\"" + label(node).replaceAll("\"", "\'") + "\"" + "];")
    })
  }

  def label(edge: Edge): String = {
    if (edge.incomingTypeDefined()) {
      val emitter = new WeaveTypeEmitter(WeaveTypeEmitterConfig(nameOnly = true, skipMetadataConstraints = true, printTypeParamInstanceId = true, printConstrains = true, skipWeaveTypeMetadata = true))
      val typeText = emitter.toString(edge.incomingType())
      if (typeText.length > MAX_LABEL_LENGTH) {
        typeText.substring(0, MAX_LABEL_LENGTH / 2) + "..." + typeText.substring(typeText.length - (MAX_LABEL_LENGTH / 2), typeText.length)
      } else {
        typeText
      }
    } else {
      if (edge.label.isEmpty)
        "Not Defined"
      else
        s"[${edge.label}]"
    }
  }

  def label(node: TypeNode): String = {
    node.typeResolver match {
      case rt: RetyperResolver => s"Retyper(${rt.ref.referencedNode.name}, ${rt.branch})"
      case _ =>
        val astNode: AstNode = node.astNode
        astNode match {
          case varName: OpNode                      => s"${astNode.getClass.getSimpleName}(${varName.opId.name})"
          case varName: NameIdentifier              => s"${astNode.getClass.getSimpleName}(${varName.name})"
          case literal: LiteralValueAstNode         => s"${astNode.getClass.getSimpleName}(${literal.literalValue})"
          case referenceNode: VariableReferenceNode => s"${astNode.getClass.getSimpleName}(${referenceNode.variable.name})"
          case _                                    => astNode.getClass.getSimpleName
        }
    }
  }

  def id(node: TypeNode): String = {
    Math.abs(System.identityHashCode(node)).toString
  }

}
