package org.mule.weave.v2.editor.quickfix

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.ImportedElement
import org.mule.weave.v2.parser.ast.header.directives.VersionDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

class InsertImportModuleQuickFix(val editorSupport: WeaveEditorSupport, val module: NameIdentifier) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {

    editorSupport.astDocument() match {
      case Some(ast: DocumentNode) => {
        val directives = ast.header.directives
        runAddImport(directives, document)
      }
      case Some(ast: ModuleNode) => {
        val directives = ast.directives
        runAddImport(directives, document)
      }
      case None =>
    }
  }

  private def runAddImport(directives: Seq[DirectiveNode], document: WeaveTextDocument): Unit = {
    //Represents (true, position) start position
    //Represents (false, position) end position
    val startPositionPair: Option[(Boolean, Int)] = directives
      .flatMap({
        case id: ImportDirective if (!AstNodeHelper.isInjectedNode(id)) => {
          Some((true, id.location().startPosition.index))
        }
        case _ => None
      })
      .headOption
      .orElse({
        directives
          .flatMap({
            case id: VersionDirective => {
              Some((false, id.location().endPosition.index))
            }
            case _ => None
          })
          .headOption
      })

    val newImport = ImportDirective(ImportedElement(module))
    val noHeader = directives.isEmpty || directives.forall(AstNodeHelper.isInjectedNode(_))
    if (noHeader) {
      document.insert(s"%dw 2.0\n${CodeGenerator.generate(newImport).trim}\n---\n", 0)
    } else {
      if (startPositionPair.isEmpty) {
        document.insert(CodeGenerator.generate(newImport).trim + "\n", 0)
      } else {
        if (startPositionPair.get._1) {
          document.insert(CodeGenerator.generate(newImport).trim + "\n", startPositionPair.get._2)
        } else {
          document.insert("\n" + CodeGenerator.generate(newImport).trim, startPositionPair.get._2)
        }
      }
    }

  }
}
