package org.mule.weave.v2.scope

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ParsingContext

import scala.collection.mutable.ListBuffer

/**
  * This class is in charge of building a Variable Scope. It is being used by the Factory to construct a VariableScope
  */
class VariableScopeBuilder(val parsingContext: ParsingContext, val parentScope: Option[VariableScopeBuilder], val name: Option[String], val astNode: AstNode, val index: Int) {
  private val children: ListBuffer[VariableScopeBuilder] = ListBuffer()
  private val references: ListBuffer[NameIdentifier] = ListBuffer()
  private val declarations: ListBuffer[NameIdentifier] = ListBuffer()
  private val importedModules: ListBuffer[(ImportDirective, VariableScope)] = ListBuffer()

  def addImportedModule(id: ImportDirective, vs: VariableScope): VariableScopeBuilder = {
    importedModules += ((id, vs))
    this
  }

  def addReference(ref: NameIdentifier): VariableScopeBuilder = {
    references.+=(ref)
    this
  }

  def addDeclaration(decl: NameIdentifier): VariableScopeBuilder = {
    declarations.+=(decl)
    this
  }

  def addDeclarations(decl: Seq[NameIdentifier]): VariableScopeBuilder = {
    declarations.++=(decl)
    this
  }

  def createChild(location: AstNode): VariableScopeBuilder = {
    val child = VariableScopeBuilder(parsingContext, this, location)
    children += child
    child
  }

  def createChild(name: String, location: AstNode): VariableScopeBuilder = {
    val child = VariableScopeBuilder(parsingContext, this, name, location)
    children += child
    child
  }

  def build(maybeParent: Option[VariableScope] = None): VariableScope = {
    val result = new VariableScope( //Build Variable Scope
      parsingContext,
      maybeParent,
      name,
      astNode,
      index,
      references,
      declarations,
      importedModules)
    result.withChildren(children.map(_.build(Some(result))))
  }
}
object VariableScopeBuilder {
  def apply(parsingContext: ParsingContext, astNode: AstNode): VariableScopeBuilder = {
    new VariableScopeBuilder(parsingContext, None, None, astNode, 0)
  }

  def apply(parsingContext: ParsingContext, name: String, astNode: AstNode): VariableScopeBuilder = {
    new VariableScopeBuilder(parsingContext, None, Some(name), astNode, 0)
  }

  def apply(parsingContext: ParsingContext, parent: VariableScopeBuilder, astNode: AstNode): VariableScopeBuilder = {
    new VariableScopeBuilder(parsingContext, Some(parent), None, astNode, parent.index + 1)
  }

  def apply(parsingContext: ParsingContext, parent: VariableScopeBuilder, name: String, astNode: AstNode): VariableScopeBuilder = {
    new VariableScopeBuilder(parsingContext, Some(parent), Some(name), astNode, parent.index + 1)
  }
}
