package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.{ Message, MessageCollector, TypeCoercedMessage }
import org.mule.weave.v2.ts.WeaveTypeTraverse.equalsWith

case class Edge(source: TypeNode, target: TypeNode, label: Option[EdgeLabel] = None, expected: Option[WeaveType] = None, supportsCoercion: Boolean = true) {
  source.addOutgoingEdge(this)
  target.addIncomingEdge(this)

  private var _incomingType: Option[WeaveType] = None

  private var _expectedPropagatedType: Option[WeaveType] = None

  private var _error = false

  /**
    * If this node has an error or not
    *
    * @return
    */
  def error(): Boolean = _error

  /**
    * If this edge is cross different graphs
    *
    * @return
    */
  def crossGraphEdge(): Boolean = {
    source.parent() != target.parent()
  }

  def remove(): Unit = {
    source.removeOutputEdge(this)
    target.removeIncomingEdge(this)
  }

  /**
    * Propagates the type to the target
    *
    * @param actualType The type to propagate
    * @param ctx        The context
    */
  def propagateType(actualType: WeaveType, ctx: WeaveTypeResolutionContext): Unit = {
    if (expected.isDefined) {
      val expectedType: WeaveType = expected.get
      val messageCollector = MessageCollector()
      if (!TypeHelper(ctx).canBeAssignedTo(actualType, expectedType, ctx, strict = false, messageCollector)) {
        val errorMessages: Seq[(WeaveLocation, Message)] = messageCollector.errorMessages
        if (supportsCoercion) {
          val maybeCoercedType: Option[WeaveType] = TypeCoercer.coerce(expectedType, actualType, ctx)
          if (maybeCoercedType.isDefined) {
            ctx.warning(TypeCoercedMessage(expectedType, actualType), source, actualType.location())
            val coercedType = maybeCoercedType.get
            WeaveTypeCloneHelper.copyAdditionalTypeInformation(actualType, coercedType)
            _incomingType = Some(coercedType)
            ctx.currentExecutor.scheduleNode(target)
          } else {
            _error = true
            _incomingType = Some(actualType)
            errorMessages.foreach((pair) => {
              ctx.error(pair._2, source, pair._1)
            })
          }
        } else {
          _error = true
          _incomingType = Some(actualType)
          errorMessages.foreach((pair) => {
            ctx.error(pair._2, source, pair._1)
          })

        }
      } else {
        _incomingType = Some(actualType)
        ctx.currentExecutor.scheduleNode(target)
      }
    } else {
      _incomingType = Some(actualType)
      ctx.currentExecutor.scheduleNode(target)
    }
  }

  /**
    * Propagate the expected type to the source
    *
    * @param expectedType The expected type
    * @param ctx          The context
    */
  def reversePropagate(expectedType: WeaveType, ctx: WeaveTypeResolutionContext): Unit = {
    if (_expectedPropagatedType.isEmpty || !equalsWith(expectedType, _expectedPropagatedType.get)) {
      updateExpectedType(expectedType)
      ctx.currentExecutor.scheduleNode(source)
    }
  }

  def updateExpectedType(expectedType: WeaveType): Unit = {
    if (expected.isEmpty || TypeHelper.canBeAssignedTo(expectedType, expected.get, null)) {
      _expectedPropagatedType = Some(expectedType)
    }
  }

  /**
    * Returns the type that was inferred to be expected by the source node
    *
    * @return
    */
  def mayBeExpectedPropagatedType(): Option[WeaveType] = {
    _expectedPropagatedType.orElse(expected)
  }

  /**
    * Returns true if expected type is defined
    *
    * @return
    */
  def expectedPropagatedTypeDefined(): Boolean = {
    mayBeExpectedPropagatedType().isDefined
  }

  /**
    * Returns the inferred type by the source
    *
    * @return
    */
  def mayBeIncomingType(): Option[WeaveType] = {
    _incomingType
  }

  def incomingType(): WeaveType = {
    _incomingType.get
  }

  def incomingTypeDefined(): Boolean = {
    _incomingType.isDefined
  }
}

object Edge {
  def apply(source: TypeNode, target: TypeNode, label: EdgeLabel): Edge = new Edge(source, target, Some(label))

  def apply(source: TypeNode, target: TypeNode, label: EdgeLabel, expected: WeaveType): Edge = new Edge(source, target, Some(label), Some(expected))
}
