package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.parser.InvalidAmountTypeParametersMessage
import org.mule.weave.v2.parser.InvalidDynamicReturnMessage
import org.mule.weave.v2.parser.InvalidMethodTypesMessage
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MultipleValidFunctions
import org.mule.weave.v2.parser.NotEnoughArgumentMessage
import org.mule.weave.v2.parser.TooManyArgumentMessage
import org.mule.weave.v2.parser.TypeCoercedMessage
import org.mule.weave.v2.parser.TypeMessage
import org.mule.weave.v2.parser.TypeMismatch
import org.mule.weave.v2.parser.UnfulfilledConstraint
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.ConcreteTypeParamInstanceCounter.nextId
import org.mule.weave.v2.ts.ConstrainProblem
import org.mule.weave.v2.ts.Constraint
import org.mule.weave.v2.ts.ConstraintResult
import org.mule.weave.v2.ts.ConstraintSet
import org.mule.weave.v2.ts.DynamicReturnType
import org.mule.weave.v2.ts.Edge
import org.mule.weave.v2.ts.EdgeLabels
import org.mule.weave.v2.ts.ErrorResult
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.FunctionTypeHelper
import org.mule.weave.v2.ts.FunctionTypeParameter
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.NoSolutionSet
import org.mule.weave.v2.ts.NothingType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.SolutionResult
import org.mule.weave.v2.ts.Substitution
import org.mule.weave.v2.ts.TypeCoercer
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.TypeType
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.WeaveTypeResolver
import org.mule.weave.v2.ts.WeaveTypeTraverse
import org.mule.weave.v2.utils.SeqUtils

import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

/**
  * This node handles the resolution of calling a function with a given set of arguments.
  */
object FunctionCallNodeResolver extends WeaveTypeResolver {

  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val functionEdge: Edge = getIncomingFunctionEdge(node)
    val argEdges: Seq[Edge] = argumentEdges(node)
    val typeParamEdges: Seq[Edge] = getTypeParamEdges(node)

    resolveFunctionCall(node, ctx, functionEdge, argEdges, typeParamEdges)
  }

  def resolveFunctionCall(node: TypeNode, ctx: WeaveTypeResolutionContext, functionEdge: Edge, argEdges: Seq[Edge], typeParamEdges: Seq[Edge]): Option[WeaveType] = {
    val typeParams: Seq[Option[WeaveType]] = typeParamEdges.map(tpE => {
      tpE.incomingType() match {
        case TypeType(actualType) => {
          Some(actualType.withLocation(tpE.source.astNode.location()))
        }
        case _ => None // This shouldn't happen as our grammar parses type params as type expressions, all of them should be a TypeType
      }
    })

    if (typeParams.exists(p => !p.isDefined)) {
      None
    } else {
      val arguments: Seq[WeaveType] = argEdges.map((paramEdge) => {
        val weaveType = paramEdge.incomingType()
        weaveType.withLocation(paramEdge.source.astNode.location())
        weaveType
      })
      resolveFunctionType(node, ctx, functionEdge.incomingType(), arguments, typeParams.map(_.get))
    }
  }

  def getIncomingFunctionEdge(node: TypeNode): Edge = {
    node.incomingEdge(EdgeLabels.FUNCTION).get
  }

  def argumentEdges(node: TypeNode): Seq[Edge] = {
    node.incomingEdges(EdgeLabels.ARGUMENT)
  }

  def getTypeParamEdges(node: TypeNode): Seq[Edge] = {
    node.incomingEdges(EdgeLabels.TYPE_PARAMETER)
  }

  def calculateExpectedTypeByParameter(functionType: WeaveType, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): scala.Seq[WeaveType] = {
    functionType match {
      case ft: FunctionType => {
        setNoImplicitBounds(ft)
        //Try to find the overloaded function that is accepted by the expected type
        val result = if (incomingExpectedType.isDefined) {
          val actualFunctionType = if (ft.isOverloaded()) {
            val matchedFunction = ft.overloads.find((ft) => {
              TypeHelper(ctx).canBeSubstituted(ft.returnType, incomingExpectedType.get, ctx)
            })
            matchedFunction.getOrElse(ft)
          } else {
            ft
          }
          val constrains: ConstraintSet = Constraint.collectConstrains(incomingExpectedType.get, actualFunctionType.returnType, ctx)
          val typeParameters = TypeHelper(ctx).collectTypeParameters(functionType)
          val constraintResult = constrains.resolve(ctx, coerce = false, typeParameters)
          constraintResult match {
            case ErrorResult(_) => {
              actualFunctionType.params.map(_.wtype)
            }
            case SolutionResult(substitution, _) => {
              actualFunctionType.params.map((param) => substitution.apply(ctx, param.wtype))
            }
          }
        } else {
          ft.params.map(_.wtype)
        }
        resetImplicitBounds(ft)
        result
      }
      case TypeParameter(_, Some(topType), _, _, _) => {
        calculateExpectedTypeByParameter(topType, incomingExpectedType, ctx)
      }
      case TypeParameter(_, _, Some(baseType), _, _) => {
        calculateExpectedTypeByParameter(baseType, incomingExpectedType, ctx)
      }
      case UnionType(of) => {
        val possibleParams = of.map((t) => calculateExpectedTypeByParameter(t, incomingExpectedType, ctx))
        if (possibleParams.exists(_.isEmpty)) {
          Seq()
        } else {
          possibleParams.reduce((value, acc) => {
            value.zip(acc).map((tuple) => TypeHelper(ctx).unify(Seq(tuple._1, tuple._2)))
          })
        }
      }
      case rt: ReferenceType => {
        calculateExpectedTypeByParameter(rt.resolveType(), incomingExpectedType, ctx)
      }
      case it: IntersectionType => {
        // TODO Specialize this case
        TypeHelper.resolveIntersection(it.of) match {
          case IntersectionType(of) => Seq()
          case otherType =>
            calculateExpectedTypeByParameter(otherType, incomingExpectedType, ctx)
        }
      }
      case _ => Seq()
    }

  }

  override def resolveExpectedType(node: TypeNode, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): Seq[(Edge, WeaveType)] = {
    getIncomingFunctionEdge(node).mayBeIncomingType() match {
      case Some(weaveType) => {
        val paramsTypes: Seq[WeaveType] = calculateExpectedTypeByParameter(weaveType, incomingExpectedType, ctx)
        argumentEdges(node).zip(paramsTypes)
      }
      case _ => Seq()
    }
  }

  private def resolveFunctionType(node: TypeNode, ctx: WeaveTypeResolutionContext, functionType: WeaveType, arguments: Seq[WeaveType], typeParams: Seq[WeaveType]): Option[WeaveType] = {
    functionType match {
      case ft: FunctionType => {
        resolveReturnType(ft, arguments, typeParams, node, ctx)
      }
      case TypeParameter(_, Some(topType), _, _, _) => {
        resolveFunctionType(node, ctx, topType, arguments, typeParams)
      }
      case TypeParameter(_, _, Some(baseType), _, _) => {
        resolveFunctionType(node, ctx, baseType, arguments, typeParams)
      }
      case UnionType(of) => {
        val options = of.map((t) => resolveFunctionType(node, ctx, t, arguments, typeParams))
        if (options.exists(_.isEmpty)) {
          None
        } else {
          Some(TypeHelper(ctx).unify(options.flatten))
        }
      }
      case rt: ReferenceType => {
        resolveFunctionType(node, ctx, rt.resolveType(), arguments, typeParams)
      }
      case _: NothingType => None
      case _: AnyType     => None
      case it: IntersectionType => {
        // TODO Specialize this case
        TypeHelper.resolveIntersection(it.of) match {
          case IntersectionType(of) => None
          case otherType =>
            resolveFunctionType(node, ctx, otherType, arguments, typeParams)
        }
      }
      case _ => {
        val expectedType: FunctionType = FunctionType(Seq(), arguments.map(FunctionTypeParameter("arg", _)), AnyType())
        if (!TypeHelper(ctx).canBeAssignedTo(functionType, expectedType, ctx))
          ctx.error(TypeMismatch(expectedType, functionType), node, functionType.location())
        None
      }
    }
  }

  def checkParametersCount(args: Seq[WeaveType], params: Seq[FunctionTypeParameter], ctx: WeaveTypeResolutionContext): Boolean = {
    val requiredParams: Seq[FunctionTypeParameter] = params.filter(!_.optional)
    if (args.size > params.size) {
      false
    } else if (args.size < requiredParams.size) {
      false
    } else {
      true
    }
  }

  /**
    * Checks if the function type parameter application count is valid. We don't allow partial type parameter application
    * so it is valid if:
    *   1. Type parameters are not defined and will be derivde from arguments
    *   2. All parameters are defined
    *
    *   @param types: The types specified at the function call.
    *   @param typeParams: Generic types specified in function definition
    *   @return type specification count is valid for type params
    */
  def checkTypeParameterCount(types: Seq[WeaveType], typeParams: Seq[TypeParameter]): Boolean = {
    types.isEmpty || (types.size == typeParams.size)
  }

  def filterIrrelevantProblems(problems: Seq[(FunctionType, Seq[(WeaveLocation, Message)])]): Seq[(FunctionType, Seq[(WeaveLocation, Message)])] = {
    problems.filterNot((pair) => {
      pair._2.forall({
        case (_, TypeMismatch(expectedType, _, _)) => expectedType.isInstanceOf[NullType]
        case _                                     => false
      })
    })
  }

  /**
    * Resolves the return type of invoking the provided function with the given arguments
    *
    * @param ft       The function to be invoked
    * @param args     The arguments
    * @param thisNode The node where this resolution needs to be done
    * @param ctx      The context
    * @return The return type
    */
  def resolveReturnType(ft: FunctionType, args: Seq[WeaveType], typeParams: Seq[WeaveType], thisNode: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val allDefinitions: Seq[FunctionType] = if (ft.overloads.isEmpty) Seq(ft) else ft.overloads
    val withCorrectArity: Seq[FunctionType] = allDefinitions.filter(fun => {
      checkParametersCount(args, fun.params, ctx) && checkTypeParameterCount(typeParams, fun.typeParams)
    })
    if (withCorrectArity.isEmpty) {
      addCardinalityErrors(args, typeParams, allDefinitions, thisNode, ctx)
      None
    } else {
      // We need to replace any abstract type parameter with a concrete one
      // as there is no real instance we create a fake one.
      // We also ignore DynamicTypeParameters, they should be kept abstract  so that they can be substituted later on.
      val fixedAbstractTypeParams = args.map((wt) => {
        val typeParameters = TypeHelper.collectAbstractTypeParameters(wt)
        val abstractToConcrete = typeParameters.filter(!FunctionTypeHelper.isDynamicTypeParameter(_)).map((atp) => {
          val concreteType = atp.copy(atp.name, atp.top, atp.bottom, Some(nextId()), atp.noImplicitBounds)
          (atp, concreteType)
        })
        Substitution(abstractToConcrete).apply(ctx, wt)
      })
      val argTypes: Seq[Seq[WeaveType]] = flattenUnion(fixedAbstractTypeParams)
      //We only support any function call with Dynamic or Nothing if it is only one
      val nothingOrDynamic: Boolean = argTypes.exists((args) => args.exists((arg) => arg.isInstanceOf[NothingType]))
      if (nothingOrDynamic && withCorrectArity.size != 1) {
        None
      } else {
        //Implement all the combination of all the types this is for union types
        val argTypeCombinations: Array[Seq[WeaveType]] = SeqUtils.combine(argTypes).toArray
        val responseTypes: Seq[ResolutionResult] =
          argTypeCombinations
            .map((typeCombination) => {
              resolve(thisNode, withCorrectArity, typeCombination, typeParams, ctx)
            })

        if (responseTypes.exists(_.failure)) {
          val name: String = ft.name.getOrElse(calculateFunctionName(thisNode))
          val problems: Seq[(FunctionType, Seq[(WeaveLocation, Message)])] = filterIrrelevantProblems(responseTypes.flatMap(result => {
            if (result.typeParamFailure) {
              // If there was a type param failure then messages are related to constraints.
              result.messages.map(problemMessage => {
                val wrappedProblem = problemMessage._2.map(msg => {
                  val transformedMessage = msg._2 match {
                    case t: TypeMessage => UnfulfilledConstraint(t)
                    case t              => t
                  }
                  (msg._1, transformedMessage)
                })
                (problemMessage._1, wrappedProblem)
              })
            } else {
              result.messages
            }
          }))
          if (problems.size == 1 && problems.head._2.size == 1) {
            problems.head._2.head match {
              case (_, idr: InvalidDynamicReturnMessage) => {
                ctx.error(idr.reason, thisNode, idr.messageLocation)
              }
              case (location, typeMessage: TypeMessage) => {
                ctx.error(typeMessage.addTrace(name, problems.head._1), thisNode, location)
              }
              case (location, _) => {
                ctx.error(InvalidMethodTypesMessage(name, args, problems).addTrace(name, problems.head._1), thisNode, location)
              }
            }
          } else {
            val errorLocations: Seq[WeaveLocation] = problems.flatMap((funcMessages) => funcMessages._2.map(_._1))
            val startLocation: WeaveLocation = if (errorLocations.isEmpty) UnknownLocation else errorLocations.minBy(_.startPosition.index)
            val endLocation: WeaveLocation = if (errorLocations.isEmpty) UnknownLocation else errorLocations.maxBy(_.endPosition.index)
            //We build a new location with all the possibilities this way if they are all together we can give a much better idea of where the problem is
            val newLocation = WeaveLocation(startLocation.startPosition, endLocation.endPosition, startLocation.resourceName)
            ctx.error(InvalidMethodTypesMessage(name, args, problems), thisNode, newLocation)
          }
          None
        } else {
          val results: Seq[WeaveType] = responseTypes.flatMap(_.result)
          if (results.isEmpty) {
            None
          } else {
            //We remove old DynamicReturnTypes
            thisNode
              .incomingEdges(EdgeLabels.DYNAMIC_RETURN_TYPE)
              .foreach((edge) => {
                edge.remove()
              })
            //We connect all the DynamicReturnTypes
            val dynamicReturnTypes: Seq[DynamicReturnType] = results
              .flatMap((wtype) => {
                WeaveTypeTraverse.shallowCollectAll(wtype, {
                  case drt: DynamicReturnType => Seq(drt)
                  case _                      => Seq()
                })
              })

            dynamicReturnTypes
              .foreach((drt) => {
                val maybeNode = FunctionTypeHelper.getFunctionSubGraphFor(drt, ctx)
                maybeNode.foreach((returnType) => {
                  Edge(returnType, thisNode, EdgeLabels.DYNAMIC_RETURN_TYPE)
                })
              })
            val result = TypeHelper(ctx).unify(results)
            //If it is DynamicReturnType then we should not return anything and wait for the graph to stabilize
            if (!isDynamicReturnType(result)) {
              Some(result)
            } else {
              None
            }
          }
        }
      }
    }
  }

  private def flattenUnion(args: Seq[WeaveType]): Seq[Seq[WeaveType]] = {
    args.map {
      case ut @ UnionType(of) => flattenUnion(of).flatten.map(_.withLocation(ut.location()))
      case actualType         => Seq(actualType)
    }
  }

  /**
    * We can not return a DynamicReturnType as it will be never resolved.
    * We replace it with Nothing as for now is like this part does not return any value.
    * DynamicReturnType can only be as a ReturnType of a FunctionType.
    * As a standalone type there is no way to substitute its parameters with a specific type so that it will be resolved.
    * That is why it will be enough with just adding the arrow.
    */
  private def isDynamicReturnType(result: WeaveType): Boolean = {
    result match {
      case _: DynamicReturnType => true
      case UnionType(of)        => of.exists(isDynamicReturnType)
      case _                    => false
    }
  }

  private def addCardinalityErrors(args: Seq[WeaveType], typeParams: Seq[WeaveType], allDefinitions: Seq[FunctionType], node: TypeNode, ctx: WeaveTypeResolutionContext): Unit = {
    val functionType = getIncomingFunctionEdge(node).incomingType()
    allDefinitions.foreach((fun) => {
      val requiredParams: Seq[FunctionTypeParameter] = fun.params.filter(!_.optional)
      if (args.size > fun.params.size) {
        ctx.error(TooManyArgumentMessage(fun.params.map(_.wtype), args, fun, functionType), node)
        false
      } else if (args.size < requiredParams.size) {
        ctx.error(NotEnoughArgumentMessage(requiredParams.map(_.wtype), args, fun, functionType), node)
        false
      } else if (typeParams.nonEmpty && typeParams.size != fun.typeParams.size) {
        ctx.error(InvalidAmountTypeParametersMessage(fun.typeParams.size, typeParams.size, fun, functionType), node)
      } else {
        true
      }
    })
  }

  def expandWithDefaultValues(invokedArguments: Seq[WeaveType], params: Seq[FunctionTypeParameter]): Seq[WeaveType] = {
    val expandedArguments: Seq[WeaveType] =
      if (params.nonEmpty && params.size != invokedArguments.size) {
        if (params.head.optional) {
          params.zipWithIndex.map((param: (FunctionTypeParameter, Int)) => {
            val index: Int = param._2
            val delta: Int = params.length - invokedArguments.size
            if (index < delta) {
              param._1.defaultValueType.getOrElse(param._1.wtype)
            } else {
              invokedArguments.apply(index - delta)
            }
          })
        } else {
          params.zipWithIndex.map((param: (FunctionTypeParameter, Int)) => {
            val index: Int = param._2
            if (invokedArguments.size > index) {
              invokedArguments.apply(index)
            } else {
              param._1.defaultValueType.getOrElse(param._1.wtype)
            }
          })
        }
      } else {
        invokedArguments
      }
    expandedArguments
  }

  def calculateFunctionName(node: TypeNode): String = {
    node.astNode match {
      case fcn: FunctionCallNode =>
        node.incomingEdges().head.source.astNode match {
          case VariableReferenceNode(variable, _) => variable.name
          case _                                  => FunctionTypeHelper.ANONYMOUS_FUNCTION_NAME
        }
      case _ => FunctionTypeHelper.ANONYMOUS_FUNCTION_NAME
    }
  }

  private def resolve(node: TypeNode, operators: Seq[FunctionType], actualTypes: Seq[WeaveType], typeParams: Seq[WeaveType], ctx: WeaveTypeResolutionContext): ResolutionResult = {
    val matchedOperator = collectValidOptions(operators, actualTypes, typeParams, node, ctx)
    if (matchedOperator.head.success) {
      matchedOperator.head
    } else {
      //First we try with simple coercions on declared types
      val functionWithCoercion = collectValidOptions(operators, actualTypes, typeParams, node, ctx, coerce = true, collectFirstSuccess = false)
      if (functionWithCoercion.size > 1) {
        multipleMatches(functionWithCoercion, actualTypes, node, ctx)
      } else if (functionWithCoercion.head.success) {
        functionWithCoercion.head
      } else {
        //We try with full coercion
        val functionsWithAllCoercions = collectValidOptions(operators, actualTypes, typeParams, node, ctx, coerce = true, coerceOnSubstitution = true, collectFirstSuccess = false)
        if (functionsWithAllCoercions.size > 1) {
          multipleMatches(functionsWithAllCoercions, actualTypes, node, ctx)
        } else {
          functionsWithAllCoercions.head
        }
      }
    }
  }

  private def multipleMatches(functionWithCoercion: mutable.Seq[ResolutionResult], actualTypes: Seq[WeaveType], node: TypeNode, ctx: WeaveTypeResolutionContext) = {
    //If all return types are the same with keep with that
    val types = TypeHelper(ctx).dedupTypes(functionWithCoercion.flatMap(_.result))
    if (types.size > 1) {
      val functionTypes = functionWithCoercion.flatMap(_.functionType)
      ctx.warning(MultipleValidFunctions(actualTypes, functionTypes), node)
      ResolutionResult(success = true, typeParametersSuccess = true, None, Some(AnyType()))
    } else {
      functionWithCoercion.head
    }
  }

  private def substituteResult(substitution: Substitution, returnType: WeaveType, ctx: WeaveTypeResolutionContext): WeaveType = {
    var substitute: WeaveType = Constraint.substitute(returnType, substitution, ctx, resolveDR = true)
    //Let's substitute with the type params in the context
    substitute = TypeHelper(ctx).simplifyIntersections(substitute)
    //Cleanup type parameters that were not resolved inside union types
    //We collect type params from the original return type to see what are still there
    val parameters = TypeHelper(ctx).collectTypeParameters(returnType)
    TypeHelper(ctx).simplifyUnions(TypeHelper(ctx).cleanupUnionTypeWithParameters(substitute, parameters))
  }

  def setNoImplicitBounds(weaveType: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(weaveType, {
      case tp: TypeParameter => {
        tp.noImplicitBounds = true
        tp
      }
    })
  }

  def resetImplicitBounds(weaveType: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(weaveType, {
      case tp: TypeParameter => {
        tp.noImplicitBounds = false
        tp
      }
    })
  }

  private def collectValidOptions(functions: Seq[FunctionType], arguments: Seq[WeaveType], typeParams: Seq[WeaveType], node: TypeNode, ctx: WeaveTypeResolutionContext, coerce: Boolean = false, coerceOnSubstitution: Boolean = false, collectFirstSuccess: Boolean = true): mutable.Seq[ResolutionResult] = {
    //Search valid operator
    val validOptions: ArrayBuffer[ResolutionResult] = ArrayBuffer[ResolutionResult]()
    var collectedProblems: Seq[(FunctionType, Seq[(WeaveLocation, Message)])] = Seq()
    var tpCollectedProblems: Seq[(FunctionType, Seq[(WeaveLocation, Message)])] = Seq()
    val functionTypes: Iterator[FunctionType] = functions.iterator

    while (functionTypes.hasNext && !(collectFirstSuccess && validOptions.headOption.nonEmpty)) {
      val functionType: FunctionType = functionTypes.next()
      var warningMessages = Seq[Message]()
      var calledArguments = Seq[WeaveType]()
      val paramTypes: Seq[WeaveType] = functionType.params.map(p => setNoImplicitBounds(p.wtype))
      val expandedArguments: Seq[WeaveType] = expandWithDefaultValues(arguments, functionType.params)
      var alreadyMissmatch = false

      //First we check that type parameter application is valid
      val typeParamConstraints: ConstraintSet = functionType.typeParams
        .zip(typeParams)
        .map(typeParamTuple => Constraint.collectConstrains(typeParamTuple._1, typeParamTuple._2, ctx))
        .foldLeft[ConstraintSet](ConstrainProblem(Seq()))(_.merge(_))

      typeParamConstraints match {
        case NoSolutionSet(problems) => {
          tpCollectedProblems = tpCollectedProblems :+ (functionType, problems)
        }
        case _ => {
          typeParamConstraints.resolve(ctx, coerce, functionType.typeParams) match {
            case ErrorResult(problems) => {
              //If type parameter application is invalid we collect problems for this function
              //* and skip checking typing of arguments
              tpCollectedProblems = tpCollectedProblems :+ (functionType, problems)
            }
            case SolutionResult(substitution, _) => {
              //We update function parameter typing with what we know given the type parameter application
              val updatedParamTypes = paramTypes.map(substitution.apply(ctx, _))
              val constraintSet: ConstraintSet = updatedParamTypes
                .zip(expandedArguments)
                .map({
                  case (paramType, argType) => {
                    val constrains: ConstraintSet = Constraint.collectConstrains(paramType, argType, ctx)
                    constrains match {
                      case NoSolutionSet(_) if coerce => {
                        val maybeType = TypeCoercer.coerce(paramType, argType, ctx)
                        maybeType match {
                          //If already a missmatch do not try to coerce
                          case Some(coercedArg) if !alreadyMissmatch => {
                            warningMessages = warningMessages :+ TypeCoercedMessage(paramType, argType)
                            calledArguments = calledArguments :+ coercedArg
                            Constraint.collectConstrains(paramType, coercedArg, ctx)
                          }
                          case _ => {
                            alreadyMissmatch = true
                            constrains
                          }
                        }
                      }
                      case _ => {
                        calledArguments = calledArguments :+ argType
                        constrains
                      }
                    }
                  }
                })
                .foldLeft[ConstraintSet](typeParamConstraints)(_.merge(_))

              val solution: ConstraintResult = constraintSet.resolve(ctx, coerceOnSubstitution, functionType.typeParams)
              solution match {
                case ErrorResult(problems) =>
                  collectedProblems = collectedProblems :+ (functionType, problems)
                case SolutionResult(substitution, warnings) => {
                  //Log warning messages
                  warningMessages.foreach((warMessage) => ctx.warning(warMessage, node))
                  val returnType: Option[WeaveType] = resolveReturnType(node, ctx, functionType, calledArguments, substitution, warnings)
                  validOptions += ResolutionResult(success = true, typeParametersSuccess = true, Some(functionType), returnType)
                }
              }
              //close the types
              paramTypes.map(resetImplicitBounds)
            }
          }
        }
      }
    }
    if (validOptions.isEmpty) {
      /**
        * Collected problems means that a function was matched against its generics
        * In that case we decide to prioritize its error over the errors with functions
        * that didn't match. eg
        *
        *   fun f<T <: String>(x: T) = x
        *   fun f<T <: {}>(x: T) = x
        *   fun f<T <: {a: String}>(x: T) = x
        *   ---
        *   f<{a: String}>({a: {}})
        *
        * In this case we want to show errors regarding the second and third definition of f,
        * as the type parameter application is valid for those, we omit showing errors for the
        * first definition
        */
      if (collectedProblems.isEmpty) {
        validOptions.+=(ResolutionResult(typeParametersSuccess = false, messages = tpCollectedProblems))
      } else {
        validOptions.+=(ResolutionResult(messages = collectedProblems))
      }
    }
    validOptions
  }

  private def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext, functionType: FunctionType, calledArguments: Seq[WeaveType], substitution: Substitution, warnings: Seq[Message]) = {
    warnings.foreach((warMessage) => ctx.warning(warMessage, node))
    val resolvedReturnType: WeaveType = substituteResult(substitution, functionType.returnType, ctx)
    val result = functionType.customReturnTypeResolver match {
      case Some(crtr) =>
        crtr.resolve(calledArguments, ctx, node, resolvedReturnType).map(substituteResult(substitution, _, ctx))
      case _ =>
        Some(resolvedReturnType)
    }
    result
  }
}
