package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.ts.Edge
import org.mule.weave.v2.ts.EdgeLabels
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.WeaveTypeResolver

object HeadTailObjectTypeResolver extends WeaveTypeResolver {

  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val headKeyType: WeaveType = headKeyEdge(node).incomingType()
    val headValueType: WeaveType = headValueEdge(node).incomingType()
    val tailType: WeaveType = tailEdge(node).incomingType()
    val headKvpType = KeyValuePairType(headKeyType, headValueType)
    val keyValuePairTypes: Seq[KeyValuePairType] = resolveTailType(tailType, headKvpType)
    Some(ObjectType(keyValuePairTypes))
  }

  private def headValueEdge(node: TypeNode) = {
    node.incomingEdges(EdgeLabels.HEAD_VALUE).head
  }

  private def tailEdge(node: TypeNode) = {
    node.incomingEdges(EdgeLabels.TAIL).head
  }

  private def headKeyEdge(node: TypeNode) = {
    node.incomingEdges(EdgeLabels.HEAD_KEY).head
  }

  private def resolveTailType(tailType: WeaveType, headKvpType: KeyValuePairType): Seq[KeyValuePairType] = {
    val keyValuePairTypes = tailType match {
      case ObjectType(t, _, _) => t :+ headKvpType
      case rt: ReferenceType   => resolveTailType(rt.resolveType(), headKvpType)
      case _                   => Seq(headKvpType)
    }
    keyValuePairTypes
  }

  override def resolveExpectedType(node: TypeNode, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): Seq[(Edge, WeaveType)] = {
    incomingExpectedType match {
      case Some(objectType: ObjectType) => {
        val objectValues: WeaveType = TypeHelper.unify(objectType.properties.map(_.value))
        Seq((headValueEdge(node), objectValues), (tailEdge(node), objectType))
      }
      case _ => Seq()
    }
  }
}
