package org.mule.weave.v2.utils

import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.scope.VariableScope

object VariableScopeDotEmitter {

  def print(scope: VariableScope): String = {
    val writer: StringCodeWriter = new StringCodeWriter()
    val name = scope.name.getOrElse(scope.astNode.getClass.getSimpleName)
    writer.indent()
    writer.print(s"digraph ${sanitize(name)} {\n")
    writer.println("node [style=filled];")
    printDeclarations(writer, scope)
    writer.dedent()
    writer.println("label = \"" + scope.parsingContext.nameIdentifier + "\";")
    writer.print("}")
    writer.toString()
  }

  private def printScope(writer: StringCodeWriter, scope: VariableScope) = {
    val name = scope.name.getOrElse(scope.astNode.getClass.getSimpleName)
    writer.println()

    writer.printIndent()
    writer.print(s"subgraph cluster${sanitize(name)} {\n")
    writer.indent()
    writer.printIndent()
    writer.println("node [style=filled];")
    printDeclarations(writer, scope)
    writer.dedent()
    writer.printIndent()
    writer.println("label = \"" + sanitize(name) + "\";")
    writer.println("}")
    writer.toString()
  }

  private def printDeclarations(builder: StringCodeWriter, scope: VariableScope): Unit = {
    scope.declarations.foreach((nameIdentifier) => {
      val name = "[D] " + nameIdentifier.name + "  " + nameIdentifier.location().startPosition.line + ":" + nameIdentifier.location().endPosition.column
      builder.printIndent()
      builder.println(s"${id(nameIdentifier)} [label=" + "\"" + sanitize(name) + "\"" + "][fillcolor = yellow];")
    })
    scope.references.foreach((nameIdentifier) => {
      builder.printIndent()
      val name = "[R] " + nameIdentifier.name + "  " + nameIdentifier.location().startPosition.line + ":" + nameIdentifier.location().endPosition.column
      builder.println(s"${id(nameIdentifier)} [label=" + "\"" + sanitize(name) + "\"" + "][fillcolor = green];")
    })

    scope.children().foreach((scope) => printScope(builder, scope))

    var arrow = false
    scope.references.foreach((nameIdentifier) => {
      scope.resolveVariable(nameIdentifier) match {
        case Some(r) =>
          if (r.isLocalReference) {
            arrow = true
            builder.printIndent()
            builder.println(s"${id(nameIdentifier)} -> ${id(r.referencedNode)}")
          }
        case _ =>
      }
    })
    if (arrow) {
      builder.printIndent()
      builder.println(";")
    }
  }

  private def sanitize(name: String) = {
    name.replaceAll("\"", "\'")
  }

  def id(node: NameIdentifier): String = {
    Math.abs(System.identityHashCode(node)).toString
  }
}

