package org.mule.weave.v2.weavedoc

import java.lang

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.codegen.CodeGeneratorSettings
import org.mule.weave.v2.codegen.InfixOptions
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.InvalidSyntaxMessage
import org.mule.weave.v2.parser.InvalidWeaveDocSyntaxMessage
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.ParserPosition
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.sdk.WeaveResource
import org.parboiled2.ErrorFormatter
import org.parboiled2.ParseError
import org.parboiled2.ParserInput

import scala.util.Failure
import scala.util.Success
import org.parboiled2.ErrorFormatter
import org.parboiled2.ParseError
import org.parboiled2.ParserInput

object WeaveDocParser {

  def parseAst(documentation: String, messageCollector: MessageCollector): Option[WeaveDocNode] = {
    val grammar = new WeaveDocGrammar(documentation)
    val result = grammar.weaveDoc.run()
    result match {
      case Failure(t) => {
        t match {
          case y: ParseError => {
            val format: String = y.format(grammar.input, new ErrorFormatter() {
              override def formatErrorLine(sb: lang.StringBuilder, error: ParseError, input: ParserInput): lang.StringBuilder = {
                sb
              }
            })
            val start = ParserPosition(y.position, grammar.input)
            val end = ParserPosition(y.principalPosition, grammar.input)
            val location: WeaveLocation = WeaveLocation(start, end, NameIdentifier.anonymous)
            messageCollector.error(InvalidWeaveDocSyntaxMessage(format), location)
          }
          case _ =>
        }
        None
      }
      case Success(value) => Some(value)
    }
  }

  def parseDocumentation(documentation: String): WeaveDocumentation = {
    val maybeNode = parseAst(documentation, new MessageCollector)
    maybeNode match {
      case Some(weavedoc) => {
        val smallDescription = weavedoc.description.map(_.content)
        val description = weavedoc.content.map(_.content)
        val parameters = weavedoc.sections.collectFirst {
          case psn: ParametersSectionNode =>
            val params = psn.params.map(p => {
              WeaveDocParameter(p.name.content, p.weaveType.map(_.content), p.description.content)
            })
            WeaveDocParameters(psn.style.content, params)
        }

        val examples = weavedoc.sections.collect({
          case en: ExampleSectionNode => {
            val description = en.descriptionNode.map(_.content)
            val input: Seq[CodeBlockWithDescription] = en.sections.collectFirst({
              case isn: InputSectionNode => {
                isn.content.map(in => {
                  val code = CodeBlock(in.code.content, in.code.info.map(_.langType))
                  CodeBlockWithDescription(in.description.map(_.content), code)
                })
              }
            }).getOrElse(Seq())
            val output: Seq[CodeBlockWithDescription] = en.sections.collectFirst({
              case osn: OutputSectionNode => {
                osn.content.map(out => {
                  val code = CodeBlock(out.code.content, out.code.info.map(_.langType))
                  CodeBlockWithDescription(out.description.map(_.content), code)
                })
              }
            }).getOrElse(Seq())
            val source = en.sections.collectFirst({
              case source: SourceSectionNode => {
                CodeBlock(source.content.content, source.content.info.map(_.langType))
              }
            })
            WeaveDocExample(description, input, output, source)
          }
        })

        val moreExamples = weavedoc.sections.collectFirst {
          case mesn: MoreExamplesSectionNode =>
            WeaveDocMoreExamples(mesn.descriptionNode.content)
        }
        WeaveDocumentation(smallDescription, description, parameters, examples, moreExamples)
      }
      case None => WeaveDocumentation(Some(documentation))
    }
  }

}

case class WeaveDocumentation(smallDescription: Option[String], description: Option[String] = None, parameters: Option[WeaveDocParameters] = None, examples: Seq[WeaveDocExample] = Seq(),
  moreExamples: Option[WeaveDocMoreExamples] = None) {
  def asInfix(parsingContext: ParsingContext) = {
    this.copy(examples = this.examples.map(_.asInfix(parsingContext)))
  }

  def asPrefix(parsingContext: ParsingContext) = {
    this.copy(examples = this.examples.map(_.asPrefix(parsingContext)))
  }
}

case class WeaveDocParameters(style: String, params: Seq[WeaveDocParameter] = Seq())

case class WeaveDocParameter(name: String, weaveType: Option[String] = None, description: String = "")

case class WeaveDocExample(description: Option[String], input: Seq[CodeBlockWithDescription], output: Seq[CodeBlockWithDescription], code: Option[CodeBlock]) {
  def asInfix(parsingContext: ParsingContext) = {
    this.copy(code = code.map(_.asInfixExample(parsingContext)))
  }

  def asPrefix(parsingContext: ParsingContext) = {
    this.copy(code = code.map(_.asPrefixExample(parsingContext)))
  }
}

case class CodeBlockWithDescription(description: Option[String], code: CodeBlock)

case class CodeBlock(script: String, langType: Option[String]) {
  def asInfixExample(parsingContext: ParsingContext): CodeBlock = {
    val value = DocumentParser(0).parse(WeaveResource("", script), parsingContext)
    if (value.hasResult()) {
      val document = value.getResult()
      val result = CodeGenerator.generate(document.astNode, CodeGeneratorSettings(InfixOptions.ALWAYS))
      CodeBlock(result, langType)
    } else {
      this
    }
  }

  def asPrefixExample(parsingContext: ParsingContext): CodeBlock = {
    val value = DocumentParser(0).parse(WeaveResource("", script), parsingContext)
    if (value.hasResult()) {
      val document = value.getResult()
      val result = CodeGenerator.generate(document.astNode, CodeGeneratorSettings(InfixOptions.ALWAYS))
      CodeBlock(result, langType)
    } else {
      this
    }
  }
}

case class WeaveDocMoreExamples(description: String)
