package org.mule.weave.v2.editor

import org.mule.weave.v2.api.tooling.annotation.DWAnnotationProcessor
import org.mule.weave.v2.completion.DataFormatDescriptorProvider
import org.mule.weave.v2.editor.indexing.SimpleWeaveIndexService
import org.mule.weave.v2.editor.indexing.WeaveIndexService
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ModuleParser
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.module.EmptyModuleLoaderProvider
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.parser.phase.DependencyGraph
import org.mule.weave.v2.parser.phase.ImplicitInputData
import org.mule.weave.v2.parser.phase.ModuleLoader
import org.mule.weave.v2.parser.phase.ModuleLoaderManager
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.WithDependencyGraphParsingPhasesManager
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.CacheBuilder
import org.mule.weave.v2.utils.DependencyGraphDotEmitter
import org.mule.weave.v2.utils.WeaveTypeEmitter
import org.mule.weave.v2.utils.WeaveTypeParser
import org.mule.weave.v2.versioncheck.SVersion

import scala.collection.mutable

/**
  * Represents an editing session of a weave project. It keeps the state of all open editors and interacts with the events of the VirtualFileSystem.
  *
  * @param vfs                The file system that contains the project being edited
  * @param dataFormatProvider The service that provides the runtime information
  * @param specificLoaders    Extension to inject custom module loaders
  */
class WeaveToolingService(val vfs: VirtualFileSystem, var dataFormatProvider: DataFormatDescriptorProvider, val specificLoaders: Array[_ <: ModuleLoaderFactory] = Array[ModuleLoaderFactory]()) {

  private var maxAmountOfOpenEditors = 100

  private val configuration: WeaveToolingConfiguration = WeaveToolingConfiguration()

  private val editorsCache = CacheBuilder[String, WeaveDocumentToolingService]()
    .maximumSize(maxAmountOfOpenEditors)
    .build()

  private val dependencyGraph: DependencyGraph = new DependencyGraph()

  /**
    * Holds the default instance of the index service to be used
    */
  private var defaultIndexService: Option[WeaveIndexService] = None

  private lazy val indexService: WeaveIndexService = defaultIndexService.getOrElse(new SimpleWeaveIndexService(vfs, (nameIdentifier) => createParsingContext(nameIdentifier)))

  private val modulesManager: WithDependencyGraphParsingPhasesManager = {
    //Client module loaders
    val clientModuleLoaders = specificLoaders.map((moduleResolver) => {
      moduleResolver.createModuleLoader()
    })
    val loaderManager = ModuleLoaderManager(ModuleLoader(vfs.asResourceResolver) +: clientModuleLoaders, EmptyModuleLoaderProvider)
    val parsingPhasesManager = ModuleParsingPhasesManager(loaderManager)
    WithDependencyGraphParsingPhasesManager(parsingPhasesManager, dependencyGraph)
  }

  private val annotationProcessors: mutable.Map[NameIdentifier, DWAnnotationProcessor] = mutable.HashMap()

  {
    vfs.changeListener(new ChangeListener() {

      override def onChanged(vf: VirtualFile): Unit = {
        invalidateModule(vf)
      }

      override def onDeleted(vf: VirtualFile): Unit = {
        close(vf)
        invalidateModule(vf)
      }

    })
  }

  /**
    * Register a parsing annotation processor
    *
    * @param nameIdentifier The nameIdentifier of the specified annotation to process
    * @param ap             The annotation processor
    * @return this
    */
  def registerAnnotationProcessor(nameIdentifier: NameIdentifier, ap: DWAnnotationProcessor): WeaveToolingService = {
    this.annotationProcessors.put(nameIdentifier, ap)
    this
  }

  def withSymbolsIndexService(symbols: WeaveIndexService): WeaveToolingService = {
    this.defaultIndexService = Some(symbols)
    this
  }

  /**
    * Returns the modules that depends
    *
    * @param nameIdentifier The nameIdentifier of the module
    * @return The list of Modules or Mappings that depends from the given module
    */
  def dependantsOf(nameIdentifier: NameIdentifier): Seq[NameIdentifier] = {
    dependencyGraph.getDependants(nameIdentifier)
  }

  /**
    * Invalidates the caches of the given module
    *
    * @param nameIdentifier of the module name
    */
  def invalidateModule(nameIdentifier: NameIdentifier): Unit = {
    val weaveResource = vfs.asResourceResolver.resolve(nameIdentifier)
    if (weaveResource.isDefined) {
      invalidateModule(nameIdentifier, weaveResource.get.url())
    }
  }

  /**
    * Invalidates the caches of the given module
    *
    * @param virtualFile The module name
    */
  def invalidateModule(virtualFile: VirtualFile): Unit = {
    //Invalidate data for this element
    val nameIdentifier = virtualFile.getNameIdentifier
    val url: String = virtualFile.url()
    invalidateModule(nameIdentifier, url)
  }

  private def invalidateModule(nameIdentifier: NameIdentifier, url: String): Unit = {
    modulesManager.invalidateModule(nameIdentifier)
    invalidateEditor(url)

    //Invalidate data for its dependencies
    dependencyGraph
      .getDependants(nameIdentifier)
      .foreach((dependency) => {
        modulesManager.invalidateModule(dependency)
        val maybeResource = vfs.asResourceResolver.resolve(dependency)
        if (maybeResource.isDefined) {
          invalidateEditor(maybeResource.get.url())
        }
      })

    //Invalidate dependency graph information
    dependencyGraph.invalidateModule(nameIdentifier)
  }

  private def invalidateEditor(url: String): Unit = {
    editorsCache.remove(url)
  }

  /**
    * Returns the dependency graph of all the loaded projects in a Dot representation
    *
    * @return
    */
  def dependencyGraphString(): String = {
    DependencyGraphDotEmitter.print(dependencyGraph)
  }

  /**
    * Returns the list of open editors
    *
    * @return All the open editors
    */
  def openEditors(): Seq[WeaveDocumentToolingService] = {
    editorsCache.values().toSeq
  }

  /**
    * Changes the language level
    *
    * @param version The language version
    * @return
    */
  def updateLanguageLevel(version: Option[SVersion]): WeaveToolingService = {
    if (configuration.languageLevel != version) {
      configuration.languageLevel = version
      invalidateAll()
    }
    this
  }

  /**
    * Specifies the max amount of open concurrent editors
    *
    * @param maxOpenEditor The max amount
    * @return This instance
    */
  def maxAmountOfOpenEditors(maxOpenEditor: Int): WeaveToolingService = {
    this.maxAmountOfOpenEditors = maxOpenEditor
    this
  }

  /**
    * Invalidates all cached state
    *
    * @return This instance
    */
  def invalidateAll(): WeaveToolingService = {
    modulesManager.invalidateAll()
    editorsCache.clear()
    this
  }

  /**
    * Opens a Document Editor Service
    *
    * @param path The path of the file to be opened
    * @return This instance
    */
  def open(path: String): WeaveDocumentToolingService = {
    val file = vfs.file(path)
    if (file == null) {
      throw new IllegalArgumentException(s"Unable to find file with path ${path}")
    }
    open(file)
  }

  /**
    * Opens a Document Editor Service
    *
    * @param path           The path of the file to be opened
    * @param inputs         The implicits input types
    * @param expectedOutput The expected output
    * @return This instance
    */
  def open(path: String, inputs: ImplicitInput, expectedOutput: Option[WeaveType]): WeaveDocumentToolingService = {
    val file = vfs.file(path)
    if (file == null) {
      throw new IllegalArgumentException(s"Unable to find file with path ${path}")
    }
    open(file, inputs, expectedOutput)
  }

  /**
    * Close the editor of this file
    *
    * @param file The file of the editor to be closed
    */
  def close(file: VirtualFile): WeaveToolingService = {
    close(file.url())
    this
  }

  /**
    * Close the editor of this file
    *
    * @param fileUrl The url of the file
    */
  def close(fileUrl: String): Unit = {
    editorsCache.remove(fileUrl)
  }

  /**
    * Close all editors
    */
  def closeAll(): Unit = {
    editorsCache.clear()
  }

  /**
    * Opens an editor in memory. This means that this editor is not going to be cached. Useful for short life editors
    *
    * @param currentFile    The file to open
    * @param inputs         The implicit inputs
    * @param expectedOutput The output type
    * @return
    */
  def openInMemory(currentFile: VirtualFile, inputs: ImplicitInput = ImplicitInput(), expectedOutput: Option[WeaveType] = None): WeaveDocumentToolingService = {
    val nameIdentifier = currentFile.getNameIdentifier
    val parsingContext = createParsingContext(nameIdentifier)
    val service = WeaveDocumentToolingService(parsingContext, currentFile, vfs, dataFormatProvider, configuration, indexService, inputs, expectedOutput)
    service
  }

  def open(currentFile: VirtualFile, inputs: ImplicitInput = ImplicitInput(), expectedOutput: Option[WeaveType] = None): WeaveDocumentToolingService = {
    if (currentFile == null) {
      throw new IllegalArgumentException("File can not be empty")
    }
    val editor = getEditorFor(currentFile, inputs, expectedOutput)
    editor
  }

  private def getEditorFor(currentFile: VirtualFile, inputs: ImplicitInput, expectedOutput: Option[WeaveType]): WeaveDocumentToolingService = {
    val url = currentFile.url()
    val maybeEditor = editorsCache.getIfPresent(url)
    // Invalidate editor if input or output are different
    if (maybeEditor.isDefined) {
      val editor = maybeEditor.get
      if (!editor.inputs.equals(inputs) || !editor.expectedOutput.equals(expectedOutput)) {
        invalidateEditor(url)
      }
    }
    editorsCache.get(
      currentFile.url(),
      (_) => {
        val nameIdentifier = currentFile.getNameIdentifier
        val parsingContext = createParsingContext(nameIdentifier)
        WeaveDocumentToolingService(parsingContext, currentFile, vfs, dataFormatProvider, configuration, dependencyGraph, indexService, inputs, expectedOutput)
      })
  }

  def createParsingContext(nameIdentifier: NameIdentifier): ParsingContext = {
    val context = ParsingContext(nameIdentifier, new MessageCollector, modulesManager, strictMode = false)
    annotationProcessors.foreach((pp) => {
      context.registerAnnotationProcessor(pp._1, pp._2)
    })
    context
  }

  /**
    * Parses a string representation of a WeaveType
    *
    * @param typeText The string of the type
    * @return The type if it can be parsed
    */
  @deprecated(message = "use loadType as this string representation does not support namespaces", since = "2.1.3")
  def parseType(typeText: String): Option[WeaveType] = {
    val parsingContext = createParsingContext(NameIdentifier.anonymous)
    WeaveTypeParser.parse(typeText, parsingContext)
  }

  /**
    * Parses a type serialized in a catalog mode.
    */
  def loadType(catalog: String): Option[WeaveType] = {
    val value = ModuleParser.parse(ModuleParser.scopePhase(), WeaveResource.anonymous(catalog), createParsingContext(NameIdentifier.anonymous))
    if (value.hasResult()) {
      val astNode = value.getResult().astNode
      val typeDirective = AstNodeHelper.collectDirectChildrenWith(astNode, classOf[TypeDirective]).find((typeDirective) => typeDirective.variable.name.equals(WeaveTypeEmitter.ROOT_WEAVE_TYPE_NAME))
      if (typeDirective.nonEmpty) {
        Some(WeaveType(typeDirective.get.typeExpression, new ScopeGraphTypeReferenceResolver(value.getResult().scope)))
      } else {
        None
      }
    } else {
      None
    }
  }

}

object WeaveToolingService {

  val MAX_OPEN_EDITORS = 100

  def apply(vfs: VirtualFileSystem, dataFormatProvider: DataFormatDescriptorProvider, specificLoaders: Array[SpecificModuleResourceResolver]): WeaveToolingService = {
    new WeaveToolingService(vfs, dataFormatProvider, specificLoaders)
  }
}

/**
  * The configuration
  *
  * @param languageLevel The language level
  */
case class WeaveToolingConfiguration(var languageLevel: Option[SVersion] = None) {}

class ImplicitInput {

  private val inputs = new mutable.ArrayBuffer[ImplicitInputData]()

  def addInput(name: String, weaveType: WeaveType): ImplicitInput = {
    inputs.+=(ImplicitInputData(name, Some(weaveType), None))
    this
  }

  def addInput(name: String, weaveType: WeaveType, mimeType: MimeType): ImplicitInput = {
    inputs.+=(ImplicitInputData(name, Some(weaveType), Some(mimeType)))
    this
  }

  def implicitInputs(): Seq[ImplicitInputData] = {
    inputs
  }

  def clear(): Unit = {
    inputs.clear()
  }

  def canEqual(other: Any): Boolean = other.isInstanceOf[ImplicitInput]

  override def equals(other: Any): Boolean = other match {
    case that: ImplicitInput =>
      (that canEqual this) &&
        inputs == that.inputs
    case _ => false
  }

  override def hashCode(): Int = {
    val state = Seq(inputs)
    state.map(_.hashCode()).foldLeft(0)((a, b) => 31 * a + b)
  }
}

object ImplicitInput {
  def apply(): ImplicitInput = new ImplicitInput()

  def apply(inputs: Map[String, WeaveType]): ImplicitInput = {
    val result = new ImplicitInput()
    inputs.foreach((pair) => {
      result.addInput(pair._1, pair._2)
    })
    result
  }
}

/**
  * Creates a module loader
  */
trait ModuleLoaderFactory {
  def createModuleLoader(): ModuleLoader
}

case class DefaultModuleLoaderFactory(ml: ModuleLoader) extends ModuleLoaderFactory {
  override def createModuleLoader(): ModuleLoader = ml
}

case class SpecificModuleResourceResolver(name: String, resourceResolver: WeaveResourceResolver) extends ModuleLoaderFactory {
  override def createModuleLoader(): ModuleLoader = {
    ModuleLoader(resourceResolver, name)
  }
}
