package org.mule.weave.v2.grammar

import org.mule.weave.v2.parser.ErrorAstNode
import org.parboiled2._

trait Tokens extends WhiteSpaceHandling with ErrorRecovery {
  this: org.mule.weave.v2.grammar.Tokens with Parser =>

  val ValidStartName = CharPredicate.Alpha
  val ValidNameChar = CharPredicate.AlphaNum ++ '_'

  def dot: Rule0 = namedRule(".") {
    atomic(ch('.'))
  }

  def squareBracketOpen: Rule0 = rule {
    ch('[') ~ ws
  }

  def atToken: Rule0 = rule {
    ws ~ ch('@')
  }

  def starToken: Rule0 = rule {
    ws ~ ch('*')
  }

  def squareBracketEnd: Rule0 = rule {
    ws ~ ch(']')
  }

  def commaSep: Rule0 = namedRule(",") {
    ws ~ ch(',') ~ ws
  }

  def attributesStart: Rule0 = rule {
    str("@(") ~ ws
  }

  def attributesEnd: Rule0 = rule {
    ws ~ ch(')')
  }

  def curlyBracketsStart: Rule0 = namedRule("{") {
    atomic(ch('{') ~ ws)
  }

  def curlyBracketsEnd: Rule0 = namedRule("}") {
    ws ~ atomic(ch('}'))
  }

  def objFieldSep: Rule0 = namedRule(":") {
    atomic(ch(':') ~ ws)
  }

  def dollar: Rule0 = namedRule("$") {
    atomic(ch('$'))
  }

  def variableAsngStart: Rule0 = namedRule("&") {
    atomic(ch('&'))
  }

  def parenStart: Rule0 = rule {
    ch('(') ~ ws
  }

  def backTick: Rule0 = rule {
    ch('`')
  }

  def parenEnd: Rule0 = rule {
    ws ~ ch(')')
  }

  def regexStart: Rule0 = namedRule("/") {
    atomic(ch('/'))
  }

  def regexEnd: Rule0 = namedRule("/") {
    atomic(ch('/'))
  }

  def questionMark: Rule0 = namedRule("?") {
    atomic(ch('?'))
  }

  def exclamationMark: Rule0 = namedRule("!") {
    atomic(ch('!'))
  }

  def star: Rule0 = namedRule("*") {
    atomic(ch('*'))
  }

  def assignment: Rule0 = namedRule("=") {
    ws ~ atomic(ch('=') ~ !ch('='))
  }

  def lambdaMark: Rule0 = namedRule("->") {
    atomic(ws ~ str("->") ~ ws)
  }

  def caseKeyword: Rule0 = namedRule("case") {
    (ws ~ str("case") ~ fws)
  }

  def atKeyword: Rule0 = namedRule("at") {
    atomic(ws ~ str("at") ~ ws)
  }

  def isKeyword: Rule0 = namedRule("is") {
    atomic(str("is") ~ fws)
  }

  def equals: Rule0 = namedRule("==") {
    atomic(str("==") ~ ws)
  }

  def similar: Rule0 = namedRule("~=") {
    atomic(str("~=") ~ ws)
  }

  def notEquals: Rule0 = namedRule("!=") {
    atomic(str("!=") ~ ws)
  }

  def lessThan: Rule0 = namedRule("<") {
    atomic(ch('<') ~ !anyOf("<=~") ~ ws)
  }

  def greaterThan: Rule0 = namedRule(">") {
    atomic(ch('>') ~ !anyOf("<=") ~ ws)
  }

  def plus: Rule0 = namedRule("+") {
    atomic(ch('+') ~ !ch('+') ~ ws)
  }

  def append: Rule0 = namedRule("++") {
    atomic(str("++") ~ ws)
  }

  def minus: Rule0 = namedRule("-") {
    atomic(ch('-') ~ !(ch('-') | ch('>')) ~ ws)
  }

  def tilde: Rule0 = namedRule("~") {
    atomic(ws ~ ch('~') ~ ws)
  }

  def pipe: Rule0 = namedRule("|") {
    atomic(ws ~ ch('|') ~ ws)
  }

  def amp: Rule0 = namedRule("&") {
    atomic(ws ~ ch('&') ~ ws)
  }

  def dateSeparator: Rule0 = namedRule("|") {
    atomic(ch('|'))
  }

  def remove: Rule0 = namedRule("--") {
    atomic(str("--") ~ !ch('-') ~ ws)
  }

  def rightShift: Rule0 = namedRule(">>") {
    atomic(str(">>") ~ ws)
  }

  def leftShift: Rule0 = namedRule("<<") {
    atomic(str("<<") ~ ws)
  }

  def metadataInjector: Rule0 = namedRule("<~") {
    atomic(str("<~") ~ ws)
  }

  def multiply: Rule0 = namedRule("*") {
    atomic(ch('*') ~ ws)
  }

  def divide: Rule0 = namedRule("/") {
    atomic(ch('/') ~ ws)
  }

  /**
    * Tries to parse `%dw`, pushing any missing postfix
    * @return
    */
  def versionDirectiveName: Rule1[Option[ErrorAstNode]] = namedRule("dw") {
    ch('%') ~!~
      ((ch('d') ~ (
        (ch('w') ~ push(None))
        | missingToken("Missing version token i.e. %dw", "w") ~> (a => Some(a)) ~ injectErrorPosition))
        | missingToken("Missing version token i.e. %dw", "dw") ~> (a => Some(a)) ~ injectErrorPosition) ~ fws
  }

  def namespaceDirectiveName: Rule0 = namedRule("ns") {
    atomic(str(Tokens.NS) ~ fws)
  }

  def outputDirectiveName: Rule0 = namedRule("output") {
    atomic(str(Tokens.OUTPUT) ~ fws)
  }

  def typeDirectiveName: Rule0 = namedRule("type") {
    atomic(str(Tokens.TYPE) ~ fws)
  }

  def inputDirectiveName: Rule0 = namedRule("input") {
    atomic(str(Tokens.INPUT) ~ fws)
  }

  def varDirectiveName: Rule0 = namedRule("var") {
    atomic(str(Tokens.VAR) ~ fws)
  }

  def functionDirectiveName: Rule0 = namedRule("fun") {
    atomic(str(Tokens.FUNCTION) ~ fws)
  }

  def importDirectiveName: Rule0 = namedRule("import") {
    atomic(str(Tokens.IMPORT) ~ fws)
  }

  def annotationDirectiveName: Rule0 = namedRule("annotation") {
    atomic(str(Tokens.ANNOTATION) ~ fws)
  }

  def fromKeyword: Rule0 = namedRule("from") {
    atomic(str("from"))
  }

  def headerSeparator: Rule0 = namedRule("---") {
    atomic(str(Tokens.DOCUMENT_SEPARATOR))
  }

  def lessOrEqualThan: Rule0 = namedRule("<=") {
    atomic(str("<=") ~ ws)
  }

  def greaterOrEqualThan: Rule0 = namedRule(">=") {
    atomic(str(">=") ~ ws)
  }

  def nullKeyword: Rule0 = namedRule("null") {
    atomic(str("null") ~ !ValidNameChar)
  }

  def trueKeyword: Rule0 = namedRule("true") {
    atomic(str("true") ~ !ValidNameChar)
  }

  def falseKeyword: Rule0 = namedRule("false") {
    atomic(str("false") ~ !ValidNameChar)
  }

  def ifKeyword: Rule0 = namedRule("if") {
    atomic(str("if") ~ !ValidNameChar)
  }

  def unlessKeyword: Rule0 = namedRule("unless") {
    atomic(str("unless") ~ !ValidNameChar)
  }

  def elseKeyword: Rule0 = namedRule("else") {
    atomic(str("else") ~ (fws | !ValidNameChar))
  }

  def asKeyword: Rule0 = namedRule("as") {
    atomic(str("as") ~ fws)
  }

  def defaultKeyword: Rule0 = namedRule("default") {
    atomic(str("default") ~ (fws | !ValidNameChar))
  }

  def andKeyword: Rule0 = namedRule("and") {
    atomic(str("and") ~ (fws | !ValidNameChar))
  }

  def orKeyword: Rule0 = namedRule("or") {
    atomic(str("or") ~ (fws | !ValidNameChar))
  }

  def notKeyword: Rule0 = namedRule("not") {
    atomic(str("not") ~ fws)
  }

  def matchKeyword: Rule0 = namedRule("match") {
    atomic(str("match") ~ (fws | !ValidNameChar))
  }

  def updateKeyword: Rule0 = namedRule("update") {
    atomic(str("update") ~ (fws | !ValidNameChar))
  }

  def matchesKeyword: Rule0 = namedRule("matches") {
    atomic(str("matches") ~ (fws | !ValidNameChar))
  }

  def usingKeyword: Rule0 = namedRule("using") {
    atomic(str("using") ~ (fws | !ValidNameChar))
  }

  def doBlockKeyword: Rule0 = namedRule("do") {
    atomic(str("do") ~ (fws | !ValidNameChar))
  }

  def baseTypeKeyword: Rule0 = namedRule("baseType") {
    atomic(ws ~ str("<:") ~ ws)
  }

  def prefixSeparatorKeyword: Rule0 = namedRule("#") {
    atomic(ws ~ ch('#') ~ ws)
  }

  def withKeyword = namedRule("with") {
    atomic(fws ~ str("with") ~ fws)
  }
}

object Tokens {
  val NS = "ns"
  val INPUT = "input"
  val OUTPUT = "output"
  val IMPORT = "import"
  val ANNOTATION = "annotation"
  val DOCUMENT_SEPARATOR = "---"
  val FUNCTION = "fun"
  val TYPE = "type"
  val VAR = "var"
  val VERSION = "%dw"
}
