package org.mule.weave.v2.grammar

import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.ast.CommentNode
import org.mule.weave.v2.parser.ast.CommentType
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.ParserPosition
import org.mule.weave.v2.parser.location.WeaveLocation
import org.parboiled2._
import org.parboiled2.support.hlist.::
import org.parboiled2.support.hlist.HNil

import scala.collection.mutable

trait WhiteSpaceHandling extends StringBuilding {
  this: org.mule.weave.v2.grammar.WhiteSpaceHandling with Parser =>

  val _comments: mutable.HashMap[Int, CommentNode] = new mutable.HashMap[Int, CommentNode]()

  def attachDocumentation: Boolean

  def input: SafeStringBasedParserInput

  def comment: Rule0 = rule {
    lineComment | docComment | blockComment
  }

  def lineComment: Rule0 = rule {
    push(ParserPosition(cursor, stringInput)) ~ quiet(ch('/') ~ ch('/') ~!~ clearSB() ~ zeroOrMore(!eol ~ ANY ~ (test(!attachDocumentation) | appendSB()))) ~ insertComment(CommentType.LineComment, sb.toString.trim)
  }

  def docComment: Rule0 = rule {
    push(ParserPosition(cursor, stringInput)) ~ quiet(str("/**") ~!~ clearSB() ~ zeroOrMore(!str("*/") ~ ANY ~ (test(!attachDocumentation) | appendSB())) ~ str("*/")) ~ insertComment(CommentType.DocComment, extractWeaveDocText())
  }

  private def stringInput = {
    input.asInstanceOf[SafeStringBasedParserInput]
  }

  private def extractWeaveDocText() = {
    if (attachDocumentation) {
      val result = new mutable.StringBuilder()
      var newLine = true
      var i = 0
      //Remove first whitespaces
      while (i < sb.length() && Character.isWhitespace(sb.charAt(i))) {
        i = i + 1
      }
      while (i < sb.length()) {
        val c = sb.charAt(i)
        c match {
          case '\n' =>
            result.append(c)
            newLine = true
          case '\r' => {
            result.append(c)
            newLine = true
            if (i + 1 < sb.length() && sb.charAt(i + 1) == '\n') {
              result.append(sb.charAt(i + 1))
              i = i + 1
            }
          }
          case _ if Character.isWhitespace(c) && newLine =>
          //Ignore whitespaces between new line
          case '*' if newLine =>
            //We only remove the first whitespaces after the *
            if (i + 1 < sb.length() && (sb.charAt(i + 1) == ' ')) {
              i = i + 1
            }
            newLine = false
          case _ => {
            newLine = false
            result.append(c)
          }
        }
        i = i + 1
      }
      result.mkString.trim
    } else {
      ""
    }
  }

  def blockComment: Rule0 = rule {
    push(ParserPosition(cursor, stringInput)) ~ quiet(str("/*") ~!~ clearSB() ~ zeroOrMore(!str("*/") ~ ANY ~ (test(!attachDocumentation) | appendSB())) ~ str("*/")) ~ insertComment(CommentType.BlockComment, extractWeaveDocText())
  }

  def insertComment(commentType: CommentType.Value, commentText: String): Rule[ParserPosition :: HNil, HNil] = rule {
    run {
      (startPos: ParserPosition) =>
        {
          if (attachDocumentation && !_comments.contains(startPos.index)) {
            val endPos = ParserPosition(cursor, stringInput)
            val location = WeaveLocation(startPos, endPos, resourceName)
            val commentNode = createCommentNode(commentType, commentText, location)
            _comments.put(startPos.index, commentNode)
          }
          HNil
        }
    }
  }

  def resourceName: NameIdentifier

  def createCommentNode(commentType: CommentType.Value, commentText: String, location: WeaveLocation): CommentNode = {
    val commentNode = CommentNode(commentText, commentType)
    commentNode._location = Some(location)
    commentNode
  }

  def comments: Rule0 = rule {
    quiet(zeroOrMore(comment).separatedBy(eol))
  }

  def fcomments: Rule0 = rule {
    quiet(oneOrMore(comment).separatedBy(eol))
  }

  def ws: Rule0 = rule {
    quiet(zeroOrMore(comment | whiteSpaceOrNewLineChar))
  }

  def fws: Rule0 = rule {
    quiet(oneOrMore(whiteSpaceOrNewLineChar ~ optional(fcomments)))
  }

  def wsnoeol: Rule0 = rule {
    quiet(zeroOrMore(comment | whiteSpaceChar) ~ optional(eol ~ fcomments))
  }

  def eol: Rule0 = rule {
    quiet(oneOrMore(newLineChar))
  }

  private val whiteSpaceChar = CharPredicate(" \f\t")
  private val newLineChar = CharPredicate("\r\n")
  val whiteSpaceOrNewLineChar = whiteSpaceChar ++ newLineChar
}
