package org.mule.weave.v2.grammar.literals

import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.grammar.Tokens
import org.mule.weave.v2.grammar.Variables
import org.mule.weave.v2.parser.annotation.InterpolationExpressionAnnotation
import org.mule.weave.v2.parser.annotation.QuotedStringAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.structure.StringInterpolationNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.parboiled2.CharPredicate._
import org.parboiled2._
import org.parboiled2.support.hlist.HNil

trait StringLiteral extends PositionTracking with Tokens with StringBuilding with Variables with BaseExpression {
  this: StringLiteral with Grammar =>

  val createStringNode = (text: String) => {
    StringNode(text)
  }

  val createStringInterpolationNode = (elements: Seq[AstNode]) => {
    elements match {
      case Seq()              => StringNode("")
      case Seq(x: StringNode) => x //When it's a simple (not interpolated) string
      case _                  => StringInterpolationNode(elements)
    }
  }

  val createForcedStringInterpolationNode = (elements: Seq[AstNode]) => {
    StringInterpolationNode(elements)
  }

  val markSingleQuote = (node: AstNode) => {
    node.annotate(QuotedStringAnnotation('\''))
    node
  }

  val markDoubleQuote = (node: AstNode) => {
    node.annotate(QuotedStringAnnotation('"'))
    node
  }

  val markInterpolationExpression = (node: AstNode) => {
    node.annotate(InterpolationExpressionAnnotation())
    node
  }

  val markVariableInterpolationExpression = (node: AstNode) => {
    node.annotate(InterpolationExpressionAnnotation(false))
    node
  }

  val markBackTickQuote = (node: AstNode) => {
    node.annotate(QuotedStringAnnotation('`'))
    node
  }

  private val charsSyntax = CharPredicate("$&*#{}[]%^=()")
  private val charsReserved = CharPredicate("`@")
  private val charsNonStart = charsSyntax ++ charsReserved
  private val alphaNumUnderscore = AlphaNum ++ CharPredicate("_")

  def interpolationVariableRef: Rule1[AstNode] = namedRule("$variable") {
    atomic(dollarNamedVariable | dollarVariable) ~> markVariableInterpolationExpression
  }

  private def interpolation: Rule[HNil, org.parboiled2.support.hlist.::[AstNode, HNil]] = namedRule("$( <expression> )") {
    str("$(") ~!~ ws ~ interpolationExpr ~ ws ~!~ ch(')')
  }

  private def interpolationExpr = rule {
    atomic(expr) ~> markInterpolationExpression
  }

  def string: Rule1[AstNode] = namedRule("String") {
    pushPosition ~ (doubleQuotedString | singleQuotedString | backTickString) ~ injectPosition
  }

  def doubleQuotedString: Rule1[AstNode] = namedRule("\"") {
    ((ch('"') ~!~ zeroOrMore(atomic(doubleQuoteStringSegment ~!~ MATCH) | interpolation | (interpolationVariableRef ~ !ch('(')))) ~ ch('"') ~> createStringInterpolationNode) ~> markDoubleQuote
  }

  def backTickString: Rule1[AstNode] = namedRule("`") {
    ((ch('`') ~!~ zeroOrMore(atomic(backTickStringSegment ~!~ MATCH) | interpolation | (interpolationVariableRef ~ !ch('('))) ~!~ ch('`')) ~> createForcedStringInterpolationNode) ~> markBackTickQuote
  }

  def singleQuotedString: Rule1[AstNode] = namedRule("'") {
    ((ch('\'') ~!~ zeroOrMore(atomic(singleQuoteStringSegment ~!~ MATCH) | interpolation | (interpolationVariableRef ~ !ch('('))) ~ ch('\'')) ~> createStringInterpolationNode) ~> markSingleQuote
  }

  private def singleQuoteEscapedChar = rule {
    ch('\\') ~ (singleQuote | escapeChar | unicode)
  }

  private def singleQuote = rule {
    ch('\'') ~ appendSB('\\') ~ appendSB()
  }

  private def doubleQuote = rule {
    ch('"') ~ appendSB('\\') ~ appendSB()
  }

  private def escapeChar = rule {
    anyOf("\\$/bfnrt") ~ appendSB(-2) ~ appendSB()
  }

  private def unicode = rule {
    ch('u') ~ 4.times(CharPredicate.HexDigit) ~ appendSB("\\u") ~ appendSB(-4) ~ appendSB(-3) ~ appendSB(-2) ~ appendSB()
  }

  private def doubleQuoteEscapedChar = rule {
    ch('\\') ~ (doubleQuote | escapeChar | unicode)
  }

  private def backTickEscapedChar = rule {
    ch('\\') ~ anyOf("\\$`") ~ appendSB('\\') ~ appendSB()
  }

  private def singleQuoteStringSegment: Rule1[AstNode] = rule {
    pushPosition ~ (clearSB() ~ oneOrMore(singleQuoteEscapedChar | (noneOf("'$") ~ appendSB())) ~ push(sb.toString) ~> createStringNode ~> markSingleQuote) ~ injectPosition
  }

  private def doubleQuoteStringSegment: Rule1[AstNode] = rule {
    pushPosition ~ (clearSB() ~ oneOrMore(doubleQuoteEscapedChar | (noneOf("\"$") ~ appendSB())) ~ push(sb.toString) ~> createStringNode ~> markDoubleQuote) ~ injectPosition
  }

  private def backTickStringSegment: Rule1[AstNode] = rule {
    pushPosition ~ (clearSB() ~ oneOrMore(backTickEscapedChar | (noneOf("`$") ~ appendSB())) ~ push(sb.toString) ~> createStringNode ~> markBackTickQuote) ~ injectPosition
  }

  def nameString: Rule1[AstNode] = namedRule("NameIdentifier") {
    pushPosition ~ (nameIdentifier ~> createStringNode) ~ injectPosition
  }
}
