package org.mule.weave.v2.inspector

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.grammar.EqOpId
import org.mule.weave.v2.grammar.GreaterThanOpId
import org.mule.weave.v2.grammar.SimilarOpId
import org.mule.weave.v2.inspector.ReferenceUtils.isReferencingTo
import org.mule.weave.v2.parser.UsingSizeOfToCompareEmpty
import org.mule.weave.v2.parser.UsingSizeOfToCompareNonEmpty
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.NameIdentifier.CORE_MODULE
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware
import org.mule.weave.v2.scope.ScopesNavigator

object SizeOfEqualsZeroInspector extends CodeInspector[AstNodeResultAware[_] with ScopeNavigatorResultAware] {

  def isReferencingSizeOf(variable: NameIdentifier, scopeData: ScopesNavigator): Boolean = {
    isReferencingTo(variable, CORE_MODULE.::("sizeOf"), scopeData)
  }

  override def inspect(node: AstNode, scopeData: AstNodeResultAware[_] with ScopeNavigatorResultAware, parsingContext: ParsingContext): Unit = {
    node match {
      case BinaryOpNode(op, FunctionCallNode(vrn: VariableReferenceNode, params, _, _), NumberNode("0", _), _) if ((op == EqOpId || op == SimilarOpId || op == GreaterThanOpId) && isReferencingSizeOf(vrn.variable, scopeData.scope)) => {
        if (params.args.length == 1) {
          if (op == GreaterThanOpId) {
            val param: AstNode = params.args.head
            parsingContext.messageCollector.warning(UsingSizeOfToCompareNonEmpty(param, node), node.location())
          } else {
            val param: AstNode = params.args.head
            parsingContext.messageCollector.warning(UsingSizeOfToCompareEmpty(param, node), node.location())
          }
        }
      }
      case _ =>
    }
  }
}

class SizeOfEqualsZeroFixAction(expression: AstNode, containerNode: AstNode) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {
    val startIndex = containerNode.location().startPosition.index
    document.delete(startIndex, containerNode.location().endPosition.index)
    document.insert("isEmpty(" + CodeGenerator.generate(expression) + ")", startIndex)
  }
}

class SizeOfGraterToZeroFixAction(expression: AstNode, containerNode: AstNode) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {
    val startIndex = containerNode.location().startPosition.index
    document.delete(startIndex, containerNode.location().endPosition.index)
    document.insert("!isEmpty(" + CodeGenerator.generate(expression) + ")", startIndex)
  }
}
