package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.utils.CacheBuilder

import scala.collection.mutable

/**
  * Holds the information of the modules dependencies
  */
class DependencyGraph {

  private val dependants = CacheBuilder.apply[NameIdentifier, mutable.Set[NameIdentifier]]().build()

  def modules(): Seq[NameIdentifier] = {
    dependants.keys().toSeq
  }

  /**
    * Invalidates all information
    */
  def invalidateAll(): Unit = {
    dependants.clear()
  }

  /**
    * Returns the list of modules that depends on the specified module
    * @param nameIdentifier The name of the module
    * @return
    */
  def getDependants(nameIdentifier: NameIdentifier): Seq[NameIdentifier] = {
    dependants.get(nameIdentifier, (_) => mutable.Set()).toSeq
  }

  /**
    * Invalidates the module information
    * @param nameIdentifier the module information to be invalidated
    */
  def invalidateModule(nameIdentifier: NameIdentifier): Unit = {
    dependants
      .values()
      .foreach((deps) => {
        //Remove this module from the list of all the dependants
        deps.-(nameIdentifier)
      })
  }

  /**
    * Loads the dependencies of the given module
    * @param ni The name of the module
    * @param parseResult The result
    */
  def loadDependenciesFrom(ni: NameIdentifier, parseResult: PhaseResult[ParsingResult[_ <: AstNode]]): Unit = {
    if (parseResult.hasResult()) {
      invalidateModule(ni) //Remove all information of this
      val value: ParsingResult[_ <: AstNode] = parseResult.getResult()
      val imports: Seq[ImportDirective] = AstNodeHelper.collectChildrenWith(value.astNode, classOf[ImportDirective])
      val variableReferenceNodes: Seq[VariableReferenceNode] = AstNodeHelper.collectChildrenWith(value.astNode, classOf[VariableReferenceNode])
      val typeReferenceNode: Seq[TypeReferenceNode] = AstNodeHelper.collectChildrenWith(value.astNode, classOf[TypeReferenceNode])
      val annotationRef: Seq[AnnotationNode] = AstNodeHelper.collectChildrenWith(value.astNode, classOf[AnnotationNode])

      val deps =
        imports.map(_.importedModule.elementName) ++
          variableReferenceNodes.flatMap((vrn) => {
            vrn.variable.parent()
          }) ++
          typeReferenceNode.flatMap((trn) => {
            trn.variable.parent()
          }) ++
          annotationRef.flatMap((an) => {
            an.name.parent()
          })

      deps.foreach((dep) => {
        moduleDependsOn(ni, dep)
      })
    }
  }

  private def moduleDependsOn(source: NameIdentifier, dependsOn: NameIdentifier): Unit = {
    val currentDeps = dependants.get(dependsOn, (_) => mutable.Set())
    currentDeps.+=(source)
  }
}
