package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation

/**
  * Helper class with functions to clone a type and copy its internal properties
  */
object WeaveTypeCloneHelper {

  /**
    * Clones the given type
    * @param from The type to be cloned
    * @param location The location to be set to this type
    * @tparam T The new type instance
    * @return The new type
    */
  def clone[T <: WeaveType](from: T, location: WeaveLocation): T = {
    val weaveType = from.cloneType()
    weaveType.label(from.label())
    copyAdditionalTypeInformation(from, weaveType, location)
    weaveType.asInstanceOf[T]
  }

  /**
    * Copies all the additional information from one to the other.
    *  - ParentKey
    *  - Location
    *  - Constraints
    *  - Metadata
    *  - Optional
    * @param from the source
    * @param to The target
    * @return The target type with all the metadata copied
    */
  def copyAdditionalTypeInformation[T <: WeaveType](from: WeaveType, to: T): Unit = {
    copyAdditionalTypeInformation(from, to, from.location())
  }

  private def copyAdditionalTypeInformation[T <: WeaveType](from: WeaveType, to: T, location: WeaveLocation): Unit = {
    doCopyAdditionalTypeInformation(from, to, location)
  }

  private def doCopyAdditionalTypeInformation[T <: WeaveType](from: WeaveType, to: T, location: WeaveLocation): Unit = {
    if (to.parentKey.isEmpty) {
      to.parentKey = from.parentKey
    }
    to.withOptional(from.isOptional())
    copyLocation(to, location)
    copyMetadataTo(from, to)
  }

  def copyLocation[T <: WeaveType](to: T, location: WeaveLocation): Unit = {
    if (!(location eq UnknownLocation)) {
      to.withLocation(location)
    }
  }

  def copyMetadataTo[T <: WeaveType](from: WeaveType, to: T): Unit = {
    from.copyMetadataConstraintsTo(to)
    from.copyMetadataTo(to)
  }
}
