package org.mule.weave.v2.ts.updaters

import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NameValuePairType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeCloneHelper

import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

/**
  * Will handle the update of the structure and returns the new structure updated
  */
sealed trait TypeUpdater {

  var forceCreate = false

  /**
    * The list of updater children
    *
    * @return The children
    */
  def children(): ArrayBuffer[TypeUpdater]

  /**
    * Adds a new child type updater
    *
    * @param typeUpdater The child type updater
    */
  def addChild(typeUpdater: TypeUpdater): Unit = {
    children().+=(typeUpdater)
  }

}

object UpdaterDispatcher {

  def createAttributes(attributeUpdaters: ArrayBuffer[TypeUpdater], context: (String, WeaveType) => Option[WeaveType]): Seq[NameValuePairType] = {
    attributeUpdaters.collect({
      case au: AttributeFieldUpdater if (au.forceCreate) => au
    }).map(_.createNameValuePair(context))
  }

  private def doCreate(valueUpdaters: ArrayBuffer[TypeUpdater], context: (String, WeaveType) => Option[WeaveType]): WeaveType = {

    val objectFields = valueUpdaters.collect({
      case fieldUpdater: ObjectFieldUpdater => fieldUpdater
    }).map((ofu) => {
      ofu.createKVP(context)
    })

    val maybeObjectType =
      if (objectFields.isEmpty)
        None
      else
        Some(ObjectType(objectFields))

    val arrayTypes = valueUpdaters.collect({
      case fieldUpdater: IndexFieldUpdater => fieldUpdater
    }).map((fi) => {
      create(fi.children, context)
    })

    val maybeArrayType =
      if (arrayTypes.isEmpty)
        None
      else
        Some(ArrayType(TypeHelper.unify(arrayTypes)))

    maybeObjectType match {
      case Some(ot) => {
        maybeArrayType match {
          case Some(at) => TypeHelper.unify(Seq(ot, at))
          case None     => ot
        }
      }
      case None => maybeArrayType.getOrElse(NullType())
    }

  }

  def create(valueUpdaters: ArrayBuffer[TypeUpdater], context: (String, WeaveType) => Option[WeaveType]): WeaveType = {
    val forceCreateUpdaters = valueUpdaters.filter(_.forceCreate)

    val terminalUpdaters = forceCreateUpdaters.collect({
      case terminalNodeUpdater: TerminalNodeUpdater => terminalNodeUpdater
    })
    if (terminalUpdaters.nonEmpty) {
      val newValueType = TypeHelper.unify(terminalUpdaters.map(_.update(NullType(), context)))
      val totalMatch = terminalUpdaters.exists(!_.conditional)
      if (totalMatch || terminalUpdaters.size == forceCreateUpdaters.size) {
        newValueType
      } else {
        //This is a partial match it may match some
        TypeHelper.unify(Seq(doCreate(forceCreateUpdaters, context), newValueType))
      }
    } else {
      doCreate(forceCreateUpdaters, context)
    }
  }

  /**
    * Dispatches the update to the correct updater
    *
    * @param toUpdate The type to be updated
    * @return The updated type
    */
  def dispatchUpdate(toUpdate: WeaveType, children: mutable.ArrayBuffer[TypeUpdater], context: (String, WeaveType) => Option[WeaveType]): WeaveType = {
    toUpdate match {
      case rt: ReferenceType => {
        dispatchUpdate(rt.resolveType(), children, context)
      }
      case ut: UnionType => {
        val weaveTypes = ut.of.map((wt) => dispatchUpdate(wt, children, context))
        val unified = TypeHelper.unify(weaveTypes)
        WeaveTypeCloneHelper.copyAdditionalTypeInformation(ut, unified)
        unified
      }
      case it: IntersectionType => {
        val resolvedType = TypeHelper.resolveIntersection(it.of)
        resolvedType match {
          case it: IntersectionType => {
            val types = it.of.map((wt) => dispatchUpdate(wt, children, context))
            val copied = it.copy(of = types)
            WeaveTypeCloneHelper.copyAdditionalTypeInformation(it, copied)
            copied
          }
          case newType => dispatchUpdate(newType, children, context)
        }
      }
      case tp: TypeParameter if (tp.top.isDefined) => {
        dispatchUpdate(tp.top.get, children, context)
      }
      case _ => {
        val terminalNodeUpdaters = children.collect({ case tn: TerminalNodeUpdater => tn })
        if (terminalNodeUpdaters.nonEmpty) {
          val weaveType = TypeHelper.unify(terminalNodeUpdaters.map(_.update(toUpdate, context)))
          val totalMatch = terminalNodeUpdaters.exists(!_.conditional)
          if (totalMatch) {
            weaveType
          } else {
            //This is a partial match it may match some
            TypeHelper.unify(Seq(doUpdate(toUpdate, children, context), weaveType))
          }
        } else {
          doUpdate(toUpdate, children, context)
        }
      }
    }

  }

  private def doUpdate(toUpdate: WeaveType, children: mutable.ArrayBuffer[TypeUpdater], context: (String, WeaveType) => Option[WeaveType]): WeaveType = {

    toUpdate match {
      case ot: ObjectType => {
        val candidates = children.collect({
          case objectFieldUpdater: ObjectFieldUpdater => objectFieldUpdater
        })

        val missingCandidates = ArrayBuffer(candidates: _*)

        val newProps = ot.properties.map((kvpt) => {
          candidates
            .zipWithIndex // Find the first updater that matches
            .find((updater) => updater._1.accepts(kvpt)) //
            .map((updater) => {
              missingCandidates.update(updater._2, null)
              updater._1.update(kvpt, context)
            })
            .getOrElse(kvpt)
        })

        val missingKeyValuePairs = missingCandidates.filter((u) => u match {
          case ofu: ObjectFieldUpdater if (ofu.forceCreate) => true
          case _ => false
        })
          .map((u) => u.createKVP(context))
        val newObjectType = ot.copy(properties = newProps ++ missingKeyValuePairs)
        WeaveTypeCloneHelper.copyAdditionalTypeInformation(ot, newObjectType)
        newObjectType
      }

      case kt: KeyType => {

        val candidates = children.collect({
          case fieldUpdater: AttributeFieldUpdater => fieldUpdater
        })

        val missingCandidates = ArrayBuffer(candidates: _*)

        val newProps = kt.attrs.map((nvpt) => {
          candidates
            .zipWithIndex // Find the first updater that matches
            .find((child) => child._1.accepts(nvpt)) //
            .map((updater) => {
              missingCandidates.update(updater._2, null)
              updater._1.update(nvpt, context) match {
                case nvpt: NameValuePairType => nvpt
                case v                       => nvpt.copy(value = v)
              }
            })
            .getOrElse(nvpt)
        })

        val missingKeyValuePairs = missingCandidates.filter((u) => u match {
          case ofu: AttributeFieldUpdater if (ofu.forceCreate) => true
          case _ => false
        })
          .map((u) => u.createNameValuePair(context))
        kt.copy(attrs = newProps ++ missingKeyValuePairs)
      }

      case at: ArrayType => {
        val indexUpdaters = children.collect({
          case iu: IndexFieldUpdater => iu
        })
        val arrayComponent = at.of

        //Index updater is the union of all the branches and the current type
        //[1,2,3] update { [0] -> true, [1] -> "test"} needs to return Array<Boolean | String | Number>
        val updatedComponent = TypeHelper.unify(
          indexUpdaters.map(iu => {
            UpdaterDispatcher.dispatchUpdate(arrayComponent, iu.children, context)
          }) :+ (arrayComponent))
        at.copy(updatedComponent)
      }

      case updated => updated
    }
  }
}

class RootFieldUpdater(val children: mutable.ArrayBuffer[TypeUpdater]) extends TypeUpdater {
  def update(toUpdate: WeaveType, context: (String, WeaveType) => Option[WeaveType]): WeaveType = {
    UpdaterDispatcher.dispatchUpdate(toUpdate, children, context)
  }
}

object RootFieldUpdater {
  def apply(): RootFieldUpdater = new RootFieldUpdater(ArrayBuffer())
}

class ObjectFieldUpdater(val maybeQName: Option[QName], val valueUpdaters: mutable.ArrayBuffer[TypeUpdater], val attributeUpdaters: mutable.ArrayBuffer[TypeUpdater], multiSelection: Boolean) extends TypeUpdater {

  def createKVP(context: (String, WeaveType) => Option[WeaveType]): KeyValuePairType = {
    KeyValuePairType(KeyType(NameType(maybeQName), UpdaterDispatcher.createAttributes(attributeUpdaters, context)), UpdaterDispatcher.create(valueUpdaters, context))
  }

  def accepts(toUpdate: WeaveType): Boolean = {
    toUpdate match {
      case KeyValuePairType(KeyType(NameType(maybeName), _), _, _, _) => {
        maybeQName match {
          case Some(keyName) if (maybeName.isDefined) => {
            maybeName.get.selectedBy(keyName)
          }
          case _ => {
            true
          }
        }
      }
      case _ => false
    }
  }

  def update(kvpt: KeyValuePairType, context: (String, WeaveType) => Option[WeaveType]): KeyValuePairType = {
    val newValueType = UpdaterDispatcher.dispatchUpdate(kvpt.value, valueUpdaters, context)
    val newKeyType = UpdaterDispatcher.dispatchUpdate(kvpt.key, attributeUpdaters, context)
    var optional = kvpt.optional
    if (this.forceCreate) {
      optional = !this.forceCreate
    }
    if (kvpt.repeated && !multiSelection) {
      kvpt.copy(key = newKeyType, value = TypeHelper.unify(Seq(newValueType, kvpt.value)), optional)
    } else {
      kvpt.copy(key = newKeyType, value = newValueType, optional)
    }
  }

  override def children(): ArrayBuffer[TypeUpdater] = valueUpdaters

  override def addChild(typeUpdater: TypeUpdater): Unit = {
    typeUpdater match {
      case afu: AttributeFieldUpdater => attributeUpdaters.+=(afu)
      case _                          => valueUpdaters.+=(typeUpdater)
    }
  }
}

object ObjectFieldUpdater {
  def apply(keyName: QName, multiSelection: Boolean): ObjectFieldUpdater = new ObjectFieldUpdater(Some(keyName), ArrayBuffer(), ArrayBuffer(), multiSelection)

  def apply(keyName: Option[QName], multiSelection: Boolean): ObjectFieldUpdater = new ObjectFieldUpdater(keyName, ArrayBuffer(), ArrayBuffer(), multiSelection)

  def apply(keyName: Option[QName], parent: TypeUpdater, multiSelection: Boolean): ObjectFieldUpdater = {
    val updater = ObjectFieldUpdater(keyName, multiSelection)
    parent.addChild(updater)
    updater
  }
}

class IndexFieldUpdater(val children: mutable.ArrayBuffer[TypeUpdater]) extends TypeUpdater {

}

object IndexFieldUpdater {
  def apply(): IndexFieldUpdater = new IndexFieldUpdater(ArrayBuffer())

  def apply(parent: TypeUpdater): IndexFieldUpdater = {
    val result = IndexFieldUpdater()
    parent.addChild(result)
    result
  }
}

class AttributeFieldUpdater(val attributeName: QName, val children: mutable.ArrayBuffer[TypeUpdater]) extends TypeUpdater {

  def createNameValuePair(context: (String, WeaveType) => Option[WeaveType]): NameValuePairType = {
    NameValuePairType(NameType(Some(attributeName)), UpdaterDispatcher.create(children, context))
  }

  def update(toUpdate: NameValuePairType, context: (String, WeaveType) => Option[WeaveType]): WeaveType = {
    toUpdate match {
      case nvpt: NameValuePairType => {
        val newValue = UpdaterDispatcher.dispatchUpdate(nvpt.value, children, context)
        nvpt.copy(value = newValue)
      }
      case _ => toUpdate

    }
  }

  def accepts(toUpdate: WeaveType): Boolean = {
    toUpdate match {
      case NameValuePairType(NameType(Some(name)), _, _) => name.selectedBy(attributeName)
      case _ => false
    }
  }

}

object AttributeFieldUpdater {
  def apply(attributeName: QName): AttributeFieldUpdater = new AttributeFieldUpdater(attributeName, ArrayBuffer())

  def apply(attributeName: QName, parent: TypeUpdater): AttributeFieldUpdater = {
    val result = AttributeFieldUpdater(attributeName)
    parent.addChild(result)
    result
  }
}

class TerminalNodeUpdater(branchIndex: Int, val conditional: Boolean, forceCreateTerminal: Boolean, val children: mutable.ArrayBuffer[TypeUpdater] = ArrayBuffer()) extends TypeUpdater {

  this.forceCreate = forceCreateTerminal

  def update(toUpdate: WeaveType, context: (String, WeaveType) => Option[WeaveType]): WeaveType = {
    val maybeType = context(String.valueOf(branchIndex), toUpdate)
    maybeType.getOrElse(toUpdate)
  }
}

object TerminalNodeUpdater {
  def apply(branchIndex: Int, conditional: Boolean, forceCreate: Boolean): TerminalNodeUpdater = new TerminalNodeUpdater(branchIndex, conditional, forceCreate)

  def apply(branchIndex: Int, conditional: Boolean, forceCreate: Boolean, parent: TypeUpdater): TerminalNodeUpdater = {
    val result = TerminalNodeUpdater(branchIndex, conditional, forceCreate)
    parent.addChild(result)
    result
  }
}