package org.mule.weave.v2.utils

import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.operators.OpNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode

object AstGraphDotEmitter {

  def print(node: AstNode, name: String = "AST"): String = {
    val writer: StringCodeWriter = new StringCodeWriter()
    writer.println(s"digraph ${name.replaceAll("\"", "\'")} {")
    writer.indent()
    writer.println("  node [fixedsize=shape fontsize=10]")

    writer.println(s"${id(node)} [label=" + "\"" + label(node).replaceAll("\"", "\'") + "\"" + "];")
    printNodes(writer, node)
    printEdges(writer, node)

    writer.printIndent()
    writer.dedent()
    writer.println()
    writer.println("}")
    writer.codeContent()
  }

  def printEdges(writer: StringCodeWriter, container: AstNode): Unit = {
    container.children().foreach((node) => {
      writer.println(s"${id(container)} -> ${id(node)}")
      printEdges(writer, node)
    })
  }

  def printNodes(writer: StringCodeWriter, node: AstNode): Unit = {
    node.children().foreach((node) => {
      writer.println(s"${id(node)} [label=" + "\"" + label(node).replaceAll("\"", "\'") + "\"" + "];")
      printNodes(writer, node)
    })
  }

  def label(astNode: AstNode): String = {
    astNode match {
      case varName: OpNode                      => s"${astNode.getClass.getSimpleName}(${varName.opId.name})"
      case varName: NameIdentifier              => s"${astNode.getClass.getSimpleName}(${varName.name})"
      case literal: LiteralValueAstNode         => s"${astNode.getClass.getSimpleName}(${literal.literalValue})"
      case referenceNode: VariableReferenceNode => s"${astNode.getClass.getSimpleName}(${referenceNode.variable.name})"
      case _                                    => astNode.getClass.getSimpleName
    }
  }

  def id(node: AstNode): String = {
    Math.abs(System.identityHashCode(node)).toString
  }

}
