package org.mule.weave.v2.utils

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.codegen.CodeWriter
import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.ts._

import scala.collection.mutable

class WeaveTypeEmitter(config: WeaveTypeEmitterConfig = WeaveTypeEmitterConfig()) {

  val emittedTypes: mutable.Map[String, WeaveType] = mutable.Map()
  var typesToEmit: mutable.Map[String, WeaveType] = mutable.Map()
  var namespaces: mutable.Map[String, String] = mutable.Map()

  def mergeEmitterMetadata(other: WeaveTypeEmitter): Unit = {
    emittedTypes ++= other.emittedTypes
    typesToEmit ++= other.typesToEmit
    namespaces ++= other.namespaces
  }

  def toString(wtype: WeaveType, insideKey: Boolean = false): String = {
    val builder = new StringCodeWriter()
    val text = generate(wtype, builder, RecursionDetector((id, _) => {
      builder.print(id.name).toString
    }), insideKey)

    val namespaceString = new StringCodeWriter()
    if (namespaces.nonEmpty) {
      namespaceString.println()
      namespaces.foreach((namespace) => {
        namespaceString.println(s"ns ${namespace._2} ${namespace._1}")
      })
    }

    if (namespaceString.toString.nonEmpty) {
      s"$text ${namespaceString.toString}"
    } else {
      text
    }
  }

  def generateTypeCatalog(types: Map[String, WeaveType], typeDeclarations: StringCodeWriter = new StringCodeWriter()): String = {

    types.foreach((entry) => {
      if (!emittedTypes.contains(entry._1)) {
        generateType(entry._1, entry._2, typeDeclarations)
        typeDeclarations.println()
        typeDeclarations.println()
      }
    })

    val header = new StringCodeWriter()
    header.println("%dw 2.0")
    if (namespaces.nonEmpty) {
      header.println("// namespaces")
      namespaces.foreach((namespace) => {
        header.println(s"ns ${namespace._2} ${namespace._1}")
      })
    }
    header.toString + typeDeclarations.toString
  }

  def generateType(rootName: String, wtype: WeaveType, builder: StringCodeWriter = new StringCodeWriter()): String = {
    val typeParameters = TypeHelper.collectTypeParameters(wtype)
    val recursionDetector = RecursionDetector[String]((id, _) => {
      id.name
    })

    builder.print(s"\ntype ${rootName} ")
    if (typeParameters.nonEmpty) {
      val boundedTypeEmitter = new WeaveTypeEmitter(config.copy(printTypeBounds = true))
      builder.print("<")
      builder.printForeachWithSeparator(", ", typeParameters.distinct, (tp: WeaveType) => {
        boundedTypeEmitter.generate(tp, builder, recursionDetector)
      })
      mergeEmitterMetadata(boundedTypeEmitter)
      builder.print("> ")
    }
    builder.print("= ")
    emittedTypes.+=((rootName, wtype))
    generate(wtype, builder, RecursionDetector[String]((id, _) => {
      id.name
    }))
    val types = typesToEmit.toMap
    typesToEmit = mutable.Map[String, WeaveType]()
    generateTypeCatalog(types, builder)
  }

  def getPrefixFor(namespace: String): String = {
    if (namespaces.contains(namespace)) {
      namespaces(namespace)
    } else {
      val prefix = "ns" + namespaces.size
      namespaces.put(namespace, prefix)
      prefix
    }
  }

  private def generate(wtype: WeaveType, builder: StringCodeWriter, recursionDetector: RecursionDetector[String], insideKey: Boolean = false, depth: Int = 0): String = {
    if (config.maxDepth.isDefined && depth > config.maxDepth.get) {
      builder.print("...")
    } else if (config.nameOnly && wtype.label().isDefined) {
      builder.print(wtype.label().get)
    } else {
      val weaveTypeMetadataConstraints = if (config.skipMetadataConstraints) Seq.empty else wtype.metadataConstraints()
      val weaveTypeMetadata = filterWeaveTypeMetadata(wtype)
      wtype match {
        case ObjectType(properties, close, ordered) => {
          if (properties.isEmpty && (config.emptyObjectWithText || !close && !ordered)) {
            builder.print("Object")
          } else {
            builder.print("{")
            if (ordered)
              builder.print("-")
            if (close)
              builder.print("|")
            builder.indent()
            //In case is one property key value pair we don't print the new lines so simple types keep simple to read
            if (inlineObject(properties)) {
              builder.printSpace()
              properties.zipWithIndex.foreach((indexProp) => {
                generate(indexProp._1, builder, recursionDetector, insideKey, depth + 1)
              })
              builder.printSpace()
            } else {
              properties.zipWithIndex.foreach((indexProp) => {
                if (indexProp._2 > 0) {
                  builder.printSpace(",")
                }
                if (config.prettyPrint) {
                  builder.println()
                  builder.printIndent()
                }
                generate(indexProp._1, builder, recursionDetector, insideKey, depth + 1)
              })
            }
            builder.dedent()
            if (config.prettyPrint && !inlineObject(properties)) {
              builder.println()
              builder.printIndent()
            }
            if (close)
              builder.print("|")
            if (ordered)
              builder.print("-")
            builder.print("}")
          }
        }
        case KeyValuePairType(key, value, optional, repeated) => {
          generate(key, builder, recursionDetector, insideKey = true, depth)
          if (repeated) {
            builder.print("*")
          }
          if (optional) {
            builder.print("?")
          }
          builder.printSpace(":")
          generate(value, builder, recursionDetector, insideKey, depth + 1)
        }
        case KeyType(name, attrs) => {
          if (insideKey) {
            generate(name, builder, recursionDetector, insideKey, depth + 1)
            if (attrs.nonEmpty) {
              builder.print(" @(")
              attrs.zipWithIndex.foreach((attr) => {
                if (attr._2 > 0) {
                  builder.printSpace(",")
                }
                generate(attr._1, builder, recursionDetector, insideKey, depth + 1)
              })
              builder.print(")")
            }
          } else {
            builder.print("Key")
          }
        }
        case NameValuePairType(name, value, optional) => {
          generate(name, builder, recursionDetector, insideKey = true, depth)
          if (optional) {
            builder.print("?")
          }
          builder.print(": ")
          generate(value, builder, recursionDetector, insideKey, depth + 1)
        }
        case NameType(name) => {
          name match {
            case Some(qName) => {
              if (qName.ns.isDefined) {
                builder.print(getPrefixFor(qName.ns.get))
                builder.print("#")
              }
              if (StringEscapeHelper.keyRequiresQuotes(qName.name)) {
                builder.printQuoted(qName.name)
              } else {
                builder.print(qName.name)
              }
            }
            case None => if (insideKey) builder.print("_") else builder.print("Name")
          }
        }
        case ArrayType(of) => {
          builder.print("Array")
          printTypeParameter(of, builder, recursionDetector, insideKey, depth)
        }
        case UnionType(of) =>
          if (weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print("(")
          }

          if (config.simplifyTypes) {
            val generatedTypes = of.map(item => generate(item, new StringCodeWriter(), recursionDetector, insideKey, depth + 1))
            val distinct = generatedTypes.distinct
            distinct.zipWithIndex.foreach {
              case (item, index) =>
                if (index > 0) {
                  builder.print(" | ")
                }
                builder.print(item)
            }
          } else {
            of.zipWithIndex.foreach {
              case (item, index) =>
                if (index > 0) {
                  builder.print(" | ")
                }
                generate(item, builder, recursionDetector, insideKey, depth + 1)
            }
          }
          if (weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print(")")
          }
        case IntersectionType(of) =>
          if (weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print("(")
          }
          of.zipWithIndex.foreach((item) => {
            if (item._2 > 0) {
              builder.print(" & ")
            }
            generate(item._1, builder, recursionDetector, insideKey, depth + 1)
          })
          if (weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print(")")
          }
        case StringType(Some(value)) if (config.useLiteralType) => {
          builder.print(StringEscapeHelper.escapeString(value))
        }
        case StringType(_) => {
          builder.print("String")
        }
        case AnyType() => builder.print("Any")
        case BooleanType(Some(value), constraints) if (config.useLiteralType) => {
          builder.print(value.toString)
          printConstrains(builder, constraints)
        }
        case BooleanType(_, constraints) => {
          builder.print("Boolean")
          printConstrains(builder, constraints)
        }
        case NumberType(Some(value)) if (config.useLiteralType) => builder.print(value)
        case NumberType(_)                                      => builder.print("Number")
        case RangeType()                                        => builder.print("Range")
        case RegexType()                                        => builder.print("Regex")
        case UriType(_)                                         => builder.print("Uri")
        case NullType()                                         => builder.print("Null")
        case DateTimeType()                                     => builder.print("DateTime")
        case LocalDateTimeType()                                => builder.print("LocalDateTime")
        case LocalDateType()                                    => builder.print("Date")
        case LocalTimeType()                                    => builder.print("LocalTime")
        case TimeType()                                         => builder.print("Time")
        case TimeZoneType()                                     => builder.print("TimeZone")
        case PeriodType()                                       => builder.print("Period")
        case BinaryType()                                       => builder.print("Binary")
        case TypeParameter(name, tt, bt, id, _) => {
          builder.print(name)
          if (config.printTypeParamInstanceId && id.isDefined) {
            builder.print("@" + id.get.toString)
          }

          if (config.printTypeBounds) {
            if (bt.isDefined) {
              builder.print(" :> ")
              generate(bt.get, builder, recursionDetector, insideKey, depth + 1)
            } else if (tt.isDefined) {
              builder.print(" <: ")
              generate(tt.get, builder, recursionDetector, insideKey, depth + 1)
            }
          }
        }
        case FunctionType(typeParams, args, returnType, _, _, _) => {

          if (config.printFunctionConstrains && typeParams.nonEmpty) {
            val constraintEmitter = new WeaveTypeEmitter(config.copy(printTypeBounds = true))
            builder.print("<")
            builder.printForeachWithSeparator[TypeParameter](", ", typeParams, tp => {
              constraintEmitter.generate(tp, builder, recursionDetector, insideKey, depth + 1)
            })
            mergeEmitterMetadata(constraintEmitter)
            builder.print(">")
          }

          builder.print("(")
          args.zipWithIndex.foreach((item) => {
            if (item._2 > 0) {
              builder.print(", ")
            }

            builder.print(item._1.name)
            if (item._1.optional) {
              builder.print("?")
            }
            if (!FunctionTypeHelper.isDynamicTypeParameter(item._1.wtype)) {
              builder.printSpace(":")
              generate(item._1.wtype, builder, recursionDetector, insideKey, depth + 1)
            }
          })
          builder.printSpace(")")
          builder.printSpace("->")
          generate(returnType, builder, recursionDetector, insideKey, depth + 1)
        }
        case TypeType(t) =>
          builder.print("Type")
          printTypeParameter(t, builder, recursionDetector, insideKey, depth)
        case NothingType()        => builder.print("Nothing")
        case NamespaceType(_, _)  => builder.print("Namespace")
        case _: DynamicReturnType => builder.print(s"?")
        case tsrt: TypeSelectorType => {
          generate(tsrt.referencedType, builder, recursionDetector, insideKey, depth + 1)
          builder.print(".")
          if (StringEscapeHelper.keyRequiresQuotes(tsrt.refName)) {
            builder.printQuoted(tsrt.refName)
          } else {
            builder.print(tsrt.refName)
          }
        }
        case rt: SimpleReferenceType => {
          if (config.followReference) {
            val text = recursionDetector.resolve(
              rt,
              (referencedType) => {
                val builder = new StringCodeWriter()
                val refName = rt.refName
                if (config.skipRecursive) {
                  val weaveType = referencedType.baseType()
                  weaveType match {
                    case _: ObjectType => builder.print("{}")
                    case _: ArrayType  => builder.print("Array<Any>")
                    case _             => builder.print("Any")
                  }
                } else if (config.prettyPrint) {
                  builder.print(refName.name)
                  rt.typeParams match {
                    case Some(tps) => {
                      builder.print("<")
                      builder.startIgnoringNewLines()
                      builder.printForeachWithSeparator(", ", tps, (tp: WeaveType) => {
                        generate(tp, builder, recursionDetector, insideKey, depth + 1)
                      })
                      builder.endIgnoringNewLines()
                      builder.print(">")
                    }
                    case None =>
                  }
                  if (!emittedTypes.contains(refName.name)) {
                    typesToEmit.+=((refName.name, rt.resolveType()))
                  }
                } else {
                  generate(referencedType, builder, recursionDetector, insideKey, depth + 1)
                }
                builder.toString
              })
            builder.print(text)
          } else {

            builder.print(rt.refName.name)
            rt.typeParams match {
              case None => {}
              case Some(value) => {
                builder.print("<")
                builder.printForeachWithSeparator(",", value, (wt: WeaveType) => generate(wt, builder, recursionDetector, insideKey, depth + 1))
                builder.print(">")
              }
            }
          }
        }
        case _ => builder.print(wtype.getClass.getSimpleName)
      }
      if (weaveTypeMetadataConstraints.nonEmpty) {
        builder.print(" {")
        builder.printForeachWithSeparator(
          ",\n",
          weaveTypeMetadataConstraints,
          (m: Any) => {
            m match {
              case constraint: MetadataConstraint =>
                builder.printQuoted(constraint.name).printSpace(":").print(StringEscapeHelper.escapeString(constraint.value.toString))
              case _ => // Nothing to do
            }

          })
        builder.print("}")
      }

      if (weaveTypeMetadata.nonEmpty) {
        val cfg = WeaveTypeMetadataEmitterConfig(prettyPrint = config.prettyPrint)
        val emitter = WeaveTypeMetadataEmitter(cfg)
        builder.print(" <~ { ")
        builder.printForeachWithSeparator(
          ",\n",
          weaveTypeMetadata,
          (m: Any) => {
            m match {
              case metadata: Metadata =>
                builder.print(emitter.toString(metadata))
              case _ => // Nothing to do
            }

          })
        builder.print("}")
      }
    }
    builder.codeContent()
  }

  private def printConstrains(builder: StringCodeWriter, constraints: VariableConstraints) = {
    if (config.printConstrains) {
      builder.println(" match { ")
      builder.println("   case true => ")
      constraints.positiveConstrains().foreach((con) => {
        builder.print(con._1.referencedNode.name).print(" ")
        con._2.foreach((vc) => {
          builder.print(constrainToString(vc))
        })
      })

      builder.println("   case false => ")
      constraints.negativeConstrains().foreach((con) => {
        builder.print(con._1.referencedNode.name).print(" ")
        con._2.foreach((vc) => {
          builder.print(constrainToString(vc))
        })
      })
      builder.print("}")
    }
  }

  private def constrainToString(vc: VariableConstraint): String = {
    vc match {
      case ConditionalConstraint(const, _) => constrainToString(const) + " if(...) "
      case IsConstraint(typ)               => s"is ${typ.toString(false, true)}"
      case NotConstraint(typ)              => s"is ^${typ.toString(false, true)}"
    }
  }

  private def filterWeaveTypeMetadata(wtype: WeaveType): Seq[Metadata] = {
    var result: Seq[Metadata] = Seq()
    if (!config.skipWeaveTypeMetadata) {
      val filteredMedatada = wtype.metadata().filterNot(metadata => config.weaveTypeMetadataToIgnore.contains(metadata.name))
      if (!FunctionTypeHelper.isDynamicTypeParameter(wtype) && filteredMedatada.nonEmpty) {
        result = filteredMedatada
      }
    }
    result
  }

  private def printTypeParameter(t: WeaveType, builder: StringCodeWriter, recursionDetector: RecursionDetector[String], insideKey: Boolean, depth: Int) = {
    builder.print("<")
    builder.startIgnoringNewLines()
    generate(t, builder, recursionDetector, insideKey, depth + 1)
    builder.endIgnoringNewLines()
    builder.print(">")
  }

  private def inlineObject(properties: Seq[KeyValuePairType]) = {
    properties.isEmpty || (properties.size == 1 && isSimpleType(properties.head.value))
  }

  /**
    * Returns false if this type contains any complex object (an object with more than one kvp) otherwise returns true
    *
    * @param weaveType The type to check
    * @return True if not compex
    */
  def isSimpleType(weaveType: WeaveType): Boolean = {
    !WeaveTypeTraverse.exists(weaveType, {
      case ot: ObjectType => ot.properties.size > 1
      case _              => false
    })
  }

  def isImplicitParameter(variable: String): Boolean = {
    variable.matches("""\$+""")
  }

  def printQName(builder: StringCodeWriter, name: QName): CodeWriter = {
    builder.print(name.name)
  }

}

case class WeaveTypeEmitterConfig(
  prettyPrint: Boolean = false,
  nameOnly: Boolean = false,
  generateMultiTypes: Boolean = true,
  skipMetadataConstraints: Boolean = false,
  emptyObjectWithText: Boolean = false,
  useLiteralType: Boolean = true,
  skipRecursive: Boolean = false,
  simplifyTypes: Boolean = false,
  printTypeBounds: Boolean = false,
  printFunctionConstrains: Boolean = false,
  printTypeParamInstanceId: Boolean = false,
  printConstrains: Boolean = false,
  weaveTypeMetadataToIgnore: Seq[String] = Seq(),
  skipWeaveTypeMetadata: Boolean = false,
  followReference: Boolean = true,
  maxDepth: Option[Int] = None)

class WeaveTypeEmitterConfigBuilder {
  private var prettyPrint: Boolean = false
  private var nameOnly: Boolean = false
  private var generateMultiTypes: Boolean = true
  private var skipMetadataConstraints: Boolean = false
  private var emptyObjectWithText: Boolean = false
  private var useLiteralType: Boolean = true
  private var skipRecursive: Boolean = false
  private var simplifiedTypes: Boolean = false
  private var printTypeBounds: Boolean = false
  private var printFunctionConstrains: Boolean = false
  private var weaveTypeMetadataToIgnore: Array[String] = Array()
  private var skipWeaveTypeMetadata: Boolean = false

  def withPrettyPrint(prettyPrint: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.prettyPrint = prettyPrint
    this
  }

  def withNameOnly(nameOnly: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.nameOnly = nameOnly
    this
  }

  def withGenerateMultiTypes(generateMultiTypes: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.generateMultiTypes = generateMultiTypes
    this
  }

  def withSkipMetadataConstraints(skipMetadataConstraints: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.skipMetadataConstraints = skipMetadataConstraints
    this
  }

  def withEmptyObjectWithText(emptyObjectWithText: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.emptyObjectWithText = emptyObjectWithText
    this
  }

  def withUserLiteralType(useLiteralType: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.useLiteralType = useLiteralType
    this
  }

  def withSkipRecursive(skipRecursive: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.skipRecursive = skipRecursive
    this
  }

  def withSimplifiedTypes(simplifiedTypes: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.simplifiedTypes = simplifiedTypes
    this
  }

  def withPrintTypeBound(printTypeBounds: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.printTypeBounds = printTypeBounds
    this
  }

  def withPrintFunctionConstrains(printFunctionConstrains: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.printFunctionConstrains = printFunctionConstrains
    this
  }

  def withWeaveTypeMetadataToIgnore(weaveTypeMetadataToIgnore: Array[String]): WeaveTypeEmitterConfigBuilder = {
    this.weaveTypeMetadataToIgnore = weaveTypeMetadataToIgnore
    this
  }

  def withSkipWeaveTypeMetadata(skipWeaveTypeMetadata: Boolean): WeaveTypeEmitterConfigBuilder = {
    this.skipWeaveTypeMetadata = skipWeaveTypeMetadata
    this
  }

  def build(): WeaveTypeEmitterConfig = {
    WeaveTypeEmitterConfig(
      prettyPrint = prettyPrint,
      nameOnly = nameOnly,
      generateMultiTypes = generateMultiTypes,
      skipMetadataConstraints = skipMetadataConstraints,
      emptyObjectWithText = emptyObjectWithText,
      useLiteralType = useLiteralType,
      skipRecursive = skipRecursive,
      simplifyTypes = simplifiedTypes,
      printTypeBounds = printTypeBounds,
      printFunctionConstrains = printFunctionConstrains,
      weaveTypeMetadataToIgnore = weaveTypeMetadataToIgnore.toSeq,
      skipWeaveTypeMetadata = skipWeaveTypeMetadata)
  }
}

object WeaveTypeEmitter {

  val ROOT_WEAVE_TYPE_NAME = "rootWeaveType___"
  val DEFAULT_MESSAGE_CONFIG_BUILDER = new WeaveTypeEmitterConfigBuilder()
    .withNameOnly(true)
    .withGenerateMultiTypes(false)
    .withSkipMetadataConstraints(true)
    .withEmptyObjectWithText(true)
    .withPrintFunctionConstrains(false)
    .withSkipWeaveTypeMetadata(true)

  val MESSAGE_CONFIG = DEFAULT_MESSAGE_CONFIG_BUILDER.build()

  def toString(
    wtype: WeaveType,
    prettyPrint: Boolean = false,
    nameOnly: Boolean = false,
    skipWeaveTypeMetadata: Boolean = true): String = {
    val emitter = new WeaveTypeEmitter(WeaveTypeEmitterConfig(prettyPrint = prettyPrint, nameOnly = nameOnly, printTypeBounds = true, skipWeaveTypeMetadata = skipWeaveTypeMetadata))
    emitter.toString(wtype)
  }

  def toCatalogString(types: Map[String, WeaveType]): String = {
    new WeaveTypeEmitter(WeaveTypeEmitterConfig(prettyPrint = true)).generateTypeCatalog(types)
  }

  def toCatalogString(name: String, wtype: WeaveType): String = {
    val config = WeaveTypeEmitterConfig(prettyPrint = true)
    new WeaveTypeEmitter(config).generateTypeCatalog(Map(name -> wtype))
  }

  @WeaveApi(Seq("data-weave-agent"))
  def toCatalogString(wtype: WeaveType): String = {
    toCatalogString(ROOT_WEAVE_TYPE_NAME, wtype)
  }

  def toCatalogString(config: WeaveTypeEmitterConfig, wtype: WeaveType): String = {
    new WeaveTypeEmitter(config).generateTypeCatalog(Map(ROOT_WEAVE_TYPE_NAME -> wtype))
  }

  /**
    * Generate a String representation of a Type that can be presented to a User. It is going to get shrinked if the maxLength gets
    *
    * @param theType The WeaveType to emmit
    * @param prefix  The prefix to be used
    * @param suffix   The suffix to be used
    * @return
    */
  def toPresentableString(theType: WeaveType, prefix: String = "`", suffix: String = "`", maxLength: Int = 40): String = {
    theType match {
      case ft: FunctionType => ft.toString(DEFAULT_MESSAGE_CONFIG_BUILDER.withPrintFunctionConstrains(true).build())
      case it: IntersectionType =>
        prefix + it.of
          .map(ip => {
            val typeString = toPresentableString(ip, prefix = "", suffix = "", maxLength)
            StringHelper.shorten(typeString.replaceAll("\n", " "), maxLength / it.of.length)
          })
          .mkString(" & ") + suffix
      case tp: TypeParameter => prefix + tp.toString(MESSAGE_CONFIG) + suffix
      case _ =>
        val typeString = theType.toString(MESSAGE_CONFIG)
        prefix + StringHelper.shorten(typeString.replaceAll("\n", " "), maxLength) + suffix
    }
  }
}
