package org.mule.weave.v2.editor

import org.mule.weave.v2.CalculatedConstraints
import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.codegen.CodeGeneratorSettings
import org.mule.weave.v2.codegen.InfixOptions
import org.mule.weave.v2.completion.AutoCompletionService
import org.mule.weave.v2.completion.DataFormatDescriptorProvider
import org.mule.weave.v2.completion.Suggestion
import org.mule.weave.v2.completion.SuggestionResult
import org.mule.weave.v2.editor.indexing.EmptyIndexService
import org.mule.weave.v2.editor.indexing.LocatedResult
import org.mule.weave.v2.editor.indexing.WeaveIdentifier
import org.mule.weave.v2.editor.indexing.WeaveIndexService
import org.mule.weave.v2.editor.quickfix.CreateFunctionDeclarationQuickFix
import org.mule.weave.v2.editor.quickfix.CreateNamespaceDeclarationQuickFix
import org.mule.weave.v2.editor.quickfix.CreateTypeDeclarationQuickFix
import org.mule.weave.v2.editor.quickfix.CreateVariableDeclarationQuickFix
import org.mule.weave.v2.editor.quickfix.InsertDefaultQuickFix
import org.mule.weave.v2.editor.quickfix.InsertImportModuleQuickFix
import org.mule.weave.v2.editor.quickfix.InsertImportQuickFix
import org.mule.weave.v2.editor.refactor.RefactorService
import org.mule.weave.v2.formatting.FormattingOptions
import org.mule.weave.v2.formatting.FormattingService
import org.mule.weave.v2.hover.HoverMessage
import org.mule.weave.v2.hover.HoverService
import org.mule.weave.v2.inspector.CodeInspector
import org.mule.weave.v2.inspector.DefaultInspector
import org.mule.weave.v2.inspector.EqualsBooleanInspector
import org.mule.weave.v2.inspector.IfNotNullInspector
import org.mule.weave.v2.inspector.ReduceConcatObjectsInspector
import org.mule.weave.v2.inspector.ReplaceUsingWithDoInspector
import org.mule.weave.v2.inspector.SizeOfEqualsZeroInspector
import org.mule.weave.v2.inspector.TypeOfInspector
import org.mule.weave.v2.inspector.UnnecessaryDefaultInspector
import org.mule.weave.v2.inspector.UnnecessaryDoubleNegationInspector
import org.mule.weave.v2.inspector.UnnecessaryIfBlockInspector
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.InvalidMethodTypesMessage
import org.mule.weave.v2.parser.InvalidReferenceMessage
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.QuickFixAwareMessage
import org.mule.weave.v2.parser.TypeMismatch
import org.mule.weave.v2.parser.annotation.InternalAstNodeAnnotation
import org.mule.weave.v2.parser.annotation.LabelsAstNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.AstNodeHelper.collectChildrenWith
import org.mule.weave.v2.parser.ast.DirectivesCapableNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.functions.DoBlockNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.functions.OverloadedFunctionNode
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.ImportedElement
import org.mule.weave.v2.parser.ast.header.directives.ImportedElements
import org.mule.weave.v2.parser.ast.header.directives.InputDirective
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.patterns.PatternExpressionNode
import org.mule.weave.v2.parser.ast.patterns.PatternMatcherNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.ObjectNode
import org.mule.weave.v2.parser.ast.types.DynamicReturnTypeNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.SimpleParserPosition
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.parser.phase.DependencyGraph
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.ScopeGraphResult
import org.mule.weave.v2.parser.phase.TypeCheckingResult
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.DependenciesAnalyzerService
import org.mule.weave.v2.scope.Reference
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.scope.VariableDependency
import org.mule.weave.v2.scope.VariableScope
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.signature.FunctionSignatureHelpService
import org.mule.weave.v2.signature.FunctionSignatureResult
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.Constraint
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.FunctionTypeParameter
import org.mule.weave.v2.ts.RecursionDetector
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeInformationProviderService
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.resolvers.FunctionCallNodeResolver
import org.mule.weave.v2.utils.AsciiDocMigrator
import org.mule.weave.v2.utils.AstGraphDotEmitter
import org.mule.weave.v2.utils.DataGraphDotEmitter
import org.mule.weave.v2.utils.VariableScopeDotEmitter
import org.mule.weave.v2.weavedoc.WeaveDocParser
import org.mule.weave.v2.weavedoc.WeaveDocumentation

import scala.annotation.tailrec
import scala.collection.mutable.ArrayBuffer

class WeaveDocumentToolingService(
  private val rootParsingContext: ParsingContext,
  val file: VirtualFile,
  private val virtualFileSystem: VirtualFileSystem,
  var dataFormatProvider: DataFormatDescriptorProvider,
  var configuration: WeaveToolingConfiguration = WeaveToolingConfiguration(),
  dependencyGraph: DependencyGraph = new DependencyGraph(),
  indexService: WeaveIndexService = EmptyIndexService,
  val inputs: ImplicitInput = ImplicitInput(),
  val expectedOutput: Option[WeaveType] = None) {

  private val inspector: DefaultInspector =
    DefaultInspector(
      scopeInspectors = Seq(TypeOfInspector, IfNotNullInspector, SizeOfEqualsZeroInspector, ReplaceUsingWithDoInspector, UnnecessaryDefaultInspector, EqualsBooleanInspector, UnnecessaryIfBlockInspector, UnnecessaryDoubleNegationInspector),
      typeInspectors = Seq[CodeInspector[TypeCheckingResult[_ <: AstNode]]](ReduceConcatObjectsInspector))
  private val editor: WeaveEditorSupport = WeaveEditorSupport(file.asResource(), () => buildParsingContext(), virtualFileSystem, dependencyGraph, inspector)
  private lazy val typeInfoProvider = new TypeInformationProviderService(editor)
  private lazy val documentNameIdentifier: NameIdentifier = file.getNameIdentifier

  /**
    * Returns the root documentation of the current open document
    *
    * @return The root documentation if any
    */
  def documentation(): Option[WeaveDoc] = {
    editor
      .astDocument()
      .flatMap((node) => node.weaveDoc.map((doc) => WeaveDoc(doc.literalValue)))
  }

  /**
    * Returns all the functions that are declared in this document
    *
    * @return The Array of all the functions
    */
  def availableFunctions(): Array[FunctionDefinition] = {

    def toWeaveType(referenceResolver: ScopeGraphTypeReferenceResolver, rt: WeaveTypeNode) = {
      rt match {
        case _: DynamicReturnTypeNode => AnyType()
        case _                        => WeaveType(rt, referenceResolver)
      }
    }

    def toFunctionDefinition(fqn: NameIdentifier, fn: FunctionNode, docs: String, referenceResolver: ScopeGraphTypeReferenceResolver): FunctionDefinition = {
      val params = fn.params.paramList
        .map(fp => {
          val name = fp.variable.name
          val maybeType = fp.wtype.map((wtn) => WeaveType(wtn, referenceResolver))
          val defaultValue = fp.defaultValue.map(CodeGenerator.generate)
          FunctionParameterDefinition(name, maybeType, defaultValue)
        })
        .toArray

      val labels: Array[String] = fn.annotation(classOf[LabelsAstNodeAnnotation]).map(_.labels).getOrElse(Array())
      FunctionDefinition(fqn, docs, params, fn.returnType.map(rt => {
        toWeaveType(referenceResolver, rt)
      }), labels)
    }

    def functionTypeToFunctionDefinition(labels: Array[String], nameIdentifier: NameIdentifier, ft: FunctionType): FunctionDefinition = {
      val asciidocDoc: String = ft.getDocumentation().getOrElse("")
      val params: Array[FunctionParameterDefinition] = ft.params
        .map(fpt => {
          val defaultValue: Option[String] = fpt.defaultValueType.map(_.toString)
          FunctionParameterDefinition(fpt.name, Option(fpt.wtype), defaultValue)
        })
        .toArray
      val returnType: Option[WeaveType] = Option(ft.returnType)
      FunctionDefinition(nameIdentifier, asciidocDoc, params, returnType, labels)
    }

    val parentNameIdentifier = rootParsingContext.nameIdentifier
    val maybeScopeResult = scopeGraph()

    if (maybeScopeResult.isDefined) {
      val scopeResult = maybeScopeResult.get
      val headerNode = scopeResult.rootScope.astNavigator().documentNode match {
        case dn: DocumentNode => dn.header
        case mn               => mn
      }

      val referenceResolver = new ScopeGraphTypeReferenceResolver(scopeResult)
      val varDirectives: Seq[VarDirective] = AstNodeHelper.collectDirectChildrenWith(headerNode, classOf[VarDirective])
      val vars: Array[FunctionDefinition] = varDirectives
        .flatMap(vd => {
          if (vd.annotation(classOf[InternalAstNodeAnnotation]).isDefined) { //If it is internal ignore it
            Seq()
          } else {
            val labels: Array[String] = vd.annotation(classOf[LabelsAstNodeAnnotation]).map(_.labels).getOrElse(Array())
            val maybeFunctionType = typeInfoProvider.typeOf(vd.value)
            maybeFunctionType match {
              case Some(ft: FunctionType) =>
                val nameIdentifier = NameIdentifier.withParent(parentNameIdentifier, vd.variable.name)
                if (ft.overloads.isEmpty) {
                  Seq(functionTypeToFunctionDefinition(labels, nameIdentifier, ft))
                } else {
                  ft.overloads.map(oft => functionTypeToFunctionDefinition(labels, nameIdentifier, oft))
                }
              case _ =>
                Seq()
            }
          }
        })
        .toArray
      val functionDirectiveNodes: Seq[FunctionDirectiveNode] = AstNodeHelper.collectDirectChildrenWith(headerNode, classOf[FunctionDirectiveNode])
      val functions = functionDirectiveNodes
        .flatMap(fd => {
          if (fd.annotation(classOf[InternalAstNodeAnnotation]).isDefined) { //If it is internal ignore it
            Seq()
          } else {
            val docs = fd.weaveDoc.map(_.literalValue).getOrElse("")
            val fqn = NameIdentifier.withParent(parentNameIdentifier, fd.variable.name)
            fd.literal match {
              case fn: FunctionNode =>
                Seq(toFunctionDefinition(fqn, fn, docs, referenceResolver))
              case ofn: OverloadedFunctionNode =>
                ofn.functionDirectives.map(fn => {
                  val docs = fn.weaveDoc.map(_.literalValue).getOrElse("")
                  toFunctionDefinition(fqn, fn.literal.asInstanceOf[FunctionNode], docs, referenceResolver)
                })
            }
          }
        })
        .toArray
      vars ++ functions
    } else {
      Array()
    }
  }

  /**
    * Retruns true if the model is a mapping
    *
    * @return True if it is a mapping.
    */
  def isMapping(): Boolean = {
    editor.astDocument().exists(_.isInstanceOf[DocumentNode])
  }

  /**
    * Returns the type that was inferred from the mapping body
    *
    * @return The type of this mapping body
    */
  def typeOfMapping(): Option[WeaveType] = {
    if (isMapping()) {
      val typeCheckResult = editor.runTypeCheck(withRecovery = false)
      val node = typeCheckResult.getResult().astNode
      typeCheckResult
        .getResult()
        .typeGraph
        .findNode(node)
        .flatMap((tn) => {
          tn.resultType()
        })
    } else {
      None
    }
  }

  def extractVariable(startOffset: Int, endOffset: Int): Option[CodeRefactor] = {
    new RefactorService(editor).extractVariable(startOffset, endOffset)
  }

  def extractConstant(startOffset: Int, endOffset: Int): Option[CodeRefactor] = {
    new RefactorService(editor).extractVariable(startOffset, endOffset, toRootScope = true)
  }

  def extractFunction(startOffset: Int, endOffset: Int): Option[CodeRefactor] = {
    new RefactorService(editor).extractFunction(startOffset, endOffset)
  }

  private def calculateConstrains(function: WeaveType, arguments: Seq[WeaveType], typeGraph: TypeGraph): Array[CalculatedConstraints] = {
    val context = new WeaveTypeResolutionContext(typeGraph)

    def resolveConstrain(ft: FunctionType): Option[CalculatedConstraints] = {
      val types = ft.params.map(_.wtype)
      val expandedArgs = FunctionCallNodeResolver.expandWithDefaultValues(arguments, ft.params)
      if (expandedArgs.size == types.size) {
        val constraintSet = types
          .zip(expandedArgs)
          .map((ea) => {
            Constraint.collectConstrains(ea._1, ea._2, context)
          })
          .reduce(_.merge(_))
        Some(CalculatedConstraints(ft, expandedArgs.toArray, constraintSet))
      } else {
        None
      }
    }

    function match {
      case ft: FunctionType => {
        if (ft.overloads.isEmpty) {
          resolveConstrain(ft) match {
            case Some(value) => Array(value)
            case None        => Array()
          }
        } else {
          ft.overloads.flatMap((ft) => resolveConstrain(ft)).toArray
        }
      }
      case ut: UnionType     => ut.of.flatMap((t) => calculateConstrains(t, arguments, typeGraph)).toArray
      case ut: ReferenceType => calculateConstrains(ut.resolveType(), arguments, typeGraph)
      case _                 => Array()
    }
  }

  /**
    * Returns the list of list of constraints of the function call in the given loaction
    *
    * @param start The start location of the function call
    * @param end   The end location of the location
    * @return The array of constraints
    */
  def collectFunctionCallConstrains(start: Int, end: Int): Array[CalculatedConstraints] = {
    editor
      .astNavigator()
      .flatMap((nav) => {
        val maybeNode = nav.nodeAt(start, end)
        maybeNode.flatMap((node) => {
          editor
            .typeGraph()
            .flatMap((tg) => {
              val maybeNode = tg.findNode(node)
              maybeNode.flatMap((node) => {
                if (node.typeResolver eq FunctionCallNodeResolver) {
                  val arguments: Seq[WeaveType] = FunctionCallNodeResolver.argumentEdges(node).flatMap(_.mayBeIncomingType())
                  val function = FunctionCallNodeResolver.getIncomingFunctionEdge(node).mayBeIncomingType()
                  function match {
                    case Some(value) => Some(calculateConstrains(value, arguments, tg))
                    case None        => None
                  }

                } else {
                  None
                }
              })
            })
        })
      })
      .getOrElse(Array())
  }

  /**
    * Validates the docs of this document and check they
    *
    * @return
    */
  def validateDocs(): ValidationMessages = {
    editor.astDocument() match {
      case Some(ast) => {
        val docs = AstNodeHelper.collectChildren(ast, (node) => node.hasWeaveDoc).map(_.weaveDoc.get)
        docs
          .map((comment) => {
            val messageCollector = new MessageCollector
            val value = comment.literalValue.linesIterator.map((line) => "  " + line).mkString("\n")
            WeaveDocParser.parseAst("   \n" + value, messageCollector)
            val commentLocation = comment.location()
            val mappedErrors = shiftMessagesPosition(commentLocation, messageCollector.errorMessages)
            val mappedWarnings = shiftMessagesPosition(commentLocation, messageCollector.warningMessages)
            ValidationMessages(asMessages(mappedErrors), asMessages(mappedWarnings))
          })
          .foldLeft(ValidationMessages(Array(), Array()))(_.concat(_))
      }
      case None => {
        ValidationMessages(Array(), Array())
      }
    }
  }

  private def shiftMessagesPosition(commentLocation: WeaveLocation, messages: Seq[(WeaveLocation, Message)]): Seq[(WeaveLocation, Message)] = {
    val indexShift = commentLocation.startPosition.index
    val lineShift = commentLocation.startPosition.line
    messages.map((message) => {
      val startPosition = message._1.startPosition
      val endPosition = message._1.endPosition
      val newStartPosition = SimpleParserPosition(indexShift + startPosition.index, lineShift + startPosition.line, startPosition.column, startPosition.source)
      val newEndPosition = SimpleParserPosition(indexShift + endPosition.index, lineShift + endPosition.line, endPosition.column, endPosition.source)
      (WeaveLocation(newStartPosition, newEndPosition, commentLocation.resourceName), message._2)
    })
  }

  def cursorAt(cursorLocation: Int): Unit = {
    this.editor.updateCursor(cursorLocation)
  }

  /**
    * Returns the weavedoc scaffold of the given element.
    *
    * @param startLocation The start offset of the element
    * @param endLocation   The end offset of the element
    * @return
    */
  def scaffoldDocs(startLocation: Int, endLocation: Int): Option[String] = {
    this.editor.astNavigator() match {
      case Some(navigator) =>
        val maybeNode = navigator.nodeAt(startLocation, endLocation)
        maybeNode match {
          case Some(functionDirectiveNode: FunctionDirectiveNode) =>
            val literal = functionDirectiveNode.literal
            val namesWithTypes: Seq[(String, String)] = literal match {
              case fn: FunctionNode =>
                fn.params.paramList.map(p => (p.variable.name, p.wtype.map(wtype => CodeGenerator.generate(wtype)).getOrElse("Any")))
              case ofn: OverloadedFunctionNode =>
                ofn.functions.head.params.paramList.map(p => (p.variable.name, p.wtype.map(wtype => CodeGenerator.generate(wtype)).getOrElse("Any")))
              case _ => Seq()
            }

            val params = {
              if (namesWithTypes.isEmpty) {
                "*"
              } else {
                s"""*
                   |* === Parameters
                   |*
                   |* [%header, cols="1,1,3"]
                   |* |===
                   |* | Name | Type | Description
                   |${namesWithTypes.map(nameWithType => "* | `" + nameWithType._1 + "` | " + toValidTypeDoc(nameWithType._2) + " | ").mkString("\n")}
                   |* |===
                   |*""".stripMargin.trim
              }
            }

            val doc =
              s"""
                 |/**
                 |* Describes the `${functionDirectiveNode.variable.name}` function purpose.
                 |$params
                 |* === Example
                 |*
                 |* This example shows how the `${functionDirectiveNode.variable.name}` function behaves under different inputs.
                 |*
                 |* ==== Source
                 |*
                 |* [source,DataWeave,linenums]
                 |* ----
                 |* %dw 2.0
                 |* output application/json
                 |* ---
                 |*
                 |*
                 |* ----
                 |*
                 |* ==== Output
                 |*
                 |* [source,Json,linenums]
                 |* ----
                 |*
                 |* ----
                 |*
                 |*/
            """.stripMargin
            Some(doc)
          case _ => None
        }
      case _ => None

    }

  }

  private def toValidTypeDoc(weaveType: String): String = {
    if (weaveType.nonEmpty) {
      weaveType
        .replace("-", "&#45;")
        .replace("+", "&#43;")
        .replace("|", "&#124;")
        .replace(">", "&#62;")
        .replace("(", "&#40;")
        .replace(")", "&#41;")
    } else {
      weaveType
    }
  }

  /**
    * Returns the relative file path of the test for a given Document
    * @return The path of the test file
    */
  def getTestPathFromDefinition(): String = {
    WeaveUnitTestSuite.testFilePathFromNameIdentifier(documentNameIdentifier.name)
  }

  /**
    * Returns the initial test content for a given function
    * @param startLocation The start of the function location
    * @param endLocation The end of the function location
    * @return
    */
  def createUnitTestFromDefinition(startLocation: Int, endLocation: Int): Option[String] = {
    functionDirectiveAt(startLocation, endLocation)
      .map((node) => {
        val testPath = WeaveUnitTestSuite.expectedPath(node, documentNameIdentifier.name).testPath
        WeaveUnitTestSuite.initialTest(node, documentNameIdentifier.name, testPath)
      })
  }

  def addUnitTestFromDefinition(startLocation: Int, endLocation: Int, testAst: AstNode): Option[WeaveUnitTestAddition] = {
    functionDirectiveAt(startLocation, endLocation)
      .flatMap((node) => {
        val testPath = WeaveUnitTestSuite.expectedPath(node, documentNameIdentifier.name).testPath
        WeaveUnitTestSuite.fromExisting(node, testAst, testPath)
      })
  }

  private def functionDirectiveAt(startLocation: Int, endLocation: Int): Option[FunctionDirectiveNode] = {
    nodeAtLocation(startLocation, endLocation)
      .flatMap({
        case fd: FunctionDirectiveNode => Some(fd)
        case ModuleNode(_, directives) => {
          directives.collectFirst({
            case fd: FunctionDirectiveNode => fd
          })
        }
        case _ => None
      })
  }

  private def nodeAtLocation(startLocation: Int, endLocation: Int): Option[AstNode] = {
    this.editor
      .astNavigator()
      .flatMap((navigator) => {
        navigator.nodeAt(startLocation, endLocation)
      })
  }

  /**
    * Returns the information of the function signature being called.
    *
    * @param offset the offset where the cursor is located
    */
  def signatureInfo(offset: Int): Option[FunctionSignatureResult] = {
    editor.updateCursor(offset)
    new FunctionSignatureHelpService(editor).functionSignatureAt(offset)
  }

  /**
    * Returns the suggestions for the element at the given offset
    *
    * @param offset The offset of the ast node
    * @return The suggestions
    */
  def completion(offset: Int): SuggestionResult = {
    editor.updateCursor(offset)
    val service = new AutoCompletionService(editor, dataFormatProvider, indexService)
    service.suggest()
  }

  /**
    * Returns the offset of a given line and column.
    * Line is 0 base
    *
    * @param line   The line number
    * @param column The column number
    * @return The offset
    */
  def offsetOf(line: Int, column: Int): Int = {
    editor.offsetOf(line, column)
  }

  /**
    * Returns the node at the given line and column
    *
    * @param line   The line of the search node
    * @param column The column of the search node
    * @return The node if found
    */
  def nodeAt(line: Int, column: Int, clazz: Option[Class[_ <: AstNode]]): Option[AstNode] = {
    editor
      .astNavigator()
      .flatMap((navigator) => {
        navigator.nodeAt(offsetOf(line, column), clazz)
      })
  }

  /**
    * Returns the node found at the given range
    *
    * @param startOffset The start range
    * @param endOffset   The end range
    * @return The node if found
    */
  def nodeAt(startOffset: Int, endOffset: Int): Option[AstNode] = {
    nodeAtLocation(startOffset, endOffset)
  }

  /**
    * The position of the given offset
    */
  def positionOf(offset: Int): Position = {
    editor.positionOf(offset)
  }

  /**
    * The Suggestions at the given offset
    */
  def completionItems(offset: Int): Array[Suggestion] = {
    completion(offset).suggestions
  }

  /**
    * Returns the list of symbols
    *
    * @return
    */
  def documentSymbol(): Array[SymbolInformation] = {
    editor
      .astNavigator()
      .map((navigator) => {
        val directiveNodes = collectChildrenWith(navigator.documentNode, classOf[DirectiveNode])
        directiveNodes.collect({
          //We should ignore overloaded as is an artifiicial node type
          case fd: FunctionDirectiveNode if (!fd.literal.isInstanceOf[OverloadedFunctionNode]) => {
            SymbolInformation(fd.variable.name, SymbolKind.Function, fd.location())
          }
          case vd: VarDirective => {
            SymbolInformation(vd.variable.name, SymbolKind.Variable, vd.location())
          }
          case ns: NamespaceDirective => {
            SymbolInformation(ns.prefix.name, SymbolKind.Namespace, ns.location())
          }
          case ns: AnnotationDirectiveNode => {
            SymbolInformation(ns.nameIdentifier.name, SymbolKind.Method, ns.location())
          }
          case td: TypeDirective => {
            SymbolInformation(td.variable.name, SymbolKind.Interface, td.location())
          }
        })

      })
      .getOrElse(Seq())
      .toArray
  }

  /**
    * Transforms an example into infix notation
    *
    * @param exampleContent The content of the exampe
    * @return The content of the new code
    */
  def asInfixExample(exampleContent: String): String = {
    val parsingContext = buildParsingContext()
    val value = DocumentParser(0).parse(WeaveResource("", exampleContent), parsingContext)
    if (value.hasResult()) {
      val document = value.getResult()
      CodeGenerator.generate(document.astNode, CodeGeneratorSettings(InfixOptions.ALWAYS))
    } else {
      exampleContent
    }
  }

  /**
    * Transform an example in prefix notation
    *
    * @param exampleContent The example to be transformed
    * @return The content of the new code
    */
  def asPrefixExample(exampleContent: String): String = {
    val value = DocumentParser(0).parse(WeaveResource("", exampleContent), buildParsingContext())
    if (value.hasResult()) {
      val document = value.getResult()
      CodeGenerator.generate(document.astNode, CodeGeneratorSettings(InfixOptions.ALWAYS))
    } else {
      exampleContent
    }
  }

  /**
    * Returns the list of visible variables at a given point
    *
    * @param location the location
    * @return The array of all the visible variables
    */
  def visibleLocalVariables(location: Int): Array[VisibleElement] = {

    def collectVariables(variableScope: VariableScope): Seq[NameIdentifier] = {
      variableScope.declarations() ++
        (variableScope.parentScope match {
          case Some(parentScope) => collectVariables(parentScope)
          case None              => Seq()
        })
    }

    scopeOf(location) match {
      case Some(variableScope) => {
        collectVariables(variableScope).distinct
          .filter(_.name != DocumentParser.FAKE_VARIABLE_NAME)
          .map((v) => {
            val maybeType = typeInfoProvider.typeOf(v)
            maybeType match {
              case Some(wtype) => VisibleElement(v.name, wtype)
              case None        => VisibleElement(v.name, AnyType())
            }
          })
          .toArray
      }
      case None => Array.empty
    }
  }

  /**
    * Returns the code formatted according to DW standard
    *
    * @return
    */
  def formatting(): Option[ReformatResult] = {
    val context = buildParsingContext()
    val value = DocumentParser().parse(file.asResource(), context)
    if (value.hasResult()) {
      val astNode = value.getResult().astNode
      Some(ReformatResult(astNode.location(), CodeGenerator.generate(astNode)))
    } else {
      //If we can not parse it then we just return the same
      None
    }
  }

  def format(startLocation: Int, endLocation: Int, weaveTextDocument: WeaveTextDocument, options: FormattingOptions): Unit = {
    val service = new FormattingService(options)
    editor
      .astNavigator(false)
      .foreach((navigator) => {
        navigator
          .nodeAt(startLocation, endLocation)
          .foreach((node) => {
            service.format(node, weaveTextDocument)
          })
      })
  }

  def formatDocument(weaveTextDocument: WeaveTextDocument, options: FormattingOptions): Unit = {
    val service = new FormattingService(options)
    editor
      .astNavigator(false)
      .foreach((navigator) => service.format(navigator.documentNode, weaveTextDocument))
  }

  /**
    * Returns information, Documentation, TypeInformation of that node
    *
    * @param offset The offset location the astnode
    * @return
    */
  def hoverResult(offset: Int): Option[HoverMessage] = {
    val maybeMessage = new HoverService(editor, dataFormatProvider).hover(offset)
    maybeMessage
  }

  /**
    * Return the elements referencing the element at the given offset
    *
    * @param offset The offset of the element
    * @return
    */
  def references(offset: Int): Array[Reference] = {
    val result = scopeGraph()
      .map((reference) => {
        val navigator = reference.rootScope.astNavigator()
        val maybeNode = navigator.nodeAt(offset)
        maybeNode match {
          case Some(ni: NameIdentifier) => {
            referencesOf(reference, ni)
          }
          case _ => Seq()
        }
      })
      .getOrElse(Seq())
    result.toArray
  }

  /**
    * Return the element that the astnode at the offset references to
    *
    * @param offset the offset zero based of the element
    * @return
    */
  def definition(offset: Int): Option[Reference] = {
    definitionLink(offset).map(_.reference)
  }

  /**
    * Return the element that the astnode at the offset references to
    *
    * @param offset The offset of the element
    * @return
    */
  def definitionLink(offset: Int): Option[Link] = {
    val maybeScopesNavigator = scopeGraph()
    maybeScopesNavigator
      .flatMap((scopeNavigator) => {
        val navigator = scopeNavigator.rootScope.astNavigator()
        val maybeNode = navigator.nodeAt(offset)
        maybeNode match {
          case Some(ni: NameIdentifier) => {
            navigator.parentOf(ni) match {
              case Some(ie: ImportedElement) if (navigator.parentOf(ie).exists(_.isInstanceOf[ImportedElements])) => {
                val id = navigator.parentWithType(ie, classOf[ImportDirective]).get
                val moduleScope = buildParsingContext().getScopeGraphForModule(id.importedModule.elementName)
                moduleScope.mayBeResult.flatMap((scope) => {
                  val maybeReference = scope.scope.rootScope.resolveLocalVariable(ni)
                  maybeReference.map((ref) => Link(ni, ref.copy(moduleSource = Some(id.importedModule.elementName))))
                })
              }
              case Some(id: ImportedElement) if (navigator.parentOf(id).exists(_.isInstanceOf[ImportDirective])) => {
                val importDirective = navigator.parentWithType(id, classOf[ImportDirective]).get
                val moduleScope: PhaseResult[ScopeGraphResult[ModuleNode]] = buildParsingContext().getScopeGraphForModule(importDirective.importedModule.elementName)
                moduleScope.mayBeResult.map((scope) => {
                  Link(ni, Reference(NameIdentifier(ni.name), scope.scope.rootScope, Some(importDirective.importedModule.elementName)))
                })
              }
              case _ => {
                //User is making click on the module
                if (isReferencingParentModule(offset, ni)) {
                  val maybeModuleName: Option[NameIdentifier] = ni.parent()
                  editor
                    .resolveFullQualifiedName(maybeModuleName.get)
                    .map((moduleName) => {
                      Link(ni, Reference(NameIdentifier(maybeModuleName.get.name), scopeNavigator.rootScope, Some(moduleName)))
                    })
                } else {
                  val maybeReference = scopeNavigator.resolveVariable(ni)
                  maybeReference.map(Link(ni, _))
                }
              }
            }
          }
          case _ => None
        }
      })
  }

  private def isReferencingParentModule(offset: Int, ni: NameIdentifier) = {
    val maybeModuleName: Option[NameIdentifier] = ni.parent()
    maybeModuleName.isDefined && maybeModuleName.get.location().contains(offset)
  }

  /**
    * Returns all the definitions that this location points to It can be multiple in the case of the overloaded functions.
    * In this case it will return the list of possible functions that can be resolved based on the parameters.
    *
    * @param offset The offset location
    * @return
    */
  def definitions(offset: Int): Array[Link] = {
    editor
      .astNavigator()
      .map((navigator) => {
        val maybeNodeAtCursor: Option[AstNode] = navigator.nodeAt(offset)
        maybeNodeAtCursor match {
          case Some(functionCallName: NameIdentifier) if (isReferencingParentModule(offset, functionCallName)) => {
            definitionLink(offset).toSeq
          }
          case Some(functionCallName: NameIdentifier) if (isFunctionCallReference(navigator, functionCallName)) => {
            definitionLink(offset).toSeq
              .flatMap((link) => {
                val astNavigator = link.reference.scope.astNavigator()
                astNavigator.parentOf(link.reference.referencedNode) match {
                  case Some(FunctionDirectiveNode(_, ofn: OverloadedFunctionNode, _)) if (editor.typeGraph().isDefined) => {
                    resolveFunctionDispatch(ofn, offset, navigator, functionCallName, link)
                  }
                  case _ => {
                    Some(link)
                  }
                }
              })
          }
          case Some(_) => {
            val maybeDefinitionLink = definitionLink(offset)
            if (editor.typeGraph().isDefined) {
              val maybeLinks = maybeDefinitionLink.map((link) => {
                val weaveType = typeInfoProvider.typeOf(offset)
                if (weaveType != null) {
                  weaveType match {
                    case ft: FunctionType if (ft.overloads.size > 1) => {
                      collectOverloads(link, ft)
                    }
                    case _ => {
                      Seq(link)
                    }
                  }
                } else {
                  Seq(link)
                }
              })
              maybeLinks.getOrElse(Seq())
            } else {
              maybeDefinitionLink.toSeq
            }

          }
          case _ => Seq()
        }

      })
      .getOrElse(Seq())
      .toArray

  }

  private def isFunctionCallReference(navigator: AstNavigator, functionCallName: NameIdentifier) = {
    navigator.granParentOf(functionCallName).exists(_.isInstanceOf[FunctionCallNode])
  }

  private def collectOverloads(link: Link, ft: FunctionType): Seq[Link] = {
    val links: Seq[Link] = ft.overloads.flatMap((ft) => {
      val astNavigator = link.reference.scope.astNavigator()
      val location = ft.location()
      val maybeLink: Option[Link] = astNavigator
        .nodeAt(location.startPosition.index, location.endPosition.index)
        .collect({
          case ni: NameIdentifier => ni
        })
        .map((ni) => {
          link.copy(reference = link.reference.copy(ni))
        })
      maybeLink
    })
    links
  }

  private def resolveFunctionDispatch(ofn: OverloadedFunctionNode, offset: Int, navigator: AstNavigator, functionCallName: NameIdentifier, link: Link): Seq[Link] = {
    val functionType: WeaveType = typeInfoProvider.typeOf(offset)
    functionType match {
      case ft: FunctionType if (ft.overloads.size > 1) => {
        val functionCall = navigator.granParentOf(functionCallName).map(_.asInstanceOf[FunctionCallNode]).get
        val argumentTypes: Seq[Option[WeaveType]] = functionCall.args.args.map((arg) => {
          typeInfoProvider.typeOf(arg)
        })
        val functionTypes = ft.overloads.filter((ft) => {
          val paramsArity = ft.params.size
          val argumentsArity = functionCall.args.args.size
          if (paramsArity >= argumentsArity) {
            val head = ft.params.headOption.exists(_.optional)
            val tail = ft.params.lastOption.exists(_.optional)

            val amountOptionalParams = ft.params.count(_.optional)
            val amountMissingParams = paramsArity - argumentsArity
            if (amountMissingParams > amountOptionalParams) {
              false
            } else {
              val allArguments: Seq[Option[WeaveType]] =
                if (head && paramsArity > argumentsArity) {
                  (0 until (amountOptionalParams - amountMissingParams)).map((_) => None) ++ argumentTypes
                } else if (tail && paramsArity > argumentsArity) {
                  argumentTypes ++ (0 until (amountOptionalParams - amountMissingParams)).map((_) => None)
                } else {
                  argumentTypes
                }
              if (allArguments.size != paramsArity) {
                false
              } else {
                allArguments
                  .zip(ft.params)
                  .forall((assignmentExpected) => {
                    val (assignment, expected) = assignmentExpected
                    if (assignment.isEmpty) {
                      true
                    } else {
                      canBeAssigned(assignment.get, expected)
                    }
                  })
              }
            }

          } else {
            false
          }

        })
        if (functionTypes.isEmpty) {
          Seq(link)
        } else {
          functionTypes.map((ft) => {
            val matchedFunction = ofn.functionDirectives.find((fn) => {
              fn.location().contains(ft.location())
            })
            matchedFunction match {
              case Some(value) => {
                link.copy(reference = link.reference.copy(value.variable))
              }
              case None => link
            }
          })
        }
      }
      case _ => {
        Seq(link)
      }
    }
  }

  private def canBeAssigned(assignment: WeaveType, expected: FunctionTypeParameter, recursionDetector: RecursionDetector[Boolean] = RecursionDetector((_, _) => false)): Boolean = {
    assignment match {
      case ut: UnionType => {
        ut.of.exists((wt) => canBeAssigned(wt, expected, recursionDetector))
      }
      case rt: ReferenceType => {
        recursionDetector.resolve(rt, (wt) => canBeAssigned(wt, expected, recursionDetector))
      }
      case _ => {
        TypeHelper.canBeSubstituted(assignment, expected.wtype, null)
      }
    }
  }

  /**
    * Returns the type of the element at that offset
    *
    * @param offset The offset of the element
    * @return The type or null if no type was found
    */
  def typeOf(offset: Int): WeaveType = {
    new TypeInformationProviderService(editor).typeOf(offset)
  }

  /**
    * Returns the type of the given astNode
    *
    * @param astNode astNode to search the type of
    * @return
    */
  def typeOf(astNode: AstNode): Option[WeaveType] = {
    new TypeInformationProviderService(editor).typeOf(astNode)
  }

  /**
    * Returns the type of the expression between that range
    *
    * @param startOffset The start index of the element
    * @param endOffset   The end index of the element
    * @return
    */
  def typeOf(startOffset: Int, endOffset: Int): WeaveType = {
    new TypeInformationProviderService(editor).typeOf(startOffset, endOffset)
  }

  /**
    * Type checks this document and returns the list of messages
    *
    * @return
    */
  def typeCheck(): ValidationMessages = {
    val typeCheckResult = editor.runTypeCheck(withRecovery = false)
    ValidationMessages(asMessages(typeCheckResult.errorMessages()), asMessages(typeCheckResult.warningMessages()))
  }

  /**
    * Parses this document and return the list of messages
    *
    * @return
    */
  def parseCheck(): ValidationMessages = {
    val typeCheckResult = editor.runParse()
    ValidationMessages(asMessages(typeCheckResult.errorMessages()), asMessages(typeCheckResult.warningMessages()))
  }

  /**
    * Scope checks this document and return the list of messages
    *
    * @return
    */
  def scopeCheck(): ValidationMessages = {
    val typeCheckResult = editor.runScopePhase(withRecovery = false)
    ValidationMessages(asMessages(typeCheckResult.errorMessages()), asMessages(typeCheckResult.warningMessages()))
  }

  /**
    * Returns the quick fixes that are available for a given message
    *
    * @param message The message to check if there is any quick fix available
    * @return The array of quick fixes
    */
  def getQuickFix(message: Message): Array[QuickFix] = {
    message match {
      case qfa: QuickFixAwareMessage => qfa.quickFixes()
      case InvalidReferenceMessage(reference) => {
        val maybeNavigator = editor.astNavigator()
        maybeNavigator match {
          case Some(astNavigator) => {
            val mayBeNode: Option[AstNode] = astNavigator.nodeAt(reference.location().startPosition.index, Some(reference.getClass))
            mayBeNode.map((node) => handleInvalidReferenceMessage(astNavigator, node).toArray).getOrElse(Array())
          }
          case None => Array()
        }
      }
      case im: InvalidMethodTypesMessage => {
        im.problems
          .flatMap((problem) => {
            if (problem._2.size == 1) {
              problem._2
                .flatMap((m) => getQuickFix(m._2).toSeq)
                .toArray[QuickFix]
            } else {
              Array[QuickFix]()
            }
          })
          .distinct
          .toArray
      }
      case tm: TypeMismatch => {
        if (tm.isNullMismatch && !(tm.actualType.location() eq UnknownLocation)) {
          Array(QuickFix(s"Insert `default` expression.", s"Insert `default` expression.", new InsertDefaultQuickFix(tm.expectedType, tm.actualType, tm.actualType.location())))
        } else {
          Array[QuickFix]()
        }
      }
      //      case UnableToResolveModule(nameIdentifier) =>
      //      case FunctionInvalidDefaultValueMessage() =>
      case _ => Array[QuickFix]()
    }
  }

  /**
    * Returns a dot representation of the AST
    *
    * @return
    */
  def astString(): Option[String] = {
    editor.astNavigator().map(_.documentNode).map(AstGraphDotEmitter.print(_))
  }

  /**
    * Returns the Ast of this Document
    *
    * @return
    */
  def ast(): Option[AstNode] = {
    editor.astDocument()
  }

  /**
    * Returns the scope graph of this Document
    *
    * @return
    */
  def scopeGraph(): Option[ScopesNavigator] = {
    editor.scopeGraph()
  }

  /**
    * Returns the input directive that has the same name as the one this document.
    *
    * @param inputName The name of the input
    * @return The input directive if any
    */
  def inputOf(inputName: String): Option[MappingInput] = {
    editor.astNavigator().map(_.documentNode) match {
      case Some(ast: DocumentNode) => {
        val directives: Seq[InputDirective] = AstNodeHelper.getInputs(ast)
        directives
          .find((id) => id.variable.name == inputName)
          .map((id) => {
            val properties: Array[DataFormatProperty] = id.options
              .map((ods) => {
                ods.flatMap((od) => {
                  od.value match {
                    case lv: LiteralValueAstNode => Some(DataFormatProperty(od.name.name, lv.literalValue))
                    case _                       => None
                  }
                })
              })
              .getOrElse(Seq.empty)
              .toArray

            val maybeMimeType: Option[DataFormatId] = id.mime.flatMap((mime) => {
              if (mime.mime.equals(MimeType.INTERNAL_MIME_TYPE)) {
                None
              } else {
                Some(DataFormatMimeType(mime.mime))
              }
            }).orElse(id.dataFormat.map((dfId) => DataFormatName(dfId.id)))
            val maybeType = typeOf(id.variable)
            MappingInput(inputName, maybeType, maybeMimeType, properties)
          })
      }
      case _ => None
    }
  }

  /**
    * Returns the folding regions for a given document
    *
    * @return
    */
  def foldingRegions(): Array[FoldingRegion] = {
    editor.astDocument() match {
      case Some(astNode) => {
        val collector = new ArrayBuffer[FoldingRegion]()
        AstNodeHelper.traverse(
          astNode,
          (astNode) => {
            astNode match {
              case dn: DoBlockNode => {
                collector.+=(FoldingRegion(dn.location(), RegionKind.CODE_BLOCKS))
              }
              case dn: DirectivesCapableNode => {
                val directives = dn.directives
                var i = 0
                var startPosition: Position = null
                var endPosition: Position = null
                while (i < directives.length) {
                  directives(i) match {
                    case id: ImportDirective => {
                      if (startPosition == null) {
                        startPosition = id.location().startPosition
                      }
                      endPosition = id.location().startPosition
                    }
                    case _ => {
                      if (startPosition != null && endPosition != null) {
                        collector.+=(FoldingRegion(WeaveLocation(startPosition, endPosition, documentNameIdentifier), RegionKind.IMPORT))
                        startPosition = null
                        endPosition = null
                      }
                    }
                  }
                  i = i + 1
                }
                //If is the only thing no more directives
                if (startPosition != null && endPosition != null) {
                  collector.+=(FoldingRegion(WeaveLocation(startPosition, endPosition, documentNameIdentifier), RegionKind.IMPORT))
                  startPosition = null
                  endPosition = null
                }
              }
              case dn: FunctionNode => {
                collector.+=(FoldingRegion(dn.body.location(), RegionKind.FUNCTIONS))
              }
              case dn: ObjectNode => {
                collector.+=(FoldingRegion(dn.location(), RegionKind.OBJECTS))
              }
              case dn: ArrayNode => {
                collector.+=(FoldingRegion(dn.location(), RegionKind.ARRAYS))
              }
              case pm: PatternMatcherNode => {
                collector.+=(FoldingRegion(pm.location(), RegionKind.PATTERN_MATCHER))
              }
              case pm: PatternExpressionNode => {
                collector.+=(FoldingRegion(pm.location(), RegionKind.CODE_BLOCKS))
              }
              case _ => {}
            }
            astNode.comments.foreach((c) => {
              collector.+=(FoldingRegion(c.location(), RegionKind.COMMENTS))
            })
            true
          })
        collector.toArray
      }
      case None => Array.empty
    }
  }

  /**
    * Returns a dot representation of the type graph
    *
    * @return
    */
  def typeGraphString(): Option[String] = {
    editor
      .typeGraph()
      .map((typeGraph) => {
        DataGraphDotEmitter.print(typeGraph)
      })
  }

  /**
    * Returns a dot representation of the scope graph
    *
    * @return
    */
  def scopeGraphString(): Option[String] = {
    scopeGraph()
      .map((scopeGraph) => {
        VariableScopeDotEmitter.print(scopeGraph.rootScope)
      })
  }

  @tailrec
  private def handleInvalidReferenceMessage(astNavigator: AstNavigator, refreshedRef: AstNode): Seq[QuickFix] = {
    refreshedRef match {
      case vrn: VariableReferenceNode => {
        val variableName = vrn.variable
        astNavigator.parentOf(refreshedRef) match {
          case Some(parent) => {
            parent match {
              case fcn: FunctionCallNode => {
                val quickFixes: Seq[QuickFix] = createImportQuickFixes(variableName)
                quickFixes :+
                  QuickFix(
                    s"Create Function `${variableName.name}`",
                    s"Declare New Function `${variableName.name}`",
                    new CreateFunctionDeclarationQuickFix(editor, fcn, variableName.name))
              }
              case _ =>
                val quickFixes: Seq[QuickFix] = createImportQuickFixes(variableName)
                quickFixes :+
                  QuickFix(
                    s"Create Variable `${variableName.name}`.",
                    s"Declare `${variableName.name}` Variable.",
                    new CreateVariableDeclarationQuickFix(editor, vrn, variableName.name))
            }
          }
          case _ =>
            val quickFixes: Seq[QuickFix] = createImportQuickFixes(variableName)
            quickFixes :+
              QuickFix(
                s"Create Variable `${variableName.name}`.",
                s"Declare `${variableName.name}` Variable.",
                new CreateVariableDeclarationQuickFix(editor, vrn, variableName.name))
        }
      }
      case trn: TypeReferenceNode => {
        val quickFixes: Seq[QuickFix] = createImportQuickFixes(trn.variable)
        quickFixes :+
          QuickFix(
            s"Create Type `${trn.variable.name}`.", //
            s"Declare `${trn.variable.name}` Type.", //
            new CreateTypeDeclarationQuickFix(editor, trn, trn.variable.name) //
          )
      }
      case ns: NamespaceNode => {
        val quickFixes: Seq[QuickFix] = createImportQuickFixes(ns.prefix)
        quickFixes :+
          QuickFix(s"Create Namespace `${ns.prefix.name}`.", s"Declare `${ns.prefix.name}` Namespace.", new CreateNamespaceDeclarationQuickFix(editor, ns, ns.prefix.name))
      }
      case ni: NameIdentifier => {
        handleInvalidReferenceMessage(astNavigator, astNavigator.parentOf(ni).get)
      }
      case _ => Seq()
    }
  }

  private def createImportQuickFixes(variableName: NameIdentifier): Seq[QuickFix] = {
    if (variableName.parent().isEmpty) {
      indexService
        .searchDefinitions(variableName.localName().name)
        .groupBy((definition) => definition.moduleName.name)
        .values
        .map(_.head)
        .map((definition) => {
          QuickFix(
            s"Import `${definition.value.value}` from `${definition.moduleName}`.",
            s"Import `${definition.value.value}` from `${definition.moduleName}`.",
            new InsertImportQuickFix(editor, definition.value.value, definition.moduleName))
        })
        .toSeq
    } else if (variableName.parent().get.parent().isEmpty) {
      indexService
        .searchDefinitions(variableName.localName().name)
        .filter((definition) => {
          definition.moduleName.localName().name.equals(variableName.parent().get.localName().name)
        })
        .groupBy((definition) => definition.moduleName.name)
        .values
        .map(_.head)
        .map((definition) => {
          QuickFix(
            s"Import module `${definition.moduleName.toString}`.",
            s"Import module `${definition.moduleName.toString}`.",
            new InsertImportModuleQuickFix(editor, definition.moduleName))
        })
        .toSeq
    } else {
      Seq()
    }
  }

  private def asMessages(errors: Seq[(WeaveLocation, Message)]): Array[ValidationMessage] = {
    errors
      .filter(_._1.resourceName.equals(documentNameIdentifier))
      .map((pair) => {
        val message = pair._2
        ValidationMessage(pair._1, message, getQuickFix(message))
      })
      .toArray
  }

  /**
    * Return the list of references (working as locations) that need to be changed
    *
    * @param offset  the offset of the element to rename
    * @param newName The new name of the element
    * @return
    */
  def rename(offset: Int, newName: String): Array[Reference] = {
    def resolveDeclarationRefs(ni: NameIdentifier, refService: ScopesNavigator): Seq[Reference] = {
      val maybeScope = refService.scopeOf(ni)
      val variableScope = maybeScope.getOrElse(refService.rootScope)
      referencesOf(refService, ni) ++ Seq(Reference(ni, variableScope, Some(documentNameIdentifier)))
    }

    val result = scopeGraph()
      .map((refService) => {
        val navigator = refService.rootScope.astNavigator()
        navigator.nodeAt(offset) match {
          case Some(ni: NameIdentifier) => {
            //Check if it is a definition or it is a usage
            val isDeclaration = isVariableDeclaration(ni, navigator)
            if (isDeclaration) {
              resolveDeclarationRefs(ni, refService)
            } else {
              refService.resolveVariable(ni) match {
                case Some(reference) => {
                  referencesOf(refService, reference.referencedNode) ++ Seq(reference)
                }
                case _ => {
                  resolveDeclarationRefs(ni, refService)
                }
              }
            }
          }
          case _ => {
            Seq()
          }
        }
      })
    result.map(_.toArray).getOrElse(Array())
  }

  private def isVariableDeclaration(ni: NameIdentifier, navigator: AstNavigator) = {
    val maybeNode = navigator.parentOf(ni)
    maybeNode match {
      case Some(_: FunctionParameter)     => true
      case Some(_: VarDirective)          => true
      case Some(_: FunctionDirectiveNode) => true
      case _                              => false
    }
  }

  def referencesOf(scopesNavigator: ScopesNavigator, ni: NameIdentifier): Seq[Reference] = {
    //If it is a root declaration and it can be referenced from outside
    //We need to search in other locations and see if it does reference to this ni
    val isRootDeclaration = scopesNavigator
      .scopeOf(ni)
      .exists((scope) => {
        scope.isRootScope() && scope.astNode.isInstanceOf[ModuleNode]
      })

    if (isRootDeclaration) {
      //Search for symbols with this name
      val references: Array[LocatedResult[WeaveIdentifier]] = indexService.searchReferences(ni.localName().name)

      val remoteReferences: Array[Reference] =
        references
          .filter((locatedResource) => {
            !locatedResource.moduleName.equals(documentNameIdentifier)
          })
          .flatMap((locatedReference) => {
            val nameMaybeResource = virtualFileSystem.asResourceResolver.resolve(locatedReference.moduleName)
            if (nameMaybeResource.isEmpty) {
              None
            } else {
              val scopeResult: PhaseResult[ScopeGraphResult[_ <: AstNode]] = DocumentParser().runScopePhases(nameMaybeResource.get, buildParsingContextFor(locatedReference.moduleName))
              val mayBeResult: Option[ScopeGraphResult[_ <: AstNode]] = scopeResult.mayBeResult
              mayBeResult match {
                case Some(scopeGraphResult) => {
                  val theReferencedScope: ScopesNavigator = scopeGraphResult.scope
                  val nameIdentifier: Option[AstNode] = theReferencedScope.astNavigator().nodeAt(locatedReference.value.startLocation, Some(classOf[NameIdentifier]))
                  nameIdentifier match {
                    case Some(theNameIdentifier: NameIdentifier) => {
                      val maybeReference: Option[Reference] = theReferencedScope.resolveVariable(theNameIdentifier)
                      maybeReference match {
                        //TODO comparison is not eq since they are different instances
                        case Some(ref) if (referenceSameElement(ref, ni)) => {
                          Some(Reference(theNameIdentifier, theReferencedScope.rootScope, Some(locatedReference.moduleName)))
                        }
                        case _ => None
                      }
                    }
                    case _ => None
                  }
                }
                case None => None
              }
            }
          })
      remoteReferences ++ //This returns local references
        scopesNavigator.resolveLocalReferencedBy(ni)
    } else {

      //This returns local references
      scopesNavigator.resolveLocalReferencedBy(ni)
    }
  }

  private def referenceSameElement(ref: Reference, ni: NameIdentifier) = {
    ref.isCrossModule &&
      ref.moduleSource.get.equals(documentNameIdentifier) &&
      ref.referencedNode.location().startPosition.index == ni.location().startPosition.index
  }

  /**
    * Returns the scope at a given location
    *
    * @param location The location where to search the scope
    * @return The scope
    */
  def scopeOf(location: Int): Option[VariableScope] = {
    val value = scopeGraph()
    if (value.isDefined) {
      val scope = value.get
      val astNavigator = scope.rootScope.astNavigator()
      val maybeNode = astNavigator.nodeAt(location)
      if (maybeNode.isDefined) {
        scope.scopeOf(maybeNode.get)
      } else {
        None
      }
    } else {
      None
    }
  }

  /**
    * The scope at a given location
    *
    * @param start The start position of the node
    * @param end   The end position of the node
    * @return The scope if available
    */
  def scopeOf(start: Int, end: Int): Option[VariableScope] = {
    val value = scopeGraph()
    if (value.isDefined) {
      val scope = value.get
      val astNavigator = scope.rootScope.astNavigator()
      val maybeNode = astNavigator.nodeAt(start, end)
      if (maybeNode.isDefined) {
        scope.scopeOf(maybeNode.get)
      } else {
        None
      }
    } else {
      None
    }
  }

  /**
    * Returns the list of variable that are declared under the specified node that are not locally resolved
    */
  def externalScopeDependencies(startBlock: Int, endBlock: Int, topScope: Option[VariableScope]): Array[VariableDependency] = {
    new DependenciesAnalyzerService(editor).externalScopeDependencies(startBlock, endBlock, topScope)
  }

  def buildParsingContext(): ParsingContext = {
    val context = rootParsingContext.withMessageCollector(new MessageCollector)
    inputs
      .implicitInputs()
      .foreach((implicitInputData) => {
        context.addImplicitInput(implicitInputData)
      })
    if (expectedOutput.isDefined) {
      context.expectedOutputType = expectedOutput
    }
    context.languageLevel = configuration.languageLevel
    context
  }

  def buildParsingContextFor(nameIdentifier: NameIdentifier): ParsingContext = {
    val context = rootParsingContext.withNameIdentifier(nameIdentifier)
    context.languageLevel = configuration.languageLevel
    context
  }

  override def toString = s"WeaveDocumentToolingService($file)"
}

case class WeaveDoc(asciidocDoc: String) {
  def markdownDoc(): String = {
    AsciiDocMigrator.toMarkDown(asciidocDoc)
  }

  def parseDoc(): WeaveDocumentation = {
    WeaveDocParser.parseDocumentation(asciidocDoc)
  }
}

case class Link(linkLocation: NameIdentifier, reference: Reference)

object WeaveDocumentToolingService {

  def apply(rootParsingContext: ParsingContext, file: VirtualFile, vfs: VirtualFileSystem, dataFormatProvider: DataFormatDescriptorProvider, languageLevel: WeaveToolingConfiguration, globalSymbols: WeaveIndexService): WeaveDocumentToolingService = {
    new WeaveDocumentToolingService(rootParsingContext, file, vfs, dataFormatProvider, languageLevel, indexService = globalSymbols)
  }

  def apply(rootParsingContext: ParsingContext, file: VirtualFile, vfs: VirtualFileSystem, dataFormatProvider: DataFormatDescriptorProvider, languageLevel: WeaveToolingConfiguration, globalSymbols: WeaveIndexService, inputs: ImplicitInput, expectedOutput: Option[WeaveType]): WeaveDocumentToolingService = {
    new WeaveDocumentToolingService(rootParsingContext, file, vfs, dataFormatProvider, languageLevel, indexService = globalSymbols, inputs = inputs, expectedOutput = expectedOutput)
  }

  def apply(rootParsingContext: ParsingContext, file: VirtualFile, vfs: VirtualFileSystem, dataFormatProvider: DataFormatDescriptorProvider, languageLevel: WeaveToolingConfiguration, dependencyGraph: DependencyGraph, globalSymbols: WeaveIndexService): WeaveDocumentToolingService = {
    new WeaveDocumentToolingService(rootParsingContext, file, vfs, dataFormatProvider, languageLevel, dependencyGraph, globalSymbols)
  }

  def apply(rootParsingContext: ParsingContext, file: VirtualFile, vfs: VirtualFileSystem, dataFormatProvider: DataFormatDescriptorProvider, languageLevel: WeaveToolingConfiguration, dependencyGraph: DependencyGraph, globalSymbols: WeaveIndexService, inputs: ImplicitInput, expectedOutput: Option[WeaveType]): WeaveDocumentToolingService = {
    new WeaveDocumentToolingService(rootParsingContext, file, vfs, dataFormatProvider, languageLevel, dependencyGraph, globalSymbols, inputs, expectedOutput)
  }
}

case class VisibleElement(name: String, weaveType: WeaveType)
