package org.mule.weave.v2.mapping

import org.mule.weave.v2.grammar.MultiValueSelectorOpId
import org.mule.weave.v2.grammar.ValueSelectorOpId
import org.mule.weave.v2.mapping.QNameType.QNameType
import org.mule.weave.v2.parser.annotation.QuotedStringAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.KeyNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.utils.StringEscapeHelper

import scala.annotation.tailrec

/**
  * This is a way to address (refer to) an element in a document (independent from the format).
  */
case class NamePathElement(_type: QNameType, name: String, ns: Option[String], parentMaybe: Option[NamePathElement]) {

  private var cachedPath: Array[NamePathElement] = _

  def path(): Array[NamePathElement] = {
    if (cachedPath == null) {
      val maybeParentPath = parentMaybe.map(_.path())
      val calculatedPath = maybeParentPath match {
        case Some(parentPath) => parentPath :+ this
        case None             => Array(this)
      }
      cachedPath = calculatedPath
    }
    cachedPath
  }

  def relativePath(): Array[NamePathElement] = {
    val maybeParentPath = parentMaybe
      .filterNot(x => x.isArray() || x.isRepeated() || x.isRoot())
      .map(_.relativePath())
    maybeParentPath match {
      case Some(parentPath) => parentPath :+ this
      case None             => Array(this)
    }
  }

  def relativePathFrom(other: NamePathElement): Array[NamePathElement] = {
    val maybeParentPath = parentMaybe
      .filterNot(_ == other)
      .map(_.relativePathFrom(other))
    maybeParentPath match {
      case Some(parentPath) => parentPath :+ this
      case None             => Array(this)
    }
  }

  lazy val cardinality: Int = {
    val _path = path()
    _path.count(_.isArrayOrRepeated())
  }

  /**
    * Returns the cardinality difference from the given QName
    * E.g. if the other QName has cardinality 2 and this has 3, the result will be 1
    */
  def cardinalityFrom(other: NamePathElement): Int = {
    if (other.contains(this)) {
      cardinality - other.cardinality
    } else {
      cardinality
    }
  }

  /**
    * A QName contains another one if it's a parent of it or equal to it.
    */
  def contains(other: NamePathElement): Boolean = {
    val path = this.path()
    val otherPath = other.path()
    if (path.length > otherPath.length) {
      return false
    }
    contains(path, otherPath)
  }

  @tailrec
  private def contains(path: Array[NamePathElement], otherPath: Array[NamePathElement]): Boolean = {
    path match {
      case Array() =>
        true
      case htail =>
        val x = htail.head
        val xs = htail.tail

        if (!x.equals(otherPath.head)) {
          false
        } else {
          contains(xs, otherPath.tail)
        }
    }
  }

  def isSelectable: Boolean = true

  def toKey: KeyNode = {
    val maybeNode: Option[NamespaceNode] = ns.map(x => NamespaceNode(NameIdentifier(x)))
    KeyNode(StringNode(name), maybeNode)
  }

  private def getNameNode() = {
    val str = nameWithNs()
    val stringNode = StringNode(str)
    if (StringEscapeHelper.keyRequiresQuotes(str)) {
      stringNode.annotate(QuotedStringAnnotation('"'))
    }
    NameNode(stringNode)
  }

  def toSelector(index: Int): AstNode = {
    parentMaybe match {
      case _ if name == NameIdentifier.$.name =>
        VariableReferenceNode(s"value$index")

      case Some(parent) if parent.isSelectable && !parent.isArray() && !parent.isRepeated() =>
        val nameNode = getNameNode()

        val op = if (this.isRepeated()) MultiValueSelectorOpId else ValueSelectorOpId
        BinaryOpNode(op, parent.toSelector(index), nameNode)

      case Some(parent) if parent.isSelectable && parent.isArrayOrRepeated() =>
        val mapParameter = VariableReferenceNode(s"value$index")
        val nameNode = getNameNode()
        BinaryOpNode(ValueSelectorOpId, mapParameter, nameNode)

      //      case Some(parent) if parent.isSelectable && parent.isRepeated() =>
      //        //TODO: implement the case where the parent is a repeated element
      //        val $ = VariableReferenceNode(NameIdentifier.$)
      //        val nameNode = NameNode(StringNode(nameWithNs()))
      //        BinaryOpNode(ValueSelectorOpId, $, nameNode)
      case _ =>
        VariableReferenceNode(name)
    }
  }

  def isRoot(): Boolean = false

  def isArray(): Boolean = _type == QNameType.Array

  def isRepeated(): Boolean = _type == QNameType.Repeated

  def isArrayOrRepeated(): Boolean = isArray() || isRepeated()

  def isAttribute(): Boolean = _type == QNameType.Attribute

  def isObject(): Boolean = _type == QNameType.Object

  def hasArrays(): Boolean = path().exists(_.isArray())

  def hasRepeated(): Boolean = path().exists(_.isRepeated())

  def nameWithNs(): String = {
    ns match {
      case Some(namespace) => s"$namespace#$name"
      case None            => name
    }
  }

  def segmentString(): String = {
    typePrefix + nameWithNs() + typeSuffix
  }

  private def typePrefix = {
    _type match {
      case QNameType.Attribute => "@"
      case _                   => ""
    }
  }

  private def typeSuffix = {
    _type match {
      case QNameType.Array    => "/[]"
      case QNameType.Repeated => "/*"
      case _                  => ""
    }
  }

  override def toString: String = {
    path().map(_.segmentString()).mkString("/")
  }

  override def equals(obj: scala.Any): Boolean = {
    super.equals(obj) || this.toString == obj.toString
  }
}

class RootNamePathElement(_type: QNameType) extends NamePathElement(_type, "", None, None) {
  override def toString: String = {
    _type match {
      case QNameType.Array    => "/[]"
      case QNameType.Repeated => "/*"
      case _                  => "/"
    }
  }

  override def isRoot(): Boolean = true

  override def isSelectable: Boolean = false
}

object NamePathElement {

  private val Attr = "^@(.*)".r
  //  private val Arr = """^\[\](.*)""".r
  //  private val Repeated = """^\*(.*)""".r
  private val NsName = "(?:^([^#]+)#)?(.+)".r //namespaces have the form ns0#foo

  def fromString(str: String) = {
    val segments = str.split("/").toList.filter(_.nonEmpty)
    if (segments.isEmpty) {
      new RootNamePathElement(QNameType.Object)
    } else {
      val (qNameType, isModifier) = checkType(segments.head)
      val remainder = if (isModifier) segments.tail else segments
      val root = new RootNamePathElement(qNameType)
      fromStringRec(remainder, Some(root))
    }
  }

  @tailrec
  private def fromStringRec(segments: List[String], parent: Option[NamePathElement]): NamePathElement = {
    segments match {
      case Attr(NsName(ns, name)) :: tail => //attribute
        val qName = NamePathElement(QNameType.Attribute, name, ns, parent)
        fromStringRec(tail, Some(qName))

      case NsName(ns, name) :: segment2 :: tail => //2 segments or more
        val (qNameType, isModifier) = checkType(segment2)
        val remainder = if (isModifier) tail else segment2 :: tail
        val qName = NamePathElement(qNameType, name, ns, parent)
        fromStringRec(remainder, Some(qName))

      case NsName(ns, name) :: Nil => //1 segment
        NamePathElement(QNameType.Object, name, ns, parent)

      case Nil =>
        parent.getOrElse(throw new IllegalArgumentException(""))
    }
  }

  /**
    * Returns a pair with (QNameType, isModifier)
    */
  def checkType(segment: String): (QNameType, Boolean) = {
    segment match {
      case "[]" => (QNameType.Array, true) //TODO: support arrays of arrays
      case "*"  => (QNameType.Repeated, true)
      case _    => (QNameType.Object, false)
    }
  }

  def apply(_type: QNameType, name: String, ns: String, parent: Option[NamePathElement]): NamePathElement = {
    val maybeNs = if (ns == null || ns.isEmpty) None else Some(ns)
    NamePathElement(_type, name, maybeNs, parent)
  }
}

object QNameType extends Enumeration {
  type QNameType = Value
  val Array, Object, Repeated, Attribute = Value
}