package org.mule.weave.v2.parser.ast.variables

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.api.tooling.ast.DWAstNodeKind
import org.mule.weave.v2.api.tooling.location.ResourceIdentifier
import org.mule.weave.v2.parser.annotation.InjectedNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.location.SimpleParserPosition
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.sdk.NameIdentifierHelper

import java.io.File
import java.util.StringTokenizer
import scala.collection.mutable.ArrayBuffer

/**
  * This class represents a fully qualified name. A name is build from a localName and its container module.
  * The parts of a name is separated by `::`. It may also contain a custom loader.
  */
@WeaveApi(Seq("data-weave-agent"))
case class NameIdentifier(name: String, loader: Option[String] = None) extends AstNode with ResourceIdentifier {

  assert(name != null, "Name should not be null")

  /**
    * The local name of this NameIdentifier
    *
    * @return The local name
    */
  def localName(): NameIdentifier = {
    NameIdentifier(nameElements().last)
  }

  /**
    * The name parts of this name
    *
    * @return All the parts
    */
  def nameElements(): Array[String] = {
    val nameElements: ArrayBuffer[String] = nameElementsSeq()
    nameElements.toArray
  }

  private def nameElementsSeq() = {
    //Avoid using split method as that one ends up using regex and is slow on turtles
    val elementsTokenizer: StringTokenizer = new StringTokenizer(name, NameIdentifier.SEPARATOR);
    // 8 elements should be more than enough for most of the use cases for parts of name identifier
    val nameElements = new ArrayBuffer[String](8)
    while (elementsTokenizer.hasMoreElements) {
      nameElements.+=(elementsTokenizer.nextToken())
    }
    nameElements
  }

  /**
    * If this name references to the current module or document or reference to an external module
    *
    * @return True if references an external module
    */
  def isLocalReference(): Boolean = {
    parent().isEmpty
  }

  /**
    * Then name of the container module if any
    *
    * @return The name of the corresponding container module if it is not local
    */
  def parent(): Option[NameIdentifier] = {
    val parentElements = nameElementsSeq().take(nameElements().length - 1)
    if (parentElements.isEmpty) {
      None
    } else {
      val parentIdentifier = NameIdentifier(parentElements.mkString(NameIdentifier.SEPARATOR), loader)
      val startPosition = location().startPosition
      val endPosition = location().endPosition
      parentIdentifier._location = Some(
        WeaveLocation(
          startPosition,
          SimpleParserPosition(startPosition.index + parentIdentifier.fullQualifiedName().length, endPosition.line, endPosition.column, endPosition.source),
          location().resourceName))
      Some(parentIdentifier)
    }
  }

  /**
    * The FQN name of this nameidentifer as a String
    *
    * @return The FQN string
    */
  def fullQualifiedName(): String = {
    loader.map(_ + "!").getOrElse("") + name
  }

  override def toString = s"$name"

  override def doClone(): NameIdentifier = copy()

  override def cloneAst(): NameIdentifier = super.cloneAst().asInstanceOf[NameIdentifier]

  def ::(localName: String): NameIdentifier = {
    NameIdentifier(name + NameIdentifier.SEPARATOR + localName)
  }

  def child(localName: String): NameIdentifier = {
    ::(localName)
  }

  override def getFQNIdentifier: String = {
    fullQualifiedName()
  }

  override def getKind(): String = DWAstNodeKind.NAME_IDENTIFIER
}

object NameIdentifier {
  val SEPARATOR: String = "::"

  val ANONYMOUS_NAME: NameIdentifier = NameIdentifier("anonymous")

  val SINCE_ANNOTATION = NameIdentifier("Since")

  val CORE_MODULE: NameIdentifier = NameIdentifier(ParsingContext.CORE_MODULE)
  val OBJECTS_MODULE: NameIdentifier = NameIdentifier("dw::core::Objects")
  val ARRAYS_MODULE: NameIdentifier = NameIdentifier("dw::core::Arrays")
  val CRYPTO_MODULE: NameIdentifier = NameIdentifier("dw::Crypto")
  val RUNTIME_MODULE: NameIdentifier = NameIdentifier("dw::Runtime")
  val SYSTEM_MODULE: NameIdentifier = NameIdentifier("dw::System")
  val DATA_FORMAT_MODULE: NameIdentifier = NameIdentifier("dw::extension::DataFormat")
  val INTERNAL_ANNOTATION: NameIdentifier = CORE_MODULE.::("Internal")
  val EXPERIMENTAL_ANNOTATION: NameIdentifier = CORE_MODULE.::("Experimental")
  val DEPRECATED_ANNOTATION: NameIdentifier = CORE_MODULE.::("Deprecated")
  val INTERCEPTOR_ANNOTATION: NameIdentifier = CORE_MODULE.::("Interceptor")
  val UNTRUSTED_CODE_ANNOTATION: NameIdentifier = CORE_MODULE.::("UntrustedCode")
  val RUNTIME_PRIVILEGE_ANNOTATION: NameIdentifier = CORE_MODULE.::("RuntimePrivilege")
  val ANNOTATION_TARGET_ANNOTATION: NameIdentifier = CORE_MODULE.::("AnnotationTarget")
  val METADATA_ANNOTATION: NameIdentifier = CORE_MODULE.::("Metadata")

  val INSERTED_FAKE_VARIABLE_NAME = "inserted_Weave_Fake_Variable_______"

  def INSERTED_FAKE_VARIABLE = NameIdentifier(INSERTED_FAKE_VARIABLE_NAME)

  def fromPath(path: String): NameIdentifier = {
    NameIdentifierHelper.fromWeaveFilePath(path, File.separator)
  }

  def fromElements(elements: Array[String]): NameIdentifier = {
    NameIdentifier(elements.mkString(SEPARATOR))
  }

  def fromElements(elements: Seq[String], loader: Option[String]): NameIdentifier = {
    NameIdentifier(elements.mkString(SEPARATOR), loader)
  }

  def fromFQN(fqn: String): NameIdentifier = {
    val strings = fqn.split('!')
    if (strings.nonEmpty) {
      var loader: Option[String] = None
      var elementIndex = 0
      var elements: Seq[String] = Seq()
      if (strings.length > 1) {
        loader = Some(strings(0))
        elementIndex = 1
      }
      elements = strings(elementIndex).split(SEPARATOR)
      fromElements(elements, loader)
    } else {
      NameIdentifier("")
    }
  }

  def withParent(parent: NameIdentifier, name: String) = {
    NameIdentifier(parent.name + SEPARATOR + name, parent.loader)
  }

  def anonymous = ANONYMOUS_NAME

  def isStarImport(nameIdentifier: NameIdentifier): Boolean = nameIdentifier.name == $star.name

  def $ = {
    val nameIdentifier = NameIdentifier("$")
    nameIdentifier.annotate(InjectedNodeAnnotation())
    nameIdentifier
  }

  def $$ = {
    val nameIdentifier = NameIdentifier("$$")
    nameIdentifier.annotate(InjectedNodeAnnotation())
    nameIdentifier
  }

  def $star = {
    NameIdentifier("*")
  }
}
