package org.mule.weave.v2.scaffolding

import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.utils.StringEscapeHelper

/**
  * Represents the path of an element
  *
  * @param elements The elements of the path
  */
case class WeaveTypePath(elements: Seq[WeaveTypeElement]) {

  /**
    * The String representation of this path
    * @return
    */
  def pathString(): String = {
    elements.map(_.pathString()).mkString("/")
  }

  /**
    * A new path with the given element as a new member of this path
    * @param child The new member of the path
    * @return The new path
    */
  def child(child: WeaveTypeElement): WeaveTypePath = {
    WeaveTypePath(elements :+ child)
  }
}

/**
  * Represent an element of a path
  */
sealed trait WeaveTypeElement {

  /**
    * Returns the path string of an element
    * @return
    */
  def pathString(): String
}

case class ArrayTypeElement() extends WeaveTypeElement {
  override def pathString(): String = "[_]"
}

case class FieldTypeElement(name: QName) extends WeaveTypeElement {
  override def pathString(): String = {
    val localName = StringEscapeHelper.escapeString(name.name)
    if (name.ns.isDefined) {
      s"{${name.ns.get}}${localName}"
    } else {
      s"${localName}"
    }
  }
}

case class AttributeTypeElement(name: QName) extends WeaveTypeElement {
  override def pathString(): String = {
    val localName = StringEscapeHelper.escapeString(name.name)
    if (name.ns.isDefined) {
      s"@{${name.ns.get}}${localName}"
    } else {
      s"@${localName}"
    }
  }
}
