import * from dw::http::Server
import * from dw::http::Types
import * from dw::core::Objects
import * from dw::Runtime
import * from dw::http::Interceptors
import dw::core::Binaries

type TypedValue = {
  content: String,
  contentType: String,
  encoding?: String
}

type Named = {
  name: String
}

type Configurable = {
  "options"?: Object
}

type WeaveInput = Named & TypedValue & Configurable

type TransformRequest = {
  inputs: Array<WeaveInput>,
  script: String
}

type ResourceCapable = {
  resources?: Dictionary<String>
}

type TypeRequest = TypedValue & Configurable & ResourceCapable

type ModuleContent = {content?: String, moduleIdentifier: String, errorMessage?: String}

type DataFormat = {
   name: String,
   defaultMimeType: String,
   mimeTypes : Array<String>,
   readOptions : Array<DataFormatOption>,
   writerOptions: Array<DataFormatOption>,
   dataType : String  //String, Boolean, Number,
}

type DataFormatOption = {
  name: String,
  defaultValue: String | Null,
  description: String | Null,
  possibleValues: Array<String>
}

var defaultServerConfig =
{
  "host": "localhost",
  "port": 8080,
  "contentType": "application/json"
}

fun exec(script: String, inputs: Array<Any>, resources: Dictionary<String>): TypedValue = native("playground::ExecFunction")

fun resolveModule(loader: String, module: String): ModuleContent = native("playground::ModuleResolver")

fun listDataFormats(): Array<DataFormat> = native("playground::ListDataFormats")

fun weaveTypeOf(sampleData: String, contentType: String, encoding: String, options: Object = {}): String | Null = native("playground::WeaveTypeOfFunction")

fun decodeInputs(inputs: Array<WeaveInput>) =
  inputs map ((in) -> in mapObject ((value, key) ->
    {
      (key): key match {
        case "content" -> Binaries::fromBase64(value)
        else -> value
      }
    }
  ))

fun buildTransformResponse(requestBody: TransformRequest) =
  {
    body: try(() -> exec(requestBody.script, decodeInputs(requestBody.inputs), requestBody.resources default {})),
    headers: {
      "Content-Type": "application/json"
    }
  }

fun buildTypesResponse(responseBody: Any) = {
  body: {
    success: responseBody != null,
    result: responseBody
  },
  headers: {
    "Content-Type": "application/json"
  }
}

fun start(serverConfig: ApiConfig = defaultServerConfig) =
  api(serverConfig mergeWith {interceptors: [CORS()]},
    {
      "/transform": {
        POST: ((request) -> buildTransformResponse(request.body as TransformRequest))
      },
      "/": {
        GET: ((request) -> serveResource("/index.html"))
      },
      "/type": {
        POST: ((request) ->
          buildTypesResponse(
            request.body match {
              case x is TypeRequest -> weaveTypeOf(x.content, x.contentType, x.encoding default "UTF-8", x.options default {})
              else -> null
            }
          )
        )
      },
      "/module": {
        POST: ((request) ->
          {
            body: resolveModule(request.body.loader, request.body.module)
          }
        )
      },
      "/dataformats": {
         GET: (request) ->
          {
            body: listDataFormats(),
            headers: {
             "Content-Type": "application/json"
            }
         }
      },
      //TODO: need uri params to extract parts of the path
      "/.+": {
        GET: ((request) -> serveResource(request.path))
      }
    }
  )
  //wait 60000

