package org.mule.weave.v2.module.protobuf.utils

import com.google.protobuf.Descriptors.FieldDescriptor
import com.google.protobuf.DynamicMessage
import com.google.protobuf.Message
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.protobuf.exception.ProtoBufParsingException
import org.mule.weave.v2.module.protobuf.exception.ProtoBufWritingException
import org.mule.weave.v2.parser.location.UnknownLocation

import java.util
import scala.collection.JavaConverters.seqAsJavaListConverter
import scala.collection.convert.ImplicitConversions.`collection AsScalaIterable`

trait FieldParser[ProtoBufType, DWType] {
  def accepts(fdesc: FieldDescriptor): Boolean

  def fromDw(value: Value[_], fdesc: FieldDescriptor)(implicit ctx: EvaluationContext): ProtoBufType = {
    try {
      doFromDw(value, fdesc)
    } catch {
      case e: ExecutionException =>
        throw new ProtoBufWritingException(value.location(), e.message)
    }
  }

  protected def doFromDw(value: Value[_], fdesc: FieldDescriptor)(implicit ctx: EvaluationContext): ProtoBufType

  def toDw(msg: Message, fdesc: FieldDescriptor): Value[DWType] = {
    if (!accepts(fdesc)) {
      throw new ProtoBufParsingException(s"Can't transform ${msg.getDescriptorForType.getFullName}", UnknownLocation)
    } else {
      doToDw(msg, fdesc)
    }
  }

  protected def doToDw(msg: Message, fdesc: FieldDescriptor): Value[DWType]
}

object FieldParser {
  val messageParsers = Seq(MapParser)

  def parseFieldFrom(msg: Message, fieldDescriptor: FieldDescriptor): Option[Value[Any]] =
    messageParsers.find(_.accepts(fieldDescriptor)).map(_.toDw(msg, fieldDescriptor))

  def writeField(value: Value[_], fieldDescriptor: FieldDescriptor)(implicit ctx: EvaluationContext): Option[Any] =
    messageParsers.find(_.accepts(fieldDescriptor)).map(_.fromDw(value, fieldDescriptor))
}

object MapParser extends FieldParser[java.util.List[DynamicMessage], ObjectSeq] {
  override def accepts(fdesc: FieldDescriptor): Boolean = fdesc.isMapField

  override def doFromDw(value: Value[_], fdesc: FieldDescriptor)(implicit ctx: EvaluationContext): java.util.List[DynamicMessage] = {
    val keyDesc = fdesc.getMessageType.findFieldByName("key")
    val valueDesc = fdesc.getMessageType.findFieldByName("value")
    val objValue = ObjectType.coerce(value).evaluate

    objValue
      .toIterator()
      .map(kvp => {
        val key = DWToProtoConverter.DWValueToProto(kvp._1, keyDesc)
        val value = DWToProtoConverter.DWValueToProto(kvp._2, valueDesc)
        val innerBuiled = DynamicMessage.newBuilder(fdesc.getMessageType)
        innerBuiled.setField(keyDesc, key)
        innerBuiled.setField(valueDesc, value)

        innerBuiled.build()
      })
      .toList
      .asJava
  }

  override def doToDw(msg: Message, fdesc: FieldDescriptor): Value[ObjectSeq] = {
    val fields: util.List[Message] = msg.getField(fdesc).asInstanceOf[java.util.List[Message]]
    ObjectValue(
      ObjectSeq(
        fields
          .map(msg => {
            val key = msg.getField(msg.getDescriptorForType.findFieldByName("key")).toString
            val value = ProtoToDWConverter.protoToDWValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")))
            KeyValuePair(KeyValue(key), value)
          })
          .toArray))
  }
}
