package org.mule.weave.v2.module.protobuf.utils

import com.google.protobuf
import com.google.protobuf.BoolValue
import com.google.protobuf.ByteString
import com.google.protobuf.BytesValue
import com.google.protobuf.Descriptors.Descriptor
import com.google.protobuf.Descriptors.GenericDescriptor
import com.google.protobuf.DoubleValue
import com.google.protobuf.Empty
import com.google.protobuf.FloatValue
import com.google.protobuf.Int32Value
import com.google.protobuf.Int64Value
import com.google.protobuf.ListValue
import com.google.protobuf.Message
import com.google.protobuf.NullValue
import com.google.protobuf.StringValue
import com.google.protobuf.Struct
import com.google.protobuf.Timestamp
import com.google.protobuf.UInt32Value
import com.google.protobuf.UInt64Value
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.LocalDateTimeType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.PeriodType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.BooleanValue
import org.mule.weave.v2.model.values.LocalDateTimeValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.PeriodValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.module.protobuf.exception.ProtoBufParsingException
import org.mule.weave.v2.module.protobuf.exception.ProtoBufWritingException
import org.mule.weave.v2.parser.location.UnknownLocation

import java.time.Duration
import java.time.LocalDateTime
import java.time.ZoneOffset
import java.time.temporal.ChronoUnit
import java.time.temporal.TemporalAmount
import scala.collection.JavaConverters.asJavaIterableConverter
import scala.collection.convert.ImplicitConversions.`collection AsScalaIterable`

trait MessageParser[+ProtoBufType <: Message, +DWType] {
  def accepts(msg: Message): Boolean = accepts(msg.getDescriptorForType)

  def accepts(desc: GenericDescriptor): Boolean = {
    desc.getFullName == descriptorName
  }

  def descriptorName: String

  def fromDw(value: Value[_])(implicit ctx: EvaluationContext): ProtoBufType =
    try {
      doFromDw(value)
    } catch {
      case e: ExecutionException =>
        throw new ProtoBufWritingException(value.location(), e.message)
    }

  protected def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): ProtoBufType

  def toDw(msg: Message): Value[DWType] = {
    if (!accepts(msg)) {
      throw new ProtoBufParsingException(s"Can't transform ${msg.getDescriptorForType.getFullName}", UnknownLocation)
    } else {
      doToDw(msg)
    }
  }

  protected def doToDw(msg: Message): Value[DWType]
}

object MessageParser {
  val messageParsers: Seq[MessageParser[Message, Any]] = Seq(
    BoolValueParser,
    DoubleValueParser,
    FloatValueParser,
    Int32ValueParser,
    Int64ValueParser,
    UInt32ValueParser,
    UInt64ValueParser,
    BytesValueParser,
    DurationParser,
    TimestampParser,
    EmptyParser,
    ValueParser,
    ListValueParser,
    StringValueParser,
    StructValueParser)

  def parseMessage(msg: Message): Option[Value[Any]] =
    messageParsers.find(_.accepts(msg)).map(_.toDw(msg))

  def writeMessage(value: Value[_], desc: Descriptor)(implicit ctx: EvaluationContext): Option[Message] =
    messageParsers.find(_.accepts(desc)).map(_.fromDw(value))
}

object BoolValueParser extends MessageParser[BoolValue, Boolean] {
  override def descriptorName: String = "google.protobuf.BoolValue"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): BoolValue =
    BoolValue.newBuilder().setValue(BooleanType.coerce(value).evaluate).build()

  override protected def doToDw(msg: Message): Value[Boolean] =
    BooleanValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[Boolean])
}

object BytesValueParser extends MessageParser[BytesValue, SeekableStream] {
  override def descriptorName: String = "google.protobuf.BytesValue"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): BytesValue =
    BytesValue.newBuilder().setValue(ByteString.readFrom(BinaryType.coerce(value).evaluate)).build()

  override protected def doToDw(msg: Message): Value[SeekableStream] =
    BinaryValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[ByteString].toByteArray)
}

trait NumberParser[ProtoBufType <: Message] extends MessageParser[ProtoBufType, Number]

object DoubleValueParser extends NumberParser[DoubleValue] {
  override def descriptorName: String = "google.protobuf.DoubleValue"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): DoubleValue =
    DoubleValue.newBuilder().setValue(NumberType.coerce(value).evaluate.toDouble).build()

  override def doToDw(msg: Message): Value[Number] =
    NumberValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[Double])
}

object FloatValueParser extends NumberParser[FloatValue] {
  override def descriptorName: String = "google.protobuf.FloatValue"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): FloatValue =
    FloatValue.newBuilder().setValue(NumberType.coerce(value).evaluate.toFloat).build()

  override def doToDw(msg: Message): Value[Number] =
    NumberValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[Float])
}

object Int32ValueParser extends NumberParser[Int32Value] {
  override def descriptorName: String = "google.protobuf.Int32Value"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): Int32Value =
    Int32Value.newBuilder().setValue(NumberType.coerce(value).evaluate.toInt).build()

  override def doToDw(msg: Message): Value[Number] =
    NumberValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[Int])
}

object Int64ValueParser extends NumberParser[Int64Value] {
  override def descriptorName: String = "google.protobuf.Int64Value"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): Int64Value =
    Int64Value.newBuilder().setValue(NumberType.coerce(value).evaluate.toLong).build()

  override def doToDw(msg: Message): Value[Number] =
    NumberValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[Long])
}

object UInt32ValueParser extends NumberParser[UInt32Value] {
  override def descriptorName: String = "google.protobuf.UInt32Value"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): UInt32Value =
    UInt32Value.newBuilder().setValue(NumberType.coerce(value).evaluate.toInt).build()

  override def doToDw(msg: Message): Value[Number] =
    NumberValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[Int])
}

object UInt64ValueParser extends NumberParser[UInt64Value] {
  override def descriptorName: String = "google.protobuf.UInt64Value"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): UInt64Value =
    UInt64Value.newBuilder().setValue(NumberType.coerce(value).evaluate.toLong).build()

  override def doToDw(msg: Message): Value[Number] =
    NumberValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[Long])
}

object DurationParser extends MessageParser[protobuf.Duration, TemporalAmount] {
  override def descriptorName: String = "google.protobuf.Duration"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): protobuf.Duration = {
    val dwPeriod = PeriodType.coerce(value).evaluate
    dwPeriod match {
      case duration: Duration =>
        protobuf.Duration
          .newBuilder()
          .setSeconds(duration.get(ChronoUnit.SECONDS))
          .setNanos(duration.get(ChronoUnit.NANOS).toInt)
          .build()
      case _ =>
        throw new ProtoBufWritingException(value.location(), "Can't write as protobuf.Duration")

    }
  }

  override def doToDw(msg: Message): Value[TemporalAmount] = {
    val nanos = msg.getField(msg.getDescriptorForType.findFieldByName("nanos")).asInstanceOf[Int]
    val seconds = msg.getField(msg.getDescriptorForType.findFieldByName("seconds")).asInstanceOf[Long]
    PeriodValue(Duration.ofSeconds(seconds).plusNanos(nanos))
  }
}

object TimestampParser extends MessageParser[protobuf.Timestamp, LocalDateTime] {
  override def descriptorName: String = "google.protobuf.Timestamp"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): Timestamp = {
    val dateTime = LocalDateTimeType.coerce(value).evaluate.toInstant(ZoneOffset.UTC)
    protobuf.Timestamp.newBuilder().setSeconds(dateTime.getEpochSecond).setNanos(dateTime.getNano).build()
  }

  override def doToDw(msg: Message): Value[LocalDateTime] = {
    val nanos = msg.getField(msg.getDescriptorForType.findFieldByName("nanos")).asInstanceOf[Int]
    val seconds = msg.getField(msg.getDescriptorForType.findFieldByName("seconds")).asInstanceOf[Long]
    LocalDateTimeValue(LocalDateTime.ofEpochSecond(seconds, nanos, ZoneOffset.UTC))
  }
}

object EmptyParser extends MessageParser[Empty, ObjectSeq] {
  override def descriptorName: String = "google.protobuf.Empty"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): Empty = {
    val obj = ObjectType.coerce(value).evaluate
    if (obj.isEmpty())
      Empty.newBuilder().build()
    else
      throw new ProtoBufWritingException(value.location(), "Can't write as protobuf.Empty, since the object is not empty.")
  }

  override def doToDw(msg: Message): Value[ObjectSeq] = ObjectValue(ObjectSeq.empty)
}

object ValueParser extends MessageParser[protobuf.Value, Any] {
  override def descriptorName: String = "google.protobuf.Value"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): protobuf.Value = {
    val builder = protobuf.Value.newBuilder()
    if (NullType.accepts(value))
      builder.setNullValue(NullValue.NULL_VALUE)
    else if (NumberType.accepts(value))
      builder.setNumberValue(NumberType.coerce(value).evaluate.toDouble)
    else if (BooleanType.accepts(value))
      builder.setBoolValue(BooleanType.coerce(value).evaluate)
    else if (StringType.accepts(value))
      builder.setStringValue(StringType.coerce(value).evaluate.toString)
    else if (ObjectType.accepts(value)) {
      val struct = StructValueParser.fromDw(value)
      builder.setStructValue(struct.asInstanceOf[Struct])
    } else if (ArrayType.accepts(value)) {
      val list = ListValueParser.fromDw(value)
      builder.setListValue(list)
    } else
      throw new ProtoBufWritingException(
        value.location(),
        "Can't write as Value, since it's not any of Null, Number, Boolean, String, Object or Array")

    builder.build()
  }

  override def doToDw(msg: Message): Value[Any] = {
    if (msg.hasField(msg.getDescriptorForType.findFieldByName("null_value")))
      values.NullValue
    else if (msg.hasField(msg.getDescriptorForType.findFieldByName("number_value")))
      values.NumberValue(msg.getField(msg.getDescriptorForType.findFieldByName("number_value")).asInstanceOf[Double])
    else if (msg.hasField(msg.getDescriptorForType.findFieldByName("string_value")))
      values.StringValue(msg.getField(msg.getDescriptorForType.findFieldByName("string_value")).asInstanceOf[String])
    else if (msg.hasField(msg.getDescriptorForType.findFieldByName("bool_value")))
      values.BooleanValue(msg.getField(msg.getDescriptorForType.findFieldByName("bool_value")).asInstanceOf[Boolean])
    else if (msg.hasField(msg.getDescriptorForType.findFieldByName("struct_value"))) {
      val a = msg.getField(msg.getDescriptorForType.findFieldByName("struct_value")).asInstanceOf[Message]
      StructValueParser.toDw(a)
    } else if (msg.hasField(msg.getDescriptorForType.findFieldByName("list_value"))) {
      val a = msg.getField(msg.getDescriptorForType.findFieldByName("list_value")).asInstanceOf[Message]
      ListValueParser.toDw(a)
    } else
      throw new ProtoBufParsingException("Can't read as Value since none of the fields was set.", UnknownLocation)
  }
}

object ListValueParser extends MessageParser[ListValue, ArraySeq] {
  override def descriptorName: String = "google.protobuf.ListValue"

  override def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): ListValue = {
    val values = ArrayType.coerce(value).evaluate
    val builder = ListValue.newBuilder()

    builder.addAllValues(values.toIterator().map(v => ValueParser.fromDw(v)).toIterable.asJava)

    builder.build()
  }

  override def doToDw(msg: Message): Value[ArraySeq] = {
    val fieldDesc = msg.getDescriptorForType.findFieldByName("values")

    ArrayValue(
      msg.getField(fieldDesc).asInstanceOf[java.util.List[Message]].map(msg => ValueParser.toDw(msg)).toIterator,
      UnknownLocationCapable)
  }
}

object StringValueParser extends MessageParser[StringValue, CharSequence] {
  override def descriptorName: String = "google.protobuf.StringValue"

  override protected def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): StringValue =
    StringValue.newBuilder().setValue(StringType.coerce(value).evaluate.toString).build()

  override protected def doToDw(msg: Message): Value[CharSequence] =
    values.StringValue(msg.getField(msg.getDescriptorForType.findFieldByName("value")).asInstanceOf[String])
}

object StructValueParser extends MessageParser[Struct, ObjectSeq] {
  override def descriptorName: String = "google.protobuf.Struct"

  override protected def doFromDw(value: Value[_])(implicit ctx: EvaluationContext): Struct = {
    val values = ObjectType.coerce(value).evaluate
    val builder = Struct.newBuilder()

    values.toIterator().foreach((kvp) => {
      val value = ValueParser.fromDw(kvp._2)
      builder.putFields(kvp._1.evaluate.name, value)
    })
    builder.build()
  }

  override protected def doToDw(msg: Message): Value[ObjectSeq] = {
    MapParser.toDw(msg, msg.getDescriptorForType.findFieldByName("fields"))
  }
}

