package org.mule.weave.v2.module.protobuf

import com.google.protobuf.DescriptorProtos
import com.google.protobuf.Descriptors
import com.google.protobuf.InvalidProtocolBufferException
import org.mule.weave.v2.core.exception.MissingRequiredSettingException
import org.mule.weave.v2.core.exception.SchemaNotFoundException
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.option.ConfigurableSchemaSetting
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.module.option.Settings
import org.mule.weave.v2.module.option.StringModuleOption
import org.mule.weave.v2.module.protobuf.ProtoBufSettings.MESSAGE_TYPE
import org.mule.weave.v2.module.protobuf.exception.MessageTypeNotFoundException
import org.mule.weave.v2.module.protobuf.exception.ProtoBufParsingException
import org.mule.weave.v2.module.reader.ConfigurableDeferred
import org.mule.weave.v2.parser.location.UnknownLocation

import scala.collection.JavaConverters.collectionAsScalaIterableConverter

class ProtoBufSettings(override val dataFormat: DataFormat[_, _]) extends Settings with ConfigurableDeferred with ConfigurableSchemaSetting {
  var messageType: Option[String] = None

  lazy val messageDescriptor: Descriptors.Descriptor = {

    this.schema match {
      case Some(ConfigurableSchemaSetting.SchemaResult(resultSchema)) =>
        val set = try {
          DescriptorProtos.FileDescriptorSet.parseFrom(resultSchema)
        } catch {
          case pe: InvalidProtocolBufferException =>
            throw new ProtoBufParsingException(pe.getMessage, UnknownLocation)
        }
        val files = set.getFileList.asScala
        // Yes, files are in topological order, so this guarantees that dependencies are found correctly
        val descriptors = files.foldLeft(Seq[Descriptors.FileDescriptor]())((deps, f) => deps :+ Descriptors.FileDescriptor.buildFrom(f, deps.toArray))

        val messageTypes = descriptors.flatMap(_.getMessageTypes.asScala)
        this.messageType match {
          case Some(messageType) =>
            messageTypes
              .find((message: Descriptors.Descriptor) => message.getFullName == messageType)
              .getOrElse(throw new MessageTypeNotFoundException(messageType, UnknownLocation))
          case None =>
            throw new MissingRequiredSettingException(MESSAGE_TYPE, "ProtoBuf", UnknownLocation)
        }
      case Some(ConfigurableSchemaSetting.SchemaNotSet(location)) =>
        throw new MissingRequiredSettingException("descriptorUrl", "ProtoBuf", location)
      case Some(ConfigurableSchemaSetting.SchemaNotFound(location, schemaUrl)) =>
        throw new SchemaNotFoundException(location, schemaUrl)
      case None =>
        throw new MissingRequiredSettingException("descriptorUrl", "ProtoBuf", UnknownLocation)
    }

  }

  override def schemaUrlDescriptionUrl: String = "data-format/protobuf/descriptorUrl.asciidoc"
  override val schemaExtension: String = "-protobuf.desc"
  override val propertyName: String = ProtoBufSettings.DESCRIPTOR_URL

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions ++
      Map(
        StringModuleOption(ProtoBufSettings.MESSAGE_TYPE, descriptionUrl = "data-format/protobuf/messageType.asciidoc", required = true, defaultValue = null))
  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case ProtoBufSettings.MESSAGE_TYPE =>
        this.messageType = Option(value).map(_.asInstanceOf[String])
      case _ => super.writeSettingsValue(settingName, value)
    }
  }

}

object ProtoBufSettings {
  val MESSAGE_TYPE: String = "messageType"
  val DESCRIPTOR_URL: String = "descriptorUrl"
}