package org.mule.weave.v2.helper

import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.extension.ParsingContextCreator
import org.mule.weave.v2.interpreted.extension.WeaveBasedDataFormatExtensionLoaderService
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.SPIWeaveServicesProvider
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.WeaveServicesProvider
import org.mule.weave.v2.model.service.LanguageLevelService
import org.mule.weave.v2.model.service.RuntimeSettings
import org.mule.weave.v2.model.service.SettingsService
import org.mule.weave.v2.matchers.WeaveMatchers._
import org.mule.weave.v2.module.CompositeDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.core.json.JsonDataFormat
import org.mule.weave.v2.module.DataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DefaultDataFormatExtensionsLoaderService
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.phase.CompilationException
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.runtime.CompilationResult
import org.mule.weave.v2.runtime.ExecutableWeaveHelper
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.scalatest.funspec.AnyFunSpec
import org.scalatest.matchers.should.Matchers

import java.io.File
import java.io.FileReader
import java.io.FileWriter
import java.io.PrintWriter
import java.io.StringWriter
import java.util.Properties

class AbstractExceptionsTest extends AnyFunSpec
    with Matchers
    with FolderBasedTest
    with ParsingContextTestAware
    with ReadCapableTest
    with ConfigPropertyAwareTest {

  scenarios.foreach {
    case (scenario, inputs, transform, expected, configProperty) => {
      it(scenario) {
        beforeTest()
        try {
          val moduleNodeLoader: RuntimeModuleNodeCompiler = RuntimeModuleNodeCompiler()
          val languageLevelService = createLanguageLevelService(configProperty)
          val runtimeProperties = new Properties(System.getProperties)
          //Reduce the stack size so that the stack overflow can be controled
          runtimeProperties.setProperty(RuntimeSettings.STACK_SIZE, 12.toString)
          val settingsService = createSettingsService(configProperty, languageLevelService, Some(runtimeProperties))
          val context: ParsingContext = createTestParsingContext(transform, settingsService.parsingContext())
          val moduleParserManager = context.moduleParserManager
          val loaderService = WeaveBasedDataFormatExtensionLoaderService(ParsingContextCreator(moduleParserManager), ClassLoaderWeaveResourceResolver.noContextClassloader(), moduleNodeLoader)
          val manager = ServiceManager(
            WeaveServicesProvider(
              WeaveServicesProvider(
                Map[Class[_], Any](
                  //Map Of Services
                  classOf[LanguageLevelService] -> languageLevelService,
                  classOf[SettingsService] -> settingsService,
                  classOf[DataFormatExtensionsLoaderService] -> CompositeDataFormatExtensionsLoaderService(DefaultDataFormatExtensionsLoaderService, loaderService))),
              new SPIWeaveServicesProvider))

          implicit val evaluationContext: EvaluationContext = EvaluationContext(manager)
          val properties = new Properties()
          properties.load(new FileReader(expected))

          val exceptionName = properties.getProperty("exception")
          val msg = properties.getProperty("message")
          val msgPart = properties.getProperty("message.part")
          val thrown = intercept[Exception] {
            inputs.foreach((f) => {
              context.addImplicitInput(FileHelper.baseName(f), None)
            })
            val compilerResult: PhaseResult[CompilationResult[DocumentNode]] = WeaveCompiler.compile(WeaveResourceFactory.fromFile(transform), context)
            if (compilerResult.hasErrors()) {
              throw new CompilationException(compilerResult.messages())
            }
            val executable = compilerResult.getResult().executable
            val readers = if (scenario.endsWith("-in")) {
              ExecutableWeaveHelper.buildReaders(executable, buildReaders(inputs))
            } else {
              buildReaders(inputs)
            }
            executable.writeWith(executable.implicitWriterOption().getOrElse(new JsonDataFormat().writer(None)), readers, Map())
          }
          val stackTrace: StringWriter = new StringWriter()
          thrown.printStackTrace(new PrintWriter(stackTrace))
          val actualExceptionClass = thrown.getClass
          val actualMsg = thrown.getMessage
          if (msg != null) {
            if (!actualMsg.trim.equals(msg.trim)) {
              print(actualMsg)
              if (shouldUpdateResult) {
                val expectedExceptionFile: File = new File(new File(getTestCaseFolder(expected), scenario), expected.getName)
                System.out.println("WARNING -------------------- Updating ->" + expectedExceptionFile.getAbsolutePath)
                val properties = new Properties()
                properties.put("exception", actualExceptionClass.getName)
                properties.put("message", actualMsg)
                properties.store(new FileWriter(expectedExceptionFile), "")
              }
            }
          }

          val expectedExceptionClass = Class.forName(exceptionName)
          assert(expectedExceptionClass.isAssignableFrom(actualExceptionClass), s"'${actualExceptionClass.getName}' is not a subclass of '$exceptionName'")
          if (msg != null) {
            actualMsg should matchString(msg)(after being whiteSpaceNormalised)
          } else {
            assert(actualMsg.contains(msgPart))
          }

        } finally {
          afterTest()
        }
      }
    }
  }

  def scenarios: Seq[(String, Array[File], File, File, Option[File])] = {
    val folders = getTestFoldersFor(getClass)
    folders
      .flatMap(
        (folder) =>
          folder.listFiles
            .filter((file: File) => {
              file.isDirectory && file.listFiles().exists((f) => f.getName.equals("expected.properties"))
            })
            .filter(acceptScenario)
            .map((file: File) => {
              (file.getName, file.listFiles.filter(_.getName.matches("in[0-9]+\\.[a-zA-Z]+")), new File(file, "transform.dwl"), new File(file, "expected.properties"), configProperty(file))
            }))
      .sortBy((testEntry) => testEntry._1)
  }

  def beforeTest(): Unit = {}

  def afterTest(): Unit = {}

}
