package org.mule.weave.v2.helper

import java.io.BufferedInputStream
import java.io.InputStream
import java.sql
import java.sql.Time
import java.sql.Timestamp
import java.util
import java.util.Date
import java.util.GregorianCalendar
import java.util.Map
import java.util.Optional
import org.apache.commons.beanutils.BeanMap
import org.apache.commons.beanutils.PropertyUtilsBean

import java.util.OptionalDouble
import java.util.OptionalInt
import java.util.OptionalLong
import scala.collection.JavaConversions._
import scala.collection.JavaConverters._

object MapComparator {
  def diff(from: java.util.Map[Any, Any], to: java.util.Map[Any, Any]): Difference = {
    if (from == to) return Identical()
    if (from.size() != to.size()) return Different("Size is different " + from.size() + " - " + to.size(), "size()", from, to)

    val i: java.util.Iterator[util.Map.Entry[Any, Any]] = from.entrySet.iterator
    while (i.hasNext) {
      val e: util.Map.Entry[Any, Any] = i.next
      val key: Any = e.getKey
      val value: Any = e.getValue
      if (value == null) {
        if (!(to.get(key) == null && to.containsKey(key))) {
          return Different("Key missing is " + key.toString, key.toString, value, to.get(key))
        }
      } else {
        val difference: Difference = JavaComparator.diff(value, to.get(key))
        if (!difference.identical) {
          return Different("Key `" + key.toString + "` has different value.", key.toString, value, to.get(key), Some(difference.asInstanceOf[Different]))
        }
      }
    }
    Identical()
  }
}

object ArrayComparator {
  def diff(from: Array[Any], to: Array[Any]): Difference = {
    if (from sameElements to) return Identical()
    if (from.length != to.length) return Different("Size is different", "size()", from.length, to.length)

    val sourceIterator: Iterator[Any] = from.iterator
    val toIterator: Iterator[Any] = to.iterator
    var index = 0
    while (sourceIterator.hasNext) {
      val source: Any = sourceIterator.next()
      val target: Any = toIterator.next()
      val difference: Difference = JavaComparator.diff(source, target)
      if (!difference.identical) {
        return Different("Index different is " + index, index.toString, source, target, Some(difference.asInstanceOf[Different]))
      }
      index = index + 1
    }
    Identical()
  }
}

object IteratorComparator {
  def diff(sourceIterator: Iterator[Any], toIterator: Iterator[Any]): Difference = {
    var index = 0
    while (sourceIterator.hasNext && toIterator.hasNext) {
      val source: Any = sourceIterator.next()
      val target: Any = toIterator.next()
      val difference: Difference = JavaComparator.diff(source, target)
      if (!difference.identical) {
        return Different("Index different is " + index, index.toString, source, target, Some(difference.asInstanceOf[Different]))
      }
      index = index + 1
    }
    if (sourceIterator.hasNext || toIterator.hasNext) {
      return Different("length is different", "length()", index, index)
    }
    Identical()
  }
}

object BeanComparator {
  def diff(from: AnyRef, to: AnyRef): Difference = {
    if (from eq to) return Identical()
    val map = new BeanMap(to)
    if ((to != null && from == null) || (to == null && from != null))
      Different("Values are not equal", "value", from, to)
    else {
      //Size > 1 since get class should be ignore
      if (map.size() == 1 || from.isInstanceOf[String] || isOfTypeTime(to) || from.isInstanceOf[OptionalInt] || from.isInstanceOf[OptionalLong] || from.isInstanceOf[OptionalDouble]) {
        //If is empty then is not a bean treat as a simple value
        if (from.equals(to)) {
          Identical()
        } else {
          Different("Values are not equal", "value", from, to)
        }
      } else {
        val propUtils = new PropertyUtilsBean()
        for (propNameObject <- map.keySet()) {
          val propertyName = propNameObject.toString
          if (!propertyName.equals("class")) {
            if (!propUtils.isReadable(from, propertyName)) {
              return Different("Property  " + propertyName + " is not present in " + from.getClass, propertyName, from, to, None)
            }
            val sourceValue = propUtils.getProperty(from, propertyName)
            val targetValue = propUtils.getProperty(to, propertyName)
            val difference: Difference = JavaComparator.diff(sourceValue, targetValue)
            if (!difference.identical) {
              return Different("Property different is " + propertyName, propertyName, sourceValue, targetValue, Some(difference.asInstanceOf[Different]))
            }
          }
        }
        Identical()
      }
    }
  }

  private def isOfTypeTime(to: AnyRef) = {
    to.isInstanceOf[GregorianCalendar] || to.isInstanceOf[Date] || to.isInstanceOf[sql.Date] || to.isInstanceOf[Timestamp] || to.isInstanceOf[Time]
  }

}

object JavaComparator {
  def diff(from: Any, to: Any): Difference = {
    if (from.asInstanceOf[AnyRef] eq to.asInstanceOf[AnyRef]) {
      Identical()
    } else {
      from match {
        case op: Optional[Any]               => diff(op.orElse(null), to.asInstanceOf[Optional[Any]].orElse(null))
        case array: Array[_]                 => IteratorComparator.diff(array.iterator, to.asInstanceOf[Array[_]].iterator)
        case is: InputStream                 => InputStreamComparator.diff(is, to.asInstanceOf[InputStream])
        case map: java.util.Map[Any, Any]    => MapComparator.diff(map, to.asInstanceOf[Map[Any, Any]])
        case coll: java.util.Collection[Any] => IteratorComparator.diff(coll.iterator.asScala, to.asInstanceOf[util.Collection[Any]].iterator().asScala)
        case iter: java.util.Iterator[Any]   => IteratorComparator.diff(iter.asScala, to.asInstanceOf[util.Iterator[Any]].asScala)
        case cs: CharSequence                => CharSequenceComparator.diff(cs, to.asInstanceOf[CharSequence])
        case _                               => BeanComparator.diff(from.asInstanceOf[AnyRef], to.asInstanceOf[AnyRef])
      }
    }
  }
}

trait Difference {
  val identical: Boolean
}

case class Different(description: String, id: String, source: Any, target: Any, cause: Option[Different] = None) extends Difference {
  override val identical: Boolean = false

  def message(): String = {
    description + cause.map((cause) => " reason:\n" + cause.message()).getOrElse("")
  }

  def path(): String = {
    if (cause.isDefined) {
      id + "/" + cause.get.path()
    } else {
      id
    }
  }

  def finalCause(): Different = {
    if (cause.isDefined) {
      cause.get.finalCause()
    } else {
      this
    }
  }
}

case class Identical() extends Difference {
  override val identical: Boolean = true
}

object InputStreamComparator {

  def contentEquals(in1: InputStream, in2: InputStream): Difference = {

    var input1 = in1
    var input2 = in2

    if (!input1.isInstanceOf[BufferedInputStream]) input1 = new BufferedInputStream(input1)
    if (!input2.isInstanceOf[BufferedInputStream]) input2 = new BufferedInputStream(input2)
    var ch = input1.read
    var i = 0
    while (-1 != ch) {
      val ch2 = input2.read
      if (ch != ch2) {
        return Different(s"The content of the Stream is different at byte ${i}", "content", in1, in2)
      }
      ch = input1.read
      i = i + 1
    }
    val ch2 = input2.read
    if (ch2 == -1) {
      Identical()

    } else {
      Different(s"The length of the files are not the same.", "content", in1, in2)
    }
  }

  def diff(from: InputStream, to: InputStream): Difference = {
    contentEquals(from, to)
  }
}

object CharSequenceComparator {
  def diff(from: CharSequence, to: CharSequence): Difference = {
    if (from.toString.equals(to.toString)) {
      Identical()
    } else {
      Different("The content of CharSequence is different", "value", from, to)
    }
  }
}