package org.mule.weave.v2.interpreted.node.executors

import org.mule.weave.v2.core.functions.BaseTernaryFunctionValue
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.UnexpectedFunctionCallTypesException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.findMatchingFunctionWithCoercion
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValuesHelper

import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicReference

class TernaryFunctionExecutor(
  override val node: ValueNode[_],
  val name: String,
  val firstArgConstantType: Boolean,
  val secondArgConstantType: Boolean,
  val thirdArgConstantType: Boolean,
  override val showInStacktrace: Boolean = false,
  override val location: WeaveLocation) extends TernaryExecutor with Product4[ValueNode[_], String, Boolean, Boolean] {

  private val validated: AtomicBoolean = new AtomicBoolean(false)

  private val cachedCoercedDispatch: AtomicReference[(Type, Type, Type, Seq[Int])] = new AtomicReference()

  override def executeTernary(fv: Value[_], sv: Value[_], tv: Value[_])(implicit ctx: ExecutionContext): Value[Any] = {
    val activeFrame: Frame = ctx.executionStack().activeFrame()
    try {
      activeFrame.updateCallSite(node)
      val functionValue = target().asInstanceOf[BaseTernaryFunctionValue]
      //This is the fast path
      if (validated.get()) {
        //If values are literal we do not need to validate every just the first time when we load the operation
        //Then is ok every time as its type will NEVER change
        val firstValue = if (!firstArgConstantType && functionValue.firstParam.typeRequiresMaterialization) {
          fv.materialize
        } else {
          fv
        }

        val secondValue = if (!secondArgConstantType && functionValue.secondParam.typeRequiresMaterialization) {
          sv.materialize
        } else {
          sv
        }

        val thirdValue = if (!thirdArgConstantType && functionValue.thirdParam.typeRequiresMaterialization) {
          tv.materialize
        } else {
          tv
        }

        if ((firstArgConstantType || functionValue.First.accepts(firstValue)) &&
          (secondArgConstantType || functionValue.Second.accepts(secondValue)) &&
          (thirdArgConstantType || functionValue.Third.accepts(thirdValue))) {
          return doCall(functionValue, firstValue, secondValue, thirdValue)
        }
      }

      val firstValue = if (functionValue.firstParam.typeRequiresMaterialization) {
        fv.materialize
      } else {
        fv
      }

      val secondValue = if (functionValue.secondParam.typeRequiresMaterialization) {
        sv.materialize
      } else {
        sv
      }

      val thirdValue = if (functionValue.thirdParam.typeRequiresMaterialization) {
        tv.materialize
      } else {
        tv
      }

      if (cachedCoercedDispatch.get() != null) {
        val coercedOperation = cachedCoercedDispatch.get
        //If values are literal we do not need to validate every just the first time when we load the coerced operation
        //Then is ok every time as its type will NEVER change
        if ((firstArgConstantType || coercedOperation._1.accepts(firstValue)) &&
          (secondArgConstantType || coercedOperation._2.accepts(secondValue)) &&
          (thirdArgConstantType || coercedOperation._3.accepts(thirdValue))) {
          val maybeFirstValue = if (functionValue.First.accepts(firstValue)) Some(firstValue) else functionValue.First.coerceMaybe(firstValue)
          val maybeSecondValue = if (functionValue.Second.accepts(secondValue)) Some(secondValue) else functionValue.Second.coerceMaybe(secondValue)
          val maybeThirdValue = if (functionValue.Third.accepts(thirdValue)) Some(thirdValue) else functionValue.Third.coerceMaybe(thirdValue)
          if (maybeFirstValue.isDefined && maybeSecondValue.isDefined && maybeThirdValue.isDefined) {
            return doCall(functionValue, maybeFirstValue.get, maybeSecondValue.get, maybeThirdValue.get)
          }
        }
      }

      if (functionValue.First.accepts(firstValue) &&
        functionValue.Second.accepts(secondValue) &&
        functionValue.Third.accepts(thirdValue)) {
        validated.set(true)
        doCall(functionValue, firstValue, secondValue, thirdValue)
      } else {
        //SLOW PATH
        val materializedValues: Array[Value[Any]] = ValuesHelper.array(firstValue.materialize, secondValue.materialize, thirdValue.materialize)
        val functionToCallWithCoercion: Option[(Int, Array[Value[_]], Seq[Int])] = findMatchingFunctionWithCoercion(materializedValues, Array(functionValue), this)
        functionToCallWithCoercion match {
          case Some((_, argumentsWithCoercion, paramsToCoerce)) => {
            //Cache the coercion use the base type to avoid Memory Leaks as Types may have references to Streams or Objects
            cachedCoercedDispatch.set((firstValue.valueType.baseType, secondValue.valueType.baseType, thirdValue.valueType.baseType, paramsToCoerce))
            val firstCoercedValue: Value[_] = argumentsWithCoercion(0)
            val secondCoercedValue = argumentsWithCoercion(1)
            val thirdCoercedValue = argumentsWithCoercion(2)
            doCall(functionValue, firstCoercedValue, secondCoercedValue, thirdCoercedValue)
          }
          case None =>
            throw new UnexpectedFunctionCallTypesException(location, name, materializedValues, Seq(functionValue.functionParamTypes.map(_.theType)))
        }
      }
    } finally {
      activeFrame.cleanCallSite()
    }
  }

  private def doCall(functionValue: BaseTernaryFunctionValue, firstValue: Value[_], secondValue: Value[_], thirdValue: Value[_])(implicit ctx: ExecutionContext) = {
    try {
      functionValue.call(firstValue, secondValue, thirdValue)
    } catch {
      case ex: ExecutionException =>
        if (showInStacktrace) {
          ex.addCallToStacktrace(location, name())
        }
        throw ex
    }
  }

  def target()(implicit ctx: ExecutionContext): FunctionValue = {
    node.execute.asInstanceOf[FunctionValue]
  }

  override def _1: ValueNode[_] = node

  override def _2: String = name

  override def _3: Boolean = firstArgConstantType

  override def _4: Boolean = secondArgConstantType

  override def execute(arguments: Array[Value[_]])(implicit ctx: ExecutionContext): Value[Any] = {
    executeTernary(arguments(0), arguments(1), arguments(2))
  }

  override def name()(implicit ctx: ExecutionContext): String = {
    this.name
  }

}
