package org.mule.weave.v2.runtime

import org.mule.weave.v2.core.exception.InvalidMimeTypeException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.module.MimeType

object ExecutableWeaveHelper {

  /**
    * Build readers for the given inputs.
    */
  def buildReaders(inputs: Map[String, WeaveInput])(implicit ctx: EvaluationContext): Map[String, Reader] = {
    for ((inputName, input) <- inputs) yield {
      val contentType = input.contentType
      val inputDataFormat = DataFormatManager.byContentType(contentType).getOrElse(throw new InvalidMimeTypeException(s"Invalid content type $contentType", UnknownLocation))
      val reader = inputDataFormat.reader(input.sourceProvider)
      for ((name, value) <- input.options) {
        reader.setOption(UnknownLocation, name, value)
      }
      (inputName, reader.asInstanceOf[Reader])
    }
  }

  /**
    * Looks for the given inputs within the script declared ones and creates a reader of the declared type.
    */
  def buildReaders(executableWeave: ExecutableWeave[_], inputs: Map[String, AnyRef])(implicit ctx: EvaluationContext): Map[String, Reader] = {
    val declaredInputs: Map[String, DataFormat[_, _]] = executableWeave.declaredInputs()
    inputs.map((input) => {
      val inputModule: DataFormat[_, _] = declaredInputs.getOrElse(input._1, throw new RuntimeException(s"No input directive found for ${input._1}"))
      (input._1, inputModule.reader(SourceProvider(input._2)))
    })
  }

  /**
    * Looks for the declared output within the script and creates a writer of the declared type.
    */
  def buildWriter(executableWeave: ExecutableWeave[_], output: AnyRef = null)(implicit ctx: EvaluationContext): Writer = {
    buildWriter(executableWeave, output, None)
  }

  def buildWriter(executableWeave: ExecutableWeave[_], output: AnyRef, defaultOutputType: Option[DataFormat[_, _]])(implicit ctx: EvaluationContext): Writer = {
    executableWeave
      .implicitWriterOption(Option(output))
      .orElse(defaultOutputType.map(_.writer(Option(output))))
      .getOrElse(throw new RuntimeException(s"No output directive found."))
  }

  /**
    * Build a writer of the given type.
    */
  def buildWriter(outputType: String)(implicit ctx: EvaluationContext): Writer = {
    val outputDataFormat = DataFormatManager.byContentType(outputType).getOrElse(throw new InvalidMimeTypeException(s"Invalid content type $outputType", UnknownLocation))
    outputDataFormat.writer(None, MimeType.fromSimpleString(outputType))
  }

}

case class WeaveInput(name: String, contentType: String, sourceProvider: SourceProvider, options: Map[String, Any] = Map()) {
  def toTuple: (String, WeaveInput) = (name, this)
}

object WeaveInput {
  implicit def toTuple(input: WeaveInput): (String, WeaveInput) = input.toTuple
}
