package org.mule.weave.v2.interpreted

import org.mule.weave.v2.interpreted.exception.WeaveStackOverflowException
import org.mule.weave.v2.interpreted.listener.NotificationManager
import org.mule.weave.v2.interpreted.node.Module
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBindings
import org.mule.weave.v2.interpreted.node.structure.header.VariableTable
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.capabilities.DelegateLocationCapable
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.reader.ResourceManager
import org.mule.weave.v2.module.writer.EmptyWriter
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.ast.WeaveLocationCapable
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.LocationCapable

/**
  * Module information
  */
class ModuleContext(val variableTable: VariableTable, val modulesTable: VariableTable, val nameIdentifier: NameIdentifier) {
  def getVariableSlot(name: String): Int = variableTable.indexOf(name)

  def getVariableName(slotIndex: Int): String = variableTable.apply(slotIndex).name

  val modules = new Array[Module](modulesTable.size)

  def updateModule(moduleSlot: Int, module: Module): Unit = {
    modules.update(moduleSlot, module)
  }
}

object ModuleContext {
  def apply(variableTable: VariableTable, modulesTable: VariableTable, nameIdentifier: NameIdentifier): ModuleContext =
    new ModuleContext(variableTable, modulesTable, nameIdentifier)
}

/**
  * A Virtual Frame representing the variables that are currently visible
  */
class Frame(val content: Array[Value[_]], val moduleContext: ModuleContext, val delegate: LocationCapable, val name: Option[String]) extends DelegateLocationCapable {

  private var callSite: WeaveLocationCapable = _

  def cleanCallSite(): Unit = {
    this.callSite = null
  }

  def updateVariable(slot: Int, value: Value[_]): Unit = {
    content.update(slot, value)
  }

  def updateCallSite(callSite: WeaveLocationCapable) = {
    this.callSite = callSite
  }

  def currentCallSite(): Option[WeaveLocationCapable] = {
    Option(callSite)
  }

  def variableAt(slot: Int): Value[_] = {
    content(slot)
  }

  def child(locationCapable: LocationCapable): Frame = {
    val content = this.content.clone()
    Frame(content, moduleContext, locationCapable, None)
  }

  def child(locationCapable: LocationCapable, name: Option[String]): Frame = {
    val content = this.content.clone()
    Frame(content, moduleContext, locationCapable, name)
  }

  def child(variables: Array[(Int, Value[_])], locationCapable: LocationCapable, name: Option[String]): Frame = {
    val content = this.content.clone()
    val frame = Frame(content, moduleContext, locationCapable, name)
    frame.updateAll(variables)
  }

  private def updateAll(variables: Array[(Int, Value[_])]): Frame = {
    var i = 0
    while (i < variables.length) {
      val v = variables(i)
      content.update(v._1, v._2)
      i = i + 1
    }
    this
  }

  def variableAt(moduleSlot: Int, slot: Int): Value[_] = {
    val module = moduleContext.modules(moduleSlot)
    if (module != null) {
      module.getVariable(slot)
    } else {
      null
    }
  }

  def updateModule(moduleSlot: Int, module: Module): Unit = {
    moduleContext.updateModule(moduleSlot, module)
  }

  override def clone(): Frame = {
    Frame(content.clone(), moduleContext, delegate, name)
  }

}

object Frame {
  def apply(content: Array[Value[_]], moduleContext: ModuleContext, delegate: LocationCapable, name: Option[String]): Frame = {
    //Every instance of a frame contains
    new Frame(content, moduleContext, delegate, name)
  }
}

/**
  * Represents the Interpreted mode execution context.
  * This is the context to what all the ValueNodes will have access to
  */
trait ExecutionContext extends EvaluationContext {

  /**
    * Return if values should be materialized in each step
    *
    * @return If true the engine will materialize the values
    */
  def materializeValues(): Boolean

  /**
    * The writer
    *
    * @return
    */
  def writer: Option[Writer]

  /**
    * Returns the current execution thread stack
    *
    * @return
    */
  def executionStack(): ExecutionThreadStack

  /**
    * Returns the engine notification manager
    *
    * @return The notification manager
    */
  def notificationManager(): NotificationManager

  /**
    * Runs the callback under the specified frame
    *
    * @param frame    The frame to be used
    * @param callback The callback
    * @tparam T The return type
    * @return The result
    */
  def runInFrame[T](frame: Frame, callback: => T): T = {
    if (executionStack().activeFrame() eq frame) {
      callback
    } else {
      executionStack().pushFrame(frame)
      try {
        callback
      } finally {
        executionStack().dropFrame()
      }
    }
  }

  /**
    * Runs the callback under the specified frame
    *
    * @param callback The callback
    * @tparam T The return type
    * @return The result
    */
  def runInNewFrame[T](locationCapable: LocationCapable, name: Option[String], callback: => T): T = {
    executionStack().pushFrame(executionStack().activeFrame().child(locationCapable, name))
    try {
      callback
    } finally {
      executionStack().dropFrame()
    }
  }

  /**
    * Runs the callback under the specified frame
    *
    * @param callback The callback
    * @tparam T The return type
    * @return The result
    */
  def runInNewFrame[T](locationCapable: LocationCapable, callback: => T): T = {
    executionStack().pushFrame(executionStack().activeFrame().child(locationCapable))
    try {
      callback
    } finally {
      executionStack().dropFrame()
    }
  }

  /**
    * Spawn a new thread for this execution context
    *
    * @return
    */
  def spawnNewThread(): ExecutionContext = {
    val frame = executionStack().activeFrame().clone()
    ExecutionContext(
      writer,
      materializeValues(),
      ExecutionThreadStack(frame, executionStack().maxExecutionStack),
      notificationManager(),
      EvaluationContext(ServiceManager.withResourceManager(ResourceManager(), serviceManager)))
  }
}

/**
  * Represents the execution thread stack
  */
trait ExecutionThreadStack {

  /**
    * The max amount of elements in the stack
    * @return
    */
  def maxExecutionStack: Int

  /**
    * Returns the name of this execution thread stack
    *
    * @return
    */
  def name(): String

  /**
    * Updates the module at the given slot
    *
    * @param slot   The slot
    * @param module The module
    */
  def setModule(slot: Int, module: Module): Unit

  /**
    * Sets the value at a given slot in the active frame
    *
    * @param slot  The variable slot
    * @param value The value
    */
  def setVariable(slot: Int, value: Value[_]): Unit

  /**
    * Return the active frame
    *
    * @return The active frame
    */
  def activeFrame(): Frame

  /**
    * Returns the variable at the specified module
    *
    * @param moduleSlot   The module slot
    * @param variableSlot The variable slot
    * @return The variable or null if not present
    */
  def getVariable(moduleSlot: Int, variableSlot: Int): Value[_]

  /**
    * Returns the variable at the specifed slot or null if not present
    *
    * @param slot The slot
    * @return
    */
  def getVariable(slot: Int): Value[_]

  /**
    * Return the active frame
    *
    * @return The active frame
    */
  def frames(): Array[Frame]

  /**
    * Push a new frame in the execution stack
    *
    * @param frame The frame
    */
  def pushFrame(frame: Frame): Unit

  /**
    * Drops current frame
    */
  def dropFrame(): Unit
}

class DefaultExecutionThreadStack(val thread: Thread, initialFrame: Frame, val maxExecutionStack: Int) extends ExecutionThreadStack {

  private var stack: Array[Frame] = new Array[Frame](Math.max((maxExecutionStack / 16), Math.min(maxExecutionStack, 8)))
  private var index: Int = -1
  pushFrame(initialFrame)

  override def setModule(slot: Int, module: Module): Unit = {
    activeFrame().updateModule(slot, module)
  }

  override def getVariable(moduleSlot: Int, slot: Int): Value[_] = {
    activeFrame().variableAt(moduleSlot, slot)
  }

  override def getVariable(slot: Int): Value[_] = {
    activeFrame().variableAt(slot)
  }

  override def setVariable(slot: Int, value: Value[_]): Unit = {
    activeFrame().updateVariable(slot, value)
  }

  override def activeFrame(): Frame = {
    stack(index)
  }

  override def frames(): Array[Frame] = {
    stack.slice(0, index + 1)
  }

  override def pushFrame(frame: Frame): Unit = {
    if (index < stack.length - 1) {
      index = index + 1
      stack.update(index, frame)
    } else if (index < maxExecutionStack - 1) {
      index = index + 1
      val tmpStack = new Array[Frame](stack.length * 2)
      System.arraycopy(stack, 0, tmpStack, 0, stack.length)
      stack = tmpStack
      stack.update(index, frame)
    } else {
      throw new WeaveStackOverflowException(frame, maxExecutionStack)
    }
  }

  override def dropFrame(): Unit = {
    index = index - 1
  }

  override def name(): String = thread.getName
}

object ExecutionThreadStack {

  def apply(initialFrame: Frame, maxExecutionStack: Int) = new DefaultExecutionThreadStack(Thread.currentThread(), initialFrame, maxExecutionStack)

  def apply(content: Array[Value[_]], moduleContext: ModuleContext, delegate: LocationCapable, name: Option[String], maxExecutionStack: Int): DefaultExecutionThreadStack = {
    new DefaultExecutionThreadStack(Thread.currentThread(), Frame(content, moduleContext, delegate, name), maxExecutionStack)
  }
}

/**
  * Default Execution Context Implementation
  */
class DefaultExecutionContext(val writer: Option[Writer], val materializeValues: Boolean, val executionStack: ExecutionThreadStack, val notificationManager: NotificationManager, val evalCtx: EvaluationContext) extends ExecutionContext {

  override def initialNestedExecutionCount(): Int = {
    evalCtx.initialNestedExecutionCount()
  }

  override def newAsyncExecution(): Unit = {
    evalCtx.newAsyncExecution()
  }

  override def asyncExecutionCount(): Int = {
    evalCtx.asyncExecutionCount()
  }

  override def serviceManager: ServiceManager = {
    evalCtx.serviceManager
  }

  override def close(): Unit = {
    evalCtx.close()
  }
}

object ExecutionContext {

  val INIT_METHOD_NAME = "init"

  def apply(writer: Option[Writer], materializeValues: Boolean, executionStack: ExecutionThreadStack, notificationManager: NotificationManager, evalCtx: EvaluationContext): ExecutionContext = {
    new DefaultExecutionContext(writer, materializeValues, executionStack, notificationManager, evalCtx)
  }

  def apply(
    writer: Writer = EmptyWriter,
    variableTable: VariableTable = new VariableTable(),
    moduleTable: VariableTable = new VariableTable(),
    externalBindings: ExternalBindings = new ExternalBindings(),
    values: Map[String, Value[_]] = Map.empty,
    notificationManager: NotificationManager = NotificationManager.EMPTY,
    evaluationContext: EvaluationContext = EvaluationContext(),
    materializedValue: Boolean = false,
    location: LocationCapable = UnknownLocationCapable,
    resource: NameIdentifier = NameIdentifier.anonymous): ExecutionContext = {

    val stackSize = evaluationContext.serviceManager.settingsService.execution().stackSize

    val threadStack = ExecutionThreadStack(new Array[Value[_]](variableTable.size), ModuleContext(variableTable, moduleTable, resource), location, Some(INIT_METHOD_NAME), stackSize)
    //Initialize the frame
    var i = 0
    val variables = externalBindings.variables()
    while (i < variables.size) {
      val binding = variables(i)
      val value = values.get(binding.name.name)
      value match {
        case Some(theValue) => {
          threadStack.setVariable(binding.name.slot, if (binding.needsMaterialize) theValue.materialize(evaluationContext) else theValue)
          theValue match {
            case closeable: AutoCloseable => evaluationContext.serviceManager.resourceManager.registerCloseable(closeable)
            case _                        =>
          }
        }
        case None =>
      }
      i = i + 1
    }
    new DefaultExecutionContext(Some(writer), materializedValue, threadStack, notificationManager, evaluationContext)
  }

  def apply(frame: Frame, evaluationContext: EvaluationContext): ExecutionContext = {
    val stackSize = evaluationContext.serviceManager.settingsService.execution().stackSize
    new DefaultExecutionContext(None, false, ExecutionThreadStack(frame, stackSize), NotificationManager.EMPTY, evaluationContext)
  }
}

object EmptyExecutionContext {
  def apply(): ExecutionContext = {
    ExecutionContext()
  }
}
