package org.mule.weave.v2.interpreted.debugger.server

import java.io.PrintWriter
import java.io.StringWriter

import org.mule.weave.v2.debugger.DebuggerFrame
import org.mule.weave.v2.debugger.commands.DebuggerCommand
import org.mule.weave.v2.debugger.event.DebuggerEvent
import org.mule.weave.v2.debugger.event.OnFrameEvent
import org.mule.weave.v2.debugger.event.UnexpectedServerErrorEvent
import org.mule.weave.v2.interpreted.debugger.server.DebuggerConverters.toDebuggerPosition
import org.mule.weave.v2.interpreted.debugger.server.tcp.TcpServerProtocol
import org.mule.weave.v2.interpreted.listener.SessionListener
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.WeaveLocation

import scala.collection.mutable.ListBuffer
import scala.util.Failure
import scala.util.Success
import scala.util.Try

class DefaultWeaveDebuggingSession(protocol: ServerProtocol = TcpServerProtocol()) extends WeaveDebuggingSession {

  var debuggerExecutor: WeaveDebuggerExecutor = _
  @volatile
  var active: Boolean = false
  val breakpointManager: WeaveBreakpointManager = new DefaultWeaveBreakpointManager(this)
  val sessionListener: ListBuffer[SessionListener] = new ListBuffer[SessionListener]
  var initialized: Boolean = false

  override def initSession(): Unit = {
    sessionListener.foreach((listener) => {
      listener.onSessionInitialized()
    })
    initialized = true
  }

  def addSessionListener(listener: SessionListener): Unit = {
    sessionListener += listener
    if (initialized) {
      listener.onSessionInitialized()
    }
  }

  override def onExecutionPaused(toArray: Array[DebuggerFrame], weaveLocation: WeaveLocation, reason: Int): Unit = {
    val startPosition: Position = weaveLocation.startPosition
    val endPosition: Position = weaveLocation.endPosition
    send(OnFrameEvent(toArray, toDebuggerPosition(weaveLocation.resourceName, startPosition), toDebuggerPosition(weaveLocation.resourceName, endPosition), reason))
  }

  def send(event: DebuggerEvent): Unit = {
    if (started()) {
      protocol.send(event)
    }
  }

  def started(): Boolean = active

  def start(debuggerContext: WeaveDebuggerExecutor): Unit = {
    this.debuggerExecutor = debuggerContext
    this.active = true
    this.protocol.addCommandHandler(classOf[DebuggerCommand], new CommandHandler[DebuggerCommand] {
      override def handle(command: DebuggerCommand): Unit = {
        val interpreter = new DefaultWeaveDebuggerCommandInterpreter(DefaultWeaveDebuggingSession.this)
        val maybeEventResult: Try[Option[DebuggerEvent]] = Try({
          command.call(interpreter)
        })
        maybeEventResult match {
          case Failure(exception) => {
            println("Unexpected error : " + exception)
            exception.printStackTrace()
            val stringWriter = new StringWriter()
            exception.printStackTrace(new PrintWriter(stringWriter))
            protocol.send(UnexpectedServerErrorEvent(stringWriter.toString))
          }
          case Success(maybeEvent) => {
            maybeEvent match {
              case Some(response) => {
                //Correlate the response with the request
                response.commandId = Some(command.id)
                protocol.send(response)
              }
              case None =>
            }
          }
        }
      }
    })
    if (!this.protocol.isStarted()) {
      this.protocol.start()
    }
  }

  def stop(): Unit = {
    protocol.disconnect()
    this.active = false
    //make sure we resume any message
    this.debuggerExecutor.resume()
  }

  override def getWeaveBreakpointManager(): WeaveBreakpointManager = {
    breakpointManager
  }

  override def getWeaveDebuggerExecutor(): WeaveDebuggerExecutor = {
    debuggerExecutor
  }
}
