package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.interpreted.marker.ReferenceAnnotation
import org.mule.weave.v2.interpreted.marker.ReferenceValue
import org.mule.weave.v2.interpreted.node.NameSlot
import org.mule.weave.v2.interpreted.node.VariableReferenceNode
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBindings
import org.mule.weave.v2.interpreted.node.structure.header.VariableTable
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.ParseException

import scala.collection.mutable.ArrayBuffer

trait EngineVariableTransformations extends AstTransformation with EngineDirectiveTransformations {

  val _variablesTable: VariableTable

  val _modulesNameTable: VariableTable

  val _externalBindings: ExternalBindings

  val _modules: ArrayBuffer[VariableTable]

  def createNameSlot(name: String): NameSlot = {
    _variablesTable.createVariable(name)
  }

  def createModuleSlot(name: String): NameSlot = {
    _modulesNameTable.createVariable(name)
  }

  def transformVariableReferenceNode(variable: org.mule.weave.v2.parser.ast.variables.VariableReferenceNode): VariableReferenceNode = {
    val maybeSlot = transformReference(variable.variable)
    if (maybeSlot.isDefined) {
      VariableReferenceNode(maybeSlot.get)
    } else {
      throw new ParseException(s"Unable to resolve variable ${variable.variable.name}", variable.location())
    }
  }

  override def transformReference(reference: NameIdentifier): Option[NameSlot] = {
    reference.annotation(classOf[ReferenceAnnotation])
      .flatMap(annotation =>
        transformReferenceValue(annotation.referenceValue))
  }

  def transformReferenceValue(reference: ReferenceValue): Option[NameSlot] = {
    reference.moduleFQN match {
      case None =>
        val nameSlot = createNameSlot(reference.localVariableName)
        Some(nameSlot)
      case Some(moduleFQN) =>
        val moduleSlot = createModuleSlot(moduleFQN.name)
        if (moduleSlot.slot < _modules.length) {
          val nameSlot = _modules(moduleSlot.slot)
            .createVariable(reference.localVariableName)
            .fqn(moduleSlot)
          Some(nameSlot)
        } else {
          throw new RuntimeException(s"Module for ${reference.localVariableName} was not defined.")
        }
    }
  }

  def transformNameSlot(name: NameIdentifier): NameSlot = {
    createNameSlot(name.name)
  }

}
