package org.mule.weave.v2.runtime.utils

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.OutputDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResource

@WeaveApi(Seq("Tooling"))
class WeaveRequirementsChecker(ctx: ParsingContext) {

  def this() = {
    this(ParsingContextFactory.createParsingContext(NameIdentifier.anonymous))
  }

  def requiresJava(script: String): Boolean = {
    val resource = WeaveResource.anonymous(script)

    val phaseResult = MappingParser.parse(MappingParser.parsingPhase(), resource, ctx)
    if (phaseResult.hasResult()) {
      val astNode = phaseResult.getResult().astNode
      val directives = astNode.header.directives
      if (directives.exists(directiveRequiresJava)) {
        true
      } else {
        val nameIdentifiers = AstNodeHelper.collectChildrenWith(astNode, classOf[NameIdentifier])
        nameIdentifiers.exists((nameIdentifier) => {
          nameIdentifier.loader.exists(_.equals("java")) || (nameIdentifier.nameElements().length > 1 && !nameIdentifier.name.startsWith("dw" + NameIdentifier.SEPARATOR))
        })
      }
    } else {
      //If doesn't compile requires java
      false
    }
  }

  private def directiveRequiresJava(dir: DirectiveNode) = {
    dir match {
      case importNode: ImportDirective =>
        val element = importNode.importedModule
        val name = element.elementName.name
        !name.startsWith("dw" + NameIdentifier.SEPARATOR) || element.elementName.loader.exists(_.equals("java"))
      case outputNode: OutputDirective =>
        val maybeMime = outputNode.mime
        if (maybeMime.isDefined) {
          val mimeType = MimeType.fromSimpleString(maybeMime.get.mime)
          MimeType.fromSimpleString("application/java").includes(mimeType)
        } else {
          "java".equals(outputNode.dataFormat.map(_.id).orNull)
        }
      case _ =>
        false
    }
  }

}
