package org.mule.weave.v2.interpreted.debugger.server.tcp

import org.mule.weave.v2.debugger.client.tcp.TcpClientProtocol
import org.mule.weave.v2.debugger.commands.ClientCommand
import org.mule.weave.v2.debugger.event.RemoteServerMessage
import org.mule.weave.v2.interpreted.debugger.server.ClientConnectionListener
import org.mule.weave.v2.interpreted.debugger.server.CommandHandler
import org.mule.weave.v2.interpreted.debugger.server.ServerProtocol

import java.io.EOFException
import java.io.IOException
import java.io.ObjectInputStream
import java.io.ObjectOutputStream
import java.net.InetAddress
import java.net.ServerSocket
import java.net.Socket
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.util.Try

class TcpServerProtocol(val serverPort: Int = TcpServerProtocol.DEFAULT_PORT) extends ServerProtocol {

  val handlers = ArrayBuffer[(Class[_], CommandHandler[_])]()

  var serverSocket: ServerSocket = _
  var clientSocket: Socket = _
  @volatile var isStopped: Boolean = true
  var listener: mutable.Seq[ClientConnectionListener] = mutable.Seq()

  override def addClientConnectionListener(clientConnectionListener: ClientConnectionListener): Unit = {
    listener = listener.:+(clientConnectionListener)
  }

  override def send(event: RemoteServerMessage): Unit = {
    synchronized({
      try {
        if (clientSocket != null) {
          val outputStream: ObjectOutputStream = new ObjectOutputStream(clientSocket.getOutputStream)
          outputStream.reset()
          outputStream.writeObject(event)
          outputStream.flush()
        }
      } catch {
        case io: IOException => {
          listener.foreach(_.onConnectionError(io))
        }
      }
    })
  }

  def waitForCommand(): Option[ClientCommand[_, _]] = {
    try {
      if (clientSocket != null) {
        val readObject: Any = new ObjectInputStream(clientSocket.getInputStream).readObject()
        readObject match {
          case readObject: ClientCommand[_, _] => Some(readObject)
          case _                               => None
        }
      } else {
        None
      }
    } catch {
      case eof: EOFException => {
        disconnect()
        None
      }
      case e: IOException => {
        listener.foreach(_.onConnectionError(e))
        None
      }
    }
  }

  override def isStarted(): Boolean = {
    !isStopped
  }

  override def disconnect(): Unit = {
    synchronized {
      isStopped = true
      if (clientSocket != null) {
        try {
          listener.foreach((listener) => listener.onClientDisconnected())
          clientSocket.close()
        } catch {
          case e: IOException =>
        }
        clientSocket = null
      }
      if (serverSocket != null) {
        try {
          serverSocket.close()
        } catch {
          case e: IOException => {
            listener.foreach(_.onConnectionError(e))
          }
        }
        serverSocket = null
      }
    }
  }

  def dispatchCommand(command: ClientCommand[_, _]): Unit = {
    handlers
      .find((pairs) => {
        pairs._1.isAssignableFrom(command.getClass)
      })
      .foreach((pair) => {
        val value = pair._2
        Try(value.handle(command.asInstanceOf[value.CommandType]))
      })
  }

  override def start(daemon: Boolean = true): Unit = {
    createServerSocket()
    isStopped = false
    val runner: Thread = new Thread(
      () => {
        while (!isStopped) {
          try {
            println(s"[dw-debugger] Starting debugger at: `${serverPort}`")
            clientSocket = serverSocket.accept()
            if (clientSocket != null) {
              val address: InetAddress = clientSocket.getInetAddress
              if (address != null) {
                println(s"[dw-debugger] Client connected from: `${address.getHostAddress}`")
              }
            }
            listener.foreach((listener) => listener.onClientConnected())
            startListeningForCommands()
          } catch {
            case e: IOException => {
              listener.foreach(_.onConnectionError(e))
              disconnect()
            }
          }
        }
      },
      "Debugger Server Poll")
    runner.setDaemon(daemon)
    runner.start()
  }

  private def startListeningForCommands(): Unit = {
    while (!isStopped) {
      val command = waitForCommand()
      command match {
        case Some(value) => dispatchCommand(value)
        case None        =>
      }

    }
  }

  private def createServerSocket() {
    try {
      this.serverSocket = new ServerSocket(this.serverPort)
    } catch {
      case e: IOException => {
        throw new RuntimeException("Cannot open port " + serverPort, e)
      }
    }
  }

  override def addCommandHandler[T <: ClientCommand[_, _]](clazz: Class[T], handler: CommandHandler[T]): Unit = {
    handlers.+=((clazz, handler))
  }
}

object TcpServerProtocol {
  val DEFAULT_PORT: Int = TcpClientProtocol.DEFAULT_PORT

  def apply(serverPort: Int = DEFAULT_PORT) = new TcpServerProtocol(serverPort)
}
