package org.mule.weave.v2.interpreted.extension

import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.core.util.ObjectValueUtils
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.service.CharsetProviderService
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.MaterializedBinaryValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValuesHelper
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.option.Settings
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.reader.SourceProviderAwareReader
import org.mule.weave.v2.module.writer.DeferredWriter
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.module.MimeType

import java.io.OutputStream
import java.nio.charset.Charset

class WeaveBasedDataFormatProxy(delegate: Value[ObjectSeq], val readerSettingsFactory: () => WeaveBasedDataFormatSettings, writerSettingsFactory: () => WeaveBasedDataFormatWriterSettings,
  formatName: String,
  mimeTypesValue: Seq[MimeType],
  fileExtensionsValue: Seq[String],
  labelValue: Option[String],
  binaryDataFormatValue: Boolean,
  defaultCharsetValue: Option[Charset])
    extends DataFormat[WeaveBasedDataFormatSettings, WeaveBasedDataFormatWriterSettings] {

  def readerValue(implicit ctx: EvaluationContext): Array[Value[_]] => Value[_] = ObjectValueUtils
    .selectFunction(delegate.evaluate, "reader")
    .getOrElse(throw new WeaveRuntimeException("Missing field `reader` with function value was found.", delegate.location().asInstanceOf[WeaveLocation]))

  def writerValue(implicit ctx: EvaluationContext): Array[Value[_]] => Value[_] = ObjectValueUtils
    .selectFunction(delegate.evaluate, "writer")
    .getOrElse(throw new WeaveRuntimeException("Missing field `writer` with function value was found.", delegate.location().asInstanceOf[WeaveLocation]))

  override def defaultCharset: Option[CharsetProviderService#Charset] = defaultCharsetValue

  override def binaryFormat: Boolean = binaryDataFormatValue

  override def name(): String = formatName

  override def label(): String = labelValue.getOrElse(super.label())

  override val defaultMimeType: MimeType = mimeTypesValue.head

  override val acceptedMimeTypes: Seq[MimeType] = mimeTypesValue

  override def readerSettings(): WeaveBasedDataFormatSettings = readerSettingsFactory()

  override def writerSettings(): WeaveBasedDataFormatWriterSettings = writerSettingsFactory()

  override def fileExtensions: Seq[String] = fileExtensionsValue

  override def reader(source: SourceProvider)(implicit readerContext: EvaluationContext): Reader = {
    new WeaveBasedReaderProxy(readerValue, source, createReaderSettings(), readerContext, name(), this)
  }

  override def writer(target: Option[Any], outputMimeType: MimeType)(implicit ctx: EvaluationContext): Writer = {
    DeferredWriter(
      (tp: TargetProvider, settings: WeaveBasedDataFormatWriterSettings) => new WeaveBasedWriterProxy(writerValue, settings, tp, name(), this),
      TargetProvider(target),
      createWriterSettings())
  }
}

class WeaveBasedReaderProxy(callback: Array[Value[_]] => Value[_], override val sourceProvider: SourceProvider, readerSettings: WeaveBasedDataFormatSettings, ctx: EvaluationContext, name: String, df: WeaveBasedDataFormatProxy) extends Reader with SourceProviderAwareReader {

  override def dataFormat: Option[DataFormat[_, _]] = Some(df)

  override protected def doRead(name: String): Value[_] = {
    callback(ValuesHelper.array(BinaryValue.streaming(sourceProvider.asInputStream(ctx))(ctx), StringValue(sourceProvider.charset.name()), readerSettings.toObjectValue()))
  }

  override val settings: Settings = readerSettings

  override def getName(): String = name
}

class WeaveBasedWriterProxy(callback: Array[Value[_]] => Value[_], writerSettings: WeaveBasedDataFormatWriterSettings, tp: TargetProvider, name: String, df: WeaveBasedDataFormatProxy) extends Writer {

  var resultStream: OutputStream = _

  override val settings: Settings = writerSettings

  protected override def doWriteValue(value: Value[_])(implicit ctx: EvaluationContext): Unit = {
    val functionCallResult: Value[_] = callback(ValuesHelper.array(value, writerSettings.toObjectValue()))
    val writerResult: SeekableStream = functionCallResult match {
      case m: MaterializedBinaryValue => m.underlyingStream()
      case b                          => b.evaluate.asInstanceOf[SeekableStream]
    }
    resultStream = writerResult.copyTo(Some(tp.asOutputStream))
  }

  override def result: Any = resultStream

  override def close(): Unit = {}

  override def getName(): String = {
    name
  }

  override def dataFormat: Option[DataFormat[_, _]] = Some(df)
}
