package org.mule.weave.v2.interpreted.profiler

import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.END_POSITION
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.EXCEPTION_PROPERTY
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.NAME_PROPERTY
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.NODE_PROPERTY
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.POST_EXECUTION
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.PRE_EXECUTION
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.START_POSITION
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.data
import org.mule.weave.v2.core.telemetry.service.api.TelemetryService
import org.mule.weave.v2.grammar.AsOpId
import org.mule.weave.v2.grammar.DescendantsSelectorOpId
import org.mule.weave.v2.grammar.FilterSelectorOpId
import org.mule.weave.v2.grammar.IsOpId
import org.mule.weave.v2.grammar.LeftShiftOpId
import org.mule.weave.v2.grammar.RangeSelectorOpId
import org.mule.weave.v2.grammar.RightShiftOpId
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.listener.WeaveExecutionListener
import org.mule.weave.v2.interpreted.node.BinaryOpNode
import org.mule.weave.v2.interpreted.node.ChainedBinaryOpNode
import org.mule.weave.v2.interpreted.node.FunctionCallNode
import org.mule.weave.v2.interpreted.node.UnaryOpNode
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.pattern.PatternMatcherNode
import org.mule.weave.v2.interpreted.node.pattern.PatternNode
import org.mule.weave.v2.interpreted.node.structure.DocumentNode
import org.mule.weave.v2.interpreted.node.updater.DynamicUpdaterValueNode
import org.mule.weave.v2.interpreted.node.updater.StaticUpdaterValueNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.runtime.ExecutableWeave

class ExecutionTelemetryListener extends WeaveExecutionListener {

  def isTelemetryAwareNode(node: ValueNode[_])(implicit ctx: ExecutionContext): Boolean = {
    node match {
      case _: FunctionCallNode        => true
      case _: DocumentNode            => true
      case con: ChainedBinaryOpNode   => con.operations.exists((n) => shouldMeasureOp(n.name()))
      case _: DynamicUpdaterValueNode => true
      case _: StaticUpdaterValueNode  => true
      case _: PatternMatcherNode      => true
      case _: PatternNode             => true
      case uon: UnaryOpNode =>
        //We only do telemetry on slow ops
        uon.operationName == DescendantsSelectorOpId.name
      case uon: BinaryOpNode =>
        //We only do telemetry on slow ops
        shouldMeasureOp(uon.operationName)
      case _ => false
    }
  }

  private def shouldMeasureOp(operationName: String): Boolean = {
    operationName == FilterSelectorOpId.name ||
      operationName == RangeSelectorOpId.name ||
      operationName == IsOpId.name ||
      operationName == RightShiftOpId.name ||
      operationName == LeftShiftOpId.name ||
      operationName == AsOpId.name
  }

  def nodeName(node: ValueNode[_])(implicit ctx: ExecutionContext): String = {
    node match {
      case fcn: FunctionCallNode      => fcn.functionName()
      case _: DocumentNode            => "main"
      case bon: BinaryOpNode          => bon.operationName
      case uon: UnaryOpNode           => uon.operationName
      case _: PatternMatcherNode      => "match"
      case _: DynamicUpdaterValueNode => "update"
      case _: StaticUpdaterValueNode  => "update"
      case con: ChainedBinaryOpNode   => con.operations.filter((n) => shouldMeasureOp(n.name())).head.name()
      case _                          => ""
    }
  }

  override def preExecution(node: ValueNode[_])(implicit ctx: ExecutionContext): Unit = {
    val eventHandler: Option[TelemetryService] = getEventHandlerService(ctx)
    if (eventHandler.isDefined && isTelemetryAwareNode(node)) {
      eventHandler.get.publishEvent(
        PRE_EXECUTION,
        node.location(),
        System.identityHashCode(node).toString,
        data(
          NODE_PROPERTY,
          node.getClass.getSimpleName,
          START_POSITION,
          node.location().startPosition.line + ":" + node.location().startPosition.column,
          END_POSITION,
          node.location().endPosition.line + ":" + node.location().endPosition.column,
          NAME_PROPERTY,
          nodeName(node)))
    }
  }

  override def postExecution(node: ValueNode[_], result: Value[_])(implicit ctx: ExecutionContext): Unit = {
    val eventHandler: Option[TelemetryService] = getEventHandlerService(ctx)
    if (eventHandler.isDefined && isTelemetryAwareNode(node)) {
      eventHandler.get
        .publishEvent(POST_EXECUTION, node.location(), System.identityHashCode(node).toString, Array.empty)
    }
  }

  override def postExecution(node: ValueNode[_], e: Exception)(implicit ctx: ExecutionContext): Unit = {
    val eventHandler: Option[TelemetryService] = getEventHandlerService(ctx)
    if (eventHandler.isDefined && isTelemetryAwareNode(node)) {
      eventHandler.get.publishEvent(POST_EXECUTION, node.location(), System.identityHashCode(node).toString, data(EXCEPTION_PROPERTY, e.getClass.getName + ": " + e.getMessage))
    }
  }

  /**
    * When the execution ended
    */
  override def onExecutionEnded(executableWeave: ExecutableWeave[_ <: AstNode])(implicit ctx: EvaluationContext): Unit = {
    getEventHandlerService(ctx).foreach((eh) => eh.flush())
  }

  private def getEventHandlerService(ctx: EvaluationContext): Option[TelemetryService] = {
    ctx.serviceManager.telemetryService
  }

}
