package org.mule.weave.v2.telemetry.service

import com.lmax.disruptor.EventFactory
import com.lmax.disruptor.EventHandler
import com.lmax.disruptor.RingBuffer
import com.lmax.disruptor.dsl.Disruptor
import org.mule.weave.v2.core.telemetry.service.api.TelemetryService
import org.mule.weave.v2.core.util.IntervalExecutor
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.ServiceRegistration
import org.mule.weave.v2.model.service.RuntimeSettings.prop
import org.mule.weave.v2.module.core.json.JsonDataFormat
import org.mule.weave.v2.module.core.json.writer.JsonWriter
import org.mule.weave.v2.module.core.json.writer.JsonWriterSettings
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.WeaveLocation

import java.io.File
import java.io.FileOutputStream
import java.util.concurrent.Executors
import java.util.concurrent.ThreadFactory
import java.util.concurrent.TimeUnit
import scala.collection.mutable
import scala.util.Try

class DefaultTelemetryService extends TelemetryService {

  private var disruptor: Disruptor[DisruptorTelemetryEvent] = _
  private var myRingBuffer: RingBuffer[DisruptorTelemetryEvent] = _
  private var eventWriter: TelemetryEventWriter = _
  private val bufferFullLogger: IntervalExecutor = new IntervalExecutor(15, TimeUnit.MINUTES)

  @volatile
  private var initialized: Boolean = false

  def ringBuffer(): RingBuffer[DisruptorTelemetryEvent] = this.myRingBuffer

  def isInitialized(): Boolean = initialized

  def initIfRequired()(implicit ctx: EvaluationContext): Unit = {
    if (!initialized) {
      synchronized {
        if (!initialized) {
          val serviceManager: ServiceManager = ctx.serviceManager
          val threadFactory: ThreadFactory = Executors.defaultThreadFactory()
          val factory: DisruptorTelemetryEventFactory = new DisruptorTelemetryEventFactory()
          val settings = ctx.serviceManager.settingsService
          val bufferSize: Int = settings.telemetry().bufferSize
          disruptor = new Disruptor[DisruptorTelemetryEvent](factory, bufferSize, threadFactory)
          eventWriter = new FileTelemetryEventWriter(serviceManager.workingDirectoryService.telemetryDirectory())
          disruptor.handleEventsWith(new TelemetryEventHandler(eventWriter))
          disruptor.start()
          myRingBuffer = disruptor.getRingBuffer()
          initialized = true
        }
      }
    }
  }

  override def publishEvent(kind: String, location: Location, id: String, data: Array[String] = Array.empty)(implicit ctx: EvaluationContext): Unit = {
    if (ctx.serviceManager.settingsService.telemetry().enabled) {
      val threadId = Thread.currentThread().getName
      val timeStamp = System.nanoTime()
      initIfRequired()
      val locationString = location match {
        case weaveLocation: WeaveLocation => weaveLocation.resourceWithLocation()
        case _                            => "Unknown"
      }
      if (ctx.serviceManager.settingsService.telemetry().sync) {
        eventWriter.write(kind, locationString, threadId, id, timeStamp, data)
      } else {
        val tryPublishEvent: Boolean = myRingBuffer.tryPublishEvent((profilerEvent, sequence: Long) => {
          profilerEvent.kind = kind
          profilerEvent.location = locationString
          profilerEvent.threadId = threadId
          profilerEvent.id = id
          profilerEvent.timeStamp = timeStamp
          profilerEvent.data = data
        })
        if (!tryPublishEvent) {
          val loggingService = ctx.serviceManager.loggingService
          if (loggingService.isInfoEnabled()) {
            bufferFullLogger.trigger(() => {
              loggingService.logInfo(s"Ignoring event as the ring bugger is full. Consider increasing the buffer size with ${prop("telemetry.bufferSize")} ")
            })
          }
        }
      }
    }

  }

  override def flush(): Unit = {
    if (eventWriter != null) {
      disruptor.shutdown()
      eventWriter.flush()
    }
  }

  override def close(): Unit = {
    if (eventWriter != null) {
      //Put a timeout for shutdown to avoid any hang
      Try(
        disruptor.shutdown(10, TimeUnit.SECONDS))
      eventWriter.close()
    }
  }
}

class DisruptorTelemetryEvent(var kind: String = "", var location: String = "", var threadId: String = "", var id: String = " ", var timeStamp: Long = -1, var data: Array[String] = Array.empty)

class DisruptorTelemetryEventFactory extends EventFactory[DisruptorTelemetryEvent] {
  override def newInstance(): DisruptorTelemetryEvent = {
    new DisruptorTelemetryEvent()
  }
}

class TelemetryEventHandler(writer: TelemetryEventWriter) extends EventHandler[DisruptorTelemetryEvent] {
  val currentEvent = new mutable.HashMap[String, DisruptorTelemetryEvent]()
  val metrics = new mutable.HashMap[String, DisruptorTelemetryEvent]()

  override def onEvent(event: DisruptorTelemetryEvent, sequence: Long, endOfBatch: Boolean): Unit = {
    val data: Array[String] = event.data
    if (data == null) {
      return
    }
    writer.write(event.kind, event.location, event.threadId, event.id, event.timeStamp, event.data)
  }
}

/**
  * Handles how the events are being persisted
  */
trait TelemetryEventWriter {

  /**
    * Writes an telemtery event
    */
  def write(kind: String, location: String, threadId: String, id: String, timeStamp: Long, data: Array[String]): Unit

  /**
    * Flushes any intermediate buffer
    */
  def flush(): Unit

  /**
    * Closes any open connection
    */
  def close(): Unit
}

class FileTelemetryEventWriter(directory: File) extends TelemetryEventWriter {

  private val INLINE_WRITER_SETTINGS = {
    val settings = new JsonWriterSettings(new JsonDataFormat)
    settings.encoding = Some("UTF-8")
    settings.indent = false
    settings
  }
  private val localContext = EvaluationContext()
  private val outputStream = localContext.registerCloseable(new FileOutputStream(new File(directory, "events.log")))
  private val jsonWriter = new JsonWriter(outputStream, INLINE_WRITER_SETTINGS)(localContext)

  def write(kind: String, location: String, threadId: String, id: String, timeStamp: Long, data: Array[String]): Unit = {
    jsonWriter.writeOpenObject()
    jsonWriter.writeKey("kind")
    jsonWriter.writeQuoteString(kind)
    jsonWriter.writeComma()

    jsonWriter.writeKey("threadId")
    jsonWriter.writeQuoteString(threadId)
    jsonWriter.writeComma()

    jsonWriter.writeKey("id")
    jsonWriter.writeQuoteString(id)
    jsonWriter.writeComma()

    jsonWriter.writeKey("timeStamp")

    jsonWriter.writeNumber(timeStamp)(localContext)
    jsonWriter.writeComma()

    jsonWriter.writeKey("location")

    jsonWriter.writeQuoteString(location)
    jsonWriter.writeComma()

    jsonWriter.writeKey("data")
    jsonWriter.writeOpenObject()
    var i = 0

    while (i + 1 < data.length) {
      if (i > 0) {
        jsonWriter.writeComma()
      }
      jsonWriter.writeKey(data(i))
      i = i + 1
      jsonWriter.writeQuoteString(data(i))
      i = i + 1

    }
    jsonWriter.writeCloseObject()

    jsonWriter.writeCloseObject()
    jsonWriter.writeNewLine()
  }

  /**
    * Flushes any intermediate buffer
    */
  override def flush(): Unit = {
    jsonWriter.flush()
  }

  /**
    * Closes any open connection
    */
  override def close(): Unit = {
    localContext.close()
  }
}

/**
  * Registers the service
  */
class TelemetryServiceRegistration extends ServiceRegistration[TelemetryService] {
  override def service: Class[TelemetryService] = classOf[TelemetryService]

  override def implementation = new DefaultTelemetryService()
}
