/**
* This module provides functions that allows the user to handle values as if they were Trees
*
* *Since Version: 2.2.2*
*/
%dw 2.0

var OBJECT_TYPE = "Object"

var ATTRIBUTE_TYPE = "Attribute"

var ARRAY_TYPE = "Array"

/**
* Represents an Array of PathElement, that identifies the location of a Node in a Tree
*/
type Path = Array<PathElement>

/**
* Represents a specific selection node in a Path
*/
type PathElement = {|
        //{options: ["Object","Attribute","Array"]}
        kind: String ,
        selector: String | Number,
        namespace: Namespace | Null
    |}


/**
* Transforms a `Path` to a `String` representation.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | path | The path to be transformed to `String`
* |===
*
* === Example
*
* This example transforms a `Path` to a `String` representation
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Tree
* output application/json
* ---
* asExpressionString([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: ATTRIBUTE_TYPE, selector: "name", namespace: null}])
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ".user.@name"
* ----
**/
fun asExpressionString(path: Path): String =
    path reduce ((item, accumulator = "") -> do {
        var selectorExp = item.kind match {
            case "Attribute" -> ".@$(item.selector)"
            case "Array" -> "[$(item.selector)]"
            case "Object" -> ".$(item.selector)"
        }
        ---
        if(isEmpty(accumulator))
            selectorExp
        else
           accumulator ++ selectorExp
     })


/**
* Maps all nodes whose values are leafs.
* Node of type leafs are all elements where the value is not an Object or an Array
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value to map
* | callback | The mapper function
* |===
*
* === Example
*
* This example transforms all the String values to upper
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
*  import * from dw::util::Tree
*   output application/json
*   ---
*  {
*      user: [{
*          name: "mariano",
*          lastName: "achaval"
*      }],
*      group: "data-weave"
*  } mapLeafValues (value, path) -> upper(value)
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "user": [
*      {
*        "name": "MARIANO",
*        "lastName": "ACHAVAL"
*      }
*    ],
*    "group": "DATA-WEAVE"
*  }
* ----
**/
fun mapLeafValues(value: Any, callback: (value: Any, path: Path) -> Any): Any = do {

    fun doMapAttributes(key: Key, path: Path, callback: (value: Any, path: Path) -> Any) =
        key.@ mapObject (value, key) -> {
            (key) : doMapChildValues(value, path << {kind: ATTRIBUTE_TYPE, selector: key as String, namespace: key.#}, callback)
        }

    fun doMapChildValues(value: Any, path: Path, callback: (value: Any, path: Path) -> Any) = do {
        value match {
            case obj is  Object -> obj mapObject ((value, key, index) -> {
                (key)
                    @((doMapAttributes(key,path,callback))):
                        doMapChildValues(value, path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#}, callback)
            })
            case arr is Array -> arr map ((item, index) -> doMapChildValues(item, path << {kind: ARRAY_TYPE, selector: index, namespace: null}, callback))
            else -> callback(value, path)
        }
    }
    ---
    doMapChildValues(value, [], callback)
}


//fun mapNodeKeys(value: Any, callback: (key: Key, path: Path) -> Key | String) = do {
//    fun doMapNodeKeys(value: Any, path: Path, callback: (value: Key, path: Path) -> Key | String) = do {
//       value match {
//            case obj is  Object -> obj mapObject ((value, key, index) -> {
//                (callback(key, path))
//                    @((doMapNodeKeys(key.@, path << {kind: ATTRIBUTE_TYPE, selector: key as String, namespace: key.#}, callback))):
//                        doMapNodeKeys(value, path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#}, callback)
//            })
//            case arr is Array -> arr map ((item, index) -> doMapNodeKeys(item, path << {kind: ARRAY_TYPE, selector: index, namespace: null}, callback))
//            else -> value
//       }
//    }
//    ---
//    doMapNodeKeys(value, [], callback)
//}


/**
* Returns true if any Node of the tree validates against the specified criteria
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value to search
* | callback | The criteria
* |===
*
* === Example
*
* This example checks if there is any user with name "Peter"
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
** %dw 2.0
*  import * from dw::util::Tree
*   output application/json
*   ---
*  {
*      user: [{
*          name: "mariano",
*          lastName: "achaval",
*          friends: [
*              {
*                  name: "julian"
*              },
*              {
*                  name: "tom"
*              }
*          ]
*      },
*      {
*          name: "leandro",
*          lastName: "shokida",
*          friends: [
*              {
*                  name: "peter"
*              },
*              {
*                  name: "robert"
*              }
*          ]
*
*      }
*      ],
*
*  } nodeExists ((value, path) -> path[-1].selector == "name" and value == "peter")
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
fun nodeExists(value: Any, callback: (value: Any, path: Path) -> Boolean):Boolean = do {

    fun existsInObject(value:Object,path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case {k:v ~ xs} -> do{
                var currentPath = path << {kind: OBJECT_TYPE, selector: k as String, namespace: k.#}
                var exists = callback(v, currentPath) or treeExists(v, currentPath, callback) or existsInAttribute(k.@ default {}, currentPath, callback)
                ---
                if(exists)
                    true
                else
                   treeExists(xs, path, callback)
            }
            case {} -> false
        }
    }

    fun existsInAttribute(value:Object, path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case {k:v ~ xs} -> do{
                var currentPath = path << {kind: ATTRIBUTE_TYPE, selector: k as String, namespace: k.#}
                var exists = callback(v, currentPath)
                ---
                if(exists)
                    true
                else
                   treeExists(xs, path, callback)
            }
            case {} -> false
        }
    }

    fun existsInArray(value:Array,path: Path,callback: (value: Any, path: Path) -> Boolean, index: Number):Boolean = do {
        value match {
            case [v ~ xs] -> do{
                var currentPath = path << {kind: ARRAY_TYPE, selector: index, namespace: null}
                var exists = callback(v, currentPath) or treeExists(v, currentPath, callback)
                ---
                if(exists)
                    true
                else
                   existsInArray(xs, path, callback, index + 1)
            }
            case [] -> false
        }
    }

    fun treeExists(value: Any, path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case obj is  Object -> existsInObject(obj,path,callback)
            case arr is Array -> existsInArray(arr,path,callback, 0)
            else -> callback(value, path) as Boolean
        }
    }
    ---
    treeExists(value, [], callback)
}






