/**
* This utility module provides functions that enable you to handle values
* as though they are tree data structures.
*
* The module is included with Mule runtime. To use it, you must import it into
* your DataWeave code, for example, by adding the line
* `import * from dw::util::Tree` to the header of your script.
*
* _Introduced in DataWeave 2.2.2. Supported by Mule 4.2.2 and later._
*
*/
%dw 2.0

@Since(version = "2.2.2")
var OBJECT_TYPE = "Object"

@Since(version = "2.2.2")
var ATTRIBUTE_TYPE = "Attribute"

@Since(version = "2.2.2")
var ARRAY_TYPE = "Array"

/**
* Represents an array of `PathElement` types that identify the location of a node in a tree.
*/
@Since(version = "2.2.2")
type Path = Array<PathElement>

/**
* Represents a specific selection of a node in a path.
*/
@Since(version = "2.2.2")
type PathElement = {|
        //{options: ["Object","Attribute","Array"]}
        kind: String ,
        selector: String | Number,
        namespace: Namespace | Null
    |}


/**
* Transforms a path to a string representation.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | path | The path to transform to a string.
* |===
*
* === Example
*
* This example transforms a path to a string representation.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Tree
* output application/json
* ---
* asExpressionString([{kind: OBJECT_TYPE, selector: "user", namespace: null}, {kind: ATTRIBUTE_TYPE, selector: "name", namespace: null}])
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ".user.@name"
* ----
**/
@Since(version = "2.2.2")
fun asExpressionString(path: Path): String =
    path reduce ((item, accumulator = "") -> do {
        var selectorExp = item.kind match {
            case "Attribute" -> ".@$(item.selector)"
            case "Array" -> "[$(item.selector)]"
            case "Object" -> ".$(item.selector)"
        }
        ---
        if(isEmpty(accumulator))
            selectorExp
        else
           accumulator ++ selectorExp
     })


/**
* Maps the terminal (leaf) nodes in the tree.
*
*
* Leafs nodes cannot have an object or an array as a value.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | The value to map.
* | `callback` | The mapper function.
* |===
*
* === Example
*
* This example transforms all the string values to upper case.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* %dw 2.0
*  import * from dw::util::Tree
*   output application/json
*   ---
*  {
*      user: [{
*          name: "mariano",
*          lastName: "achaval"
*      }],
*      group: "data-weave"
*  } mapLeafValues (value, path) -> upper(value)
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "user": [
*      {
*        "name": "MARIANO",
*        "lastName": "ACHAVAL"
*      }
*    ],
*    "group": "DATA-WEAVE"
*  }
* ----
**/
@Since(version = "2.2.2")
fun mapLeafValues(value: Any, callback: (value: Any, path: Path) -> Any): Any = do {

    fun doMapAttributes(key: Key, path: Path, callback: (value: Any, path: Path) -> Any) =
        key.@ mapObject (value, key) -> {
            (key) : doMapChildValues(value, path << {kind: ATTRIBUTE_TYPE, selector: key as String, namespace: key.#}, callback)
        }

    fun doMapChildValues(value: Any, path: Path, callback: (value: Any, path: Path) -> Any) = do {
        value match {
            case obj is  Object -> obj mapObject ((value, key, index) -> {
                (key)
                    @((doMapAttributes(key,path,callback))):
                        doMapChildValues(value, path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#}, callback)
            })
            case arr is Array -> arr map ((item, index) -> doMapChildValues(item, path << {kind: ARRAY_TYPE, selector: index, namespace: null}, callback))
            else -> callback(value, path)
        }
    }
    ---
    doMapChildValues(value, [], callback)
}


//fun mapNodeKeys(value: Any, callback: (key: Key, path: Path) -> Key | String) = do {
//    fun doMapNodeKeys(value: Any, path: Path, callback: (value: Key, path: Path) -> Key | String) = do {
//       value match {
//            case obj is  Object -> obj mapObject ((value, key, index) -> {
//                (callback(key, path))
//                    @((doMapNodeKeys(key.@, path << {kind: ATTRIBUTE_TYPE, selector: key as String, namespace: key.#}, callback))):
//                        doMapNodeKeys(value, path << {kind: OBJECT_TYPE, selector: key as String, namespace: key.#}, callback)
//            })
//            case arr is Array -> arr map ((item, index) -> doMapNodeKeys(item, path << {kind: ARRAY_TYPE, selector: index, namespace: null}, callback))
//            else -> value
//       }
//    }
//    ---
//    doMapNodeKeys(value, [], callback)
//}


/**
* Returns `true` if any node in the tree validates against the specified criteria.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | The value to search.
* | `callback` | The criteria.
* |===
*
* === Example
*
* This example checks for any user with the name `peter`.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
** %dw 2.0
*  import * from dw::util::Tree
*   output application/json
*   ---
*  {
*      user: [{
*          name: "mariano",
*          lastName: "achaval",
*          friends: [
*              {
*                  name: "julian"
*              },
*              {
*                  name: "tom"
*              }
*          ]
*      },
*      {
*          name: "leandro",
*          lastName: "shokida",
*          friends: [
*              {
*                  name: "peter"
*              },
*              {
*                  name: "robert"
*              }
*          ]
*
*      }
*      ],
*
*  } nodeExists ((value, path) -> path[-1].selector == "name" and value == "peter")
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
@Since(version = "2.2.2")
fun nodeExists(value: Any, callback: (value: Any, path: Path) -> Boolean):Boolean = do {

    fun existsInObject(value:Object,path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case {k:v ~ xs} -> do{
                var currentPath = path << {kind: OBJECT_TYPE, selector: k as String, namespace: k.#}
                var exists = callback(v, currentPath) or treeExists(v, currentPath, callback) or existsInAttribute(k.@ default {}, currentPath, callback)
                ---
                if(exists)
                    true
                else
                   treeExists(xs, path, callback)
            }
            case {} -> false
        }
    }

    fun existsInAttribute(value:Object, path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case {k:v ~ xs} -> do{
                var currentPath = path << {kind: ATTRIBUTE_TYPE, selector: k as String, namespace: k.#}
                var exists = callback(v, currentPath)
                ---
                if(exists)
                    true
                else
                   treeExists(xs, path, callback)
            }
            case {} -> false
        }
    }

    fun existsInArray(value:Array,path: Path,callback: (value: Any, path: Path) -> Boolean, index: Number):Boolean = do {
        value match {
            case [v ~ xs] -> do{
                var currentPath = path << {kind: ARRAY_TYPE, selector: index, namespace: null}
                var exists = callback(v, currentPath) or treeExists(v, currentPath, callback)
                ---
                if(exists)
                    true
                else
                   existsInArray(xs, path, callback, index + 1)
            }
            case [] -> false
        }
    }

    fun treeExists(value: Any, path: Path, callback: (value: Any, path: Path) -> Boolean):Boolean = do {
        value match {
            case obj is  Object -> existsInObject(obj,path,callback)
            case arr is Array -> existsInArray(arr,path,callback, 0)
            else -> callback(value, path) as Boolean
        }
    }
    ---
    treeExists(value, [], callback)
}
