/**
* A utility module that provides mathematical functions.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::util::Math` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
* Variable `E` sets the value of mathematical constant `e`,
* the base of natural logarithms.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*/
@Since(version = "2.4.0")
var E = 2.7182818284590452354

/**
* Variable `PI` sets the value of constant value pi, the ratio
* of the circumference of a circle to its diameter.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*/
@Since(version = "2.4.0")
var PI = 3.14159265358979323846

/**
* Returns the trigonometric sine of an angle from a given number of radians.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angle` | Number of radians in an angle.
* |===
*
* === Example
*
* This example shows how `sin` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "sin0": sin(0),
*   "sin13": sin(0.13),
*   "sin-1": sin(-1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*{
*   "sin0": 0.0,
*   "sin13": 0.12963414261969486,
*   "sin-1": -0.8414709848078965
* }
* ----
**/
@Since(version = "2.4.0")
fun sin(angle: Number):Number = native("system::SinFunctionValue")

/**
* Returns the trigonometric cosine of an angle from a given number of radians.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angle` | Number of radians in an angle.
* |===
*
* === Example
*
* This example shows how `cos` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "cos0": cos(0),
*   "cos13": cos(0.13),
*   "cos-1": cos(-1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "cos0": 1.0,
*   "cos13": 0.9915618937147881,
*   "cos-1": 0.5403023058681398
* }
* ----
**/
@Since(version = "2.4.0")
fun cos(angle: Number):Number = native("system::CosFunctionValue")

/**
* Returns the trigonometric tangent of an angle from a given number of radians.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angle` | Number of radians in an angle.
* |===
*
* === Example
*
* This example shows how `tan` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*    "tan0": tan(0),
*    "tan13": tan(0.13),
*    "tan-1": tan(-1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "tan0": 0.0,
*    "tan13": 0.13073731800446006,
*    "tan-1": -1.5574077246549023
*  }
* ----
**/
@Since(version = "2.4.0")
fun tan(angle: Number):Number = native("system::TanFunctionValue")

/**
* Returns an arc sine value that can range from `-pi/2` through `pi/2`.
*
*
* If the absolute value of the input is greater than 1, the result
* is `null`.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angle` | Number to convert into its arc sine value.
* |===
*
* === Example
*
* This example shows how `asin` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "asin0": asin(0),
*   "asin13": asin(0.13),
*   "asin-1": asin(-1),
*   "asin1.1": asin(1.1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "asin0": 0.0,
*    "asin13": 0.1303689797031455,
*    "asin-1": -1.5707963267948966,
*    "asin1.1": null
*  }
* ----
**/
@Since(version = "2.4.0")
fun asin(angle: Number):Number = native("system::ASinFunctionValue")

/**
* Returns an arc cosine value that can range from `0.0` through pi.
*
*
* If the absolute value of the input is greater than `1`,
* the result is `null`.
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angle` | Number to convert into it arc cosine value.
* |===
*
* === Example
*
* This example shows how `acos` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "acos0": acos(0),
*   "acos13": acos(0.13),
*   "acos-1": acos(-1),
*   "acos1": acos(1),
*   "acos1.1": acos(1.1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "acos0": 1.5707963267948966,
*    "acos13": 1.440427347091751,
*    "acos-1": 3.141592653589793,
*    "acos1": 0.0,
*    "acos1.1": null
*  }
* ----
**/
@Since(version = "2.4.0")
fun acos(angle: Number):Number | NaN = native("system::ACosFunctionValue")

/**
* Returns an arc tangent value that can range from `-pi/2` through `pi/2`.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angle` | Number to convert into its arc tangent value.
* |===
*
* === Example
*
* This example shows how `atan` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "atan0":  atan(0),
*   "atan13": atan(0.13),
*   "atan-1": atan(-1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "atan0": 0.0,
*    "atan13": 0.12927500404814307,
*    "atan-1": -0.7853981633974483
* }
* ----
**/
@Since(version = "2.4.0")
fun atan(angle: Number):Number = native("system::ATanFunctionValue")

/**
* Converts a given number of degrees in an angle to an approximately
* equivalent number of radians.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angdeg` | Number of degrees to convert into radians.
* |===
*
* === Example
*
* This example shows how `toRadians` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "toRadians10":  toRadians(10),
*   "toRadians013": toRadians(0.13),
*   "toRadians-20": toRadians(-20)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* {
*    "toRadians10": 0.1745329251994329576922222222222222,
*    "toRadians013": 0.002268928027592628449998888888888889,
*    "toRadians-20": -0.3490658503988659153844444444444444
*  }
* ----
**/
@Since(version = "2.4.0")
fun toRadians(angdeg:Number): Number = angdeg / 180.0 * PI

/**
* Converts an angle measured in radians to an approximately
* equivalent number of degrees.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `angrad` | Number of radians to convert to degrees.
* |===
*
* === Example
*
* This example shows how `toDegrees` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "toDegrees0.17":  toDegrees(0.174),
*   "toDegrees0": toDegrees(0),
*   "toDegrees-20": toDegrees(-0.20)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "toDegrees0.17": 9.969465635276323832571267395889251,
*    "toDegrees0": 0E+19,
*    "toDegrees-20": -11.45915590261646417536927286883822
*  }
* ----
**/
@Since(version = "2.4.0")
fun toDegrees(angrad:Number):Number = angrad * 180.0 / PI

/**
*  Returns the natural logarithm (base `e`) of a number.
*
*
* If the input value is less than or equal to zero,
* the result is `NaN` (or `null`).
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `a` | Number to convert into its natural logarithm.
* |===
*
* === Example
*
* This example shows how `logn` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*    "logn10":  logn(10),
*    "logn13": logn(0.13),
*    "logn-20": logn(-20)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "logn10": 2.302585092994046,
*    "logn13": -2.0402208285265546,
*    "logn-20": null
* }
* ----
**/
@Since(version = "2.4.0")
fun logn(a: Number): Number | NaN = native("system::LognFunctionValue")

/**
* Returns the logarithm base 10 of a number.
*
*
* _Introduced in DataWeave 2.4.0. Supported by Mule 4.4.0 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `a` | A `Number` value that serves as input to the function.
* |===
*
* === Example
*
* This example shows how `log10` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Math
* output application/json
* ---
* {
*   "log1010": log10(10),
*   "log1013": log10(0.13),
*   "log10-20": log10(-20)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*  {
*     "log1010": 1.0,
*     "log1013": -0.8860566476931632,
*     "log10-20": null
*  }
* ----
**/
@Since(version = "2.4.0")
fun log10(a: Number): Number | NaN = native("system::Log10FunctionValue")
