package org.mule.common.metadata.parser.json;

import static java.lang.String.*;

/**
 * JSONType is implemented by classes which represent individual JSON schemas or types.
 * <p/>
 * A JSONType determines a subset of Objects, through its <tt>contains</tt> method.
 * <p/>
 * Given a JSONType, users should call the {@link org.mule.common.metadata.parser.json.JSONType.contains} method on values to see if each
 * value is in the type.
 *
 * @author Timothy Danford
 */
public interface JSONType
{

    /**
     * The central method of JSONType; the type encompasses the values for which <tt>contains</tt> returns <tt>true</tt>.
     *
     * @return <tt>true</tt>if the given Object satisfies the type, <tt>false</tt> otherwise.
     */
    public boolean contains(Object obj);

    /**
     * Returns a non-null String explanation, suitable for display to a user, explaining why the given object fails to conform to the JSONType's <tt>contains</tt> method.
     *
     * @return an explanatory string if <tt>contains(arg)</tt> is <tt>false</tt>, or <tt>null</tt> if <tt>contains(arg)</tt> is <tt>true</tt>.
     * @arg obj A value for which <tt>contains(obj)</tt> returns <tt>false</tt>.
     */
    public java.lang.String explain(Object obj);

    /**
     * @return <tt>true</tt> if the type will accept "missing" (i.e. <tt>null</tt>) values,
     * <tt>false</tt> otherwise.
     */
    public boolean isOptional();

    public boolean isJSONPrimitive();

    public boolean isJSONArray();

    public boolean isJSONObject();

    public boolean isJSONPointer();

    public static class Everything extends AbstractType
    {

        public boolean contains(Object obj)
        {
            return obj != null;
        }

        @Override
        public boolean isJSONPrimitive()
        {
            return true;
        }

        @Override
        public boolean isJSONArray()
        {
            return false;
        }

        @Override
        public boolean isJSONObject()
        {
            return false;
        }

        @Override
        public boolean isJSONPointer()
        {
            return false;
        }
    }

    public static class Empty extends AbstractType
    {

        public boolean contains(Object obj)
        {
            return false;
        }

        public java.lang.String explain(Object obj)
        {
            return java.lang.String.format(
                    "REJECT: \"%s\" not in Empty", java.lang.String.valueOf(obj));
        }

        @Override
        public boolean isJSONPrimitive()
        {
            return false;
        }

        @Override
        public boolean isJSONArray()
        {
            return false;
        }

        @Override
        public boolean isJSONObject()
        {
            return false;
        }

        @Override
        public boolean isJSONPointer()
        {
            return false;
        }
    }

    public static class StringType extends AbstractType
    {

        public boolean contains(Object obj)
        {
            return obj != null && obj instanceof java.lang.String;
        }

        public java.lang.String explain(Object obj)
        {
            java.lang.String typeName = "String";
            if (contains(obj))
            {
                return null;
            }
            else if (obj == null)
            {
                return java.lang.String.format("REJECT: null value");
            }
            else
            {
                return java.lang.String.format(
                        "REJECT: \"%s\" has type %s not in %s",
                        java.lang.String.valueOf(obj),
                        obj.getClass().getSimpleName(),
                        typeName);
            }
        }

        @Override
        public boolean isJSONPrimitive()
        {
            return true;
        }

        @Override
        public boolean isJSONArray()
        {
            return false;
        }

        @Override
        public boolean isJSONObject()
        {
            return false;
        }

        @Override
        public boolean isJSONPointer()
        {
            return false;
        }
    }

    public static class IntegerType extends AbstractType
    {

        public boolean contains(Object obj)
        {
            if (obj == null)
            {
                return false;
            }
            if (obj instanceof java.lang.Integer)
            {
                return true;
            }
            try
            {
                java.lang.Integer.parseInt(obj.toString());
                return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }

        public java.lang.String explain(Object obj)
        {
            if (obj == null)
            {
                return format("REJECT: null value");
            }
            if (obj instanceof java.lang.Integer)
            {
                return null;
            }
            try
            {
                java.lang.Integer.parseInt(obj.toString());
                return null;
            }
            catch (NumberFormatException e)
            {
                return format("REJECT: %s", e.getMessage());
            }
        }

        @Override
        public boolean isJSONPrimitive()
        {
            return true;
        }

        @Override
        public boolean isJSONArray()
        {
            return false;
        }

        @Override
        public boolean isJSONObject()
        {
            return false;
        }

        @Override
        public boolean isJSONPointer()
        {
            return false;
        }
    }

    public static class DoubleType extends AbstractType
    {

        public boolean contains(Object obj)
        {
            if (obj == null)
            {
                return false;
            }
            if (obj instanceof java.lang.Double)
            {
                return true;
            }
            try
            {
                java.lang.Double.parseDouble(obj.toString());
                return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }

        public java.lang.String explain(Object obj)
        {
            if (obj == null)
            {
                return format("REJECT: null value");
            }
            if (obj instanceof java.lang.Double)
            {
                return null;
            }
            try
            {
                java.lang.Double.parseDouble(obj.toString());
                return null;
            }
            catch (NumberFormatException e)
            {
                return format("REJECT: %s", e.getMessage());
            }
        }

        @Override
        public boolean isJSONPrimitive()
        {
            return true;
        }

        @Override
        public boolean isJSONArray()
        {
            return false;
        }

        @Override
        public boolean isJSONObject()
        {
            return false;
        }

        @Override
        public boolean isJSONPointer()
        {
            return false;
        }
    }

    public static class NumberType extends AbstractType
    {

        public boolean contains(Object obj)
        {
            if (obj == null)
            {
                return false;
            }
            if (obj instanceof java.lang.Double)
            {
                return true;
            }
            try
            {
                java.lang.Double.parseDouble(obj.toString());
                return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }

        public java.lang.String explain(Object obj)
        {
            if (obj == null)
            {
                return format("REJECT: null value");
            }
            if (obj instanceof java.lang.Double)
            {
                return null;
            }
            try
            {
                java.lang.Double.parseDouble(obj.toString());
                return null;
            }
            catch (NumberFormatException e)
            {
                return format("REJECT: %s", e.getMessage());
            }
        }

        @Override
        public boolean isJSONPrimitive()
        {
            return true;
        }

        @Override
        public boolean isJSONArray()
        {
            return false;
        }

        @Override
        public boolean isJSONObject()
        {
            return false;
        }

        @Override
        public boolean isJSONPointer()
        {
            return false;
        }
    }

    public static class BooleanType extends AbstractType
    {

        public boolean contains(Object obj)
        {
            if (obj == null)
            {
                return false;
            }
            if (obj instanceof java.lang.Boolean)
            {
                return true;
            }
            try
            {
                java.lang.Boolean.parseBoolean(obj.toString());
                return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }

        public java.lang.String explain(Object obj)
        {
            if (obj == null)
            {
                return format("REJECT: null value");
            }
            if (obj instanceof java.lang.Boolean)
            {
                return null;
            }
            try
            {
                java.lang.Boolean.parseBoolean(obj.toString());
                return null;
            }
            catch (NumberFormatException e)
            {
                return format("REJECT: %s", e.getMessage());
            }
        }

        @Override
        public boolean isJSONPrimitive()
        {
            return true;
        }

        @Override
        public boolean isJSONArray()
        {
            return false;
        }

        @Override
        public boolean isJSONObject()
        {
            return false;
        }

        @Override
        public boolean isJSONPointer()
        {
            return false;
        }
    }
}
