/**
 * marked - a markdown parser
 * Copyright (c) 2011-2014, Christopher Jeffrey. (MIT Licensed)
 * https://github.com/chjj/marked
 */

;(function() {

/**
 * Block-Level Grammar
 */

var block = {
  newline: /^\n+/,
  code: /^( {4}[^\n]+\n*)+/,
  fences: noop,
  hr: /^( *[-*_]){3,} *(?:\n+|$)/,
  heading: /^ *(#{1,6}) *([^\n]+?) *#* *(?:\n+|$)/,
  nptable: noop,
  lheading: /^([^\n]+)\n *(=|-){2,} *(?:\n+|$)/,
  blockquote: /^( *>[^\n]+(\n(?!def)[^\n]+)*\n*)+/,
  list: /^( *)(bull) [\s\S]+?(?:hr|def|\n{2,}(?! )(?!\1bull )\n*|\s*$)/,
  html: /^ *(?:comment *(?:\n|\s*$)|closed *(?:\n{2,}|\s*$)|closing *(?:\n{2,}|\s*$))/,
  def: /^ *\[([^\]]+)\]: *<?([^\s>]+)>?(?: +["(]([^\n]+)[")])? *(?:\n+|$)/,
  table: noop,
  paragraph: /^((?:[^\n]+\n?(?!hr|heading|lheading|blockquote|tag|def))+)\n*/,
  text: /^[^\n]+/
};

block.bullet = /(?:[*+-]|\d+\.)/;
block.item = /^( *)(bull) [^\n]*(?:\n(?!\1bull )[^\n]*)*/;
block.item = replace(block.item, 'gm')
  (/bull/g, block.bullet)
  ();

block.list = replace(block.list)
  (/bull/g, block.bullet)
  ('hr', '\\n+(?=\\1?(?:[-*_] *){3,}(?:\\n+|$))')
  ('def', '\\n+(?=' + block.def.source + ')')
  ();

block.blockquote = replace(block.blockquote)
  ('def', block.def)
  ();

block._tag = '(?!(?:'
  + 'a|em|strong|small|s|cite|q|dfn|abbr|data|time|code'
  + '|var|samp|kbd|sub|sup|i|b|u|mark|ruby|rt|rp|bdi|bdo'
  + '|span|br|wbr|ins|del|img)\\b)\\w+(?!:/|[^\\w\\s@]*@)\\b';

block.html = replace(block.html)
  ('comment', /<!--[\s\S]*?-->/)
  ('closed', /<(tag)[\s\S]+?<\/\1>/)
  ('closing', /<tag(?:"[^"]*"|'[^']*'|[^'">])*?>/)
  (/tag/g, block._tag)
  ();

block.paragraph = replace(block.paragraph)
  ('hr', block.hr)
  ('heading', block.heading)
  ('lheading', block.lheading)
  ('blockquote', block.blockquote)
  ('tag', '<' + block._tag)
  ('def', block.def)
  ();

/**
 * Normal Block Grammar
 */

block.normal = merge({}, block);

/**
 * GFM Block Grammar
 */

block.gfm = merge({}, block.normal, {
  fences: /^ *(`{3,}|~{3,})[ \.]*(\S+)? *\n([\s\S]*?)\s*\1 *(?:\n+|$)/,
  paragraph: /^/,
  heading: /^ *(#{1,6}) +([^\n]+?) *#* *(?:\n+|$)/
});

block.gfm.paragraph = replace(block.paragraph)
  ('(?!', '(?!'
    + block.gfm.fences.source.replace('\\1', '\\2') + '|'
    + block.list.source.replace('\\1', '\\3') + '|')
  ();

/**
 * GFM + Tables Block Grammar
 */

block.tables = merge({}, block.gfm, {
  nptable: /^ *(\S.*\|.*)\n *([-:]+ *\|[-| :]*)\n((?:.*\|.*(?:\n|$))*)\n*/,
  table: /^ *\|(.+)\n *\|( *[-:]+[-| :]*)\n((?: *\|.*(?:\n|$))*)\n*/
});

/**
 * Block Lexer
 */

function Lexer(options) {
  this.tokens = [];
  this.tokens.links = {};
  this.options = options || marked.defaults;
  this.rules = block.normal;

  if (this.options.gfm) {
    if (this.options.tables) {
      this.rules = block.tables;
    } else {
      this.rules = block.gfm;
    }
  }
}

/**
 * Expose Block Rules
 */

Lexer.rules = block;

/**
 * Static Lex Method
 */

Lexer.lex = function(src, options) {
  var lexer = new Lexer(options);
  return lexer.lex(src);
};

/**
 * Preprocessing
 */

Lexer.prototype.lex = function(src) {
  src = src
    .replace(/\r\n|\r/g, '\n')
    .replace(/\t/g, '    ')
    .replace(/\u00a0/g, ' ')
    .replace(/\u2424/g, '\n');

  return this.token(src, true);
};

/**
 * Lexing
 */

Lexer.prototype.token = function(src, top, bq) {
  var src = src.replace(/^ +$/gm, '')
    , next
    , loose
    , cap
    , bull
    , b
    , item
    , space
    , i
    , l;

  while (src) {
    // newline
    if (cap = this.rules.newline.exec(src)) {
      src = src.substring(cap[0].length);
      if (cap[0].length > 1) {
        this.tokens.push({
          type: 'space'
        });
      }
    }

    // code
    if (cap = this.rules.code.exec(src)) {
      src = src.substring(cap[0].length);
      cap = cap[0].replace(/^ {4}/gm, '');
      this.tokens.push({
        type: 'code',
        text: !this.options.pedantic
          ? cap.replace(/\n+$/, '')
          : cap
      });
      continue;
    }

    // fences (gfm)
    if (cap = this.rules.fences.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'code',
        lang: cap[2],
        text: cap[3] || ''
      });
      continue;
    }

    // heading
    if (cap = this.rules.heading.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'heading',
        depth: cap[1].length,
        text: cap[2]
      });
      continue;
    }

    // table no leading pipe (gfm)
    if (top && (cap = this.rules.nptable.exec(src))) {
      src = src.substring(cap[0].length);

      item = {
        type: 'table',
        header: cap[1].replace(/^ *| *\| *$/g, '').split(/ *\| */),
        align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
        cells: cap[3].replace(/\n$/, '').split('\n')
      };

      for (i = 0; i < item.align.length; i++) {
        if (/^ *-+: *$/.test(item.align[i])) {
          item.align[i] = 'right';
        } else if (/^ *:-+: *$/.test(item.align[i])) {
          item.align[i] = 'center';
        } else if (/^ *:-+ *$/.test(item.align[i])) {
          item.align[i] = 'left';
        } else {
          item.align[i] = null;
        }
      }

      for (i = 0; i < item.cells.length; i++) {
        item.cells[i] = item.cells[i].split(/ *\| */);
      }

      this.tokens.push(item);

      continue;
    }

    // lheading
    if (cap = this.rules.lheading.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'heading',
        depth: cap[2] === '=' ? 1 : 2,
        text: cap[1]
      });
      continue;
    }

    // hr
    if (cap = this.rules.hr.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'hr'
      });
      continue;
    }

    // blockquote
    if (cap = this.rules.blockquote.exec(src)) {
      src = src.substring(cap[0].length);

      this.tokens.push({
        type: 'blockquote_start'
      });

      cap = cap[0].replace(/^ *> ?/gm, '');

      // Pass `top` to keep the current
      // "toplevel" state. This is exactly
      // how markdown.pl works.
      this.token(cap, top, true);

      this.tokens.push({
        type: 'blockquote_end'
      });

      continue;
    }

    // list
    if (cap = this.rules.list.exec(src)) {
      src = src.substring(cap[0].length);
      bull = cap[2];

      this.tokens.push({
        type: 'list_start',
        ordered: bull.length > 1
      });

      // Get each top-level item.
      cap = cap[0].match(this.rules.item);

      next = false;
      l = cap.length;
      i = 0;

      for (; i < l; i++) {
        item = cap[i];

        // Remove the list item's bullet
        // so it is seen as the next token.
        space = item.length;
        item = item.replace(/^ *([*+-]|\d+\.) +/, '');

        // Outdent whatever the
        // list item contains. Hacky.
        if (~item.indexOf('\n ')) {
          space -= item.length;
          item = !this.options.pedantic
            ? item.replace(new RegExp('^ {1,' + space + '}', 'gm'), '')
            : item.replace(/^ {1,4}/gm, '');
        }

        // Determine whether the next list item belongs here.
        // Backpedal if it does not belong in this list.
        if (this.options.smartLists && i !== l - 1) {
          b = block.bullet.exec(cap[i + 1])[0];
          if (bull !== b && !(bull.length > 1 && b.length > 1)) {
            src = cap.slice(i + 1).join('\n') + src;
            i = l - 1;
          }
        }

        // Determine whether item is loose or not.
        // Use: /(^|\n)(?! )[^\n]+\n\n(?!\s*$)/
        // for discount behavior.
        loose = next || /\n\n(?!\s*$)/.test(item);
        if (i !== l - 1) {
          next = item.charAt(item.length - 1) === '\n';
          if (!loose) loose = next;
        }

        this.tokens.push({
          type: loose
            ? 'loose_item_start'
            : 'list_item_start'
        });

        // Recurse.
        this.token(item, false, bq);

        this.tokens.push({
          type: 'list_item_end'
        });
      }

      this.tokens.push({
        type: 'list_end'
      });

      continue;
    }

    // html
    if (cap = this.rules.html.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: this.options.sanitize
          ? 'paragraph'
          : 'html',
        pre: !this.options.sanitizer
          && (cap[1] === 'pre' || cap[1] === 'script' || cap[1] === 'style'),
        text: cap[0]
      });
      continue;
    }

    // def
    if ((!bq && top) && (cap = this.rules.def.exec(src))) {
      src = src.substring(cap[0].length);
      this.tokens.links[cap[1].toLowerCase()] = {
        href: cap[2],
        title: cap[3]
      };
      continue;
    }

    // table (gfm)
    if (top && (cap = this.rules.table.exec(src))) {
      src = src.substring(cap[0].length);

      item = {
        type: 'table',
        header: cap[1].replace(/^ *| *\| *$/g, '').split(/ *\| */),
        align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
        cells: cap[3].replace(/(?: *\| *)?\n$/, '').split('\n')
      };

      for (i = 0; i < item.align.length; i++) {
        if (/^ *-+: *$/.test(item.align[i])) {
          item.align[i] = 'right';
        } else if (/^ *:-+: *$/.test(item.align[i])) {
          item.align[i] = 'center';
        } else if (/^ *:-+ *$/.test(item.align[i])) {
          item.align[i] = 'left';
        } else {
          item.align[i] = null;
        }
      }

      for (i = 0; i < item.cells.length; i++) {
        item.cells[i] = item.cells[i]
          .replace(/^ *\| *| *\| *$/g, '')
          .split(/ *\| */);
      }

      this.tokens.push(item);

      continue;
    }

    // top-level paragraph
    if (top && (cap = this.rules.paragraph.exec(src))) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'paragraph',
        text: cap[1].charAt(cap[1].length - 1) === '\n'
          ? cap[1].slice(0, -1)
          : cap[1]
      });
      continue;
    }

    // text
    if (cap = this.rules.text.exec(src)) {
      // Top-level should never reach here.
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'text',
        text: cap[0]
      });
      continue;
    }

    if (src) {
      throw new
        Error('Infinite loop on byte: ' + src.charCodeAt(0));
    }
  }

  return this.tokens;
};

/**
 * Inline-Level Grammar
 */

var inline = {
  escape: /^\\([\\`*{}\[\]()#+\-.!_>])/,
  autolink: /^<([^ >]+(@|:\/)[^ >]+)>/,
  url: noop,
  tag: /^<!--[\s\S]*?-->|^<\/?\w+(?:"[^"]*"|'[^']*'|[^'">])*?>/,
  link: /^!?\[(inside)\]\(href\)/,
  reflink: /^!?\[(inside)\]\s*\[([^\]]*)\]/,
  nolink: /^!?\[((?:\[[^\]]*\]|[^\[\]])*)\]/,
  strong: /^__([\s\S]+?)__(?!_)|^\*\*([\s\S]+?)\*\*(?!\*)/,
  em: /^\b_((?:[^_]|__)+?)_\b|^\*((?:\*\*|[\s\S])+?)\*(?!\*)/,
  code: /^(`+)\s*([\s\S]*?[^`])\s*\1(?!`)/,
  br: /^ {2,}\n(?!\s*$)/,
  del: noop,
  text: /^[\s\S]+?(?=[\\<!\[_*`]| {2,}\n|$)/
};

inline._inside = /(?:\[[^\]]*\]|[^\[\]]|\](?=[^\[]*\]))*/;
inline._href = /\s*<?([\s\S]*?)>?(?:\s+['"]([\s\S]*?)['"])?\s*/;

inline.link = replace(inline.link)
  ('inside', inline._inside)
  ('href', inline._href)
  ();

inline.reflink = replace(inline.reflink)
  ('inside', inline._inside)
  ();

/**
 * Normal Inline Grammar
 */

inline.normal = merge({}, inline);

/**
 * Pedantic Inline Grammar
 */

inline.pedantic = merge({}, inline.normal, {
  strong: /^__(?=\S)([\s\S]*?\S)__(?!_)|^\*\*(?=\S)([\s\S]*?\S)\*\*(?!\*)/,
  em: /^_(?=\S)([\s\S]*?\S)_(?!_)|^\*(?=\S)([\s\S]*?\S)\*(?!\*)/
});

/**
 * GFM Inline Grammar
 */

inline.gfm = merge({}, inline.normal, {
  escape: replace(inline.escape)('])', '~|])')(),
  url: /^(https?:\/\/[^\s<]+[^<.,:;"')\]\s])/,
  del: /^~~(?=\S)([\s\S]*?\S)~~/,
  text: replace(inline.text)
    (']|', '~]|')
    ('|', '|https?://|')
    ()
});

/**
 * GFM + Line Breaks Inline Grammar
 */

inline.breaks = merge({}, inline.gfm, {
  br: replace(inline.br)('{2,}', '*')(),
  text: replace(inline.gfm.text)('{2,}', '*')()
});

/**
 * Inline Lexer & Compiler
 */

function InlineLexer(links, options) {
  this.options = options || marked.defaults;
  this.links = links;
  this.rules = inline.normal;
  this.renderer = this.options.renderer || new Renderer;
  this.renderer.options = this.options;

  if (!this.links) {
    throw new
      Error('Tokens array requires a `links` property.');
  }

  if (this.options.gfm) {
    if (this.options.breaks) {
      this.rules = inline.breaks;
    } else {
      this.rules = inline.gfm;
    }
  } else if (this.options.pedantic) {
    this.rules = inline.pedantic;
  }
}

/**
 * Expose Inline Rules
 */

InlineLexer.rules = inline;

/**
 * Static Lexing/Compiling Method
 */

InlineLexer.output = function(src, links, options) {
  var inline = new InlineLexer(links, options);
  return inline.output(src);
};

/**
 * Lexing/Compiling
 */

InlineLexer.prototype.output = function(src) {
  var out = ''
    , link
    , text
    , href
    , cap;

  while (src) {
    // escape
    if (cap = this.rules.escape.exec(src)) {
      src = src.substring(cap[0].length);
      out += cap[1];
      continue;
    }

    // autolink
    if (cap = this.rules.autolink.exec(src)) {
      src = src.substring(cap[0].length);
      if (cap[2] === '@') {
        text = cap[1].charAt(6) === ':'
          ? this.mangle(cap[1].substring(7))
          : this.mangle(cap[1]);
        href = this.mangle('mailto:') + text;
      } else {
        text = escape(cap[1]);
        href = text;
      }
      out += this.renderer.link(href, null, text);
      continue;
    }

    // url (gfm)
    if (!this.inLink && (cap = this.rules.url.exec(src))) {
      src = src.substring(cap[0].length);
      text = escape(cap[1]);
      href = text;
      out += this.renderer.link(href, null, text);
      continue;
    }

    // tag
    if (cap = this.rules.tag.exec(src)) {
      if (!this.inLink && /^<a /i.test(cap[0])) {
        this.inLink = true;
      } else if (this.inLink && /^<\/a>/i.test(cap[0])) {
        this.inLink = false;
      }
      src = src.substring(cap[0].length);
      out += this.options.sanitize
        ? this.options.sanitizer
          ? this.options.sanitizer(cap[0])
          : escape(cap[0])
        : cap[0]
      continue;
    }

    // link
    if (cap = this.rules.link.exec(src)) {
      src = src.substring(cap[0].length);
      this.inLink = true;
      out += this.outputLink(cap, {
        href: cap[2],
        title: cap[3]
      });
      this.inLink = false;
      continue;
    }

    // reflink, nolink
    if ((cap = this.rules.reflink.exec(src))
        || (cap = this.rules.nolink.exec(src))) {
      src = src.substring(cap[0].length);
      link = (cap[2] || cap[1]).replace(/\s+/g, ' ');
      link = this.links[link.toLowerCase()];
      if (!link || !link.href) {
        out += cap[0].charAt(0);
        src = cap[0].substring(1) + src;
        continue;
      }
      this.inLink = true;
      out += this.outputLink(cap, link);
      this.inLink = false;
      continue;
    }

    // strong
    if (cap = this.rules.strong.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.strong(this.output(cap[2] || cap[1]));
      continue;
    }

    // em
    if (cap = this.rules.em.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.em(this.output(cap[2] || cap[1]));
      continue;
    }

    // code
    if (cap = this.rules.code.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.codespan(escape(cap[2], true));
      continue;
    }

    // br
    if (cap = this.rules.br.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.br();
      continue;
    }

    // del (gfm)
    if (cap = this.rules.del.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.del(this.output(cap[1]));
      continue;
    }

    // text
    if (cap = this.rules.text.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.text(escape(this.smartypants(cap[0])));
      continue;
    }

    if (src) {
      throw new
        Error('Infinite loop on byte: ' + src.charCodeAt(0));
    }
  }

  return out;
};

/**
 * Compile Link
 */

InlineLexer.prototype.outputLink = function(cap, link) {
  var href = escape(link.href)
    , title = link.title ? escape(link.title) : null;

  return cap[0].charAt(0) !== '!'
    ? this.renderer.link(href, title, this.output(cap[1]))
    : this.renderer.image(href, title, escape(cap[1]));
};

/**
 * Smartypants Transformations
 */

InlineLexer.prototype.smartypants = function(text) {
  if (!this.options.smartypants) return text;
  return text
    // em-dashes
    .replace(/---/g, '\u2014')
    // en-dashes
    .replace(/--/g, '\u2013')
    // opening singles
    .replace(/(^|[-\u2014/(\[{"\s])'/g, '$1\u2018')
    // closing singles & apostrophes
    .replace(/'/g, '\u2019')
    // opening doubles
    .replace(/(^|[-\u2014/(\[{\u2018\s])"/g, '$1\u201c')
    // closing doubles
    .replace(/"/g, '\u201d')
    // ellipses
    .replace(/\.{3}/g, '\u2026');
};

/**
 * Mangle Links
 */

InlineLexer.prototype.mangle = function(text) {
  if (!this.options.mangle) return text;
  var out = ''
    , l = text.length
    , i = 0
    , ch;

  for (; i < l; i++) {
    ch = text.charCodeAt(i);
    if (Math.random() > 0.5) {
      ch = 'x' + ch.toString(16);
    }
    out += '&#' + ch + ';';
  }

  return out;
};

/**
 * Renderer
 */

function Renderer(options) {
  this.options = options || {};
}

Renderer.prototype.code = function(code, lang, escaped) {
  if (this.options.highlight) {
    var out = this.options.highlight(code, lang);
    if (out != null && out !== code) {
      escaped = true;
      code = out;
    }
  }

  if (!lang) {
    return '<pre><code>'
      + (escaped ? code : escape(code, true))
      + '\n</code></pre>';
  }

  return '<pre><code class="'
    + this.options.langPrefix
    + escape(lang, true)
    + '">'
    + (escaped ? code : escape(code, true))
    + '\n</code></pre>\n';
};

Renderer.prototype.blockquote = function(quote) {
  return '<blockquote>\n' + quote + '</blockquote>\n';
};

Renderer.prototype.html = function(html) {
  return html;
};

Renderer.prototype.heading = function(text, level, raw) {
  return '<h'
    + level
    + ' id="'
    + this.options.headerPrefix
    + raw.toLowerCase().replace(/[^\w]+/g, '-')
    + '">'
    + text
    + '</h'
    + level
    + '>\n';
};

Renderer.prototype.hr = function() {
  return this.options.xhtml ? '<hr/>\n' : '<hr>\n';
};

Renderer.prototype.list = function(body, ordered) {
  var type = ordered ? 'ol' : 'ul';
  return '<' + type + '>\n' + body + '</' + type + '>\n';
};

Renderer.prototype.listitem = function(text) {
  return '<li>' + text + '</li>\n';
};

Renderer.prototype.paragraph = function(text) {
  return '<p>' + text + '</p>\n';
};

Renderer.prototype.table = function(header, body) {
  return '<table>\n'
    + '<thead>\n'
    + header
    + '</thead>\n'
    + '<tbody>\n'
    + body
    + '</tbody>\n'
    + '</table>\n';
};

Renderer.prototype.tablerow = function(content) {
  return '<tr>\n' + content + '</tr>\n';
};

Renderer.prototype.tablecell = function(content, flags) {
  var type = flags.header ? 'th' : 'td';
  var tag = flags.align
    ? '<' + type + ' style="text-align:' + flags.align + '">'
    : '<' + type + '>';
  return tag + content + '</' + type + '>\n';
};

// span level renderer
Renderer.prototype.strong = function(text) {
  return '<strong>' + text + '</strong>';
};

Renderer.prototype.em = function(text) {
  return '<em>' + text + '</em>';
};

Renderer.prototype.codespan = function(text) {
  return '<code>' + text + '</code>';
};

Renderer.prototype.br = function() {
  return this.options.xhtml ? '<br/>' : '<br>';
};

Renderer.prototype.del = function(text) {
  return '<del>' + text + '</del>';
};

Renderer.prototype.link = function(href, title, text) {
  if (this.options.sanitize) {
    try {
      var prot = decodeURIComponent(unescape(href))
        .replace(/[^\w:]/g, '')
        .toLowerCase();
    } catch (e) {
      return '';
    }
    if (prot.indexOf('javascript:') === 0 || prot.indexOf('vbscript:') === 0) {
      return '';
    }
  }
  var out = '<a href="' + href + '"';
  if (title) {
    out += ' title="' + title + '"';
  }
  out += '>' + text + '</a>';
  return out;
};

Renderer.prototype.image = function(href, title, text) {
  var out = '<img src="' + href + '" alt="' + text + '"';
  if (title) {
    out += ' title="' + title + '"';
  }
  out += this.options.xhtml ? '/>' : '>';
  return out;
};

Renderer.prototype.text = function(text) {
  return text;
};

/**
 * Parsing & Compiling
 */

function Parser(options) {
  this.tokens = [];
  this.token = null;
  this.options = options || marked.defaults;
  this.options.renderer = this.options.renderer || new Renderer;
  this.renderer = this.options.renderer;
  this.renderer.options = this.options;
}

/**
 * Static Parse Method
 */

Parser.parse = function(src, options, renderer) {
  var parser = new Parser(options, renderer);
  return parser.parse(src);
};

/**
 * Parse Loop
 */

Parser.prototype.parse = function(src) {
  this.inline = new InlineLexer(src.links, this.options, this.renderer);
  this.tokens = src.reverse();

  var out = '';
  while (this.next()) {
    out += this.tok();
  }

  return out;
};

/**
 * Next Token
 */

Parser.prototype.next = function() {
  return this.token = this.tokens.pop();
};

/**
 * Preview Next Token
 */

Parser.prototype.peek = function() {
  return this.tokens[this.tokens.length - 1] || 0;
};

/**
 * Parse Text Tokens
 */

Parser.prototype.parseText = function() {
  var body = this.token.text;

  while (this.peek().type === 'text') {
    body += '\n' + this.next().text;
  }

  return this.inline.output(body);
};

/**
 * Parse Current Token
 */

Parser.prototype.tok = function() {
  switch (this.token.type) {
    case 'space': {
      return '';
    }
    case 'hr': {
      return this.renderer.hr();
    }
    case 'heading': {
      return this.renderer.heading(
        this.inline.output(this.token.text),
        this.token.depth,
        this.token.text);
    }
    case 'code': {
      return this.renderer.code(this.token.text,
        this.token.lang,
        this.token.escaped);
    }
    case 'table': {
      var header = ''
        , body = ''
        , i
        , row
        , cell
        , flags
        , j;

      // header
      cell = '';
      for (i = 0; i < this.token.header.length; i++) {
        flags = { header: true, align: this.token.align[i] };
        cell += this.renderer.tablecell(
          this.inline.output(this.token.header[i]),
          { header: true, align: this.token.align[i] }
        );
      }
      header += this.renderer.tablerow(cell);

      for (i = 0; i < this.token.cells.length; i++) {
        row = this.token.cells[i];

        cell = '';
        for (j = 0; j < row.length; j++) {
          cell += this.renderer.tablecell(
            this.inline.output(row[j]),
            { header: false, align: this.token.align[j] }
          );
        }

        body += this.renderer.tablerow(cell);
      }
      return this.renderer.table(header, body);
    }
    case 'blockquote_start': {
      var body = '';

      while (this.next().type !== 'blockquote_end') {
        body += this.tok();
      }

      return this.renderer.blockquote(body);
    }
    case 'list_start': {
      var body = ''
        , ordered = this.token.ordered;

      while (this.next().type !== 'list_end') {
        body += this.tok();
      }

      return this.renderer.list(body, ordered);
    }
    case 'list_item_start': {
      var body = '';

      while (this.next().type !== 'list_item_end') {
        body += this.token.type === 'text'
          ? this.parseText()
          : this.tok();
      }

      return this.renderer.listitem(body);
    }
    case 'loose_item_start': {
      var body = '';

      while (this.next().type !== 'list_item_end') {
        body += this.tok();
      }

      return this.renderer.listitem(body);
    }
    case 'html': {
      var html = !this.token.pre && !this.options.pedantic
        ? this.inline.output(this.token.text)
        : this.token.text;
      return this.renderer.html(html);
    }
    case 'paragraph': {
      return this.renderer.paragraph(this.inline.output(this.token.text));
    }
    case 'text': {
      return this.renderer.paragraph(this.parseText());
    }
  }
};

/**
 * Helpers
 */

function escape(html, encode) {
  return html
    .replace(!encode ? /&(?!#?\w+;)/g : /&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&#39;');
}

function unescape(html) {
  return html.replace(/&([#\w]+);/g, function(_, n) {
    n = n.toLowerCase();
    if (n === 'colon') return ':';
    if (n.charAt(0) === '#') {
      return n.charAt(1) === 'x'
        ? String.fromCharCode(parseInt(n.substring(2), 16))
        : String.fromCharCode(+n.substring(1));
    }
    return '';
  });
}

function replace(regex, opt) {
  regex = regex.source;
  opt = opt || '';
  return function self(name, val) {
    if (!name) return new RegExp(regex, opt);
    val = val.source || val;
    val = val.replace(/(^|[^\[])\^/g, '$1');
    regex = regex.replace(name, val);
    return self;
  };
}

function noop() {}
noop.exec = noop;

function merge(obj) {
  var i = 1
    , target
    , key;

  for (; i < arguments.length; i++) {
    target = arguments[i];
    for (key in target) {
      if (Object.prototype.hasOwnProperty.call(target, key)) {
        obj[key] = target[key];
      }
    }
  }

  return obj;
}


/**
 * Marked
 */

function marked(src, opt, callback) {
  if (callback || typeof opt === 'function') {
    if (!callback) {
      callback = opt;
      opt = null;
    }

    opt = merge({}, marked.defaults, opt || {});

    var highlight = opt.highlight
      , tokens
      , pending
      , i = 0;

    try {
      tokens = Lexer.lex(src, opt)
    } catch (e) {
      return callback(e);
    }

    pending = tokens.length;

    var done = function(err) {
      if (err) {
        opt.highlight = highlight;
        return callback(err);
      }

      var out;

      try {
        out = Parser.parse(tokens, opt);
      } catch (e) {
        err = e;
      }

      opt.highlight = highlight;

      return err
        ? callback(err)
        : callback(null, out);
    };

    if (!highlight || highlight.length < 3) {
      return done();
    }

    delete opt.highlight;

    if (!pending) return done();

    for (; i < tokens.length; i++) {
      (function(token) {
        if (token.type !== 'code') {
          return --pending || done();
        }
        return highlight(token.text, token.lang, function(err, code) {
          if (err) return done(err);
          if (code == null || code === token.text) {
            return --pending || done();
          }
          token.text = code;
          token.escaped = true;
          --pending || done();
        });
      })(tokens[i]);
    }

    return;
  }
  try {
    if (opt) opt = merge({}, marked.defaults, opt);
    return Parser.parse(Lexer.lex(src, opt), opt);
  } catch (e) {
    e.message += '\nPlease report this to https://github.com/chjj/marked.';
    if ((opt || marked.defaults).silent) {
      return '<p>An error occured:</p><pre>'
        + escape(e.message + '', true)
        + '</pre>';
    }
    throw e;
  }
}

/**
 * Options
 */

marked.options =
marked.setOptions = function(opt) {
  merge(marked.defaults, opt);
  return marked;
};

marked.defaults = {
  gfm: true,
  tables: true,
  breaks: false,
  pedantic: false,
  sanitize: false,
  sanitizer: null,
  mangle: true,
  smartLists: false,
  silent: false,
  highlight: null,
  langPrefix: 'lang-',
  smartypants: false,
  headerPrefix: '',
  renderer: new Renderer,
  xhtml: false
};

/**
 * Expose
 */

marked.Parser = Parser;
marked.parser = Parser.parse;

marked.Renderer = Renderer;

marked.Lexer = Lexer;
marked.lexer = Lexer.lex;

marked.InlineLexer = InlineLexer;
marked.inlineLexer = InlineLexer.output;

marked.parse = marked;

if (typeof module !== 'undefined' && typeof exports === 'object') {
  module.exports = marked;
} else if (typeof define === 'function' && define.amd) {
  define(function() { return marked; });
} else {
  this.marked = marked;
}

}).call(function() {
  return this || (typeof window !== 'undefined' ? window : global);
}());

(function webpackUniversalModuleDefinition(root, factory) {
	if(typeof exports === 'object' && typeof module === 'object')
		module.exports = factory(require("fs"));
	else if(typeof define === 'function' && define.amd)
		define(["fs"], factory);
	else if(typeof exports === 'object')
		exports["Parser"] = factory(require("fs"));
	else
		root["RAML"] = root["RAML"] || {}, root["RAML"]["Parser"] = factory(root["fs"]);
})(this, function(__WEBPACK_EXTERNAL_MODULE_36__) {
return /******/ (function(modules) { // webpackBootstrap
/******/ 	// The module cache
/******/ 	var installedModules = {};
/******/
/******/ 	// The require function
/******/ 	function __webpack_require__(moduleId) {
/******/
/******/ 		// Check if module is in cache
/******/ 		if(installedModules[moduleId])
/******/ 			return installedModules[moduleId].exports;
/******/
/******/ 		// Create a new module (and put it into the cache)
/******/ 		var module = installedModules[moduleId] = {
/******/ 			exports: {},
/******/ 			id: moduleId,
/******/ 			loaded: false
/******/ 		};
/******/
/******/ 		// Execute the module function
/******/ 		modules[moduleId].call(module.exports, module, module.exports, __webpack_require__);
/******/
/******/ 		// Flag the module as loaded
/******/ 		module.loaded = true;
/******/
/******/ 		// Return the exports of the module
/******/ 		return module.exports;
/******/ 	}
/******/
/******/
/******/ 	// expose the modules object (__webpack_modules__)
/******/ 	__webpack_require__.m = modules;
/******/
/******/ 	// expose the module cache
/******/ 	__webpack_require__.c = installedModules;
/******/
/******/ 	// __webpack_public_path__
/******/ 	__webpack_require__.p = "";
/******/
/******/ 	// Load entry module and return exports
/******/ 	return __webpack_require__(0);
/******/ })
/************************************************************************/
/******/ ([
/* 0 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var apiLoader = __webpack_require__(9);
	var path = __webpack_require__(15);
	var PromiseConstructor = __webpack_require__(30);
	/**
	 * RAML 1.0 top-level AST interfaces.
	 */
	exports.api10 = __webpack_require__(12);
	/**
	 * RAML 0.8 top-level AST interfaces.
	 */
	exports.api08 = __webpack_require__(13);
	function loadApiSync(apiPath, arg1, arg2) {
	    return apiLoader.loadApi(apiPath, arg1, arg2).getOrElse(null);
	}
	exports.loadApiSync = loadApiSync;
	function loadRAMLSync(ramlPath, arg1, arg2) {
	    return apiLoader.loadApi(ramlPath, arg1, arg2).getOrElse(null);
	}
	exports.loadRAMLSync = loadRAMLSync;
	function optionsForContent(content, arg2) {
	    return {
	        fsResolver: {
	            content: function (pathStr) {
	                if (pathStr == path.resolve("/", "#local.raml")) {
	                    return content;
	                }
	                if (arg2) {
	                    if (arg2.fsResolver) {
	                        return arg2.fsResolver.content(pathStr);
	                    }
	                }
	            },
	            contentAsync: function (pathStr) {
	                if (pathStr == path.resolve("/", "#local.raml")) {
	                    return Promise.resolve(content);
	                }
	                if (arg2) {
	                    if (arg2.fsResolver) {
	                        return arg2.fsResolver.contentAsync(pathStr);
	                    }
	                }
	            }
	        },
	        httpResolver: arg2 ? arg2.httpResolver : null,
	        rejectOnErrors: arg2 ? arg2.rejectOnErrors : false,
	        attributeDefaults: arg2 ? arg2.attributeDefaults : true
	    };
	}
	/**
	 * Load RAML synchronously. May load both Api and Typed fragments. If the 'rejectOnErrors' option is set to true, [[ApiLoadingError]] is thrown for RAML which contains errors.
	 * @param content content of the raml
	 * @param options Load options  (note you should path a resolvers if you want includes to be resolved)
	 * @return RAMLLanguageElement instance.
	 **/
	function parseRAMLSync(content, arg2) {
	    return apiLoader.loadApi("/#local.raml", [], optionsForContent(content, arg2)).getOrElse(null);
	}
	exports.parseRAMLSync = parseRAMLSync;
	/**
	 * Load RAML asynchronously. May load both Api and Typed fragments. If the 'rejectOnErrors' option is set to true, [[ApiLoadingError]] is thrown for RAML which contains errors.
	 * @param content content of the raml
	 * @param options Load options  (note you should path a resolvers if you want includes to be resolved)
	 * @return RAMLLanguageElement instance.
	 **/
	function parseRAML(content, arg2) {
	    return apiLoader.loadApi("/#local.raml", [], optionsForContent(content, arg2)).getOrElse(null);
	}
	exports.parseRAML = parseRAML;
	function loadApi(apiPath, arg1, arg2) {
	    return apiLoader.loadApiAsync(apiPath, arg1, arg2);
	}
	exports.loadApi = loadApi;
	function loadRAML(ramlPath, arg1, arg2) {
	    return apiLoader.loadRAMLAsync(ramlPath, arg1, arg2);
	}
	exports.loadRAML = loadRAML;
	/**
	 * Gets AST node by runtime type, if runtime type matches any.
	 * @param runtimeType - runtime type to find the match for
	 */
	function getLanguageElementByRuntimeType(runtimeType) {
	    return apiLoader.getLanguageElementByRuntimeType(runtimeType);
	}
	exports.getLanguageElementByRuntimeType = getLanguageElementByRuntimeType;
	/**
	 * Check if the AST node represents fragment
	 */
	function isFragment(node) {
	    return exports.api10.isFragment(node);
	}
	exports.isFragment = isFragment;
	/**
	 * Convert fragment representing node to FragmentDeclaration instance.
	 */
	function asFragment(node) {
	    return exports.api10.asFragment(node);
	}
	exports.asFragment = asFragment;
	/**
	 * High-level AST interfaces.
	 */
	exports.hl = __webpack_require__(10);
	/**
	 * Low-level AST interfaces.
	 */
	exports.ll = __webpack_require__(11);
	/**
	 * Search functionality, operates on high AST level.
	 */
	exports.search = __webpack_require__(1);
	/**
	 * High-level stub node factory methods.
	 */
	exports.stubs = __webpack_require__(2);
	exports.utils = __webpack_require__(3);
	/**
	 * Low-level project factory.
	 */
	exports.project = __webpack_require__(4);
	/**
	 * Helpers for classification of high-level AST entity types.
	 */
	exports.universeHelpers = __webpack_require__(14);
	/**
	 * Definition system.
	 */
	exports.ds = __webpack_require__(35);
	/**
	 * Schema utilities.
	 */
	exports.schema = __webpack_require__(5);
	/**
	 * A set of constants describing definition system entities.
	 * @hidden
	 **/
	exports.universes = exports.ds.universesInfo;
	/**
	 * Exposed parser model modification methods. Operate on high-level.
	 */
	exports.parser = __webpack_require__(6);
	/**
	 * Applies traits and resources types to API on high-level.
	 * Top-level expansion should be performed via calling expand() method of API node.
	 */
	exports.expander = __webpack_require__(7);
	/**
	 * Exposed part of custom methods applied to top-level AST during generation.
	 * Not to be used by parser clients.
	 */
	exports.wrapperHelper = __webpack_require__(8);
	if (typeof Promise === 'undefined' && typeof window !== 'undefined') {
	    window.Promise = PromiseConstructor;
	}
	//# sourceMappingURL=index.js.map

/***/ },
/* 1 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var search = __webpack_require__(29);
	var hlimpl = __webpack_require__(16);
	(function (LocationKind) {
	    LocationKind[LocationKind["VALUE_COMPLETION"] = 0] = "VALUE_COMPLETION";
	    LocationKind[LocationKind["KEY_COMPLETION"] = 1] = "KEY_COMPLETION";
	    LocationKind[LocationKind["PATH_COMPLETION"] = 2] = "PATH_COMPLETION";
	    LocationKind[LocationKind["DIRECTIVE_COMPLETION"] = 3] = "DIRECTIVE_COMPLETION";
	    LocationKind[LocationKind["VERSION_COMPLETION"] = 4] = "VERSION_COMPLETION";
	    LocationKind[LocationKind["ANNOTATION_COMPLETION"] = 5] = "ANNOTATION_COMPLETION";
	    LocationKind[LocationKind["SEQUENCE_KEY_COPLETION"] = 6] = "SEQUENCE_KEY_COPLETION";
	    LocationKind[LocationKind["INCOMMENT"] = 7] = "INCOMMENT";
	})(exports.LocationKind || (exports.LocationKind = {}));
	var LocationKind = exports.LocationKind;
	function determineCompletionKind(text, offset) {
	    return search.determineCompletionKind(text, offset);
	}
	exports.determineCompletionKind = determineCompletionKind;
	function enumValues(c, n) {
	    return search.enumValues(c, n);
	}
	exports.enumValues = enumValues;
	function globalDeclarations(n) {
	    return search.globalDeclarations(n);
	}
	exports.globalDeclarations = globalDeclarations;
	function qName(n, p) {
	    return hlimpl.qName(n, p);
	}
	exports.qName = qName;
	function subTypesWithLocals(t, n) {
	    return search.subTypesWithLocals(t, n);
	}
	exports.subTypesWithLocals = subTypesWithLocals;
	function nodesDeclaringType(t, n) {
	    return search.nodesDeclaringType(t, n);
	}
	exports.nodesDeclaringType = nodesDeclaringType;
	function isExampleNodeContent(n) {
	    return search.isExampleNodeContent(n);
	}
	exports.isExampleNodeContent = isExampleNodeContent;
	function findExampleContentType(n) {
	    return search.findExampleContentType(n);
	}
	exports.findExampleContentType = findExampleContentType;
	function parseDocumentationContent(n, t) {
	    return search.parseDocumentationContent(n, t);
	}
	exports.parseDocumentationContent = parseDocumentationContent;
	function referenceTargets(p0, c) {
	    return search.referenceTargets(p0, c);
	}
	exports.referenceTargets = referenceTargets;
	function findUsages(unit, offset) {
	    return search.findUsages(unit, offset);
	}
	exports.findUsages = findUsages;
	function findDeclaration(unit, offset, nodePart) {
	    return search.findDeclaration(unit, offset, nodePart);
	}
	exports.findDeclaration = findDeclaration;
	//# sourceMappingURL=searchProxy.js.map

/***/ },
/* 2 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var hlimpl = __webpack_require__(16);
	var jsyaml = __webpack_require__(23);
	var stubs = __webpack_require__(17);
	function createStubNode(t, p, key, unit) {
	    if (key === void 0) { key = null; }
	    return stubs.createStubNode(t, p, key, unit);
	}
	exports.createStubNode = createStubNode;
	function createStub(parent, property, key) {
	    return stubs.createStub(parent, property, key);
	}
	exports.createStub = createStub;
	function createStubNoParentPatch(parent, property, key) {
	    return stubs.createStub0(parent, property, key);
	}
	exports.createStubNoParentPatch = createStubNoParentPatch;
	function createResourceStub(parent, key) {
	    return stubs.createResourceStub(parent, key);
	}
	exports.createResourceStub = createResourceStub;
	function createMethodStub(parent, key) {
	    return stubs.createMethodStub(parent, key);
	}
	exports.createMethodStub = createMethodStub;
	function createResponseStub(parent, key) {
	    return stubs.createResourceStub(parent, key);
	}
	exports.createResponseStub = createResponseStub;
	function createBodyStub(parent, key) {
	    return stubs.createBodyStub(parent, key);
	}
	exports.createBodyStub = createBodyStub;
	function createUriParameterStub(parent, key) {
	    return stubs.createUriParameterStub(parent, key);
	}
	exports.createUriParameterStub = createUriParameterStub;
	function createQueryParameterStub(parent, key) {
	    return stubs.createQueryParameterStub(parent, key);
	}
	exports.createQueryParameterStub = createQueryParameterStub;
	function createASTPropImpl(node, parent, _def, _prop, fk) {
	    if (fk === void 0) { fk = false; }
	    return new hlimpl.ASTPropImpl(node, parent, _def, _prop, fk);
	}
	exports.createASTPropImpl = createASTPropImpl;
	function createASTNodeImpl(node, parent, _def, _prop) {
	    return new hlimpl.ASTNodeImpl(node, parent, _def, _prop);
	}
	exports.createASTNodeImpl = createASTNodeImpl;
	function createVirtualASTPropImpl(node, parent, _def, _prop) {
	    return new VirtualAttr(node, parent, _def, _prop);
	}
	exports.createVirtualASTPropImpl = createVirtualASTPropImpl;
	function createVirtualNodeImpl(node, parent, _def, _prop) {
	    return new VirtualNode(node, parent, _def, _prop);
	}
	exports.createVirtualNodeImpl = createVirtualNodeImpl;
	var VirtualAttr = (function (_super) {
	    __extends(VirtualAttr, _super);
	    function VirtualAttr() {
	        _super.apply(this, arguments);
	    }
	    VirtualAttr.prototype.value = function () {
	        return "";
	    };
	    return VirtualAttr;
	}(hlimpl.ASTPropImpl));
	var VirtualNode = (function (_super) {
	    __extends(VirtualNode, _super);
	    function VirtualNode() {
	        _super.apply(this, arguments);
	    }
	    VirtualNode.prototype.value = function () {
	        return "";
	    };
	    return VirtualNode;
	}(hlimpl.ASTNodeImpl));
	function createMapping(name, value) {
	    return jsyaml.createMapping(name, value);
	}
	exports.createMapping = createMapping;
	function createMap() {
	    return jsyaml.createMap([]);
	}
	exports.createMap = createMap;
	function createAttr(_property, val) {
	    return stubs.createAttr(_property, val);
	}
	exports.createAttr = createAttr;
	//# sourceMappingURL=stubProxy.js.map

/***/ },
/* 3 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var rs = __webpack_require__(24);
	var hlimpl = __webpack_require__(16);
	var defs = __webpack_require__(35);
	var jsyaml = __webpack_require__(23);
	var expander = __webpack_require__(26);
	var linter = __webpack_require__(27);
	var builder = __webpack_require__(28);
	function hasAsyncRequests() {
	    return rs.hasAsyncRequests();
	}
	exports.hasAsyncRequests = hasAsyncRequests;
	function addLoadCallback(x) {
	    rs.addLoadCallback(x);
	}
	exports.addLoadCallback = addLoadCallback;
	function getTransformerNames() {
	    return expander.getTransformNames();
	}
	exports.getTransformerNames = getTransformerNames;
	exports.updateType = function (node) {
	    var type = builder.doDescrimination(node);
	    if (type == null && node.property()) {
	        type = node.property().range();
	    }
	    if (type) {
	        node.patchType(type);
	    }
	};
	function getFragmentDefenitionName(node) {
	    return hlimpl.getFragmentDefenitionName(node);
	}
	exports.getFragmentDefenitionName = getFragmentDefenitionName;
	function genStructuredValue(name, parent, pr) {
	    if (pr.range() instanceof defs.ReferenceType) {
	        var t = pr.range();
	        var mockNode = jsyaml.createNode(name);
	        return new hlimpl.StructuredValue(mockNode, parent, pr);
	    }
	    else
	        return name;
	}
	exports.genStructuredValue = genStructuredValue;
	function parseUrl(u) {
	    return new linter.UrlParameterNameValidator().parseUrl(u);
	}
	exports.parseUrl = parseUrl;
	//# sourceMappingURL=utils.js.map

/***/ },
/* 4 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var limpl = __webpack_require__(23);
	function createProject(path, r, h) {
	    return new limpl.Project(path, r, h);
	}
	exports.createProject = createProject;
	//# sourceMappingURL=project.js.map

/***/ },
/* 5 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var def = __webpack_require__(35);
	var refResolvers = __webpack_require__(25);
	var schemaGenImpl = __webpack_require__(18);
	var contentprovider = __webpack_require__(31);
	var su = def.getSchemaUtils();
	function createSchema(c, u) {
	    return su.createSchema(c, new contentprovider.ContentProvider(u));
	}
	exports.createSchema = createSchema;
	function getXMLSchema(c) {
	    return su.getXMLSchema(c);
	}
	exports.getXMLSchema = getXMLSchema;
	function getJSONSchema(c, u) {
	    return su.getJSONSchema(c, new contentprovider.ContentProvider(u));
	}
	exports.getJSONSchema = getJSONSchema;
	function completeReference(includePath, includeReference, content) {
	    return refResolvers.completeReference(includePath, includeReference, content);
	}
	exports.completeReference = completeReference;
	function getIncludePath(p) {
	    return refResolvers.getIncludePath(p);
	}
	exports.getIncludePath = getIncludePath;
	function getIncludeReference(p) {
	    return refResolvers.getIncludeReference(p);
	}
	exports.getIncludeReference = getIncludeReference;
	function createSchemaModelGenerator() {
	    return new schemaGenImpl.SchemaToModelGenerator();
	}
	exports.createSchemaModelGenerator = createSchemaModelGenerator;
	function createModelToSchemaGenerator() {
	    return new schemaGenImpl.ModelToSchemaGenerator();
	}
	exports.createModelToSchemaGenerator = createModelToSchemaGenerator;
	//# sourceMappingURL=schema.js.map

/***/ },
/* 6 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	exports.modify = __webpack_require__(19);
	//# sourceMappingURL=parser.js.map

/***/ },
/* 7 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var ex = __webpack_require__(26);
	function expandTraitsAndResourceTypes(api) {
	    return ex.expandTraitsAndResourceTypes(api);
	}
	exports.expandTraitsAndResourceTypes = expandTraitsAndResourceTypes;
	//# sourceMappingURL=expanderStub.js.map

/***/ },
/* 8 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var wh = __webpack_require__(32);
	function completeRelativeUri(res) {
	    return wh.completeRelativeUri(res);
	}
	exports.completeRelativeUri = completeRelativeUri;
	//  function schema(body:api.TypeDeclaration, api:api.Api):Opt<SchemaDef> {
	//     return wh.schema(body,api);
	// } 
	//# sourceMappingURL=wrapperHelperStub.js.map

/***/ },
/* 9 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../typings/main.d.ts" />
	"use strict";
	var RamlWrapper1Impl = __webpack_require__(20);
	var path = __webpack_require__(15);
	var Opt = __webpack_require__(21);
	var jsyaml = __webpack_require__(23);
	var hlimpl = __webpack_require__(16);
	var llimpl = __webpack_require__(23);
	var expander = __webpack_require__(26);
	var universeDef = __webpack_require__(22);
	var ramlServices = __webpack_require__(33);
	var universeProvider = __webpack_require__(34);
	/***
	 * Load API synchronously. Detects RAML version and uses corresponding parser.
	 * @param apiPath Path to API: local file system path or Web URL
	 * @param options Load options
	 * @return Opt&lt;Api&gt;, where Api belongs to RAML 1.0 or RAML 0.8 model.
	 ***/
	function loadApi(apiPath, arg1, arg2) {
	    var api = loadRAMLInternal(apiPath, arg1, arg2);
	    // if (false) {
	    //     //TODO loaded RAML is API
	    //     throw new Error("Loaded RAML is not API");
	    // } else {
	    return new Opt(api);
	    // }
	}
	exports.loadApi = loadApi;
	/***
	 * Load RAML synchronously. Detects RAML version and uses corresponding parser.
	 * @param ramlPath Path to RAML: local file system path or Web URL
	 * @param options Load options
	 * @return Opt&lt;RAMLLanguageElement&gt;, where RAMLLanguageElement belongs to RAML 1.0 or RAML 0.8 model.
	 ***/
	function loadRAML(ramlPath, arg1, arg2) {
	    var result = loadRAMLInternal(ramlPath, arg1, arg2);
	    return new Opt(result);
	}
	exports.loadRAML = loadRAML;
	function loadRAMLInternal(apiPath, arg1, arg2) {
	    var gotArray = Array.isArray(arg1);
	    var extensionsAndOverlays = (gotArray ? arg1 : null);
	    var options = (gotArray ? arg2 : arg1);
	    options = options || {};
	    var project = getProject(apiPath, options);
	    var pr = apiPath.indexOf("://");
	    var unitName = (pr != -1 && pr < 6) ? apiPath : path.basename(apiPath);
	    var unit = project.unit(unitName);
	    if (arg2 && !extensionsAndOverlays) {
	        extensionsAndOverlays = null;
	    }
	    var api;
	    if (unit) {
	        if (extensionsAndOverlays && extensionsAndOverlays.length > 0) {
	            var extensionUnits = [];
	            extensionsAndOverlays.forEach(function (currentPath) {
	                if (!currentPath || currentPath.trim().length == 0) {
	                    throw new Error("Extensions and overlays list should contain legal file paths");
	                }
	            });
	            extensionsAndOverlays.forEach(function (unitPath) {
	                extensionUnits.push(project.unit(path.basename(unitPath)));
	            });
	            //calling to perform the checks, we do not actually need the api itself
	            extensionUnits.forEach(function (extensionUnit) { return toApi(extensionUnit, options); });
	            api = toApi(expander.mergeAPIs(unit, extensionUnits, hlimpl.OverlayMergeMode.MERGE), options);
	        }
	        else {
	            api = toApi(unit, options);
	            api.highLevel().setMergeMode(hlimpl.OverlayMergeMode.MERGE);
	        }
	    }
	    if (!unit) {
	        throw new Error("Can not resolve :" + apiPath);
	    }
	    if (options.rejectOnErrors && api && api.errors().filter(function (x) { return !x.isWarning; }).length) {
	        throw toError(api);
	    }
	    if (options.attributeDefaults != null && api) {
	        api.setAttributeDefaults(options.attributeDefaults);
	    }
	    else if (api) {
	        api.setAttributeDefaults(true);
	    }
	    return api;
	}
	/***
	 * Load API asynchronously. Detects RAML version and uses corresponding parser.
	 * @param apiPath Path to API: local file system path or Web URL
	 * @param options Load options
	 * @return Promise&lt;Api&gt;, where Api belongs to RAML 1.0 or RAML 0.8 model.
	 ***/
	function loadApiAsync(apiPath, arg1, arg2) {
	    var ramlPromise = loadRAMLAsync(apiPath, arg1, arg2);
	    return ramlPromise.then(function (loadedRaml) {
	        // if (false) {
	        //     //TODO check that loaded RAML is API
	        //     return Promise.reject("Specified RAML is not API");
	        // } else {
	        return loadedRaml;
	        // }
	    });
	}
	exports.loadApiAsync = loadApiAsync;
	/***
	 * Load API asynchronously. Detects RAML version and uses corresponding parser.
	 * @param ramlPath Path to RAML: local file system path or Web URL
	 * @param options Load options
	 * @return Promise&lt;RAMLLanguageElement&gt;, where RAMLLanguageElement belongs to RAML 1.0 or RAML 0.8 model.
	 ***/
	function loadRAMLAsync(ramlPath, arg1, arg2) {
	    var gotArray = Array.isArray(arg1);
	    var extensionsAndOverlays = (gotArray ? arg1 : null);
	    var options = (gotArray ? arg2 : arg1);
	    options = options || {};
	    var project = getProject(ramlPath, options);
	    var pr = ramlPath.indexOf("://");
	    var unitName = (pr != -1 && pr < 6) ? ramlPath : path.basename(ramlPath);
	    if (arg2 && !extensionsAndOverlays) {
	        extensionsAndOverlays = null;
	    }
	    if (!extensionsAndOverlays || extensionsAndOverlays.length == 0) {
	        return fetchAndLoadApiAsync(project, unitName, options).then(function (masterApi) {
	            masterApi.highLevel().setMergeMode(hlimpl.OverlayMergeMode.MERGE);
	            if (options.attributeDefaults != null && masterApi) {
	                masterApi.setAttributeDefaults(options.attributeDefaults);
	            }
	            else if (masterApi) {
	                masterApi.setAttributeDefaults(true);
	            }
	            return masterApi;
	        });
	    }
	    else {
	        extensionsAndOverlays.forEach(function (currentPath) {
	            if (!currentPath || currentPath.trim().length == 0) {
	                throw new Error("Extensions and overlays list should contain legal file paths");
	            }
	        });
	        return fetchAndLoadApiAsync(project, unitName, options).then(function (masterApi) {
	            var apiPromises = [];
	            extensionsAndOverlays.forEach(function (extensionUnitPath) {
	                apiPromises.push(fetchAndLoadApiAsync(project, path.basename(extensionUnitPath), options));
	            });
	            return Promise.all(apiPromises).then(function (apis) {
	                var overlayUnits = [];
	                apis.forEach(function (currentApi) { return overlayUnits.push(currentApi.highLevel().lowLevel().unit()); });
	                var result = expander.mergeAPIs(masterApi.highLevel().lowLevel().unit(), overlayUnits, hlimpl.OverlayMergeMode.MERGE);
	                if (options.attributeDefaults != null && result) {
	                    result.setAttributeDefaults(options.attributeDefaults);
	                }
	                else if (result) {
	                    result.setAttributeDefaults(true);
	                }
	                return result;
	            }).then(function (mergedHighLevel) {
	                return toApi(mergedHighLevel, options);
	            });
	        });
	    }
	}
	exports.loadRAMLAsync = loadRAMLAsync;
	/**
	 * Gets AST node by runtime type, if runtime type matches any.
	 * @param runtimeType
	 */
	function getLanguageElementByRuntimeType(runtimeType) {
	    if (runtimeType == null) {
	        return null;
	    }
	    var highLevelNode = runtimeType.getAdapter(ramlServices.RAMLService).getDeclaringNode();
	    if (highLevelNode == null) {
	        return null;
	    }
	    return highLevelNode.wrapperNode();
	}
	exports.getLanguageElementByRuntimeType = getLanguageElementByRuntimeType;
	function fetchAndLoadApiAsync(project, unitName, options) {
	    return llimpl.fetchIncludesAndMasterAsync(project, unitName).then(function (x) {
	        try {
	            var api = toApi(x, options);
	            if (options.rejectOnErrors && api && api.errors().filter(function (x) { return !x.isWarning; }).length) {
	                return Promise.reject(toError(api));
	            }
	            return api;
	        }
	        catch (err) {
	            return Promise.reject(err);
	        }
	    });
	}
	function getProject(apiPath, options) {
	    options = options || {};
	    var includeResolver = options.fsResolver;
	    var httpResolver = options.httpResolver;
	    var projectRoot = path.dirname(apiPath);
	    var project = new jsyaml.Project(projectRoot, includeResolver, httpResolver);
	    return project;
	}
	;
	function toApi(unitOrHighlevel, options, checkApisOverlays) {
	    if (checkApisOverlays === void 0) { checkApisOverlays = false; }
	    if (!unitOrHighlevel) {
	        return null;
	    }
	    var unit = null;
	    var highLevel = null;
	    if (unitOrHighlevel.isRAMLUnit) {
	        unit = unitOrHighlevel;
	    }
	    else {
	        highLevel = unitOrHighlevel;
	        unit = highLevel.lowLevel().unit();
	    }
	    var api;
	    var contents = unit.contents();
	    var ramlFirstLine = contents.match(/^\s*#%RAML\s+(\d\.\d)\s*(\w*)\s*$/m);
	    if (!ramlFirstLine) {
	        //TODO throw sensible error
	        return null;
	    }
	    var verStr = ramlFirstLine[1];
	    var ramlFileType = ramlFirstLine[2];
	    var typeName;
	    var apiImpl;
	    var ramlVersion;
	    if (verStr == '0.8') {
	        ramlVersion = 'RAML08';
	    }
	    else if (verStr == '1.0') {
	        ramlVersion = 'RAML10';
	    }
	    if (!ramlVersion) {
	        throw new Error("Unknown version of RAML expected to see one of '#%RAML 0.8' or '#%RAML 1.0'");
	    }
	    if (ramlVersion == 'RAML08' && checkApisOverlays) {
	        throw new Error('Extensions and overlays are not supported in RAML 0.8.');
	    }
	    //if (!ramlFileType || ramlFileType.trim() === "") {
	    //    if (verStr=='0.8') {
	    //        typeName = universeDef.Universe08.Api.name;
	    //        apiImpl = RamlWrapper08.ApiImpl;
	    //    } else if(verStr=='1.0'){
	    //        typeName = universeDef.Universe10.Api.name;
	    //        apiImpl = RamlWrapper1.ApiImpl;
	    //    }
	    //} else if (ramlFileType === "Overlay") {
	    //    apiImpl = RamlWrapper1.OverlayImpl;
	    //    typeName = universeDef.Universe10.Overlay.name;
	    //} else if (ramlFileType === "Extension") {
	    //    apiImpl = RamlWrapper1.ExtensionImpl;
	    //    typeName = universeDef.Universe10.Extension.name;
	    //}
	    var universe = universeProvider(ramlVersion);
	    var apiType = universe.type(typeName);
	    if (!highLevel) {
	        highLevel = hlimpl.fromUnit(unit);
	    }
	    //api = new apiImpl(highLevel);
	    api = highLevel.wrapperNode();
	    return api;
	}
	;
	function toError(api) {
	    var error = new Error('Api contains errors.');
	    error.parserErrors = api.errors();
	    return error;
	}
	exports.toError = toError;
	function loadApis1(projectRoot, cacheChildren, expandTraitsAndResourceTypes) {
	    if (cacheChildren === void 0) { cacheChildren = false; }
	    if (expandTraitsAndResourceTypes === void 0) { expandTraitsAndResourceTypes = true; }
	    var universe = universeProvider("RAML10");
	    var apiType = universe.type(universeDef.Universe10.Api.name);
	    var p = new jsyaml.Project(projectRoot);
	    var result = [];
	    p.units().forEach(function (x) {
	        var lowLevel = x.ast();
	        if (cacheChildren) {
	            lowLevel = llimpl.toChildCahcingNode(lowLevel);
	        }
	        var api = new RamlWrapper1Impl.ApiImpl(new hlimpl.ASTNodeImpl(lowLevel, null, apiType, null));
	        if (expandTraitsAndResourceTypes) {
	            api = expander.expandTraitsAndResourceTypes(api);
	        }
	        result.push(api);
	    });
	    return result;
	}
	exports.loadApis1 = loadApis1;
	//# sourceMappingURL=apiLoader.js.map

/***/ },
/* 10 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	(function (NodeKind) {
	    NodeKind[NodeKind["BASIC"] = 0] = "BASIC";
	    NodeKind[NodeKind["NODE"] = 1] = "NODE";
	    NodeKind[NodeKind["ATTRIBUTE"] = 2] = "ATTRIBUTE";
	})(exports.NodeKind || (exports.NodeKind = {}));
	var NodeKind = exports.NodeKind;
	(function (RAMLVersion) {
	    RAMLVersion[RAMLVersion["RAML10"] = 0] = "RAML10";
	    RAMLVersion[RAMLVersion["RAML08"] = 1] = "RAML08";
	})(exports.RAMLVersion || (exports.RAMLVersion = {}));
	var RAMLVersion = exports.RAMLVersion;
	(function (IssueCode) {
	    IssueCode[IssueCode["UNRESOLVED_REFERENCE"] = 0] = "UNRESOLVED_REFERENCE";
	    IssueCode[IssueCode["YAML_ERROR"] = 1] = "YAML_ERROR";
	    IssueCode[IssueCode["UNKNOWN_NODE"] = 2] = "UNKNOWN_NODE";
	    IssueCode[IssueCode["MISSING_REQUIRED_PROPERTY"] = 3] = "MISSING_REQUIRED_PROPERTY";
	    IssueCode[IssueCode["PROPERTY_EXPECT_TO_HAVE_SINGLE_VALUE"] = 4] = "PROPERTY_EXPECT_TO_HAVE_SINGLE_VALUE";
	    //TODO IMPLEMENT
	    IssueCode[IssueCode["KEY_SHOULD_BE_UNIQUE_INTHISCONTEXT"] = 5] = "KEY_SHOULD_BE_UNIQUE_INTHISCONTEXT";
	    IssueCode[IssueCode["UNABLE_TO_RESOLVE_INCLUDE_FILE"] = 6] = "UNABLE_TO_RESOLVE_INCLUDE_FILE";
	    IssueCode[IssueCode["INVALID_VALUE_SCHEMA"] = 7] = "INVALID_VALUE_SCHEMA";
	    IssueCode[IssueCode["MISSED_CONTEXT_REQUIREMENT"] = 8] = "MISSED_CONTEXT_REQUIREMENT";
	    IssueCode[IssueCode["NODE_HAS_VALUE"] = 9] = "NODE_HAS_VALUE";
	    IssueCode[IssueCode["ONLY_OVERRIDE_ALLOWED"] = 10] = "ONLY_OVERRIDE_ALLOWED";
	    IssueCode[IssueCode["ILLEGAL_PROPERTY_VALUE"] = 11] = "ILLEGAL_PROPERTY_VALUE";
	    IssueCode[IssueCode["ILLEGAL_PROPERTY"] = 12] = "ILLEGAL_PROPERTY";
	    IssueCode[IssueCode["INVALID_PROPERTY"] = 13] = "INVALID_PROPERTY";
	})(exports.IssueCode || (exports.IssueCode = {}));
	var IssueCode = exports.IssueCode;
	//# sourceMappingURL=highLevelAST.js.map

/***/ },
/* 11 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var ASTDelta = (function () {
	    function ASTDelta() {
	    }
	    return ASTDelta;
	}());
	exports.ASTDelta = ASTDelta;
	(function (CommandKind) {
	    CommandKind[CommandKind["ADD_CHILD"] = 0] = "ADD_CHILD";
	    CommandKind[CommandKind["REMOVE_CHILD"] = 1] = "REMOVE_CHILD";
	    CommandKind[CommandKind["MOVE_CHILD"] = 2] = "MOVE_CHILD";
	    CommandKind[CommandKind["CHANGE_KEY"] = 3] = "CHANGE_KEY";
	    CommandKind[CommandKind["CHANGE_VALUE"] = 4] = "CHANGE_VALUE";
	    CommandKind[CommandKind["INIT_RAML_FILE"] = 5] = "INIT_RAML_FILE";
	})(exports.CommandKind || (exports.CommandKind = {}));
	var CommandKind = exports.CommandKind;
	var TextChangeCommand = (function () {
	    function TextChangeCommand(offset, replacementLength, text, unit, target) {
	        if (target === void 0) { target = null; }
	        this.offset = offset;
	        this.replacementLength = replacementLength;
	        this.text = text;
	        this.unit = unit;
	        this.target = target;
	    }
	    return TextChangeCommand;
	}());
	exports.TextChangeCommand = TextChangeCommand;
	var CompositeCommand = (function () {
	    function CompositeCommand() {
	        this.commands = [];
	    }
	    return CompositeCommand;
	}());
	exports.CompositeCommand = CompositeCommand;
	(function (InsertionPointType) {
	    InsertionPointType[InsertionPointType["NONE"] = 0] = "NONE";
	    InsertionPointType[InsertionPointType["START"] = 1] = "START";
	    InsertionPointType[InsertionPointType["END"] = 2] = "END";
	    InsertionPointType[InsertionPointType["POINT"] = 3] = "POINT";
	})(exports.InsertionPointType || (exports.InsertionPointType = {}));
	var InsertionPointType = exports.InsertionPointType;
	var ASTChangeCommand = (function () {
	    function ASTChangeCommand(kind, target, value, position) {
	        this.toSeq = false;
	        this.kind = kind;
	        this.target = target;
	        this.value = value;
	        this.position = position;
	    }
	    return ASTChangeCommand;
	}());
	exports.ASTChangeCommand = ASTChangeCommand;
	function setAttr(t, value) {
	    return new ASTChangeCommand(CommandKind.CHANGE_VALUE, t, value, -1);
	}
	exports.setAttr = setAttr;
	function setAttrStructured(t, value) {
	    return new ASTChangeCommand(CommandKind.CHANGE_VALUE, t, value.lowLevel(), -1);
	}
	exports.setAttrStructured = setAttrStructured;
	function setKey(t, value) {
	    return new ASTChangeCommand(CommandKind.CHANGE_KEY, t, value, -1);
	}
	exports.setKey = setKey;
	function removeNode(t, child) {
	    return new ASTChangeCommand(CommandKind.REMOVE_CHILD, t, child, -1);
	}
	exports.removeNode = removeNode;
	function insertNode(t, child, insertAfter, toSeq) {
	    if (insertAfter === void 0) { insertAfter = null; }
	    if (toSeq === void 0) { toSeq = false; }
	    var s = new ASTChangeCommand(CommandKind.ADD_CHILD, t, child, -1);
	    s.insertionPoint = insertAfter;
	    s.toSeq = toSeq;
	    return s;
	}
	exports.insertNode = insertNode;
	function initRamlFile(root, newroot) {
	    return new ASTChangeCommand(CommandKind.INIT_RAML_FILE, root, newroot, -1);
	}
	exports.initRamlFile = initRamlFile;
	var LineMapperImpl = (function () {
	    function LineMapperImpl(content, absPath) {
	        this.content = content;
	        this.absPath = absPath;
	    }
	    LineMapperImpl.prototype.position = function (_pos) {
	        var pos = _pos;
	        this.initMapping();
	        for (var i = 0; i < this.mapping.length; i++) {
	            var lineLength = this.mapping[i];
	            if (pos < lineLength) {
	                return {
	                    line: i,
	                    column: pos,
	                    position: _pos
	                };
	            }
	            pos -= lineLength;
	        }
	        if (pos == 1) {
	            //sometimes YAML library reports an error at a position of document length + 1, no idea what they want
	            //to tell us that way
	            return {
	                line: this.mapping.length - 1,
	                column: this.mapping[this.mapping.length - 1] - 1,
	                position: _pos - 1
	            };
	        }
	        throw new Error("Character position exceeds text length: " + _pos + " > + " + this.content.length + ".\nUnit path: " + this.absPath);
	    };
	    LineMapperImpl.prototype.initMapping = function () {
	        if (this.mapping != null) {
	            return;
	        }
	        if (this.content == null) {
	            throw new Error("Line Mapper has been given null content" + (this.absPath != null
	                ? ('. Path: ' + this.absPath) : ' and null path.'));
	        }
	        this.mapping = [];
	        var ind = 0;
	        var l = this.content.length;
	        for (var i = 0; i < l; i++) {
	            if (this.content.charAt(i) == '\r') {
	                if (i < l - 1 && this.content.charAt(i + 1) == '\n') {
	                    this.mapping.push(i - ind + 2);
	                    ind = i + 2;
	                    i++;
	                }
	                else {
	                    this.mapping.push(i - ind + 1);
	                    ind = i + 1;
	                }
	            }
	            else if (this.content.charAt(i) == '\n') {
	                this.mapping.push(i - ind + 1);
	                ind = i + 1;
	            }
	        }
	        this.mapping.push(l - ind);
	    };
	    return LineMapperImpl;
	}());
	exports.LineMapperImpl = LineMapperImpl;
	//# sourceMappingURL=lowLevelAST.js.map

/***/ },
/* 12 */
/***/ function(module, exports, __webpack_require__) {

	/**
	 * <p>See <a href="http://raml.org">http://raml.org</a> for more information about RAML.</p>
	 *
	 * <p>This parser is at a beta state of development, as part of the API Workbench development cycle (<a href="http://apiworkbench.com">http://apiworkbench.com</a>).</p>
	 *
	 * <p><a href="https://github.com/raml-org/raml-js-parser-2/blob/master/documentation/GettingStarted.md">Getting Started Guide</a> describes the first steps with the parser.</p>
	 *
	 * <h2>Installation</h2>
	 *
	 * <pre><code>git clone https://github.com/raml-org/raml-js-parser-2
	 *
	 * cd raml-js-parser-2
	 *
	 * npm install
	 *
	 * node test/test.js  //here you should observe JSON representation of XKCD API in your console
	 *
	 * node test/testAsync.js  //same as above but in asynchronous mode
	 * </code></pre>
	 *
	 * <h2>Usage</h2>
	 *
	 * <ul>
	 * <li>For parser usage example refer to <code>test/test.js</code></li>
	 * <li>For asynchrounous usage example refer to <code>test/testAsync.js</code></li>
	 * </ul>
	 **/
	"use strict";
	/**
	 * Custom type guard for Api. Returns true if node is instance of Api. Returns false otherwise.
	 * Also returns false for super interfaces of Api.
	 */
	function isApi(node) {
	    return node.kind() == "Api" && node.RAMLVersion() == "RAML10";
	}
	exports.isApi = isApi;
	/**
	 * Custom type guard for LibraryBase. Returns true if node is instance of LibraryBase. Returns false otherwise.
	 * Also returns false for super interfaces of LibraryBase.
	 */
	function isLibraryBase(node) {
	    return node.kind() == "LibraryBase" && node.RAMLVersion() == "RAML10";
	}
	exports.isLibraryBase = isLibraryBase;
	/**
	 * Custom type guard for Annotable. Returns true if node is instance of Annotable. Returns false otherwise.
	 * Also returns false for super interfaces of Annotable.
	 */
	function isAnnotable(node) {
	    return node.kind() == "Annotable" && node.RAMLVersion() == "RAML10";
	}
	exports.isAnnotable = isAnnotable;
	/**
	 * Custom type guard for AnnotationRef. Returns true if node is instance of AnnotationRef. Returns false otherwise.
	 * Also returns false for super interfaces of AnnotationRef.
	 */
	function isAnnotationRef(node) {
	    return node.kind() == "AnnotationRef" && node.RAMLVersion() == "RAML10";
	}
	exports.isAnnotationRef = isAnnotationRef;
	/**
	 * Custom type guard for Reference. Returns true if node is instance of Reference. Returns false otherwise.
	 * Also returns false for super interfaces of Reference.
	 */
	function isReference(node) {
	    return node.kind() == "Reference" && node.RAMLVersion() == "RAML10";
	}
	exports.isReference = isReference;
	/**
	 * Custom type guard for ValueType. Returns true if node is instance of ValueType. Returns false otherwise.
	 * Also returns false for super interfaces of ValueType.
	 */
	function isValueType(node) {
	    return node.kind() == "ValueType" && node.RAMLVersion() == "RAML10";
	}
	exports.isValueType = isValueType;
	/**
	 * Custom type guard for StringType. Returns true if node is instance of StringType. Returns false otherwise.
	 * Also returns false for super interfaces of StringType.
	 */
	function isStringType(node) {
	    return node.kind() == "StringType" && node.RAMLVersion() == "RAML10";
	}
	exports.isStringType = isStringType;
	/**
	 * Custom type guard for UriTemplate. Returns true if node is instance of UriTemplate. Returns false otherwise.
	 * Also returns false for super interfaces of UriTemplate.
	 */
	function isUriTemplate(node) {
	    return node.kind() == "UriTemplate" && node.RAMLVersion() == "RAML10";
	}
	exports.isUriTemplate = isUriTemplate;
	/**
	 * Custom type guard for RelativeUriString. Returns true if node is instance of RelativeUriString. Returns false otherwise.
	 * Also returns false for super interfaces of RelativeUriString.
	 */
	function isRelativeUriString(node) {
	    return node.kind() == "RelativeUriString" && node.RAMLVersion() == "RAML10";
	}
	exports.isRelativeUriString = isRelativeUriString;
	/**
	 * Custom type guard for FullUriTemplateString. Returns true if node is instance of FullUriTemplateString. Returns false otherwise.
	 * Also returns false for super interfaces of FullUriTemplateString.
	 */
	function isFullUriTemplateString(node) {
	    return node.kind() == "FullUriTemplateString" && node.RAMLVersion() == "RAML10";
	}
	exports.isFullUriTemplateString = isFullUriTemplateString;
	/**
	 * Custom type guard for StatusCodeString. Returns true if node is instance of StatusCodeString. Returns false otherwise.
	 * Also returns false for super interfaces of StatusCodeString.
	 */
	function isStatusCodeString(node) {
	    return node.kind() == "StatusCodeString" && node.RAMLVersion() == "RAML10";
	}
	exports.isStatusCodeString = isStatusCodeString;
	/**
	 * Custom type guard for FixedUriString. Returns true if node is instance of FixedUriString. Returns false otherwise.
	 * Also returns false for super interfaces of FixedUriString.
	 */
	function isFixedUriString(node) {
	    return node.kind() == "FixedUriString" && node.RAMLVersion() == "RAML10";
	}
	exports.isFixedUriString = isFixedUriString;
	/**
	 * Custom type guard for ContentType. Returns true if node is instance of ContentType. Returns false otherwise.
	 * Also returns false for super interfaces of ContentType.
	 */
	function isContentType(node) {
	    return node.kind() == "ContentType" && node.RAMLVersion() == "RAML10";
	}
	exports.isContentType = isContentType;
	/**
	 * Custom type guard for MarkdownString. Returns true if node is instance of MarkdownString. Returns false otherwise.
	 * Also returns false for super interfaces of MarkdownString.
	 */
	function isMarkdownString(node) {
	    return node.kind() == "MarkdownString" && node.RAMLVersion() == "RAML10";
	}
	exports.isMarkdownString = isMarkdownString;
	/**
	 * Custom type guard for SchemaString. Returns true if node is instance of SchemaString. Returns false otherwise.
	 * Also returns false for super interfaces of SchemaString.
	 */
	function isSchemaString(node) {
	    return node.kind() == "SchemaString" && node.RAMLVersion() == "RAML10";
	}
	exports.isSchemaString = isSchemaString;
	/**
	 * Custom type guard for MimeType. Returns true if node is instance of MimeType. Returns false otherwise.
	 * Also returns false for super interfaces of MimeType.
	 */
	function isMimeType(node) {
	    return node.kind() == "MimeType" && node.RAMLVersion() == "RAML10";
	}
	exports.isMimeType = isMimeType;
	/**
	 * Custom type guard for AnyType. Returns true if node is instance of AnyType. Returns false otherwise.
	 * Also returns false for super interfaces of AnyType.
	 */
	function isAnyType(node) {
	    return node.kind() == "AnyType" && node.RAMLVersion() == "RAML10";
	}
	exports.isAnyType = isAnyType;
	/**
	 * Custom type guard for NumberType. Returns true if node is instance of NumberType. Returns false otherwise.
	 * Also returns false for super interfaces of NumberType.
	 */
	function isNumberType(node) {
	    return node.kind() == "NumberType" && node.RAMLVersion() == "RAML10";
	}
	exports.isNumberType = isNumberType;
	/**
	 * Custom type guard for IntegerType. Returns true if node is instance of IntegerType. Returns false otherwise.
	 * Also returns false for super interfaces of IntegerType.
	 */
	function isIntegerType(node) {
	    return node.kind() == "IntegerType" && node.RAMLVersion() == "RAML10";
	}
	exports.isIntegerType = isIntegerType;
	/**
	 * Custom type guard for NullType. Returns true if node is instance of NullType. Returns false otherwise.
	 * Also returns false for super interfaces of NullType.
	 */
	function isNullType(node) {
	    return node.kind() == "NullType" && node.RAMLVersion() == "RAML10";
	}
	exports.isNullType = isNullType;
	/**
	 * Custom type guard for TimeOnlyType. Returns true if node is instance of TimeOnlyType. Returns false otherwise.
	 * Also returns false for super interfaces of TimeOnlyType.
	 */
	function isTimeOnlyType(node) {
	    return node.kind() == "TimeOnlyType" && node.RAMLVersion() == "RAML10";
	}
	exports.isTimeOnlyType = isTimeOnlyType;
	/**
	 * Custom type guard for DateOnlyType. Returns true if node is instance of DateOnlyType. Returns false otherwise.
	 * Also returns false for super interfaces of DateOnlyType.
	 */
	function isDateOnlyType(node) {
	    return node.kind() == "DateOnlyType" && node.RAMLVersion() == "RAML10";
	}
	exports.isDateOnlyType = isDateOnlyType;
	/**
	 * Custom type guard for DateTimeOnlyType. Returns true if node is instance of DateTimeOnlyType. Returns false otherwise.
	 * Also returns false for super interfaces of DateTimeOnlyType.
	 */
	function isDateTimeOnlyType(node) {
	    return node.kind() == "DateTimeOnlyType" && node.RAMLVersion() == "RAML10";
	}
	exports.isDateTimeOnlyType = isDateTimeOnlyType;
	/**
	 * Custom type guard for DateTimeType. Returns true if node is instance of DateTimeType. Returns false otherwise.
	 * Also returns false for super interfaces of DateTimeType.
	 */
	function isDateTimeType(node) {
	    return node.kind() == "DateTimeType" && node.RAMLVersion() == "RAML10";
	}
	exports.isDateTimeType = isDateTimeType;
	/**
	 * Custom type guard for FileType. Returns true if node is instance of FileType. Returns false otherwise.
	 * Also returns false for super interfaces of FileType.
	 */
	function isFileType(node) {
	    return node.kind() == "FileType" && node.RAMLVersion() == "RAML10";
	}
	exports.isFileType = isFileType;
	/**
	 * Custom type guard for BooleanType. Returns true if node is instance of BooleanType. Returns false otherwise.
	 * Also returns false for super interfaces of BooleanType.
	 */
	function isBooleanType(node) {
	    return node.kind() == "BooleanType" && node.RAMLVersion() == "RAML10";
	}
	exports.isBooleanType = isBooleanType;
	/**
	 * Custom type guard for AnnotationTarget. Returns true if node is instance of AnnotationTarget. Returns false otherwise.
	 * Also returns false for super interfaces of AnnotationTarget.
	 */
	function isAnnotationTarget(node) {
	    return node.kind() == "AnnotationTarget" && node.RAMLVersion() == "RAML10";
	}
	exports.isAnnotationTarget = isAnnotationTarget;
	/**
	 * Custom type guard for TraitRef. Returns true if node is instance of TraitRef. Returns false otherwise.
	 * Also returns false for super interfaces of TraitRef.
	 */
	function isTraitRef(node) {
	    return node.kind() == "TraitRef" && node.RAMLVersion() == "RAML10";
	}
	exports.isTraitRef = isTraitRef;
	/**
	 * Custom type guard for Trait. Returns true if node is instance of Trait. Returns false otherwise.
	 * Also returns false for super interfaces of Trait.
	 */
	function isTrait(node) {
	    return node.kind() == "Trait" && node.RAMLVersion() == "RAML10";
	}
	exports.isTrait = isTrait;
	/**
	 * Custom type guard for MethodBase. Returns true if node is instance of MethodBase. Returns false otherwise.
	 * Also returns false for super interfaces of MethodBase.
	 */
	function isMethodBase(node) {
	    return node.kind() == "MethodBase" && node.RAMLVersion() == "RAML10";
	}
	exports.isMethodBase = isMethodBase;
	/**
	 * Custom type guard for Operation. Returns true if node is instance of Operation. Returns false otherwise.
	 * Also returns false for super interfaces of Operation.
	 */
	function isOperation(node) {
	    return node.kind() == "Operation" && node.RAMLVersion() == "RAML10";
	}
	exports.isOperation = isOperation;
	/**
	 * Custom type guard for TypeDeclaration. Returns true if node is instance of TypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of TypeDeclaration.
	 */
	function isTypeDeclaration(node) {
	    return node.kind() == "TypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isTypeDeclaration = isTypeDeclaration;
	/**
	 * Custom type guard for ModelLocation. Returns true if node is instance of ModelLocation. Returns false otherwise.
	 * Also returns false for super interfaces of ModelLocation.
	 */
	function isModelLocation(node) {
	    return node.kind() == "ModelLocation" && node.RAMLVersion() == "RAML10";
	}
	exports.isModelLocation = isModelLocation;
	/**
	 * Custom type guard for LocationKind. Returns true if node is instance of LocationKind. Returns false otherwise.
	 * Also returns false for super interfaces of LocationKind.
	 */
	function isLocationKind(node) {
	    return node.kind() == "LocationKind" && node.RAMLVersion() == "RAML10";
	}
	exports.isLocationKind = isLocationKind;
	/**
	 * Custom type guard for ExampleSpec. Returns true if node is instance of ExampleSpec. Returns false otherwise.
	 * Also returns false for super interfaces of ExampleSpec.
	 */
	function isExampleSpec(node) {
	    return node.kind() == "ExampleSpec" && node.RAMLVersion() == "RAML10";
	}
	exports.isExampleSpec = isExampleSpec;
	/**
	 * Custom type guard for XMLFacetInfo. Returns true if node is instance of XMLFacetInfo. Returns false otherwise.
	 * Also returns false for super interfaces of XMLFacetInfo.
	 */
	function isXMLFacetInfo(node) {
	    return node.kind() == "XMLFacetInfo" && node.RAMLVersion() == "RAML10";
	}
	exports.isXMLFacetInfo = isXMLFacetInfo;
	/**
	 * Custom type guard for UsesDeclaration. Returns true if node is instance of UsesDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of UsesDeclaration.
	 */
	function isUsesDeclaration(node) {
	    return node.kind() == "UsesDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isUsesDeclaration = isUsesDeclaration;
	/**
	 * Custom type guard for ArrayTypeDeclaration. Returns true if node is instance of ArrayTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of ArrayTypeDeclaration.
	 */
	function isArrayTypeDeclaration(node) {
	    return node.kind() == "ArrayTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isArrayTypeDeclaration = isArrayTypeDeclaration;
	/**
	 * Custom type guard for UnionTypeDeclaration. Returns true if node is instance of UnionTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of UnionTypeDeclaration.
	 */
	function isUnionTypeDeclaration(node) {
	    return node.kind() == "UnionTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isUnionTypeDeclaration = isUnionTypeDeclaration;
	/**
	 * Custom type guard for ObjectTypeDeclaration. Returns true if node is instance of ObjectTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of ObjectTypeDeclaration.
	 */
	function isObjectTypeDeclaration(node) {
	    return node.kind() == "ObjectTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isObjectTypeDeclaration = isObjectTypeDeclaration;
	/**
	 * Custom type guard for StringTypeDeclaration. Returns true if node is instance of StringTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of StringTypeDeclaration.
	 */
	function isStringTypeDeclaration(node) {
	    return node.kind() == "StringTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isStringTypeDeclaration = isStringTypeDeclaration;
	/**
	 * Custom type guard for BooleanTypeDeclaration. Returns true if node is instance of BooleanTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of BooleanTypeDeclaration.
	 */
	function isBooleanTypeDeclaration(node) {
	    return node.kind() == "BooleanTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isBooleanTypeDeclaration = isBooleanTypeDeclaration;
	/**
	 * Custom type guard for NumberTypeDeclaration. Returns true if node is instance of NumberTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of NumberTypeDeclaration.
	 */
	function isNumberTypeDeclaration(node) {
	    return node.kind() == "NumberTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isNumberTypeDeclaration = isNumberTypeDeclaration;
	/**
	 * Custom type guard for IntegerTypeDeclaration. Returns true if node is instance of IntegerTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of IntegerTypeDeclaration.
	 */
	function isIntegerTypeDeclaration(node) {
	    return node.kind() == "IntegerTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isIntegerTypeDeclaration = isIntegerTypeDeclaration;
	/**
	 * Custom type guard for DateOnlyTypeDeclaration. Returns true if node is instance of DateOnlyTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of DateOnlyTypeDeclaration.
	 */
	function isDateOnlyTypeDeclaration(node) {
	    return node.kind() == "DateOnlyTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isDateOnlyTypeDeclaration = isDateOnlyTypeDeclaration;
	/**
	 * Custom type guard for TimeOnlyTypeDeclaration. Returns true if node is instance of TimeOnlyTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of TimeOnlyTypeDeclaration.
	 */
	function isTimeOnlyTypeDeclaration(node) {
	    return node.kind() == "TimeOnlyTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isTimeOnlyTypeDeclaration = isTimeOnlyTypeDeclaration;
	/**
	 * Custom type guard for DateTimeOnlyTypeDeclaration. Returns true if node is instance of DateTimeOnlyTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of DateTimeOnlyTypeDeclaration.
	 */
	function isDateTimeOnlyTypeDeclaration(node) {
	    return node.kind() == "DateTimeOnlyTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isDateTimeOnlyTypeDeclaration = isDateTimeOnlyTypeDeclaration;
	/**
	 * Custom type guard for DateTimeTypeDeclaration. Returns true if node is instance of DateTimeTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of DateTimeTypeDeclaration.
	 */
	function isDateTimeTypeDeclaration(node) {
	    return node.kind() == "DateTimeTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isDateTimeTypeDeclaration = isDateTimeTypeDeclaration;
	/**
	 * Custom type guard for DateTypeDeclaration. Returns true if node is instance of DateTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of DateTypeDeclaration.
	 */
	function isDateTypeDeclaration(node) {
	    return node.kind() == "DateTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isDateTypeDeclaration = isDateTypeDeclaration;
	/**
	 * Custom type guard for FileTypeDeclaration. Returns true if node is instance of FileTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of FileTypeDeclaration.
	 */
	function isFileTypeDeclaration(node) {
	    return node.kind() == "FileTypeDeclaration" && node.RAMLVersion() == "RAML10";
	}
	exports.isFileTypeDeclaration = isFileTypeDeclaration;
	/**
	 * Custom type guard for Response. Returns true if node is instance of Response. Returns false otherwise.
	 * Also returns false for super interfaces of Response.
	 */
	function isResponse(node) {
	    return node.kind() == "Response" && node.RAMLVersion() == "RAML10";
	}
	exports.isResponse = isResponse;
	/**
	 * Custom type guard for SecuritySchemePart. Returns true if node is instance of SecuritySchemePart. Returns false otherwise.
	 * Also returns false for super interfaces of SecuritySchemePart.
	 */
	function isSecuritySchemePart(node) {
	    return node.kind() == "SecuritySchemePart" && node.RAMLVersion() == "RAML10";
	}
	exports.isSecuritySchemePart = isSecuritySchemePart;
	/**
	 * Custom type guard for SecuritySchemeRef. Returns true if node is instance of SecuritySchemeRef. Returns false otherwise.
	 * Also returns false for super interfaces of SecuritySchemeRef.
	 */
	function isSecuritySchemeRef(node) {
	    return node.kind() == "SecuritySchemeRef" && node.RAMLVersion() == "RAML10";
	}
	exports.isSecuritySchemeRef = isSecuritySchemeRef;
	/**
	 * Custom type guard for AbstractSecurityScheme. Returns true if node is instance of AbstractSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of AbstractSecurityScheme.
	 */
	function isAbstractSecurityScheme(node) {
	    return node.kind() == "AbstractSecurityScheme" && node.RAMLVersion() == "RAML10";
	}
	exports.isAbstractSecurityScheme = isAbstractSecurityScheme;
	/**
	 * Custom type guard for SecuritySchemeSettings. Returns true if node is instance of SecuritySchemeSettings. Returns false otherwise.
	 * Also returns false for super interfaces of SecuritySchemeSettings.
	 */
	function isSecuritySchemeSettings(node) {
	    return node.kind() == "SecuritySchemeSettings" && node.RAMLVersion() == "RAML10";
	}
	exports.isSecuritySchemeSettings = isSecuritySchemeSettings;
	/**
	 * Custom type guard for OAuth1SecuritySchemeSettings. Returns true if node is instance of OAuth1SecuritySchemeSettings. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth1SecuritySchemeSettings.
	 */
	function isOAuth1SecuritySchemeSettings(node) {
	    return node.kind() == "OAuth1SecuritySchemeSettings" && node.RAMLVersion() == "RAML10";
	}
	exports.isOAuth1SecuritySchemeSettings = isOAuth1SecuritySchemeSettings;
	/**
	 * Custom type guard for OAuth2SecuritySchemeSettings. Returns true if node is instance of OAuth2SecuritySchemeSettings. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth2SecuritySchemeSettings.
	 */
	function isOAuth2SecuritySchemeSettings(node) {
	    return node.kind() == "OAuth2SecuritySchemeSettings" && node.RAMLVersion() == "RAML10";
	}
	exports.isOAuth2SecuritySchemeSettings = isOAuth2SecuritySchemeSettings;
	/**
	 * Custom type guard for OAuth2SecurityScheme. Returns true if node is instance of OAuth2SecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth2SecurityScheme.
	 */
	function isOAuth2SecurityScheme(node) {
	    return node.kind() == "OAuth2SecurityScheme" && node.RAMLVersion() == "RAML10";
	}
	exports.isOAuth2SecurityScheme = isOAuth2SecurityScheme;
	/**
	 * Custom type guard for OAuth1SecurityScheme. Returns true if node is instance of OAuth1SecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth1SecurityScheme.
	 */
	function isOAuth1SecurityScheme(node) {
	    return node.kind() == "OAuth1SecurityScheme" && node.RAMLVersion() == "RAML10";
	}
	exports.isOAuth1SecurityScheme = isOAuth1SecurityScheme;
	/**
	 * Custom type guard for PassThroughSecurityScheme. Returns true if node is instance of PassThroughSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of PassThroughSecurityScheme.
	 */
	function isPassThroughSecurityScheme(node) {
	    return node.kind() == "PassThroughSecurityScheme" && node.RAMLVersion() == "RAML10";
	}
	exports.isPassThroughSecurityScheme = isPassThroughSecurityScheme;
	/**
	 * Custom type guard for BasicSecurityScheme. Returns true if node is instance of BasicSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of BasicSecurityScheme.
	 */
	function isBasicSecurityScheme(node) {
	    return node.kind() == "BasicSecurityScheme" && node.RAMLVersion() == "RAML10";
	}
	exports.isBasicSecurityScheme = isBasicSecurityScheme;
	/**
	 * Custom type guard for DigestSecurityScheme. Returns true if node is instance of DigestSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of DigestSecurityScheme.
	 */
	function isDigestSecurityScheme(node) {
	    return node.kind() == "DigestSecurityScheme" && node.RAMLVersion() == "RAML10";
	}
	exports.isDigestSecurityScheme = isDigestSecurityScheme;
	/**
	 * Custom type guard for CustomSecurityScheme. Returns true if node is instance of CustomSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of CustomSecurityScheme.
	 */
	function isCustomSecurityScheme(node) {
	    return node.kind() == "CustomSecurityScheme" && node.RAMLVersion() == "RAML10";
	}
	exports.isCustomSecurityScheme = isCustomSecurityScheme;
	/**
	 * Custom type guard for Method. Returns true if node is instance of Method. Returns false otherwise.
	 * Also returns false for super interfaces of Method.
	 */
	function isMethod(node) {
	    return node.kind() == "Method" && node.RAMLVersion() == "RAML10";
	}
	exports.isMethod = isMethod;
	/**
	 * Custom type guard for ResourceTypeRef. Returns true if node is instance of ResourceTypeRef. Returns false otherwise.
	 * Also returns false for super interfaces of ResourceTypeRef.
	 */
	function isResourceTypeRef(node) {
	    return node.kind() == "ResourceTypeRef" && node.RAMLVersion() == "RAML10";
	}
	exports.isResourceTypeRef = isResourceTypeRef;
	/**
	 * Custom type guard for ResourceType. Returns true if node is instance of ResourceType. Returns false otherwise.
	 * Also returns false for super interfaces of ResourceType.
	 */
	function isResourceType(node) {
	    return node.kind() == "ResourceType" && node.RAMLVersion() == "RAML10";
	}
	exports.isResourceType = isResourceType;
	/**
	 * Custom type guard for ResourceBase. Returns true if node is instance of ResourceBase. Returns false otherwise.
	 * Also returns false for super interfaces of ResourceBase.
	 */
	function isResourceBase(node) {
	    return node.kind() == "ResourceBase" && node.RAMLVersion() == "RAML10";
	}
	exports.isResourceBase = isResourceBase;
	/**
	 * Custom type guard for Resource. Returns true if node is instance of Resource. Returns false otherwise.
	 * Also returns false for super interfaces of Resource.
	 */
	function isResource(node) {
	    return node.kind() == "Resource" && node.RAMLVersion() == "RAML10";
	}
	exports.isResource = isResource;
	/**
	 * Custom type guard for DocumentationItem. Returns true if node is instance of DocumentationItem. Returns false otherwise.
	 * Also returns false for super interfaces of DocumentationItem.
	 */
	function isDocumentationItem(node) {
	    return node.kind() == "DocumentationItem" && node.RAMLVersion() == "RAML10";
	}
	exports.isDocumentationItem = isDocumentationItem;
	/**
	 * Custom type guard for Library. Returns true if node is instance of Library. Returns false otherwise.
	 * Also returns false for super interfaces of Library.
	 */
	function isLibrary(node) {
	    return node.kind() == "Library" && node.RAMLVersion() == "RAML10";
	}
	exports.isLibrary = isLibrary;
	/**
	 * Custom type guard for Overlay. Returns true if node is instance of Overlay. Returns false otherwise.
	 * Also returns false for super interfaces of Overlay.
	 */
	function isOverlay(node) {
	    return node.kind() == "Overlay" && node.RAMLVersion() == "RAML10";
	}
	exports.isOverlay = isOverlay;
	/**
	 * Custom type guard for Extension. Returns true if node is instance of Extension. Returns false otherwise.
	 * Also returns false for super interfaces of Extension.
	 */
	function isExtension(node) {
	    return node.kind() == "Extension" && node.RAMLVersion() == "RAML10";
	}
	exports.isExtension = isExtension;
	/**
	 * Check if the AST node represents fragment
	 */
	function isFragment(node) {
	    return node.highLevel().parent() == null;
	}
	exports.isFragment = isFragment;
	/**
	 * Convert fragment representing node to FragmentDeclaration instance.
	 */
	function asFragment(node) {
	    return isFragment(node) ? node : null;
	}
	exports.asFragment = asFragment;
	//# sourceMappingURL=raml10parserapi.js.map

/***/ },
/* 13 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	/**
	 * Custom type guard for Api. Returns true if node is instance of Api. Returns false otherwise.
	 * Also returns false for super interfaces of Api.
	 */
	function isApi(node) {
	    return node.kind() == "Api" && node.RAMLVersion() == "RAML08";
	}
	exports.isApi = isApi;
	/**
	 * Custom type guard for FullUriTemplateString. Returns true if node is instance of FullUriTemplateString. Returns false otherwise.
	 * Also returns false for super interfaces of FullUriTemplateString.
	 */
	function isFullUriTemplateString(node) {
	    return node.kind() == "FullUriTemplateString" && node.RAMLVersion() == "RAML08";
	}
	exports.isFullUriTemplateString = isFullUriTemplateString;
	/**
	 * Custom type guard for UriTemplate. Returns true if node is instance of UriTemplate. Returns false otherwise.
	 * Also returns false for super interfaces of UriTemplate.
	 */
	function isUriTemplate(node) {
	    return node.kind() == "UriTemplate" && node.RAMLVersion() == "RAML08";
	}
	exports.isUriTemplate = isUriTemplate;
	/**
	 * Custom type guard for StringType. Returns true if node is instance of StringType. Returns false otherwise.
	 * Also returns false for super interfaces of StringType.
	 */
	function isStringType(node) {
	    return node.kind() == "StringType" && node.RAMLVersion() == "RAML08";
	}
	exports.isStringType = isStringType;
	/**
	 * Custom type guard for ValueType. Returns true if node is instance of ValueType. Returns false otherwise.
	 * Also returns false for super interfaces of ValueType.
	 */
	function isValueType(node) {
	    return node.kind() == "ValueType" && node.RAMLVersion() == "RAML08";
	}
	exports.isValueType = isValueType;
	/**
	 * Custom type guard for AnyType. Returns true if node is instance of AnyType. Returns false otherwise.
	 * Also returns false for super interfaces of AnyType.
	 */
	function isAnyType(node) {
	    return node.kind() == "AnyType" && node.RAMLVersion() == "RAML08";
	}
	exports.isAnyType = isAnyType;
	/**
	 * Custom type guard for NumberType. Returns true if node is instance of NumberType. Returns false otherwise.
	 * Also returns false for super interfaces of NumberType.
	 */
	function isNumberType(node) {
	    return node.kind() == "NumberType" && node.RAMLVersion() == "RAML08";
	}
	exports.isNumberType = isNumberType;
	/**
	 * Custom type guard for BooleanType. Returns true if node is instance of BooleanType. Returns false otherwise.
	 * Also returns false for super interfaces of BooleanType.
	 */
	function isBooleanType(node) {
	    return node.kind() == "BooleanType" && node.RAMLVersion() == "RAML08";
	}
	exports.isBooleanType = isBooleanType;
	/**
	 * Custom type guard for Reference. Returns true if node is instance of Reference. Returns false otherwise.
	 * Also returns false for super interfaces of Reference.
	 */
	function isReference(node) {
	    return node.kind() == "Reference" && node.RAMLVersion() == "RAML08";
	}
	exports.isReference = isReference;
	/**
	 * Custom type guard for ResourceTypeRef. Returns true if node is instance of ResourceTypeRef. Returns false otherwise.
	 * Also returns false for super interfaces of ResourceTypeRef.
	 */
	function isResourceTypeRef(node) {
	    return node.kind() == "ResourceTypeRef" && node.RAMLVersion() == "RAML08";
	}
	exports.isResourceTypeRef = isResourceTypeRef;
	/**
	 * Custom type guard for ResourceType. Returns true if node is instance of ResourceType. Returns false otherwise.
	 * Also returns false for super interfaces of ResourceType.
	 */
	function isResourceType(node) {
	    return node.kind() == "ResourceType" && node.RAMLVersion() == "RAML08";
	}
	exports.isResourceType = isResourceType;
	/**
	 * Custom type guard for Method. Returns true if node is instance of Method. Returns false otherwise.
	 * Also returns false for super interfaces of Method.
	 */
	function isMethod(node) {
	    return node.kind() == "Method" && node.RAMLVersion() == "RAML08";
	}
	exports.isMethod = isMethod;
	/**
	 * Custom type guard for MethodBase. Returns true if node is instance of MethodBase. Returns false otherwise.
	 * Also returns false for super interfaces of MethodBase.
	 */
	function isMethodBase(node) {
	    return node.kind() == "MethodBase" && node.RAMLVersion() == "RAML08";
	}
	exports.isMethodBase = isMethodBase;
	/**
	 * Custom type guard for Response. Returns true if node is instance of Response. Returns false otherwise.
	 * Also returns false for super interfaces of Response.
	 */
	function isResponse(node) {
	    return node.kind() == "Response" && node.RAMLVersion() == "RAML08";
	}
	exports.isResponse = isResponse;
	/**
	 * Custom type guard for StatusCodeString. Returns true if node is instance of StatusCodeString. Returns false otherwise.
	 * Also returns false for super interfaces of StatusCodeString.
	 */
	function isStatusCodeString(node) {
	    return node.kind() == "StatusCodeString" && node.RAMLVersion() == "RAML08";
	}
	exports.isStatusCodeString = isStatusCodeString;
	/**
	 * Custom type guard for Parameter. Returns true if node is instance of Parameter. Returns false otherwise.
	 * Also returns false for super interfaces of Parameter.
	 */
	function isParameter(node) {
	    return node.kind() == "Parameter" && node.RAMLVersion() == "RAML08";
	}
	exports.isParameter = isParameter;
	/**
	 * Custom type guard for ParameterLocation. Returns true if node is instance of ParameterLocation. Returns false otherwise.
	 * Also returns false for super interfaces of ParameterLocation.
	 */
	function isParameterLocation(node) {
	    return node.kind() == "ParameterLocation" && node.RAMLVersion() == "RAML08";
	}
	exports.isParameterLocation = isParameterLocation;
	/**
	 * Custom type guard for MarkdownString. Returns true if node is instance of MarkdownString. Returns false otherwise.
	 * Also returns false for super interfaces of MarkdownString.
	 */
	function isMarkdownString(node) {
	    return node.kind() == "MarkdownString" && node.RAMLVersion() == "RAML08";
	}
	exports.isMarkdownString = isMarkdownString;
	/**
	 * Custom type guard for StringTypeDeclaration. Returns true if node is instance of StringTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of StringTypeDeclaration.
	 */
	function isStringTypeDeclaration(node) {
	    return node.kind() == "StringTypeDeclaration" && node.RAMLVersion() == "RAML08";
	}
	exports.isStringTypeDeclaration = isStringTypeDeclaration;
	/**
	 * Custom type guard for BooleanTypeDeclaration. Returns true if node is instance of BooleanTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of BooleanTypeDeclaration.
	 */
	function isBooleanTypeDeclaration(node) {
	    return node.kind() == "BooleanTypeDeclaration" && node.RAMLVersion() == "RAML08";
	}
	exports.isBooleanTypeDeclaration = isBooleanTypeDeclaration;
	/**
	 * Custom type guard for NumberTypeDeclaration. Returns true if node is instance of NumberTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of NumberTypeDeclaration.
	 */
	function isNumberTypeDeclaration(node) {
	    return node.kind() == "NumberTypeDeclaration" && node.RAMLVersion() == "RAML08";
	}
	exports.isNumberTypeDeclaration = isNumberTypeDeclaration;
	/**
	 * Custom type guard for IntegerTypeDeclaration. Returns true if node is instance of IntegerTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of IntegerTypeDeclaration.
	 */
	function isIntegerTypeDeclaration(node) {
	    return node.kind() == "IntegerTypeDeclaration" && node.RAMLVersion() == "RAML08";
	}
	exports.isIntegerTypeDeclaration = isIntegerTypeDeclaration;
	/**
	 * Custom type guard for DateTypeDeclaration. Returns true if node is instance of DateTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of DateTypeDeclaration.
	 */
	function isDateTypeDeclaration(node) {
	    return node.kind() == "DateTypeDeclaration" && node.RAMLVersion() == "RAML08";
	}
	exports.isDateTypeDeclaration = isDateTypeDeclaration;
	/**
	 * Custom type guard for FileTypeDeclaration. Returns true if node is instance of FileTypeDeclaration. Returns false otherwise.
	 * Also returns false for super interfaces of FileTypeDeclaration.
	 */
	function isFileTypeDeclaration(node) {
	    return node.kind() == "FileTypeDeclaration" && node.RAMLVersion() == "RAML08";
	}
	exports.isFileTypeDeclaration = isFileTypeDeclaration;
	/**
	 * Custom type guard for BodyLike. Returns true if node is instance of BodyLike. Returns false otherwise.
	 * Also returns false for super interfaces of BodyLike.
	 */
	function isBodyLike(node) {
	    return node.kind() == "BodyLike" && node.RAMLVersion() == "RAML08";
	}
	exports.isBodyLike = isBodyLike;
	/**
	 * Custom type guard for SchemaString. Returns true if node is instance of SchemaString. Returns false otherwise.
	 * Also returns false for super interfaces of SchemaString.
	 */
	function isSchemaString(node) {
	    return node.kind() == "SchemaString" && node.RAMLVersion() == "RAML08";
	}
	exports.isSchemaString = isSchemaString;
	/**
	 * Custom type guard for JSonSchemaString. Returns true if node is instance of JSonSchemaString. Returns false otherwise.
	 * Also returns false for super interfaces of JSonSchemaString.
	 */
	function isJSonSchemaString(node) {
	    return node.kind() == "JSonSchemaString" && node.RAMLVersion() == "RAML08";
	}
	exports.isJSonSchemaString = isJSonSchemaString;
	/**
	 * Custom type guard for XMLSchemaString. Returns true if node is instance of XMLSchemaString. Returns false otherwise.
	 * Also returns false for super interfaces of XMLSchemaString.
	 */
	function isXMLSchemaString(node) {
	    return node.kind() == "XMLSchemaString" && node.RAMLVersion() == "RAML08";
	}
	exports.isXMLSchemaString = isXMLSchemaString;
	/**
	 * Custom type guard for ExampleString. Returns true if node is instance of ExampleString. Returns false otherwise.
	 * Also returns false for super interfaces of ExampleString.
	 */
	function isExampleString(node) {
	    return node.kind() == "ExampleString" && node.RAMLVersion() == "RAML08";
	}
	exports.isExampleString = isExampleString;
	/**
	 * Custom type guard for JSONExample. Returns true if node is instance of JSONExample. Returns false otherwise.
	 * Also returns false for super interfaces of JSONExample.
	 */
	function isJSONExample(node) {
	    return node.kind() == "JSONExample" && node.RAMLVersion() == "RAML08";
	}
	exports.isJSONExample = isJSONExample;
	/**
	 * Custom type guard for XMLExample. Returns true if node is instance of XMLExample. Returns false otherwise.
	 * Also returns false for super interfaces of XMLExample.
	 */
	function isXMLExample(node) {
	    return node.kind() == "XMLExample" && node.RAMLVersion() == "RAML08";
	}
	exports.isXMLExample = isXMLExample;
	/**
	 * Custom type guard for XMLBody. Returns true if node is instance of XMLBody. Returns false otherwise.
	 * Also returns false for super interfaces of XMLBody.
	 */
	function isXMLBody(node) {
	    return node.kind() == "XMLBody" && node.RAMLVersion() == "RAML08";
	}
	exports.isXMLBody = isXMLBody;
	/**
	 * Custom type guard for JSONBody. Returns true if node is instance of JSONBody. Returns false otherwise.
	 * Also returns false for super interfaces of JSONBody.
	 */
	function isJSONBody(node) {
	    return node.kind() == "JSONBody" && node.RAMLVersion() == "RAML08";
	}
	exports.isJSONBody = isJSONBody;
	/**
	 * Custom type guard for SecuritySchemeRef. Returns true if node is instance of SecuritySchemeRef. Returns false otherwise.
	 * Also returns false for super interfaces of SecuritySchemeRef.
	 */
	function isSecuritySchemeRef(node) {
	    return node.kind() == "SecuritySchemeRef" && node.RAMLVersion() == "RAML08";
	}
	exports.isSecuritySchemeRef = isSecuritySchemeRef;
	/**
	 * Custom type guard for AbstractSecurityScheme. Returns true if node is instance of AbstractSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of AbstractSecurityScheme.
	 */
	function isAbstractSecurityScheme(node) {
	    return node.kind() == "AbstractSecurityScheme" && node.RAMLVersion() == "RAML08";
	}
	exports.isAbstractSecurityScheme = isAbstractSecurityScheme;
	/**
	 * Custom type guard for SecuritySchemePart. Returns true if node is instance of SecuritySchemePart. Returns false otherwise.
	 * Also returns false for super interfaces of SecuritySchemePart.
	 */
	function isSecuritySchemePart(node) {
	    return node.kind() == "SecuritySchemePart" && node.RAMLVersion() == "RAML08";
	}
	exports.isSecuritySchemePart = isSecuritySchemePart;
	/**
	 * Custom type guard for TraitRef. Returns true if node is instance of TraitRef. Returns false otherwise.
	 * Also returns false for super interfaces of TraitRef.
	 */
	function isTraitRef(node) {
	    return node.kind() == "TraitRef" && node.RAMLVersion() == "RAML08";
	}
	exports.isTraitRef = isTraitRef;
	/**
	 * Custom type guard for Trait. Returns true if node is instance of Trait. Returns false otherwise.
	 * Also returns false for super interfaces of Trait.
	 */
	function isTrait(node) {
	    return node.kind() == "Trait" && node.RAMLVersion() == "RAML08";
	}
	exports.isTrait = isTrait;
	/**
	 * Custom type guard for SecuritySchemeSettings. Returns true if node is instance of SecuritySchemeSettings. Returns false otherwise.
	 * Also returns false for super interfaces of SecuritySchemeSettings.
	 */
	function isSecuritySchemeSettings(node) {
	    return node.kind() == "SecuritySchemeSettings" && node.RAMLVersion() == "RAML08";
	}
	exports.isSecuritySchemeSettings = isSecuritySchemeSettings;
	/**
	 * Custom type guard for OAuth1SecuritySchemeSettings. Returns true if node is instance of OAuth1SecuritySchemeSettings. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth1SecuritySchemeSettings.
	 */
	function isOAuth1SecuritySchemeSettings(node) {
	    return node.kind() == "OAuth1SecuritySchemeSettings" && node.RAMLVersion() == "RAML08";
	}
	exports.isOAuth1SecuritySchemeSettings = isOAuth1SecuritySchemeSettings;
	/**
	 * Custom type guard for FixedUri. Returns true if node is instance of FixedUri. Returns false otherwise.
	 * Also returns false for super interfaces of FixedUri.
	 */
	function isFixedUri(node) {
	    return node.kind() == "FixedUri" && node.RAMLVersion() == "RAML08";
	}
	exports.isFixedUri = isFixedUri;
	/**
	 * Custom type guard for OAuth2SecuritySchemeSettings. Returns true if node is instance of OAuth2SecuritySchemeSettings. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth2SecuritySchemeSettings.
	 */
	function isOAuth2SecuritySchemeSettings(node) {
	    return node.kind() == "OAuth2SecuritySchemeSettings" && node.RAMLVersion() == "RAML08";
	}
	exports.isOAuth2SecuritySchemeSettings = isOAuth2SecuritySchemeSettings;
	/**
	 * Custom type guard for OAuth2SecurityScheme. Returns true if node is instance of OAuth2SecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth2SecurityScheme.
	 */
	function isOAuth2SecurityScheme(node) {
	    return node.kind() == "OAuth2SecurityScheme" && node.RAMLVersion() == "RAML08";
	}
	exports.isOAuth2SecurityScheme = isOAuth2SecurityScheme;
	/**
	 * Custom type guard for OAuth1SecurityScheme. Returns true if node is instance of OAuth1SecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of OAuth1SecurityScheme.
	 */
	function isOAuth1SecurityScheme(node) {
	    return node.kind() == "OAuth1SecurityScheme" && node.RAMLVersion() == "RAML08";
	}
	exports.isOAuth1SecurityScheme = isOAuth1SecurityScheme;
	/**
	 * Custom type guard for BasicSecurityScheme. Returns true if node is instance of BasicSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of BasicSecurityScheme.
	 */
	function isBasicSecurityScheme(node) {
	    return node.kind() == "BasicSecurityScheme" && node.RAMLVersion() == "RAML08";
	}
	exports.isBasicSecurityScheme = isBasicSecurityScheme;
	/**
	 * Custom type guard for DigestSecurityScheme. Returns true if node is instance of DigestSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of DigestSecurityScheme.
	 */
	function isDigestSecurityScheme(node) {
	    return node.kind() == "DigestSecurityScheme" && node.RAMLVersion() == "RAML08";
	}
	exports.isDigestSecurityScheme = isDigestSecurityScheme;
	/**
	 * Custom type guard for CustomSecurityScheme. Returns true if node is instance of CustomSecurityScheme. Returns false otherwise.
	 * Also returns false for super interfaces of CustomSecurityScheme.
	 */
	function isCustomSecurityScheme(node) {
	    return node.kind() == "CustomSecurityScheme" && node.RAMLVersion() == "RAML08";
	}
	exports.isCustomSecurityScheme = isCustomSecurityScheme;
	/**
	 * Custom type guard for MimeType. Returns true if node is instance of MimeType. Returns false otherwise.
	 * Also returns false for super interfaces of MimeType.
	 */
	function isMimeType(node) {
	    return node.kind() == "MimeType" && node.RAMLVersion() == "RAML08";
	}
	exports.isMimeType = isMimeType;
	/**
	 * Custom type guard for RelativeUriString. Returns true if node is instance of RelativeUriString. Returns false otherwise.
	 * Also returns false for super interfaces of RelativeUriString.
	 */
	function isRelativeUriString(node) {
	    return node.kind() == "RelativeUriString" && node.RAMLVersion() == "RAML08";
	}
	exports.isRelativeUriString = isRelativeUriString;
	/**
	 * Custom type guard for GlobalSchema. Returns true if node is instance of GlobalSchema. Returns false otherwise.
	 * Also returns false for super interfaces of GlobalSchema.
	 */
	function isGlobalSchema(node) {
	    return node.kind() == "GlobalSchema" && node.RAMLVersion() == "RAML08";
	}
	exports.isGlobalSchema = isGlobalSchema;
	/**
	 * Custom type guard for RAMLSimpleElement. Returns true if node is instance of RAMLSimpleElement. Returns false otherwise.
	 * Also returns false for super interfaces of RAMLSimpleElement.
	 */
	function isRAMLSimpleElement(node) {
	    return node.kind() == "RAMLSimpleElement" && node.RAMLVersion() == "RAML08";
	}
	exports.isRAMLSimpleElement = isRAMLSimpleElement;
	/**
	 * Custom type guard for DocumentationItem. Returns true if node is instance of DocumentationItem. Returns false otherwise.
	 * Also returns false for super interfaces of DocumentationItem.
	 */
	function isDocumentationItem(node) {
	    return node.kind() == "DocumentationItem" && node.RAMLVersion() == "RAML08";
	}
	exports.isDocumentationItem = isDocumentationItem;
	/**
	 * Custom type guard for Resource. Returns true if node is instance of Resource. Returns false otherwise.
	 * Also returns false for super interfaces of Resource.
	 */
	function isResource(node) {
	    return node.kind() == "Resource" && node.RAMLVersion() == "RAML08";
	}
	exports.isResource = isResource;
	//# sourceMappingURL=raml08parserapi.js.map

/***/ },
/* 14 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var universe = __webpack_require__(22);
	/////////////////////// properties
	function isDocumentationProperty(p) {
	    return p.nameId() === universe.Universe10.Api.properties.documentation.name ||
	        p.nameId() === universe.Universe08.Api.properties.documentation.name;
	}
	exports.isDocumentationProperty = isDocumentationProperty;
	function isUsagePropertyName(name) {
	    return name === universe.Universe10.Trait.properties.usage.name ||
	        name === universe.Universe08.Trait.properties.usage.name ||
	        name === universe.Universe10.ResourceType.properties.usage.name ||
	        name === universe.Universe08.ResourceType.properties.usage.name ||
	        name === universe.Universe10.Library.properties.usage.name ||
	        name === universe.Universe10.Overlay.properties.usage.name ||
	        name === universe.Universe10.Extension.properties.usage.name;
	}
	exports.isUsagePropertyName = isUsagePropertyName;
	function isUsageProperty(p) {
	    return isUsagePropertyName(p.nameId());
	}
	exports.isUsageProperty = isUsageProperty;
	function isMasterRefProperty(p) {
	    return p.nameId() == universe.Universe10.Overlay.properties.extends.name ||
	        p.nameId() == universe.Universe10.Extension.properties.extends.name;
	}
	exports.isMasterRefProperty = isMasterRefProperty;
	function isDescriptionPropertyName(name) {
	    return name === universe.Universe10.TypeDeclaration.properties.description.name ||
	        name === "description";
	    //TODO too long to actually list every element having a description, so a couple of checks to cause compile error, and a simple equals check. Also we do not want to affect performance that much.
	}
	exports.isDescriptionPropertyName = isDescriptionPropertyName;
	function isDescriptionProperty(p) {
	    return isDescriptionPropertyName(p.nameId());
	}
	exports.isDescriptionProperty = isDescriptionProperty;
	function isRequiredPropertyName(name) {
	    return name === universe.Universe10.TypeDeclaration.properties.required.name ||
	        name === universe.Universe08.Parameter.properties.required.name ||
	        name === "required";
	    //TODO too long to actually list every element having displayname, so a couple of checks to cause compile error, and a simple equals check. Also we do not want to affect performance that much.
	}
	exports.isRequiredPropertyName = isRequiredPropertyName;
	function isDisplayNamePropertyName(name) {
	    return name === universe.Universe10.TypeDeclaration.properties.displayName.name ||
	        name === "displayName";
	    //TODO too long to actually list every element having displayname, so a couple of checks to cause compile error, and a simple equals check. Also we do not want to affect performance that much.
	}
	exports.isDisplayNamePropertyName = isDisplayNamePropertyName;
	function isDisplayNameProperty(p) {
	    return isDisplayNamePropertyName(p.nameId());
	}
	exports.isDisplayNameProperty = isDisplayNameProperty;
	function isRequiredProperty(p) {
	    return isRequiredPropertyName(p.nameId());
	}
	exports.isRequiredProperty = isRequiredProperty;
	function isTitlePropertyName(name) {
	    return name === universe.Universe10.Api.properties.title.name ||
	        name === universe.Universe08.Api.properties.title.name ||
	        name === universe.Universe10.DocumentationItem.properties.title.name ||
	        name === universe.Universe08.DocumentationItem.properties.title.name ||
	        name === universe.Universe10.Overlay.properties.title.name ||
	        name === universe.Universe10.Extension.properties.title.name;
	}
	exports.isTitlePropertyName = isTitlePropertyName;
	function isTitleProperty(p) {
	    return isTitlePropertyName(p.nameId());
	}
	exports.isTitleProperty = isTitleProperty;
	function isHeadersProperty(p) {
	    return isHeadersPropertyName(p.nameId());
	}
	exports.isHeadersProperty = isHeadersProperty;
	function isHeadersPropertyName(name) {
	    return name === universe.Universe08.MethodBase.properties.headers.name ||
	        name === universe.Universe08.Response.properties.headers.name ||
	        name === universe.Universe08.SecuritySchemePart.properties.headers.name ||
	        name === universe.Universe10.MethodBase.properties.headers.name ||
	        name === universe.Universe10.Response.properties.headers.name;
	}
	exports.isHeadersPropertyName = isHeadersPropertyName;
	function isFormParametersProperty(p) {
	    return isFormParametersPropertyName(p.nameId());
	}
	exports.isFormParametersProperty = isFormParametersProperty;
	function isFormParametersPropertyName(name) {
	    return name === universe.Universe08.BodyLike.properties.formParameters.name;
	}
	exports.isFormParametersPropertyName = isFormParametersPropertyName;
	function isQueryParametersProperty(p) {
	    return isQueryParametersPropertyName(p.nameId());
	}
	exports.isQueryParametersProperty = isQueryParametersProperty;
	function isQueryParametersPropertyName(name) {
	    return name === universe.Universe08.MethodBase.properties.queryParameters.name ||
	        name === universe.Universe08.SecuritySchemePart.properties.queryParameters.name ||
	        name === universe.Universe10.MethodBase.properties.queryParameters.name;
	}
	exports.isQueryParametersPropertyName = isQueryParametersPropertyName;
	function isAnnotationsProperty(p) {
	    return p.nameId() === universe.Universe10.Api.properties.annotations.name ||
	        //p.nameId() === universe.Universe10.AbstractSecurityScheme.properties.annotations.name ||
	        p.nameId() === universe.Universe10.TypeDeclaration.properties.annotations.name ||
	        p.nameId() === universe.Universe10.Response.properties.annotations.name;
	}
	exports.isAnnotationsProperty = isAnnotationsProperty;
	function isAnnotationProperty(p) {
	    return p.nameId() === universe.Universe10.AnnotationRef.properties.annotation.name;
	}
	exports.isAnnotationProperty = isAnnotationProperty;
	function isIsProperty(p) {
	    return p.nameId() === universe.Universe10.MethodBase.properties.is.name ||
	        p.nameId() === universe.Universe08.Method.properties.is.name ||
	        p.nameId() === universe.Universe10.ResourceBase.properties.is.name ||
	        p.nameId() === universe.Universe08.ResourceType.properties.is.name ||
	        p.nameId() === universe.Universe08.Resource.properties.is.name;
	}
	exports.isIsProperty = isIsProperty;
	function isSecuredByProperty(p) {
	    return p.nameId() === universe.Universe10.Api.properties.securedBy.name ||
	        p.nameId() === universe.Universe08.Api.properties.securedBy.name ||
	        p.nameId() === universe.Universe10.MethodBase.properties.securedBy.name ||
	        p.nameId() === universe.Universe08.MethodBase.properties.securedBy.name ||
	        p.nameId() === universe.Universe08.ResourceType.properties.securedBy.name ||
	        p.nameId() === universe.Universe08.Resource.properties.securedBy.name ||
	        p.nameId() === universe.Universe10.ResourceBase.properties.securedBy.name;
	}
	exports.isSecuredByProperty = isSecuredByProperty;
	function isSecuritySchemesProperty(p) {
	    return p.nameId() === universe.Universe10.LibraryBase.properties.securitySchemes.name ||
	        p.nameId() === universe.Universe08.Api.properties.securitySchemes.name;
	}
	exports.isSecuritySchemesProperty = isSecuritySchemesProperty;
	function isSecuritySchemeProperty(p) {
	    return p.nameId() === universe.Universe10.SecuritySchemeRef.properties.securityScheme.name ||
	        p.nameId() === universe.Universe08.SecuritySchemeRef.properties.securityScheme.name;
	}
	exports.isSecuritySchemeProperty = isSecuritySchemeProperty;
	function isTypeProperty(p) {
	    return p.nameId() === universe.Universe10.AbstractSecurityScheme.properties.type.name ||
	        p.nameId() === universe.Universe08.AbstractSecurityScheme.properties.type.name ||
	        p.nameId() === universe.Universe08.ResourceType.properties.type.name ||
	        p.nameId() === universe.Universe08.Resource.properties.type.name ||
	        p.nameId() === universe.Universe08.Parameter.properties.type.name ||
	        p.nameId() === universe.Universe10.ResourceBase.properties.type.name ||
	        p.nameId() === universe.Universe10.TypeDeclaration.properties.type.name;
	}
	exports.isTypeProperty = isTypeProperty;
	function isPropertiesProperty(p) {
	    return p.nameId() === universe.Universe10.ObjectTypeDeclaration.properties.properties.name;
	}
	exports.isPropertiesProperty = isPropertiesProperty;
	function isResponsesProperty(p) {
	    return p.nameId() === universe.Universe10.MethodBase.properties.responses.name ||
	        p.nameId() === universe.Universe08.MethodBase.properties.responses.name;
	}
	exports.isResponsesProperty = isResponsesProperty;
	function isProtocolsProperty(p) {
	    return p.nameId() === universe.Universe10.Api.properties.protocols.name ||
	        p.nameId() === universe.Universe08.Api.properties.protocols.name ||
	        p.nameId() === universe.Universe10.MethodBase.properties.protocols.name;
	}
	exports.isProtocolsProperty = isProtocolsProperty;
	function isNameProperty(p) {
	    return p.nameId() === universe.Universe10.TypeDeclaration.properties.name.name ||
	        p.nameId() === universe.Universe10.TypeDeclaration.properties.name.name ||
	        p.nameId() === universe.Universe08.AbstractSecurityScheme.properties.name.name ||
	        p.nameId() === universe.Universe10.AbstractSecurityScheme.properties.name.name ||
	        p.nameId() === universe.Universe08.Trait.properties.name.name ||
	        p.nameId() === universe.Universe10.Trait.properties.name.name ||
	        p.nameId() === "name";
	    //TODO too long to actually list every element having a name, so a couple of checks to cause compile error, and a simple equals check. Also we do not want to affect performance that much.
	}
	exports.isNameProperty = isNameProperty;
	function isBodyProperty(p) {
	    return p.nameId() === universe.Universe10.MethodBase.properties.body.name ||
	        p.nameId() === universe.Universe08.MethodBase.properties.body.name ||
	        p.nameId() === universe.Universe10.Response.properties.body.name ||
	        p.nameId() === universe.Universe08.Response.properties.body.name;
	}
	exports.isBodyProperty = isBodyProperty;
	function isSchemaProperty(p) {
	    return p.nameId() === universe.Universe08.BodyLike.properties.schema.name ||
	        p.nameId() === universe.Universe08.XMLBody.properties.schema.name ||
	        p.nameId() === universe.Universe08.JSONBody.properties.schema.name ||
	        p.nameId() === universe.Universe10.TypeDeclaration.properties.schema.name;
	}
	exports.isSchemaProperty = isSchemaProperty;
	function isTraitsProperty(p) {
	    return p.nameId() === universe.Universe08.Api.properties.traits.name ||
	        p.nameId() === universe.Universe10.LibraryBase.properties.traits.name;
	}
	exports.isTraitsProperty = isTraitsProperty;
	function isTraitProperty(p) {
	    return p.nameId() === universe.Universe08.TraitRef.properties.trait.name ||
	        p.nameId() === universe.Universe10.TraitRef.properties.trait.name;
	}
	exports.isTraitProperty = isTraitProperty;
	function isResourceTypesProperty(p) {
	    return p.nameId() === universe.Universe08.Api.properties.resourceTypes.name ||
	        p.nameId() === universe.Universe10.LibraryBase.properties.resourceTypes.name;
	}
	exports.isResourceTypesProperty = isResourceTypesProperty;
	function isResourceTypeProperty(p) {
	    return p.nameId() === universe.Universe08.ResourceTypeRef.properties.resourceType.name ||
	        p.nameId() === universe.Universe10.ResourceTypeRef.properties.resourceType.name;
	}
	exports.isResourceTypeProperty = isResourceTypeProperty;
	function isFacetsProperty(p) {
	    return p.nameId() === universe.Universe10.TypeDeclaration.properties.facets.name;
	}
	exports.isFacetsProperty = isFacetsProperty;
	function isSchemasProperty(p) {
	    return p.nameId() === universe.Universe08.Api.properties.schemas.name ||
	        p.nameId() === universe.Universe10.LibraryBase.properties.schemas.name;
	}
	exports.isSchemasProperty = isSchemasProperty;
	//export function isSignatureProperty(p:hl.IProperty) : boolean {
	//    return p.nameId() === universe.Universe10.Method.properties.signature.name ||
	//    p.nameId() === universe.Universe10.Resource.properties.signature.name;
	//}
	function isResourcesProperty(p) {
	    return p.nameId() === universe.Universe10.Api.properties.resources.name ||
	        p.nameId() === universe.Universe08.Api.properties.resources.name ||
	        p.nameId() === universe.Universe10.Resource.properties.resources.name ||
	        p.nameId() === universe.Universe08.Resource.properties.resources.name;
	}
	exports.isResourcesProperty = isResourcesProperty;
	function isMethodsProperty(p) {
	    return p.nameId() === universe.Universe10.ResourceBase.properties.methods.name ||
	        p.nameId() === universe.Universe08.Resource.properties.methods.name ||
	        p.nameId() === universe.Universe08.ResourceType.properties.methods.name;
	}
	exports.isMethodsProperty = isMethodsProperty;
	function isTypesProperty(p) {
	    return p.nameId() === universe.Universe10.LibraryBase.properties.types.name;
	}
	exports.isTypesProperty = isTypesProperty;
	function isExampleProperty(p) {
	    return p.nameId() === universe.Universe10.TypeDeclaration.properties.example.name ||
	        p.nameId() === "example";
	    //TODO too long to actually list every element having an example, so a couple of checks to cause compile error, and a simple equals check. Also we do not want to affect performance that much.
	}
	exports.isExampleProperty = isExampleProperty;
	function isEnumProperty(p) {
	    return p.nameId() === universe.Universe10.StringTypeDeclaration.properties.enum.name
	        || p.nameId() === universe.Universe10.NumberTypeDeclaration.properties.enum.name
	        || p.nameId() === universe.Universe08.StringTypeDeclaration.properties.enum.name;
	}
	exports.isEnumProperty = isEnumProperty;
	function isExamplesProperty(p) {
	    return p.nameId() === universe.Universe10.TypeDeclaration.properties.example.name || p.nameId() === universe.Universe10.TypeDeclaration.properties.examples.name;
	}
	exports.isExamplesProperty = isExamplesProperty;
	function isValueProperty(p) {
	    return p.nameId() === universe.Universe08.GlobalSchema.properties.value.name;
	}
	exports.isValueProperty = isValueProperty;
	function isUriParametersProperty(p) {
	    return p.nameId() === universe.Universe08.Api.properties.uriParameters.name ||
	        p.nameId() === universe.Universe08.ResourceType.properties.uriParameters.name ||
	        p.nameId() === universe.Universe08.Resource.properties.uriParameters.name ||
	        p.nameId() === universe.Universe10.ResourceBase.properties.uriParameters.name;
	}
	exports.isUriParametersProperty = isUriParametersProperty;
	function isBaseUriParametersProperty(p) {
	    return p.nameId() === universe.Universe08.Resource.properties.baseUriParameters.name ||
	        p.nameId() === universe.Universe08.Api.properties.baseUriParameters.name ||
	        p.nameId() === universe.Universe10.Api.properties.baseUriParameters.name;
	}
	exports.isBaseUriParametersProperty = isBaseUriParametersProperty;
	function isRAMLVersionProperty(p) {
	    return p.nameId() === universe.Universe08.Api.properties.RAMLVersion.name ||
	        p.nameId() === universe.Universe10.Api.properties.RAMLVersion.name;
	}
	exports.isRAMLVersionProperty = isRAMLVersionProperty;
	function isUsesProperty(p) {
	    return p.nameId() === universe.Universe10.FragmentDeclaration.properties.uses.name;
	}
	exports.isUsesProperty = isUsesProperty;
	function isAnnotationTypesProperty(p) {
	    return p.nameId() === universe.Universe10.LibraryBase.properties.annotationTypes.name;
	}
	exports.isAnnotationTypesProperty = isAnnotationTypesProperty;
	function isRepeatProperty(p) {
	    return p.nameId() === universe.Universe08.Parameter.properties.repeat.name ||
	        p.nameId() === universe.Universe10.TypeDeclaration.properties.repeat.name;
	}
	exports.isRepeatProperty = isRepeatProperty;
	/////////////////////// types
	function isMethodType(type) {
	    return type.key() == universe.Universe10.Method ||
	        type.key() == universe.Universe08.Method;
	}
	exports.isMethodType = isMethodType;
	function isApiType(type) {
	    return type.key() == universe.Universe10.Api ||
	        type.key() == universe.Universe08.Api;
	}
	exports.isApiType = isApiType;
	function isBooleanTypeType(type) {
	    return type.key() == universe.Universe10.BooleanType ||
	        type.key() == universe.Universe08.BooleanType;
	}
	exports.isBooleanTypeType = isBooleanTypeType;
	function isMarkdownStringType(type) {
	    return type.key() == universe.Universe10.MarkdownString ||
	        type.key() == universe.Universe08.MarkdownString;
	}
	exports.isMarkdownStringType = isMarkdownStringType;
	function isResourceType(type) {
	    return type.key() == universe.Universe10.Resource ||
	        type.key() == universe.Universe08.Resource;
	}
	exports.isResourceType = isResourceType;
	function isTraitType(type) {
	    return type.key() == universe.Universe10.Trait ||
	        type.key() == universe.Universe08.Trait;
	}
	exports.isTraitType = isTraitType;
	function isTraitRefType(type) {
	    return type.key() == universe.Universe10.TraitRef ||
	        type.key() == universe.Universe08.TraitRef;
	}
	exports.isTraitRefType = isTraitRefType;
	function isResourceTypeRefType(type) {
	    return type.key() == universe.Universe10.ResourceTypeRef ||
	        type.key() == universe.Universe08.ResourceTypeRef;
	}
	exports.isResourceTypeRefType = isResourceTypeRefType;
	function isGlobalSchemaType(type) {
	    return type.key() == universe.Universe08.GlobalSchema;
	}
	exports.isGlobalSchemaType = isGlobalSchemaType;
	function isSecuritySchemaType(type) {
	    return type.key() == universe.Universe10.AbstractSecurityScheme ||
	        type.key() == universe.Universe08.AbstractSecurityScheme;
	}
	exports.isSecuritySchemaType = isSecuritySchemaType;
	function isSecuritySchemeRefType(type) {
	    return type.key() == universe.Universe10.SecuritySchemeRef ||
	        type.key() == universe.Universe08.SecuritySchemeRef;
	}
	exports.isSecuritySchemeRefType = isSecuritySchemeRefType;
	function isTypeDeclarationType(type) {
	    return type.key() == universe.Universe10.TypeDeclaration;
	}
	exports.isTypeDeclarationType = isTypeDeclarationType;
	function isResponseType(type) {
	    return type.key() == universe.Universe10.Response ||
	        type.key() == universe.Universe08.Response;
	}
	exports.isResponseType = isResponseType;
	function isBodyLikeType(type) {
	    return type.key() == universe.Universe08.BodyLike;
	}
	exports.isBodyLikeType = isBodyLikeType;
	function isOverlayType(type) {
	    return type.key() == universe.Universe10.Overlay;
	}
	exports.isOverlayType = isOverlayType;
	function isAnnotationTypeType(type) {
	    return false;
	}
	exports.isAnnotationTypeType = isAnnotationTypeType;
	function isResourceTypeType(type) {
	    return type.key() == universe.Universe10.ResourceType ||
	        type.key() == universe.Universe08.ResourceType;
	}
	exports.isResourceTypeType = isResourceTypeType;
	function isSchemaStringType(type) {
	    return type.key() == universe.Universe10.SchemaString ||
	        type.key() == universe.Universe08.SchemaString;
	}
	exports.isSchemaStringType = isSchemaStringType;
	function isMethodBaseType(type) {
	    return type.key() == universe.Universe10.MethodBase ||
	        type.key() == universe.Universe08.MethodBase;
	}
	exports.isMethodBaseType = isMethodBaseType;
	function isLibraryType(type) {
	    return type.key() == universe.Universe10.Library;
	}
	exports.isLibraryType = isLibraryType;
	function isStringTypeType(type) {
	    return type.key() == universe.Universe10.StringType ||
	        type.key() == universe.Universe08.StringType;
	}
	exports.isStringTypeType = isStringTypeType;
	function isExampleSpecType(type) {
	    return type.key() == universe.Universe10.ExampleSpec;
	}
	exports.isExampleSpecType = isExampleSpecType;
	function isExtensionType(type) {
	    return type.key() == universe.Universe10.Extension;
	}
	exports.isExtensionType = isExtensionType;
	function isTypeDeclarationTypeOrDescendant(type) {
	    return type.isAssignableFrom(universe.Universe10.TypeDeclaration.name);
	}
	exports.isTypeDeclarationTypeOrDescendant = isTypeDeclarationTypeOrDescendant;
	function isDocumentationType(type) {
	    return type.key() == universe.Universe10.DocumentationItem ||
	        type.key() == universe.Universe08.DocumentationItem;
	}
	exports.isDocumentationType = isDocumentationType;
	function isAnnotationRefTypeOrDescendant(type) {
	    return type.isAssignableFrom(universe.Universe10.AnnotationRef.name);
	}
	exports.isAnnotationRefTypeOrDescendant = isAnnotationRefTypeOrDescendant;
	function isApiSibling(type) {
	    return type.isAssignableFrom(universe.Universe10.Api.name) || type.isAssignableFrom(universe.Universe08.Api.name);
	}
	exports.isApiSibling = isApiSibling;
	function isLibraryBaseSibling(type) {
	    return type.isAssignableFrom(universe.Universe10.LibraryBase.name);
	}
	exports.isLibraryBaseSibling = isLibraryBaseSibling;
	function isResourceBaseSibling(type) {
	    return type.isAssignableFrom(universe.Universe10.ResourceBase.name) || type.isAssignableFrom(universe.Universe08.Resource.name);
	}
	exports.isResourceBaseSibling = isResourceBaseSibling;
	function isObjectTypeDeclarationSibling(type) {
	    return type.isAssignableFrom(universe.Universe10.ObjectTypeDeclaration.name);
	}
	exports.isObjectTypeDeclarationSibling = isObjectTypeDeclarationSibling;
	function isTypeDeclarationSibling(type) {
	    return type.isAssignableFrom(universe.Universe10.TypeDeclaration.name);
	}
	exports.isTypeDeclarationSibling = isTypeDeclarationSibling;
	function isMethodBaseSibling(type) {
	    return type.isAssignableFrom(universe.Universe10.MethodBase.name) ||
	        type.isAssignableFrom(universe.Universe08.MethodBase.name);
	}
	exports.isMethodBaseSibling = isMethodBaseSibling;
	function isSecuritySchemePartType(type) {
	    return type.key() == universe.Universe10.SecuritySchemePart ||
	        type.key() == universe.Universe08.SecuritySchemePart;
	}
	exports.isSecuritySchemePartType = isSecuritySchemePartType;
	function isMediaTypeProperty(p) {
	    return p.nameId() === universe.Universe08.Api.properties.mediaType.name ||
	        p.nameId() === universe.Universe10.Api.properties.mediaType.name;
	}
	exports.isMediaTypeProperty = isMediaTypeProperty;
	function isRAML08Type(type) {
	    return type.universe().version() == "RAML08";
	}
	exports.isRAML08Type = isRAML08Type;
	function isRAML10Type(type) {
	    return type.universe().version() == "RAML10";
	}
	exports.isRAML10Type = isRAML10Type;
	//# sourceMappingURL=universeHelpers.js.map

/***/ },
/* 15 */
/***/ function(module, exports, __webpack_require__) {

	/* WEBPACK VAR INJECTION */(function(process) {// Copyright Joyent, Inc. and other Node contributors.
	//
	// Permission is hereby granted, free of charge, to any person obtaining a
	// copy of this software and associated documentation files (the
	// "Software"), to deal in the Software without restriction, including
	// without limitation the rights to use, copy, modify, merge, publish,
	// distribute, sublicense, and/or sell copies of the Software, and to permit
	// persons to whom the Software is furnished to do so, subject to the
	// following conditions:
	//
	// The above copyright notice and this permission notice shall be included
	// in all copies or substantial portions of the Software.
	//
	// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
	// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
	// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
	// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
	// USE OR OTHER DEALINGS IN THE SOFTWARE.

	// resolves . and .. elements in a path array with directory names there
	// must be no slashes, empty elements, or device names (c:\) in the array
	// (so also no leading and trailing slashes - it does not distinguish
	// relative and absolute paths)
	function normalizeArray(parts, allowAboveRoot) {
	  // if the path tries to go above the root, `up` ends up > 0
	  var up = 0;
	  for (var i = parts.length - 1; i >= 0; i--) {
	    var last = parts[i];
	    if (last === '.') {
	      parts.splice(i, 1);
	    } else if (last === '..') {
	      parts.splice(i, 1);
	      up++;
	    } else if (up) {
	      parts.splice(i, 1);
	      up--;
	    }
	  }

	  // if the path is allowed to go above the root, restore leading ..s
	  if (allowAboveRoot) {
	    for (; up--; up) {
	      parts.unshift('..');
	    }
	  }

	  return parts;
	}

	// Split a filename into [root, dir, basename, ext], unix version
	// 'root' is just a slash, or nothing.
	var splitPathRe =
	    /^(\/?|)([\s\S]*?)((?:\.{1,2}|[^\/]+?|)(\.[^.\/]*|))(?:[\/]*)$/;
	var splitPath = function(filename) {
	  return splitPathRe.exec(filename).slice(1);
	};

	// path.resolve([from ...], to)
	// posix version
	exports.resolve = function() {
	  var resolvedPath = '',
	      resolvedAbsolute = false;

	  for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
	    var path = (i >= 0) ? arguments[i] : process.cwd();

	    // Skip empty and invalid entries
	    if (typeof path !== 'string') {
	      throw new TypeError('Arguments to path.resolve must be strings');
	    } else if (!path) {
	      continue;
	    }

	    resolvedPath = path + '/' + resolvedPath;
	    resolvedAbsolute = path.charAt(0) === '/';
	  }

	  // At this point the path should be resolved to a full absolute path, but
	  // handle relative paths to be safe (might happen when process.cwd() fails)

	  // Normalize the path
	  resolvedPath = normalizeArray(filter(resolvedPath.split('/'), function(p) {
	    return !!p;
	  }), !resolvedAbsolute).join('/');

	  return ((resolvedAbsolute ? '/' : '') + resolvedPath) || '.';
	};

	// path.normalize(path)
	// posix version
	exports.normalize = function(path) {
	  var isAbsolute = exports.isAbsolute(path),
	      trailingSlash = substr(path, -1) === '/';

	  // Normalize the path
	  path = normalizeArray(filter(path.split('/'), function(p) {
	    return !!p;
	  }), !isAbsolute).join('/');

	  if (!path && !isAbsolute) {
	    path = '.';
	  }
	  if (path && trailingSlash) {
	    path += '/';
	  }

	  return (isAbsolute ? '/' : '') + path;
	};

	// posix version
	exports.isAbsolute = function(path) {
	  return path.charAt(0) === '/';
	};

	// posix version
	exports.join = function() {
	  var paths = Array.prototype.slice.call(arguments, 0);
	  return exports.normalize(filter(paths, function(p, index) {
	    if (typeof p !== 'string') {
	      throw new TypeError('Arguments to path.join must be strings');
	    }
	    return p;
	  }).join('/'));
	};


	// path.relative(from, to)
	// posix version
	exports.relative = function(from, to) {
	  from = exports.resolve(from).substr(1);
	  to = exports.resolve(to).substr(1);

	  function trim(arr) {
	    var start = 0;
	    for (; start < arr.length; start++) {
	      if (arr[start] !== '') break;
	    }

	    var end = arr.length - 1;
	    for (; end >= 0; end--) {
	      if (arr[end] !== '') break;
	    }

	    if (start > end) return [];
	    return arr.slice(start, end - start + 1);
	  }

	  var fromParts = trim(from.split('/'));
	  var toParts = trim(to.split('/'));

	  var length = Math.min(fromParts.length, toParts.length);
	  var samePartsLength = length;
	  for (var i = 0; i < length; i++) {
	    if (fromParts[i] !== toParts[i]) {
	      samePartsLength = i;
	      break;
	    }
	  }

	  var outputParts = [];
	  for (var i = samePartsLength; i < fromParts.length; i++) {
	    outputParts.push('..');
	  }

	  outputParts = outputParts.concat(toParts.slice(samePartsLength));

	  return outputParts.join('/');
	};

	exports.sep = '/';
	exports.delimiter = ':';

	exports.dirname = function(path) {
	  var result = splitPath(path),
	      root = result[0],
	      dir = result[1];

	  if (!root && !dir) {
	    // No dirname whatsoever
	    return '.';
	  }

	  if (dir) {
	    // It has a dirname, strip trailing slash
	    dir = dir.substr(0, dir.length - 1);
	  }

	  return root + dir;
	};


	exports.basename = function(path, ext) {
	  var f = splitPath(path)[2];
	  // TODO: make this comparison case-insensitive on windows?
	  if (ext && f.substr(-1 * ext.length) === ext) {
	    f = f.substr(0, f.length - ext.length);
	  }
	  return f;
	};


	exports.extname = function(path) {
	  return splitPath(path)[3];
	};

	function filter (xs, f) {
	    if (xs.filter) return xs.filter(f);
	    var res = [];
	    for (var i = 0; i < xs.length; i++) {
	        if (f(xs[i], i, xs)) res.push(xs[i]);
	    }
	    return res;
	}

	// String.prototype.substr - negative index don't work in IE8
	var substr = 'ab'.substr(-1) === 'b'
	    ? function (str, start, len) { return str.substr(start, len) }
	    : function (str, start, len) {
	        if (start < 0) start = str.length + start;
	        return str.substr(start, len);
	    }
	;
	
	/* WEBPACK VAR INJECTION */}.call(exports, __webpack_require__(56)))

/***/ },
/* 16 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../typings/main.d.ts" />
	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var defs = __webpack_require__(35);
	var hl = __webpack_require__(10);
	var _ = __webpack_require__(61);
	var proxy = __webpack_require__(37);
	var def = defs;
	var builder = __webpack_require__(28);
	var search = __webpack_require__(29);
	var mutators = __webpack_require__(38);
	var linter = __webpack_require__(27);
	var expander = __webpack_require__(26);
	var typeBuilder = __webpack_require__(39);
	var universes = __webpack_require__(22);
	var jsyaml = __webpack_require__(23);
	var textutil = __webpack_require__(40);
	var services = def;
	var yaml = __webpack_require__(72);
	var factory10 = __webpack_require__(41);
	var factory08 = __webpack_require__(42);
	var universeHelpers = __webpack_require__(14);
	var resourceRegistry = __webpack_require__(24);
	var rTypes = defs.rt;
	var contentprovider = __webpack_require__(31);
	function qName(x, context) {
	    //var dr=search.declRoot(context);
	    var nm = x.name();
	    if (x.lowLevel().unit() != context.lowLevel().unit()) {
	        var root = context;
	        while (true) {
	            if (root.lowLevel().includePath() || root.parent() == null) {
	                if (!root.unitMap) {
	                    root.unitMap = {};
	                    root.asElement().elements().forEach(function (x) {
	                        if (x.definition().key() == universes.Universe10.UsesDeclaration) {
	                            var mm = x.attr("value");
	                            var unit = root.lowLevel().unit().resolve(mm.value());
	                            if (unit != null) {
	                                root.unitMap[unit.absolutePath()] = x.attr("key").value();
	                            }
	                        }
	                    });
	                }
	                var prefix = root.unitMap[x.lowLevel().unit().absolutePath()];
	                if (prefix) {
	                    return prefix + "." + nm;
	                }
	            }
	            if (!root.parent()) {
	                break;
	            }
	            else {
	                root = root.parent();
	            }
	        }
	    }
	    return nm;
	}
	exports.qName = qName;
	var BasicASTNode = (function () {
	    function BasicASTNode(_node, _parent) {
	        this._node = _node;
	        this._parent = _parent;
	        this._implicit = false;
	        this.values = {};
	        if (_node) {
	            _node.setHighLevelParseResult(this);
	        }
	    }
	    BasicASTNode.prototype.getKind = function () {
	        return hl.NodeKind.BASIC;
	    };
	    BasicASTNode.prototype.asAttr = function () {
	        return null;
	    };
	    BasicASTNode.prototype.asElement = function () {
	        return null;
	    };
	    BasicASTNode.prototype.hashkey = function () {
	        if (!this._hashkey)
	            this._hashkey = this.parent() ? this.parent().hashkey() + "/" + this.name() : this.name();
	        return this._hashkey;
	    };
	    BasicASTNode.prototype.root = function () {
	        if (this.parent()) {
	            return this.parent().root();
	        }
	        return this;
	    };
	    BasicASTNode.prototype.version = function () {
	        return "";
	    };
	    BasicASTNode.prototype.getLowLevelStart = function () {
	        if (this.lowLevel().kind() === jsyaml.Kind.SCALAR) {
	            return this.lowLevel().start();
	        }
	        return this.lowLevel().keyStart();
	    };
	    BasicASTNode.prototype.getLowLevelEnd = function () {
	        if (this.lowLevel().kind() === jsyaml.Kind.SCALAR) {
	            return this.lowLevel().end();
	        }
	        return this.lowLevel().keyEnd();
	    };
	    BasicASTNode.prototype.isSameNode = function (n) {
	        if (n) {
	            if (n.lowLevel().actual() == this.lowLevel().actual()) {
	                return true;
	            }
	        }
	        return false;
	    };
	    BasicASTNode.prototype.checkContextValue = function (name, value, thisObj) {
	        var vl = this.computedValue(name);
	        if (vl && vl.indexOf(value) != -1) {
	            return true; //FIXME
	        }
	        return value == vl || value == 'false';
	    };
	    BasicASTNode.prototype.printDetails = function (indent) {
	        return (indent ? indent : "") + "Unkown\n";
	    };
	    /**
	     * Used for test comparison of two trees. Touching this will require AST tests update.
	     * @param indent
	     * @returns {string}
	     */
	    BasicASTNode.prototype.testSerialize = function (indent) {
	        return (indent ? indent : "") + "Unkown\n";
	    };
	    BasicASTNode.prototype.errors = function () {
	        var errors = [];
	        var q = {
	            accept: function (c) {
	                errors.push(c);
	            },
	            begin: function () {
	            },
	            end: function () {
	            }
	        };
	        this.validate(q);
	        return errors;
	    };
	    BasicASTNode.prototype.markCh = function () {
	        var n = this.lowLevel();
	        n = n._node ? n._node : n;
	        if (n['markCh']) {
	            return true;
	        }
	        n['markCh'] = 1;
	    };
	    BasicASTNode.prototype.unmarkCh = function () {
	        var n = this.lowLevel();
	        n = n._node ? n._node : n;
	        delete n['markCh'];
	    };
	    BasicASTNode.prototype.validate = function (v) {
	        linter.validate(this, v);
	    };
	    BasicASTNode.prototype.allowRecursive = function () {
	        return false;
	    };
	    BasicASTNode.prototype.setComputed = function (name, v) {
	        this.values[name] = v;
	    };
	    BasicASTNode.prototype.computedValue = function (name) {
	        var vl = this.values[name];
	        if (!vl && this.parent()) {
	            return this.parent().computedValue(name);
	        }
	        return vl;
	    };
	    BasicASTNode.prototype.lowLevel = function () {
	        return this._node;
	    };
	    BasicASTNode.prototype.name = function () {
	        var c = this.lowLevel().key();
	        if (!c) {
	            return "";
	        }
	        return c;
	    };
	    BasicASTNode.prototype.optional = function () {
	        var llNode = this.lowLevel();
	        var ownValue = llNode.optional();
	        if (llNode.key() != null) {
	            return ownValue;
	        }
	        var p = this.property();
	        if (!p || !p.isMultiValue()) {
	            return ownValue;
	        }
	        var llParent = llNode.parent();
	        while (llParent && llParent.highLevelNode() == null) {
	            if (llParent.kind() == yaml.Kind.MAPPING) {
	                return llParent.optional();
	            }
	            llParent = llParent.parent();
	        }
	        return ownValue;
	    };
	    BasicASTNode.prototype.parent = function () {
	        return this._parent;
	    };
	    BasicASTNode.prototype.setParent = function (parent) {
	        this._parent = parent;
	    };
	    BasicASTNode.prototype.isElement = function () {
	        return false;
	    };
	    BasicASTNode.prototype.directChildren = function () {
	        return this.children();
	    };
	    BasicASTNode.prototype.children = function () {
	        return [];
	    };
	    BasicASTNode.prototype.isAttached = function () {
	        return this.parent() != null;
	    };
	    BasicASTNode.prototype.isImplicit = function () {
	        return this._implicit;
	    };
	    BasicASTNode.prototype.isAttr = function () {
	        return false;
	    };
	    BasicASTNode.prototype.isUnknown = function () {
	        return true;
	    };
	    BasicASTNode.prototype.id = function () {
	        var _this = this;
	        if (this._parent) {
	            var parentId = this.parent().id();
	            parentId += "." + this.name();
	            var sameName = this.parent().directChildren().filter(function (x) { return x.name() == _this.name(); });
	            if (sameName.length > 1) {
	                var ind = sameName.indexOf(this);
	                parentId += "[" + ind + "]";
	            }
	            return parentId;
	        }
	        return "";
	    };
	    BasicASTNode.prototype.localId = function () {
	        return this.name();
	    };
	    BasicASTNode.prototype.fullLocalId = function () {
	        var _this = this;
	        if (this._parent) {
	            var result = "." + this.name();
	            var sameName = this.parent().directChildren().filter(function (x) { return x.name() == _this.name(); });
	            if (sameName.length > 1) {
	                var ind = sameName.indexOf(this);
	                result += "[" + ind + "]";
	            }
	            return result;
	        }
	        return this.localId();
	    };
	    BasicASTNode.prototype.property = function () {
	        return null;
	    };
	    return BasicASTNode;
	}());
	exports.BasicASTNode = BasicASTNode;
	var StructuredValue = (function () {
	    function StructuredValue(node, _parent, _pr, kv) {
	        if (kv === void 0) { kv = null; }
	        this.node = node;
	        this._parent = _parent;
	        this.kv = kv;
	        this._pr = _pr;
	    }
	    StructuredValue.prototype.valueName = function () {
	        var res = null;
	        if (this.kv) {
	            res = this.kv;
	        }
	        res = this.node.key();
	        if (this._pr && this._pr.isAnnotation()) {
	            if (res && res.charAt(0) == '(') {
	                res = res.substring(1, res.length - 1);
	            }
	        }
	        return res;
	    };
	    StructuredValue.prototype.children = function () {
	        return this.node.children().map(function (x) { return new StructuredValue(x, null, null); });
	    };
	    StructuredValue.prototype.lowLevel = function () {
	        return this.node;
	    };
	    StructuredValue.prototype.toHighLevel = function (parent) {
	        if (!parent && this._parent)
	            parent = this._parent;
	        if (this._hl) {
	            return this._hl;
	        }
	        var vn = this.valueName();
	        var cands = search.referenceTargets(this._pr, parent).filter(function (x) { return qName(x, parent) == vn; });
	        if (cands && cands[0]) {
	            var tp = cands[0].localType();
	            var node = new ASTNodeImpl(this.node, parent, tp, this._pr);
	            if (this._pr) {
	                this._pr.childRestrictions().forEach(function (y) {
	                    node.setComputed(y.name, y.value);
	                });
	            }
	            this._hl = node;
	            return node;
	        }
	        //if (this._pr.range()){
	        //    var node=new ASTNodeImpl(parent.lowLevel(),parent,this._pr.range(),this._pr);
	        //    if (this._pr){
	        //        this._pr.childRestrictions().forEach(y=>{
	        //            node.setComputed(y.name,y.value)
	        //        })
	        //    }
	        //    return node;
	        //}
	        return null;
	    };
	    StructuredValue.prototype.toHighLevel2 = function (parent) {
	        if (!parent && this._parent)
	            parent = this._parent;
	        var vn = this.valueName();
	        var cands = search.referenceTargets(this._pr, parent).filter(function (x) { return qName(x, parent) == vn; });
	        if (cands && cands[0]) {
	            var tp = cands[0].localType();
	            var node = new ASTNodeImpl(this.node, parent, tp, this._pr);
	            if (this._pr) {
	                this._pr.childRestrictions().forEach(function (y) {
	                    node.setComputed(y.name, y.value);
	                });
	            }
	            return node;
	        }
	        if (this._pr.range()) {
	            var node = new ASTNodeImpl(this.node.parent(), parent, this._pr.range(), this._pr);
	            if (this._pr) {
	                this._pr.childRestrictions().forEach(function (y) {
	                    node.setComputed(y.name, y.value);
	                });
	            }
	            return node;
	        }
	        return null;
	    };
	    return StructuredValue;
	}());
	exports.StructuredValue = StructuredValue;
	var ASTPropImpl = (function (_super) {
	    __extends(ASTPropImpl, _super);
	    function ASTPropImpl(node, parent, _def, _prop, fromKey) {
	        if (fromKey === void 0) { fromKey = false; }
	        _super.call(this, node, parent);
	        this._def = _def;
	        this._prop = _prop;
	        this.fromKey = fromKey;
	    }
	    ASTPropImpl.prototype.definition = function () {
	        return this._def;
	    };
	    ASTPropImpl.prototype.asAttr = function () {
	        return this;
	    };
	    ASTPropImpl.prototype.errors = function () {
	        var errors = [];
	        var q = {
	            accept: function (c) {
	                if (c.node === this) {
	                    errors.push(c);
	                }
	            },
	            begin: function () {
	            },
	            end: function () {
	            }
	        };
	        this.parent().validate(q);
	        return errors;
	    };
	    ASTPropImpl.prototype.isString = function () {
	        if (this._def) {
	            if (this._def.key() === universes.Universe08.StringType || this._def.key() == universes.Universe10.StringType) {
	                return true;
	            }
	        }
	        return false;
	    };
	    ASTPropImpl.prototype.isAnnotatedScalar = function () {
	        if (!this.property().isAnnotation() && !this.property().isKey()) {
	            return this.lowLevel().isAnnotatedScalar();
	        }
	        return false;
	    };
	    ASTPropImpl.prototype.annotations = function () {
	        var ch = this.lowLevel().children();
	        var annotations = [];
	        var u = this.definition().universe().type(universes.Universe10.Annotable.name);
	        if (!u) {
	            return annotations;
	        }
	        var pr = u.property("annotations");
	        for (var i = 0; i < ch.length; i++) {
	            var child = ch[i];
	            var key = child.key();
	            if (key[0] == ("(") && key[key.length - 1] == (")")) {
	                var attr = new ASTPropImpl(child, this.parent(), pr.range(), pr);
	                annotations.push(attr);
	            }
	        }
	        return annotations;
	    };
	    ASTPropImpl.prototype.getKind = function () {
	        return hl.NodeKind.ATTRIBUTE;
	    };
	    ASTPropImpl.prototype.owningWrapper = function () {
	        return {
	            node: this.parent().wrapperNode(),
	            property: this.name()
	        };
	    };
	    ASTPropImpl.prototype.patchType = function (t) {
	        this._def = t;
	    };
	    ASTPropImpl.prototype.findReferenceDeclaration = function () {
	        var _this = this;
	        var targets = search.referenceTargets(this.property(), this.parent());
	        var vl = this.value();
	        if (vl instanceof StructuredValue) {
	            var st = vl;
	            var nm = st.valueName();
	        }
	        else {
	            var nm = "" + vl;
	        }
	        var t = _.find(targets, function (x) { return qName(x, _this.parent()) == nm; });
	        return t;
	    };
	    ASTPropImpl.prototype.findReferencedValue = function () {
	        var c = this.findReferenceDeclaration();
	        if (c) {
	            var vl = c.attr("value");
	            var ck = c.definition().key();
	            if (ck === universes.Universe08.GlobalSchema) {
	                if (vl) {
	                    var actualValue = vl.value();
	                    if (actualValue) {
	                        var rf = linter.isValid(this._def, this.parent(), actualValue, vl.property());
	                        return rf;
	                    }
	                }
	                return null;
	            }
	        }
	        return c;
	    };
	    ASTPropImpl.prototype.isElement = function () {
	        return false;
	    };
	    ASTPropImpl.prototype.property = function () {
	        return this._prop;
	    };
	    ASTPropImpl.prototype.convertMultivalueToString = function (value) {
	        //|\n  xxx\n  yyy\n  zzz
	        var gap = 0;
	        var pos = 2;
	        while (value[pos] == ' ') {
	            gap++;
	            pos++;
	        }
	        //console.log('gap: ' + gap);
	        var lines = textutil.splitOnLines(value);
	        lines = lines.map(function (line) {
	            //console.log('line: ' + line);
	            return line.substring(gap, line.length);
	        });
	        return lines.join('');
	    };
	    ASTPropImpl.prototype.overrideValue = function (value) {
	        this._value = value;
	    };
	    ASTPropImpl.prototype.value = function () {
	        if (this._value) {
	            return this._value;
	        }
	        if (this._computed) {
	            return this.computedValue(this.property().nameId());
	        }
	        if (this.fromKey) {
	            return this._node.key();
	        }
	        if (this.property().isAnnotation() && this._node.key() && this._node.key() != 'annotations') {
	            return new StructuredValue(this._node, this.parent(), this._prop);
	        }
	        var isString = this.property() != null && universeHelpers.isStringTypeType(this.property().range());
	        var actualValue = this._node.value(isString); //TODO FIXME
	        if (this.property().isSelfNode()) {
	            if (!actualValue || actualValue instanceof jsyaml.ASTNode) {
	                actualValue = this._node;
	                if (actualValue.children().length == 0) {
	                    actualValue = null;
	                }
	            }
	        }
	        if (actualValue instanceof jsyaml.ASTNode || actualValue instanceof proxy.LowLevelProxyNode) {
	            var isAnnotatedScalar = false;
	            if (!this.property().range().hasStructure()) {
	                if (this._node.isAnnotatedScalar()) {
	                    this._node.children().forEach(function (x) {
	                        if (x.key() === "value") {
	                            actualValue = x.value(isString);
	                            isAnnotatedScalar = true;
	                        }
	                    });
	                }
	            }
	            if (!isAnnotatedScalar) {
	                if (this._sval) {
	                    return this._sval;
	                }
	                this._sval = new StructuredValue(actualValue, this.parent(), this._prop);
	                return this._sval;
	            }
	        }
	        if (typeof (actualValue) == 'string' && textutil.isMultiLineValue(actualValue)) {
	            var res = this.convertMultivalueToString(actualValue);
	            //console.log('converted: [' + textutil.replaceNewlines(res) + ']');
	            return res;
	        }
	        return actualValue;
	    };
	    ASTPropImpl.prototype.name = function () {
	        return this._prop.nameId();
	    };
	    ASTPropImpl.prototype.printDetails = function (indent) {
	        var className = this.definition().nameId();
	        var definitionClassName = this.property().range().nameId();
	        var result = (indent ? indent : "") +
	            (this.name() + " : " + className
	                + "[" + definitionClassName + "]"
	                + "  =  " + this.value()) + (this.property().isKey() && this.optional() ? "?" : "");
	        "\n";
	        if (this.value() instanceof StructuredValue) {
	            var structuredHighLevel = this.value().toHighLevel();
	            if (structuredHighLevel && structuredHighLevel.printDetails) {
	                result += structuredHighLevel.printDetails(indent + "\t");
	            }
	        }
	        return result;
	    };
	    /**
	     * Used for test comparison of two trees. Touching this will require AST tests update.
	     * @param indent
	     * @returns {string}
	     */
	    ASTPropImpl.prototype.testSerialize = function (indent) {
	        var className = this.definition().nameId();
	        var result = (indent ? indent : "") +
	            (this.name() + " : " + className
	                + "  =  " + this.value()) +
	            "\n";
	        if (this.value() instanceof StructuredValue) {
	            var structuredHighLevel = this.value().toHighLevel();
	            if (structuredHighLevel && structuredHighLevel.testSerialize) {
	                result += structuredHighLevel.testSerialize((indent ? indent : "") + "  ");
	            }
	            else {
	                var lowLevel = this.value().lowLevel();
	                var dumpObject = lowLevel.dumpToObject();
	                var dump = JSON.stringify(dumpObject);
	                var indentedDump = "";
	                var dumpLines = dump.split("\n");
	                dumpLines.forEach(function (dumpLine) { return indentedDump += ((indent ? indent : "") + "  " + dumpLine + "\n"); });
	                result += indentedDump + "\n";
	            }
	        }
	        return result;
	    };
	    ASTPropImpl.prototype.isAttr = function () {
	        return true;
	    };
	    ASTPropImpl.prototype.isUnknown = function () {
	        return false;
	    };
	    ASTPropImpl.prototype.setValue = function (value) {
	        mutators.setValue(this, value);
	    };
	    ASTPropImpl.prototype.setKey = function (value) {
	        mutators.setKey(this, value);
	    };
	    ASTPropImpl.prototype.children = function () {
	        return [];
	    };
	    ASTPropImpl.prototype.addStringValue = function (value) {
	        mutators.addStringValue(this, value);
	    };
	    ASTPropImpl.prototype.addStructuredValue = function (sv) {
	        mutators.addStructuredValue(this, sv);
	    };
	    ASTPropImpl.prototype.addValue = function (value) {
	        if (!this.property().isMultiValue())
	            throw new Error("setValue(string) only apply to multi-values properties");
	        if (typeof value == 'string') {
	            this.addStringValue(value);
	        }
	        else {
	            this.addStructuredValue(value);
	        }
	    };
	    ASTPropImpl.prototype.isEmbedded = function () {
	        var keyname = this.lowLevel().asMapping().key.value;
	        //console.log('propery: ' + this.property().name());
	        //console.log('mapping: ' + keyname);
	        return this.property().canBeValue() && keyname != this.property().nameId();
	    };
	    ASTPropImpl.prototype.remove = function () {
	        mutators.removeAttr(this);
	    };
	    ASTPropImpl.prototype.setValues = function (values) {
	        mutators.setValues(this, values);
	    };
	    ASTPropImpl.prototype.isEmpty = function () {
	        if (!this.property().isMultiValue())
	            throw new Error("isEmpty() only apply to multi-values attributes");
	        //console.log('remove: ' + this.name());
	        var node = this.parent();
	        var llnode = node.lowLevel();
	        //node.lowLevel().show('Parent:');
	        var attrs = node.attributes(this.name());
	        //console.log('attributes: ' + attrs.length);
	        if (attrs.length == 0) {
	            return true;
	        }
	        else if (attrs.length == 1) {
	            var anode = attrs[0].lowLevel();
	            //console.log('attribute : ' + anode.kindName());
	            //anode.show("ATTR:");
	            if (anode.isMapping() && anode.value() == null) {
	                // that's crazy but it means zero length array indeed )
	                return true;
	            }
	            else {
	                return false;
	            }
	        }
	        else {
	            return false;
	        }
	    };
	    return ASTPropImpl;
	}(BasicASTNode));
	exports.ASTPropImpl = ASTPropImpl;
	var nodeBuilder = new builder.BasicNodeBuilder();
	(function (OverlayMergeMode) {
	    OverlayMergeMode[OverlayMergeMode["MERGE"] = 0] = "MERGE";
	    OverlayMergeMode[OverlayMergeMode["AGGREGATE"] = 1] = "AGGREGATE";
	})(exports.OverlayMergeMode || (exports.OverlayMergeMode = {}));
	var OverlayMergeMode = exports.OverlayMergeMode;
	var LowLevelWrapperForTypeSystem = (function (_super) {
	    __extends(LowLevelWrapperForTypeSystem, _super);
	    function LowLevelWrapperForTypeSystem(_node, _highLevelRoot) {
	        _super.call(this);
	        this._node = _node;
	        this._highLevelRoot = _highLevelRoot;
	    }
	    LowLevelWrapperForTypeSystem.prototype.contentProvider = function () {
	        var root = this._node && this._node.unit() && ((this._node.includePath && this._node.includePath()) ? this._node.unit().resolve(this._node.includePath()) : this._node.unit());
	        return new contentprovider.ContentProvider(root);
	    };
	    LowLevelWrapperForTypeSystem.prototype.key = function () {
	        var vl = this._node.key();
	        if (this._node.optional()) {
	            vl = vl + "?";
	        }
	        return vl;
	    };
	    LowLevelWrapperForTypeSystem.prototype.value = function () {
	        var vk = this._node.valueKind();
	        if (vk === yaml.Kind.SEQ) {
	            return this.children().map(function (x) { return x.value(); });
	        }
	        else if (vk === yaml.Kind.MAP || vk === yaml.Kind.ANCHOR_REF) {
	            var vl = this._node.dumpToObject(true);
	            return vl[this.key()];
	        }
	        else if (this._node.kind() == yaml.Kind.MAP) {
	            var vl = this._node.dumpToObject(true);
	            return vl;
	        }
	        return this._node.value();
	    };
	    LowLevelWrapperForTypeSystem.prototype.children = function () {
	        var _this = this;
	        if (this.key() == "uses" && !this._node.parent().parent()) {
	            return this._node.children().map(function (x) { return new UsesNodeWrapperFoTypeSystem(x, _this._highLevelRoot); });
	        }
	        return this._node.children().map(function (x) { return new LowLevelWrapperForTypeSystem(x, _this._highLevelRoot); });
	    };
	    LowLevelWrapperForTypeSystem.prototype.childWithKey = function (k) {
	        var mm = this.children();
	        for (var i = 0; i < mm.length; i++) {
	            if (mm[i].key() == k) {
	                return mm[i];
	            }
	        }
	        return null;
	    };
	    LowLevelWrapperForTypeSystem.prototype.kind = function () {
	        if (this._node.valueKind() == yaml.Kind.MAPPING) {
	            return rTypes.NodeKind.MAP;
	        }
	        if (this._node.valueKind() == yaml.Kind.MAP) {
	            return rTypes.NodeKind.MAP;
	        }
	        if (this._node.kind() == yaml.Kind.MAP) {
	            return rTypes.NodeKind.MAP;
	        }
	        if (this._node.valueKind() == yaml.Kind.SEQ) {
	            return rTypes.NodeKind.ARRAY;
	        }
	        if (this._node.valueKind() == yaml.Kind.INCLUDE_REF) {
	            if (this._node.children().length > 0) {
	                //we can safely assume that it is map in the type system in this case
	                return rTypes.NodeKind.MAP;
	            }
	        }
	        return rTypes.NodeKind.SCALAR;
	    };
	    LowLevelWrapperForTypeSystem.prototype.getSource = function () {
	        if (!this._node)
	            return null;
	        var highLevelNode = this._node.highLevelNode();
	        if (!highLevelNode) {
	            var position = this._node.start();
	            var result = search.deepFindNode(this._highLevelRoot, position, position, true, false);
	            if (result) {
	                this._node.setHighLevelParseResult(result);
	                if (result instanceof ASTNodeImpl) {
	                    this._node.setHighLevelNode(result);
	                }
	            }
	            return result;
	        }
	    };
	    return LowLevelWrapperForTypeSystem;
	}(defs.SourceProvider));
	exports.LowLevelWrapperForTypeSystem = LowLevelWrapperForTypeSystem;
	var UsesNodeWrapperFoTypeSystem = (function (_super) {
	    __extends(UsesNodeWrapperFoTypeSystem, _super);
	    function UsesNodeWrapperFoTypeSystem() {
	        _super.apply(this, arguments);
	    }
	    UsesNodeWrapperFoTypeSystem.prototype.children = function () {
	        var s = this._node.unit().resolve(this.value());
	        if (s && s.isRAMLUnit()) {
	            return new LowLevelWrapperForTypeSystem(s.ast(), this._highLevelRoot).children();
	        }
	        return [];
	    };
	    UsesNodeWrapperFoTypeSystem.prototype.anchor = function () {
	        return this._node.actual();
	    };
	    UsesNodeWrapperFoTypeSystem.prototype.childWithKey = function (k) {
	        var mm = this.children();
	        for (var i = 0; i < mm.length; i++) {
	            if (mm[i].key() == k) {
	                return mm[i];
	            }
	        }
	        return null;
	    };
	    return UsesNodeWrapperFoTypeSystem;
	}(LowLevelWrapperForTypeSystem));
	exports.UsesNodeWrapperFoTypeSystem = UsesNodeWrapperFoTypeSystem;
	var ASTNodeImpl = (function (_super) {
	    __extends(ASTNodeImpl, _super);
	    function ASTNodeImpl(node, parent, _def, _prop) {
	        _super.call(this, node, parent);
	        this._def = _def;
	        this._prop = _prop;
	        this._expanded = false;
	        this._allowQuestion = false;
	        this._subTypesCache = null;
	        this._auxChecked = false;
	        /**
	         * Depending on the merge mode, overlays and extensions are either merged with the master, or their trees are joined via aggregation
	         * @type {OverlayMergeMode}
	         */
	        this.overlayMergeMode = OverlayMergeMode.MERGE;
	        if (node) {
	            node.setHighLevelNode(this);
	        }
	        if (node instanceof proxy.LowLevelProxyNode) {
	            this._expanded = true;
	        }
	    }
	    ASTNodeImpl.prototype.validate = function (v) {
	        var k = this.definition().key();
	        if (k == universes.Universe10.Api || k == universes.Universe08.Api || k == universes.Universe10.Overlay || k == universes.Universe10.Extension) {
	            if (!this.isExpanded()) {
	                var nm = expander.expandTraitsAndResourceTypes(this.wrapperNode());
	                var hlnode = nm.highLevel();
	                hlnode._expanded = true;
	                hlnode.validate(v);
	                return;
	            }
	        }
	        linter.validate(this, v);
	    };
	    ASTNodeImpl.prototype.types = function () {
	        var _this = this;
	        if (!this._types) {
	            if (this.parent() && (this.definition().key() !== universes.Universe10.Library)) {
	                return this.parent().types();
	            }
	            else {
	                var c = this.lowLevel().actual();
	                if (c.types) {
	                    return c.types;
	                }
	                this._types = rTypes.parseFromAST(new LowLevelWrapperForTypeSystem(this.lowLevel(), this));
	                this._types.types().forEach(function (x) {
	                    var convertedType = typeBuilder.convertType(_this, x);
	                    // if (defs.instanceOfHasExtra(convertedType)) {
	                    convertedType.putExtra(defs.USER_DEFINED_EXTRA, true);
	                    // }
	                });
	                c.types = this._types;
	            }
	        }
	        return this._types;
	    };
	    ASTNodeImpl.prototype.setTypes = function (t) {
	        this._types = t;
	    };
	    ASTNodeImpl.prototype.parsedType = function () {
	        if (!this._ptype) {
	            if (this.property() && this.property().nameId() == universes.Universe10.MethodBase.properties.body.name) {
	                this._ptype = rTypes.parseTypeFromAST(this.name(), new LowLevelWrapperForTypeSystem(this.lowLevel(), this), this.types(), true, false, false);
	            }
	            else {
	                var annotation = this.property() && this.property().nameId() == universes.Universe10.LibraryBase.properties.annotationTypes.name;
	                var tl = (!this.property()) || (this.property().nameId() == universes.Universe10.LibraryBase.properties.types.name || this.property().nameId() == universes.Universe10.LibraryBase.properties.schemas.name);
	                this._ptype = rTypes.parseTypeFromAST(this.name(), new LowLevelWrapperForTypeSystem(this.lowLevel(), this), this.types(), false, annotation, tl);
	            }
	            if (this.property() && universeHelpers.isTypesProperty(this.property())
	                && this.parent() && universeHelpers.isApiType(this.parent().definition())) {
	                //top level types declared via "types"
	                // this._ptype.setExtra()
	                if (this._ptype.putExtra) {
	                    this._ptype.putExtra(defs.DEFINED_IN_TYPES_EXTRA, true);
	                }
	            }
	            var potentialHasExtra = this._ptype;
	            potentialHasExtra.putExtra(defs.USER_DEFINED_EXTRA, true);
	        }
	        this._ptype.putExtra(defs.SOURCE_EXTRA, this);
	        return this._ptype;
	    };
	    ASTNodeImpl.prototype.localType = function () {
	        return typeBuilder.typeFromNode(this);
	    };
	    ASTNodeImpl.prototype.patchProp = function (pr) {
	        this._prop = pr;
	    };
	    ASTNodeImpl.prototype.getKind = function () {
	        return hl.NodeKind.NODE;
	    };
	    ASTNodeImpl.prototype.wrapperNode = function () {
	        if (!this._wrapperNode) {
	            //forcing discrimination
	            this.children();
	            this._wrapperNode = this.buildWrapperNode();
	        }
	        return this._wrapperNode;
	    };
	    ASTNodeImpl.prototype.asElement = function () {
	        return this;
	    };
	    ASTNodeImpl.prototype.buildWrapperNode = function () {
	        var ramlVersion = this.definition().universe().version();
	        if (ramlVersion == 'RAML10') {
	            return factory10.buildWrapperNode(this);
	        }
	        else if (ramlVersion == 'RAML08') {
	            return factory08.buildWrapperNode(this);
	        }
	        return null;
	    };
	    ASTNodeImpl.prototype.propertiesAllowedToUse = function () {
	        var _this = this;
	        return this.definition().allProperties().filter(function (x) { return _this.isAllowedToUse(x); });
	    };
	    ASTNodeImpl.prototype.isAllowedToUse = function (p) {
	        var _this = this;
	        var ok = true;
	        if (p.getAdapter(services.RAMLPropertyService).isSystem()) {
	            return false;
	        }
	        p.getContextRequirements().forEach(function (y) {
	            if (y.name.indexOf('(') != -1) {
	                //TODO HANDLE IT LATER
	                return true;
	            }
	            var vl = _this.computedValue(y.name);
	            if (vl) {
	                ok = ok && (vl == y.value);
	            }
	            else {
	                if (y.value) {
	                    ok = false;
	                }
	            }
	        });
	        return ok;
	    };
	    ASTNodeImpl.prototype.allowRecursive = function () {
	        if (this.definition().getAdapter(services.RAMLService).isUserDefined()) {
	            return true;
	        }
	        return false;
	    };
	    ASTNodeImpl.prototype.setWrapperNode = function (node) {
	        this._wrapperNode = node;
	    };
	    ASTNodeImpl.prototype.setAssociatedType = function (d) {
	        this._associatedDef = d;
	    };
	    ASTNodeImpl.prototype.associatedType = function () {
	        return this._associatedDef;
	    };
	    ASTNodeImpl.prototype.knownIds = function () {
	        //initializing ids if needed
	        //TODO refactor workaround
	        this.isAuxilary();
	        if (this._knownIds) {
	            return this._knownIds;
	        }
	        else {
	            return {};
	        }
	    };
	    ASTNodeImpl.prototype.findById = function (id) {
	        var _this = this;
	        var v = this._knownIds;
	        if (!v) {
	            this._knownIds = {};
	            var all = search.allChildren(this);
	            all.forEach(function (x) { return _this._knownIds[x.id()] = x; });
	        }
	        return this._knownIds[id];
	    };
	    ASTNodeImpl.prototype.isAuxilary = function () {
	        if (this._isAux) {
	            return true;
	        }
	        if (this._auxChecked) {
	            return false;
	        }
	        this._auxChecked = true;
	        var masterApi = this.getMaster();
	        if (masterApi) {
	            this._isAux = true;
	            this.initilizeKnownIDs(masterApi);
	            return true;
	        }
	        return false;
	    };
	    ASTNodeImpl.prototype.initilizeKnownIDs = function (api) {
	        var _this = this;
	        this._knownIds = {};
	        var allChildren = search.allChildren(api);
	        allChildren.forEach(function (x) { return _this._knownIds[x.id()] = x; });
	        this._knownIds[""] = api;
	    };
	    ASTNodeImpl.prototype.getMaster = function () {
	        if (this.masterApi) {
	            return this.masterApi;
	        }
	        return this.calculateMasterByRef();
	    };
	    /**
	     * Forcefully sets a master unit for this API, which may be different from the one, current unit points to
	     * via masterRef.
	     * @param master
	     */
	    ASTNodeImpl.prototype.overrideMaster = function (master) {
	        this.masterApi = master;
	        this.resetAuxilaryState();
	    };
	    ASTNodeImpl.prototype.setMergeMode = function (mergeMode) {
	        this.overlayMergeMode = mergeMode;
	        this.resetAuxilaryState();
	    };
	    ASTNodeImpl.prototype.getMergeMode = function () {
	        return this.overlayMergeMode;
	    };
	    ASTNodeImpl.prototype.calculateMasterByRef = function () {
	        var unit = this.lowLevel().unit();
	        if (!unit)
	            return null;
	        var masterReferenceNode = unit.getMasterReferenceNode();
	        if (!masterReferenceNode || !masterReferenceNode.value()) {
	            return null;
	        }
	        var masterPath = masterReferenceNode.value();
	        var masterUnit = this.lowLevel().unit().project().resolve(this.lowLevel().unit().path(), masterPath);
	        if (!masterUnit) {
	            return null;
	        }
	        var result = fromUnit(masterUnit);
	        result.setMergeMode(this.overlayMergeMode);
	        return result;
	    };
	    ASTNodeImpl.prototype.resetAuxilaryState = function () {
	        this._isAux = false;
	        this._auxChecked = false;
	        this._knownIds = null;
	        this.clearChildrenCache();
	    };
	    ASTNodeImpl.prototype.printDetails = function (indent) {
	        var result = "";
	        if (!indent)
	            indent = "";
	        var classname = this.definition().nameId();
	        var definitionClasName = this.property() ? this.property().range().nameId() : "";
	        var parentPropertyName = this.property() ? this.property().nameId() : "";
	        result += indent + classname + "[" + definitionClasName + "]" + " <--- " + parentPropertyName + "\n";
	        this.children().forEach(function (child) {
	            result += child.printDetails(indent + "\t");
	        });
	        return result;
	    };
	    /**
	     * Used for test comparison of two trees. Touching this will require AST tests update.
	     * @param indent
	     * @returns {string}
	     */
	    ASTNodeImpl.prototype.testSerialize = function (indent) {
	        var result = "";
	        if (!indent)
	            indent = "";
	        var classname = this.definition().nameId();
	        var parentPropertyName = this.property() ? this.property().nameId() : "";
	        result += indent + classname + " <-- " + parentPropertyName + "\n";
	        this.children().forEach(function (child) {
	            if (child.testSerialize) {
	                result += child.testSerialize(indent + "  ");
	            }
	        });
	        return result;
	    };
	    ASTNodeImpl.prototype.getExtractedChildren = function () {
	        var r = this.root();
	        if (r.isAuxilary()) {
	            if (r._knownIds) {
	                var i = r._knownIds[this.id()];
	                if (i) {
	                    return i.children();
	                }
	            }
	            return [];
	        }
	        return [];
	    };
	    ASTNodeImpl.prototype.allowsQuestion = function () {
	        return this._allowQuestion || this.definition().getAdapter(services.RAMLService).getAllowQuestion();
	    };
	    ASTNodeImpl.prototype.findReferences = function () {
	        var _this = this;
	        var rs = [];
	        search.refFinder(this.root(), this, rs);
	        //TODO FIX ME
	        if (rs.length > 1) {
	            rs = rs.filter(function (x) { return x != _this && x.parent() != _this; });
	        }
	        //filtering out equal results
	        var filteredReferences = [];
	        rs.forEach(function (ref) {
	            if (!_.find(filteredReferences, function (existing) { return existing == ref; })) {
	                filteredReferences.push(ref);
	            }
	        });
	        return filteredReferences;
	    };
	    ASTNodeImpl.prototype.setNamePatch = function (s) {
	        this._patchedName = s;
	    };
	    ASTNodeImpl.prototype.isNamePatch = function () {
	        return this._patchedName;
	    };
	    ASTNodeImpl.prototype.name = function () {
	        if (this._patchedName) {
	            return this._patchedName;
	        }
	        var ka = _.find(this.directChildren(), function (x) { return x.property() && x.property().getAdapter(services.RAMLPropertyService).isKey(); });
	        if (ka && ka instanceof ASTPropImpl) {
	            var c = ka.value();
	            if (c) {
	                var io = c.indexOf(':');
	                if (io != -1) {
	                    return c.substring(0, io);
	                }
	            }
	            return c;
	        }
	        return _super.prototype.name.call(this);
	    };
	    ASTNodeImpl.prototype.findElementAtOffset = function (n) {
	        return this._findNode(this, n, n);
	    };
	    ASTNodeImpl.prototype.isElement = function () {
	        return true;
	    };
	    ASTNodeImpl.prototype.universe = function () {
	        if (this._universe) {
	            return this._universe;
	        }
	        return this.definition().universe();
	    };
	    ASTNodeImpl.prototype.setUniverse = function (u) {
	        this._universe = u;
	    };
	    ASTNodeImpl.prototype._findNode = function (n, offset, end) {
	        var _this = this;
	        if (n == null) {
	            return null;
	        }
	        if (n.lowLevel()) {
	            //var node:ASTNode=<ASTNode>n;
	            if (n.lowLevel().start() <= offset && n.lowLevel().end() >= end) {
	                var res = n;
	                //TODO INCLUDES
	                n.elements().forEach(function (x) {
	                    if (x.lowLevel().unit() != n.lowLevel().unit()) {
	                        return;
	                    }
	                    var m = _this._findNode(x, offset, end);
	                    if (m) {
	                        res = m;
	                    }
	                });
	                return res;
	            }
	        }
	        return null;
	    };
	    ASTNodeImpl.prototype.isStub = function () {
	        return (!this.lowLevel().unit()) || this.lowLevel().unit().isStubUnit();
	    };
	    ASTNodeImpl.prototype.add = function (node) {
	        mutators.addToNode(this, node);
	    };
	    ASTNodeImpl.prototype.remove = function (node) {
	        mutators.removeNodeFrom(this, node);
	    };
	    ASTNodeImpl.prototype.dump = function (flavor) {
	        return this._node.dump();
	    };
	    ASTNodeImpl.prototype.patchType = function (d) {
	        this._def = d;
	        var ass = this._associatedDef;
	        this._associatedDef = null;
	        this._children = null;
	    };
	    ASTNodeImpl.prototype.children = function () {
	        var lowLevel = this.lowLevel();
	        if (lowLevel && lowLevel.isValueInclude && lowLevel.isValueInclude() && resourceRegistry.isWaitingFor(lowLevel.includePath())) {
	            this._children = null;
	            return [];
	        }
	        if (this._children) {
	            return this.mergeChildren(this._children, this.getExtractedChildren());
	        }
	        if (this._node) {
	            this._children = nodeBuilder.process(this, this._node.children());
	            this._children = this._children.filter(function (x) { return x != null; });
	            //FIXME
	            return this.mergeChildren(this._children, this.getExtractedChildren());
	        }
	        return [];
	    };
	    ASTNodeImpl.prototype.mergeChildren = function (originalChildren, masterChildren) {
	        var _this = this;
	        var root = this.root();
	        if (root.overlayMergeMode == OverlayMergeMode.AGGREGATE) {
	            //simply joining the sets
	            return originalChildren.concat(masterChildren);
	        }
	        else if (root.overlayMergeMode == OverlayMergeMode.MERGE) {
	            var result = [];
	            originalChildren.forEach(function (originalChild) {
	                var masterCounterpart = _.find(masterChildren, function (masterChild) { return masterChild.fullLocalId() == originalChild.fullLocalId(); });
	                if (!masterCounterpart) {
	                    //we dont have a counterpart, so simply adding to result
	                    result.push(originalChild);
	                }
	                else {
	                    //there is a counterpart, so deciding what to do:
	                    _this.mergeChild(result, originalChild, masterCounterpart);
	                }
	            });
	            masterChildren.forEach(function (masterChild) {
	                var originalCounterpart = _.find(originalChildren, function (originalChild) { return masterChild.fullLocalId() == originalChild.fullLocalId(); });
	                if (!originalCounterpart) {
	                    //we dont have a counterpart, so simply adding to result
	                    result.push(masterChild);
	                }
	            });
	            return result;
	        }
	        return null;
	    };
	    ASTNodeImpl.prototype.mergeChild = function (result, originalChild, masterChild) {
	        if (originalChild.getKind() != masterChild.getKind()) {
	            //should not happen theoretically
	            result.push(originalChild);
	            result.push(masterChild);
	            return;
	        }
	        if (originalChild.getKind() == hl.NodeKind.NODE) {
	            result.push(originalChild);
	            return;
	        }
	        else if (originalChild.getKind() == hl.NodeKind.ATTRIBUTE) {
	            //if ((<ASTPropImpl>originalChild).name() == "displayName" ||
	            //    (<ASTPropImpl>originalChild).name() == "title") {
	            //    console.log("OriginalChildForDisplayName: " + (<ASTPropImpl>originalChild).value())
	            //    console.log("MasterChildForDisplayName: " + (<ASTPropImpl>masterChild).value())
	            //
	            //}
	            result.push(originalChild);
	            return;
	        }
	        else if (originalChild.getKind() == hl.NodeKind.BASIC) {
	            //we do not know what to do with basic nodes, so adding both.
	            result.push(originalChild);
	            result.push(masterChild);
	            return;
	        }
	    };
	    ASTNodeImpl.prototype.directChildren = function () {
	        if (this._children) {
	            return this._children;
	        }
	        if (this._node) {
	            this._children = nodeBuilder.process(this, this._node.children());
	            return this._children;
	        }
	        return [];
	    };
	    ASTNodeImpl.prototype.resetChildren = function () {
	        this._children = null;
	    };
	    ASTNodeImpl.prototype.isEmptyRamlFile = function () {
	        var llroot = this.lowLevel().root();
	        return llroot.isScalar();
	    };
	    ASTNodeImpl.prototype.initRamlFile = function () {
	        mutators.initEmptyRAMLFile(this);
	    };
	    ASTNodeImpl.prototype.createAttr = function (n, v) {
	        mutators.createAttr(this, n, v);
	    };
	    ASTNodeImpl.prototype.isAttr = function () {
	        return false;
	    };
	    ASTNodeImpl.prototype.isUnknown = function () {
	        return false;
	    };
	    ASTNodeImpl.prototype.value = function () {
	        return this._node.value();
	    };
	    ASTNodeImpl.prototype.valuesOf = function (propName) {
	        var pr = this._def.property(propName);
	        if (pr != null) {
	            return this.elements().filter(function (x) { return x.property() == pr; });
	        }
	        return [];
	    };
	    ASTNodeImpl.prototype.attr = function (n) {
	        return _.find(this.attrs(), function (y) { return y.name() == n; });
	    };
	    ASTNodeImpl.prototype.attrOrCreate = function (name) {
	        var a = this.attr(name);
	        if (!a)
	            this.createAttr(name, '');
	        return this.attr(name);
	    };
	    ASTNodeImpl.prototype.attrValue = function (n) {
	        var a = this.attr(n);
	        return a ? a.value() : null;
	    };
	    ASTNodeImpl.prototype.attributes = function (n) {
	        return _.filter(this.attrs(), function (y) { return y.name() == n; });
	    };
	    ASTNodeImpl.prototype.attrs = function () {
	        var rs = this.children().filter(function (x) { return x.isAttr(); });
	        if (this._patchedName) {
	            var kp = _.find(this.definition().allProperties(), function (x) { return x.getAdapter(services.RAMLPropertyService).isKey(); });
	            if (kp) {
	                var mm = new ASTPropImpl(this.lowLevel(), this, kp.range(), kp, true);
	                mm._value = this._patchedName;
	                return [mm].concat(rs);
	            }
	        }
	        return rs;
	    };
	    ASTNodeImpl.prototype.elements = function () {
	        return this.children()
	            .filter(function (x) { return !x.isAttr() && !x.isUnknown(); });
	    };
	    ASTNodeImpl.prototype.element = function (n) {
	        var r = this.elementsOfKind(n);
	        if (r.length > 0) {
	            return r[0];
	        }
	        return null;
	    };
	    ASTNodeImpl.prototype.elementsOfKind = function (n) {
	        var r = this.elements().filter(function (x) { return x.property().nameId() == n; });
	        return r;
	    };
	    ASTNodeImpl.prototype.definition = function () {
	        return this._def;
	    };
	    ASTNodeImpl.prototype.property = function () {
	        return this._prop;
	    };
	    ASTNodeImpl.prototype.isExpanded = function () {
	        return this._expanded;
	    };
	    ASTNodeImpl.prototype.copy = function () {
	        return new ASTNodeImpl(this.lowLevel().copy(), this.parent(), this.definition(), this.property());
	    };
	    ASTNodeImpl.prototype.clearChildrenCache = function () {
	        this._children = null;
	    };
	    ASTNodeImpl.prototype.optionalProperties = function () {
	        var def = this.definition();
	        if (def == null) {
	            return [];
	        }
	        var result = [];
	        var map = {};
	        var children = this.lowLevel().children();
	        children.forEach(function (x) {
	            if (x.optional()) {
	                map[x.key()] = true;
	            }
	        });
	        var props = def.allProperties();
	        props.forEach(function (x) {
	            var prop = x;
	            if (map[prop.nameId()]) {
	                result.push(prop.nameId());
	            }
	        });
	        return result;
	    };
	    return ASTNodeImpl;
	}(BasicASTNode));
	exports.ASTNodeImpl = ASTNodeImpl;
	exports.universeProvider = __webpack_require__(34);
	var getDefinitionSystemType = function (contents, ast) {
	    var spec = "";
	    var ptype = "Api";
	    var num = 0;
	    var pt = 0;
	    for (var n = 0; n < contents.length; n++) {
	        var c = contents.charAt(n);
	        if (c == '\r' || c == '\n') {
	            if (spec) {
	                ptype = contents.substring(pt, n).trim();
	            }
	            else {
	                spec = contents.substring(0, n).trim();
	            }
	            break;
	        }
	        if (c == ' ') {
	            num++;
	            if (!spec && num == 2) {
	                spec = contents.substring(0, n);
	                pt = n;
	            }
	        }
	    }
	    var localUniverse = spec == "#%RAML 1.0" ? new def.Universe(null, "RAML10", exports.universeProvider("RAML10"), "RAML10") : new def.Universe(null, "RAML08", exports.universeProvider("RAML08"));
	    if (ptype == 'API') {
	        ptype = "Api";
	    }
	    if (ptype == 'NamedExample') {
	        ptype = "ExampleSpec";
	    }
	    if (ptype == 'DataType') {
	        ptype = "TypeDeclaration";
	    }
	    if (ptype == 'SecurityScheme') {
	        ptype = "AbstractSecurityScheme";
	    }
	    localUniverse.setTopLevel(ptype);
	    localUniverse.setTypedVersion(spec);
	    // localUniverse.setDescription(spec);
	    return { ptype: ptype, localUniverse: localUniverse };
	};
	function getFragmentDefenitionName(highLevelNode) {
	    var contents = highLevelNode.lowLevel() && highLevelNode.lowLevel().unit() && highLevelNode.lowLevel().unit().contents();
	    if (contents == null) {
	        return null;
	    }
	    return getDefinitionSystemType(contents, highLevelNode.lowLevel()).ptype;
	}
	exports.getFragmentDefenitionName = getFragmentDefenitionName;
	function fromUnit(l) {
	    if (l == null)
	        return null;
	    var contents = l.contents();
	    var ast = l.ast();
	    var __ret = getDefinitionSystemType(contents, ast);
	    var ptype = __ret.ptype;
	    var localUniverse = __ret.localUniverse;
	    var apiType = localUniverse.type(ptype);
	    if (!apiType)
	        apiType = localUniverse.type("Api");
	    var api = new ASTNodeImpl(ast, null, apiType, null);
	    api.setUniverse(localUniverse);
	    //forcing discrimination
	    api.children();
	    return api;
	}
	exports.fromUnit = fromUnit;
	//# sourceMappingURL=highLevelImpl.js.map

/***/ },
/* 17 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var jsyaml = __webpack_require__(23);
	var hlimpl = __webpack_require__(16);
	var yaml = __webpack_require__(72);
	//*** THis is not needed any more *///
	function createStub0(parent, property, key) {
	    var p = parent.definition().property(property);
	    if (!p)
	        return null;
	    var nc = p.range();
	    var node = createStubNode(nc, p, key);
	    return node;
	}
	exports.createStub0 = createStub0;
	function genStructuredValue(type, name, mappings, parent) {
	    var map = yaml.newMap(mappings.map(function (mapping) { return yaml.newMapping(yaml.newScalar(mapping.key), yaml.newScalar(mapping.value)); }));
	    var node = new jsyaml.ASTNode(map, (parent ? parent.lowLevel().unit() : null), parent ? parent.lowLevel() : null, null, null);
	    return new hlimpl.StructuredValue(node, parent, parent ? parent.definition().property(type) : null, name);
	}
	exports.genStructuredValue = genStructuredValue;
	function createStub(parent, property, key) {
	    var p = parent.definition().property(property);
	    if (!p)
	        return null;
	    var nc = p.range();
	    var su = parent.lowLevel().unit().stub();
	    var node = createStubNode(nc, p, key, su);
	    node.isInEdit = true;
	    node.lowLevel()._unit = su;
	    node._parent = parent.copy();
	    node._parent.lowLevel()._unit = su;
	    return node;
	}
	exports.createStub = createStub;
	function createResourceStub(parent, key) {
	    return createStub(parent, "resources", key);
	}
	exports.createResourceStub = createResourceStub;
	function createMethodStub(parent, key) {
	    return createStub(parent, 'methods', key);
	}
	exports.createMethodStub = createMethodStub;
	function createResponseStub(parent, key) {
	    return createStub(parent, 'responses', key);
	}
	exports.createResponseStub = createResponseStub;
	function createBodyStub(parent, key) {
	    return createStub(parent, 'body', key);
	}
	exports.createBodyStub = createBodyStub;
	function createUriParameterStub(parent, key) {
	    return createStub(parent, 'uriParameters', key);
	}
	exports.createUriParameterStub = createUriParameterStub;
	function createQueryParameterStub(parent, key) {
	    return createStub(parent, 'queryParameters', key);
	}
	exports.createQueryParameterStub = createQueryParameterStub;
	function createAttr(_property, val) {
	    var lowLevel = jsyaml.createMapping(_property.nameId(), val);
	    var nm = new hlimpl.ASTPropImpl(lowLevel, null, _property.range(), _property);
	    return nm;
	}
	exports.createAttr = createAttr;
	function createStubNode(t, p, key, unit) {
	    if (key === void 0) { key = null; }
	    var lowLevel = jsyaml.createNode(key ? key : "key");
	    var nm = new hlimpl.ASTNodeImpl(lowLevel, null, t, p);
	    if (!lowLevel.unit()) {
	        lowLevel._unit = unit;
	    }
	    nm.children();
	    return nm;
	}
	exports.createStubNode = createStubNode;
	//# sourceMappingURL=stubs.js.map

/***/ },
/* 18 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var _ = __webpack_require__(61);
	var textutil = __webpack_require__(40);
	var wrapper = __webpack_require__(20);
	var SchemaToModelGenerator = (function () {
	    function SchemaToModelGenerator() {
	    }
	    SchemaToModelGenerator.prototype.generateText = function (schema) {
	        var _this = this;
	        var obj = JSON.parse(schema);
	        var items = obj['items'];
	        if (!items)
	            return '';
	        var text = '';
	        var itemsArray = (items instanceof Array) ? items : [items];
	        text += 'types:\n';
	        itemsArray.forEach(function (e) {
	            //console.log('Item: ' + e.title + ' ' + e.type);
	            text += '  - ' + e.title + ':\n';
	            text += _this.generateObj(e, 3);
	        });
	        return text;
	    };
	    SchemaToModelGenerator.prototype.generateObj = function (e, lev) {
	        var text = '';
	        text += textutil.indent(lev, 'type: ' + e.type) + '\n';
	        if (!e.properties)
	            return;
	        text += textutil.indent(lev, 'properties:\n');
	        //console.log('props: ' + e.properties);
	        for (var name in e.properties) {
	            //console.log('  property: ' + JSON.stringify(p));
	            var p = e.properties[name];
	            text += textutil.indent(lev + 1, name + ':\n');
	            //console.log('  property ' + name + ': ' + p.type);
	            if (p.type == 'object') {
	                text += this.generateObj(p, lev + 2);
	            }
	            else {
	                text += textutil.indent(lev + 2, 'type: ' + p.type) + '\n';
	            }
	        }
	        return text;
	    };
	    SchemaToModelGenerator.prototype.generateTo = function (api, schema, title) {
	        var _this = this;
	        var obj = JSON.parse(schema);
	        var items = obj['items'];
	        if (!items) {
	            if (obj.title)
	                title = obj.title;
	            //console.log('title: ' + title);
	            var type = new wrapper.ObjectTypeDeclarationImpl(title);
	            this.generateObjTo(type, obj);
	            new wrapper.BasicSecuritySchemeImpl(api).addToProp(type, 'types');
	            return [title];
	        }
	        else {
	            var itemsArray = (items instanceof Array) ? items : [items];
	            var types = [];
	            itemsArray.forEach(function (e) {
	                var type = new wrapper.ObjectTypeDeclarationImpl(e.title);
	                _this.generateObjTo(type, e);
	                new wrapper.BasicSecuritySchemeImpl(api).addToProp(type, 'types');
	                types.push(e.title);
	            });
	            return types;
	        }
	    };
	    SchemaToModelGenerator.prototype.generateObjTo = function (type, e) {
	        type.setType(e.type);
	        //console.log('type: ' + type.definition().name());
	        //util.showTypeProperties(type.definition());
	        //type.attrOrCreate('type').setValue(e.type);
	        if (!e.properties)
	            return;
	        //text += textutil.indent(lev, 'properties:\n');
	        //console.log('props: ' + e.properties);
	        for (var name in e.properties) {
	            var p = e.properties[name];
	            var field = this.makeTypeField(name, p);
	            if (p.type == 'array') {
	            }
	            //console.log('  property: ' + JSON.stringify(p));
	            //text += textutil.indent(lev+1, name + ':\n');
	            //console.log('  property ' + name + ': ' + p.type);
	            type.addToProp(field, 'properties');
	        }
	    };
	    SchemaToModelGenerator.prototype.makeTypeField = function (name, p) {
	        var field = this.makeType(name, p.type);
	        if (p.type)
	            field.setType(p.type);
	        if (p.type == 'number') {
	            var n = field;
	            if (p.minimum != undefined)
	                (n).setMinimum(p.minimum);
	            if (p.maximum != undefined)
	                (n).setMaximum(p.maximum);
	        }
	        if (p.type == 'array') {
	            var itype = p.items.type;
	            field.setType(itype + '[]');
	            var a = field;
	            if (p.minItems != undefined)
	                a.setMinItems(p.minItems);
	            if (p.maxItems != undefined)
	                a.setMaxItems(p.maxItems);
	            if (p.uniqueItems != undefined)
	                a.setUniqueItems(p.uniqueItems);
	        }
	        if (p.type == 'object') {
	            this.generateObjTo(field, p);
	        }
	        return field;
	    };
	    SchemaToModelGenerator.prototype.makeType = function (name, type) {
	        if (type == 'number')
	            return new wrapper.NumberTypeDeclarationImpl(name);
	        if (type == 'string')
	            return new wrapper.StringTypeDeclarationImpl(name);
	        if (type == 'array')
	            return new wrapper.ArrayTypeDeclarationImpl(name);
	        return new wrapper.ObjectTypeDeclarationImpl(name);
	    };
	    SchemaToModelGenerator.prototype.generateItemsTo = function (a, obj) {
	        var items = obj['items'];
	        if (!items)
	            return;
	        var itemsArray = (items instanceof Array) ? items : [items];
	        itemsArray.forEach(function (item) {
	            //TODO add items here
	        });
	    };
	    return SchemaToModelGenerator;
	}());
	exports.SchemaToModelGenerator = SchemaToModelGenerator;
	var ModelToSchemaGenerator = (function () {
	    function ModelToSchemaGenerator() {
	    }
	    ModelToSchemaGenerator.prototype.generateSchema = function (node) {
	        var obj = this.generateType(node);
	        obj["$schema"] = "http://json-schema.org/draft-04/schema#";
	        return obj;
	    };
	    ModelToSchemaGenerator.prototype.isSimpleType = function (name) {
	        return name == 'string' || name == 'number' || name == 'boolean' || name == 'object';
	    };
	    ModelToSchemaGenerator.prototype.generateType = function (type) {
	        var typelist = this.allTypes(type);
	        var obj = {};
	        obj.title = type.attrValue('name');
	        var properties = {};
	        if (type.attrValue('type')) {
	            // support multiple inheritance
	            var types = type.attributes('type');
	            //var properties = {};
	            var inherited = false;
	            var simple = false;
	            for (var i in types) {
	                var typevalue = types[i].value();
	                var ll = types[i].lowLevel();
	                obj.type = '';
	                if (ll.isValueInclude()) {
	                    var schema = JSON.parse(typevalue);
	                    obj.type = 'object';
	                    obj.properties = schema.properties;
	                }
	                else {
	                    //console.log('type value: ' + typevalue);
	                    if (this.isSimpleType(typevalue)) {
	                        obj.type = typevalue;
	                        simple = true;
	                        if (inherited)
	                            throw new Error("couldn't mix user defined and basic types in inheritance");
	                    }
	                    else {
	                        var t = this.resolveType(type, typevalue);
	                        if (t) {
	                            var ppp = this.generateTypeExp(typevalue, typelist);
	                            //console.log('res: ' + typevalue + '\n' + JSON.stringify(ppp));
	                            //_.extend(properties, ppp.properties);
	                            if (obj.properties) {
	                                _.extend(obj.properties, ppp.properties);
	                            }
	                            else {
	                                obj.properties = ppp.properties;
	                            }
	                            obj.type = 'object';
	                            inherited = true;
	                            if (simple)
	                                throw new Error("couldn't mix user defined and basic types in inheritance");
	                        }
	                        else {
	                            var ppp = this.generateTypeExp(typevalue, typelist);
	                            //console.log('xtype: ' + typevalue + '\n' + JSON.stringify(ppp));
	                            obj.type = 'object';
	                            inherited = true;
	                            if (ppp.anyOf) {
	                                obj.anyOf = ppp.anyOf;
	                            }
	                        }
	                    }
	                }
	            }
	        }
	        else {
	            obj.type = 'object';
	        }
	        //var ps = this.generateProperties(type);
	        //this.generatePropertiesTo(properties, type);
	        var ownProps = this.generateProperties(type);
	        //_.extend(properties, ownProps);
	        //if(Object.getOwnPropertyNames(properties).length > 0) {
	        if (obj.properties) {
	            //console.log('own.p : ' + obj.properties);
	            //console.log('own: ' + JSON.stringify(ownProps));
	            //console.log('obj.p1: ' + obj.properties);
	            _.extend(obj.properties, ownProps);
	        }
	        else {
	            obj.properties = ownProps;
	        }
	        return obj;
	    };
	    ModelToSchemaGenerator.prototype.makeUnion = function (typelist, types) {
	        var _this = this;
	        var anyof = [];
	        typelist.forEach(function (t) {
	            t = t.trim();
	            if (types[t]) {
	                anyof.push({
	                    type: 'object',
	                    properties: _this.generateType(types[t]).properties
	                });
	            }
	            else {
	                anyof.push({ type: t });
	            }
	        });
	        return anyof;
	    };
	    ModelToSchemaGenerator.prototype.generateTypeExp = function (type, types) {
	        var obj = {};
	        //console.log('type: ' + type);
	        if (textutil.endsWith(type, '[]')) {
	            obj.type = 'array';
	            obj.items = { type: type.substring(0, type.length - 2) };
	        }
	        else if (type.indexOf('|') > 0) {
	            var typelist = type.split('|');
	            obj.anyOf = this.makeUnion(typelist, types);
	        }
	        else {
	            if (types[type]) {
	                var schema = this.generateType(types[type]);
	                obj.type = 'object';
	                obj.properties = schema.properties;
	            }
	            else {
	                obj.type = type;
	            }
	        }
	        return obj;
	    };
	    ModelToSchemaGenerator.prototype.allTypes = function (node) {
	        var api = node.root();
	        var modelTypes = api.elementsOfKind('types');
	        var types = {};
	        modelTypes.forEach(function (type) {
	            //console.log('  types: ' + type.name());
	            types[type.name()] = type;
	        });
	        return types;
	    };
	    ModelToSchemaGenerator.prototype.resolveType = function (node, name) {
	        var types = this.allTypes(node);
	        return types[name];
	    };
	    ModelToSchemaGenerator.prototype.generateProperty = function (node, optional) {
	        var _this = this;
	        var types = this.allTypes(node);
	        var obj = {};
	        var props = node.definition().allProperties();
	        props.forEach(function (p) {
	            if (p.nameId() == 'name')
	                return;
	            var value = node.attrValue(p.nameId());
	            if (!(value != null && value != undefined && value != 'undefined'))
	                return;
	            if (p.nameId() == 'type') {
	                var t = _this.generateTypeExp(value, types);
	                _.extend(obj, t);
	            }
	            else if (p.nameId() == 'enum') {
	                var values = node.attributes('enum');
	                var arr = values.map(function (a) { return a.value(); });
	                obj.enum = arr;
	            }
	            else {
	                obj[p.nameId()] = value;
	            }
	        });
	        if (optional)
	            obj.required = false;
	        var elements = node.elements();
	        var properties = this.generateProperties(node);
	        if (Object.getOwnPropertyNames(properties).length > 0) {
	            obj.properties = properties;
	        }
	        return obj;
	    };
	    ModelToSchemaGenerator.prototype.generateProperties = function (node) {
	        var _this = this;
	        var props = {};
	        var elements = node.elements();
	        var empty = true;
	        elements.forEach(function (p) {
	            var name = p.attrValue('name').trim();
	            var optional = p.optional();
	            props[name] = _this.generateProperty(p, optional);
	            empty = false;
	        });
	        return props;
	    };
	    return ModelToSchemaGenerator;
	}());
	exports.ModelToSchemaGenerator = ModelToSchemaGenerator;
	//# sourceMappingURL=schemaModelGen.js.map

/***/ },
/* 19 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var parser10impl = __webpack_require__(20);
	var defSys = __webpack_require__(35);
	var jsyaml = __webpack_require__(23);
	var ll = __webpack_require__(11);
	function createTypeDeclaration(typeName) {
	    return new parser10impl.TypeDeclarationImpl(typeName);
	}
	exports.createTypeDeclaration = createTypeDeclaration;
	function createObjectTypeDeclaration(typeName) {
	    return new parser10impl.ObjectTypeDeclarationImpl(typeName);
	}
	exports.createObjectTypeDeclaration = createObjectTypeDeclaration;
	//TODO it would be probably better to generate modification interface too
	function setTypeDeclarationSchema(type, schema) {
	    type.setSchema(schema);
	}
	exports.setTypeDeclarationSchema = setTypeDeclarationSchema;
	function setTypeDeclarationExample(type, example) {
	    var exampleSpecType = defSys.getUniverse("RAML10").type(defSys.universesInfo.Universe10.ExampleSpec.name);
	    var examplePropName = defSys.universesInfo.Universe10.TypeDeclaration.properties.example.name;
	    var hlParent = type.highLevel();
	    var llParent = hlParent.lowLevel();
	    var exampleNodes = hlParent.children().filter(function (x) { return x.lowLevel().key() == examplePropName; });
	    var llNode = jsyaml.createNode(examplePropName);
	    ll.setAttr(llNode, example);
	    if (exampleNodes.length > 0) {
	        ll.removeNode(llParent, exampleNodes[0].lowLevel());
	        exampleNodes[0]._node = llNode;
	    }
	    ll.insertNode(llParent, llNode);
	}
	exports.setTypeDeclarationExample = setTypeDeclarationExample;
	function addChild(parent, child) {
	    parent.add(child);
	}
	exports.addChild = addChild;
	//# sourceMappingURL=parserMod.js.map

/***/ },
/* 20 */
/***/ function(module, exports, __webpack_require__) {

	/**
	 * <p>See <a href="http://raml.org">http://raml.org</a> for more information about RAML.</p>
	 *
	 * <p>This parser is at a beta state of development, as part of the API Workbench development cycle (<a href="http://apiworkbench.com">http://apiworkbench.com</a>).</p>
	 *
	 * <p><a href="https://github.com/raml-org/raml-js-parser-2/blob/master/documentation/GettingStarted.md">Getting Started Guide</a> describes the first steps with the parser.</p>
	 *
	 * <h2>Installation</h2>
	 *
	 * <pre><code>git clone https://github.com/raml-org/raml-js-parser-2
	 *
	 * cd raml-js-parser-2
	 *
	 * npm install
	 *
	 * node test/test.js  //here you should observe JSON representation of XKCD API in your console
	 *
	 * node test/testAsync.js  //same as above but in asynchronous mode
	 * </code></pre>
	 *
	 * <h2>Usage</h2>
	 *
	 * <ul>
	 * <li>For parser usage example refer to <code>test/test.js</code></li>
	 * <li>For asynchrounous usage example refer to <code>test/testAsync.js</code></li>
	 * </ul>
	 **/
	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var stubs = __webpack_require__(17);
	var def = __webpack_require__(35);
	var core = __webpack_require__(45);
	var apiLoader = __webpack_require__(9);
	var helper = __webpack_require__(32);
	var AnnotableImpl = (function (_super) {
	    __extends(AnnotableImpl, _super);
	    function AnnotableImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
	     **/
	    AnnotableImpl.prototype.annotations = function () {
	        return _super.prototype.attributes.call(this, 'annotations', function (attr) { return new AnnotationRefImpl(attr); });
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    AnnotableImpl.prototype.wrapperClassName = function () { return "AnnotableImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    AnnotableImpl.prototype.kind = function () { return "Annotable"; };
	    /**
	     * @return RAML version of the node
	     **/
	    AnnotableImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    AnnotableImpl.prototype.scalarsAnnotations = function () { return new AnnotableScalarsAnnotationsImpl(this.highLevel()); };
	    return AnnotableImpl;
	}(core.BasicNodeImpl));
	exports.AnnotableImpl = AnnotableImpl;
	var ValueTypeImpl = (function (_super) {
	    __extends(ValueTypeImpl, _super);
	    function ValueTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ValueTypeImpl.prototype.wrapperClassName = function () { return "ValueTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ValueTypeImpl.prototype.kind = function () { return "ValueType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ValueTypeImpl.prototype.RAMLVersion = function () { return "undefined"; };
	    /**
	     * @return JS representation of the node value
	     **/
	    ValueTypeImpl.prototype.value = function () { return this.attr.value(); };
	    return ValueTypeImpl;
	}(core.AttributeNodeImpl));
	exports.ValueTypeImpl = ValueTypeImpl;
	var StringTypeImpl = (function (_super) {
	    __extends(StringTypeImpl, _super);
	    function StringTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    StringTypeImpl.prototype.wrapperClassName = function () { return "StringTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    StringTypeImpl.prototype.kind = function () { return "StringType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    StringTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * @return String representation of the node value
	     **/
	    StringTypeImpl.prototype.value = function () { return this.attr.value(); };
	    return StringTypeImpl;
	}(ValueTypeImpl));
	exports.StringTypeImpl = StringTypeImpl;
	/**
	 * This type currently serves both for absolute and relative urls
	 **/
	var UriTemplateImpl = (function (_super) {
	    __extends(UriTemplateImpl, _super);
	    function UriTemplateImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    UriTemplateImpl.prototype.wrapperClassName = function () { return "UriTemplateImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    UriTemplateImpl.prototype.kind = function () { return "UriTemplate"; };
	    /**
	     * @return RAML version of the node
	     **/
	    UriTemplateImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return UriTemplateImpl;
	}(StringTypeImpl));
	exports.UriTemplateImpl = UriTemplateImpl;
	/**
	 * This  type describes relative uri templates
	 **/
	var RelativeUriStringImpl = (function (_super) {
	    __extends(RelativeUriStringImpl, _super);
	    function RelativeUriStringImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    RelativeUriStringImpl.prototype.wrapperClassName = function () { return "RelativeUriStringImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    RelativeUriStringImpl.prototype.kind = function () { return "RelativeUriString"; };
	    /**
	     * @return RAML version of the node
	     **/
	    RelativeUriStringImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return RelativeUriStringImpl;
	}(UriTemplateImpl));
	exports.RelativeUriStringImpl = RelativeUriStringImpl;
	/**
	 * This  type describes absolute uri templates
	 **/
	var FullUriTemplateStringImpl = (function (_super) {
	    __extends(FullUriTemplateStringImpl, _super);
	    function FullUriTemplateStringImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    FullUriTemplateStringImpl.prototype.wrapperClassName = function () { return "FullUriTemplateStringImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    FullUriTemplateStringImpl.prototype.kind = function () { return "FullUriTemplateString"; };
	    /**
	     * @return RAML version of the node
	     **/
	    FullUriTemplateStringImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return FullUriTemplateStringImpl;
	}(UriTemplateImpl));
	exports.FullUriTemplateStringImpl = FullUriTemplateStringImpl;
	var StatusCodeStringImpl = (function (_super) {
	    __extends(StatusCodeStringImpl, _super);
	    function StatusCodeStringImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    StatusCodeStringImpl.prototype.wrapperClassName = function () { return "StatusCodeStringImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    StatusCodeStringImpl.prototype.kind = function () { return "StatusCodeString"; };
	    /**
	     * @return RAML version of the node
	     **/
	    StatusCodeStringImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return StatusCodeStringImpl;
	}(StringTypeImpl));
	exports.StatusCodeStringImpl = StatusCodeStringImpl;
	/**
	 * This  type describes fixed uris
	 **/
	var FixedUriStringImpl = (function (_super) {
	    __extends(FixedUriStringImpl, _super);
	    function FixedUriStringImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    FixedUriStringImpl.prototype.wrapperClassName = function () { return "FixedUriStringImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    FixedUriStringImpl.prototype.kind = function () { return "FixedUriString"; };
	    /**
	     * @return RAML version of the node
	     **/
	    FixedUriStringImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return FixedUriStringImpl;
	}(StringTypeImpl));
	exports.FixedUriStringImpl = FixedUriStringImpl;
	var ContentTypeImpl = (function (_super) {
	    __extends(ContentTypeImpl, _super);
	    function ContentTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ContentTypeImpl.prototype.wrapperClassName = function () { return "ContentTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ContentTypeImpl.prototype.kind = function () { return "ContentType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ContentTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return ContentTypeImpl;
	}(StringTypeImpl));
	exports.ContentTypeImpl = ContentTypeImpl;
	/**
	 * [GitHub Flavored Markdown](https://help.github.com/articles/github-flavored-markdown/)
	 **/
	var MarkdownStringImpl = (function (_super) {
	    __extends(MarkdownStringImpl, _super);
	    function MarkdownStringImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    MarkdownStringImpl.prototype.wrapperClassName = function () { return "MarkdownStringImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    MarkdownStringImpl.prototype.kind = function () { return "MarkdownString"; };
	    /**
	     * @return RAML version of the node
	     **/
	    MarkdownStringImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return MarkdownStringImpl;
	}(StringTypeImpl));
	exports.MarkdownStringImpl = MarkdownStringImpl;
	/**
	 * Schema at this moment only two subtypes are supported (json schema and xsd)
	 **/
	var SchemaStringImpl = (function (_super) {
	    __extends(SchemaStringImpl, _super);
	    function SchemaStringImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    SchemaStringImpl.prototype.wrapperClassName = function () { return "SchemaStringImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    SchemaStringImpl.prototype.kind = function () { return "SchemaString"; };
	    /**
	     * @return RAML version of the node
	     **/
	    SchemaStringImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return SchemaStringImpl;
	}(StringTypeImpl));
	exports.SchemaStringImpl = SchemaStringImpl;
	/**
	 * This sub type of the string represents mime types
	 **/
	var MimeTypeImpl = (function (_super) {
	    __extends(MimeTypeImpl, _super);
	    function MimeTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    MimeTypeImpl.prototype.wrapperClassName = function () { return "MimeTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    MimeTypeImpl.prototype.kind = function () { return "MimeType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    MimeTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return MimeTypeImpl;
	}(StringTypeImpl));
	exports.MimeTypeImpl = MimeTypeImpl;
	var AnyTypeImpl = (function (_super) {
	    __extends(AnyTypeImpl, _super);
	    function AnyTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    AnyTypeImpl.prototype.wrapperClassName = function () { return "AnyTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    AnyTypeImpl.prototype.kind = function () { return "AnyType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    AnyTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return AnyTypeImpl;
	}(ValueTypeImpl));
	exports.AnyTypeImpl = AnyTypeImpl;
	var NumberTypeImpl = (function (_super) {
	    __extends(NumberTypeImpl, _super);
	    function NumberTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    NumberTypeImpl.prototype.wrapperClassName = function () { return "NumberTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    NumberTypeImpl.prototype.kind = function () { return "NumberType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    NumberTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * @return Number representation of the node value
	     **/
	    NumberTypeImpl.prototype.value = function () { return this.attr.value(); };
	    return NumberTypeImpl;
	}(ValueTypeImpl));
	exports.NumberTypeImpl = NumberTypeImpl;
	var IntegerTypeImpl = (function (_super) {
	    __extends(IntegerTypeImpl, _super);
	    function IntegerTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    IntegerTypeImpl.prototype.wrapperClassName = function () { return "IntegerTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    IntegerTypeImpl.prototype.kind = function () { return "IntegerType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    IntegerTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return IntegerTypeImpl;
	}(ValueTypeImpl));
	exports.IntegerTypeImpl = IntegerTypeImpl;
	var NullTypeImpl = (function (_super) {
	    __extends(NullTypeImpl, _super);
	    function NullTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    NullTypeImpl.prototype.wrapperClassName = function () { return "NullTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    NullTypeImpl.prototype.kind = function () { return "NullType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    NullTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return NullTypeImpl;
	}(ValueTypeImpl));
	exports.NullTypeImpl = NullTypeImpl;
	var TimeOnlyTypeImpl = (function (_super) {
	    __extends(TimeOnlyTypeImpl, _super);
	    function TimeOnlyTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    TimeOnlyTypeImpl.prototype.wrapperClassName = function () { return "TimeOnlyTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    TimeOnlyTypeImpl.prototype.kind = function () { return "TimeOnlyType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    TimeOnlyTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return TimeOnlyTypeImpl;
	}(ValueTypeImpl));
	exports.TimeOnlyTypeImpl = TimeOnlyTypeImpl;
	var DateOnlyTypeImpl = (function (_super) {
	    __extends(DateOnlyTypeImpl, _super);
	    function DateOnlyTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DateOnlyTypeImpl.prototype.wrapperClassName = function () { return "DateOnlyTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DateOnlyTypeImpl.prototype.kind = function () { return "DateOnlyType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DateOnlyTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return DateOnlyTypeImpl;
	}(ValueTypeImpl));
	exports.DateOnlyTypeImpl = DateOnlyTypeImpl;
	var DateTimeOnlyTypeImpl = (function (_super) {
	    __extends(DateTimeOnlyTypeImpl, _super);
	    function DateTimeOnlyTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DateTimeOnlyTypeImpl.prototype.wrapperClassName = function () { return "DateTimeOnlyTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DateTimeOnlyTypeImpl.prototype.kind = function () { return "DateTimeOnlyType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DateTimeOnlyTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return DateTimeOnlyTypeImpl;
	}(ValueTypeImpl));
	exports.DateTimeOnlyTypeImpl = DateTimeOnlyTypeImpl;
	var DateTimeTypeImpl = (function (_super) {
	    __extends(DateTimeTypeImpl, _super);
	    function DateTimeTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DateTimeTypeImpl.prototype.wrapperClassName = function () { return "DateTimeTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DateTimeTypeImpl.prototype.kind = function () { return "DateTimeType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DateTimeTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return DateTimeTypeImpl;
	}(ValueTypeImpl));
	exports.DateTimeTypeImpl = DateTimeTypeImpl;
	var FileTypeImpl = (function (_super) {
	    __extends(FileTypeImpl, _super);
	    function FileTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    FileTypeImpl.prototype.wrapperClassName = function () { return "FileTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    FileTypeImpl.prototype.kind = function () { return "FileType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    FileTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return FileTypeImpl;
	}(ValueTypeImpl));
	exports.FileTypeImpl = FileTypeImpl;
	var BooleanTypeImpl = (function (_super) {
	    __extends(BooleanTypeImpl, _super);
	    function BooleanTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    BooleanTypeImpl.prototype.wrapperClassName = function () { return "BooleanTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    BooleanTypeImpl.prototype.kind = function () { return "BooleanType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    BooleanTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * @return Boolean representation of the node value
	     **/
	    BooleanTypeImpl.prototype.value = function () { return this.attr.value(); };
	    return BooleanTypeImpl;
	}(ValueTypeImpl));
	exports.BooleanTypeImpl = BooleanTypeImpl;
	/**
	 * Elements to which this Annotation can be applied (enum)
	 **/
	var AnnotationTargetImpl = (function (_super) {
	    __extends(AnnotationTargetImpl, _super);
	    function AnnotationTargetImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    AnnotationTargetImpl.prototype.wrapperClassName = function () { return "AnnotationTargetImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    AnnotationTargetImpl.prototype.kind = function () { return "AnnotationTarget"; };
	    /**
	     * @return RAML version of the node
	     **/
	    AnnotationTargetImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return AnnotationTargetImpl;
	}(ValueTypeImpl));
	exports.AnnotationTargetImpl = AnnotationTargetImpl;
	var ReferenceImpl = (function (_super) {
	    __extends(ReferenceImpl, _super);
	    function ReferenceImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ReferenceImpl.prototype.wrapperClassName = function () { return "ReferenceImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ReferenceImpl.prototype.kind = function () { return "Reference"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ReferenceImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * @return StructuredValue object representing the node value
	     **/
	    ReferenceImpl.prototype.value = function () { return core.toStructuredValue(this.attr); };
	    ReferenceImpl.prototype.structuredValue = function () {
	        return helper.structuredValue(this);
	    };
	    ReferenceImpl.prototype.name = function () {
	        return helper.referenceName(this);
	    };
	    return ReferenceImpl;
	}(core.AttributeNodeImpl));
	exports.ReferenceImpl = ReferenceImpl;
	var TraitRefImpl = (function (_super) {
	    __extends(TraitRefImpl, _super);
	    function TraitRefImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    TraitRefImpl.prototype.wrapperClassName = function () { return "TraitRefImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    TraitRefImpl.prototype.kind = function () { return "TraitRef"; };
	    /**
	     * @return RAML version of the node
	     **/
	    TraitRefImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    TraitRefImpl.prototype.trait = function () {
	        return helper.referencedTrait(this);
	    };
	    return TraitRefImpl;
	}(ReferenceImpl));
	exports.TraitRefImpl = TraitRefImpl;
	var OperationImpl = (function (_super) {
	    __extends(OperationImpl, _super);
	    function OperationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createOperation(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * An APIs resources MAY be filtered (to return a subset of results) or altered (such as transforming  a response body from JSON to XML format) by the use of query strings. If the resource or its method supports a query string, the query string MUST be defined by the queryParameters property
	     **/
	    OperationImpl.prototype.queryParameters = function () {
	        return _super.prototype.elements.call(this, 'queryParameters');
	    };
	    /**
	     * Headers that allowed at this position
	     **/
	    OperationImpl.prototype.headers = function () {
	        return _super.prototype.elements.call(this, 'headers');
	    };
	    /**
	     * Specifies the query string needed by this method. Mutually exclusive with queryParameters.
	     **/
	    OperationImpl.prototype.queryString = function () {
	        return _super.prototype.element.call(this, 'queryString');
	    };
	    /**
	     * Information about the expected responses to a request
	     **/
	    OperationImpl.prototype.responses = function () {
	        return _super.prototype.elements.call(this, 'responses');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    OperationImpl.prototype.wrapperClassName = function () { return "OperationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    OperationImpl.prototype.kind = function () { return "Operation"; };
	    /**
	     * @return RAML version of the node
	     **/
	    OperationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return OperationImpl;
	}(AnnotableImpl));
	exports.OperationImpl = OperationImpl;
	var TypeDeclarationImpl = (function (_super) {
	    __extends(TypeDeclarationImpl, _super);
	    function TypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * name of the parameter
	     **/
	    TypeDeclarationImpl.prototype.name = function () {
	        return _super.prototype.attribute.call(this, 'name', this.toString);
	    };
	    /**
	     * @hidden
	     * Set name value
	     **/
	    TypeDeclarationImpl.prototype.setName = function (param) {
	        this.highLevel().attrOrCreate("name").setValue("" + param);
	        return this;
	    };
	    /**
	     * The displayName attribute specifies the type display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
	     **/
	    TypeDeclarationImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    TypeDeclarationImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    /**
	     * When extending from a type you can define new facets (which can then be set to concrete values by subtypes).
	     **/
	    TypeDeclarationImpl.prototype.facets = function () {
	        return _super.prototype.elements.call(this, 'facets');
	    };
	    /**
	     * Alias for the equivalent "type" property, for compatibility with RAML 0.8. Deprecated - API definitions should use the "type" property, as the "schema" alias for that property name may be removed in a future RAML version. The "type" property allows for XML and JSON schemas.
	     * @hidden
	     **/
	    TypeDeclarationImpl.prototype.schema_original = function () {
	        return _super.prototype.attributes.call(this, 'schema', this.toString);
	    };
	    /**
	     * @hidden
	     * Set schema value
	     **/
	    TypeDeclarationImpl.prototype.setSchema = function (param) {
	        this.highLevel().attrOrCreate("schema").setValue("" + param);
	        return this;
	    };
	    /**
	     * A base type which the current type extends, or more generally a type expression.
	     * @hidden
	     **/
	    TypeDeclarationImpl.prototype.type_original = function () {
	        return _super.prototype.attributes.call(this, 'type', this.toString);
	    };
	    /**
	     * @hidden
	     * Set type value
	     **/
	    TypeDeclarationImpl.prototype.setType = function (param) {
	        this.highLevel().attrOrCreate("type").setValue("" + param);
	        return this;
	    };
	    /**
	     * Location of the parameter (can not be edited by user)
	     **/
	    TypeDeclarationImpl.prototype.location = function () {
	        return _super.prototype.attribute.call(this, 'location', function (attr) { return new ModelLocationImpl(attr); });
	    };
	    /**
	     * Kind of location
	     **/
	    TypeDeclarationImpl.prototype.locationKind = function () {
	        return _super.prototype.attribute.call(this, 'locationKind', function (attr) { return new LocationKindImpl(attr); });
	    };
	    /**
	     * Provides default value for a property
	     **/
	    TypeDeclarationImpl.prototype["default"] = function () {
	        return _super.prototype.attribute.call(this, 'default', this.toAny);
	    };
	    /**
	     * @hidden
	     * Set default value
	     **/
	    TypeDeclarationImpl.prototype.setDefault = function (param) {
	        this.highLevel().attrOrCreate("default").setValue("" + param);
	        return this;
	    };
	    /**
	     * An example of this type instance represented as string or yaml map/sequence. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the examples property is present.
	     * @hidden
	     **/
	    TypeDeclarationImpl.prototype.example_original = function () {
	        return _super.prototype.element.call(this, 'example');
	    };
	    /**
	     * An example of this type instance represented as string. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the example property is present.
	     * @hidden
	     **/
	    TypeDeclarationImpl.prototype.examples_original = function () {
	        return _super.prototype.elements.call(this, 'examples');
	    };
	    /**
	     * The repeat attribute specifies that the parameter can be repeated. If the parameter can be used multiple times, the repeat parameter value MUST be set to 'true'. Otherwise, the default value is 'false' and the parameter may not be repeated.
	     **/
	    TypeDeclarationImpl.prototype.repeat = function () {
	        return _super.prototype.attribute.call(this, 'repeat', this.toBoolean);
	    };
	    /**
	     * @hidden
	     * Set repeat value
	     **/
	    TypeDeclarationImpl.prototype.setRepeat = function (param) {
	        this.highLevel().attrOrCreate("repeat").setValue("" + param);
	        return this;
	    };
	    /**
	     * Sets if property is optional or not
	     **/
	    TypeDeclarationImpl.prototype.required = function () {
	        return _super.prototype.attribute.call(this, 'required', this.toBoolean);
	    };
	    /**
	     * @hidden
	     * Set required value
	     **/
	    TypeDeclarationImpl.prototype.setRequired = function (param) {
	        this.highLevel().attrOrCreate("required").setValue("" + param);
	        return this;
	    };
	    /**
	     * A longer, human-friendly description of the type
	     **/
	    TypeDeclarationImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    TypeDeclarationImpl.prototype.xml = function () {
	        return _super.prototype.element.call(this, 'xml');
	    };
	    /**
	     * Restrictions on where annotations of this type can be applied. If this property is specified, annotations of this type may only be applied on a property corresponding to one of the target names specified as the value of this property.
	     **/
	    TypeDeclarationImpl.prototype.allowedTargets = function () {
	        return _super.prototype.attributes.call(this, 'allowedTargets', function (attr) { return new AnnotationTargetImpl(attr); });
	    };
	    /**
	     * Whether the type represents annotation
	     **/
	    TypeDeclarationImpl.prototype.isAnnotation = function () {
	        return _super.prototype.attribute.call(this, 'isAnnotation', this.toBoolean);
	    };
	    /**
	     * @hidden
	     * Set isAnnotation value
	     **/
	    TypeDeclarationImpl.prototype.setIsAnnotation = function (param) {
	        this.highLevel().attrOrCreate("isAnnotation").setValue("" + param);
	        return this;
	    };
	    /**
	     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
	     **/
	    TypeDeclarationImpl.prototype.annotations = function () {
	        return _super.prototype.attributes.call(this, 'annotations', function (attr) { return new AnnotationRefImpl(attr); });
	    };
	    TypeDeclarationImpl.prototype.uses = function () {
	        return _super.prototype.elements.call(this, 'uses');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    TypeDeclarationImpl.prototype.wrapperClassName = function () { return "TypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    TypeDeclarationImpl.prototype.kind = function () { return "TypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    TypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Runtime representation of type represented by this AST node
	     **/
	    TypeDeclarationImpl.prototype.runtimeType = function () {
	        return helper.runtimeType(this);
	    };
	    /**
	     * validate an instance against type
	     **/
	    TypeDeclarationImpl.prototype.validateInstance = function (value) {
	        return helper.validateInstance(this, value);
	    };
	    /**
	     * validate an instance against type
	     **/
	    TypeDeclarationImpl.prototype.validateInstanceWithDetailedStatuses = function (value) {
	        return helper.validateInstanceWithDetailedStatuses(this, value);
	    };
	    /**
	     * An example of this type instance represented as string or yaml map/sequence. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the examples property is present.
	     **/
	    TypeDeclarationImpl.prototype.example = function () {
	        return helper.getTypeExample(this);
	    };
	    /**
	     * An example of this type instance represented as string. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the example property is present.
	     **/
	    TypeDeclarationImpl.prototype.examples = function () {
	        return helper.getTypeExamples(this);
	    };
	    TypeDeclarationImpl.prototype.fixedFacets = function () {
	        return helper.typeFixedFacets(this);
	    };
	    /**
	     * A base type which the current type extends, or more generally a type expression.
	     **/
	    TypeDeclarationImpl.prototype["type"] = function () {
	        return helper.typeValue(this);
	    };
	    /**
	     * Alias for the equivalent "type" property, for compatibility with RAML 0.8. Deprecated - API definitions should use the "type" property, as the "schema" alias for that property name may be removed in a future RAML version. The "type" property allows for XML and JSON schemas.
	     **/
	    TypeDeclarationImpl.prototype.schema = function () {
	        return helper.schemaValue(this);
	    };
	    /**
	     * Inlined supertype definition.
	     **/
	    TypeDeclarationImpl.prototype.structuredType = function () {
	        return helper.typeStructuredValue(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    TypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new TypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return TypeDeclarationImpl;
	}(AnnotableImpl));
	exports.TypeDeclarationImpl = TypeDeclarationImpl;
	var ModelLocationImpl = (function () {
	    function ModelLocationImpl(attr) {
	        this.attr = attr;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ModelLocationImpl.prototype.wrapperClassName = function () { return "ModelLocationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ModelLocationImpl.prototype.kind = function () { return "ModelLocation"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ModelLocationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return ModelLocationImpl;
	}());
	exports.ModelLocationImpl = ModelLocationImpl;
	var LocationKindImpl = (function () {
	    function LocationKindImpl(attr) {
	        this.attr = attr;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    LocationKindImpl.prototype.wrapperClassName = function () { return "LocationKindImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    LocationKindImpl.prototype.kind = function () { return "LocationKind"; };
	    /**
	     * @return RAML version of the node
	     **/
	    LocationKindImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return LocationKindImpl;
	}());
	exports.LocationKindImpl = LocationKindImpl;
	var XMLFacetInfoImpl = (function (_super) {
	    __extends(XMLFacetInfoImpl, _super);
	    function XMLFacetInfoImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createXMLFacetInfo(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * If attribute is set to true, a type instance should be serialized as an XML attribute. It can only be true for scalar types.
	     **/
	    XMLFacetInfoImpl.prototype.attribute = function () {
	        return _super.prototype.attribute.call(this, 'attribute', this.toBoolean);
	    };
	    /**
	     * @hidden
	     * Set attribute value
	     **/
	    XMLFacetInfoImpl.prototype.setAttribute = function (param) {
	        this.highLevel().attrOrCreate("attribute").setValue("" + param);
	        return this;
	    };
	    /**
	     * If wrapped is set to true, a type instance should be wrapped in its own XML element. It can not be true for scalar types and it can not be true at the same moment when attribute is true.
	     **/
	    XMLFacetInfoImpl.prototype.wrapped = function () {
	        return _super.prototype.attribute.call(this, 'wrapped', this.toBoolean);
	    };
	    /**
	     * @hidden
	     * Set wrapped value
	     **/
	    XMLFacetInfoImpl.prototype.setWrapped = function (param) {
	        this.highLevel().attrOrCreate("wrapped").setValue("" + param);
	        return this;
	    };
	    /**
	     * Allows to override the name of the XML element or XML attribute in it's XML representation.
	     **/
	    XMLFacetInfoImpl.prototype.name = function () {
	        return _super.prototype.attribute.call(this, 'name', this.toString);
	    };
	    /**
	     * @hidden
	     * Set name value
	     **/
	    XMLFacetInfoImpl.prototype.setName = function (param) {
	        this.highLevel().attrOrCreate("name").setValue("" + param);
	        return this;
	    };
	    /**
	     * Allows to configure the name of the XML namespace.
	     **/
	    XMLFacetInfoImpl.prototype.namespace = function () {
	        return _super.prototype.attribute.call(this, 'namespace', this.toString);
	    };
	    /**
	     * @hidden
	     * Set namespace value
	     **/
	    XMLFacetInfoImpl.prototype.setNamespace = function (param) {
	        this.highLevel().attrOrCreate("namespace").setValue("" + param);
	        return this;
	    };
	    /**
	     * Allows to configure the prefix which will be used during serialization to XML.
	     **/
	    XMLFacetInfoImpl.prototype.prefix = function () {
	        return _super.prototype.attribute.call(this, 'prefix', this.toString);
	    };
	    /**
	     * @hidden
	     * Set prefix value
	     **/
	    XMLFacetInfoImpl.prototype.setPrefix = function (param) {
	        this.highLevel().attrOrCreate("prefix").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    XMLFacetInfoImpl.prototype.wrapperClassName = function () { return "XMLFacetInfoImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    XMLFacetInfoImpl.prototype.kind = function () { return "XMLFacetInfo"; };
	    /**
	     * @return RAML version of the node
	     **/
	    XMLFacetInfoImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    XMLFacetInfoImpl.prototype.scalarsAnnotations = function () { return new XMLFacetInfoScalarsAnnotationsImpl(this.highLevel()); };
	    return XMLFacetInfoImpl;
	}(AnnotableImpl));
	exports.XMLFacetInfoImpl = XMLFacetInfoImpl;
	/**
	 * Annotable scalar properties annotations accessor
	 **/
	var AnnotableScalarsAnnotationsImpl = (function () {
	    function AnnotableScalarsAnnotationsImpl(node) {
	        this.node = node;
	    }
	    /**
	     * Annotable.annotations annotations
	     **/
	    AnnotableScalarsAnnotationsImpl.prototype.annotations = function () {
	        var attrs = this.node.attributes("annotations");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return AnnotableScalarsAnnotationsImpl;
	}());
	exports.AnnotableScalarsAnnotationsImpl = AnnotableScalarsAnnotationsImpl;
	/**
	 * XMLFacetInfo scalar properties annotations accessor
	 **/
	var XMLFacetInfoScalarsAnnotationsImpl = (function (_super) {
	    __extends(XMLFacetInfoScalarsAnnotationsImpl, _super);
	    function XMLFacetInfoScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * XMLFacetInfo.attribute annotations
	     **/
	    XMLFacetInfoScalarsAnnotationsImpl.prototype.attribute = function () {
	        var attr = this.node.attr("attribute");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * XMLFacetInfo.wrapped annotations
	     **/
	    XMLFacetInfoScalarsAnnotationsImpl.prototype.wrapped = function () {
	        var attr = this.node.attr("wrapped");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * XMLFacetInfo.name annotations
	     **/
	    XMLFacetInfoScalarsAnnotationsImpl.prototype.name = function () {
	        var attr = this.node.attr("name");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * XMLFacetInfo.namespace annotations
	     **/
	    XMLFacetInfoScalarsAnnotationsImpl.prototype.namespace = function () {
	        var attr = this.node.attr("namespace");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * XMLFacetInfo.prefix annotations
	     **/
	    XMLFacetInfoScalarsAnnotationsImpl.prototype.prefix = function () {
	        var attr = this.node.attr("prefix");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return XMLFacetInfoScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.XMLFacetInfoScalarsAnnotationsImpl = XMLFacetInfoScalarsAnnotationsImpl;
	var UsesDeclarationImpl = (function (_super) {
	    __extends(UsesDeclarationImpl, _super);
	    function UsesDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createUsesDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Name prefix (without dot) used to refer imported declarations
	     **/
	    UsesDeclarationImpl.prototype.key = function () {
	        return _super.prototype.attribute.call(this, 'key', this.toString);
	    };
	    /**
	     * @hidden
	     * Set key value
	     **/
	    UsesDeclarationImpl.prototype.setKey = function (param) {
	        this.highLevel().attrOrCreate("key").setValue("" + param);
	        return this;
	    };
	    /**
	     * Content of the schema
	     **/
	    UsesDeclarationImpl.prototype.value = function () {
	        return _super.prototype.attribute.call(this, 'value', this.toString);
	    };
	    /**
	     * @hidden
	     * Set value value
	     **/
	    UsesDeclarationImpl.prototype.setValue = function (param) {
	        this.highLevel().attrOrCreate("value").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    UsesDeclarationImpl.prototype.wrapperClassName = function () { return "UsesDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    UsesDeclarationImpl.prototype.kind = function () { return "UsesDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    UsesDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Returns the root node of the AST, uses statement refers.
	     **/
	    UsesDeclarationImpl.prototype.ast = function () {
	        return helper.referencedNode(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    UsesDeclarationImpl.prototype.scalarsAnnotations = function () { return new UsesDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return UsesDeclarationImpl;
	}(AnnotableImpl));
	exports.UsesDeclarationImpl = UsesDeclarationImpl;
	/**
	 * UsesDeclaration scalar properties annotations accessor
	 **/
	var UsesDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(UsesDeclarationScalarsAnnotationsImpl, _super);
	    function UsesDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * UsesDeclaration.value annotations
	     **/
	    UsesDeclarationScalarsAnnotationsImpl.prototype.value = function () {
	        var attr = this.node.attr("value");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return UsesDeclarationScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.UsesDeclarationScalarsAnnotationsImpl = UsesDeclarationScalarsAnnotationsImpl;
	var ArrayTypeDeclarationImpl = (function (_super) {
	    __extends(ArrayTypeDeclarationImpl, _super);
	    function ArrayTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createArrayTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Should items in array be unique
	     **/
	    ArrayTypeDeclarationImpl.prototype.uniqueItems = function () {
	        return _super.prototype.attribute.call(this, 'uniqueItems', this.toBoolean);
	    };
	    /**
	     * @hidden
	     * Set uniqueItems value
	     **/
	    ArrayTypeDeclarationImpl.prototype.setUniqueItems = function (param) {
	        this.highLevel().attrOrCreate("uniqueItems").setValue("" + param);
	        return this;
	    };
	    /**
	     * Array component type.
	     **/
	    ArrayTypeDeclarationImpl.prototype.items = function () {
	        return _super.prototype.element.call(this, 'items');
	    };
	    /**
	     * Minimum amount of items in array
	     **/
	    ArrayTypeDeclarationImpl.prototype.minItems = function () {
	        return _super.prototype.attribute.call(this, 'minItems', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set minItems value
	     **/
	    ArrayTypeDeclarationImpl.prototype.setMinItems = function (param) {
	        this.highLevel().attrOrCreate("minItems").setValue("" + param);
	        return this;
	    };
	    /**
	     * Maximum amount of items in array
	     **/
	    ArrayTypeDeclarationImpl.prototype.maxItems = function () {
	        return _super.prototype.attribute.call(this, 'maxItems', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set maxItems value
	     **/
	    ArrayTypeDeclarationImpl.prototype.setMaxItems = function (param) {
	        this.highLevel().attrOrCreate("maxItems").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ArrayTypeDeclarationImpl.prototype.wrapperClassName = function () { return "ArrayTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ArrayTypeDeclarationImpl.prototype.kind = function () { return "ArrayTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ArrayTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ArrayTypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new ArrayTypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return ArrayTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.ArrayTypeDeclarationImpl = ArrayTypeDeclarationImpl;
	/**
	 * TypeDeclaration scalar properties annotations accessor
	 **/
	var TypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(TypeDeclarationScalarsAnnotationsImpl, _super);
	    function TypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * TypeDeclaration.displayName annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.displayName = function () {
	        var attr = this.node.attr("displayName");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.schema annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.schema = function () {
	        var attrs = this.node.attributes("schema");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * TypeDeclaration.type annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype["type"] = function () {
	        var attrs = this.node.attributes("type");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * TypeDeclaration.location annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.location = function () {
	        var attr = this.node.attr("location");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.locationKind annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.locationKind = function () {
	        var attr = this.node.attr("locationKind");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.default annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype["default"] = function () {
	        var attr = this.node.attr("default");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.repeat annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.repeat = function () {
	        var attr = this.node.attr("repeat");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.required annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.required = function () {
	        var attr = this.node.attr("required");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.description annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.description = function () {
	        var attr = this.node.attr("description");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.allowedTargets annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.allowedTargets = function () {
	        var attrs = this.node.attributes("allowedTargets");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * TypeDeclaration.isAnnotation annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.isAnnotation = function () {
	        var attr = this.node.attr("isAnnotation");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * TypeDeclaration.annotations annotations
	     **/
	    TypeDeclarationScalarsAnnotationsImpl.prototype.annotations = function () {
	        var attrs = this.node.attributes("annotations");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return TypeDeclarationScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.TypeDeclarationScalarsAnnotationsImpl = TypeDeclarationScalarsAnnotationsImpl;
	/**
	 * ArrayTypeDeclaration scalar properties annotations accessor
	 **/
	var ArrayTypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(ArrayTypeDeclarationScalarsAnnotationsImpl, _super);
	    function ArrayTypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * ArrayTypeDeclaration.uniqueItems annotations
	     **/
	    ArrayTypeDeclarationScalarsAnnotationsImpl.prototype.uniqueItems = function () {
	        var attr = this.node.attr("uniqueItems");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ArrayTypeDeclaration.minItems annotations
	     **/
	    ArrayTypeDeclarationScalarsAnnotationsImpl.prototype.minItems = function () {
	        var attr = this.node.attr("minItems");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ArrayTypeDeclaration.maxItems annotations
	     **/
	    ArrayTypeDeclarationScalarsAnnotationsImpl.prototype.maxItems = function () {
	        var attr = this.node.attr("maxItems");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return ArrayTypeDeclarationScalarsAnnotationsImpl;
	}(TypeDeclarationScalarsAnnotationsImpl));
	exports.ArrayTypeDeclarationScalarsAnnotationsImpl = ArrayTypeDeclarationScalarsAnnotationsImpl;
	var UnionTypeDeclarationImpl = (function (_super) {
	    __extends(UnionTypeDeclarationImpl, _super);
	    function UnionTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createUnionTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    UnionTypeDeclarationImpl.prototype.wrapperClassName = function () { return "UnionTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    UnionTypeDeclarationImpl.prototype.kind = function () { return "UnionTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    UnionTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return UnionTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.UnionTypeDeclarationImpl = UnionTypeDeclarationImpl;
	var ObjectTypeDeclarationImpl = (function (_super) {
	    __extends(ObjectTypeDeclarationImpl, _super);
	    function ObjectTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createObjectTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * The properties that instances of this type may or must have.
	     **/
	    ObjectTypeDeclarationImpl.prototype.properties = function () {
	        return _super.prototype.elements.call(this, 'properties');
	    };
	    /**
	     * The minimum number of properties allowed for instances of this type.
	     **/
	    ObjectTypeDeclarationImpl.prototype.minProperties = function () {
	        return _super.prototype.attribute.call(this, 'minProperties', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set minProperties value
	     **/
	    ObjectTypeDeclarationImpl.prototype.setMinProperties = function (param) {
	        this.highLevel().attrOrCreate("minProperties").setValue("" + param);
	        return this;
	    };
	    /**
	     * The maximum number of properties allowed for instances of this type.
	     **/
	    ObjectTypeDeclarationImpl.prototype.maxProperties = function () {
	        return _super.prototype.attribute.call(this, 'maxProperties', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set maxProperties value
	     **/
	    ObjectTypeDeclarationImpl.prototype.setMaxProperties = function (param) {
	        this.highLevel().attrOrCreate("maxProperties").setValue("" + param);
	        return this;
	    };
	    /**
	     * JSON schema style syntax for declaring maps
	     **/
	    ObjectTypeDeclarationImpl.prototype.additionalProperties = function () {
	        return _super.prototype.element.call(this, 'additionalProperties');
	    };
	    /**
	     * JSON schema style syntax for declaring key restricted maps
	     **/
	    ObjectTypeDeclarationImpl.prototype.patternProperties = function () {
	        return _super.prototype.elements.call(this, 'patternProperties');
	    };
	    /**
	     * Type property name to be used as discriminator, or boolean
	     **/
	    ObjectTypeDeclarationImpl.prototype.discriminator = function () {
	        return _super.prototype.attribute.call(this, 'discriminator', this.toString);
	    };
	    /**
	     * @hidden
	     * Set discriminator value
	     **/
	    ObjectTypeDeclarationImpl.prototype.setDiscriminator = function (param) {
	        this.highLevel().attrOrCreate("discriminator").setValue("" + param);
	        return this;
	    };
	    /**
	     * The value of discriminator for the type.
	     **/
	    ObjectTypeDeclarationImpl.prototype.discriminatorValue = function () {
	        return _super.prototype.attribute.call(this, 'discriminatorValue', this.toString);
	    };
	    /**
	     * @hidden
	     * Set discriminatorValue value
	     **/
	    ObjectTypeDeclarationImpl.prototype.setDiscriminatorValue = function (param) {
	        this.highLevel().attrOrCreate("discriminatorValue").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ObjectTypeDeclarationImpl.prototype.wrapperClassName = function () { return "ObjectTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ObjectTypeDeclarationImpl.prototype.kind = function () { return "ObjectTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ObjectTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ObjectTypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new ObjectTypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return ObjectTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.ObjectTypeDeclarationImpl = ObjectTypeDeclarationImpl;
	/**
	 * ObjectTypeDeclaration scalar properties annotations accessor
	 **/
	var ObjectTypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(ObjectTypeDeclarationScalarsAnnotationsImpl, _super);
	    function ObjectTypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * ObjectTypeDeclaration.minProperties annotations
	     **/
	    ObjectTypeDeclarationScalarsAnnotationsImpl.prototype.minProperties = function () {
	        var attr = this.node.attr("minProperties");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ObjectTypeDeclaration.maxProperties annotations
	     **/
	    ObjectTypeDeclarationScalarsAnnotationsImpl.prototype.maxProperties = function () {
	        var attr = this.node.attr("maxProperties");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ObjectTypeDeclaration.discriminator annotations
	     **/
	    ObjectTypeDeclarationScalarsAnnotationsImpl.prototype.discriminator = function () {
	        var attr = this.node.attr("discriminator");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ObjectTypeDeclaration.discriminatorValue annotations
	     **/
	    ObjectTypeDeclarationScalarsAnnotationsImpl.prototype.discriminatorValue = function () {
	        var attr = this.node.attr("discriminatorValue");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return ObjectTypeDeclarationScalarsAnnotationsImpl;
	}(TypeDeclarationScalarsAnnotationsImpl));
	exports.ObjectTypeDeclarationScalarsAnnotationsImpl = ObjectTypeDeclarationScalarsAnnotationsImpl;
	/**
	 * Value must be a string
	 **/
	var StringTypeDeclarationImpl = (function (_super) {
	    __extends(StringTypeDeclarationImpl, _super);
	    function StringTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createStringTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Regular expression that this string should path
	     **/
	    StringTypeDeclarationImpl.prototype.pattern = function () {
	        return _super.prototype.attribute.call(this, 'pattern', this.toString);
	    };
	    /**
	     * @hidden
	     * Set pattern value
	     **/
	    StringTypeDeclarationImpl.prototype.setPattern = function (param) {
	        this.highLevel().attrOrCreate("pattern").setValue("" + param);
	        return this;
	    };
	    /**
	     * Minimum length of the string
	     **/
	    StringTypeDeclarationImpl.prototype.minLength = function () {
	        return _super.prototype.attribute.call(this, 'minLength', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set minLength value
	     **/
	    StringTypeDeclarationImpl.prototype.setMinLength = function (param) {
	        this.highLevel().attrOrCreate("minLength").setValue("" + param);
	        return this;
	    };
	    /**
	     * Maximum length of the string
	     **/
	    StringTypeDeclarationImpl.prototype.maxLength = function () {
	        return _super.prototype.attribute.call(this, 'maxLength', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set maxLength value
	     **/
	    StringTypeDeclarationImpl.prototype.setMaxLength = function (param) {
	        this.highLevel().attrOrCreate("maxLength").setValue("" + param);
	        return this;
	    };
	    /**
	     * (Optional, applicable only for parameters of type string) The enum attribute provides an enumeration of the parameter's valid values. This MUST be an array. If the enum attribute is defined, API clients and servers MUST verify that a parameter's value matches a value in the enum array. If there is no matching value, the clients and servers MUST treat this as an error.
	     **/
	    StringTypeDeclarationImpl.prototype.enum = function () {
	        return _super.prototype.attributes.call(this, 'enum', this.toString);
	    };
	    /**
	     * @hidden
	     * Set enum value
	     **/
	    StringTypeDeclarationImpl.prototype.setEnum = function (param) {
	        this.highLevel().attrOrCreate("enum").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    StringTypeDeclarationImpl.prototype.wrapperClassName = function () { return "StringTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    StringTypeDeclarationImpl.prototype.kind = function () { return "StringTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    StringTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    StringTypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new StringTypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return StringTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.StringTypeDeclarationImpl = StringTypeDeclarationImpl;
	/**
	 * StringTypeDeclaration scalar properties annotations accessor
	 **/
	var StringTypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(StringTypeDeclarationScalarsAnnotationsImpl, _super);
	    function StringTypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * StringTypeDeclaration.pattern annotations
	     **/
	    StringTypeDeclarationScalarsAnnotationsImpl.prototype.pattern = function () {
	        var attr = this.node.attr("pattern");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * StringTypeDeclaration.minLength annotations
	     **/
	    StringTypeDeclarationScalarsAnnotationsImpl.prototype.minLength = function () {
	        var attr = this.node.attr("minLength");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * StringTypeDeclaration.maxLength annotations
	     **/
	    StringTypeDeclarationScalarsAnnotationsImpl.prototype.maxLength = function () {
	        var attr = this.node.attr("maxLength");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * StringTypeDeclaration.enum annotations
	     **/
	    StringTypeDeclarationScalarsAnnotationsImpl.prototype.enum = function () {
	        var attrs = this.node.attributes("enum");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return StringTypeDeclarationScalarsAnnotationsImpl;
	}(TypeDeclarationScalarsAnnotationsImpl));
	exports.StringTypeDeclarationScalarsAnnotationsImpl = StringTypeDeclarationScalarsAnnotationsImpl;
	/**
	 * Value must be a boolean
	 **/
	var BooleanTypeDeclarationImpl = (function (_super) {
	    __extends(BooleanTypeDeclarationImpl, _super);
	    function BooleanTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createBooleanTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    BooleanTypeDeclarationImpl.prototype.wrapperClassName = function () { return "BooleanTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    BooleanTypeDeclarationImpl.prototype.kind = function () { return "BooleanTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    BooleanTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return BooleanTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.BooleanTypeDeclarationImpl = BooleanTypeDeclarationImpl;
	/**
	 * Value MUST be a number. Indicate floating point numbers as defined by YAML.
	 **/
	var NumberTypeDeclarationImpl = (function (_super) {
	    __extends(NumberTypeDeclarationImpl, _super);
	    function NumberTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createNumberTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * (Optional, applicable only for parameters of type number or integer) The minimum attribute specifies the parameter's minimum value.
	     **/
	    NumberTypeDeclarationImpl.prototype.minimum = function () {
	        return _super.prototype.attribute.call(this, 'minimum', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set minimum value
	     **/
	    NumberTypeDeclarationImpl.prototype.setMinimum = function (param) {
	        this.highLevel().attrOrCreate("minimum").setValue("" + param);
	        return this;
	    };
	    /**
	     * (Optional, applicable only for parameters of type number or integer) The maximum attribute specifies the parameter's maximum value.
	     **/
	    NumberTypeDeclarationImpl.prototype.maximum = function () {
	        return _super.prototype.attribute.call(this, 'maximum', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set maximum value
	     **/
	    NumberTypeDeclarationImpl.prototype.setMaximum = function (param) {
	        this.highLevel().attrOrCreate("maximum").setValue("" + param);
	        return this;
	    };
	    /**
	     * (Optional, applicable only for parameters of type string) The enum attribute provides an enumeration of the parameter's valid values. This MUST be an array. If the enum attribute is defined, API clients and servers MUST verify that a parameter's value matches a value in the enum array. If there is no matching value, the clients and servers MUST treat this as an error.
	     **/
	    NumberTypeDeclarationImpl.prototype.enum = function () {
	        return _super.prototype.attributes.call(this, 'enum', this.toString);
	    };
	    /**
	     * @hidden
	     * Set enum value
	     **/
	    NumberTypeDeclarationImpl.prototype.setEnum = function (param) {
	        this.highLevel().attrOrCreate("enum").setValue("" + param);
	        return this;
	    };
	    /**
	     * Value format
	     **/
	    NumberTypeDeclarationImpl.prototype.format = function () {
	        return _super.prototype.attribute.call(this, 'format', this.toString);
	    };
	    /**
	     * @hidden
	     * Set format value
	     **/
	    NumberTypeDeclarationImpl.prototype.setFormat = function (param) {
	        this.highLevel().attrOrCreate("format").setValue("" + param);
	        return this;
	    };
	    /**
	     * A numeric instance is valid against "multipleOf" if the result of the division of the instance by this keyword's value is an integer.
	     **/
	    NumberTypeDeclarationImpl.prototype.multipleOf = function () {
	        return _super.prototype.attribute.call(this, 'multipleOf', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set multipleOf value
	     **/
	    NumberTypeDeclarationImpl.prototype.setMultipleOf = function (param) {
	        this.highLevel().attrOrCreate("multipleOf").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    NumberTypeDeclarationImpl.prototype.wrapperClassName = function () { return "NumberTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    NumberTypeDeclarationImpl.prototype.kind = function () { return "NumberTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    NumberTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    NumberTypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new NumberTypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return NumberTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.NumberTypeDeclarationImpl = NumberTypeDeclarationImpl;
	/**
	 * Value MUST be a integer.
	 **/
	var IntegerTypeDeclarationImpl = (function (_super) {
	    __extends(IntegerTypeDeclarationImpl, _super);
	    function IntegerTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createIntegerTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Value format
	     **/
	    IntegerTypeDeclarationImpl.prototype.format = function () {
	        return _super.prototype.attribute.call(this, 'format', this.toString);
	    };
	    /**
	     * @hidden
	     * Set format value
	     **/
	    IntegerTypeDeclarationImpl.prototype.setFormat = function (param) {
	        this.highLevel().attrOrCreate("format").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    IntegerTypeDeclarationImpl.prototype.wrapperClassName = function () { return "IntegerTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    IntegerTypeDeclarationImpl.prototype.kind = function () { return "IntegerTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    IntegerTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    IntegerTypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new IntegerTypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return IntegerTypeDeclarationImpl;
	}(NumberTypeDeclarationImpl));
	exports.IntegerTypeDeclarationImpl = IntegerTypeDeclarationImpl;
	/**
	 * NumberTypeDeclaration scalar properties annotations accessor
	 **/
	var NumberTypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(NumberTypeDeclarationScalarsAnnotationsImpl, _super);
	    function NumberTypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * NumberTypeDeclaration.minimum annotations
	     **/
	    NumberTypeDeclarationScalarsAnnotationsImpl.prototype.minimum = function () {
	        var attr = this.node.attr("minimum");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * NumberTypeDeclaration.maximum annotations
	     **/
	    NumberTypeDeclarationScalarsAnnotationsImpl.prototype.maximum = function () {
	        var attr = this.node.attr("maximum");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * NumberTypeDeclaration.enum annotations
	     **/
	    NumberTypeDeclarationScalarsAnnotationsImpl.prototype.enum = function () {
	        var attrs = this.node.attributes("enum");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * NumberTypeDeclaration.format annotations
	     **/
	    NumberTypeDeclarationScalarsAnnotationsImpl.prototype.format = function () {
	        var attr = this.node.attr("format");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * NumberTypeDeclaration.multipleOf annotations
	     **/
	    NumberTypeDeclarationScalarsAnnotationsImpl.prototype.multipleOf = function () {
	        var attr = this.node.attr("multipleOf");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return NumberTypeDeclarationScalarsAnnotationsImpl;
	}(TypeDeclarationScalarsAnnotationsImpl));
	exports.NumberTypeDeclarationScalarsAnnotationsImpl = NumberTypeDeclarationScalarsAnnotationsImpl;
	/**
	 * IntegerTypeDeclaration scalar properties annotations accessor
	 **/
	var IntegerTypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(IntegerTypeDeclarationScalarsAnnotationsImpl, _super);
	    function IntegerTypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * IntegerTypeDeclaration.format annotations
	     **/
	    IntegerTypeDeclarationScalarsAnnotationsImpl.prototype.format = function () {
	        var attr = this.node.attr("format");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return IntegerTypeDeclarationScalarsAnnotationsImpl;
	}(NumberTypeDeclarationScalarsAnnotationsImpl));
	exports.IntegerTypeDeclarationScalarsAnnotationsImpl = IntegerTypeDeclarationScalarsAnnotationsImpl;
	/**
	 * the "full-date" notation of RFC3339, namely yyyy-mm-dd (no implications about time or timezone-offset)
	 **/
	var DateOnlyTypeDeclarationImpl = (function (_super) {
	    __extends(DateOnlyTypeDeclarationImpl, _super);
	    function DateOnlyTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createDateOnlyTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DateOnlyTypeDeclarationImpl.prototype.wrapperClassName = function () { return "DateOnlyTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DateOnlyTypeDeclarationImpl.prototype.kind = function () { return "DateOnlyTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DateOnlyTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return DateOnlyTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.DateOnlyTypeDeclarationImpl = DateOnlyTypeDeclarationImpl;
	/**
	 * the "partial-time" notation of RFC3339, namely hh:mm:ss[.ff...] (no implications about date or timezone-offset)
	 **/
	var TimeOnlyTypeDeclarationImpl = (function (_super) {
	    __extends(TimeOnlyTypeDeclarationImpl, _super);
	    function TimeOnlyTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createTimeOnlyTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    TimeOnlyTypeDeclarationImpl.prototype.wrapperClassName = function () { return "TimeOnlyTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    TimeOnlyTypeDeclarationImpl.prototype.kind = function () { return "TimeOnlyTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    TimeOnlyTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return TimeOnlyTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.TimeOnlyTypeDeclarationImpl = TimeOnlyTypeDeclarationImpl;
	/**
	 * combined date-only and time-only with a separator of "T", namely yyyy-mm-ddThh:mm:ss[.ff...] (no implications about timezone-offset)
	 **/
	var DateTimeOnlyTypeDeclarationImpl = (function (_super) {
	    __extends(DateTimeOnlyTypeDeclarationImpl, _super);
	    function DateTimeOnlyTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createDateTimeOnlyTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DateTimeOnlyTypeDeclarationImpl.prototype.wrapperClassName = function () { return "DateTimeOnlyTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DateTimeOnlyTypeDeclarationImpl.prototype.kind = function () { return "DateTimeOnlyTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DateTimeOnlyTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return DateTimeOnlyTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.DateTimeOnlyTypeDeclarationImpl = DateTimeOnlyTypeDeclarationImpl;
	/**
	 * a timestamp, either in the "date-time" notation of RFC3339, if format is omitted or is set to rfc3339, or in the format defined in RFC2616, if format is set to rfc2616.
	 **/
	var DateTimeTypeDeclarationImpl = (function (_super) {
	    __extends(DateTimeTypeDeclarationImpl, _super);
	    function DateTimeTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createDateTimeTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Format used for this date time rfc3339 or rfc2616
	     **/
	    DateTimeTypeDeclarationImpl.prototype.format = function () {
	        return _super.prototype.attribute.call(this, 'format', this.toString);
	    };
	    /**
	     * @hidden
	     * Set format value
	     **/
	    DateTimeTypeDeclarationImpl.prototype.setFormat = function (param) {
	        this.highLevel().attrOrCreate("format").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DateTimeTypeDeclarationImpl.prototype.wrapperClassName = function () { return "DateTimeTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DateTimeTypeDeclarationImpl.prototype.kind = function () { return "DateTimeTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DateTimeTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    DateTimeTypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new DateTimeTypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return DateTimeTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.DateTimeTypeDeclarationImpl = DateTimeTypeDeclarationImpl;
	/**
	 * DateTimeTypeDeclaration scalar properties annotations accessor
	 **/
	var DateTimeTypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(DateTimeTypeDeclarationScalarsAnnotationsImpl, _super);
	    function DateTimeTypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * DateTimeTypeDeclaration.format annotations
	     **/
	    DateTimeTypeDeclarationScalarsAnnotationsImpl.prototype.format = function () {
	        var attr = this.node.attr("format");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return DateTimeTypeDeclarationScalarsAnnotationsImpl;
	}(TypeDeclarationScalarsAnnotationsImpl));
	exports.DateTimeTypeDeclarationScalarsAnnotationsImpl = DateTimeTypeDeclarationScalarsAnnotationsImpl;
	/**
	 * Value MUST be a string representation of a date as defined in RFC2616 Section 3.3, or according to specified date format
	 **/
	var DateTypeDeclarationImpl = (function (_super) {
	    __extends(DateTypeDeclarationImpl, _super);
	    function DateTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createDateTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DateTypeDeclarationImpl.prototype.wrapperClassName = function () { return "DateTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DateTypeDeclarationImpl.prototype.kind = function () { return "DateTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DateTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return DateTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.DateTypeDeclarationImpl = DateTypeDeclarationImpl;
	/**
	 * (Applicable only to Form properties) Value is a file. Client generators SHOULD use this type to handle file uploads correctly.
	 **/
	var FileTypeDeclarationImpl = (function (_super) {
	    __extends(FileTypeDeclarationImpl, _super);
	    function FileTypeDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createFileTypeDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * It should also include a new property: fileTypes, which should be a list of valid content-type strings for the file. The file type * /* should be a valid value.
	     **/
	    FileTypeDeclarationImpl.prototype.fileTypes = function () {
	        return _super.prototype.attributes.call(this, 'fileTypes', function (attr) { return new ContentTypeImpl(attr); });
	    };
	    /**
	     * The minLength attribute specifies the parameter value's minimum number of bytes.
	     **/
	    FileTypeDeclarationImpl.prototype.minLength = function () {
	        return _super.prototype.attribute.call(this, 'minLength', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set minLength value
	     **/
	    FileTypeDeclarationImpl.prototype.setMinLength = function (param) {
	        this.highLevel().attrOrCreate("minLength").setValue("" + param);
	        return this;
	    };
	    /**
	     * The maxLength attribute specifies the parameter value's maximum number of bytes.
	     **/
	    FileTypeDeclarationImpl.prototype.maxLength = function () {
	        return _super.prototype.attribute.call(this, 'maxLength', this.toNumber);
	    };
	    /**
	     * @hidden
	     * Set maxLength value
	     **/
	    FileTypeDeclarationImpl.prototype.setMaxLength = function (param) {
	        this.highLevel().attrOrCreate("maxLength").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    FileTypeDeclarationImpl.prototype.wrapperClassName = function () { return "FileTypeDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    FileTypeDeclarationImpl.prototype.kind = function () { return "FileTypeDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    FileTypeDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    FileTypeDeclarationImpl.prototype.scalarsAnnotations = function () { return new FileTypeDeclarationScalarsAnnotationsImpl(this.highLevel()); };
	    return FileTypeDeclarationImpl;
	}(TypeDeclarationImpl));
	exports.FileTypeDeclarationImpl = FileTypeDeclarationImpl;
	/**
	 * FileTypeDeclaration scalar properties annotations accessor
	 **/
	var FileTypeDeclarationScalarsAnnotationsImpl = (function (_super) {
	    __extends(FileTypeDeclarationScalarsAnnotationsImpl, _super);
	    function FileTypeDeclarationScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * FileTypeDeclaration.fileTypes annotations
	     **/
	    FileTypeDeclarationScalarsAnnotationsImpl.prototype.fileTypes = function () {
	        var attrs = this.node.attributes("fileTypes");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * FileTypeDeclaration.minLength annotations
	     **/
	    FileTypeDeclarationScalarsAnnotationsImpl.prototype.minLength = function () {
	        var attr = this.node.attr("minLength");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * FileTypeDeclaration.maxLength annotations
	     **/
	    FileTypeDeclarationScalarsAnnotationsImpl.prototype.maxLength = function () {
	        var attr = this.node.attr("maxLength");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return FileTypeDeclarationScalarsAnnotationsImpl;
	}(TypeDeclarationScalarsAnnotationsImpl));
	exports.FileTypeDeclarationScalarsAnnotationsImpl = FileTypeDeclarationScalarsAnnotationsImpl;
	var ResponseImpl = (function (_super) {
	    __extends(ResponseImpl, _super);
	    function ResponseImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createResponse(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Responses MUST be a map of one or more HTTP status codes, where each status code itself is a map that describes that status code.
	     **/
	    ResponseImpl.prototype.code = function () {
	        return _super.prototype.attribute.call(this, 'code', function (attr) { return new StatusCodeStringImpl(attr); });
	    };
	    /**
	     * Detailed information about any response headers returned by this method
	     **/
	    ResponseImpl.prototype.headers = function () {
	        return _super.prototype.elements.call(this, 'headers');
	    };
	    /**
	     * The body of the response: a body declaration
	     **/
	    ResponseImpl.prototype.body = function () {
	        return _super.prototype.elements.call(this, 'body');
	    };
	    /**
	     * A longer, human-friendly description of the response
	     **/
	    ResponseImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    /**
	     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
	     **/
	    ResponseImpl.prototype.annotations = function () {
	        return _super.prototype.attributes.call(this, 'annotations', function (attr) { return new AnnotationRefImpl(attr); });
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ResponseImpl.prototype.wrapperClassName = function () { return "ResponseImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ResponseImpl.prototype.kind = function () { return "Response"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ResponseImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * true for codes < 400 and false otherwise
	     **/
	    ResponseImpl.prototype.isOkRange = function () {
	        return helper.isOkRange(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ResponseImpl.prototype.scalarsAnnotations = function () { return new ResponseScalarsAnnotationsImpl(this.highLevel()); };
	    return ResponseImpl;
	}(AnnotableImpl));
	exports.ResponseImpl = ResponseImpl;
	/**
	 * Response scalar properties annotations accessor
	 **/
	var ResponseScalarsAnnotationsImpl = (function (_super) {
	    __extends(ResponseScalarsAnnotationsImpl, _super);
	    function ResponseScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Response.description annotations
	     **/
	    ResponseScalarsAnnotationsImpl.prototype.description = function () {
	        var attr = this.node.attr("description");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Response.annotations annotations
	     **/
	    ResponseScalarsAnnotationsImpl.prototype.annotations = function () {
	        var attrs = this.node.attributes("annotations");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return ResponseScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.ResponseScalarsAnnotationsImpl = ResponseScalarsAnnotationsImpl;
	var SecuritySchemePartImpl = (function (_super) {
	    __extends(SecuritySchemePartImpl, _super);
	    function SecuritySchemePartImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createSecuritySchemePart(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Annotations to be applied to this security scheme part. Annotations are any property whose key begins with "(" and ends with ")" and whose name (the part between the beginning and ending parentheses) is a declared annotation name.
	     **/
	    SecuritySchemePartImpl.prototype.annotations = function () {
	        return _super.prototype.attributes.call(this, 'annotations', function (attr) { return new AnnotationRefImpl(attr); });
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    SecuritySchemePartImpl.prototype.wrapperClassName = function () { return "SecuritySchemePartImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    SecuritySchemePartImpl.prototype.kind = function () { return "SecuritySchemePart"; };
	    /**
	     * @return RAML version of the node
	     **/
	    SecuritySchemePartImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    SecuritySchemePartImpl.prototype.scalarsAnnotations = function () { return new SecuritySchemePartScalarsAnnotationsImpl(this.highLevel()); };
	    return SecuritySchemePartImpl;
	}(OperationImpl));
	exports.SecuritySchemePartImpl = SecuritySchemePartImpl;
	/**
	 * SecuritySchemePart scalar properties annotations accessor
	 **/
	var SecuritySchemePartScalarsAnnotationsImpl = (function (_super) {
	    __extends(SecuritySchemePartScalarsAnnotationsImpl, _super);
	    function SecuritySchemePartScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * SecuritySchemePart.annotations annotations
	     **/
	    SecuritySchemePartScalarsAnnotationsImpl.prototype.annotations = function () {
	        var attrs = this.node.attributes("annotations");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return SecuritySchemePartScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.SecuritySchemePartScalarsAnnotationsImpl = SecuritySchemePartScalarsAnnotationsImpl;
	var MethodBaseImpl = (function (_super) {
	    __extends(MethodBaseImpl, _super);
	    function MethodBaseImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createMethodBase(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Some method verbs expect the resource to be sent as a request body. For example, to create a resource, the request must include the details of the resource to create. Resources CAN have alternate representations. For example, an API might support both JSON and XML representations. A method's body is defined in the body property as a hashmap, in which the key MUST be a valid media type.
	     **/
	    MethodBaseImpl.prototype.body = function () {
	        return _super.prototype.elements.call(this, 'body');
	    };
	    /**
	     * A method can override the protocols specified in the resource or at the API root, by employing this property.
	     **/
	    MethodBaseImpl.prototype.protocols = function () {
	        return _super.prototype.attributes.call(this, 'protocols', this.toString);
	    };
	    /**
	     * @hidden
	     * Set protocols value
	     **/
	    MethodBaseImpl.prototype.setProtocols = function (param) {
	        this.highLevel().attrOrCreate("protocols").setValue("" + param);
	        return this;
	    };
	    /**
	     * Instantiation of applyed traits
	     **/
	    MethodBaseImpl.prototype.is = function () {
	        return _super.prototype.attributes.call(this, 'is', function (attr) { return new TraitRefImpl(attr); });
	    };
	    /**
	     * securityScheme may also be applied to a resource by using the securedBy key, which is equivalent to applying the securityScheme to all methods that may be declared, explicitly or implicitly, by defining the resourceTypes or traits property for that resource. To indicate that the method may be called without applying any securityScheme, the method may be annotated with the null securityScheme.
	     **/
	    MethodBaseImpl.prototype.securedBy = function () {
	        return _super.prototype.attributes.call(this, 'securedBy', function (attr) { return new SecuritySchemeRefImpl(attr); });
	    };
	    MethodBaseImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    MethodBaseImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    MethodBaseImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    MethodBaseImpl.prototype.wrapperClassName = function () { return "MethodBaseImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    MethodBaseImpl.prototype.kind = function () { return "MethodBase"; };
	    /**
	     * @return RAML version of the node
	     **/
	    MethodBaseImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    MethodBaseImpl.prototype.scalarsAnnotations = function () { return new MethodBaseScalarsAnnotationsImpl(this.highLevel()); };
	    return MethodBaseImpl;
	}(OperationImpl));
	exports.MethodBaseImpl = MethodBaseImpl;
	var SecuritySchemeRefImpl = (function (_super) {
	    __extends(SecuritySchemeRefImpl, _super);
	    function SecuritySchemeRefImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    SecuritySchemeRefImpl.prototype.wrapperClassName = function () { return "SecuritySchemeRefImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    SecuritySchemeRefImpl.prototype.kind = function () { return "SecuritySchemeRef"; };
	    /**
	     * @return RAML version of the node
	     **/
	    SecuritySchemeRefImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    SecuritySchemeRefImpl.prototype.securitySchemeName = function () {
	        return helper.securitySchemeName(this);
	    };
	    SecuritySchemeRefImpl.prototype.securityScheme = function () {
	        return helper.securityScheme(this);
	    };
	    return SecuritySchemeRefImpl;
	}(ReferenceImpl));
	exports.SecuritySchemeRefImpl = SecuritySchemeRefImpl;
	/**
	 * Declares globally referable security scheme definition
	 **/
	var AbstractSecuritySchemeImpl = (function (_super) {
	    __extends(AbstractSecuritySchemeImpl, _super);
	    function AbstractSecuritySchemeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createAbstractSecurityScheme(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Name of the security scheme
	     **/
	    AbstractSecuritySchemeImpl.prototype.name = function () {
	        return _super.prototype.attribute.call(this, 'name', this.toString);
	    };
	    /**
	     * @hidden
	     * Set name value
	     **/
	    AbstractSecuritySchemeImpl.prototype.setName = function (param) {
	        this.highLevel().attrOrCreate("name").setValue("" + param);
	        return this;
	    };
	    /**
	     * The securitySchemes property MUST be used to specify an API's security mechanisms, including the required settings and the authentication methods that the API supports. one authentication method is allowed if the API supports them.
	     **/
	    AbstractSecuritySchemeImpl.prototype["type"] = function () {
	        return _super.prototype.attribute.call(this, 'type', this.toString);
	    };
	    /**
	     * @hidden
	     * Set type value
	     **/
	    AbstractSecuritySchemeImpl.prototype.setType = function (param) {
	        this.highLevel().attrOrCreate("type").setValue("" + param);
	        return this;
	    };
	    /**
	     * The description MAY be used to describe a securityScheme.
	     **/
	    AbstractSecuritySchemeImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    /**
	     * A description of the request components related to Security that are determined by the scheme: the headers, query parameters or responses. As a best practice, even for standard security schemes, API designers SHOULD describe these properties of security schemes. Including the security scheme description completes an API documentation.
	     **/
	    AbstractSecuritySchemeImpl.prototype.describedBy = function () {
	        return _super.prototype.element.call(this, 'describedBy');
	    };
	    /**
	     * The displayName attribute specifies the security scheme display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
	     **/
	    AbstractSecuritySchemeImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    AbstractSecuritySchemeImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    /**
	     * The settings attribute MAY be used to provide security scheme-specific information. The required attributes vary depending on the type of security scheme is being declared. It describes the minimum set of properties which any processing application MUST provide and validate if it chooses to implement the security scheme. Processing applications MAY choose to recognize other properties for things such as token lifetime, preferred cryptographic algorithms, and more.
	     **/
	    AbstractSecuritySchemeImpl.prototype.settings = function () {
	        return _super.prototype.element.call(this, 'settings');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    AbstractSecuritySchemeImpl.prototype.wrapperClassName = function () { return "AbstractSecuritySchemeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    AbstractSecuritySchemeImpl.prototype.kind = function () { return "AbstractSecurityScheme"; };
	    /**
	     * @return RAML version of the node
	     **/
	    AbstractSecuritySchemeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    AbstractSecuritySchemeImpl.prototype.scalarsAnnotations = function () { return new AbstractSecuritySchemeScalarsAnnotationsImpl(this.highLevel()); };
	    return AbstractSecuritySchemeImpl;
	}(AnnotableImpl));
	exports.AbstractSecuritySchemeImpl = AbstractSecuritySchemeImpl;
	var SecuritySchemeSettingsImpl = (function (_super) {
	    __extends(SecuritySchemeSettingsImpl, _super);
	    function SecuritySchemeSettingsImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createSecuritySchemeSettings(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    SecuritySchemeSettingsImpl.prototype.wrapperClassName = function () { return "SecuritySchemeSettingsImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    SecuritySchemeSettingsImpl.prototype.kind = function () { return "SecuritySchemeSettings"; };
	    /**
	     * @return RAML version of the node
	     **/
	    SecuritySchemeSettingsImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return SecuritySchemeSettingsImpl;
	}(AnnotableImpl));
	exports.SecuritySchemeSettingsImpl = SecuritySchemeSettingsImpl;
	var OAuth1SecuritySchemeSettingsImpl = (function (_super) {
	    __extends(OAuth1SecuritySchemeSettingsImpl, _super);
	    function OAuth1SecuritySchemeSettingsImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createOAuth1SecuritySchemeSettings(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * The URI of the Temporary Credential Request endpoint as defined in RFC5849 Section 2.1
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.requestTokenUri = function () {
	        return _super.prototype.attribute.call(this, 'requestTokenUri', function (attr) { return new FixedUriStringImpl(attr); });
	    };
	    /**
	     * The URI of the Resource Owner Authorization endpoint as defined in RFC5849 Section 2.2
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.authorizationUri = function () {
	        return _super.prototype.attribute.call(this, 'authorizationUri', function (attr) { return new FixedUriStringImpl(attr); });
	    };
	    /**
	     * The URI of the Token Request endpoint as defined in RFC5849 Section 2.3
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.tokenCredentialsUri = function () {
	        return _super.prototype.attribute.call(this, 'tokenCredentialsUri', function (attr) { return new FixedUriStringImpl(attr); });
	    };
	    /**
	     * List of the signature methods used by the server. Available methods: HMAC-SHA1, RSA-SHA1, PLAINTEXT
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.signatures = function () {
	        return _super.prototype.attributes.call(this, 'signatures', this.toString);
	    };
	    /**
	     * @hidden
	     * Set signatures value
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.setSignatures = function (param) {
	        this.highLevel().attrOrCreate("signatures").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.wrapperClassName = function () { return "OAuth1SecuritySchemeSettingsImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.kind = function () { return "OAuth1SecuritySchemeSettings"; };
	    /**
	     * @return RAML version of the node
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    OAuth1SecuritySchemeSettingsImpl.prototype.scalarsAnnotations = function () { return new OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl(this.highLevel()); };
	    return OAuth1SecuritySchemeSettingsImpl;
	}(SecuritySchemeSettingsImpl));
	exports.OAuth1SecuritySchemeSettingsImpl = OAuth1SecuritySchemeSettingsImpl;
	/**
	 * OAuth1SecuritySchemeSettings scalar properties annotations accessor
	 **/
	var OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl = (function (_super) {
	    __extends(OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl, _super);
	    function OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * OAuth1SecuritySchemeSettings.requestTokenUri annotations
	     **/
	    OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.requestTokenUri = function () {
	        var attr = this.node.attr("requestTokenUri");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * OAuth1SecuritySchemeSettings.authorizationUri annotations
	     **/
	    OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.authorizationUri = function () {
	        var attr = this.node.attr("authorizationUri");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * OAuth1SecuritySchemeSettings.tokenCredentialsUri annotations
	     **/
	    OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.tokenCredentialsUri = function () {
	        var attr = this.node.attr("tokenCredentialsUri");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * OAuth1SecuritySchemeSettings.signatures annotations
	     **/
	    OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.signatures = function () {
	        var attrs = this.node.attributes("signatures");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl = OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl;
	var OAuth2SecuritySchemeSettingsImpl = (function (_super) {
	    __extends(OAuth2SecuritySchemeSettingsImpl, _super);
	    function OAuth2SecuritySchemeSettingsImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createOAuth2SecuritySchemeSettings(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * The URI of the Token Endpoint as defined in RFC6749 Section 3.2. Not required forby implicit grant type.
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.accessTokenUri = function () {
	        return _super.prototype.attribute.call(this, 'accessTokenUri', function (attr) { return new FixedUriStringImpl(attr); });
	    };
	    /**
	     * The URI of the Authorization Endpoint as defined in RFC6749 Section 3.1. Required forby authorization_code and implicit grant types.
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.authorizationUri = function () {
	        return _super.prototype.attribute.call(this, 'authorizationUri', function (attr) { return new FixedUriStringImpl(attr); });
	    };
	    /**
	     * A list of the Authorization grants supported by the API as defined in RFC6749 Sections 4.1, 4.2, 4.3 and 4.4, can be any of: authorization_code, password, client_credentials, implicit, or any absolute url.
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.authorizationGrants = function () {
	        return _super.prototype.attributes.call(this, 'authorizationGrants', this.toString);
	    };
	    /**
	     * @hidden
	     * Set authorizationGrants value
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.setAuthorizationGrants = function (param) {
	        this.highLevel().attrOrCreate("authorizationGrants").setValue("" + param);
	        return this;
	    };
	    /**
	     * A list of scopes supported by the security scheme as defined in RFC6749 Section 3.3
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.scopes = function () {
	        return _super.prototype.attributes.call(this, 'scopes', this.toString);
	    };
	    /**
	     * @hidden
	     * Set scopes value
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.setScopes = function (param) {
	        this.highLevel().attrOrCreate("scopes").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.wrapperClassName = function () { return "OAuth2SecuritySchemeSettingsImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.kind = function () { return "OAuth2SecuritySchemeSettings"; };
	    /**
	     * @return RAML version of the node
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    OAuth2SecuritySchemeSettingsImpl.prototype.scalarsAnnotations = function () { return new OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl(this.highLevel()); };
	    return OAuth2SecuritySchemeSettingsImpl;
	}(SecuritySchemeSettingsImpl));
	exports.OAuth2SecuritySchemeSettingsImpl = OAuth2SecuritySchemeSettingsImpl;
	/**
	 * OAuth2SecuritySchemeSettings scalar properties annotations accessor
	 **/
	var OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl = (function (_super) {
	    __extends(OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl, _super);
	    function OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * OAuth2SecuritySchemeSettings.accessTokenUri annotations
	     **/
	    OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.accessTokenUri = function () {
	        var attr = this.node.attr("accessTokenUri");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * OAuth2SecuritySchemeSettings.authorizationUri annotations
	     **/
	    OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.authorizationUri = function () {
	        var attr = this.node.attr("authorizationUri");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * OAuth2SecuritySchemeSettings.authorizationGrants annotations
	     **/
	    OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.authorizationGrants = function () {
	        var attrs = this.node.attributes("authorizationGrants");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * OAuth2SecuritySchemeSettings.scopes annotations
	     **/
	    OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl.prototype.scopes = function () {
	        var attrs = this.node.attributes("scopes");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl = OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl;
	/**
	 * Declares globally referable security scheme definition
	 **/
	var OAuth2SecuritySchemeImpl = (function (_super) {
	    __extends(OAuth2SecuritySchemeImpl, _super);
	    function OAuth2SecuritySchemeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createOAuth2SecurityScheme(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    OAuth2SecuritySchemeImpl.prototype.settings = function () {
	        return _super.prototype.element.call(this, 'settings');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    OAuth2SecuritySchemeImpl.prototype.wrapperClassName = function () { return "OAuth2SecuritySchemeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    OAuth2SecuritySchemeImpl.prototype.kind = function () { return "OAuth2SecurityScheme"; };
	    /**
	     * @return RAML version of the node
	     **/
	    OAuth2SecuritySchemeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return OAuth2SecuritySchemeImpl;
	}(AbstractSecuritySchemeImpl));
	exports.OAuth2SecuritySchemeImpl = OAuth2SecuritySchemeImpl;
	/**
	 * Declares globally referable security scheme definition
	 **/
	var OAuth1SecuritySchemeImpl = (function (_super) {
	    __extends(OAuth1SecuritySchemeImpl, _super);
	    function OAuth1SecuritySchemeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createOAuth1SecurityScheme(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    OAuth1SecuritySchemeImpl.prototype.settings = function () {
	        return _super.prototype.element.call(this, 'settings');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    OAuth1SecuritySchemeImpl.prototype.wrapperClassName = function () { return "OAuth1SecuritySchemeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    OAuth1SecuritySchemeImpl.prototype.kind = function () { return "OAuth1SecurityScheme"; };
	    /**
	     * @return RAML version of the node
	     **/
	    OAuth1SecuritySchemeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return OAuth1SecuritySchemeImpl;
	}(AbstractSecuritySchemeImpl));
	exports.OAuth1SecuritySchemeImpl = OAuth1SecuritySchemeImpl;
	/**
	 * Declares globally referable security scheme definition
	 **/
	var PassThroughSecuritySchemeImpl = (function (_super) {
	    __extends(PassThroughSecuritySchemeImpl, _super);
	    function PassThroughSecuritySchemeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createPassThroughSecurityScheme(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    PassThroughSecuritySchemeImpl.prototype.settings = function () {
	        return _super.prototype.element.call(this, 'settings');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    PassThroughSecuritySchemeImpl.prototype.wrapperClassName = function () { return "PassThroughSecuritySchemeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    PassThroughSecuritySchemeImpl.prototype.kind = function () { return "PassThroughSecurityScheme"; };
	    /**
	     * @return RAML version of the node
	     **/
	    PassThroughSecuritySchemeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return PassThroughSecuritySchemeImpl;
	}(AbstractSecuritySchemeImpl));
	exports.PassThroughSecuritySchemeImpl = PassThroughSecuritySchemeImpl;
	/**
	 * Declares globally referable security scheme definition
	 **/
	var BasicSecuritySchemeImpl = (function (_super) {
	    __extends(BasicSecuritySchemeImpl, _super);
	    function BasicSecuritySchemeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createBasicSecurityScheme(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    BasicSecuritySchemeImpl.prototype.wrapperClassName = function () { return "BasicSecuritySchemeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    BasicSecuritySchemeImpl.prototype.kind = function () { return "BasicSecurityScheme"; };
	    /**
	     * @return RAML version of the node
	     **/
	    BasicSecuritySchemeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return BasicSecuritySchemeImpl;
	}(AbstractSecuritySchemeImpl));
	exports.BasicSecuritySchemeImpl = BasicSecuritySchemeImpl;
	/**
	 * Declares globally referable security scheme definition
	 **/
	var DigestSecuritySchemeImpl = (function (_super) {
	    __extends(DigestSecuritySchemeImpl, _super);
	    function DigestSecuritySchemeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createDigestSecurityScheme(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DigestSecuritySchemeImpl.prototype.wrapperClassName = function () { return "DigestSecuritySchemeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DigestSecuritySchemeImpl.prototype.kind = function () { return "DigestSecurityScheme"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DigestSecuritySchemeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return DigestSecuritySchemeImpl;
	}(AbstractSecuritySchemeImpl));
	exports.DigestSecuritySchemeImpl = DigestSecuritySchemeImpl;
	/**
	 * Declares globally referable security scheme definition
	 **/
	var CustomSecuritySchemeImpl = (function (_super) {
	    __extends(CustomSecuritySchemeImpl, _super);
	    function CustomSecuritySchemeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createCustomSecurityScheme(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    CustomSecuritySchemeImpl.prototype.wrapperClassName = function () { return "CustomSecuritySchemeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    CustomSecuritySchemeImpl.prototype.kind = function () { return "CustomSecurityScheme"; };
	    /**
	     * @return RAML version of the node
	     **/
	    CustomSecuritySchemeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return CustomSecuritySchemeImpl;
	}(AbstractSecuritySchemeImpl));
	exports.CustomSecuritySchemeImpl = CustomSecuritySchemeImpl;
	/**
	 * AbstractSecurityScheme scalar properties annotations accessor
	 **/
	var AbstractSecuritySchemeScalarsAnnotationsImpl = (function (_super) {
	    __extends(AbstractSecuritySchemeScalarsAnnotationsImpl, _super);
	    function AbstractSecuritySchemeScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * AbstractSecurityScheme.type annotations
	     **/
	    AbstractSecuritySchemeScalarsAnnotationsImpl.prototype["type"] = function () {
	        var attr = this.node.attr("type");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * AbstractSecurityScheme.description annotations
	     **/
	    AbstractSecuritySchemeScalarsAnnotationsImpl.prototype.description = function () {
	        var attr = this.node.attr("description");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * AbstractSecurityScheme.displayName annotations
	     **/
	    AbstractSecuritySchemeScalarsAnnotationsImpl.prototype.displayName = function () {
	        var attr = this.node.attr("displayName");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return AbstractSecuritySchemeScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.AbstractSecuritySchemeScalarsAnnotationsImpl = AbstractSecuritySchemeScalarsAnnotationsImpl;
	var MethodImpl = (function (_super) {
	    __extends(MethodImpl, _super);
	    function MethodImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createMethod(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Method that can be called
	     **/
	    MethodImpl.prototype.method = function () {
	        return _super.prototype.attribute.call(this, 'method', this.toString);
	    };
	    /**
	     * @hidden
	     * Set method value
	     **/
	    MethodImpl.prototype.setMethod = function (param) {
	        this.highLevel().attrOrCreate("method").setValue("" + param);
	        return this;
	    };
	    /**
	     * The displayName attribute specifies the method display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
	     **/
	    MethodImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    MethodImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    MethodImpl.prototype.wrapperClassName = function () { return "MethodImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    MethodImpl.prototype.kind = function () { return "Method"; };
	    /**
	     * @return RAML version of the node
	     **/
	    MethodImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * For methods of Resources returns parent resource. For methods of ResourceTypes returns null.
	     **/
	    MethodImpl.prototype.parentResource = function () {
	        return helper.parentResource(this);
	    };
	    /**
	     * Api owning the resource as a sibling
	     **/
	    MethodImpl.prototype.ownerApi = function () {
	        return helper.ownerApi(this);
	    };
	    /**
	     * For methods of Resources: `{parent Resource relative path} {methodName}`.
	     * For methods of ResourceTypes: `{parent ResourceType name} {methodName}`.
	     * For other methods throws Exception.
	     **/
	    MethodImpl.prototype.methodId = function () {
	        return helper.methodId(this);
	    };
	    /**
	     * Returns security schemes, resource or method is secured with. If no security schemes are set at resource or method level,
	     * returns schemes defined with `securedBy` at API level.
	     * @deprecated
	     **/
	    MethodImpl.prototype.allSecuredBy = function () {
	        return helper.allSecuredBy(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    MethodImpl.prototype.scalarsAnnotations = function () { return new MethodScalarsAnnotationsImpl(this.highLevel()); };
	    return MethodImpl;
	}(MethodBaseImpl));
	exports.MethodImpl = MethodImpl;
	/**
	 * MethodBase scalar properties annotations accessor
	 **/
	var MethodBaseScalarsAnnotationsImpl = (function (_super) {
	    __extends(MethodBaseScalarsAnnotationsImpl, _super);
	    function MethodBaseScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * MethodBase.protocols annotations
	     **/
	    MethodBaseScalarsAnnotationsImpl.prototype.protocols = function () {
	        var attrs = this.node.attributes("protocols");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * MethodBase.is annotations
	     **/
	    MethodBaseScalarsAnnotationsImpl.prototype.is = function () {
	        var attrs = this.node.attributes("is");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * MethodBase.securedBy annotations
	     **/
	    MethodBaseScalarsAnnotationsImpl.prototype.securedBy = function () {
	        var attrs = this.node.attributes("securedBy");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * MethodBase.description annotations
	     **/
	    MethodBaseScalarsAnnotationsImpl.prototype.description = function () {
	        var attr = this.node.attr("description");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * MethodBase.displayName annotations
	     **/
	    MethodBaseScalarsAnnotationsImpl.prototype.displayName = function () {
	        var attr = this.node.attr("displayName");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return MethodBaseScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.MethodBaseScalarsAnnotationsImpl = MethodBaseScalarsAnnotationsImpl;
	/**
	 * Method scalar properties annotations accessor
	 **/
	var MethodScalarsAnnotationsImpl = (function (_super) {
	    __extends(MethodScalarsAnnotationsImpl, _super);
	    function MethodScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Method.displayName annotations
	     **/
	    MethodScalarsAnnotationsImpl.prototype.displayName = function () {
	        var attr = this.node.attr("displayName");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return MethodScalarsAnnotationsImpl;
	}(MethodBaseScalarsAnnotationsImpl));
	exports.MethodScalarsAnnotationsImpl = MethodScalarsAnnotationsImpl;
	var TraitImpl = (function (_super) {
	    __extends(TraitImpl, _super);
	    function TraitImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createTrait(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Name of the trait
	     **/
	    TraitImpl.prototype.name = function () {
	        return _super.prototype.attribute.call(this, 'name', this.toString);
	    };
	    /**
	     * @hidden
	     * Set name value
	     **/
	    TraitImpl.prototype.setName = function (param) {
	        this.highLevel().attrOrCreate("name").setValue("" + param);
	        return this;
	    };
	    /**
	     * Instructions on how and when the trait should be used.
	     **/
	    TraitImpl.prototype.usage = function () {
	        return _super.prototype.attribute.call(this, 'usage', this.toString);
	    };
	    /**
	     * @hidden
	     * Set usage value
	     **/
	    TraitImpl.prototype.setUsage = function (param) {
	        this.highLevel().attrOrCreate("usage").setValue("" + param);
	        return this;
	    };
	    /**
	     * The displayName attribute specifies the trait display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
	     **/
	    TraitImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    TraitImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    TraitImpl.prototype.uses = function () {
	        return _super.prototype.elements.call(this, 'uses');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    TraitImpl.prototype.wrapperClassName = function () { return "TraitImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    TraitImpl.prototype.kind = function () { return "Trait"; };
	    /**
	     * @return RAML version of the node
	     **/
	    TraitImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    TraitImpl.prototype.parametrizedProperties = function () {
	        return helper.getTemplateParametrizedProperties(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    TraitImpl.prototype.scalarsAnnotations = function () { return new TraitScalarsAnnotationsImpl(this.highLevel()); };
	    return TraitImpl;
	}(MethodBaseImpl));
	exports.TraitImpl = TraitImpl;
	/**
	 * Trait scalar properties annotations accessor
	 **/
	var TraitScalarsAnnotationsImpl = (function (_super) {
	    __extends(TraitScalarsAnnotationsImpl, _super);
	    function TraitScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Trait.usage annotations
	     **/
	    TraitScalarsAnnotationsImpl.prototype.usage = function () {
	        var attr = this.node.attr("usage");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Trait.displayName annotations
	     **/
	    TraitScalarsAnnotationsImpl.prototype.displayName = function () {
	        var attr = this.node.attr("displayName");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return TraitScalarsAnnotationsImpl;
	}(MethodBaseScalarsAnnotationsImpl));
	exports.TraitScalarsAnnotationsImpl = TraitScalarsAnnotationsImpl;
	var ResourceTypeRefImpl = (function (_super) {
	    __extends(ResourceTypeRefImpl, _super);
	    function ResourceTypeRefImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ResourceTypeRefImpl.prototype.wrapperClassName = function () { return "ResourceTypeRefImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ResourceTypeRefImpl.prototype.kind = function () { return "ResourceTypeRef"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ResourceTypeRefImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    ResourceTypeRefImpl.prototype.resourceType = function () {
	        return helper.referencedResourceType(this);
	    };
	    return ResourceTypeRefImpl;
	}(ReferenceImpl));
	exports.ResourceTypeRefImpl = ResourceTypeRefImpl;
	var ResourceBaseImpl = (function (_super) {
	    __extends(ResourceBaseImpl, _super);
	    function ResourceBaseImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createResourceBase(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Methods that are part of this resource type definition
	     **/
	    ResourceBaseImpl.prototype.methods = function () {
	        return _super.prototype.elements.call(this, 'methods');
	    };
	    /**
	     * A list of the traits to apply to all methods declared (implicitly or explicitly) for this resource. Individual methods may override this declaration
	     **/
	    ResourceBaseImpl.prototype.is = function () {
	        return _super.prototype.attributes.call(this, 'is', function (attr) { return new TraitRefImpl(attr); });
	    };
	    /**
	     * The resource type which this resource inherits.
	     **/
	    ResourceBaseImpl.prototype["type"] = function () {
	        return _super.prototype.attribute.call(this, 'type', function (attr) { return new ResourceTypeRefImpl(attr); });
	    };
	    ResourceBaseImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    /**
	     * The security schemes that apply to all methods declared (implicitly or explicitly) for this resource.
	     **/
	    ResourceBaseImpl.prototype.securedBy = function () {
	        return _super.prototype.attributes.call(this, 'securedBy', function (attr) { return new SecuritySchemeRefImpl(attr); });
	    };
	    /**
	     * Detailed information about any URI parameters of this resource
	     * @hidden
	     **/
	    ResourceBaseImpl.prototype.uriParameters_original = function () {
	        return _super.prototype.elements.call(this, 'uriParameters');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ResourceBaseImpl.prototype.wrapperClassName = function () { return "ResourceBaseImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ResourceBaseImpl.prototype.kind = function () { return "ResourceBase"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ResourceBaseImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Detailed information about any URI parameters of this resource
	     **/
	    ResourceBaseImpl.prototype.uriParameters = function () {
	        return helper.uriParametersPrimary(this);
	    };
	    /**
	     * Retrieve an ordered list of all uri parameters including those which are not described in the `uriParameters` node.
	     * Consider a fragment of RAML specification:
	     * ```yaml
	     * /resource/{objectId}/{propertyId}:
	     * uriParameters:
	     * objectId:
	     * ```
	     * Here `propertyId` uri parameter is not described in the `uriParameters` node,
	     * but it is among Resource.allUriParameters().
	     * @deprecated
	     **/
	    ResourceBaseImpl.prototype.allUriParameters = function () {
	        return helper.uriParameters(this);
	    };
	    /**
	     * Returns security schemes, resource or method is secured with. If no security schemes are set at resource or method level,
	     * returns schemes defined with `securedBy` at API level.
	     * @deprecated
	     **/
	    ResourceBaseImpl.prototype.allSecuredBy = function () {
	        return helper.allSecuredBy(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ResourceBaseImpl.prototype.scalarsAnnotations = function () { return new ResourceBaseScalarsAnnotationsImpl(this.highLevel()); };
	    return ResourceBaseImpl;
	}(AnnotableImpl));
	exports.ResourceBaseImpl = ResourceBaseImpl;
	var ResourceImpl = (function (_super) {
	    __extends(ResourceImpl, _super);
	    function ResourceImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createResource(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Relative URL of this resource from the parent resource
	     **/
	    ResourceImpl.prototype.relativeUri = function () {
	        return _super.prototype.attribute.call(this, 'relativeUri', function (attr) { return new RelativeUriStringImpl(attr); });
	    };
	    /**
	     * The displayName attribute specifies the resource display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
	     **/
	    ResourceImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    ResourceImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    /**
	     * A nested resource is identified as any property whose name begins with a slash ("/") and is therefore treated as a relative URI.
	     **/
	    ResourceImpl.prototype.resources = function () {
	        return _super.prototype.elements.call(this, 'resources');
	    };
	    /**
	     * A longer, human-friendly description of the resource.
	     **/
	    ResourceImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    /**
	     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
	     **/
	    ResourceImpl.prototype.annotations = function () {
	        return _super.prototype.attributes.call(this, 'annotations', function (attr) { return new AnnotationRefImpl(attr); });
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ResourceImpl.prototype.wrapperClassName = function () { return "ResourceImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ResourceImpl.prototype.kind = function () { return "Resource"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ResourceImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Path relative to API root
	     **/
	    ResourceImpl.prototype.completeRelativeUri = function () {
	        return helper.completeRelativeUri(this);
	    };
	    /**
	     * baseUri of owning Api concatenated with completeRelativeUri
	     **/
	    ResourceImpl.prototype.absoluteUri = function () {
	        return helper.absoluteUri(this);
	    };
	    /**
	     * Parent resource for non top level resources
	     **/
	    ResourceImpl.prototype.parentResource = function () {
	        return helper.parent(this);
	    };
	    /**
	     * Get child resource by its relative path
	     **/
	    ResourceImpl.prototype.childResource = function (relPath) {
	        return helper.childResource(this, relPath);
	    };
	    /**
	     * Get child method by its name
	     **/
	    ResourceImpl.prototype.childMethod = function (method) {
	        return helper.childMethod(this, method);
	    };
	    /**
	     * Api owning the resource as a sibling
	     **/
	    ResourceImpl.prototype.ownerApi = function () {
	        return helper.ownerApi(this);
	    };
	    /**
	     * Retrieve an ordered list of all absolute uri parameters. Returns a union of `Api.baseUriParameters()`
	     * for `Api` owning the `Resource` and `Resource.uriParameters()`.
	     **/
	    ResourceImpl.prototype.absoluteUriParameters = function () {
	        return helper.absoluteUriParameters(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ResourceImpl.prototype.scalarsAnnotations = function () { return new ResourceScalarsAnnotationsImpl(this.highLevel()); };
	    return ResourceImpl;
	}(ResourceBaseImpl));
	exports.ResourceImpl = ResourceImpl;
	/**
	 * ResourceBase scalar properties annotations accessor
	 **/
	var ResourceBaseScalarsAnnotationsImpl = (function (_super) {
	    __extends(ResourceBaseScalarsAnnotationsImpl, _super);
	    function ResourceBaseScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * ResourceBase.is annotations
	     **/
	    ResourceBaseScalarsAnnotationsImpl.prototype.is = function () {
	        var attrs = this.node.attributes("is");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * ResourceBase.type annotations
	     **/
	    ResourceBaseScalarsAnnotationsImpl.prototype["type"] = function () {
	        var attr = this.node.attr("type");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ResourceBase.description annotations
	     **/
	    ResourceBaseScalarsAnnotationsImpl.prototype.description = function () {
	        var attr = this.node.attr("description");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ResourceBase.securedBy annotations
	     **/
	    ResourceBaseScalarsAnnotationsImpl.prototype.securedBy = function () {
	        var attrs = this.node.attributes("securedBy");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return ResourceBaseScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.ResourceBaseScalarsAnnotationsImpl = ResourceBaseScalarsAnnotationsImpl;
	/**
	 * Resource scalar properties annotations accessor
	 **/
	var ResourceScalarsAnnotationsImpl = (function (_super) {
	    __extends(ResourceScalarsAnnotationsImpl, _super);
	    function ResourceScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Resource.displayName annotations
	     **/
	    ResourceScalarsAnnotationsImpl.prototype.displayName = function () {
	        var attr = this.node.attr("displayName");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Resource.description annotations
	     **/
	    ResourceScalarsAnnotationsImpl.prototype.description = function () {
	        var attr = this.node.attr("description");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Resource.annotations annotations
	     **/
	    ResourceScalarsAnnotationsImpl.prototype.annotations = function () {
	        var attrs = this.node.attributes("annotations");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return ResourceScalarsAnnotationsImpl;
	}(ResourceBaseScalarsAnnotationsImpl));
	exports.ResourceScalarsAnnotationsImpl = ResourceScalarsAnnotationsImpl;
	var ResourceTypeImpl = (function (_super) {
	    __extends(ResourceTypeImpl, _super);
	    function ResourceTypeImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createResourceType(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * The displayName attribute specifies the resource type display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself).
	     **/
	    ResourceTypeImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    ResourceTypeImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    /**
	     * Name of the resource type
	     **/
	    ResourceTypeImpl.prototype.name = function () {
	        return _super.prototype.attribute.call(this, 'name', this.toString);
	    };
	    /**
	     * @hidden
	     * Set name value
	     **/
	    ResourceTypeImpl.prototype.setName = function (param) {
	        this.highLevel().attrOrCreate("name").setValue("" + param);
	        return this;
	    };
	    /**
	     * Instructions on how and when the resource type should be used.
	     **/
	    ResourceTypeImpl.prototype.usage = function () {
	        return _super.prototype.attribute.call(this, 'usage', this.toString);
	    };
	    /**
	     * @hidden
	     * Set usage value
	     **/
	    ResourceTypeImpl.prototype.setUsage = function (param) {
	        this.highLevel().attrOrCreate("usage").setValue("" + param);
	        return this;
	    };
	    ResourceTypeImpl.prototype.uses = function () {
	        return _super.prototype.elements.call(this, 'uses');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ResourceTypeImpl.prototype.wrapperClassName = function () { return "ResourceTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ResourceTypeImpl.prototype.kind = function () { return "ResourceType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ResourceTypeImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    ResourceTypeImpl.prototype.parametrizedProperties = function () {
	        return helper.getTemplateParametrizedProperties(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ResourceTypeImpl.prototype.scalarsAnnotations = function () { return new ResourceTypeScalarsAnnotationsImpl(this.highLevel()); };
	    return ResourceTypeImpl;
	}(ResourceBaseImpl));
	exports.ResourceTypeImpl = ResourceTypeImpl;
	/**
	 * ResourceType scalar properties annotations accessor
	 **/
	var ResourceTypeScalarsAnnotationsImpl = (function (_super) {
	    __extends(ResourceTypeScalarsAnnotationsImpl, _super);
	    function ResourceTypeScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * ResourceType.displayName annotations
	     **/
	    ResourceTypeScalarsAnnotationsImpl.prototype.displayName = function () {
	        var attr = this.node.attr("displayName");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * ResourceType.usage annotations
	     **/
	    ResourceTypeScalarsAnnotationsImpl.prototype.usage = function () {
	        var attr = this.node.attr("usage");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return ResourceTypeScalarsAnnotationsImpl;
	}(ResourceBaseScalarsAnnotationsImpl));
	exports.ResourceTypeScalarsAnnotationsImpl = ResourceTypeScalarsAnnotationsImpl;
	/**
	 * Annotations allow you to attach information to your API
	 **/
	var AnnotationRefImpl = (function (_super) {
	    __extends(AnnotationRefImpl, _super);
	    function AnnotationRefImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    AnnotationRefImpl.prototype.wrapperClassName = function () { return "AnnotationRefImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    AnnotationRefImpl.prototype.kind = function () { return "AnnotationRef"; };
	    /**
	     * @return RAML version of the node
	     **/
	    AnnotationRefImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    AnnotationRefImpl.prototype.annotation = function () {
	        return helper.referencedAnnotation(this);
	    };
	    return AnnotationRefImpl;
	}(ReferenceImpl));
	exports.AnnotationRefImpl = AnnotationRefImpl;
	var DocumentationItemImpl = (function (_super) {
	    __extends(DocumentationItemImpl, _super);
	    function DocumentationItemImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createDocumentationItem(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Title of documentation section
	     **/
	    DocumentationItemImpl.prototype.title = function () {
	        return _super.prototype.attribute.call(this, 'title', this.toString);
	    };
	    /**
	     * @hidden
	     * Set title value
	     **/
	    DocumentationItemImpl.prototype.setTitle = function (param) {
	        this.highLevel().attrOrCreate("title").setValue("" + param);
	        return this;
	    };
	    /**
	     * Content of documentation section
	     **/
	    DocumentationItemImpl.prototype.content = function () {
	        return _super.prototype.attribute.call(this, 'content', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    DocumentationItemImpl.prototype.uses = function () {
	        return _super.prototype.elements.call(this, 'uses');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    DocumentationItemImpl.prototype.wrapperClassName = function () { return "DocumentationItemImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    DocumentationItemImpl.prototype.kind = function () { return "DocumentationItem"; };
	    /**
	     * @return RAML version of the node
	     **/
	    DocumentationItemImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    DocumentationItemImpl.prototype.scalarsAnnotations = function () { return new DocumentationItemScalarsAnnotationsImpl(this.highLevel()); };
	    return DocumentationItemImpl;
	}(AnnotableImpl));
	exports.DocumentationItemImpl = DocumentationItemImpl;
	/**
	 * DocumentationItem scalar properties annotations accessor
	 **/
	var DocumentationItemScalarsAnnotationsImpl = (function (_super) {
	    __extends(DocumentationItemScalarsAnnotationsImpl, _super);
	    function DocumentationItemScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * DocumentationItem.title annotations
	     **/
	    DocumentationItemScalarsAnnotationsImpl.prototype.title = function () {
	        var attr = this.node.attr("title");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * DocumentationItem.content annotations
	     **/
	    DocumentationItemScalarsAnnotationsImpl.prototype.content = function () {
	        var attr = this.node.attr("content");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return DocumentationItemScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.DocumentationItemScalarsAnnotationsImpl = DocumentationItemScalarsAnnotationsImpl;
	var FragmentDeclarationImpl = (function (_super) {
	    __extends(FragmentDeclarationImpl, _super);
	    function FragmentDeclarationImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createFragmentDeclaration(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    FragmentDeclarationImpl.prototype.uses = function () {
	        return _super.prototype.elements.call(this, 'uses');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    FragmentDeclarationImpl.prototype.wrapperClassName = function () { return "FragmentDeclarationImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    FragmentDeclarationImpl.prototype.kind = function () { return "FragmentDeclaration"; };
	    /**
	     * @return RAML version of the node
	     **/
	    FragmentDeclarationImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    return FragmentDeclarationImpl;
	}(AnnotableImpl));
	exports.FragmentDeclarationImpl = FragmentDeclarationImpl;
	var LibraryBaseImpl = (function (_super) {
	    __extends(LibraryBaseImpl, _super);
	    function LibraryBaseImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createLibraryBase(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Alias for the equivalent "types" property, for compatibility with RAML 0.8. Deprecated - API definitions should use the "types" property, as the "schemas" alias for that property name may be removed in a future RAML version. The "types" property allows for XML and JSON schemas.
	     **/
	    LibraryBaseImpl.prototype.schemas = function () {
	        return _super.prototype.elements.call(this, 'schemas');
	    };
	    /**
	     * Declarations of (data) types for use within this API
	     **/
	    LibraryBaseImpl.prototype.types = function () {
	        return _super.prototype.elements.call(this, 'types');
	    };
	    /**
	     * Declarations of traits for use within this API
	     * @hidden
	     **/
	    LibraryBaseImpl.prototype.traits_original = function () {
	        return _super.prototype.elements.call(this, 'traits');
	    };
	    /**
	     * Declarations of resource types for use within this API
	     * @hidden
	     **/
	    LibraryBaseImpl.prototype.resourceTypes_original = function () {
	        return _super.prototype.elements.call(this, 'resourceTypes');
	    };
	    /**
	     * Declarations of annotation types for use by annotations
	     **/
	    LibraryBaseImpl.prototype.annotationTypes = function () {
	        return _super.prototype.elements.call(this, 'annotationTypes');
	    };
	    /**
	     * Declarations of security schemes for use within this API.
	     **/
	    LibraryBaseImpl.prototype.securitySchemes = function () {
	        return _super.prototype.elements.call(this, 'securitySchemes');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    LibraryBaseImpl.prototype.wrapperClassName = function () { return "LibraryBaseImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    LibraryBaseImpl.prototype.kind = function () { return "LibraryBase"; };
	    /**
	     * @return RAML version of the node
	     **/
	    LibraryBaseImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Declarations of traits for use within this API
	     **/
	    LibraryBaseImpl.prototype.traits = function () {
	        return helper.traitsPrimary(this);
	    };
	    /**
	     * Retrieve all traits including those defined in libraries
	     * @deprecated
	     **/
	    LibraryBaseImpl.prototype.allTraits = function () {
	        return helper.allTraits(this);
	    };
	    /**
	     * Declarations of resource types for use within this API
	     **/
	    LibraryBaseImpl.prototype.resourceTypes = function () {
	        return helper.resourceTypesPrimary(this);
	    };
	    /**
	     * Retrieve all resource types including those defined in libraries
	     * @deprecated
	     **/
	    LibraryBaseImpl.prototype.allResourceTypes = function () {
	        return helper.allResourceTypes(this);
	    };
	    return LibraryBaseImpl;
	}(FragmentDeclarationImpl));
	exports.LibraryBaseImpl = LibraryBaseImpl;
	var LibraryImpl = (function (_super) {
	    __extends(LibraryImpl, _super);
	    function LibraryImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createLibrary(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * contains description of why library exist
	     **/
	    LibraryImpl.prototype.usage = function () {
	        return _super.prototype.attribute.call(this, 'usage', this.toString);
	    };
	    /**
	     * @hidden
	     * Set usage value
	     **/
	    LibraryImpl.prototype.setUsage = function (param) {
	        this.highLevel().attrOrCreate("usage").setValue("" + param);
	        return this;
	    };
	    /**
	     * Namespace which the library is imported under
	     **/
	    LibraryImpl.prototype.name = function () {
	        return _super.prototype.attribute.call(this, 'name', this.toString);
	    };
	    /**
	     * @hidden
	     * Set name value
	     **/
	    LibraryImpl.prototype.setName = function (param) {
	        this.highLevel().attrOrCreate("name").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    LibraryImpl.prototype.wrapperClassName = function () { return "LibraryImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    LibraryImpl.prototype.kind = function () { return "Library"; };
	    /**
	     * @return RAML version of the node
	     **/
	    LibraryImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    LibraryImpl.prototype.scalarsAnnotations = function () { return new LibraryScalarsAnnotationsImpl(this.highLevel()); };
	    return LibraryImpl;
	}(LibraryBaseImpl));
	exports.LibraryImpl = LibraryImpl;
	/**
	 * Library scalar properties annotations accessor
	 **/
	var LibraryScalarsAnnotationsImpl = (function (_super) {
	    __extends(LibraryScalarsAnnotationsImpl, _super);
	    function LibraryScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Library.usage annotations
	     **/
	    LibraryScalarsAnnotationsImpl.prototype.usage = function () {
	        var attr = this.node.attr("usage");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return LibraryScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.LibraryScalarsAnnotationsImpl = LibraryScalarsAnnotationsImpl;
	var ApiImpl = (function (_super) {
	    __extends(ApiImpl, _super);
	    function ApiImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createApi(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * Short plain-text label for the API
	     **/
	    ApiImpl.prototype.title = function () {
	        return _super.prototype.attribute.call(this, 'title', this.toString);
	    };
	    /**
	     * @hidden
	     * Set title value
	     **/
	    ApiImpl.prototype.setTitle = function (param) {
	        this.highLevel().attrOrCreate("title").setValue("" + param);
	        return this;
	    };
	    /**
	     * A longer, human-friendly description of the API
	     **/
	    ApiImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    /**
	     * The version of the API, e.g. 'v1'
	     **/
	    ApiImpl.prototype.version = function () {
	        return _super.prototype.attribute.call(this, 'version', this.toString);
	    };
	    /**
	     * @hidden
	     * Set version value
	     **/
	    ApiImpl.prototype.setVersion = function (param) {
	        this.highLevel().attrOrCreate("version").setValue("" + param);
	        return this;
	    };
	    /**
	     * A URI that's to be used as the base of all the resources' URIs. Often used as the base of the URL of each resource, containing the location of the API. Can be a template URI.
	     **/
	    ApiImpl.prototype.baseUri = function () {
	        return _super.prototype.attribute.call(this, 'baseUri', function (attr) { return new FullUriTemplateStringImpl(attr); });
	    };
	    /**
	     * Named parameters used in the baseUri (template)
	     * @hidden
	     **/
	    ApiImpl.prototype.baseUriParameters_original = function () {
	        return _super.prototype.elements.call(this, 'baseUriParameters');
	    };
	    /**
	     * The protocols supported by the API
	     **/
	    ApiImpl.prototype.protocols = function () {
	        return _super.prototype.attributes.call(this, 'protocols', this.toString);
	    };
	    /**
	     * @hidden
	     * Set protocols value
	     **/
	    ApiImpl.prototype.setProtocols = function (param) {
	        this.highLevel().attrOrCreate("protocols").setValue("" + param);
	        return this;
	    };
	    /**
	     * The default media type to use for request and response bodies (payloads), e.g. "application/json"
	     **/
	    ApiImpl.prototype.mediaType = function () {
	        return _super.prototype.attributes.call(this, 'mediaType', function (attr) { return new MimeTypeImpl(attr); });
	    };
	    /**
	     * The security schemes that apply to every resource and method in the API
	     **/
	    ApiImpl.prototype.securedBy = function () {
	        return _super.prototype.attributes.call(this, 'securedBy', function (attr) { return new SecuritySchemeRefImpl(attr); });
	    };
	    /**
	     * The resources of the API, identified as relative URIs that begin with a slash (/). Every property whose key begins with a slash (/), and is either at the root of the API definition or is the child property of a resource property, is a resource property, e.g.: /users, /{groupId}, etc
	     **/
	    ApiImpl.prototype.resources = function () {
	        return _super.prototype.elements.call(this, 'resources');
	    };
	    /**
	     * Additional overall documentation for the API
	     **/
	    ApiImpl.prototype.documentation = function () {
	        return _super.prototype.elements.call(this, 'documentation');
	    };
	    /**
	     * Most of RAML model elements may have attached annotations decribing additional meta data about this element
	     **/
	    ApiImpl.prototype.annotations = function () {
	        return _super.prototype.attributes.call(this, 'annotations', function (attr) { return new AnnotationRefImpl(attr); });
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ApiImpl.prototype.wrapperClassName = function () { return "ApiImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ApiImpl.prototype.kind = function () { return "Api"; };
	    /**
	     * @return RAML version of the node
	     * @hidden
	     **/
	    ApiImpl.prototype.RAMLVersion_original = function () { return "RAML10"; };
	    /**
	     * Equivalent API with traits and resource types expanded
	     **/
	    ApiImpl.prototype.expand = function () {
	        return helper.expandTraitsAndResourceTypes(this);
	    };
	    /**
	     * Get child resource by its relative path
	     **/
	    ApiImpl.prototype.childResource = function (relPath) {
	        return helper.childResource(this, relPath);
	    };
	    /**
	     * Retrieve all resources of the Api
	     **/
	    ApiImpl.prototype.allResources = function () {
	        return helper.allResources(this);
	    };
	    /**
	     * Named parameters used in the baseUri (template)
	     **/
	    ApiImpl.prototype.baseUriParameters = function () {
	        return helper.baseUriParametersPrimary(this);
	    };
	    /**
	     * Retrieve an ordered list of all base uri parameters regardless of whether they are described in `baseUriParameters` or not
	     * Consider a fragment of RAML specification:
	     * ```yaml
	     * version: v1
	     * baseUri: https://{organization}.example.com/{version}/{service}
	     * baseUriParameters:
	     * service:
	     * ```
	     * Here `version` and `organization` are base uri parameters which are not described in the `baseUriParameters` node,
	     * but they are among `Api.allBaseUriParameters()`.
	     * @deprecated
	     **/
	    ApiImpl.prototype.allBaseUriParameters = function () {
	        return helper.baseUriParameters(this);
	    };
	    /**
	     * Protocols used by the API. Returns the `protocols` property value if it is specified.
	     * Otherwise, returns protocol, specified in the base URI.
	     * @deprecated
	     **/
	    ApiImpl.prototype.allProtocols = function () {
	        return helper.allProtocols(this);
	    };
	    /**
	     * @return RAML version of the node
	     **/
	    ApiImpl.prototype.RAMLVersion = function () {
	        return helper.RAMLVersion(this);
	    };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ApiImpl.prototype.scalarsAnnotations = function () { return new ApiScalarsAnnotationsImpl(this.highLevel()); };
	    return ApiImpl;
	}(LibraryBaseImpl));
	exports.ApiImpl = ApiImpl;
	var OverlayImpl = (function (_super) {
	    __extends(OverlayImpl, _super);
	    function OverlayImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createOverlay(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * contains description of why overlay exist
	     **/
	    OverlayImpl.prototype.usage = function () {
	        return _super.prototype.attribute.call(this, 'usage', this.toString);
	    };
	    /**
	     * @hidden
	     * Set usage value
	     **/
	    OverlayImpl.prototype.setUsage = function (param) {
	        this.highLevel().attrOrCreate("usage").setValue("" + param);
	        return this;
	    };
	    /**
	     * Location of a valid RAML API definition (or overlay or extension), the overlay is applied to.
	     **/
	    OverlayImpl.prototype.extends = function () {
	        return _super.prototype.attribute.call(this, 'extends', this.toString);
	    };
	    /**
	     * @hidden
	     * Set extends value
	     **/
	    OverlayImpl.prototype.setExtends = function (param) {
	        this.highLevel().attrOrCreate("extends").setValue("" + param);
	        return this;
	    };
	    /**
	     * Short plain-text label for the API
	     **/
	    OverlayImpl.prototype.title = function () {
	        return _super.prototype.attribute.call(this, 'title', this.toString);
	    };
	    /**
	     * @hidden
	     * Set title value
	     **/
	    OverlayImpl.prototype.setTitle = function (param) {
	        this.highLevel().attrOrCreate("title").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    OverlayImpl.prototype.wrapperClassName = function () { return "OverlayImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    OverlayImpl.prototype.kind = function () { return "Overlay"; };
	    /**
	     * @return RAML version of the node
	     **/
	    OverlayImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    OverlayImpl.prototype.scalarsAnnotations = function () { return new OverlayScalarsAnnotationsImpl(this.highLevel()); };
	    return OverlayImpl;
	}(ApiImpl));
	exports.OverlayImpl = OverlayImpl;
	/**
	 * Api scalar properties annotations accessor
	 **/
	var ApiScalarsAnnotationsImpl = (function (_super) {
	    __extends(ApiScalarsAnnotationsImpl, _super);
	    function ApiScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Api.title annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.title = function () {
	        var attr = this.node.attr("title");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Api.description annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.description = function () {
	        var attr = this.node.attr("description");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Api.version annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.version = function () {
	        var attr = this.node.attr("version");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Api.baseUri annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.baseUri = function () {
	        var attr = this.node.attr("baseUri");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Api.protocols annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.protocols = function () {
	        var attrs = this.node.attributes("protocols");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * Api.mediaType annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.mediaType = function () {
	        var attrs = this.node.attributes("mediaType");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * Api.securedBy annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.securedBy = function () {
	        var attrs = this.node.attributes("securedBy");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    /**
	     * Api.annotations annotations
	     **/
	    ApiScalarsAnnotationsImpl.prototype.annotations = function () {
	        var attrs = this.node.attributes("annotations");
	        return attrs.map(function (x) {
	            var annotationAttrs = x.annotations();
	            var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	            return result;
	        });
	    };
	    return ApiScalarsAnnotationsImpl;
	}(AnnotableScalarsAnnotationsImpl));
	exports.ApiScalarsAnnotationsImpl = ApiScalarsAnnotationsImpl;
	/**
	 * Overlay scalar properties annotations accessor
	 **/
	var OverlayScalarsAnnotationsImpl = (function (_super) {
	    __extends(OverlayScalarsAnnotationsImpl, _super);
	    function OverlayScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Overlay.usage annotations
	     **/
	    OverlayScalarsAnnotationsImpl.prototype.usage = function () {
	        var attr = this.node.attr("usage");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Overlay.extends annotations
	     **/
	    OverlayScalarsAnnotationsImpl.prototype.extends = function () {
	        var attr = this.node.attr("extends");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Overlay.title annotations
	     **/
	    OverlayScalarsAnnotationsImpl.prototype.title = function () {
	        var attr = this.node.attr("title");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return OverlayScalarsAnnotationsImpl;
	}(ApiScalarsAnnotationsImpl));
	exports.OverlayScalarsAnnotationsImpl = OverlayScalarsAnnotationsImpl;
	var ExtensionImpl = (function (_super) {
	    __extends(ExtensionImpl, _super);
	    function ExtensionImpl(nodeOrKey, setAsTopLevel) {
	        _super.call(this, (typeof nodeOrKey == "string") ? createExtension(nodeOrKey) : nodeOrKey, setAsTopLevel);
	        this.nodeOrKey = nodeOrKey;
	        this.setAsTopLevel = setAsTopLevel;
	    }
	    /**
	     * contains description of why extension exist
	     **/
	    ExtensionImpl.prototype.usage = function () {
	        return _super.prototype.attribute.call(this, 'usage', this.toString);
	    };
	    /**
	     * @hidden
	     * Set usage value
	     **/
	    ExtensionImpl.prototype.setUsage = function (param) {
	        this.highLevel().attrOrCreate("usage").setValue("" + param);
	        return this;
	    };
	    /**
	     * Location of a valid RAML API definition (or overlay or extension), the extension is applied to
	     **/
	    ExtensionImpl.prototype.extends = function () {
	        return _super.prototype.attribute.call(this, 'extends', this.toString);
	    };
	    /**
	     * @hidden
	     * Set extends value
	     **/
	    ExtensionImpl.prototype.setExtends = function (param) {
	        this.highLevel().attrOrCreate("extends").setValue("" + param);
	        return this;
	    };
	    /**
	     * Short plain-text label for the API
	     **/
	    ExtensionImpl.prototype.title = function () {
	        return _super.prototype.attribute.call(this, 'title', this.toString);
	    };
	    /**
	     * @hidden
	     * Set title value
	     **/
	    ExtensionImpl.prototype.setTitle = function (param) {
	        this.highLevel().attrOrCreate("title").setValue("" + param);
	        return this;
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ExtensionImpl.prototype.wrapperClassName = function () { return "ExtensionImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ExtensionImpl.prototype.kind = function () { return "Extension"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ExtensionImpl.prototype.RAMLVersion = function () { return "RAML10"; };
	    /**
	     * Scalar properties annotations accessor
	     **/
	    ExtensionImpl.prototype.scalarsAnnotations = function () { return new ExtensionScalarsAnnotationsImpl(this.highLevel()); };
	    return ExtensionImpl;
	}(ApiImpl));
	exports.ExtensionImpl = ExtensionImpl;
	/**
	 * Extension scalar properties annotations accessor
	 **/
	var ExtensionScalarsAnnotationsImpl = (function (_super) {
	    __extends(ExtensionScalarsAnnotationsImpl, _super);
	    function ExtensionScalarsAnnotationsImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Extension.usage annotations
	     **/
	    ExtensionScalarsAnnotationsImpl.prototype.usage = function () {
	        var attr = this.node.attr("usage");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Extension.extends annotations
	     **/
	    ExtensionScalarsAnnotationsImpl.prototype.extends = function () {
	        var attr = this.node.attr("extends");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    /**
	     * Extension.title annotations
	     **/
	    ExtensionScalarsAnnotationsImpl.prototype.title = function () {
	        var attr = this.node.attr("title");
	        if (attr == null) {
	            return [];
	        }
	        var annotationAttrs = attr.annotations();
	        var result = core.attributesToValues(annotationAttrs, function (a) { return new AnnotationRefImpl(a); });
	        return result;
	    };
	    return ExtensionScalarsAnnotationsImpl;
	}(ApiScalarsAnnotationsImpl));
	exports.ExtensionScalarsAnnotationsImpl = ExtensionScalarsAnnotationsImpl;
	/**
	 * @hidden
	 **/
	function createApi(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Api");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createLibraryBase(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("LibraryBase");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createFragmentDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("FragmentDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createAnnotable(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Annotable");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createTypeInstance(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("TypeInstance");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createTypeInstanceProperty(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("TypeInstanceProperty");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createTrait(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Trait");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createMethodBase(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("MethodBase");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createOperation(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Operation");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("TypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createExampleSpec(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("ExampleSpec");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createXMLFacetInfo(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("XMLFacetInfo");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createUsesDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("UsesDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createArrayTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("ArrayTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createUnionTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("UnionTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createObjectTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("ObjectTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createStringTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("StringTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createBooleanTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("BooleanTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createNumberTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("NumberTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createIntegerTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("IntegerTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createDateOnlyTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("DateOnlyTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createTimeOnlyTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("TimeOnlyTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createDateTimeOnlyTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("DateTimeOnlyTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createDateTimeTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("DateTimeTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createDateTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("DateTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createFileTypeDeclaration(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("FileTypeDeclaration");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createResponse(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Response");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createSecuritySchemePart(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("SecuritySchemePart");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createAbstractSecurityScheme(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("AbstractSecurityScheme");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createSecuritySchemeSettings(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("SecuritySchemeSettings");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createOAuth1SecuritySchemeSettings(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("OAuth1SecuritySchemeSettings");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createOAuth2SecuritySchemeSettings(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("OAuth2SecuritySchemeSettings");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createOAuth2SecurityScheme(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("OAuth2SecurityScheme");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createOAuth1SecurityScheme(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("OAuth1SecurityScheme");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createPassThroughSecurityScheme(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("PassThroughSecurityScheme");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createBasicSecurityScheme(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("BasicSecurityScheme");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createDigestSecurityScheme(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("DigestSecurityScheme");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createCustomSecurityScheme(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("CustomSecurityScheme");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createMethod(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Method");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createResourceType(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("ResourceType");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createResourceBase(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("ResourceBase");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createResource(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Resource");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createDocumentationItem(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("DocumentationItem");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createLibrary(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Library");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createOverlay(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Overlay");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	/**
	 * @hidden
	 **/
	function createExtension(key) {
	    var universe = def.getUniverse("RAML10");
	    var nc = universe.type("Extension");
	    var node = stubs.createStubNode(nc, null, key);
	    return node;
	}
	function loadApiSync(apiPath, arg1, arg2) {
	    return apiLoader.loadApi(apiPath, arg1, arg2).getOrElse(null);
	}
	exports.loadApiSync = loadApiSync;
	function loadRAMLSync(ramlPath, arg1, arg2) {
	    return apiLoader.loadApi(ramlPath, arg1, arg2).getOrElse(null);
	}
	exports.loadRAMLSync = loadRAMLSync;
	function loadApi(apiPath, arg1, arg2) {
	    return apiLoader.loadApiAsync(apiPath, arg1, arg2);
	}
	exports.loadApi = loadApi;
	function loadRAML(ramlPath, arg1, arg2) {
	    return apiLoader.loadRAMLAsync(ramlPath, arg1, arg2);
	}
	exports.loadRAML = loadRAML;
	/**
	 * Gets AST node by runtime type, if runtime type matches any.
	 * @param runtimeType - runtime type to find the match for
	 */
	function getLanguageElementByRuntimeType(runtimeType) {
	    return apiLoader.getLanguageElementByRuntimeType(runtimeType);
	}
	exports.getLanguageElementByRuntimeType = getLanguageElementByRuntimeType;
	//# sourceMappingURL=raml10parser.js.map

/***/ },
/* 21 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../typings/main.d.ts" />
	"use strict";
	var invariant = __webpack_require__(62);
	var exists = function (v) { return (v != null); };
	var globalEmptyOpt;
	var Opt = (function () {
	    function Opt(_value) {
	        var _this = this;
	        this._value = _value;
	        this._isDefined = false;
	        this._isEmpty = true;
	        this._arr = undefined;
	        this.getOrThrow = function () {
	            invariant(_this._isDefined, 'Opt is empty. Use `Opt#getOrElse` or check `Opt#isDefined`.');
	            return _this._value;
	        };
	        this.value = function () { return _this._value; };
	        this.isDefined = function () { return _this._isDefined; };
	        this.isEmpty = function () { return _this._isEmpty; };
	        this.toArray = function () { return _this._arr || (_this._arr = _this._isDefined ? [_this._value] : []); };
	        this.getOrElse = function (v) { return _this._isDefined ? _this._value : v; };
	        this.getOrElseF = function (v) { return _this._isDefined ? _this._value : v(); };
	        this.map = function (f) { return _this._isEmpty ? Opt.empty() : new Opt(f(_this._value)); };
	        this.flatMap = function (f) { return _this.map(f).getOrElse(Opt.empty()); };
	        this.equals = function (other) {
	            invariant(other instanceof Opt, 'Expected other to be an `Opt`, but got `%s`', typeof other);
	            return (_this._isDefined === other.isDefined()) && (_this._value === other.value());
	        };
	        if (exists(this._value)) {
	            this._isDefined = true;
	            this._isEmpty = false;
	        }
	    }
	    Opt.prototype.forEach = function (fn) {
	        if (this.isDefined()) {
	            fn(this._value);
	        }
	    };
	    /**
	     * You can always create an empty option by hand just by calling new Opt<T>()
	     * but calling this method will return a global instance instead of allocating a new one each time
	     */
	    Opt.empty = function () { return (globalEmptyOpt || (globalEmptyOpt = new Opt())); };
	    return Opt;
	}());
	module.exports = Opt;
	//# sourceMappingURL=Opt.js.map

/***/ },
/* 22 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var ramlDs = __webpack_require__(35);
	module.exports = ramlDs.universesInfo;
	//# sourceMappingURL=universe.js.map

/***/ },
/* 23 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	/// <reference path="../../../typings/main.d.ts" />
	var lowlevel = __webpack_require__(11);
	var path = __webpack_require__(15);
	var fs = __webpack_require__(36);
	var yaml = __webpack_require__(72);
	var _ = __webpack_require__(61);
	var textutil = __webpack_require__(40);
	var hli = __webpack_require__(16);
	var services = __webpack_require__(33);
	var rr = __webpack_require__(24);
	var util = __webpack_require__(43);
	var URL = __webpack_require__(59);
	var refResolvers = __webpack_require__(25);
	var schemes = __webpack_require__(44);
	var universes = __webpack_require__(22);
	var Error = yaml.YAMLException;
	exports.Kind = {
	    SCALAR: yaml.Kind.SCALAR
	};
	var MarkupIndentingBuffer = (function () {
	    function MarkupIndentingBuffer(indent) {
	        this.text = '';
	        this.indent = indent;
	    }
	    MarkupIndentingBuffer.prototype.isLastNL = function () {
	        return this.text.length > 0 && this.text[this.text.length - 1] == '\n';
	    };
	    MarkupIndentingBuffer.prototype.addWithIndent = function (lev, s) {
	        if (this.isLastNL()) {
	            this.text += textutil.indent(lev);
	            this.text += this.indent;
	        }
	        this.text += s;
	    };
	    MarkupIndentingBuffer.prototype.addChar = function (ch) {
	        if (this.isLastNL()) {
	            this.text += this.indent;
	        }
	        this.text += ch;
	    };
	    MarkupIndentingBuffer.prototype.append = function (s) {
	        for (var i = 0; i < s.length; i++) {
	            this.addChar(s[i]);
	        }
	    };
	    return MarkupIndentingBuffer;
	}());
	exports.MarkupIndentingBuffer = MarkupIndentingBuffer;
	var CompilationUnit = (function () {
	    function CompilationUnit(_path, _content, _tl, _project, _apath) {
	        this._path = _path;
	        this._content = _content;
	        this._tl = _tl;
	        this._project = _project;
	        this._apath = _apath;
	    }
	    CompilationUnit.prototype.highLevel = function () {
	        if (this._hl) {
	            return this._hl;
	        }
	        this._hl = hli.fromUnit(this);
	        return this._hl;
	    };
	    CompilationUnit.prototype.isStubUnit = function () {
	        return this.stu;
	    };
	    CompilationUnit.prototype.resolveAsync = function (p) {
	        var unit = this._project.resolveAsync(this._path, p);
	        return unit;
	    };
	    CompilationUnit.prototype.getIncludeNodes = function () {
	        var ast = this.ast();
	        var arr = [];
	        ast.gatherIncludes(arr);
	        ast.children().forEach(function (x) {
	            if (x.key() == "uses") {
	                x.children().forEach(function (y) {
	                    arr.push({
	                        includePath: function () { return y.value(); }
	                    });
	                });
	            }
	        });
	        return arr;
	    };
	    CompilationUnit.prototype.cloneToProject = function (p) {
	        var newUnit = new CompilationUnit(this._path, this._content, this._tl, p, this._apath);
	        return newUnit;
	    };
	    CompilationUnit.prototype.clone = function () {
	        var newUnit = new CompilationUnit(this._path, this._content, this._tl, this.project(), this._apath);
	        return newUnit;
	    };
	    CompilationUnit.prototype.stub = function () {
	        var newUnit = new CompilationUnit(this._path, this._content, this._tl, this.project(), this._apath);
	        newUnit.stu = true;
	        return newUnit;
	    };
	    CompilationUnit.prototype.isDirty = function () {
	        return false;
	    };
	    CompilationUnit.prototype.absolutePath = function () {
	        return this._apath;
	    };
	    CompilationUnit.prototype.isRAMLUnit = function () {
	        var en = path.extname(this._path);
	        return en == '.raml' || en == '.yaml' || en == '.yml';
	    };
	    CompilationUnit.prototype.contents = function () {
	        return this._content;
	    };
	    CompilationUnit.prototype.resolve = function (p) {
	        if (typeof p != "string") {
	            p = "" + p;
	        }
	        var unit = this._project.resolve(this._path, p);
	        return unit;
	    };
	    CompilationUnit.prototype.path = function () {
	        return this._path;
	    };
	    CompilationUnit.prototype.lexerErrors = function () {
	        if (this.errors == null) {
	            this.ast();
	        }
	        return this.errors;
	    };
	    CompilationUnit.prototype.ast = function () {
	        var _this = this;
	        if (this._node) {
	            return this._node;
	        }
	        try {
	            var result = yaml.load(this._content, {});
	            this.errors = result.errors;
	            this.errors.forEach(function (x) {
	                if (x.mark) {
	                    x.mark.filePath = _this.absolutePath();
	                }
	            });
	            this._node = new ASTNode(result, this, null, null, null);
	            this._node._errors = this.errors;
	            return this._node;
	        }
	        catch (e) {
	            this.errors = [];
	            this.errors.push(new Error(e.message));
	            //console.log(this._content)
	            //console.log(e)
	            this._node = null;
	            return this._node;
	        }
	    };
	    CompilationUnit.prototype.isTopLevel = function () {
	        return this._tl;
	    };
	    CompilationUnit.prototype.updateContent = function (n) {
	        this._content = n;
	        this.errors = null;
	        this._node = null; //todo incremental update
	        this._lineMapper = null;
	    };
	    CompilationUnit.prototype.updateContentSafe = function (n) {
	        this._content = n;
	        this._lineMapper = null;
	        this._hl = null;
	    };
	    CompilationUnit.prototype.project = function () {
	        return this._project;
	    };
	    CompilationUnit.prototype.lineMapper = function () {
	        if (this._lineMapper == null) {
	            this._lineMapper = new lowlevel.LineMapperImpl(this.contents(), this.absolutePath());
	        }
	        return this._lineMapper;
	    };
	    //ramlVersion():string{
	    //    return this._ramlVersion;
	    //}
	    /**
	     * Returns true if this unit is overlay or extension, false otherwise.
	     */
	    CompilationUnit.prototype.isOverlayOrExtension = function () {
	        var contents = this.contents();
	        var spec = "";
	        var ptype = null;
	        var num = 0;
	        var pt = 0;
	        for (var n = 0; n < contents.length; n++) {
	            var c = contents.charAt(n);
	            if (c == '\r' || c == '\n') {
	                if (spec) {
	                    ptype = contents.substring(pt, n).trim();
	                }
	                else {
	                    spec = contents.substring(0, n).trim();
	                }
	                break;
	            }
	            if (c == ' ') {
	                num++;
	                if (!spec && num == 2) {
	                    spec = contents.substring(0, n);
	                    pt = n;
	                }
	            }
	        }
	        return ptype == "Extension" || ptype == "Overlay";
	    };
	    /**
	     * Returns master reference if presents, null otherwise.
	     */
	    CompilationUnit.prototype.getMasterReferenceNode = function () {
	        return _.find(this.ast().children(), function (x) { return x.key() == universes.Universe10.Overlay.properties.extends.name; });
	    };
	    return CompilationUnit;
	}());
	exports.CompilationUnit = CompilationUnit;
	var XhrSync = __webpack_require__(55).XMLHttpRequest;
	var XhrAsync = __webpack_require__(55).XMLHttpRequest;
	var XMLHttpRequestWrapper = (function () {
	    function XMLHttpRequestWrapper() {
	    }
	    XMLHttpRequestWrapper.prototype.open = function (method, url, async) {
	        this.xhr = async ? new XhrAsync() : new XhrSync();
	        this.async = async;
	        this.xhr.open(method, url, async);
	    };
	    XMLHttpRequestWrapper.prototype.setRequestHeader = function (name, value) {
	        this.xhr.setRequestHeader(name, value);
	    };
	    XMLHttpRequestWrapper.prototype.getResponseHeader = function (name) {
	        return this.xhr.getResponseHeader(name);
	    };
	    XMLHttpRequestWrapper.prototype.getAllResponseHeaders = function () {
	        return this.xhr.getAllResponseHeaders();
	    };
	    XMLHttpRequestWrapper.prototype.send = function (content) {
	        var _this = this;
	        this.xhr.onload = function () { return _this.onResponse(false); };
	        this.xhr.onerror = function () { return _this.onResponse(true); };
	        this.xhr.send(content);
	    };
	    XMLHttpRequestWrapper.prototype.onResponse = function (isError) {
	        this.statusText = this.xhr.statusText;
	        this.responseText = this.xhr.responseText;
	        this.responseType = this.xhr.responseType;
	        this.status = this.xhr.status;
	        if (isError && this.onerror) {
	            this.onerror();
	            return;
	        }
	        if (this.onload) {
	            this.onload();
	        }
	    };
	    return XMLHttpRequestWrapper;
	}());
	exports.XMLHttpRequestWrapper = XMLHttpRequestWrapper;
	function buildXHR() {
	    return new XMLHttpRequestWrapper();
	}
	exports.buildXHR = buildXHR;
	var SimpleExecutor = (function () {
	    function SimpleExecutor() {
	    }
	    SimpleExecutor.prototype.execute = function (req, doAppendParams) {
	        if (doAppendParams === void 0) { doAppendParams = true; }
	        var xhr = buildXHR();
	        var url = req.url;
	        if (doAppendParams) {
	            url = this.appendParams(req, req.url);
	        }
	        xhr.open(req.method, url, false);
	        this.doRequest(req, xhr);
	        //rheaders=xhr.getAllResponseHeaders();
	        var status = xhr.status;
	        if (status > 300 && status < 400) {
	            var locHeader = xhr.getResponseHeader('location');
	            if (locHeader) {
	                req.url = locHeader;
	                return this.execute(req, false);
	            }
	        }
	        var response = {
	            status: status,
	            statusText: xhr.statusText,
	            headers: xhr.getAllResponseHeaders().split('\n').map(function (x) {
	                var ind = x.indexOf(':');
	                return {
	                    name: x.substring(0, ind).trim(),
	                    value: x.substring(ind + 1).trim()
	                };
	            }),
	            content: {
	                text: xhr.responseText,
	                mimeType: xhr.responseType
	            }
	        };
	        return response;
	    };
	    SimpleExecutor.prototype.appendParams = function (req, url) {
	        var gotQueryParams = (req.queryString && req.queryString.length > 0);
	        if (gotQueryParams) {
	            url = url + '?';
	            var arr = [];
	            if (gotQueryParams) {
	                arr = arr.concat(req.queryString.map(function (q) {
	                    return encodeURIComponent(q.name) + '=' + encodeURIComponent(q.value);
	                }));
	            }
	            url += arr.join('&');
	        }
	        return url;
	    };
	    SimpleExecutor.prototype.log = function (varName, value) {
	    };
	    SimpleExecutor.prototype.executeAsync = function (req, doAppendParams) {
	        if (doAppendParams === void 0) { doAppendParams = true; }
	        var xhr = buildXHR();
	        var url = req.url;
	        if (doAppendParams) {
	            url = this.appendParams(req, req.url);
	        }
	        var outer = this;
	        return new Promise(function (resolve, reject) {
	            xhr.open(req.method, url, true);
	            xhr.onload = function () {
	                var status = xhr.status;
	                var response = {
	                    status: status,
	                    statusText: xhr.statusText,
	                    headers: xhr.getAllResponseHeaders().split('\n').map(function (x) {
	                        var ind = x.indexOf(':');
	                        return {
	                            name: x.substring(0, ind).trim(),
	                            value: x.substring(ind + 1).trim()
	                        };
	                    }),
	                    content: {
	                        text: xhr.responseText,
	                        mimeType: xhr.responseType
	                    }
	                };
	                resolve(response);
	            };
	            xhr.onerror = function () {
	                reject(new Error("Network Error"));
	            };
	            outer.doRequest(req, xhr);
	        });
	    };
	    SimpleExecutor.prototype.doRequest = function (req, xhr) {
	        if (req.headers) {
	            req.headers.forEach(function (x) { return xhr.setRequestHeader(x.name, x.value); });
	        }
	        if (req.postData) {
	            if (req.postData.params) {
	                var body = req.postData.params.map(function (p) { return encodeURIComponent(p.name) + '=' + encodeURIComponent(p.value); }).join('&');
	                xhr.send(body);
	            }
	            else {
	                xhr.send(req.postData.text);
	            }
	        }
	        else {
	            xhr.send();
	        }
	    };
	    return SimpleExecutor;
	}());
	exports.SimpleExecutor = SimpleExecutor;
	var HTTPResolverImpl = (function () {
	    function HTTPResolverImpl() {
	        this.executor = new SimpleExecutor();
	    }
	    HTTPResolverImpl.prototype.getResource = function (url) {
	        if (typeof atom !== 'undefined' && atom != null) {
	            var cached = rr.get(url);
	            if (cached) {
	                return cached;
	            }
	            rr.addNotify(url);
	            try {
	                var resultWithCallback = {
	                    content: ""
	                };
	                this.getResourceAsync(url).then(function (x) {
	                    try {
	                        if (x.errorMessage) {
	                            rr.set(url, { content: null, errorMessage: null });
	                        }
	                        else {
	                            x.errorMessage = null;
	                            rr.set(url, x);
	                        }
	                    }
	                    finally {
	                        if (resultWithCallback.callback) {
	                            resultWithCallback.callback(rr.get(url));
	                        }
	                        rr.removeNotity(url);
	                    }
	                }, function (e) {
	                    rr.set(url, { content: null, errorMessage: null });
	                    if (resultWithCallback.callback) {
	                        resultWithCallback.callback(rr.get(url));
	                    }
	                    rr.removeNotity(url);
	                });
	                return resultWithCallback;
	            }
	            catch (e) {
	                console.log("Error");
	                console.log(e);
	            }
	        }
	        var response = this.executor.execute({
	            method: 'get',
	            url: url
	        });
	        if (!response) {
	            throw new Error("Unable to execute GET " + url);
	        }
	        var result = this.toResponse(response, url);
	        return result;
	    };
	    HTTPResolverImpl.prototype.getResourceAsync = function (url) {
	        var _this = this;
	        return this.executor.executeAsync({
	            method: 'get',
	            url: url
	        }).then(function (x) {
	            if (!x) {
	                return Promise.reject(new Error("Unable to execute GET " + url));
	            }
	            var result = _this.toResponse(x, url);
	            return result;
	        }, function (x) {
	            return Promise.reject(new Error("Unable to execute GET " + url));
	        });
	    };
	    HTTPResolverImpl.prototype.toResponse = function (response, url) {
	        var msg = null;
	        if (response.status >= 400) {
	            msg = "GET " + url + "\nreturned error: " + response.status;
	            if (response.statusText) {
	                msg += " " + response.statusText;
	            }
	        }
	        var content = null;
	        if (response.content && response.content.text) {
	            content = response.content.text;
	        }
	        var result = {
	            content: content,
	            errorMessage: msg
	        };
	        return result;
	    };
	    return HTTPResolverImpl;
	}());
	exports.HTTPResolverImpl = HTTPResolverImpl;
	var FSResolverImpl = (function () {
	    function FSResolverImpl() {
	    }
	    FSResolverImpl.prototype.content = function (path) {
	        if (typeof path != "string") {
	            path = "" + path;
	        }
	        if (!fs.existsSync(path)) {
	            return null;
	        }
	        try {
	            return fs.readFileSync(path).toString();
	        }
	        catch (e) {
	            return null;
	        }
	    };
	    FSResolverImpl.prototype.list = function (path) {
	        return fs.readdirSync(path);
	    };
	    FSResolverImpl.prototype.contentAsync = function (path) {
	        return new Promise(function (resolve, reject) {
	            fs.readFile(path, function (err, data) {
	                if (err != null) {
	                    return reject(err);
	                }
	                var content = data.toString();
	                resolve(content);
	            });
	        });
	    };
	    FSResolverImpl.prototype.listAsync = function (path) {
	        return new Promise(function (reject, resolve) {
	            fs.readdir(path, function (err, files) {
	                if (err != null) {
	                    return reject(err);
	                }
	                resolve(files);
	            });
	        });
	    };
	    return FSResolverImpl;
	}());
	exports.FSResolverImpl = FSResolverImpl;
	function copyNode(n) {
	    if (n == null) {
	        return null;
	    }
	    switch (n.kind) {
	        case yaml.Kind.SCALAR:
	            return {
	                errors: [],
	                startPosition: n.startPosition,
	                endPosition: n.endPosition,
	                value: n.value,
	                kind: yaml.Kind.SCALAR,
	                parent: n.parent
	            };
	        case yaml.Kind.MAPPING:
	            var map = n;
	            return {
	                errors: [],
	                key: copyNode(map.key),
	                value: copyNode(map.value),
	                startPosition: map.startPosition,
	                endPosition: map.endPosition,
	                kind: yaml.Kind.MAPPING,
	                parent: map.parent
	            };
	        case yaml.Kind.MAP:
	            var ymap = n;
	            return {
	                errors: [],
	                startPosition: n.startPosition,
	                endPosition: n.endPosition,
	                mappings: ymap.mappings.map(function (x) { return copyNode(x); }),
	                kind: yaml.Kind.MAP,
	                parent: ymap.parent
	            };
	    }
	    return n;
	}
	var innerShift = function (offset, yaNode, shift) {
	    if (!yaNode)
	        return;
	    if (yaNode.startPosition >= offset) {
	        yaNode.startPosition += shift;
	    }
	    if (yaNode.endPosition > offset) {
	        yaNode.endPosition += shift;
	    }
	    //this kind is a separate case
	    if (yaNode.kind == yaml.Kind.MAPPING) {
	        var m = yaNode;
	        innerShift(offset, m.key, shift);
	        innerShift(offset, m.value, shift);
	    }
	};
	function splitOnLines(text) {
	    var lines = text.match(/^.*((\r\n|\n|\r)|$)/gm);
	    return lines;
	}
	//TODO IMPROVE INDENTS
	function stripIndent(text, indent) {
	    var lines = splitOnLines(text);
	    var rs = [];
	    for (var i = 0; i < lines.length; i++) {
	        if (i == 0) {
	            rs.push(lines[0]);
	        }
	        else {
	            rs.push(lines[i].substring(indent.length));
	        }
	    }
	    return rs.join("");
	}
	var leadingIndent = function (node, text) {
	    var leading = "";
	    var pos = node.start() - 1;
	    //if(pos > 0 && text[pos] == ' ' && text[pos-1] == '-') {
	    //    // seq here
	    //    pos -= 2;
	    //}
	    while (pos > 0) {
	        var ch = text[pos];
	        //if (ch == '\r' || ch == '\n' || ch != ' ') break;
	        //console.log('char: [' + ch + ']');
	        if (ch != ' ' && ch != '-')
	            break;
	        leading = ' ' + leading;
	        pos--;
	    }
	    return leading;
	};
	function indent(line) {
	    var rs = "";
	    for (var i = 0; i < line.length; i++) {
	        var c = line[i];
	        if (c == '\r' || c == '\n') {
	            continue;
	        }
	        if (c == ' ' || c == '\t') {
	            rs += c;
	            continue;
	        }
	        break;
	    }
	    return rs;
	}
	function indentLines(s, indent) {
	    return s.split("\n").map(function (x) {
	        if (x.trim().length == 0) {
	            return x;
	        }
	        return indent + x;
	    }).join("\n");
	}
	function extraIndent(text, indent) {
	    var lines = splitOnLines(text);
	    var rs = [];
	    for (var i = 0; i < lines.length; i++) {
	        if (i == 0) {
	            rs.push(lines[0]);
	        }
	        else {
	            if (lines[i].trim().length > 0) {
	                rs.push(indent + lines[i]);
	            }
	            else {
	                rs.push("");
	            }
	        }
	    }
	    return rs.join("");
	}
	var Project = (function () {
	    /**
	     *
	     * @param rootPath - path to folder where your root api is located
	     * @param resolver
	     * @param _httpResolver
	     */
	    function Project(rootPath, resolver, _httpResolver) {
	        this.rootPath = rootPath;
	        this.resolver = resolver;
	        this._httpResolver = _httpResolver;
	        this.listeners = [];
	        this.tlisteners = [];
	        this.pathToUnit = {};
	        this.failedUnits = {};
	        if (this.resolver == null) {
	            this.resolver = new FSResolverImpl();
	        }
	        if (this._httpResolver == null) {
	            this._httpResolver = new HTTPResolverImpl();
	        }
	    }
	    Project.prototype.cloneWithResolver = function (newResolver, httpResolver) {
	        if (httpResolver === void 0) { httpResolver = null; }
	        var newProject = new Project(this.rootPath, newResolver, httpResolver ? httpResolver : this._httpResolver);
	        for (var unitPath in this.pathToUnit) {
	            newProject.pathToUnit[unitPath] = this.pathToUnit[unitPath].cloneToProject(newProject);
	        }
	        return newProject;
	    };
	    Project.prototype.setCachedUnitContent = function (pth, cnt, tl) {
	        if (tl === void 0) { tl = true; }
	        var relPath = pth;
	        var apath = toAbsolutePath(this.rootPath, pth);
	        var unit = new CompilationUnit(relPath, cnt, tl, this, apath);
	        this.pathToUnit[apath] = unit;
	        return unit;
	    };
	    Project.prototype.resolveAsync = function (unitPath, pathInUnit) {
	        var _this = this;
	        if (!pathInUnit) {
	            return Promise.reject(new Error("Unit path is null"));
	        }
	        var includeReference = refResolvers.getIncludeReference(pathInUnit);
	        var oldPath = pathInUnit;
	        if (includeReference) {
	            pathInUnit = includeReference.getIncludePath();
	        }
	        var absPath = this.buildPath(pathInUnit, unitPath);
	        if (includeReference) {
	            var result;
	            var refPath = path.dirname(toAbsolutePath(this.rootPath, unitPath)) + '/' + includeReference.encodedName();
	            // if (this.pathToUnit[refPath]) {
	            //     result = this.pathToUnit[refPath];
	            // } else {
	            //     this.pathToUnit[refPath] = new CompilationUnit(includeReference.encodedName(), refResolvers.resolveContents(oldPath), false, this, refPath);
	            //
	            //     result = this.pathToUnit[refPath];
	            // }
	            this.pathToUnit[absPath] ? Promise.resolve(result).then(function (unit) {
	                _this.pathToUnit[refPath] = new CompilationUnit(includeReference.encodedName(), refResolvers.resolveContents(oldPath, _this.pathToUnit[absPath].contents()), false, _this, refPath);
	                return _this.pathToUnit[refPath];
	            }) : this.unitAsync(absPath, true).then(function (unit) {
	                _this.pathToUnit[absPath] = unit;
	                _this.pathToUnit[refPath] = new CompilationUnit(includeReference.encodedName(), refResolvers.resolveContents(oldPath, _this.pathToUnit[absPath].contents()), false, _this, refPath);
	                return _this.pathToUnit[refPath];
	            });
	        }
	        return this.unitAsync(absPath, true);
	    };
	    Project.prototype.resolve = function (unitPath, pathInUnit) {
	        if (!pathInUnit) {
	            return null;
	        }
	        var includeReference = refResolvers.getIncludeReference(pathInUnit);
	        var oldPath = pathInUnit;
	        if (includeReference) {
	            pathInUnit = includeReference.getIncludePath();
	        }
	        var absPath = this.buildPath(pathInUnit, unitPath);
	        if (includeReference) {
	            if (!this.pathToUnit[absPath]) {
	                this.pathToUnit[absPath] = this.unit(absPath, true);
	            }
	            var wrappedUnit = this.pathToUnit[absPath];
	            var refPath = path.dirname(toAbsolutePath(this.rootPath, unitPath)) + '/' + includeReference.encodedName();
	            if (this.pathToUnit[refPath]) {
	                return this.pathToUnit[refPath];
	            }
	            this.pathToUnit[refPath] = new CompilationUnit(includeReference.encodedName(), refResolvers.resolveContents(oldPath, wrappedUnit && wrappedUnit.contents()), false, this, refPath);
	            return this.pathToUnit[refPath];
	        }
	        return this.unit(absPath, true);
	    };
	    Project.prototype.buildPath = function (pathInUnit, unitPath) {
	        if (isWebPath(pathInUnit) || path.isAbsolute(pathInUnit)) {
	            return pathInUnit;
	        }
	        if (isWebPath(unitPath) || path.isAbsolute(unitPath)) {
	            return toAbsolutePath(path.dirname(unitPath), pathInUnit);
	        }
	        return toAbsolutePath(path.dirname(toAbsolutePath(this.rootPath, unitPath)), pathInUnit);
	    };
	    Project.prototype.units = function () {
	        var _this = this;
	        if (!this.resolver.list) {
	            throw new Error("Provided FSResolver is unable to list files. Please, use ExtendedFSResolver.");
	        }
	        var names = this.resolver.list(this.rootPath).filter(function (x) { return path.extname(x) == '.raml'; });
	        return names.map(function (x) { return _this.unit(x); }).filter(function (y) { return y.isTopLevel(); });
	    };
	    Project.prototype.unitsAsync = function () {
	        var _this = this;
	        if (!this.resolver.listAsync) {
	            return Promise.reject(new Error("Provided FSResolver is unable to list files. Please, use ExtendedFSResolver."));
	        }
	        return this.resolver.listAsync(this.rootPath).then(function (x) {
	            var promises = x.filter(function (x) { return path.extname(x) == '.raml'; }).map(function (x) { return _this.unitAsync(x).then(function (x) {
	                return x.isTopLevel() ? x : null;
	            }, function (x) {
	                return null;
	            }); });
	            return Promise.all(promises).then(function (arr) {
	                return arr.filter(function (x) { return x != null; });
	            });
	        });
	    };
	    Project.prototype.lexerErrors = function () {
	        var results = [];
	        this.units().forEach(function (x) {
	            results = results.concat(x.lexerErrors());
	        });
	        return results;
	    };
	    Project.prototype.deleteUnit = function (p, absolute) {
	        if (absolute === void 0) { absolute = false; }
	        var apath = null;
	        if (isWebPath(p)) {
	            apath = p;
	        }
	        else {
	            apath = absolute ? p : toAbsolutePath(this.rootPath, p);
	        }
	        delete this.pathToUnit[apath];
	    };
	    Project.prototype.unit = function (p, absolute) {
	        if (absolute === void 0) { absolute = false; }
	        if (absolute || isWebPath(p)) {
	            if (this.failedUnits[p] != null) {
	                throw (this.failedUnits[p]);
	            }
	        }
	        else {
	            var ap = toAbsolutePath(this.rootPath, p);
	            if (this.failedUnits[ap]) {
	                throw (this.failedUnits[ap]);
	            }
	        }
	        var cnt = null;
	        var apath = p;
	        var response;
	        if (isWebPath(p)) {
	            if (this.pathToUnit[apath]) {
	                return this.pathToUnit[apath];
	            }
	            if (this._httpResolver) {
	                response = this._httpResolver.getResource(p);
	                if (response && response.errorMessage) {
	                    throw new Error(response.errorMessage);
	                }
	                if (response) {
	                    cnt = response.content;
	                }
	                else {
	                    cnt = null;
	                }
	            }
	            else {
	                response = new HTTPResolverImpl().getResource(apath);
	                if (response) {
	                    cnt = response.content;
	                }
	                else {
	                    cnt = null;
	                }
	            }
	        }
	        else {
	            if (p.charAt(0) == '/' && !absolute) {
	                p = p.substr(1); //TODO REVIEW IT
	            }
	            var apath = toAbsolutePath(this.rootPath, p);
	            if (this.pathToUnit[apath]) {
	                return this.pathToUnit[apath];
	            }
	            if (isWebPath(apath)) {
	                if (this._httpResolver) {
	                    var resp = this._httpResolver.getResource(apath);
	                    if (resp && resp.errorMessage) {
	                        throw new Error(resp.errorMessage);
	                    }
	                    if (resp) {
	                        cnt = resp.content;
	                    }
	                    else {
	                        cnt = null;
	                    }
	                }
	                else {
	                    var resourceResponse = new HTTPResolverImpl().getResource(apath);
	                    if (resourceResponse) {
	                        cnt = resourceResponse.content;
	                    }
	                    else {
	                        cnt = null;
	                    }
	                }
	            }
	            else {
	                cnt = this.resolver.content(apath);
	            }
	        }
	        if (cnt == null) {
	            return null;
	        }
	        var tl = util.stringStartsWith(cnt, "#%RAML");
	        var relPath = (isWebPath(this.rootPath) == isWebPath(apath)) ? path.relative(this.rootPath, apath) : apath;
	        var unit = new CompilationUnit(relPath, cnt, tl, this, apath);
	        this.pathToUnit[apath] = unit;
	        if (response) {
	            response.callback = function (contentHolder) {
	                unit.updateContent(contentHolder && contentHolder.content);
	            };
	        }
	        return unit;
	    };
	    Project.prototype.unitAsync = function (p, absolute) {
	        var _this = this;
	        if (absolute === void 0) { absolute = false; }
	        var cnt = null;
	        var apath = p;
	        if (isWebPath(p)) {
	            if (this.pathToUnit[apath]) {
	                return Promise.resolve(this.pathToUnit[apath]);
	            }
	            if (this._httpResolver) {
	                var resp = this._httpResolver.getResourceAsync(apath);
	                cnt = resp.then(function (x) {
	                    if (x.errorMessage) {
	                        return Promise.reject(new Error(x.errorMessage));
	                    }
	                    return x.content;
	                });
	            }
	            else {
	                cnt = new HTTPResolverImpl().getResourceAsync(apath);
	            }
	        }
	        else {
	            if (p.charAt(0) == '/' && !absolute) {
	                p = p.substr(1); //TODO REVIEW IT
	            }
	            if (absolute) {
	                apath = p;
	            }
	            else {
	                apath = toAbsolutePath(this.rootPath, p);
	            }
	            if (this.pathToUnit[apath]) {
	                return Promise.resolve(this.pathToUnit[apath]);
	            }
	            if (isWebPath(apath)) {
	                if (this._httpResolver) {
	                    var resp = this._httpResolver.getResourceAsync(apath);
	                    cnt = resp.then(function (x) {
	                        if (x.errorMessage) {
	                            return Promise.reject(new Error(x.errorMessage));
	                        }
	                        return x.content;
	                    });
	                }
	                else {
	                    cnt = new HTTPResolverImpl().getResourceAsync(apath);
	                }
	            }
	            else {
	                cnt = this.resolver.contentAsync(apath);
	            }
	        }
	        if (cnt == null) {
	            return Promise.resolve(null);
	        }
	        var relPath = (isWebPath(this.rootPath) == isWebPath(apath)) ? path.relative(this.rootPath, apath) : apath;
	        return cnt.then(function (x) {
	            if (!x) {
	                return Promise.reject(new Error("Can note resolve " + apath));
	            }
	            var tl = util.stringStartsWith(x, "#%RAML");
	            var unit = new CompilationUnit(relPath, x, tl, _this, apath);
	            _this.pathToUnit[apath] = unit;
	            return unit;
	        }).then(function (unit) {
	            if (unit.isRAMLUnit()) {
	                return unit;
	            }
	            return schemes.isScheme(unit.contents()) ? schemes.startDownloadingReferencesAsync(unit.contents(), unit) : unit;
	        });
	    };
	    Project.prototype.visualizeNewlines = function (s) {
	        var res = '';
	        for (var i = 0; i < s.length; i++) {
	            var ch = s[i];
	            if (ch == '\r')
	                ch = '\\r';
	            if (ch == '\n')
	                ch = '\\n';
	            res += ch;
	        }
	        return res;
	    };
	    Project.prototype.indent = function (node) {
	        //node.show('NODE');
	        var text = node.unit().contents();
	        //console.log('node text: ' + textutil.replaceNewlines(text.substring(node.start(), node.end())));
	        //console.log('node parent: ' + node.parent());
	        //console.log('node unit: ' + node.unit());
	        if (node == node.root()) {
	            //console.log('node is root');
	            return '';
	        }
	        var leading = leadingIndent(node, text);
	        //console.log('leading: [' + leading + '] ' + leading.length);
	        var dmp = splitOnLines(node.dump());
	        if (dmp.length > 1) {
	            if (dmp[1].trim().length > 0) {
	                //console.log('DMP0: [' + dmp[0] + ']');
	                //console.log('DMP1: [' + dmp[1] + ']');
	                var extra = indent(dmp[1]);
	                return leading + extra;
	            }
	        }
	        //console.log('LEADING: [' + this.visualizeNewlines(leading) + '] ');
	        return leading + '  ';
	    };
	    Project.prototype.startIndent = function (node) {
	        var text = node.unit().contents();
	        //console.log('Node text:\n' + this.visualizeNewlines(text.substring(node.start(), node.end())));
	        if (node == node.root())
	            return '';
	        var dmp = splitOnLines(node.dump());
	        if (dmp.length > 0) {
	            console.log('FIRST: ' + dmp[0]);
	            var extra = indent(dmp[0]);
	            return extra + '  ';
	        }
	        //console.log('LEADING: [' + this.visualizeNewlines(leading) + '] ');
	        return '';
	    };
	    Project.prototype.canWriteInOneLine = function (node) {
	        return false;
	    };
	    Project.prototype.isOneLine = function (node) {
	        return node.text().indexOf('\n') < 0;
	    };
	    Project.prototype.recalcPositionsUp = function (target) {
	        var np = target;
	        while (np) {
	            np.recalcEndPositionFromChilds();
	            np = np.parent();
	        }
	    };
	    Project.prototype.add2 = function (target, node, toSeq, ipoint, json) {
	        if (json === void 0) { json = false; }
	        var unit = target.unit();
	        var api = target.root();
	        //console.log('api: ' + api);
	        var point = null;
	        if (ipoint) {
	            if (ipoint instanceof ASTNode) {
	                //console.log('insertion: ast node');
	                point = ipoint;
	            }
	            if (ipoint instanceof InsertionPoint) {
	                //console.log('insertion: ip');
	                point = ipoint.point;
	            }
	        }
	        //console.log('target: ' + target.kindName() + '/' + target.valueKindName() + ' node: ' + node.kindName());
	        //if(point) point.show('POINT:');
	        if (target.isValueInclude()) {
	            //console.log('insert to include ref');
	            var childs = target.children();
	            if (childs.length == 0) {
	                throw new Error("not implemented: insert into empty include ref");
	            }
	            var parent = childs[0].parent();
	            //console.log('parent: ' + parent);
	            //parent.show('INCLUDE PARENT:');
	            this.add2(parent, node, toSeq, point, json);
	            return;
	        }
	        var range = new textutil.TextRange(unit.contents(), node.start(), node.end());
	        var targetRange = new textutil.TextRange(unit.contents(), target.start(), target.end());
	        var unitText = target.unit().contents();
	        if (target.valueKind() == yaml.Kind.SEQ) {
	            target = createSeq(target.valueAsSeq(), target, target.unit());
	        }
	        var json = this.isJson(target);
	        //console.log('target: ' + target.start() + '..' + target.end());
	        var originalIndent = json ? '' : this.indent(target.isSeq() ? target.parent() : target);
	        //console.log('indent: [' + originalIndent + '] ' + originalIndent.length + '; toseq: ' + toSeq + '; json: ' + json);
	        var xindent = originalIndent;
	        var indentLength = originalIndent.length;
	        var isTargetSeq = target.isSeq() || target.isMapping() && (target.isValueSeq() || target.isValueScalar() || !target.asMapping().value); //target.valueKind() == yaml.Kind.SEQ || target.isSeq();
	        //toSeq = false;
	        //console.log('target: ' + target.kindName() + '/' + yaml.Kind[target.valueKind()] + '; toseq: ' + toSeq);
	        //target.root().show("API:");
	        //target.show("TARGET:");
	        //console.log('oindent: ' + originalIndent.length);
	        toSeq = toSeq; // || isTargetSeq;
	        if (toSeq) {
	            if (json) {
	            }
	            else {
	                if (isTargetSeq) {
	                    xindent += "  ";
	                    indentLength += 2;
	                }
	            }
	        }
	        //console.log('xindent: ' + xindent.length);
	        var buf = new MarkupIndentingBuffer(xindent);
	        //target.show('TARGET:');
	        //node.show('NODE1');
	        node.markupNode(buf, node._actualNode(), 0, json);
	        var text = buf.text;
	        //node.show('NODE2', 0, text);
	        //console.log('TEXT TO ADD0: ' + textutil.replaceNewlines(text));
	        if (toSeq) {
	            //if(target.valueKind() == yaml.Kind.SEQ) {
	            var trimText = textutil.trimEnd(text);
	            var trimLen = text.length - trimText.length;
	            if (trimLen > 0) {
	                //console.log('trim len: ' + trimLen);
	                var textlen = text.length;
	                text = text.substring(0, textlen - trimLen);
	                node.shiftNodes(textlen - trimLen, -trimLen);
	            }
	        }
	        //target.show('TARGET2');
	        //node.show('NODE2', 0, text);
	        //console.log('TEXT TO ADD1: ' + textutil.replaceNewlines(text));
	        //console.log('TEXT TO ADD:\n' + this.visualizeNewlines(text));
	        //console.log('toseq: ' + toSeq);
	        if (toSeq && !json) {
	            if (node.highLevelNode()) {
	            }
	            //console.log('target: ' + target.kindName());
	            if (target.isMapping()) {
	            }
	            if (target.isSeq() || target.isMapping() && (target.isValueSeq() || target.isValueScalar() || !target.asMapping().value)) {
	                //console.log('--- make it seq');
	                text = originalIndent + '- ' + text;
	            }
	            else {
	                //console.log('--- keep it map');
	                text = originalIndent + text;
	            }
	        }
	        else {
	            text = originalIndent + text;
	        }
	        //console.log('TEXT TO ADD2: ' + textutil.replaceNewlines(text));
	        //target.show('TARGET3');
	        var pos = target.end();
	        //console.log('insert to target end: ' + pos+ ' ; point: ' + point);
	        if (point) {
	            //point.show("POINT");
	            if (point != target) {
	                pos = point.end();
	            }
	            else {
	                if (json && toSeq) {
	                }
	                else {
	                    pos = target.keyEnd() + 1;
	                    pos = new textutil.TextRange(unitText, pos, pos).extendAnyUntilNewLines().endpos();
	                }
	            }
	        }
	        else {
	            if (json && toSeq) {
	                var seq = target.asSeq();
	                if (seq) {
	                    if (seq.items.length > 0) {
	                        pos = seq.items[seq.items.length - 1].endPosition;
	                    }
	                    else {
	                        pos = seq.endPosition - 1;
	                    }
	                }
	            }
	            else {
	                if (ipoint && (ipoint instanceof InsertionPoint)) {
	                    //ipoint.show('insertion point provided');
	                    var ip = ipoint;
	                    if (ip.type == InsertionPointType.START) {
	                        pos = target.keyEnd() + 1;
	                        pos = new textutil.TextRange(unitText, pos, pos).extendAnyUntilNewLines().endpos();
	                    }
	                }
	            }
	        }
	        //console.log('insert poition: ' + pos);
	        var insertionRange = new textutil.TextRange(unitText, 0, pos);
	        pos = insertionRange.extendToNewlines().reduceSpaces().endpos();
	        if (json && target.isSeq()) {
	            var seq = target.asSeq();
	            if (seq.items.length > 0) {
	                text = ', ' + text;
	                indentLength += 2;
	            }
	        }
	        else if (pos > 0 && unitText[pos - 1] != '\n') {
	            text = "\n" + text;
	            indentLength++;
	        }
	        var suffixLen = 0;
	        if (toSeq && !json) {
	            text += '\n';
	            suffixLen++;
	        }
	        //console.log('FINAL TEXT TO ADD: [' + textutil.replaceNewlines(text) + '] at position ' + pos);
	        var newtext = unitText.substring(0, pos) + text + unitText.substring(pos, unitText.length);
	        var cu = unit;
	        cu.updateContentSafe(newtext);
	        this.executeReplace(new textutil.TextRange(unitText, pos, pos), text, cu);
	        //console.log('shift root from position: ' + pos);
	        target.root().shiftNodes(pos, indentLength + (node.end() - node.start()) + suffixLen);
	        //console.log('node len: ' + (node.end()-node.start()));
	        //console.log('text len: ' + text.length);
	        //(<ASTNode>target.root()).shiftNodes(pos, text.length+indentLength);
	        //target.show('TARGET2:');
	        //node.show('NODE TO ADD:');
	        if (point) {
	            var childs = target.children();
	            var index = -1;
	            for (var i = 0; i < childs.length; i++) {
	                var x = childs[i];
	                if (x.start() == point.start() && x.end() == point.end()) {
	                    index = i;
	                    break;
	                }
	            }
	            //console.log('index: ' + index);
	            if (index >= 0) {
	                target.addChild(node, index + 1);
	            }
	            else {
	                target.addChild(node);
	            }
	        }
	        else {
	            target.addChild(node);
	        }
	        node.shiftNodes(0, pos + indentLength);
	        //target.show('TARGET UPDATED:');
	        this.recalcPositionsUp(target);
	        //target.show('TARGET UPDATED POSITIONS:');
	        //api.show('ROOT UPDATED POSITIONS:');
	        node.setUnit(target.unit());
	        node.visit(function (n) {
	            var node = n;
	            node.setUnit(target.unit());
	            return true;
	        });
	    };
	    Project.prototype.isJsonMap = function (node) {
	        if (!node.isMap())
	            return false;
	        var text = node.text().trim();
	        return text.length >= 2 && text[0] == '{' && text[text.length - 1] == '}';
	    };
	    Project.prototype.isJsonSeq = function (node) {
	        if (!node.isSeq())
	            return false;
	        var text = node.text().trim();
	        return text.length >= 2 && text[0] == '[' && text[text.length - 1] == ']';
	    };
	    Project.prototype.isJson = function (node) {
	        return this.isJsonMap(node) || this.isJsonSeq(node);
	    };
	    Project.prototype.remove = function (unit, target, node) {
	        var parent = node.parent();
	        node._oldText = node.dump();
	        //node.showParents('PARENTS:');
	        //console.log('REMOVE NODE: ' + node.kindName() + ' from ' + target.kindName());
	        //console.log('INITIAL SELECTION: [' + textutil.replaceNewlines(range.text()) + ']');
	        //console.log('  text: \n' + unitText.substring(startpos,endpos));
	        if (this.isOneLine(node) && node.isMapping() && node.parent().isMap()) {
	            var mapnode = node.parent();
	            if (mapnode.asMap().mappings.length == 1 && mapnode.parent() != null) {
	                //console.log('REMOVE MAP INSTEAD!');
	                this.remove(unit, mapnode.parent(), mapnode);
	                return;
	            }
	        }
	        if (this.isOneLine(node) && node.isScalar() && node.parent().isSeq()) {
	            var seqnode = node.parent();
	            var seqn = seqnode.asSeq();
	            //console.log('SEQ: ' + seqn.items.length);
	            if (seqn.items.length == 1) {
	                //console.log('REMOVE SEQ INSTEAD!');
	                this.remove(unit, seqnode.parent(), seqnode);
	                return;
	            }
	        }
	        if (target.isMapping() && node.isSeq()) {
	            //console.log('remove seq from mapping');
	            var map = target.parent();
	            //console.log('REMOVE MAPPING INSTEAD!');
	            this.remove(unit, map, target);
	            return;
	        }
	        //target.show('TARGET:');
	        //node.show('NODE:');
	        var range = new textutil.TextRange(unit.contents(), node.start(), node.end());
	        var targetRange = new textutil.TextRange(unit.contents(), target.start(), target.end());
	        var parentRange = new textutil.TextRange(unit.contents(), parent.start(), parent.end());
	        var originalStartPos = range.startpos();
	        //console.log('REMOVE TEXT: ' +  this.visualizeNewlines(range.text()));
	        if (target.isSeq()) {
	            // extend range to start of line
	            //console.log('RANGE SEQ 0: ' + textutil.replaceNewlines(range.text()));
	            var seq = (node.isSeq() ? node : node.parentOfKind(yaml.Kind.SEQ));
	            //console.log('seq: ' + seq.text() + ' json: ' + this.isJson(seq));
	            if (seq && this.isJson(seq)) {
	                range = range.extendSpaces().extendCharIfAny(',').extendSpaces();
	            }
	            else {
	                range = range.extendToStartOfLine().extendAnyUntilNewLines().extendToNewlines(); //
	            }
	        }
	        if (target.isMap()) {
	            // extend range to end of line
	            //console.log('RANGE MAP 0: [' +  this.visualizeNewlines(range.text()) + ']');
	            range = range.trimEnd().extendAnyUntilNewLines().extendToNewlines();
	            //console.log('RANGE MAP 1: [' +  this.visualizeNewlines(range.text()) + ']');
	            range = range.extendToStartOfLine().extendUntilNewlinesBack();
	        }
	        if (target.kind() == yaml.Kind.MAPPING) {
	            //console.log('RANGE MAPPING 0: ' +  this.visualizeNewlines(range.text()));
	            //console.log('NODE TEXT: ' + node.text());
	            if (this.isJson(node) && this.isOneLine(node)) {
	            }
	            else {
	                // extend range to end of line
	                //console.log('RANGE MAP 0: ' +  this.visualizeNewlines(range.text()));
	                range = range.extendSpacesUntilNewLines();
	                range = range.extendToNewlines();
	                //console.log('RANGE MAP 2: ' +  this.visualizeNewlines(range.text()));
	                range = range.extendToStartOfLine().extendUntilNewlinesBack();
	            }
	        }
	        if (node.isSeq()) {
	            //console.log('cleanup seq');
	            range = range.reduceSpaces();
	        }
	        //console.log('NODE:\n-----------\n' + range.unitText() + '\n-------------');
	        //console.log('TARGET: ' + target.kindName());
	        //target.show('TARGET');
	        //console.log('FINAL REMOVE TEXT: [' +  this.visualizeNewlines(range.text()) + ']');
	        //console.log('NEW TEXT:\n-----------\n' + range.remove() + '\n-------------');
	        var cu = unit;
	        cu.updateContentSafe(range.remove());
	        this.executeReplace(range, "", cu);
	        //node.parent().show('Before remove');
	        node.parent().removeChild(node);
	        var shift = -range.len();
	        //console.log('shift: ' + shift);
	        target.root().shiftNodes(originalStartPos, shift);
	        this.recalcPositionsUp(target);
	        //this.executeTextChange(new lowlevel.TextChangeCommand(range.startpos(), range.len(), "", unit))
	        //target.show('TARGET AFTER REMOVE:');
	        //target.root().show('API AFTER REMOVE:');
	    };
	    Project.prototype.changeKey = function (unit, attr, newval) {
	        //console.log('set key: ' + newval);
	        var range = new textutil.TextRange(attr.unit().contents(), attr.keyStart(), attr.keyEnd());
	        if (attr.kind() == yaml.Kind.MAPPING) {
	            var sc = attr._actualNode().key;
	            sc.value = newval;
	            sc.endPosition = sc.startPosition + newval.length;
	        }
	        var cu = unit;
	        this.executeReplace(range, newval, cu);
	        //console.log('new text: ' + this.visualizeNewlines(newtext));
	        var shift = newval.length - range.len();
	        //console.log('shift: ' + shift);
	        attr.root().shiftNodes(range.startpos(), shift, attr);
	        this.recalcPositionsUp(attr);
	    };
	    Project.prototype.executeReplace = function (r, txt, unit) {
	        var command = new lowlevel.TextChangeCommand(r.startpos(), r.endpos() - r.startpos(), txt, unit);
	        unit.project();
	        try {
	            this.tlisteners.forEach(function (x) { return x(command); });
	        }
	        catch (e) {
	            return false;
	        }
	        var newtext = r.replace(txt);
	        unit.updateContentSafe(newtext);
	        return true;
	    };
	    Project.prototype.changeValue = function (unit, attr, newval) {
	        //console.log('set value: ' + newval);mark
	        //console.log('ATTR ' + yaml.Kind[attr.kind()] + '; VALUE: ' + val + ' => ' + newval);
	        //attr.root().show('NODE:');
	        //console.log('TEXT:\n' + attr.unit().contents());
	        var range = new textutil.TextRange(attr.unit().contents(), attr.start(), attr.end());
	        //console.log('Range0: ' + range.startpos() + '..' + range.endpos() + ': [' + this.visualizeNewlines(range.text()) + ']');
	        //console.log('ATTR: ' + attr.kindName());
	        //attr.root().show('BEFORE');
	        var newNodeText;
	        var prefix = 0;
	        var delta = 0;
	        var replacer = null;
	        var mapping = null;
	        //console.log('attr: ' + attr.kindName());
	        if (attr.kind() == yaml.Kind.SCALAR) {
	            if (typeof newval == 'string') {
	                attr.asScalar().value = newval;
	                //range = range.withStart(attr.valueStart()).withEnd(attr.valueEnd());
	                //console.log('Range1: ' + this.visualizeNewlines(range.text()));
	                //console.log('Range0: ' + range.startpos() + '..' + range.endpos());
	                newNodeText = newval;
	            }
	            else {
	                throw new Error("not implemented");
	            }
	        }
	        else if (attr.kind() == yaml.Kind.MAPPING) {
	            //attr.show('ATTR:');
	            mapping = attr.asMapping();
	            //console.log('mapping val: ' + attr.valueKindName());
	            if (attr.isValueInclude()) {
	                var inc = attr.valueAsInclude();
	                var includeString = inc.value;
	                var includePath = includeString;
	                //console.log("attr.setValue: path: " + includePath);
	                var resolved = attr.unit().resolve(includePath);
	                if (resolved == null) {
	                    console.log("attr.setValue: couldn't resolve: " + includePath);
	                    return; // "can not resolve "+includePath
	                }
	                //console.log("attr.setValue: resolved: " + includePath);
	                if (resolved.isRAMLUnit()) {
	                    //TODO DIFFERENT DATA TYPES, inner references
	                    return;
	                }
	                //TODO for now disabling an update from outline details to JSON schema when there is a reference
	                //to an inner element of the schema
	                if (!refResolvers.getIncludeReference(includeString)) {
	                    resolved.updateContent(newval);
	                }
	                return;
	            }
	            //console.log('Range0: ' + range.startpos() + '..' + range.endpos() + ': [' + this.visualizeNewlines(range.text()) + ']');
	            if (mapping.value)
	                range = range.withStart(attr.valueStart()).withEnd(attr.valueEnd());
	            else
	                range = range.withStart(attr.keyEnd() + 1).withEnd(attr.keyEnd() + 1);
	            //console.log('Range1: ' + range.startpos() + '..' + range.endpos());
	            range = range.reduceNewlinesEnd();
	            //console.log('Range2: ' + range.startpos() + '..' + range.endpos() + ': [' + this.visualizeNewlines(range.text()) + ']');
	            if (newval == null) {
	                newNodeText = '';
	                mapping.value = null;
	            }
	            else if (typeof newval == 'string' || newval == null) {
	                var newstr = newval;
	                var ind = this.indent(attr);
	                //console.log('indent: ' + ind.length);
	                if (newstr && textutil.isMultiLine(newstr)) {
	                    newstr = '' + textutil.makeMutiLine(newstr, ind.length / 2);
	                }
	                newNodeText = newstr;
	                //var valueNode = null;
	                if (!mapping.value) {
	                    console.log('no value');
	                    mapping.value = yaml.newScalar(newstr);
	                    mapping.value.startPosition = attr.keyEnd() + 1;
	                    mapping.value.endPosition = mapping.value.startPosition + newstr.length;
	                    mapping.endPosition = mapping.value.endPosition;
	                    if (unit.contents().length > attr.keyEnd() + 1) {
	                        var vlPos = attr.keyEnd() + 1;
	                        if (unit.contents()[vlPos - 1] == ':') {
	                            newNodeText = " " + newNodeText;
	                            mapping.value.startPosition++;
	                            mapping.value.endPosition++;
	                            mapping.endPosition++;
	                            delta++;
	                        }
	                    }
	                }
	                else if (mapping.value.kind == yaml.Kind.SEQ) {
	                    console.log('seq value');
	                    var v = mapping.value.items[0];
	                    //TODO !!! assign value
	                    throw "assign value!!!";
	                }
	                else if (mapping.value.kind == yaml.Kind.SCALAR) {
	                    //console.log('scalar value');
	                    var sc = mapping.value;
	                    var oldtext = sc.value;
	                    //console.log('oldval: ' + sc.value);
	                    //console.log('newstr: ' + newstr + ' ' + newstr.length);
	                    sc.value = newstr;
	                    //console.log('value1: ' + mapping.value.startPosition + '..' + mapping.value.endPosition);
	                    mapping.value.endPosition = mapping.value.startPosition + newstr.length;
	                    //console.log('value2: ' + mapping.value.startPosition + '..' + mapping.value.endPosition);
	                    mapping.endPosition = mapping.value.endPosition;
	                    //console.log('mvalue: ' + mapping.startPosition + '..' + mapping.endPosition);
	                    //console.log('newval: ' + sc.value);
	                    delta += newstr.length - oldtext.length;
	                }
	            }
	            else {
	                var n = newval;
	                if (n.isMapping()) {
	                    newval = createMap([n.asMapping()]);
	                    n = newval;
	                }
	                else if (n.isMap()) {
	                }
	                else {
	                    throw new Error("only MAP/MAPPING nodes allowed as values");
	                }
	                //n.show('NODE1');
	                var buf = new MarkupIndentingBuffer('');
	                n.markupNode(buf, n._actualNode(), 0, true);
	                //n.show('NODE2');
	                newNodeText = '' + buf.text + '';
	                //indent++;
	                //n.shiftNodes(0, 1);
	                //console.log('node text: [[[' + newNodeText + ']]]');
	                //n.show("NN1:", 0, newNodeText);
	                //range = mapping.value? range.withStart(attr.valueStart()).withEnd(attr.valueEnd()) : range.withStart(attr.keyEnd()+1).withEnd(attr.keyEnd()+1 + newNodeText);
	                n.shiftNodes(0, range.startpos() + delta);
	                //n.show("NN2:");
	                replacer = n;
	            }
	        }
	        else {
	            console.log('Unsupported change value case: ' + attr.kindName());
	        }
	        //console.log('RangeX: ' + range.startpos() + '..' + range.endpos() + ': [' + this.visualizeNewlines(range.text()) + ']');
	        //console.log('new node text: ' + newNodeText);
	        var cu = unit;
	        //console.log('Range1: ' + range.startpos() + '..' + range.endpos());
	        //console.log('replace: ' + range.len());
	        //console.log('Range: ' + range.startpos() + '..' + range.endpos());
	        //console.log('OldText: ' + this.visualizeNewlines(cu.contents()));
	        this.executeReplace(range, newNodeText, cu);
	        //var newtext = range.replace(newNodeText);
	        //console.log('NewText: ' + this.visualizeNewlines(newtext));
	        //cu.updateContentSafe(newtext);
	        var shift = newNodeText.length - range.len();
	        //var shift = delta;
	        //attr.root().show('BEFORE SHIFT');
	        //console.log('shift: ' + shift + '; from: ' + (range.endpos() + prefix) + '; delta: ' + delta + '; prefix: ' + prefix);
	        attr.root().shiftNodes(range.endpos() + prefix, shift, attr);
	        //(<ASTNode>attr.root()).shiftNodes(range.endpos()+indent, shift);
	        //attr.show('ATTR2:');
	        if (replacer) {
	            mapping.value = replacer._actualNode();
	        }
	        this.recalcPositionsUp(attr);
	    };
	    Project.prototype.initWithRoot = function (root, newroot) {
	        var shift = root.end();
	        newroot.markup(false);
	        newroot._actualNode().startPosition = shift;
	        newroot._actualNode().endPosition = shift;
	        newroot.setUnit(root.unit());
	    };
	    Project.prototype.execute = function (cmd) {
	        var _this = this;
	        //console.log('Commands: ' + cmd.commands.length);
	        cmd.commands.forEach(function (x) {
	            //console.log('EXECUTE: kind: ' + lowlevel.CommandKind[x.kind] + '; val: ' + x.value);
	            switch (x.kind) {
	                case lowlevel.CommandKind.CHANGE_VALUE:
	                    var attr = x.target;
	                    var curval = attr.value();
	                    if (!curval) {
	                        curval = "";
	                    }
	                    var newval = x.value;
	                    console.log('set value: ' + (typeof curval) + ' ==> ' + (typeof newval));
	                    if ((typeof (curval) == 'string' ||
	                        typeof (curval) == 'number' ||
	                        typeof (curval) == 'boolean') && typeof newval == 'string') {
	                        //console.log('set value: str => str');
	                        if (curval != newval) {
	                            _this.changeValue(attr.unit(), attr, newval);
	                        }
	                    }
	                    else if ((typeof (curval) == 'string' ||
	                        typeof (curval) == 'number' ||
	                        typeof (curval) == 'boolean')
	                        && typeof (newval) != 'string') {
	                        //console.log('set value: str => obj');
	                        // change structure
	                        //this.changeValue(attr.unit(), attr, null);
	                        _this.changeValue(attr.unit(), attr, newval);
	                    }
	                    else if (typeof curval != 'string' && typeof newval == 'string') {
	                        var newstr = x.value;
	                        if (curval.kind() == yaml.Kind.MAPPING) {
	                            if (textutil.isMultiLine(newstr)) {
	                                //console.log('multiline');
	                                attr.children().forEach(function (n) {
	                                    _this.remove(attr.unit(), attr, n);
	                                });
	                                _this.changeValue(attr.unit(), attr, newstr);
	                            }
	                            else {
	                                //console.log('singleline');
	                                _this.changeKey(attr.unit(), curval, newstr);
	                            }
	                        }
	                        else {
	                            throw new Error('unsupported case: attribute value conversion: ' + (typeof curval) + ' ==> ' + (typeof newval) + ' not supported');
	                        }
	                    }
	                    else if (typeof curval != 'string' && typeof newval != 'string') {
	                        var newvalnode = newval;
	                        //(<ASTNode>curval).show("OLD:");
	                        //newvalnode.show("NEW:");
	                        if (newvalnode.isMapping()) {
	                            newval = createMap([newvalnode.asMapping()]);
	                        }
	                        //console.log('obj obj: ' + (curval == newval));
	                        if (curval == newval)
	                            break;
	                        // change structure
	                        //console.log('set value: obj => obj');
	                        var node = newval;
	                        var map = node.asMap();
	                        //console.log('attr: ' + attr.kindName() + " " + attr.dump());
	                        attr.children().forEach(function (n) {
	                            _this.remove(attr.unit(), attr, n);
	                        });
	                        node.children().forEach(function (m) {
	                            //this.add2(attr, <ASTNode>m, false, null, true);
	                        });
	                        _this.changeValue(attr.unit(), attr, newval);
	                    }
	                    else {
	                        throw new Error("shouldn't be this case: attribute value conversion " + (typeof curval) + ' ==> ' + (typeof newval) + ' not supported');
	                    }
	                    return;
	                case lowlevel.CommandKind.CHANGE_KEY:
	                    var attr = x.target;
	                    _this.changeKey(attr.unit(), attr, x.value);
	                    return;
	                case lowlevel.CommandKind.ADD_CHILD:
	                    var attr = x.target;
	                    var newValueNode = x.value;
	                    _this.add2(attr, newValueNode, x.toSeq, x.insertionPoint);
	                    return;
	                case lowlevel.CommandKind.REMOVE_CHILD:
	                    var target = x.target;
	                    var node = x.value;
	                    _this.remove(target.unit(), target, node);
	                    return;
	                case lowlevel.CommandKind.INIT_RAML_FILE:
	                    var root = x.target;
	                    var newroot = x.value;
	                    _this.initWithRoot(root, newroot);
	                    return;
	                default:
	                    console.log('UNSUPPORTED COMMAND: ' + lowlevel.CommandKind[x.kind]);
	                    return;
	            }
	        });
	    };
	    Project.prototype.replaceYamlNode = function (target, newNodeContent, offset, shift, unit) {
	        //console.log('New content:\n' + newNodeContent);
	        //target.show('OLD TARGET');
	        var newYamlNode = yaml.load(newNodeContent, {});
	        //console.log('new yaml: ' + yaml.Kind[newYamlNode.kind]);
	        this.updatePositions(target.start(), newYamlNode);
	        //console.log('Shift: ' + shift);
	        //(<ASTNode>unit.ast()).shiftNodes(offset, shift);
	        target.root().shiftNodes(offset, shift);
	        var targetParent = target.parent();
	        var targetYamlNode = target._actualNode();
	        var parent = targetYamlNode.parent;
	        newYamlNode.parent = parent;
	        if (targetParent && targetParent.kind() == yaml.Kind.MAP) {
	            //console.log('MAP!!!');
	            var targetParentMapNode = targetParent._actualNode();
	            targetParentMapNode.mappings = targetParentMapNode.mappings.map(function (x) {
	                if (x != targetYamlNode) {
	                    return x;
	                }
	                return newYamlNode;
	            });
	        }
	        target.updateFrom(newYamlNode);
	        //target.show('MEW TARGET');
	        this.recalcPositionsUp(target);
	    };
	    Project.prototype.executeTextChange2 = function (textCommand) {
	        var cu = textCommand.unit;
	        var unitText = cu.contents();
	        var target = textCommand.target;
	        if (target) {
	            var cnt = unitText.substring(target.start(), target.end());
	            var original = unitText;
	            unitText = unitText.substr(0, textCommand.offset) + textCommand.text + unitText.substr(textCommand.offset + textCommand.replacementLength);
	            var newNodeContent = cnt.substr(0, textCommand.offset - target.start()) +
	                textCommand.text + cnt.substr(textCommand.offset - target.start() + textCommand.replacementLength);
	            cu.updateContentSafe(unitText);
	            if (textCommand.offset > target.start()) {
	                try {
	                    var shift = textCommand.text.length - textCommand.replacementLength;
	                    var offset = textCommand.offset;
	                    target.unit().project().replaceYamlNode(target, newNodeContent, offset, shift, textCommand.unit);
	                }
	                catch (e) {
	                    console.log('New node contents (causes error below): \n' + newNodeContent);
	                    console.log('Reparse error: ' + e.stack);
	                }
	            }
	        }
	        else {
	            unitText = unitText.substr(0, textCommand.offset) + textCommand.text + unitText.substr(textCommand.offset + textCommand.replacementLength);
	        }
	        cu.updateContent(unitText);
	        this.listeners.forEach(function (x) {
	            x(null);
	        });
	        this.tlisteners.forEach(function (x) {
	            x(textCommand);
	        });
	    };
	    Project.prototype.executeTextChange = function (textCommand) {
	        var l0 = new Date().getTime();
	        try {
	            var oc = textCommand.unit.contents();
	            //console.log('Offset: ' + textCommand.offset + '; end: ' + (textCommand.offset + textCommand.replacementLength) + '; len: ' + textCommand.replacementLength);
	            var target = textCommand.target;
	            if (target == null) {
	                target = this.findNode(textCommand.unit.ast(), textCommand.offset, textCommand.offset + textCommand.replacementLength);
	            }
	            var cu = textCommand.unit;
	            if (target) {
	                var cnt = oc.substring(target.start(), target.end());
	                //console.log('Content: ' + cnt);
	                var original = oc;
	                oc = oc.substr(0, textCommand.offset) + textCommand.text + oc.substr(textCommand.offset + textCommand.replacementLength);
	                var newNodeContent = cnt.substr(0, textCommand.offset - target.start()) +
	                    textCommand.text + cnt.substr(textCommand.offset - target.start() + textCommand.replacementLength);
	                cu.updateContentSafe(oc);
	                //console.log('UPDATED TEXT: ' + oc);
	                var hasNewLines = breaksTheLine(original, textCommand);
	                if (textCommand.offset > target.start()) {
	                    //we can just reparse new node content;
	                    //console.log(newNodeContent)
	                    try {
	                        var newYamlNode = yaml.load(newNodeContent, {});
	                        this.updatePositions(target.start(), newYamlNode);
	                        //console.log("Positions updated")
	                        //lets shift all after it
	                        var shift = textCommand.text.length - textCommand.replacementLength;
	                        //console.log('shift: ' + shift);
	                        //console.log('offset: ' + textCommand.offset);
	                        textCommand.unit.ast().shiftNodes(textCommand.offset, shift);
	                        //console.log('Unit AST: ' + textCommand.unit.ast())
	                        if (newYamlNode != null && newYamlNode.kind == yaml.Kind.MAP) {
	                            var actualResult = newYamlNode.mappings[0];
	                            var targetYamlNode = target._actualNode();
	                            var parent = targetYamlNode.parent;
	                            var cmd = new lowlevel.ASTDelta();
	                            var unit = textCommand.unit;
	                            cmd.commands = [
	                                new lowlevel.ASTChangeCommand(lowlevel.CommandKind.CHANGE_VALUE, new ASTNode(copyNode(targetYamlNode), unit, null, null, null), new ASTNode(actualResult, unit, null, null, null), 0)
	                            ];
	                            if (parent && parent.kind == yaml.Kind.MAP) {
	                                var map = parent;
	                                map.mappings = map.mappings.map(function (x) {
	                                    if (x != targetYamlNode) {
	                                        return x;
	                                    }
	                                    return actualResult;
	                                });
	                            }
	                            actualResult.parent = parent;
	                            //updating low level ast from yaml
	                            this.recalcPositionsUp(target);
	                            target.updateFrom(actualResult);
	                            //console.log("Incremental without listeners: "+(new Date().getTime()-l0));
	                            //console.log("Notify listeners1: " + this.listeners.length + ":" + this.tlisteners.length);
	                            this.listeners.forEach(function (x) {
	                                x(cmd);
	                            });
	                            this.tlisteners.forEach(function (x) {
	                                x(textCommand);
	                            });
	                            //console.log("Incremental update processed");
	                            return;
	                        }
	                    }
	                    catch (e) {
	                        console.log('New node contents (causes error below): \n' + newNodeContent);
	                        console.log('Reparse error: ' + e.stack);
	                    }
	                }
	            }
	            else {
	                oc = oc.substr(0, textCommand.offset) + textCommand.text + oc.substr(textCommand.offset + textCommand.replacementLength);
	            }
	            var t2 = new Date().getTime();
	            //console.log("Full without listeners:"+(t2-l0));
	            //!find node in scope
	            cu.updateContent(oc);
	            //console.log("Notify listeners2: " + this.listeners.length + ":" + this.tlisteners.length);
	            this.listeners.forEach(function (x) {
	                x(null);
	            });
	            this.tlisteners.forEach(function (x) {
	                x(textCommand);
	            });
	        }
	        finally {
	            var t2 = new Date().getTime();
	        }
	    };
	    Project.prototype.updatePositions = function (offset, n) {
	        var _this = this;
	        if (n == null) {
	            return;
	        }
	        if (n.startPosition == -1) {
	            n.startPosition = offset;
	        }
	        else {
	            n.startPosition = offset + n.startPosition;
	        }
	        n.endPosition = offset + n.endPosition;
	        //console.log('SET POS: ' + n.startPosition + ".." + n.endPosition);
	        switch (n.kind) {
	            case yaml.Kind.MAP:
	                var m = n;
	                m.mappings.forEach(function (x) { return _this.updatePositions(offset, x); });
	                break;
	            case yaml.Kind.MAPPING:
	                var ma = n;
	                this.updatePositions(offset, ma.key);
	                this.updatePositions(offset, ma.value);
	                break;
	            case yaml.Kind.SCALAR:
	                break;
	            case yaml.Kind.SEQ:
	                var s = n;
	                s.items.forEach(function (x) { return _this.updatePositions(offset, x); });
	                break;
	        }
	    };
	    Project.prototype.findNode = function (n, offset, end) {
	        var _this = this;
	        if (n == null) {
	            return null;
	        }
	        var node = n;
	        if (n.start() <= offset && n.end() >= end) {
	            var res = n;
	            node.directChildren().forEach(function (x) {
	                var m = _this.findNode(x, offset, end);
	                if (m) {
	                    res = m;
	                }
	            });
	            return res;
	        }
	        return null;
	    };
	    //shiftNodes(n:lowlevel.ILowLevelASTNode, offset:number, shift:number):lowlevel.ILowLevelASTNode{
	    //    var node:ASTNode=<ASTNode>n;
	    //    if (node==null){
	    //        return null;
	    //    }
	    //    node.directChildren().forEach(x=> {
	    //        var m = this.shiftNodes(x, offset, shift);
	    //    })
	    //    var yaNode=(<ASTNode>n)._actualNode();
	    //    if(yaNode) innerShift(offset, yaNode, shift);
	    //    return null;
	    //}
	    Project.prototype.addTextChangeListener = function (listener) {
	        this.tlisteners.push(listener);
	    };
	    Project.prototype.removeTextChangeListener = function (listener) {
	        this.tlisteners = this.tlisteners.filter(function (x) { return x != listener; });
	    };
	    Project.prototype.addListener = function (listener) {
	        this.listeners.push(listener);
	    };
	    Project.prototype.removeListener = function (listener) {
	        this.listeners = this.listeners.filter(function (x) { return x != listener; });
	    };
	    return Project;
	}());
	exports.Project = Project;
	function breaksTheLine(oc, textCommand) {
	    var oldText = oc.substr(textCommand.offset, textCommand.replacementLength);
	    if (oldText.indexOf('\n') != -1) {
	        return true;
	    }
	    if (textCommand.text.indexOf('\n') != -1) {
	        return true;
	    }
	}
	function tryParseScalar(q) {
	    if (q == "true") {
	        q = true;
	    }
	    else if (q == "false") {
	        q = false;
	    }
	    else {
	        var vl = parseFloat(q);
	        if (!isNaN(vl)) {
	            if (("" + q).match("^[-+]?[0-9]*\.?[0-9]+$")) {
	                q = vl;
	            }
	        }
	    }
	    return q;
	}
	var ASTNode = (function () {
	    function ASTNode(_node, _unit, _parent, _anchor, _include, cacheChildren, _includesContents) {
	        if (cacheChildren === void 0) { cacheChildren = false; }
	        if (_includesContents === void 0) { _includesContents = false; }
	        this._node = _node;
	        this._unit = _unit;
	        this._parent = _parent;
	        this._anchor = _anchor;
	        this._include = _include;
	        this.cacheChildren = cacheChildren;
	        this._includesContents = _includesContents;
	        this._errors = [];
	        if (_node == null) {
	            console.log("null");
	        }
	    }
	    ASTNode.prototype.actual = function () {
	        return this._node;
	    };
	    ASTNode.prototype.yamlNode = function () {
	        return this._node;
	    };
	    ASTNode.prototype.includesContents = function () {
	        return this._includesContents;
	    };
	    ASTNode.prototype.setIncludesContents = function (includesContents) {
	        this._includesContents = includesContents;
	    };
	    ASTNode.prototype.gatherIncludes = function (s, inc, anc, inOneMemberMap) {
	        var _this = this;
	        if (s === void 0) { s = []; }
	        if (inc === void 0) { inc = null; }
	        if (anc === void 0) { anc = null; }
	        if (inOneMemberMap === void 0) { inOneMemberMap = true; }
	        if (this._node == null) {
	            return; //TODO FIXME
	        }
	        var kind = this._node.kind;
	        if (kind == yaml.Kind.SCALAR) {
	            if (schemes.isScheme(this._node.value)) {
	                var references = schemes.getReferences(this._node.value, this.unit());
	                references.forEach(function (reference) {
	                    var includeNode = yaml.newScalar(reference);
	                    includeNode.kind = yaml.Kind.INCLUDE_REF;
	                    var includeAST = new ASTNode(includeNode, _this.unit(), null, null, null);
	                    s.push(includeAST);
	                });
	            }
	            return;
	        }
	        else if (kind == yaml.Kind.MAP) {
	            var map = this._node;
	            if (map.mappings.length == 1 && !inOneMemberMap) {
	                new ASTNode(map.mappings[0].value, this._unit, this, inc, anc, this.cacheChildren).gatherIncludes(s);
	            }
	            else {
	                map.mappings.
	                    map(function (x) { return new ASTNode(x, _this._unit, _this, anc ? anc : _this._anchor, inc ? inc : _this._include, _this.cacheChildren); }).
	                    forEach(function (x) { return x.gatherIncludes(s); });
	            }
	        }
	        else if (kind == yaml.Kind.MAPPING) {
	            var mapping = this._node;
	            if (mapping.value == null) {
	            }
	            else {
	                new ASTNode(mapping.value, this._unit, this, anc ? anc : this._anchor, inc ? inc : this._include, this.cacheChildren).gatherIncludes(s);
	            }
	        }
	        else if (kind == yaml.Kind.SEQ) {
	            var seq = this._node;
	            seq.items.filter(function (x) { return x != null; }).map(function (x) { return new ASTNode(x, _this._unit, _this, anc ? anc : _this._anchor, inc ? inc : _this._include, _this.cacheChildren); }).forEach(function (x) { return x.gatherIncludes(s); });
	        }
	        else if (kind == yaml.Kind.INCLUDE_REF) {
	            if (this._unit) {
	                s.push(this);
	            }
	        }
	    };
	    ASTNode.prototype.setHighLevelParseResult = function (highLevelParseResult) {
	        this._highLevelParseResult = highLevelParseResult;
	    };
	    ASTNode.prototype.highLevelParseResult = function () {
	        return this._highLevelParseResult;
	    };
	    ASTNode.prototype.setHighLevelNode = function (highLevel) {
	        this._highLevelNode = highLevel;
	    };
	    ASTNode.prototype.highLevelNode = function () {
	        return this._highLevelNode;
	    };
	    ASTNode.prototype.start = function () {
	        return this._node.startPosition;
	    };
	    ASTNode.prototype.errors = function () {
	        return this._errors;
	    };
	    ASTNode.prototype.parent = function () {
	        return this._parent;
	    };
	    ASTNode.prototype.recalcEndPositionFromChilds = function () {
	        var childs = this.children();
	        //if(this.children().length == 0) return;
	        var max = 0;
	        var first = this.children()[0];
	        var last = this.children()[this.children().length - 1];
	        //this.children().forEach(n=> {
	        //    var node: ASTNode = <ASTNode>n;
	        //    if(node._node.endPosition > max) max = node._node.endPosition;
	        //});
	        if (this.isMapping()) {
	            var mapping = this.asMapping();
	            //console.log('reposition: mapping');
	            if (mapping.value) {
	                if (mapping.value.kind == yaml.Kind.MAP) {
	                    var map = mapping.value;
	                    if (map.startPosition < 0 && first) {
	                        map.startPosition = first.start();
	                    }
	                    if (last)
	                        this._node.endPosition = last._node.endPosition;
	                    //console.log('embedded map: ' + map.startPosition + ".." + map.endPosition);
	                    this._node.endPosition = Math.max(this._node.endPosition, mapping.value.endPosition);
	                }
	                else if (mapping.value.kind == yaml.Kind.SEQ) {
	                    var seq = mapping.value;
	                    if (seq.startPosition < 0) {
	                        //console.log('*** missed start position');
	                        if (seq.items.length > 0) {
	                            var pos = seq.items[0].startPosition;
	                            var range = new textutil.TextRange(this.unit().contents(), pos, pos);
	                            range = range.extendSpacesBack().extendCharIfAnyBack('-');
	                            seq.startPosition = range.startpos();
	                        }
	                        else {
	                        }
	                    }
	                    //console.log('mapping1     : ' + mapping.startPosition + ".." + mapping.endPosition);
	                    //console.log('embedded seq1: ' + seq.startPosition + ".." + seq.endPosition);
	                    if (seq.items.length > 0) {
	                        var ilast = seq.items[seq.items.length - 1];
	                        this._node.endPosition = Math.max(this._node.endPosition, seq.endPosition, ilast.endPosition);
	                        seq.endPosition = Math.max(this._node.endPosition, seq.endPosition, ilast.endPosition);
	                    }
	                }
	                else if (mapping.value.kind == yaml.Kind.SCALAR) {
	                }
	                else {
	                    if (last)
	                        this._node.endPosition = last._node.endPosition;
	                }
	            }
	        }
	        else {
	            if (last)
	                this._node.endPosition = last._node.endPosition;
	        }
	        //this._node.endPosition = max;;
	    };
	    ASTNode.prototype.isValueLocal = function () {
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            var knd = this._node.value.kind;
	            return knd != yaml.Kind.INCLUDE_REF && knd != yaml.Kind.ANCHOR_REF;
	        }
	        return true;
	    };
	    ASTNode.prototype.keyStart = function () {
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            return this._node.key.startPosition;
	        }
	        return -1;
	    };
	    ASTNode.prototype.keyEnd = function () {
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            return this._node.key.endPosition;
	        }
	        return -1;
	    };
	    ASTNode.prototype.valueStart = function () {
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            var mapping = this.asMapping();
	            if (mapping.value)
	                return mapping.value.startPosition;
	            else
	                return mapping.endPosition;
	        }
	        return -1;
	    };
	    ASTNode.prototype.valueEnd = function () {
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            var mapping = this.asMapping();
	            return mapping.value ? mapping.value.endPosition : mapping.endPosition;
	        }
	        return -1;
	    };
	    ASTNode.prototype.end = function () {
	        return this._node.endPosition;
	    };
	    ASTNode.prototype.dump = function () {
	        if (this._oldText) {
	            return this._oldText;
	        }
	        if (this._unit && this._node.startPosition > 0 && this._node.endPosition > 0) {
	            var originalText = this._unit.contents().substring(this._node.startPosition, this._node.endPosition);
	            originalText = stripIndent(originalText, leadingIndent(this, this._unit.contents()));
	            //console.log("L:");
	            //console.log(originalText);
	            return originalText;
	        }
	        return yaml.dump(this.dumpToObject(), {});
	    };
	    ASTNode.prototype.dumpToObject = function (full) {
	        if (full === void 0) { full = false; }
	        return this.dumpNode(this._node, full);
	    };
	    ASTNode.prototype.dumpNode = function (n, full) {
	        var _this = this;
	        if (full === void 0) { full = false; }
	        if (!n) {
	            return null;
	        }
	        if (n.kind == yaml.Kind.INCLUDE_REF) {
	            if (this._unit) {
	                var s = n;
	                var includePath = s.value;
	                var resolved = null;
	                try {
	                    resolved = this._unit.resolve(includePath);
	                }
	                catch (Error) {
	                }
	                if (resolved == null) {
	                    return null;
	                }
	                else if (resolved.isRAMLUnit() && this.canInclude(resolved)) {
	                    var ast = resolved.ast();
	                    if (ast) {
	                        return ast.dumpToObject(full);
	                    }
	                }
	            }
	            return null;
	        }
	        if (n.kind == yaml.Kind.SEQ) {
	            var seq = n;
	            var arr = [];
	            seq.items.forEach(function (x) { return arr.push(_this.dumpNode(x, full)); });
	            return arr;
	        }
	        if (n.kind == yaml.Kind.ANCHOR_REF) {
	            var aref = n;
	            return this.dumpNode(aref.value, full);
	        }
	        if (n.kind == yaml.Kind.MAPPING) {
	            var c = n;
	            var v = {};
	            var val = c.value;
	            var mm = this.dumpNode(val, full);
	            v["" + this.dumpNode(c.key, full)] = mm;
	            return v;
	        }
	        if (n.kind == yaml.Kind.SCALAR) {
	            var s = n;
	            var q = s.value;
	            if (s.plainScalar) {
	                q = tryParseScalar(q);
	            }
	            return q;
	        }
	        if (n.kind == yaml.Kind.MAP) {
	            var map = n;
	            var res = {};
	            if (map.mappings) {
	                map.mappings.forEach(function (x) {
	                    var ms = _this.dumpNode(x.value, full);
	                    if (ms == null) {
	                        ms = "!$$$novalue";
	                    }
	                    if ((ms + "").length > 0 || full) {
	                        res[_this.dumpNode(x.key, full) + ""] = ms;
	                    }
	                });
	            }
	            return res;
	        }
	    };
	    ASTNode.prototype.keyKind = function () {
	        if (this._node.key) {
	            return this._node.key.kind;
	        }
	        return null;
	    };
	    ASTNode.prototype._actualNode = function () {
	        return this._node;
	    };
	    ASTNode.prototype.execute = function (cmd) {
	        if (this.unit()) {
	            this.unit().project().execute(cmd);
	        }
	        else {
	            cmd.commands.forEach(function (x) {
	                switch (x.kind) {
	                    case lowlevel.CommandKind.CHANGE_VALUE:
	                        var attr = x.target;
	                        var newValue = x.value;
	                        var va = attr._actualNode();
	                        var as = attr.start();
	                        if (va.kind == yaml.Kind.MAPPING) {
	                            va.value = yaml.newScalar("" + newValue);
	                        }
	                        //this.executeTextChange(new lowlevel.TextChangeCommand(as,attr.value().length,<string>newValue,attr.unit()))
	                        return;
	                    case lowlevel.CommandKind.CHANGE_KEY:
	                        var attr = x.target;
	                        var newValue = x.value;
	                        var va = attr._actualNode();
	                        if (va.kind == yaml.Kind.MAPPING) {
	                            var sc = va.key;
	                            sc.value = newValue;
	                        }
	                        return;
	                }
	            });
	        }
	    };
	    ASTNode.prototype.updateFrom = function (n) {
	        this._node = n;
	    };
	    ASTNode.prototype.isAnnotatedScalar = function () {
	        if (this.kind() == yaml.Kind.MAPPING && this.unit()) {
	            if (this.valueKind() == yaml.Kind.MAP && this._node.value.mappings) {
	                var isScalar = this._node.value.mappings.length > 0;
	                this._node.value.mappings.forEach(function (x) {
	                    if (x.key.value === "value") {
	                        return;
	                    }
	                    if (x.key.value) {
	                        if (x.key.value.charAt(0) == '(' && x.key.value.charAt(x.key.value.length - 1) == ')') {
	                            return;
	                        }
	                    }
	                    isScalar = false;
	                });
	                return isScalar;
	            }
	        }
	        return false;
	    };
	    ASTNode.prototype.value = function (toString) {
	        if (!this._node) {
	            return "";
	        }
	        if (this._node.kind == yaml.Kind.SCALAR) {
	            //TODO WHAT IS IT IS INCLUDE ACTUALLY
	            if (!toString && ("" + this._node['valueObject'] === this._node['value'])) {
	                return this._node['valueObject'];
	            }
	            var q = this._node['value'];
	            if (!toString) {
	                if (this._node.plainScalar) {
	                    q = tryParseScalar(q);
	                }
	            }
	            return q;
	        }
	        if (this._node.kind == yaml.Kind.ANCHOR_REF) {
	            var ref = this._node;
	            return new ASTNode(ref.value, this._unit, this, null, null).value(toString);
	        }
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            var map = this._node;
	            if (map.value == null) {
	                return null;
	            }
	            if (this.isAnnotatedScalar()) {
	                var child = new ASTNode(map.value, this._unit, this, null, null);
	                var ch = child.children();
	                for (var i = 0; i < ch.length; i++) {
	                    if (ch[i].key() === "value") {
	                        return ch[i].value();
	                    }
	                }
	            }
	            return new ASTNode(map.value, this._unit, this, null, null).value(toString);
	        }
	        if (this._node.kind == yaml.Kind.INCLUDE_REF) {
	            //here we should resolve include
	            var includeString = this._node['value'];
	            var includePath = includeString;
	            var resolved = null;
	            try {
	                resolved = this._unit.resolve(includePath);
	            }
	            catch (Error) {
	                //not sure why we're returning this as a value, but that's what we do with failed units due to unknown cause below,
	                //so doing the same @Denis
	                return "can not resolve " + includePath + " due to: " + Error.message;
	            }
	            if (resolved == null) {
	                return "can not resolve " + includePath;
	            }
	            if (resolved.isRAMLUnit()) {
	                //TODO DIFFERENT DATA TYPES, inner references
	                return null;
	            }
	            var text = resolved.contents();
	            if (textutil.isMultiLineValue(text)) {
	                text = textutil.fromMutiLine(text);
	            }
	            return text;
	        }
	        if (this._node.kind == yaml.Kind.MAP) {
	            var amap = this._node;
	            if (amap.mappings.length == 1) {
	                //handle map with one member case differently
	                return new ASTNode(amap.mappings[0], this._unit, this, null, null);
	            }
	        }
	        if (this._node.kind == yaml.Kind.SEQ) {
	            var aseq = this._node;
	            if (aseq.items.length == 1 && true) {
	                //handle seq with one member case differently
	                return new ASTNode(aseq.items[0], this._unit, this, null, null).value(toString);
	            }
	        }
	        //this are only kinds which has values
	        return null;
	    };
	    ASTNode.prototype.printDetails = function (indent) {
	        var result = "";
	        if (!indent)
	            indent = "";
	        var typeName = this.kindName();
	        if (this.kind() == yaml.Kind.SCALAR) {
	            result += indent + "[" + typeName + "]" + " " + this.value() + "\n";
	        }
	        else if (this.kind() == yaml.Kind.MAPPING &&
	            this._node.value &&
	            this._node.value.kind == yaml.Kind.SCALAR) {
	            result += indent + "[" + typeName + "]" + " " + this.key()
	                + " = " + this.value() + "\n";
	        }
	        else if (this.kind() == yaml.Kind.MAPPING) {
	            result += indent + "[" + typeName + "]" + " " + this.key()
	                + " = :\n";
	            this.children().forEach(function (child) {
	                result += child.printDetails(indent + "\t");
	            });
	        }
	        else {
	            result += indent + "[" + typeName + "]" + " :\n";
	            this.children().forEach(function (child) {
	                result += child.printDetails(indent + "\t");
	            });
	        }
	        return result;
	    };
	    ASTNode.prototype.visit = function (v) {
	        this.children().forEach(function (x) {
	            if (v(x)) {
	                x.visit(v);
	            }
	        });
	    };
	    ASTNode.prototype.rawKey = function () {
	        if (!this._node) {
	            return "";
	        }
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            var map = this._node;
	            if (map.key.kind == yaml.Kind.SEQ) {
	                var items = map.key;
	                var mn = "[";
	                items.items.forEach(function (x) { return mn += x.value; });
	                return mn + "]";
	            }
	            return map.key.value;
	        }
	        //other kinds do not have keys
	        return null;
	    };
	    ASTNode.prototype.key = function () {
	        var key = this.rawKey();
	        if (key != null && util.stringEndsWith(key, '?')) {
	            key = key.substring(0, key.length - 1);
	        }
	        return key;
	    };
	    ASTNode.prototype.optional = function () {
	        var key = this.rawKey();
	        return key != null && util.stringEndsWith(key, '?');
	    };
	    ASTNode.prototype.addChild = function (n, pos) {
	        if (pos === void 0) { pos = -1; }
	        //this.show('ADD TARGET:');
	        var node = n;
	        //console.log('add-child: ' + this.kindName() + ' .add ' + node.kindName());
	        node._parent = this;
	        this._oldText = null;
	        if (this.isMap()) {
	            //console.log('pos: ' + pos);
	            var map = this.asMap();
	            if (map.mappings == null || map.mappings == undefined) {
	                map.mappings = [];
	            }
	            if (pos >= 0) {
	                map.mappings.splice(pos, 0, node.asMapping());
	            }
	            else {
	                map.mappings.push(node.asMapping());
	            }
	        }
	        else if (this.isMapping()) {
	            var mapping = this.asMapping();
	            var val = mapping.value;
	            //console.log('mapping value: ' + val);
	            if (!mapping.value && node.isMap()) {
	                mapping.value = node._actualNode();
	                return;
	            }
	            if (mapping.value && mapping.value.kind == yaml.Kind.SCALAR) {
	                // cleanup old value
	                mapping.value = null;
	                val = null;
	            }
	            if (!val) {
	                if (node.isScalar() || node.highLevelNode() && node.highLevelNode().property().getAdapter(services.RAMLPropertyParserService).isEmbedMap()) {
	                    val = yaml.newSeq();
	                }
	                else {
	                    val = yaml.newMap();
	                }
	                mapping.value = val;
	            }
	            if (val.kind == yaml.Kind.MAP) {
	                var map = val;
	                if (map.mappings == null || map.mappings == undefined) {
	                    map.mappings = [];
	                }
	                if (node.isScalar()) {
	                }
	                if (pos >= 0) {
	                    map.mappings.splice(pos, 0, node.asMapping());
	                }
	                else {
	                    map.mappings.push(node.asMapping());
	                }
	            }
	            else if (val.kind == yaml.Kind.SEQ) {
	                var seq = val;
	                if (pos >= 0) {
	                    seq.items.splice(pos, 0, node._actualNode());
	                }
	                else {
	                    seq.items.push(node._actualNode());
	                }
	            }
	            else {
	                throw new Error("Insert into mapping with " + yaml.Kind[mapping.value.kind] + " value not supported");
	            }
	        }
	        else if (this.isSeq()) {
	            var seq = this.asSeq();
	            if (pos >= 0) {
	                seq.items.splice(pos, 0, node._actualNode());
	            }
	            else {
	                seq.items.push(node._actualNode());
	            }
	        }
	        else {
	            throw new Error("Insert into " + this.kindName() + " not supported");
	        }
	    };
	    ASTNode.prototype.removeChild = function (n) {
	        this._oldText = null;
	        var node = n;
	        var ynode;
	        var index;
	        //console.log('*** REMOVE FROM: ' + this.kindName());
	        if (this.kind() == yaml.Kind.SEQ) {
	            //console.log('remove from seq');
	            var seq = this.asSeq();
	            //val = <yaml.YamlMap>((<yaml.YAMLMapping>this._node).value);
	            ynode = node._node;
	            index = seq.items.indexOf(ynode);
	            if (index > -1)
	                seq.items.splice(index, 1);
	        }
	        else if (this.kind() == yaml.Kind.MAP) {
	            //val = <yaml.YamlMap>((<yaml.YAMLMapping>this._node).value);
	            var map = this.asMap();
	            //console.log('remove from map: ' + map.mappings.length);
	            ynode = node.asMapping();
	            index = map.mappings.indexOf(ynode);
	            //console.log('  index: ' + index);
	            if (index > -1)
	                map.mappings.splice(index, 1);
	        }
	        else if (this.kind() == yaml.Kind.MAPPING) {
	            //console.log('*** REMOVE FROM MAPPING');
	            //val = <yaml.YamlMap>((<yaml.YAMLMapping>this._node).value);
	            //console.log('remove from mapping with map as value');
	            var mapping = this.asMapping();
	            //this.show("REMOVE TARGET: ***");
	            //node.show("REMOVE NODE: ***");
	            if (node._actualNode() == mapping.value) {
	                // remove right from mapping
	                //console.log('*** remove map from mapping!');
	                mapping.value = null;
	            }
	            else {
	                var map = (mapping.value);
	                ynode = node.asMapping();
	                if (map && map.mappings) {
	                    index = map.mappings.indexOf(ynode);
	                    if (index > -1)
	                        map.mappings.splice(index, 1);
	                }
	            }
	        }
	        else {
	            throw new Error("Delete from " + yaml.Kind[this.kind()] + " unsupported");
	        }
	    };
	    ASTNode.prototype.includeErrors = function () {
	        if (this._node.kind == yaml.Kind.MAPPING) {
	            var mapping = this._node;
	            if (mapping.value == null) {
	                return [];
	            }
	            return new ASTNode(mapping.value, this._unit, this, this._anchor, this._include).includeErrors();
	        }
	        var rs = [];
	        if (this._node.kind == yaml.Kind.INCLUDE_REF) {
	            var mapping = this._node;
	            if (mapping.value == null) {
	                return [];
	            }
	            var includePath = this.includePath();
	            var resolved = null;
	            try {
	                resolved = this._unit.resolve(includePath);
	            }
	            catch (Error) {
	                //known cause of failure
	                rs.push("Can not resolve " + includePath + " due to: " + Error.message);
	                return rs;
	            }
	            if (resolved == null) {
	                //unknown cause of failure
	                rs.push("Can not resolve " + includePath);
	                return rs;
	            }
	            if (resolved.isRAMLUnit()) {
	                var ast = resolved.ast();
	                if (ast) {
	                    return [];
	                }
	                else {
	                    rs.push("" + includePath + " can not be parsed");
	                }
	            }
	            else {
	            }
	        }
	        return rs;
	    };
	    ASTNode.prototype.children = function (inc, anc, inOneMemberMap) {
	        var _this = this;
	        if (inc === void 0) { inc = null; }
	        if (anc === void 0) { anc = null; }
	        if (inOneMemberMap === void 0) { inOneMemberMap = true; }
	        if (this._node == null) {
	            return []; //TODO FIXME
	        }
	        if (this.cacheChildren && this._children) {
	            return this._children;
	        }
	        var result;
	        var kind = this._node.kind;
	        if (kind == yaml.Kind.SCALAR) {
	            result = [];
	        }
	        else if (kind == yaml.Kind.MAP) {
	            var map = this._node;
	            if (map.mappings.length == 1 && !inOneMemberMap) {
	                //handle map with one member case differently
	                // q:
	                //  []
	                //   - a
	                //   - b
	                // ->
	                // q:
	                //  a
	                //  b
	                result = new ASTNode(map.mappings[0].value, this._unit, this, inc, anc, this.cacheChildren).children(null, null, true);
	            }
	            else {
	                result = map.mappings.map(function (x) { return new ASTNode(x, _this._unit, _this, anc ? anc : _this._anchor, inc ? inc : _this._include, _this.cacheChildren); });
	            }
	        }
	        else if (kind == yaml.Kind.MAPPING) {
	            var mapping = this._node;
	            if (mapping.value == null) {
	                result = [];
	            }
	            else {
	                var proxy = new ASTNode(mapping.value, this._unit, this, anc ? anc : this._anchor, inc ? inc : this._include, this.cacheChildren);
	                result = proxy.children();
	                if (proxy.includesContents()) {
	                    this.setIncludesContents(true);
	                }
	            }
	        }
	        else if (kind == yaml.Kind.SEQ) {
	            var seq = this._node;
	            result = seq.items.filter(function (x) { return x != null; }).map(function (x) { return new ASTNode(x, _this._unit, _this, anc ? anc : _this._anchor, inc ? inc : _this._include, _this.cacheChildren); });
	        }
	        else if (kind == yaml.Kind.INCLUDE_REF) {
	            if (this._unit) {
	                var includePath = this.includePath();
	                var resolved = null;
	                try {
	                    resolved = this._unit.resolve(includePath);
	                }
	                catch (Error) {
	                }
	                if (resolved == null) {
	                    result = [];
	                }
	                else if (resolved.isRAMLUnit() && this.canInclude(resolved)) {
	                    var ast = resolved.ast();
	                    if (ast) {
	                        if (this.cacheChildren) {
	                            ast = toChildCahcingNode(ast);
	                        } //else {
	                        //    ast = <ASTNode>toIncludingNode(ast);
	                        //}
	                        result = resolved.ast().children(this, null);
	                        this.setIncludesContents(true);
	                    }
	                }
	            }
	            if (!result) {
	                result = [];
	            }
	        }
	        else if (kind == yaml.Kind.ANCHOR_REF) {
	            var ref = this._node;
	            result = new ASTNode(ref.value, this._unit, this, null, null, this.cacheChildren).children();
	        }
	        else {
	            throw new Error("Should never happen; kind : " + yaml.Kind[this._node.kind]);
	        }
	        if (this.cacheChildren) {
	            this._children = result;
	        }
	        return result;
	    };
	    ASTNode.prototype.canInclude = function (unit) {
	        var includedFrom = this.includedFrom();
	        while (includedFrom != null) {
	            if (includedFrom.unit().absolutePath() == unit.absolutePath()) {
	                return false;
	            }
	            includedFrom = includedFrom.includedFrom();
	        }
	        return true;
	    };
	    ASTNode.prototype.directChildren = function (inc, anc, inOneMemberMap) {
	        var _this = this;
	        if (inc === void 0) { inc = null; }
	        if (anc === void 0) { anc = null; }
	        if (inOneMemberMap === void 0) { inOneMemberMap = true; }
	        if (this._node) {
	            switch (this._node.kind) {
	                case yaml.Kind.SCALAR:
	                    return [];
	                case yaml.Kind.MAP:
	                    {
	                        var map = this._node;
	                        if (map.mappings.length == 1 && !inOneMemberMap) {
	                            //handle map with one member case differently
	                            return new ASTNode(map.mappings[0].value, this._unit, this, inc, anc).directChildren(null, null, true);
	                        }
	                        return map.mappings.map(function (x) { return new ASTNode(x, _this._unit, _this, anc ? anc : _this._anchor, inc ? inc : _this._include); });
	                    }
	                case yaml.Kind.MAPPING:
	                    {
	                        var mapping = this._node;
	                        if (mapping.value == null) {
	                            return [];
	                        }
	                        return new ASTNode(mapping.value, this._unit, this, anc ? anc : this._anchor, inc ? inc : this._include).directChildren();
	                    }
	                case yaml.Kind.SEQ:
	                    {
	                        var seq = this._node;
	                        return seq.items.filter(function (x) { return x != null; }).map(function (x) { return new ASTNode(x, _this._unit, _this, anc ? anc : _this._anchor, inc ? inc : _this._include); });
	                    }
	                case yaml.Kind.INCLUDE_REF:
	                    {
	                        return [];
	                    }
	                case yaml.Kind.ANCHOR_REF:
	                    {
	                        return [];
	                    }
	            }
	            throw new Error("Should never happen; kind : " + yaml.Kind[this._node.kind]);
	        }
	        return [];
	    };
	    ASTNode.prototype.anchorId = function () {
	        return this._node.anchorId;
	    };
	    ASTNode.prototype.unit = function () {
	        return this._unit;
	        //if(this._unit) return this._unit;
	        //if(!this.parent()) return null;
	        //return this.parent().unit();
	    };
	    ASTNode.prototype.setUnit = function (unit) {
	        this._unit = unit;
	    };
	    ASTNode.prototype.includePath = function () {
	        var includeString = this.getIncludeString();
	        if (!includeString) {
	            return null;
	        }
	        return includeString;
	    };
	    ASTNode.prototype.includeReference = function () {
	        var includeString = this.getIncludeString();
	        if (!includeString) {
	            return null;
	        }
	        return refResolvers.getIncludeReference(includeString);
	    };
	    ASTNode.prototype.getIncludeString = function () {
	        if (this._node.kind == yaml.Kind.INCLUDE_REF) {
	            var includePath = this._node['value'];
	            return includePath;
	        }
	        else if (this._node.kind == yaml.Kind.MAPPING) {
	            var mapping = this._node;
	            if (mapping.value == null)
	                return null;
	            return new ASTNode(mapping.value, this._unit, this, null, null).getIncludeString();
	        }
	        return null;
	    };
	    ASTNode.prototype.anchoredFrom = function () {
	        return this._anchor;
	    };
	    ASTNode.prototype.includedFrom = function () {
	        return this._include;
	    };
	    ASTNode.prototype.kind = function () {
	        return this._actualNode().kind;
	    };
	    ASTNode.prototype.valueKind = function () {
	        if (this._node.kind != yaml.Kind.MAPPING) {
	            return null;
	        }
	        var map = this._node;
	        if (!map.value) {
	            return null;
	        }
	        return map.value.kind;
	    };
	    ASTNode.prototype.valueKindName = function () {
	        var kind = this.valueKind();
	        return kind != undefined ? yaml.Kind[kind] : null;
	    };
	    ASTNode.prototype.kindName = function () {
	        return yaml.Kind[this.kind()];
	    };
	    ASTNode.prototype.indent = function (lev, str) {
	        if (str === void 0) { str = ''; }
	        var leading = '';
	        //leading += '[' + lev + ']';
	        for (var i = 0; i < lev; i++)
	            leading += '  ';
	        return leading + str;
	    };
	    ASTNode.prototype.replaceNewlines = function (s, rep) {
	        if (rep === void 0) { rep = null; }
	        var res = '';
	        for (var i = 0; i < s.length; i++) {
	            var ch = s[i];
	            if (ch == '\r')
	                ch = rep == null ? '\\r' : rep;
	            if (ch == '\n')
	                ch = rep == null ? '\\n' : rep;
	            res += ch;
	        }
	        return res;
	    };
	    ASTNode.prototype.shortText = function (unittext, maxlen) {
	        if (maxlen === void 0) { maxlen = 50; }
	        var elen = this.end() - this.start();
	        var len = elen;
	        //var len = Math.min(elen,50);
	        var unit = this.unit();
	        if (!unittext && unit) {
	            unittext = unit.contents();
	        }
	        var text;
	        if (!unittext) {
	            text = '[no-unit]';
	        }
	        else {
	            var s = unittext;
	            text = s ? s.substring(this.start(), this.end()) : '[no-text]';
	        }
	        text = "[" + this.start() + ".." + this.end() + "] " + elen + " // " + text + ' //';
	        if (len < elen)
	            text += '...';
	        text = this.replaceNewlines(text);
	        return text;
	    };
	    ASTNode.prototype.nodeShortText = function (node, unittext, maxlen) {
	        if (maxlen === void 0) { maxlen = 50; }
	        var elen = node.endPosition - node.startPosition;
	        var len = elen;
	        //var len = Math.min(elen,50);
	        var unit = this.unit();
	        if (!unittext && unit) {
	            unittext = unit.contents();
	        }
	        var text;
	        if (!unittext) {
	            text = '[no-unit]';
	        }
	        else {
	            var s = unittext;
	            text = s ? s.substring(node.startPosition, node.endPosition) : '[no-text]';
	        }
	        text = "[" + node.startPosition + ".." + node.endPosition + "] " + elen + " // " + text + ' //';
	        if (len < elen)
	            text += '...';
	        text = this.replaceNewlines(text);
	        return text;
	    };
	    ASTNode.prototype.show = function (message, lev, text) {
	        if (message === void 0) { message = null; }
	        if (lev === void 0) { lev = 0; }
	        if (text === void 0) { text = null; }
	        if (message && lev == 0) {
	            console.log(message);
	        }
	        var children = this.children();
	        var desc = this.kindName();
	        var val = this._actualNode().value;
	        if (this.kind() == yaml.Kind.MAPPING) {
	            desc += '[' + this._actualNode().key.value + ']';
	        }
	        if (val) {
	            desc += "/" + yaml.Kind[val.kind];
	        }
	        else
	            desc += "";
	        if (children.length == 0) {
	            //desc += "/" + this.value();
	            console.log(this.indent(lev) + desc + " // " + this.shortText(text));
	            if (this.isMapping() && this.asMapping().value) {
	                console.log(this.indent(lev + 1) + '// ' + this.valueKindName() + ': ' + this.nodeShortText(this.asMapping().value, text));
	            }
	        }
	        else {
	            console.log(this.indent(lev) + desc + " { // " + this.shortText(text));
	            if (this.isMapping() && this.asMapping().value) {
	                console.log(this.indent(lev + 1) + '// ' + this.valueKindName() + ': ' + this.nodeShortText(this.asMapping().value, text));
	            }
	            children.forEach(function (node) {
	                var n = node;
	                n.show(null, lev + 1, text);
	            });
	            console.log(this.indent(lev) + '}');
	        }
	    };
	    ASTNode.prototype.showParents = function (message, lev) {
	        if (lev === void 0) { lev = 0; }
	        if (message && lev == 0) {
	            console.log(message);
	        }
	        var depth = 0;
	        if (this.parent()) {
	            var n = this.parent();
	            depth = n.showParents(null, lev + 1);
	        }
	        var desc = this.kindName();
	        var val = this._actualNode().value;
	        if (val)
	            desc += "/" + yaml.Kind[val.kind];
	        else
	            desc += "/null";
	        console.log(this.indent(depth) + desc + " // " + this.shortText(null));
	        return depth + 1;
	    };
	    ASTNode.prototype.inlined = function (kind) {
	        return kind == yaml.Kind.SCALAR || kind == yaml.Kind.INCLUDE_REF;
	    };
	    ASTNode.prototype.markupNode = function (xbuf, node, lev, json) {
	        if (json === void 0) { json = false; }
	        var start = xbuf.text.length;
	        //console.log('node: ' + node);
	        switch (node.kind) {
	            case yaml.Kind.MAP:
	                if (json)
	                    xbuf.append('{');
	                var mappings = node.mappings;
	                //console.log('map: ' + mappings.length);
	                for (var i = 0; i < mappings.length; i++) {
	                    if (json && i > 0)
	                        xbuf.append(', ');
	                    this.markupNode(xbuf, mappings[i], lev, json);
	                }
	                if (json)
	                    xbuf.append('}');
	                break;
	            case yaml.Kind.SEQ:
	                var items = node.items;
	                //console.log('seq: ' + items.length);
	                for (var i = 0; i < items.length; i++) {
	                    xbuf.append(this.indent(lev, '- '));
	                    //this.markupNode(xindent, pos+xbuf.text.length-(lev+1)*2, items[i], lev+1, xbuf);
	                    this.markupNode(xbuf, items[i], lev + 1, json);
	                }
	                break;
	            case yaml.Kind.MAPPING:
	                var mapping = node;
	                var val = mapping.value;
	                //console.log('mapping: ' + mapping.key.value + ' ' + val.kind);
	                if (json) {
	                    if (!val) {
	                        xbuf.append('\n');
	                        break;
	                    }
	                    xbuf.append(mapping.key.value);
	                    xbuf.append(': ');
	                    if (val.kind == yaml.Kind.SCALAR) {
	                        var sc = val;
	                        xbuf.append(sc.value);
	                    }
	                    else if (val.kind == yaml.Kind.MAP) {
	                        //var mp = <yaml.YamlMap>val;
	                        this.markupNode(xbuf, mapping.value, lev + 1, json);
	                    }
	                    else {
	                        throw new Error("markup not implemented: " + yaml.Kind[val.kind]);
	                    }
	                    break;
	                }
	                xbuf.addWithIndent(lev, mapping.key.value + ':');
	                if (!val) {
	                    xbuf.append('\n');
	                    break;
	                }
	                if (val.kind == yaml.Kind.SCALAR) {
	                    var sc = val;
	                }
	                //xbuf.append(this.indent(lev, mapping.key.value + ':'));
	                if (mapping.value) {
	                    xbuf.append(this.inlined(mapping.value.kind) ? ' ' : '\n');
	                    this.markupNode(xbuf, mapping.value, lev + 1, json);
	                }
	                else {
	                    xbuf.append('\n');
	                }
	                //console.log('xbuf: ' + xbuf);
	                break;
	            case yaml.Kind.SCALAR:
	                var sc = node;
	                if (textutil.isMultiLine(sc.value)) {
	                    xbuf.append('|\n');
	                    var lines = splitOnLines(sc.value);
	                    for (var i = 0; i < lines.length; i++) {
	                        xbuf.append(this.indent(lev, lines[i]));
	                    }
	                    xbuf.append('\n');
	                }
	                else {
	                    xbuf.append(sc.value + '\n');
	                }
	                //console.log('SCALAR: ' + textutil.replaceNewlines(sc.value));
	                break;
	            case yaml.Kind.INCLUDE_REF:
	                var ref = node;
	                xbuf.append('!include ' + ref.value + '\n');
	                break;
	            default:
	                throw new Error('Unknown node kind: ' + yaml.Kind[node.kind]);
	        }
	        while (start < xbuf.text.length && xbuf.text[start] == ' ')
	            start++;
	        node.startPosition = start;
	        node.endPosition = xbuf.text.length;
	    };
	    ASTNode.prototype.markup = function (json) {
	        if (json === void 0) { json = false; }
	        var buf = new MarkupIndentingBuffer('');
	        this.markupNode(buf, this._actualNode(), 0, json);
	        return buf.text;
	    };
	    ASTNode.prototype.root = function () {
	        var node = this;
	        while (node.parent()) {
	            var p = node.parent();
	            //if(p.isValueInclude()) break; // stop on include
	            node = p;
	        }
	        return node;
	    };
	    ASTNode.prototype.parentOfKind = function (kind) {
	        var p = this.parent();
	        while (p) {
	            if (p.kind() == kind)
	                return p;
	            p = p.parent();
	        }
	        return null;
	    };
	    ASTNode.prototype.find = function (name) {
	        var found = null;
	        //console.log('Looking for: ' + name);
	        this.directChildren().forEach(function (y) {
	            if (y.key() && y.key() == name) {
	                if (!found)
	                    found = y;
	            }
	        });
	        return found;
	    };
	    ASTNode.prototype.shiftNodes = function (offset, shift, exclude) {
	        this.directChildren().forEach(function (x) {
	            if (exclude && exclude.start() == x.start() && exclude.end() == x.end()) {
	            }
	            else {
	                var m = x.shiftNodes(offset, shift, exclude);
	            }
	        });
	        if (exclude && exclude.start() == this.start() && exclude.end() == this.end()) {
	        }
	        else {
	            var yaNode = this._actualNode();
	            if (yaNode)
	                innerShift(offset, yaNode, shift);
	        }
	        return null;
	    };
	    ASTNode.prototype.isMap = function () {
	        return this.kind() == yaml.Kind.MAP;
	    };
	    ASTNode.prototype.isMapping = function () {
	        return this.kind() == yaml.Kind.MAPPING;
	    };
	    ASTNode.prototype.isSeq = function () {
	        return this.kind() == yaml.Kind.SEQ;
	    };
	    ASTNode.prototype.isScalar = function () {
	        return this.kind() == yaml.Kind.SCALAR;
	    };
	    ASTNode.prototype.asMap = function () {
	        if (!this.isMap())
	            throw new Error("map expected instead of " + this.kindName());
	        return (this._actualNode());
	    };
	    ASTNode.prototype.asMapping = function () {
	        if (!this.isMapping())
	            throw new Error("maping expected instead of " + this.kindName());
	        return (this._actualNode());
	    };
	    ASTNode.prototype.asSeq = function () {
	        if (!this.isSeq())
	            throw new Error("seq expected instead of " + this.kindName());
	        return (this._actualNode());
	    };
	    ASTNode.prototype.asScalar = function () {
	        if (!this.isScalar())
	            throw new Error("scalar expected instead of " + this.kindName());
	        return (this._actualNode());
	    };
	    ASTNode.prototype.isValueSeq = function () {
	        return this.valueKind() == yaml.Kind.SEQ;
	    };
	    ASTNode.prototype.isValueMap = function () {
	        return this.valueKind() == yaml.Kind.MAP;
	    };
	    ASTNode.prototype.isValueInclude = function () {
	        return this.valueKind() == yaml.Kind.INCLUDE_REF;
	    };
	    ASTNode.prototype.isValueScalar = function () {
	        return this.valueKind() == yaml.Kind.SCALAR;
	    };
	    ASTNode.prototype.valueAsSeq = function () {
	        if (!this.isMapping())
	            throw new Error("mapping expected instead of " + this.kindName());
	        if (this.valueKind() != yaml.Kind.SEQ)
	            throw new Error("mappng/seq expected instead of mapping/" + this.kindName());
	        return (this.asMapping().value);
	    };
	    ASTNode.prototype.valueAsMap = function () {
	        if (!this.isMapping())
	            throw new Error("mapping expected instead of " + this.kindName());
	        if (this.valueKind() != yaml.Kind.MAP)
	            throw new Error("mappng/map expected instead of mapping/" + this.kindName());
	        return (this.asMapping().value);
	    };
	    ASTNode.prototype.valueAsScalar = function () {
	        if (!this.isMapping())
	            throw new Error("mapping expected instead of " + this.kindName());
	        if (this.valueKind() != yaml.Kind.SCALAR)
	            throw new Error("mappng/scalar expected instead of mapping/" + this.kindName());
	        return (this.asMapping().value);
	    };
	    ASTNode.prototype.valueAsInclude = function () {
	        if (!this.isMapping())
	            throw new Error("mapping expected instead of " + this.kindName());
	        if (this.valueKind() != yaml.Kind.INCLUDE_REF)
	            throw new Error("mappng/include expected instead of mapping/" + this.kindName());
	        return (this.asMapping().value);
	    };
	    ASTNode.prototype.text = function (unitText) {
	        if (unitText === void 0) { unitText = null; }
	        if (!unitText) {
	            if (!this.unit())
	                return '[no-text]';
	            unitText = this.unit().contents();
	        }
	        return unitText.substring(this.start(), this.end());
	    };
	    ASTNode.prototype.copy = function () {
	        var yn = copyNode(this._actualNode());
	        return new ASTNode(yn, this._unit, this._parent, this._anchor, this._include, this._includesContents);
	    };
	    ASTNode.prototype.nodeDefinition = function () {
	        return getDefinitionForLowLevelNode(this);
	    };
	    return ASTNode;
	}());
	exports.ASTNode = ASTNode;
	(function (InsertionPointType) {
	    InsertionPointType[InsertionPointType["NONE"] = 0] = "NONE";
	    InsertionPointType[InsertionPointType["START"] = 1] = "START";
	    InsertionPointType[InsertionPointType["END"] = 2] = "END";
	    InsertionPointType[InsertionPointType["POINT"] = 3] = "POINT";
	})(exports.InsertionPointType || (exports.InsertionPointType = {}));
	var InsertionPointType = exports.InsertionPointType;
	var InsertionPoint = (function () {
	    function InsertionPoint(type, point) {
	        if (point === void 0) { point = null; }
	        this.type = type;
	        this.point = point;
	    }
	    InsertionPoint.after = function (point) {
	        return new InsertionPoint(InsertionPointType.POINT, point);
	    };
	    InsertionPoint.atStart = function () {
	        return new InsertionPoint(InsertionPointType.START);
	    };
	    InsertionPoint.atEnd = function () {
	        return new InsertionPoint(InsertionPointType.END);
	    };
	    InsertionPoint.node = function () {
	        return new InsertionPoint(InsertionPointType.NONE);
	    };
	    InsertionPoint.prototype.show = function (msg) {
	        if (msg) {
	            console.log(msg);
	            console.log('  insertion point type: ' + InsertionPointType[this.type]);
	        }
	        else {
	            console.log('insertion point type: ' + InsertionPointType[this.type]);
	        }
	        if (this.type == InsertionPointType.POINT && this.point) {
	            this.point.show();
	        }
	    };
	    return InsertionPoint;
	}());
	exports.InsertionPoint = InsertionPoint;
	function createNode(key, parent) {
	    //console.log('create node: ' + key);
	    var node = yaml.newMapping(yaml.newScalar(key), yaml.newMap());
	    return new ASTNode(node, null, parent, null, null);
	}
	exports.createNode = createNode;
	function createMap(mappings) {
	    //console.log('create node: ' + key);
	    var node = yaml.newMap(mappings);
	    return new ASTNode(node, null, null, null, null);
	}
	exports.createMap = createMap;
	function createScalar(value) {
	    var node = yaml.newScalar(value);
	    return new ASTNode(node, null, null, null, null);
	}
	exports.createScalar = createScalar;
	function createSeq(sn, parent, unit) {
	    return new ASTNode(sn, unit, parent, null, null);
	}
	exports.createSeq = createSeq;
	/*
	export function createMappingWithMap(key:string, map: yaml.YAMLNode){
	    //console.log('create node: ' + key);
	    var node:yaml.YAMLNode=yaml.newMapping(yaml.newScalar(key),map);
	    return new ASTNode(node,null,null,null,null);
	}

	export function createMap(){
	    //console.log('create node: ' + key);
	    var node:yaml.YAMLNode=yaml.newMap();
	    return new ASTNode(node,null,null,null,null);
	}
	*/
	function createSeqNode(key) {
	    var node = yaml.newMapping(yaml.newScalar(key), yaml.newItems());
	    return new ASTNode(node, null, null, null, null);
	}
	exports.createSeqNode = createSeqNode;
	function createMapNode(key) {
	    var node = yaml.newMapping(yaml.newScalar(key), yaml.newMap());
	    return new ASTNode(node, null, null, null, null);
	}
	exports.createMapNode = createMapNode;
	function createMapping(key, v) {
	    //console.log('create mapping: ' + key);
	    var node = yaml.newMapping(yaml.newScalar(key), yaml.newScalar(v));
	    return new ASTNode(node, null, null, null, null);
	}
	exports.createMapping = createMapping;
	function toChildCahcingNode(node) {
	    if (!(node instanceof ASTNode)) {
	        return null;
	    }
	    var astNode = node;
	    var result = new ASTNode(astNode.yamlNode(), astNode.unit(), null, null, null, true);
	    result._errors = astNode._errors;
	    return result;
	}
	exports.toChildCahcingNode = toChildCahcingNode;
	function toIncludingNode(node) {
	    if (!(node instanceof ASTNode)) {
	        return null;
	    }
	    var astNode = node;
	    var result = new ASTNode(astNode.yamlNode(), astNode.unit(), null, null, null, false);
	    result._errors = astNode._errors;
	    return result;
	}
	exports.toIncludingNode = toIncludingNode;
	function getDefinitionForLowLevelNode(node) {
	    var hl = node.highLevelNode();
	    if (hl) {
	        return hl.definition();
	    }
	    var parent = node.parent();
	    if (!parent) {
	        return null;
	    }
	    var key = node.key();
	    if (!key) {
	        return null;
	    }
	    var parentDef = parent.nodeDefinition();
	    if (!parentDef) {
	        return null;
	    }
	    if (!parentDef.property) {
	        return null;
	    }
	    var prop = parentDef.property(key);
	    if (!prop) {
	        return null;
	    }
	    return prop.range();
	}
	exports.getDefinitionForLowLevelNode = getDefinitionForLowLevelNode;
	function fetchMasterReference(unit, map, errors, lMap) {
	    if (!unit.isOverlayOrExtension())
	        return;
	    var masterReferenceNode = unit.getMasterReferenceNode();
	    if (!masterReferenceNode)
	        return;
	    var masterReferenceText = masterReferenceNode.value();
	    if (!masterReferenceText)
	        return;
	    var unitPath = path.dirname(unit.absolutePath());
	    var absIncludePath = toAbsolutePath(unitPath, masterReferenceText);
	    if (map[absIncludePath]) {
	        return;
	    }
	    if (errors[absIncludePath]) {
	        masterReferenceNode.errors().push(new Error(errors[masterReferenceText]));
	        return;
	    }
	    var arr = lMap[absIncludePath];
	    if (!arr) {
	        arr = [];
	        lMap[absIncludePath] = arr;
	    }
	    arr.push(masterReferenceNode);
	}
	function fetchIncludesAndMasterAsync(project, apiPath) {
	    var map = {};
	    var errors = {};
	    var nodes = [];
	    var units = [];
	    var processUnits = function (ind) {
	        var refs = [];
	        var lMap = {};
	        while (ind < units.length) {
	            var unit = units[ind];
	            var unitPath = path.dirname(unit.absolutePath());
	            fetchMasterReference(unit, map, errors, lMap);
	            var includeNodes = unit.getIncludeNodes();
	            includeNodes.forEach(function (x) {
	                var ip = x.includePath();
	                var includeReference = refResolvers.getIncludeReference(ip);
	                if (includeReference) {
	                    refs.push(project.resolveAsync(unit.absolutePath(), ip));
	                    return;
	                }
	                if (!ip)
	                    return;
	                var absIncludePath = toAbsolutePath(unitPath, ip);
	                if (map[absIncludePath]) {
	                    return;
	                }
	                if (errors[absIncludePath]) {
	                    if (x.errors) {
	                        x.errors().push(new Error(errors[ip]));
	                        return;
	                    }
	                }
	                var arr = lMap[absIncludePath];
	                if (!arr) {
	                    arr = [];
	                    lMap[absIncludePath] = arr;
	                }
	                arr.push(x);
	            });
	            ind++;
	        }
	        var unitPaths = Object.keys(lMap);
	        if (refs.length == 0 && unitPaths.length == 0) {
	            return Promise.resolve();
	        }
	        var promises = refs;
	        unitPaths.forEach(function (unitPath) {
	            promises.push(project.unitAsync(unitPath, true).then(function (x) {
	                if (!map[x.absolutePath()]) {
	                    map[x.absolutePath()] = true;
	                    units.push(x);
	                }
	            }, function (x) {
	                lMap[unitPath].forEach(function (node) {
	                    if (node.errors) {
	                        node.errors().push(new Error(x));
	                    }
	                });
	                errors[unitPath] = x;
	                project.failedUnits[unitPath] = x;
	            }));
	        });
	        return Promise.all(promises).then(function (x) {
	            return processUnits(ind);
	        });
	    };
	    return project.unitAsync(apiPath).then(function (x) {
	        units.push(x);
	        map[x.absolutePath()] = true;
	        return processUnits(0);
	    }).then(function (x) {
	        return units.length > 0 ? units[0] : null;
	    });
	}
	exports.fetchIncludesAndMasterAsync = fetchIncludesAndMasterAsync;
	function toAbsolutePath(rootPath, relPath) {
	    if (isWebPath(relPath)) {
	        return relPath;
	    }
	    var apath;
	    if (isWebPath(rootPath)) {
	        var rp = util.stringEndsWith(rootPath, "/") ? rootPath : rootPath + "/";
	        apath = URL.resolve(rp, relPath);
	    }
	    else {
	        apath = path.resolve(rootPath, relPath);
	    }
	    return apath;
	}
	function isWebPath(str) {
	    if (str == null)
	        return false;
	    return util.stringStartsWith(str, "http://") || util.stringStartsWith(str, "https://");
	}
	//# sourceMappingURL=jsyaml2lowLevel.js.map

/***/ },
/* 24 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../../typings/main.d.ts" />
	"use strict";
	var HttpResponse = __webpack_require__(57);
	__webpack_require__(58);
	__webpack_require__(65);
	var lru = __webpack_require__(66);
	var globalCache = lru(50);
	function hasAsyncRequests() {
	    return Object.keys(notifies).length > 0;
	}
	exports.hasAsyncRequests = hasAsyncRequests;
	function addLoadCallback(listener) {
	    nlisteners.push(listener);
	}
	exports.addLoadCallback = addLoadCallback;
	var nlisteners = [];
	var notifies = {};
	function addNotify(url) {
	    notifies[url] = true;
	}
	exports.addNotify = addNotify;
	function removeNotity(url) {
	    delete notifies[url];
	    nlisteners.forEach(function (x) { return x(url); });
	}
	exports.removeNotity = removeNotity;
	function isWaitingFor(url) {
	    return notifies[url] ? true : false;
	}
	exports.isWaitingFor = isWaitingFor;
	function set(url, content) {
	    globalCache.set(url, content);
	}
	exports.set = set;
	function get(url) {
	    return globalCache.get(url);
	}
	exports.get = get;
	//# sourceMappingURL=resourceRegistry.js.map

/***/ },
/* 25 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../../typings/main.d.ts" />
	"use strict";
	var _ = __webpack_require__(61);
	var DOMParser = __webpack_require__(67).DOMParser;
	var base64 = __webpack_require__(60);
	/**
	 * Gets pure include path portion from the complete include.
	 * Does not include the reference part.
	 * @param includeString
	 */
	function getIncludePath(includeString) {
	    if (!includeString) {
	        return includeString;
	    }
	    var index = includeString.indexOf("#");
	    if (index == -1)
	        return includeString;
	    return includeString.substring(0, index);
	}
	exports.getIncludePath = getIncludePath;
	/**
	 * Gets reference portion of the include string and returns it as
	 * an array of segments. Returns null of no reference is contained in the include.
	 * @param includeString
	 */
	function getIncludeReference(includeString) {
	    if (!includeString) {
	        return null;
	    }
	    if (typeof includeString != "string") {
	        includeString = "" + includeString;
	    }
	    var index = includeString.indexOf("#");
	    if (index == -1)
	        return null;
	    var referenceString = index == includeString.length - 1 ? "" : includeString.substring(index + 1, includeString.length);
	    var segments = referenceString.split("/");
	    if (segments.length == 0)
	        return null;
	    if (segments[0].trim() == "") {
	        segments.splice(0, 1);
	    }
	    return new IncludeReferenceImpl(referenceString, getIncludePath(includeString), segments);
	}
	exports.getIncludeReference = getIncludeReference;
	/**
	 * Factory method returning all include reference resolvers, registered in the system.
	 */
	function getIncludeReferenceResolvers() {
	    return [new JSONResolver(), new XMLResolver()];
	}
	exports.getIncludeReferenceResolvers = getIncludeReferenceResolvers;
	/**
	 * Checks all resolvers, finds the suitable one, resolves the reference and returns the result
	 * of resolving. Returns null if no suitable resolver is found or resolver itself fails to resolve.
	 * @param includeString - complete include string
	 * @param content - include contents
	 */
	function resolveContents(includeString, content) {
	    if (!includeString) {
	        return content;
	    }
	    var reference = getIncludeReference(includeString);
	    if (!reference) {
	        return content;
	    }
	    var includePath = getIncludePath(includeString);
	    return resolve(includePath, reference, content).content;
	}
	exports.resolveContents = resolveContents;
	/**
	 * Checks all resolvers, finds the suitable one, resolves the reference and returns the result
	 * of resolving. Returns null if no suitable resolver is found or resolver itself fails to resolve.
	 * @param includePath
	 * @param includeReference
	 * @param content
	 */
	function resolve(includePath, includeReference, content) {
	    var resolver = _.find(getIncludeReferenceResolvers(), function (currentResolver) { return currentResolver.isApplicable(includePath, content); });
	    if (!resolver)
	        return {
	            content: content,
	            validation: []
	        };
	    return resolver.resolveReference(content, includeReference);
	}
	exports.resolve = resolve;
	function completeReference(includePath, includeReference, content) {
	    if (!content) {
	        return [];
	    }
	    var resolver = _.find(getIncludeReferenceResolvers(), function (currentResolver) { return currentResolver.isApplicable(includePath, content); });
	    if (!resolver)
	        return [];
	    return resolver.completeReference(content, includeReference);
	}
	exports.completeReference = completeReference;
	var IncludeReferenceImpl = (function () {
	    function IncludeReferenceImpl(originalString, includePath, segments) {
	        this.includePath = includePath;
	        this.segments = segments;
	        this.originalString = originalString;
	    }
	    IncludeReferenceImpl.prototype.getIncludePath = function () {
	        return this.includePath;
	    };
	    IncludeReferenceImpl.prototype.getFragments = function () {
	        return this.segments;
	    };
	    IncludeReferenceImpl.prototype.asString = function () {
	        return this.originalString;
	    };
	    IncludeReferenceImpl.prototype.encodedName = function (withExtention) {
	        if (withExtention === void 0) { withExtention = true; }
	        return base64(this.includePath + '/' + this.asString()) + (withExtention ? this.includePath.substring(this.includePath.lastIndexOf('.')) : '');
	    };
	    return IncludeReferenceImpl;
	}());
	var JSONResolver = (function () {
	    function JSONResolver() {
	    }
	    JSONResolver.prototype.isApplicable = function (includePath, content) {
	        return includePath && (endsWith(includePath.trim(), '.js') || endsWith(includePath.trim(), '.json'));
	    };
	    JSONResolver.prototype.resolveReference = function (content, reference) {
	        try {
	            var resultJson = {};
	            resultJson['$ref'] = reference.getIncludePath() + '#' + reference.asString();
	            return {
	                content: JSON.stringify(resultJson, null, 2),
	                validation: []
	            };
	        }
	        catch (Error) {
	            console.log(Error);
	        }
	        return {
	            content: content,
	            validation: []
	        };
	    };
	    JSONResolver.prototype.completeReference = function (content, reference) {
	        try {
	            var jsonRoot = JSON.parse(content);
	            var fragments = reference.getFragments();
	            if (!fragments || fragments.length == 0) {
	                return this.getChildren(jsonRoot);
	            }
	            var currentJSON = jsonRoot;
	            var emptyPrefixCompletion = reference.asString().lastIndexOf("/") == reference.asString().length - 1;
	            var limit = emptyPrefixCompletion ? fragments.length : fragments.length - 1;
	            for (var i = 0; i < fragments.length - 1; i++) {
	                var fragment = fragments[i];
	                currentJSON = this.findChild(currentJSON, fragment);
	                if (!currentJSON) {
	                    return [];
	                }
	            }
	            if (emptyPrefixCompletion) {
	                return this.getChildren(currentJSON);
	            }
	            else {
	                var lastPrefix = fragments[fragments.length - 1];
	                var result = [];
	                this.getChildren(currentJSON).forEach(function (child) {
	                    if (child.indexOf(lastPrefix) == 0) {
	                        result.push(child);
	                    }
	                });
	                return result;
	            }
	        }
	        catch (Error) {
	            console.log(Error);
	        }
	        return [];
	    };
	    JSONResolver.prototype.findChild = function (jsonObject, fragment) {
	        var decoded = fragment.replace('~1', '/');
	        decoded = fragment.replace('~0', '~');
	        return jsonObject[decoded];
	    };
	    JSONResolver.prototype.getChildren = function (jsonObject) {
	        return Object.keys(jsonObject);
	    };
	    return JSONResolver;
	}());
	var XMLResolver = (function () {
	    function XMLResolver() {
	    }
	    XMLResolver.prototype.isApplicable = function (includePath, content) {
	        return includePath && (endsWith(includePath.trim(), '.xml') || endsWith(includePath.trim(), '.xsd'));
	    };
	    XMLResolver.prototype.resolveReference = function (content, reference) {
	        try {
	            var doc = new DOMParser().parseFromString(content);
	            var requestedName = reference.asString();
	            var uniqueName = reference.encodedName(false);
	            var schema = elementChildrenByName(doc, 'xs:schema')[0];
	            var elements = elementChildrenByName(schema, 'xs:element');
	            var types = elementChildrenByName(schema, 'xs:complexType');
	            var canBeElement = _.find(elements, function (element) { return element.getAttribute('name') === requestedName; });
	            var canBeType = canBeElement ? _.find(types, function (type) { return type.getAttribute('name') === canBeElement.getAttribute('type'); }) : _.find(types, function (type) { return type.getAttribute('name') === requestedName; });
	            var element = doc.createElement('xs:element');
	            element.setAttribute('name', uniqueName);
	            if (canBeType) {
	                element.setAttribute('type', canBeType.getAttribute('name'));
	            }
	            if (canBeElement) {
	                element.setAttribute('originalname', canBeElement.getAttribute('name'));
	            }
	            element.setAttribute('requestedname', requestedName);
	            element.setAttribute('extraelement', 'true');
	            schema.appendChild(element);
	            return {
	                content: doc.toString(),
	                validation: []
	            };
	        }
	        catch (throwable) {
	            console.log(throwable);
	        }
	        return {
	            content: content,
	            validation: []
	        };
	    };
	    XMLResolver.prototype.completeReference = function (content, reference) {
	        try {
	            var doc = new DOMParser().parseFromString(content);
	            var result = [];
	            var schema = elementChildrenByName(doc, 'xs:schema')[0];
	            var elements = elementChildrenByName(schema, 'xs:element');
	            var types = elementChildrenByName(schema, 'xs:complexType');
	            elements.forEach(function (element) { return result.push(element.getAttribute('name')); });
	            types.forEach(function (type) { return result.push(type.getAttribute('name')); });
	            var emptyPrefixCompletion = reference.asString().trim().length === 0;
	            if (emptyPrefixCompletion) {
	                return result;
	            }
	            else {
	                return result.filter(function (value) { return value.indexOf(reference.asString()) === 0; });
	            }
	        }
	        catch (exception) {
	            return [];
	        }
	    };
	    return XMLResolver;
	}());
	function endsWith(input, ends) {
	    if (!input) {
	        return false;
	    }
	    if (!ends) {
	        return false;
	    }
	    return input.lastIndexOf(ends) === (input.length - ends.length);
	}
	function elementChildrenByName(parent, tagName) {
	    var elements = parent.getElementsByTagName(tagName);
	    var result = [];
	    for (var i = 0; i < elements.length; i++) {
	        var child = elements[i];
	        if (child.parentNode === parent) {
	            result.push(child);
	        }
	    }
	    return result;
	}
	//# sourceMappingURL=includeRefResolvers.js.map

/***/ },
/* 26 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var hl = __webpack_require__(10);
	var hlimpl = __webpack_require__(16);
	var util = __webpack_require__(43);
	var proxy = __webpack_require__(37);
	var RamlWrapperImpl = __webpack_require__(20);
	var RamlWrapper08Impl = __webpack_require__(46);
	var factory10 = __webpack_require__(41);
	var factory08 = __webpack_require__(42);
	var wrapperHelper = __webpack_require__(32);
	var wrapperHelper08 = __webpack_require__(47);
	var pluralize = __webpack_require__(70);
	var _ = __webpack_require__(61);
	var changeCase = __webpack_require__(71);
	function expandTraitsAndResourceTypes(api) {
	    if (!(api instanceof RamlWrapperImpl.ApiImpl || api instanceof RamlWrapper08Impl.ApiImpl)) {
	        return null;
	    }
	    return new TraitsAndResourceTypesExpander()
	        .expandTraitsAndResourceTypes(api);
	}
	exports.expandTraitsAndResourceTypes = expandTraitsAndResourceTypes;
	function mergeAPIs(masterUnit, extensionsAndOverlays, mergeMode) {
	    var masterApi = hlimpl.fromUnit(masterUnit);
	    if (!masterApi)
	        throw new Error("couldn't load api from " + masterUnit.absolutePath());
	    if (!extensionsAndOverlays || extensionsAndOverlays.length == 0) {
	        return masterApi;
	    }
	    var highLevelNodes = [];
	    for (var i = 0; i < extensionsAndOverlays.length; i++) {
	        var unit = extensionsAndOverlays[i];
	        var hlNode = hlimpl.fromUnit(unit);
	        if (!hlNode) {
	            throw new Error("couldn't load api from " + unit.absolutePath());
	        }
	        highLevelNodes.push(hlNode);
	    }
	    var lastExtensionOrOverlay = mergeHighLevelNodes(masterApi, highLevelNodes, mergeMode);
	    return lastExtensionOrOverlay;
	}
	exports.mergeAPIs = mergeAPIs;
	function mergeHighLevelNodes(masterApi, highLevelNodes, mergeMode) {
	    var currentMaster = masterApi;
	    highLevelNodes.forEach(function (currentApi) {
	        currentApi.overrideMaster(currentMaster);
	        currentApi.setMergeMode(mergeMode);
	        currentMaster = currentApi;
	    });
	    return currentMaster;
	}
	;
	var TraitsAndResourceTypesExpander = (function () {
	    function TraitsAndResourceTypesExpander() {
	    }
	    TraitsAndResourceTypesExpander.prototype.expandTraitsAndResourceTypes = function (api) {
	        var _this = this;
	        this.ramlVersion = api.highLevel().definition().universe().version();
	        var factory = this.ramlVersion == "RAML10" ? factory10 : factory08;
	        var isRAML1 = api instanceof RamlWrapperImpl.ApiImpl;
	        this.globalTraits = this.ramlVersion == "RAML10"
	            ? wrapperHelper.allTraits(api)
	            : wrapperHelper08.allTraits(api);
	        this.globalResourceTypes = this.ramlVersion == "RAML10"
	            ? wrapperHelper.allResourceTypes(api)
	            : wrapperHelper08.allResourceTypes(api);
	        //if ((!traits || traits.length == 0) && (!resourceTypes || resourceTypes.length == 0)) {
	        //    return api;
	        //}
	        if (this.globalTraits.length == 0 && this.globalResourceTypes.length == 0) {
	            return api;
	        }
	        var hlNode = this.createHighLevelNode(api.highLevel());
	        var result = factory.buildWrapperNode(hlNode);
	        result.setAttributeDefaults(api.getDefaultsCalculator().isEnabled());
	        this.traitMap = {};
	        this.resourceTypeMap = {};
	        result.highLevel().setMergeMode(api.highLevel().getMergeMode());
	        var resources = result.resources();
	        resources.forEach(function (x) { return _this.processResource(x); });
	        return result;
	    };
	    TraitsAndResourceTypesExpander.prototype.getTemplate = function (name, context, cache, globalList) {
	        var unitPath = context.lowLevel().unit().path();
	        var unitCache = cache[unitPath];
	        if (!unitCache) {
	            unitCache = {};
	            cache[unitPath] = unitCache;
	        }
	        var val = unitCache[name];
	        if (val !== undefined) {
	            return val;
	        }
	        val = null;
	        val = _.find(globalList, function (x) { return hlimpl.qName(x.highLevel(), context) == name; });
	        if (!val) {
	            val = null;
	        }
	        unitCache[name] = val;
	        return val;
	    };
	    TraitsAndResourceTypesExpander.prototype.createHighLevelNode = function (api) {
	        var highLevelNodes = [];
	        var node = api;
	        while (node) {
	            var llNode = node.lowLevel();
	            var topComposite = new proxy.LowLevelCompositeNode(llNode, null, null, this.ramlVersion);
	            var nodeType = node.definition();
	            var newNode = new hlimpl.ASTNodeImpl(topComposite, null, nodeType, null);
	            highLevelNodes.push(newNode);
	            node = node.getMaster();
	        }
	        var masterApi = highLevelNodes.pop();
	        highLevelNodes = highLevelNodes.reverse();
	        var mergeMode = api.getMergeMode();
	        return mergeHighLevelNodes(masterApi, highLevelNodes, mergeMode);
	    };
	    TraitsAndResourceTypesExpander.prototype.processResource = function (resource) {
	        var _this = this;
	        var resourceData = this.collectResourceData(resource);
	        var resourceLowLevel = resource.highLevel().lowLevel();
	        resourceData.filter(function (x) { return x.resourceType != null; }).forEach(function (x) {
	            var resourceTypeLowLevel = x.resourceType.node.highLevel().lowLevel();
	            var resourceTypeTransformer = new DefaultTransformer(resource, x.resourceType.transformer);
	            resourceLowLevel.adopt(resourceTypeLowLevel, resourceTypeTransformer);
	        });
	        var methods = resource.methods();
	        methods.forEach(function (m) {
	            var methodLowLevel = m.highLevel().lowLevel();
	            var name = m.method();
	            resourceData.forEach(function (x) {
	                var methodTraits = x.methodTraits[name];
	                if (methodTraits) {
	                    methodTraits.forEach(function (x) {
	                        var traitLowLevel = x.node.highLevel().lowLevel();
	                        var traitTransformer = new DefaultTransformer(m, x.transformer);
	                        methodLowLevel.adopt(traitLowLevel, traitTransformer);
	                    });
	                }
	                var resourceTraits = x.traits;
	                if (resourceTraits) {
	                    resourceTraits.forEach(function (x) {
	                        var traitLowLevel = x.node.highLevel().lowLevel();
	                        var traitTransformer = new DefaultTransformer(m, x.transformer);
	                        methodLowLevel.adopt(traitLowLevel, traitTransformer);
	                    });
	                }
	            });
	        });
	        var resources = resource.resources();
	        resources.forEach(function (x) { return _this.processResource(x); });
	    };
	    TraitsAndResourceTypesExpander.prototype.collectResourceData = function (obj, arr, transformer, occuredResourceTypes) {
	        var _this = this;
	        if (arr === void 0) { arr = []; }
	        if (occuredResourceTypes === void 0) { occuredResourceTypes = {}; }
	        var ownTraits = this.extractTraits(obj, transformer);
	        var methodTraitsMap = {};
	        var methods = obj.methods();
	        methods.forEach(function (x) {
	            var methodTraits = _this.extractTraits(x, transformer);
	            if (methodTraits && methodTraits.length > 0) {
	                methodTraitsMap[x.method()] = methodTraits;
	            }
	        });
	        var rtData;
	        var rt = obj.type();
	        if (rt && !occuredResourceTypes[rt.name()]) {
	            occuredResourceTypes[rt.name()] = true;
	            rtData = this.readGenerictData(rt, obj.highLevel(), this.resourceTypeMap, this.globalResourceTypes, 'resource type', transformer);
	        }
	        arr.push({
	            resourceType: rtData,
	            traits: ownTraits,
	            methodTraits: methodTraitsMap
	        });
	        if (rtData) {
	            this.collectResourceData(rtData.node, arr, rtData.transformer, occuredResourceTypes);
	        }
	        return arr;
	    };
	    TraitsAndResourceTypesExpander.prototype.extractTraits = function (obj, _transformer, occuredTraits) {
	        var _this = this;
	        if (occuredTraits === void 0) { occuredTraits = {}; }
	        var arr = [];
	        for (var i = -1; i < arr.length; i++) {
	            var gd = i < 0 ? null : arr[i];
	            var _obj = gd ? gd.node : obj;
	            var transformer = gd ? gd.transformer : _transformer;
	            if (!_obj['is']) {
	                continue;
	            }
	            _obj.is().forEach(function (x) {
	                var traitData = _this.readGenerictData(x, _obj.highLevel(), _this.traitMap, _this.globalTraits, 'trait', transformer);
	                if (traitData) {
	                    var name = traitData.name;
	                    //if (!occuredTraits[name]) {
	                    occuredTraits[name] = true;
	                    arr.push(traitData);
	                }
	            });
	        }
	        return arr;
	    };
	    TraitsAndResourceTypesExpander.prototype.readGenerictData = function (obj, context, cache, globalList, template, transformer) {
	        var value = obj.value();
	        if (typeof (value) == 'string') {
	            if (transformer) {
	                value = transformer.transform(value).value;
	            }
	            var node = this.getTemplate(value, context, cache, globalList);
	            if (node) {
	                return {
	                    name: value,
	                    transformer: null,
	                    node: node
	                };
	            }
	        }
	        else if (value instanceof hlimpl.StructuredValue) {
	            var sv = value;
	            var name = sv.valueName();
	            if (transformer) {
	                name = transformer.transform(name).value;
	            }
	            var params = {};
	            var node = this.getTemplate(name, context, cache, globalList);
	            ;
	            //var t = hlimpl.typeFromNode(node.highLevel());
	            if (node) {
	                if (this.ramlVersion == 'RAML08' && transformer) {
	                    sv.children().forEach(function (x) { return params[x.valueName()] = transformer.transform(x.lowLevel().value()).value; });
	                }
	                else {
	                    sv.children().forEach(function (x) { return params[x.valueName()] = x.lowLevel().value(); });
	                }
	                return {
	                    name: name,
	                    transformer: new ValueTransformer(template, name, params),
	                    node: node
	                };
	            }
	        }
	        return null;
	    };
	    return TraitsAndResourceTypesExpander;
	}());
	var TransformMatches = (function () {
	    function TransformMatches(name, transformer) {
	        this.name = name;
	        this.regexp = new RegExp(TransformMatches.leftTransformRegexp.source + name + TransformMatches.rightTransformRegexp.source);
	        this.transformer = transformer;
	    }
	    TransformMatches.leftTransformRegexp = /\|\s*!\s*/;
	    TransformMatches.rightTransformRegexp = /\s*$/;
	    return TransformMatches;
	}());
	var transformers = [
	    new TransformMatches("singularize", function (arg) { return pluralize.singular(arg); }),
	    new TransformMatches("pluralize", function (arg) { return pluralize.plural(arg); }),
	    new TransformMatches("uppercase", function (arg) { return arg ? arg.toUpperCase() : arg; }),
	    new TransformMatches("lowercase", function (arg) { return arg ? arg.toLowerCase() : arg; }),
	    new TransformMatches("lowercamelcase", function (arg) {
	        if (!arg) {
	            return arg;
	        }
	        return changeCase.camelCase(arg);
	    }),
	    new TransformMatches("uppercamelcase", function (arg) {
	        if (!arg) {
	            return arg;
	        }
	        var lowerCamelCase = changeCase.camelCase(arg);
	        return lowerCamelCase[0].toUpperCase() + lowerCamelCase.substring(1, lowerCamelCase.length);
	    }),
	    new TransformMatches("lowerunderscorecase", function (arg) {
	        if (!arg) {
	            return arg;
	        }
	        var snakeCase = changeCase.snake(arg);
	        return snakeCase.toLowerCase();
	    }),
	    new TransformMatches("upperunderscorecase", function (arg) {
	        if (!arg) {
	            return arg;
	        }
	        var snakeCase = changeCase.snake(arg);
	        return snakeCase.toUpperCase();
	    }),
	    new TransformMatches("lowerhyphencase", function (arg) {
	        if (!arg) {
	            return arg;
	        }
	        var paramCase = changeCase.param(arg);
	        return paramCase.toLowerCase();
	    }),
	    new TransformMatches("upperhyphencase", function (arg) {
	        if (!arg) {
	            return arg;
	        }
	        var paramCase = changeCase.param(arg);
	        return paramCase.toUpperCase();
	    })
	];
	function getTransformNames() {
	    return transformers.map(function (transformer) { return transformer.name; });
	}
	exports.getTransformNames = getTransformNames;
	function getTransformerForOccurence(occurence) {
	    var result;
	    for (var i = 0; i < transformers.length; i++) {
	        if (occurence.match(transformers[i].regexp)) {
	            result = transformers[i].transformer;
	            break;
	        }
	    }
	    return result;
	}
	exports.getTransformerForOccurence = getTransformerForOccurence;
	var TransformationBuffer = (function () {
	    function TransformationBuffer() {
	        this.buf = null;
	    }
	    TransformationBuffer.prototype.append = function (value) {
	        if (value !== "") {
	            if (this.buf != null) {
	                if (value != null) {
	                    if (typeof (this.buf) != "string") {
	                        this.buf = "" + this.buf;
	                    }
	                    this.buf += value;
	                }
	            }
	            else if (value !== "") {
	                this.buf = value;
	            }
	        }
	    };
	    TransformationBuffer.prototype.value = function () {
	        return this.buf != null ? this.buf : "";
	    };
	    return TransformationBuffer;
	}());
	var ValueTransformer = (function () {
	    function ValueTransformer(templateKind, templateName, params) {
	        this.templateKind = templateKind;
	        this.templateName = templateName;
	        this.params = params;
	    }
	    ValueTransformer.prototype.transform = function (obj) {
	        var undefParams = {};
	        var errors = [];
	        if (typeof (obj) === 'string') {
	            var str = obj;
	            var buf = new TransformationBuffer();
	            var prev = 0;
	            for (var i = str.indexOf('<<'); i >= 0; i = str.indexOf('<<', prev)) {
	                buf.append(str.substring(prev, i));
	                var i0 = i;
	                i += '<<'.length;
	                prev = str.indexOf('>>', i);
	                if (prev == -1) {
	                    break;
	                }
	                var paramOccurence = str.substring(i, prev);
	                prev += '>>'.length;
	                var originalString = str.substring(i0, prev);
	                var val;
	                var paramName;
	                var transformer = getTransformerForOccurence(paramOccurence);
	                if (transformer) {
	                    var ind = paramOccurence.lastIndexOf('|');
	                    paramName = paramOccurence.substring(0, ind).trim();
	                    val = this.params[paramName];
	                    if (val) {
	                        val = transformer(val);
	                    }
	                }
	                else {
	                    paramName = paramOccurence.trim();
	                    val = this.params[paramName];
	                }
	                if (!val) {
	                    undefParams[paramName] = true;
	                    val = originalString;
	                }
	                buf.append(val);
	            }
	            var upArr = Object.keys(undefParams);
	            if (upArr.length > 0) {
	                var errStr = upArr.join(', ').trim();
	                var message = "Undefined " + this.templateKind + " parameter" + (upArr.length > 1 ? 's' : '') + ": " + errStr;
	                var error = {
	                    code: hl.IssueCode.MISSING_REQUIRED_PROPERTY,
	                    isWarning: false,
	                    message: message,
	                    node: null,
	                    start: -1,
	                    end: -1,
	                    path: null
	                };
	                errors.push(error);
	            }
	            buf.append(str.substring(prev, str.length));
	            return { value: buf.value(), errors: errors };
	        }
	        else {
	            return { value: obj, errors: errors };
	        }
	    };
	    return ValueTransformer;
	}());
	exports.ValueTransformer = ValueTransformer;
	var DefaultTransformer = (function (_super) {
	    __extends(DefaultTransformer, _super);
	    function DefaultTransformer(owner, delegate) {
	        _super.call(this, delegate.templateKind, delegate.templateName, null);
	        this.owner = owner;
	        this.delegate = delegate;
	    }
	    DefaultTransformer.prototype.transform = function (obj) {
	        if (obj == null) {
	            return {
	                value: obj,
	                errors: []
	            };
	        }
	        var ownResult = {
	            value: obj,
	            errors: []
	        };
	        var gotDefaultParam = false;
	        defaultParameters.forEach(function (x) { return gotDefaultParam = gotDefaultParam || obj.toString().indexOf('<<' + x) >= 0; });
	        if (gotDefaultParam) {
	            this.initParams();
	            ownResult = _super.prototype.transform.call(this, obj);
	        }
	        var result = this.delegate.transform(ownResult.value);
	        return result;
	    };
	    DefaultTransformer.prototype.initParams = function () {
	        var methodName;
	        var resourcePath = "";
	        var resourcePathName;
	        var node = this.owner.highLevel().lowLevel()
	            .originalNode().originalNode();
	        while (node) {
	            var key = node.key();
	            if (key != null) {
	                if (util.stringStartsWith(key, '/')) {
	                    if (!resourcePathName) {
	                        var arr = key.split('/');
	                        resourcePathName = arr[arr.length - 1].replace(/[\/\{\}]/g, '');
	                    }
	                    resourcePath = key + resourcePath;
	                }
	                else {
	                    methodName = key;
	                }
	            }
	            node = node.parent();
	        }
	        this.params = {
	            resourcePath: resourcePath,
	            resourcePathName: resourcePathName
	        };
	        if (methodName) {
	            this.params['methodName'] = methodName;
	        }
	    };
	    return DefaultTransformer;
	}(ValueTransformer));
	exports.DefaultTransformer = DefaultTransformer;
	var defaultParameters = ['resourcePath', 'resourcePathName', 'methodName'];
	//# sourceMappingURL=expander.js.map

/***/ },
/* 27 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../../typings/main.d.ts" />
	"use strict";
	var jsyaml = __webpack_require__(23);
	var json = __webpack_require__(48);
	var stringify = __webpack_require__(63);
	var proxy = __webpack_require__(37);
	var defs = __webpack_require__(35);
	var hl = __webpack_require__(10);
	var _ = __webpack_require__(61);
	var yaml = __webpack_require__(72);
	var def = __webpack_require__(35);
	var hlimpl = __webpack_require__(16);
	var path = __webpack_require__(15);
	var fs = __webpack_require__(36);
	var universes = __webpack_require__(22);
	var universeHelpers = __webpack_require__(14);
	var universeProvider = __webpack_require__(34);
	var services = def;
	var OverloadingValidator = __webpack_require__(49);
	var expander = __webpack_require__(26);
	var search = __webpack_require__(29);
	var rtypes = def.rt;
	var util = __webpack_require__(40);
	var contentprovider = __webpack_require__(31);
	var su = def.getSchemaUtils();
	var mediaTypeParser = __webpack_require__(64);
	var xmlutil = __webpack_require__(50);
	var changeCase = __webpack_require__(71);
	var pluralize = __webpack_require__(70);
	var LinterSettings = (function () {
	    function LinterSettings() {
	        this.validateNotStrictExamples = true;
	    }
	    return LinterSettings;
	}());
	var settings = new LinterSettings();
	var loophole = __webpack_require__(76);
	function evalInSandbox(code, thisArg, args) {
	    return new loophole.Function(code).call(thisArg, args);
	}
	var MAX_RECURSION_LEVEL = 400;
	exports.RESERVED_TEMPLATE_PARAMETERS = {
	    "resourcePathName": "Part of the resource path following the rightmost \"/\"",
	    "methodName": "Method name",
	    "resourcePath": "Path of the resource"
	};
	var lintWithFile = function (customLinter, acceptor, astNode) {
	    if (fs.existsSync(customLinter)) {
	        try {
	            var content = fs.readFileSync(customLinter).toString();
	            var factr = new LinterExtensionsImpl(acceptor);
	            evalInSandbox(content, factr, null);
	            factr.visit(astNode);
	        }
	        catch (e) {
	            console.log("Error in custom linter");
	            console.log(e);
	        }
	    }
	};
	function checkPropertyQuard(n, v) {
	    var pr = n.property();
	    if (pr) {
	        pr.getContextRequirements().forEach(function (x) {
	            if (!n.checkContextValue(x.name, x.value, x.value)) {
	                v.accept(createIssue(hl.IssueCode.MISSED_CONTEXT_REQUIREMENT, x.name + " should be " + x.value + " to use property " + pr.nameId(), n));
	            }
	        });
	    }
	    return pr;
	}
	;
	function lintNode(astNode, acceptor) {
	    if (typeof fs === "undefined") {
	        return;
	    }
	    if (!fs) {
	        return;
	    }
	    var fsMethods = ['exists', 'readFile', 'writeFile', 'readdir', 'existsSync', 'readFileSync', 'writeFileSync', 'readdirSync'];
	    var existingFsFields = Object.keys(fs);
	    for (var i = 0; i < fsMethods.length; i++) {
	        if (!fs[fsMethods[i]]) {
	            return;
	        }
	    }
	    var ps = astNode.lowLevel().unit().absolutePath();
	    var dr = path.dirname(ps);
	    var customLinter = path.resolve(dr, "raml-lint.js");
	    lintWithFile(customLinter, acceptor, astNode);
	    var dir = path.resolve(dr, ".raml");
	    if (fs.existsSync(dir)) {
	        var st = fs.statSync(dir);
	        if (st.isDirectory()) {
	            var files = fs.readdirSync(dir);
	            files.forEach(function (x) {
	                if (x.indexOf("-lint.js") != -1) {
	                    lintWithFile(path.resolve(dir, x), acceptor, astNode);
	                }
	                //console.log(x);
	            });
	        }
	    }
	}
	;
	var LinterExtensionsImpl = (function () {
	    function LinterExtensionsImpl(acceptor) {
	        this.acceptor = acceptor;
	        this.nodes = {};
	    }
	    LinterExtensionsImpl.prototype.error = function (w, message) {
	        this.acceptor.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, message, w.highLevel()));
	    };
	    LinterExtensionsImpl.prototype.errorOnProperty = function (w, property, message) {
	        var pr = w.highLevel().attr(property);
	        this.acceptor.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, message, pr));
	    };
	    LinterExtensionsImpl.prototype.warningOnProperty = function (w, property, message) {
	        var pr = w.highLevel().attr(property);
	        this.acceptor.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, message, pr, true));
	    };
	    LinterExtensionsImpl.prototype.warning = function (w, message) {
	        this.acceptor.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, message, w.highLevel(), true));
	    };
	    LinterExtensionsImpl.prototype.registerRule = function (nodeType, rule) {
	        var q = this.nodes[nodeType];
	        if (!q) {
	            q = [];
	            this.nodes[nodeType] = q;
	        }
	        q.push(rule);
	    };
	    LinterExtensionsImpl.prototype.visit = function (h) {
	        var _this = this;
	        var nd = h.definition();
	        this.process(nd, h);
	        nd.allSuperTypes().forEach(function (x) { return _this.process(x, h); });
	        h.elements().forEach(function (y) { return _this.visit(y); });
	    };
	    LinterExtensionsImpl.prototype.process = function (d, h) {
	        var _this = this;
	        if (d instanceof def.NodeClass) {
	            if (!d.getAdapter(services.RAMLService).getDeclaringNode()) {
	                var rules = this.nodes[d.nameId()];
	                if (rules) {
	                    rules.forEach(function (x) { return x(h.wrapperNode(), _this); });
	                }
	            }
	        }
	    };
	    return LinterExtensionsImpl;
	}());
	var StackNode = (function () {
	    function StackNode() {
	    }
	    StackNode.prototype.toString = function () {
	        if (this.prev) {
	            return this.value + "." + this.prev.toString();
	        }
	        return this.value;
	    };
	    StackNode.prototype.last = function () {
	        if (this.prev) {
	            return this.prev.last();
	        }
	        return this;
	    };
	    return StackNode;
	}());
	function isTypeOrSchema(d) {
	    return d.nameId() == universes.Universe10.TypeDeclaration.properties.type.name || d.nameId() == universes.Universe10.TypeDeclaration.properties.schema.name;
	}
	function isExampleProp(d) {
	    if (!d.domain()) {
	        return;
	    }
	    if (d.domain().getAdapter(services.RAMLService).isUserDefined()) {
	        return false;
	    }
	    return (d.nameId() == universes.Universe10.TypeDeclaration.properties.example.name) && (d.domain().key() != universes.Universe10.DocumentationItem && d.domain().key() != universes.Universe08.DocumentationItem);
	}
	/**
	 * For descendants of templates returns template type. Returns null for all other nodes.
	 */
	function typeOfContainingTemplate(h) {
	    var declRoot = h;
	    while (true) {
	        if (declRoot.definition().getAdapter(services.RAMLService).isInlinedTemplates()) {
	            return declRoot.definition().nameId();
	        }
	        var np = declRoot.parent();
	        if (!np) {
	            break;
	        }
	        else {
	            declRoot = np;
	        }
	    }
	    return null;
	}
	function restrictUnknownNodeError(node) {
	    var parentNode = node.parent();
	    var issue = null;
	    var parentDef = parentNode.definition();
	    if (parentNode && parentDef instanceof def.UserDefinedClass) {
	        var parentProperty = parentNode.property();
	        if (universeHelpers.isIsProperty(parentProperty)
	            || universeHelpers.isTypeProperty(parentProperty)) {
	            var paramName = node.name();
	            if (exports.RESERVED_TEMPLATE_PARAMETERS[paramName] != null) {
	                //Handling reserved parameter names;
	                issue = createIssue(hl.IssueCode.INVALID_PROPERTY, "Invalid parameter name: " + paramName + " is reserved", node);
	            }
	            else {
	                issue = createIssue(hl.IssueCode.UNKNOWN_NODE, "Unused parameter: " + paramName, node);
	            }
	        }
	    }
	    if (!issue) {
	        var propName = node.name();
	        var universe08 = universeProvider("RAML08");
	        var isRAML08 = parentDef.universe().version() == universe08.version();
	        if (isRAML08) {
	            var parameterTypeName = universes.Universe08.Parameter.name;
	            var typeDeclarationTypeName = universes.Universe10.TypeDeclaration.name;
	            if (isRAML08 && parentDef.isAssignableFrom(parameterTypeName)) {
	                var possibleDefs = universe08.type(parameterTypeName).allSubTypes().filter(function (x) {
	                    return universes.Universe08[x.nameId()]['properties'][propName] != null;
	                });
	                var possibleDefsMap = {};
	                for (var i = 0; i < possibleDefs.length; i++) {
	                    var x = possibleDefs[i];
	                    if (possibleDefsMap[x.nameId()]) {
	                        continue;
	                    }
	                    var valueRequirements = x.valueRequirements();
	                    if (!(valueRequirements && valueRequirements.length != 0)) {
	                        continue;
	                    }
	                    var typeRquirements = valueRequirements.filter(function (x) { return x.name == 'type'; }).map(function (x) { return x.value; });
	                    if (typeRquirements.length == 0) {
	                        continue;
	                    }
	                    var runtimeName = typeRquirements[0];
	                    possibleDefsMap[x.nameId()] = runtimeName;
	                    x.allSubTypes().forEach(function (y) { return possibleDefs.push(y); });
	                }
	                var runtimeNames = Object.keys(possibleDefsMap).map(function (x) { return possibleDefsMap[x]; }).sort();
	                ;
	                if (runtimeNames.length > 0) {
	                    var namesStr = runtimeNames.map(function (x, i) {
	                        var x1 = "'" + x + "'";
	                        if (i == runtimeNames.length - 1) {
	                            return x1;
	                        }
	                        if (i == runtimeNames.length - 2) {
	                            return x1 + ' or ';
	                        }
	                        return x1 + ', ';
	                    }).join('');
	                    var message = "Property '" + propName + "' can only be used if type is " + namesStr;
	                    issue = createIssue(hl.IssueCode.ILLEGAL_PROPERTY, message, node);
	                }
	            }
	        }
	    }
	    return issue;
	}
	;
	function validateBasic(node, v, requiredOnly) {
	    if (requiredOnly === void 0) { requiredOnly = false; }
	    var parentNode = node.parent();
	    if (node.lowLevel()) {
	        if (node.lowLevel().keyKind() == yaml.Kind.MAP) {
	            v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, "Node key can not be map", node));
	        }
	        if (node.lowLevel().keyKind() == yaml.Kind.SEQ) {
	            if (node.lowLevel().value() == null) {
	                var isPattern = false;
	                if (node.isElement()) {
	                    if (node.asElement().definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
	                        isPattern = true;
	                    }
	                }
	                if (!isPattern) {
	                    v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, "Node key can not be sequence", node));
	                }
	            }
	        }
	        if (parentNode == null) {
	            node.lowLevel().errors().forEach(function (x) {
	                var ps = x.mark ? x.mark.position : 0;
	                var em = {
	                    code: hl.IssueCode.YAML_ERROR,
	                    message: x.message,
	                    node: null,
	                    start: ps,
	                    end: ps + 1,
	                    isWarning: false,
	                    path: node.lowLevel().unit() == node.root().lowLevel().unit() ? null : node.lowLevel().unit().path(),
	                    unit: node.lowLevel().unit()
	                };
	                v.accept(em);
	            });
	        }
	    }
	    // if (node.errorMessage){
	    //     v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, node.errorMessage, node));
	    //     return;
	    // }
	    if (node.isUnknown()) {
	        if (node.name().indexOf("<<") != -1) {
	            if (typeOfContainingTemplate(parentNode) != null) {
	                new TraitVariablesValidator().validateName(node, v);
	                return;
	            }
	        }
	        if (node.needSequence) {
	            v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, "node: " + node.name() + " should be wrapped in sequence", node));
	        }
	        if (node.needMap) {
	            v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, (node.knownProperty ? node.knownProperty.nameId() : "") + " should be a map in RAML 1.0", node));
	            return;
	        }
	        if (node.unresolvedRef) {
	            v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, "reference: " + node.lowLevel().value() + " can not be resolved", node));
	        }
	        if (node.knownProperty && node.lowLevel().value() != null) {
	            //if (!node.lowLevel().)
	            if (node.lowLevel().includeErrors().length == 0) {
	                if (node.name() == "body" && node.computedValue("mediaType")) {
	                    return;
	                }
	                v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, "property " + node.name() + " can not have scalar value", node));
	            }
	        }
	        else {
	            var issue = restrictUnknownNodeError(node);
	            if (!issue) {
	                issue = createIssue(hl.IssueCode.UNKNOWN_NODE, "Unknown node: " + node.name(), node);
	            }
	            v.accept(issue);
	        }
	    }
	    if (node.markCh() && !node.allowRecursive()) {
	        if (!node.property()) {
	            return;
	        }
	        v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, "Recursive definition: " + node.name(), node));
	        return;
	    }
	    try {
	        node.directChildren().filter(function (child) {
	            return !requiredOnly || (child.property && child.property() && child.property().isRequired());
	        }).forEach(function (x) {
	            if (x.errorMessage) {
	                v.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, x.errorMessage, x.name() ? x : node));
	                return;
	            }
	            x.validate(v);
	        });
	    }
	    finally {
	        node.unmarkCh();
	    }
	}
	exports.validateBasic = validateBasic;
	function hasTemplateArgs(node) {
	    var vl = node.value();
	    if (typeof vl == "string") {
	        if (vl.indexOf("<<") != -1) {
	            return true;
	        }
	    }
	    var x = node.children();
	    for (var i = 0; i < x.length; i++) {
	        if (hasTemplateArgs(x[i])) {
	            return true;
	        }
	    }
	    return false;
	}
	function validate(node, v) {
	    if (!node.parent()) {
	        try {
	            validateIncludes(node, v);
	        }
	        finally {
	            cleanupIncludesFlag(node, v);
	        }
	    }
	    if (node.isAttr()) {
	        new CompositePropertyValidator().validate(node, v);
	    }
	    else if (node.isElement()) {
	        var highLevelNode = node;
	        if (highLevelNode.definition().isAnnotationType() || highLevelNode.property() && highLevelNode.property().nameId() == "annotations") {
	            new FixedFacetsValidator().validate(highLevelNode, v);
	            return;
	        }
	        if (highLevelNode.definition().isAssignableFrom(universes.Universe10.UsesDeclaration.name)) {
	            var vn = highLevelNode.attr(universes.Universe10.UsesDeclaration.properties.value.name);
	            if (vn && vn.value()) {
	                var rs = highLevelNode.lowLevel().unit().resolve(vn.value());
	                if (!rs) {
	                    v.accept(createIssue(hl.IssueCode.UNRESOLVED_REFERENCE, "Can not resolve library from path:" + vn.value(), highLevelNode, false));
	                }
	                else {
	                    var issues = [];
	                    rs.highLevel().validate({
	                        begin: function () {
	                        },
	                        accept: function (x) {
	                            issues.push(x);
	                        },
	                        end: function () {
	                        }
	                    });
	                    if (issues.length > 0) {
	                        var brand = createIssue(hl.IssueCode.UNRESOLVED_REFERENCE, "Issues in the used library:" + vn.value(), highLevelNode, false);
	                        issues.forEach(function (x) { x.unit = rs; x.path = rs.absolutePath(); });
	                        brand.extras = issues;
	                        v.accept(brand);
	                    }
	                }
	            }
	        }
	        if (highLevelNode.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
	            if (typeOfContainingTemplate(highLevelNode)) {
	                if (hasTemplateArgs(highLevelNode.lowLevel())) {
	                    return;
	                }
	            }
	            highLevelNode.attrs().forEach(function (a) {
	                var range = a.property().range().key();
	                if (range == universes.Universe08.RelativeUriString || range == universes.Universe10.RelativeUriString) {
	                    new UriValidator().validate(a, v);
	                    return;
	                }
	                if (range == universes.Universe08.FullUriTemplateString || range == universes.Universe10.FullUriTemplateString) {
	                    new UriValidator().validate(a, v);
	                    return;
	                }
	                if (a.property().getAdapter(services.RAMLPropertyService).isKey()) {
	                    var nameId = node.property() && node.property().nameId();
	                    if (nameId == universes.Universe08.Resource.properties.uriParameters.name
	                        || nameId == universes.Universe08.Resource.properties.baseUriParameters.name) {
	                        //new UrlParameterNameValidator().validate(a, v);
	                        return;
	                    }
	                    if (highLevelNode.property()) {
	                        if (highLevelNode.property().nameId() ==
	                            universes.Universe10.MethodBase.properties.body.name) {
	                            new MediaTypeValidator().validate(a, v);
	                            return;
	                        }
	                    }
	                }
	            });
	            if (highLevelNode.parent() && !highLevelNode.parent().parent()) {
	                if (rtypes.builtInTypes().get(highLevelNode.name())) {
	                    v.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, "redefining a built in type:" + highLevelNode.name(), highLevelNode));
	                }
	            }
	            new RecurrentOverlayValidator().validate(highLevelNode, v);
	            new RecurrentValidateChildrenKeys().validate(highLevelNode, v);
	            new NodeSpecificValidator().validate(highLevelNode, v);
	            new TypeDeclarationValidator().validate(highLevelNode, v);
	            return;
	        }
	        if (highLevelNode.definition().isAssignableFrom(universes.Universe10.LibraryBase.name)) {
	            var hasSchemas = false;
	            var hasTypes = false;
	            var vv;
	            highLevelNode.lowLevel().children().forEach(function (x) {
	                if (x.key() == "schemas") {
	                    hasSchemas = true;
	                    vv = x;
	                }
	                if (x.key() == "types") {
	                    hasTypes = true;
	                }
	            });
	            if (hasSchemas && hasTypes) {
	                v.accept(localLowLevelError(vv, highLevelNode, hl.IssueCode.ILLEGAL_PROPERTY_VALUE, false, "types and schemas are mutually exclusive", false));
	            }
	        }
	        var hasRequireds = highLevelNode.definition().requiredProperties() && highLevelNode.definition().requiredProperties().length > 0;
	        var isAllowAny = highLevelNode.definition().getAdapter(services.RAMLService).getAllowAny();
	        if (isAllowAny) {
	            if (hasRequireds) {
	                validateBasic(node, v, true);
	            }
	        }
	        else {
	            validateBasic(node, v);
	        }
	        new UriParametersValidator().validate(highLevelNode, v);
	        new CompositeNodeValidator().validate(highLevelNode, v);
	        new TemplateCyclesDetector().validate(highLevelNode, v);
	    }
	    else {
	        validateBasic(node, v);
	    }
	    new OptionalPropertiesValidator().validate(node, v);
	}
	exports.validate = validate;
	function cleanupIncludesFlag(node, v) {
	    var val = node.lowLevel().actual();
	    delete val._inc;
	    node.children().forEach(function (x) { return cleanupIncludesFlag(x, v); });
	}
	function validateIncludes(node, v) {
	    var val = node.lowLevel().actual();
	    if (val._inc) {
	        return;
	    }
	    if (node.isElement()) {
	        var vl = node.name();
	        if (typeof vl == "string") {
	            if (vl != null && vl.indexOf(" ") != -1) {
	                v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Keys should not have spaces '" + vl + "'", node));
	            }
	        }
	    }
	    val._inc = true;
	    if (node.lowLevel()) {
	        node.lowLevel().includeErrors().forEach(function (x) {
	            var em = createIssue(hl.IssueCode.UNABLE_TO_RESOLVE_INCLUDE_FILE, x, node);
	            v.accept(em);
	        });
	    }
	    node.children().forEach(function (x) { return validateIncludes(x, v); });
	}
	var validateRegexp = function (cleanedValue, v, node) {
	    try {
	        new RegExp(cleanedValue);
	    }
	    catch (Error) {
	        v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Illegal pattern " + cleanedValue, node));
	    }
	};
	var TraitVariablesValidator = (function () {
	    function TraitVariablesValidator() {
	    }
	    TraitVariablesValidator.prototype.validateName = function (node, acceptor) {
	        var name = node.name();
	        if (name) {
	            var start = node.lowLevel().keyStart();
	            this.check(name, start, node, acceptor);
	        }
	    };
	    TraitVariablesValidator.prototype.validateValue = function (node, acceptor) {
	        var value = node.value();
	        if (typeof (value) === 'string') {
	            var start = node.lowLevel().valueStart();
	            this.check(value, start, node, acceptor);
	        }
	    };
	    TraitVariablesValidator.prototype.check = function (str, start, node, acceptor) {
	        var errors = [];
	        var prev = 0;
	        for (var i = str.indexOf('<<'); i >= 0; i = str.indexOf('<<', prev)) {
	            var i0 = i;
	            i += '<<'.length;
	            prev = str.indexOf('>>', i);
	            var paramOccurence = str.substring(i, prev);
	            var ind = paramOccurence.lastIndexOf('|');
	            var paramName = ind >= 0 ? paramOccurence.substring(0, ind) : paramOccurence;
	            if (paramName.trim().length == 0) {
	                var msg = "Trait or resource type parameter name must contain non whitespace characters";
	                var issue = createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, msg, node, false);
	                issue.start = start + i;
	                issue.end = start + prev;
	                acceptor.accept(issue);
	            }
	            if (ind != -1) {
	                ind++;
	                var transformerName = paramOccurence.substring(ind).trim();
	                var functionNames = expander.getTransformNames();
	                if (!_.find(functionNames, function (functionName) {
	                    return transformerName === functionName || transformerName === ('!' + functionName);
	                })) {
	                    var msg = "Unknown function applied to parameter: " + transformerName;
	                    var issue = createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, msg, node, false);
	                    issue.start = start + ind;
	                    issue.end = start + prev;
	                    acceptor.accept(issue);
	                }
	            }
	            prev += '>>'.length;
	        }
	        return errors;
	    };
	    return TraitVariablesValidator;
	}());
	var MethodBodyValidator = (function () {
	    function MethodBodyValidator() {
	    }
	    MethodBodyValidator.prototype.validate = function (node, validationAcceptor) {
	        var methodNode = node.parent();
	        if (!methodNode) {
	            return;
	        }
	        if (!(methodNode.definition().isAssignableFrom(universes.Universe08.Method.name) || methodNode.definition().isAssignableFrom(universes.Universe10.Method.name))) {
	            return;
	        }
	        var hasBody = _.find(methodNode.lowLevel() && methodNode.lowLevel().children() || [], function (child) {
	            var keyValue = child.key();
	            return keyValue && (universes.Universe08.MethodBase.properties.body.name === keyValue || universes.Universe10.MethodBase.properties.body.name === keyValue);
	        });
	        if (hasBody && _.find(MethodBodyValidator.methodsWithoutRequestBody, function (methodDisabled) { return methodNode.name() === methodDisabled; })) {
	            validationAcceptor.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, 'Request body is disabled for "' + methodNode.name() + '" method.', methodNode));
	        }
	    };
	    MethodBodyValidator.methodsWithoutRequestBody = ['trace'];
	    return MethodBodyValidator;
	}());
	var CompositePropertyValidator = (function () {
	    function CompositePropertyValidator() {
	    }
	    CompositePropertyValidator.prototype.validate = function (node, v) {
	        var pr = checkPropertyQuard(node, v);
	        var vl = node.value();
	        if (!node.property().range().hasStructure()) {
	            if (vl instanceof hlimpl.StructuredValue && !node.property().isSelfNode()) {
	                //TODO THIS SHOULD BE MOVED TO TYPESYSTEM FOR STS AT SOME MOMENT
	                if (isTypeOrSchema(node.property())) {
	                    if (node.property().domain().key() == universes.Universe08.BodyLike) {
	                        var structValue = vl;
	                        var newNode = new hlimpl.ASTNodeImpl(node.lowLevel(), node.parent(), node.parent().definition().universe().type(universes.Universe08.BodyLike.name), node.property());
	                        newNode.validate(v);
	                        return;
	                    }
	                }
	                v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Scalar is expected here", node));
	            }
	            else {
	                if (node.lowLevel().valueKind() != yaml.Kind.SCALAR && node.lowLevel().valueKind() != yaml.Kind.INCLUDE_REF && !node.property().getAdapter(services.RAMLPropertyService).isKey()) {
	                    if ((!node.property().isMultiValue())) {
	                        var k = node.property().range().key();
	                        if (k == universes.Universe08.StringType || k == universes.Universe08.MarkdownString || k == universes.Universe08.MimeType) {
	                            v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "property '" + node.name() + "' must be a string", node));
	                        }
	                    }
	                }
	            }
	            if (node.isAnnotatedScalar()) {
	                var fvl = new FixedFacetsValidator();
	                node.annotations().forEach(function (x) {
	                    var vl = x.value();
	                    var highLevel = vl.toHighLevel();
	                    if (!highLevel) {
	                        v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "unknown annotation " + vl.valueName(), x));
	                    }
	                    else {
	                        fvl.validate(highLevel, v);
	                    }
	                });
	            }
	        }
	        if (typeof vl == 'string' && vl.indexOf("<<") != -1) {
	            if (vl.indexOf(">>") > vl.indexOf("<<")) {
	                new TraitVariablesValidator().validateValue(node, v);
	                if (typeOfContainingTemplate(node.parent()) != null) {
	                    return;
	                }
	            }
	        }
	        new MethodBodyValidator().validate(node, v);
	        if ((node.property().range().key() == universes.Universe08.MimeType ||
	            node.property().range().key() == universes.Universe10.MimeType) ||
	            (node.property().nameId() == universes.Universe10.TypeDeclaration.properties.name.name
	                && node.parent().property().nameId() ==
	                    universes.Universe10.MethodBase.properties.body.name)) {
	            new MediaTypeValidator().validate(node, v);
	            return;
	        }
	        if (isExampleProp(node.property())) {
	            if (node.definition().universe().version() == "RAML08") {
	                var llv = node.lowLevel().value();
	                if (node.lowLevel().children().length > 0) {
	                    v.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, "example should be a string", node, false));
	                }
	            }
	            new ExampleValidator().validate(node, v);
	        }
	        if (node.property().nameId() == universes.Universe10.TypeDeclaration.properties.name.name) {
	            //TODO MOVE TO DEF SYSTEM
	            var nameId = node.parent().property() && node.parent().property().nameId();
	            if (nameId == universes.Universe08.Resource.properties.uriParameters.name
	                || nameId == universes.Universe08.Resource.properties.baseUriParameters.name) {
	                //                    new UrlParameterNameValidator().validate(node, v);
	                return;
	            }
	        }
	        var range = node.property().range().key();
	        if (range == universes.Universe08.RelativeUriString || range == universes.Universe10.RelativeUriString) {
	            new UriValidator().validate(node, v);
	            return;
	        }
	        if (range == universes.Universe08.FullUriTemplateString || range == universes.Universe10.FullUriTemplateString) {
	            new UriValidator().validate(node, v);
	            return;
	        }
	        if ("pattern" == node.name() && universes.Universe10.StringType == node.definition().key()
	            && node.parent().definition().isAssignableFrom("StringTypeDeclaration")) {
	            validateRegexp(node.value(), v, node);
	        }
	        if ("name" == node.name() && universes.Universe10.StringType == node.definition().key()
	            && (typeof node.value() == "string")
	            && node.value().indexOf("[") == 0
	            && node.value().lastIndexOf("]") == node.value().length - 1) {
	            if (node.parent() instanceof hlimpl.ASTNodeImpl &&
	                universes.Universe10.ObjectTypeDeclaration.properties.properties.name == node.parent().property().nameId()) {
	                if (node.parent().parent() instanceof hlimpl.ASTNodeImpl &&
	                    universes.Universe10.ObjectTypeDeclaration == node.parent().parent().definition().key()) {
	                    var cleanedValue = node.value().substr(1, node.value().length - 2);
	                    validateRegexp(cleanedValue, v, node);
	                }
	            }
	        }
	        if (pr.isReference() || pr.isDescriminator()) {
	            new DescriminatorOrReferenceValidator().validate(node, v);
	        }
	        else {
	            new NormalValidator().validate(node, v);
	        }
	    };
	    return CompositePropertyValidator;
	}());
	function isValid(t, h, value, p, attr) {
	    if (t.hasArrayInHierarchy()) {
	        return isValidArray(t, h, value, p, attr);
	    }
	    else if (t.hasValueTypeInHierarchy()) {
	        return isValidValueType(t, h, value, p, attr);
	    }
	    return true;
	}
	exports.isValid = isValid;
	function isValidArray(t, h, v, p, attr) {
	    if (t.arrayInHierarchy().componentType()) {
	        return isValid(t.arrayInHierarchy().componentType(), h, v, p);
	    }
	    return true;
	}
	function isValidValueType(t, h, v, p, attr) {
	    //FIXME
	    try {
	        if (t.key() == universes.Universe10.AnnotationRef) {
	            var targets = search.referenceTargets(p, h);
	            var actualAnnotation = _.find(targets, function (x) { return hlimpl.qName(x, h) == v; });
	            if (actualAnnotation != null) {
	                var attrs = actualAnnotation.attributes("allowedTargets");
	                if (attrs) {
	                    var aVals = attrs.map(function (x) { return x.value(); });
	                    if (aVals.length > 0) {
	                        var found = false;
	                        //no we should actually check that we are applying annotation properly
	                        var tps = h.definition().allSuperTypes();
	                        tps = tps.concat([h.definition()]);
	                        var tpNames = tps.map(function (x) { return x.nameId(); });
	                        aVals.forEach(function (x) {
	                            //FIXME this is deeply wrong code
	                            if (x == "API") {
	                                x = "Api";
	                            }
	                            if (x == "NamedExample") {
	                                x = "ExampleSpec";
	                            }
	                            if (x == "SecurityScheme") {
	                                x = "AbstractSecurityScheme";
	                            }
	                            if (x == "SecuritySchemeSettings") {
	                                x = "SecuritySchemeSettings";
	                            }
	                            if (_.find(tpNames, function (y) { return y == x; })) {
	                                found = true;
	                            }
	                            else {
	                                if (x == "Parameter") {
	                                    if (h.computedValue("location")) {
	                                        found = true;
	                                    }
	                                }
	                                if (x == "Field") {
	                                    if (h.computedValue("field")) {
	                                        found = true;
	                                    }
	                                }
	                            }
	                        });
	                        if (!found) {
	                            return new Error("annotation " + v + " can not be placed at this location, allowed targets are:" + aVals);
	                        }
	                    }
	                }
	            }
	            return tm;
	        }
	        if (t.key() == universes.Universe08.SchemaString || t.key() == universes.Universe10.SchemaString) {
	            var tm = su.createSchema(v, new contentprovider.ContentProvider(h.lowLevel().unit()));
	            if (tm instanceof Error) {
	                tm.canBeRef = true;
	            }
	            return tm;
	        }
	        if (t.key() == universes.Universe08.StatusCodeString || t.key() == universes.Universe10.StatusCodeString) {
	            var err = validateResponseString(v);
	            if (err != null) {
	                return err;
	            }
	        }
	        if (t.key() == universes.Universe08.BooleanType || t.isAssignableFrom(universes.Universe10.BooleanType.name)) {
	            if (!(v === 'true' || v === 'false' || v === true || v === false)) {
	                return new Error("'true' or 'false' is expected here");
	            }
	            if (attr) {
	                var stringValue = attr.lowLevel().value(true);
	                if (!(stringValue === 'true' || stringValue === 'false')) {
	                    return new Error("'true' or 'false' is expected here");
	                }
	            }
	        }
	        if (t.key() == universes.Universe08.NumberType || t.isAssignableFrom(universes.Universe10.NumberType.name)) {
	            var q = parseFloat(v);
	            if (isNaN(q)) {
	                return new Error("the value of " + p.nameId() + " must be a number");
	            }
	        }
	        if (t.key() == universes.Universe08.StringType || t.isAssignableFrom(universes.Universe10.StringType.name)) {
	            if (v === null) {
	                //checking if there is at least something in the node.
	                //We have many tests and APIs with the text like 'propertyName:' without a value. I do not know if such cases are
	                //actually valid, but not reporting this for now.
	                if (h && p) {
	                    var highLevelProperty = h.attr(p.nameId());
	                    if (highLevelProperty) {
	                        var lowLevelChildren = highLevelProperty.lowLevel().children();
	                        if (lowLevelChildren && lowLevelChildren.length > 0) {
	                            return new Error(p.nameId() + " must be a string");
	                        }
	                    }
	                }
	            }
	        }
	        return true;
	    }
	    catch (e) {
	        e.canBeRef = true; //FIXME
	        return e;
	    }
	}
	var NormalValidator = (function () {
	    function NormalValidator() {
	    }
	    NormalValidator.prototype.validate = function (node, cb) {
	        var vl = node.value();
	        var pr = node.property();
	        var range = pr.range();
	        var dnode = range.getAdapter(services.RAMLService).getDeclaringNode();
	        if (dnode) {
	            var rof = dnode.parsedType();
	            var dp = node.parent().lowLevel().dumpToObject();
	            var vl = dp[node.parent().name()];
	            var isVal = pr.canBeValue();
	            var val = isVal ? vl : vl[pr.nameId()];
	            var validateObject = rof.validate(val, true);
	            if (!validateObject.isOk()) {
	                validateObject.getErrors().forEach(function (e) { return cb.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, e.getMessage(), node, false)); });
	            }
	        }
	        var v = cb;
	        if (node.lowLevel().keyKind() != yaml.Kind.SEQ) {
	            var validation = isValid(pr.range(), node.parent(), vl, pr, node);
	        }
	        else {
	            validation = true;
	        }
	        if (validation instanceof Error) {
	            if (!validation.canBeRef) {
	                v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, validation.message, node));
	                validation = null;
	                return;
	            }
	        }
	        if (!validation || validation instanceof Error) {
	            if (pr.nameId() != 'value') {
	                if (!checkReference(pr, node, vl, v)) {
	                    if (pr.nameId() == universes.Universe10.TypeDeclaration.properties.schema.name
	                        || universes.Universe10.TypeDeclaration.properties.type.name) {
	                        if (vl && vl.trim() && (pr.domain().key() == universes.Universe08.BodyLike
	                            || pr.domain().key() == universes.Universe10.TypeDeclaration)) {
	                            var testSchema = vl.trim().charAt(0); //FIXME
	                            if (testSchema != '{' && testSchema != '<') {
	                                return;
	                            }
	                        }
	                    }
	                    var decl = node.findReferencedValue();
	                    if (decl instanceof Error) {
	                        v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, decl.message, node));
	                    }
	                    if (!decl) {
	                        if (vl) {
	                            if (pr.nameId() == universes.Universe10.TypeDeclaration.properties.schema.name) {
	                                var z = vl.trim();
	                                if (z.charAt(0) != '{' && z.charAt(0) != '<') {
	                                    if (vl.indexOf('|') != -1 || vl.indexOf('[]') != -1 || vl.indexOf("(") != -1) {
	                                        return;
	                                    }
	                                }
	                            }
	                        }
	                        if (validation instanceof Error && vl) {
	                            v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, validation.message, node));
	                            validation = null;
	                            return;
	                        }
	                        v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Empty value is not allowed here", node));
	                    }
	                }
	            }
	            else {
	                var vl = node.value();
	                var message = "Invalid value schema:" + vl;
	                if (validation instanceof Error) {
	                    message = validation.message;
	                }
	                v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, message, node));
	            }
	        }
	        var values = pr.enumOptions();
	        if (values) {
	            if (typeof vl !== 'string') {
	                return;
	            }
	            if (typeof values == 'string') {
	                if (values != vl) {
	                    if (vl && (vl.indexOf("x-") == 0) && pr.nameId() == universes.Universe08.AbstractSecurityScheme.properties.type.name) {
	                    }
	                    else {
	                        v.accept(createIssue(hl.IssueCode.UNRESOLVED_REFERENCE, "Invalid value:" + vl + " allowed values are:" + values, node));
	                    }
	                }
	            }
	            else if (values.length > 0) {
	                if (!_.find(values, function (x) { return x == vl; })) {
	                    if (vl && (vl.indexOf("x-") == 0) && pr.nameId() == universes.Universe08.AbstractSecurityScheme.properties.type.name) {
	                    }
	                    else {
	                        v.accept(createIssue(hl.IssueCode.UNRESOLVED_REFERENCE, "Invalid value:" + vl + " allowed values are:" + values.join(","), node));
	                    }
	                }
	            }
	        }
	    };
	    return NormalValidator;
	}());
	var UriValidator = (function () {
	    function UriValidator() {
	    }
	    UriValidator.prototype.validate = function (node, cb) {
	        try {
	            var values = new UrlParameterNameValidator().parseUrl(node.value());
	            if (values.some(function (x) { return x == "version"; }) && node.property().nameId() == "baseUri") {
	                var version = node.root().attr("version");
	                if (!version) {
	                    cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "missing version", node, false));
	                }
	            }
	            if (values.some(function (x) { return x.length == 0; })) {
	                cb.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, "URI parameter must have name specified", node, false));
	            }
	        }
	        catch (e) {
	            cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, e.message, node, false));
	        }
	    };
	    return UriValidator;
	}());
	var MediaTypeValidator = (function () {
	    function MediaTypeValidator() {
	    }
	    MediaTypeValidator.prototype.validate = function (node, cb) {
	        try {
	            var v = node.value();
	            if (!v) {
	                return;
	            }
	            if (v == "*/*") {
	                return;
	            }
	            if (v.indexOf("/*") == v.length - 2) {
	                v = v.substring(0, v.length - 2) + "/xxx";
	            }
	            if (node.parent() && node.parent().parent() && node.parent().parent().definition().isAssignableFrom(universes.Universe10.Trait.name)) {
	                if (v.indexOf("<<") >= 0) {
	                    return;
	                }
	            }
	            if (v == "body") {
	                if (node.parent().parent()) {
	                    var ppc = node.parent().parent().definition().key();
	                    if (ppc === universes.Universe08.Response || ppc === universes.Universe10.Response ||
	                        node.parent().parent().definition().isAssignableFrom(universes.Universe10.MethodBase.name)) {
	                        v = node.parent().computedValue("mediaType");
	                    }
	                }
	            }
	            var res = mediaTypeParser.parse(v);
	            var types = {
	                application: 1,
	                audio: 1,
	                example: 1,
	                image: 1,
	                message: 1,
	                model: 1,
	                multipart: 1,
	                text: 1,
	                video: 1
	            };
	            if (!types[res.type]) {
	                cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Unknown media type 'type'", node));
	            }
	        }
	        catch (e) {
	            cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "" + e.message, node));
	        }
	        if (node.value() && node.value() == ("multipart/form-data") || node.value() == ("application/x-www-form-urlencoded")) {
	            if (node.parent() && node.parent().parent() && node.parent().parent().property()) {
	                if (node.parent().parent().property().nameId() == universes.Universe10.MethodBase.properties.responses.name) {
	                    cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Form related media types can not be used in responses", node));
	                }
	            }
	        }
	        return;
	    };
	    return MediaTypeValidator;
	}());
	//class SignatureValidator implements PropertyValidator{
	//    validate(node:hl.IAttribute,cb:hl.ValidationAcceptor){
	//        var vl=node.value();
	//        var q = vl?vl.trim():"";
	//        if (q.length > 0 ) {
	//            try {
	//                //ramlSignature.validate(vl, node, cb);
	//            }catch (e){
	//                cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA,"Error during signature parse:"+e.message,node))
	//            }
	//            return;
	//        }
	//        return;
	//    }
	//}
	var UrlParameterNameValidator = (function () {
	    function UrlParameterNameValidator() {
	    }
	    UrlParameterNameValidator.prototype.checkBaseUri = function (node, c, vl, v) {
	        var bu = c.root().attr("baseUri");
	        if (bu) {
	            var tnv = bu.value();
	            try {
	                var pNames = this.parseUrl(tnv);
	                if (!_.find(pNames, function (x) { return x == vl; })) {
	                    v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Unused url parameter", node));
	                }
	            }
	            catch (e) {
	            }
	        }
	        else {
	            v.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Unused url parameter", node));
	        }
	    };
	    UrlParameterNameValidator.prototype.parseUrl = function (value) {
	        var result = [];
	        var temp = "";
	        var inPar = false;
	        var count = 0;
	        for (var a = 0; a < value.length; a++) {
	            var c = value[a];
	            if (c == '{') {
	                count++;
	                inPar = true;
	                continue;
	            }
	            if (c == '}') {
	                count--;
	                inPar = false;
	                result.push(temp);
	                temp = "";
	                continue;
	            }
	            if (inPar) {
	                temp += c;
	            }
	        }
	        if (count > 0) {
	            throw new Error("Unmatched '{'");
	        }
	        if (count < 0) {
	            throw new Error("Unmatched '}'");
	        }
	        return result;
	    };
	    UrlParameterNameValidator.prototype.validate = function (node, cb) {
	        var vl = node.value();
	        if (node.parent().property().nameId() == universes.Universe10.Api.properties.baseUri.name) {
	            var c = node.parent().parent();
	            this.checkBaseUri(node, c, vl, cb);
	            return;
	        }
	        var c = node.parent().parent();
	        var tn = c.name();
	        if (c.definition().key() === universes.Universe10.Api ||
	            c.definition().key() === universes.Universe08.Api) {
	            this.checkBaseUri(node, c, vl, cb);
	            return;
	        }
	        if (c.definition().key() == universes.Universe10.ResourceType ||
	            c.definition().key() == universes.Universe08.ResourceType) {
	            return;
	        }
	        try {
	            var pNames = this.parseUrl(tn);
	            var foundInLocalParameters = _.find(pNames, function (x) { return x == vl; });
	            if (!foundInLocalParameters) {
	                var baseUri = node.root().attr(universes.Universe10.Api.properties.baseUri.name);
	                if (baseUri && node.name() === universes.Universe08.Api.properties.baseUriParameters.name) {
	                    var baseUriValue = baseUri.value();
	                    if (baseUriValue) {
	                        pNames = this.parseUrl(baseUriValue);
	                        if (pNames && pNames.length > 0) {
	                            if (_.find(pNames, function (x) { return x == vl; }))
	                                return;
	                        }
	                    }
	                }
	                cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Unused url parameter '" + vl + "'", node));
	            }
	        }
	        catch (e) {
	        }
	    };
	    return UrlParameterNameValidator;
	}());
	exports.UrlParameterNameValidator = UrlParameterNameValidator;
	//TODO this should be probably moved to a more general file/module
	exports.typeToName = {};
	exports.typeToName[universes.Universe08.Trait.name] = "trait";
	exports.typeToName[universes.Universe08.ResourceType.name] = "resource type";
	exports.typeToName[universes.Universe10.Trait.name] = "trait";
	exports.typeToName[universes.Universe10.ResourceType.name] = "resource type";
	exports.typeToName[universes.Universe10.AbstractSecurityScheme.name] = "security scheme";
	exports.typeToName[universes.Universe10.Method.name] = "method";
	exports.typeToName[universes.Universe08.Method.name] = "method";
	exports.typeToName[universes.Universe10.Resource.name] = "resource";
	exports.typeToName[universes.Universe08.Resource.name] = "resource";
	exports.typeToName[universes.Universe10.Api.name] = "api";
	exports.typeToName[universes.Universe08.Api.name] = "api";
	exports.typeToName[universes.Universe10.Response.name] = "response";
	exports.typeToName[universes.Universe08.Response.name] = "response";
	exports.typeToName[universes.Universe08.BodyLike.name] = "body";
	exports.parameterPropertyToName = {};
	exports.parameterPropertyToName[universes.Universe08.MethodBase.properties.headers.name] = "header";
	exports.parameterPropertyToName[universes.Universe08.MethodBase.properties.queryParameters.name] = "query parameter";
	exports.parameterPropertyToName[universes.Universe08.Api.properties.uriParameters.name] = "uri parameter";
	exports.parameterPropertyToName[universes.Universe08.Api.properties.baseUriParameters.name] = "base uri parameter";
	exports.parameterPropertyToName[universes.Universe08.BodyLike.properties.formParameters.name] = "form parameter";
	exports.parameterPropertyToName[universes.Universe10.MethodBase.properties.headers.name] = "header";
	exports.parameterPropertyToName[universes.Universe10.MethodBase.properties.queryParameters.name] = "query parameter";
	exports.parameterPropertyToName[universes.Universe10.ResourceBase.properties.uriParameters.name] = "uri parameter";
	exports.parameterPropertyToName[universes.Universe10.Api.properties.baseUriParameters.name] = "base uri parameter";
	exports.parameterPropertyToName[universes.Universe10.MethodBase.properties.body.name] = "body";
	function getHumanReadableNodeName(astNode) {
	    if (!astNode)
	        return null;
	    if (astNode.isElement()) {
	        var element = astNode;
	        var definition = element.definition();
	        if (definition && exports.typeToName.hasOwnProperty(definition.nameId())) {
	            return exports.typeToName[definition.nameId()];
	        }
	        if (definition.isAssignableFrom(universes.Universe10.TypeDeclaration.name) ||
	            definition.isAssignableFrom(universes.Universe08.Parameter.name)) {
	            if (element.property() && exports.parameterPropertyToName.hasOwnProperty(element.property().nameId())) {
	                return exports.parameterPropertyToName[element.property().nameId()];
	            }
	            if (element.property() && element.parent() &&
	                element.property().nameId() == universes.Universe10.LibraryBase.properties.types.name &&
	                element.parent().definition() && element.parent().definition().isAssignableFrom(universes.Universe10.LibraryBase.name)) {
	                return "type";
	            }
	            if (element.property() && element.parent() &&
	                element.property().nameId() == universes.Universe10.LibraryBase.properties.securitySchemes.name &&
	                element.parent().definition() && element.parent().definition().isAssignableFrom(universes.Universe10.LibraryBase.name)) {
	                return "security scheme";
	            }
	        }
	    }
	    return null;
	}
	exports.getHumanReadableNodeName = getHumanReadableNodeName;
	function isValidPropertyValue(pr, vl, c) {
	    var node = search.declRoot(c);
	    if (!node._cach) {
	        node._cach = {};
	    }
	    var id = pr.id();
	    if (pr.domain()) {
	        id += pr.domain().nameId();
	    }
	    if (id) {
	        var cached = node._cach[id];
	        if (cached) {
	            return cached[vl] != null;
	        }
	    }
	    var vls = search.enumValues(pr, c);
	    var mm = {};
	    vls.forEach(function (x) { return mm[x] = 1; });
	    if (pr.id()) {
	        node._cach[id] = mm;
	    }
	    return mm[vl] != null;
	}
	function checkReference(pr, astNode, vl, cb) {
	    checkTraitReference(pr, astNode, cb);
	    checkResourceTypeReference(pr, astNode, cb);
	    if (!vl) {
	        return;
	    }
	    if (vl == 'null') {
	        if (pr.isAllowNull()) {
	            return;
	        }
	    }
	    var adapter = pr.getAdapter(services.RAMLPropertyService);
	    var valid = isValidPropertyValue(pr, vl, astNode.parent());
	    if (!valid && astNode.lowLevel().unit().absolutePath() !== astNode.parent().lowLevel().unit().absolutePath()) {
	        valid = isValidPropertyValue(pr, vl, hlimpl.fromUnit(astNode.lowLevel().unit()));
	    }
	    if (!valid) {
	        if (typeof vl == 'string') {
	            if ((vl.indexOf("x-") == 0) && pr.nameId() == universes.Universe10.TypeDeclaration.properties.type.name) {
	                return true;
	            }
	        }
	        var expected = (adapter.isReference && adapter.isReference() && adapter.referencesTo && adapter.referencesTo() && adapter.referencesTo().nameId && adapter.referencesTo().nameId());
	        var referencedToName = exports.typeToName[expected] || nameForNonReference(astNode);
	        var message = referencedToName ? ("Unrecognized " + referencedToName + " '" + vl + "'.") : ("Unresolved reference: " + vl);
	        var spesializedMessage = specializeReferenceError(message, pr, astNode);
	        cb.accept(createIssue(hl.IssueCode.UNRESOLVED_REFERENCE, spesializedMessage, astNode));
	        return true;
	    }
	    if (isDuplicateSibling(astNode) && universeHelpers.isTraitRefType(astNode.definition())) {
	        var message = "Duplicate trait reference '" + vl + "'.";
	        var spesializedMessage = specializeReferenceError(message, pr, astNode);
	        cb.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, spesializedMessage, astNode));
	        return true;
	    }
	    return false;
	}
	function isDuplicateSibling(attr) {
	    var ramlVersion = attr.property().domain().universe().version();
	    var siblingName;
	    if (ramlVersion == "RAML10") {
	        siblingName = stringify(json.serialize(attr.lowLevel()));
	    }
	    else {
	        siblingName = attr.value() && attr.value().valueName && attr.value().valueName();
	    }
	    if (!siblingName) {
	        return false;
	    }
	    var parent = attr.parent && attr.parent();
	    if (!parent) {
	        return false;
	    }
	    var propertyName = attr.name && attr.name();
	    if (!propertyName) {
	        return false;
	    }
	    var siblings = parent.attributes && parent.attributes(propertyName);
	    if (!siblings) {
	        return false;
	    }
	    if (siblings.length === 0) {
	        return false;
	    }
	    var count = 0;
	    siblings.forEach(function (sibling) {
	        var name;
	        if (ramlVersion == "RAML10") {
	            siblingName = stringify(json.serialize(sibling.lowLevel()));
	        }
	        else {
	            name = sibling.value && sibling.value() && sibling.value().valueName && sibling.value().valueName();
	        }
	        if (name === siblingName) {
	            count++;
	        }
	    });
	    return count > 1;
	}
	function checkTraitReference(property, astNode, acceptor) {
	    //"is" property value must be an array
	    if (!universeHelpers.isIsProperty(property)) {
	        return;
	    }
	    var lowLevel = astNode.lowLevel();
	    if (lowLevel == null) {
	        return;
	    }
	    //trying to find "is" mapping, looking 2 nodes up max
	    var isMappingNode = null;
	    var lowLevelParent = lowLevel.parent();
	    var lowLevelParentParent = lowLevelParent != null ? lowLevelParent.parent() : null;
	    if (lowLevel.kind() == yaml.Kind.MAPPING && lowLevel.key() && lowLevel.key() == "is") {
	        isMappingNode = lowLevel;
	    }
	    else if (lowLevelParent != null &&
	        lowLevelParent.kind() == yaml.Kind.MAPPING && lowLevelParent.key() && lowLevelParent.key() == "is") {
	        isMappingNode = lowLevelParent;
	    }
	    else if (lowLevelParentParent != null &&
	        lowLevelParentParent.kind() == yaml.Kind.MAPPING && lowLevelParentParent.key() && lowLevelParentParent.key() == "is") {
	        isMappingNode = lowLevelParentParent;
	    }
	    if (isMappingNode == null) {
	        return;
	    }
	    //having a single value is bad
	    if (isMappingNode.value() != null && (!isMappingNode.children() || isMappingNode.children().length == 0)) {
	        acceptor.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, "property 'is' must be an array", astNode));
	    }
	    //only maps and scalars are allowed as direct children
	    var illegalChildFound = false;
	    isMappingNode.children().forEach(function (child) {
	        if (child.kind() != yaml.Kind.SCALAR && child.kind() != yaml.Kind.MAP) {
	            illegalChildFound = true;
	        }
	    });
	    if (illegalChildFound) {
	        acceptor.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, "property 'is' must be an array", astNode));
	    }
	}
	function checkResourceTypeReference(property, astNode, acceptor) {
	    if (!universeHelpers.isTypeProperty(property)) {
	        return;
	    }
	    if (!universeHelpers.isResourceTypeRefType(astNode.definition())) {
	        return;
	    }
	    var lowLevel = astNode.lowLevel();
	    if (astNode.value() == null && lowLevel && lowLevel.children() &&
	        lowLevel.children().length == 0) {
	        if (lowLevel.kind() == yaml.Kind.MAPPING && lowLevel.valueKind() != null) {
	            //no value, no children in the mapping, but some value, that means empty map or something like this.
	            acceptor.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, "resource type name must be provided", astNode));
	        }
	    }
	    else if (astNode.value() == null && lowLevel && lowLevel.children() &&
	        lowLevel.children().length > 1) {
	        //more than a single resource type in a list / map
	        acceptor.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, "a resource or resourceType can inherit from a single resourceType", astNode));
	    }
	}
	/**
	 * Sometimes we need a more specialized message for the bad references, which diviate from a general algorithm.
	 * Like listing possible values etc.
	 * This method is responsible for such cases.
	 * @param originalMessage
	 * @param pr
	 * @param astNode
	 * @returns {string}
	 */
	function specializeReferenceError(originalMessage, property, astNode) {
	    if (property.nameId() == "type" && property.domain().universe().version() == "RAML08") {
	        if (property.domain().isAssignableFrom(universes.Universe08.Parameter.name)) {
	            return "type can be either of: string, number, integer, file, date or boolean";
	        }
	    }
	    if (astNode.parent() != null && universeHelpers.isSecuritySchemaType(astNode.parent().definition())) {
	        return originalMessage +
	            " Allowed values are:OAuth 1.0,OAuth 2.0,Basic Authentication,DigestSecurityScheme Authentication,x-{other}";
	    }
	    return originalMessage;
	}
	function nameForNonReference(astNode) {
	    var propertyName = astNode && astNode.lowLevel() && astNode.lowLevel().key();
	    if (propertyName === universes.Universe10.AbstractSecurityScheme.properties.type.name) {
	        var domain = astNode.parent() && astNode.parent().definition() && astNode.parent().definition().nameId();
	        if (domain === universes.Universe10.AbstractSecurityScheme.name) {
	            return "security scheme type";
	        }
	    }
	    else if (propertyName === universes.Universe08.BodyLike.properties.schema.name) {
	        var domain = astNode.parent() && astNode.parent().definition() && astNode.parent().definition().nameId();
	        if (domain === universes.Universe08.BodyLike.name) {
	            return "schema";
	        }
	    }
	}
	var DescriminatorOrReferenceValidator = (function () {
	    function DescriminatorOrReferenceValidator() {
	    }
	    DescriminatorOrReferenceValidator.prototype.validate = function (node, cb) {
	        var vl = node.value();
	        var valueKey = vl;
	        var pr = node.property();
	        if (typeof vl == 'string') {
	            checkReference(pr, node, vl, cb);
	            if (pr.range() instanceof defs.ReferenceType) {
	                var t = pr.range();
	                if (true) {
	                    var mockNode = jsyaml.createNode("" + vl, node.lowLevel().parent());
	                    mockNode._actualNode().startPosition = node.lowLevel().valueStart();
	                    mockNode._actualNode().endPosition = node.lowLevel().valueEnd();
	                    var stv = new hlimpl.StructuredValue(mockNode, node.parent(), node.property());
	                    var hn = stv.toHighLevel();
	                    if (hn) {
	                        hn.validate(cb);
	                    }
	                }
	            }
	        }
	        else if (vl != null) {
	            var st = vl;
	            if (st) {
	                valueKey = st.valueName();
	                var vn = st.valueName();
	                if (!checkReference(pr, node, vn, cb)) {
	                    var hnode = st.toHighLevel();
	                    if (hnode)
	                        hnode.validate(cb);
	                }
	            }
	            else {
	                valueKey = null;
	            }
	        }
	        else {
	            //there is no value, but still a reference: calling checkReference with null value
	            if (node.definition().isAssignableFrom(universes.Universe10.Reference.name)) {
	                checkReference(pr, node, null, cb);
	            }
	        }
	        if (valueKey) {
	            var validation = isValid(pr.range(), node.parent(), valueKey, pr);
	            if (validation instanceof Error) {
	                cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, validation.message, node));
	                validation = null;
	            }
	        }
	    };
	    return DescriminatorOrReferenceValidator;
	}());
	var allowOverride = { resources: 1, queryParameters: 1, headers: 1, body: 1, methods: 1, responses: 1 };
	var RAMLVersionAndFragmentValidator = (function () {
	    function RAMLVersionAndFragmentValidator() {
	    }
	    RAMLVersionAndFragmentValidator.prototype.validate = function (node, v) {
	        var u = node.universe();
	        var tv = u.getTypedVersion();
	        if (tv) {
	            if (util.startsWith(tv, "#%")) {
	                if (tv !== "#%RAML 0.8" && tv !== "#%RAML 1.0") {
	                    var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "Unknown version of RAML expected to see one of '#%RAML 0.8' or '#%RAML 1.0'", node);
	                    v.accept(i);
	                }
	                var tl = u.getTopLevel();
	                if (tl) {
	                    if (tl != node.definition().nameId()) {
	                        if (node.definition().nameId() == "Api") {
	                            var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "Unknown top level type:" + tl, node);
	                            v.accept(i);
	                        }
	                    }
	                }
	            }
	        }
	    };
	    return RAMLVersionAndFragmentValidator;
	}());
	var RequiredPropertiesAndContextRequirementsValidator = (function () {
	    function RequiredPropertiesAndContextRequirementsValidator() {
	    }
	    RequiredPropertiesAndContextRequirementsValidator.prototype.validate = function (node, v) {
	        var _this = this;
	        (node.definition()).getAdapter(services.RAMLService).getContextRequirements().forEach(function (x) {
	            if (!node.checkContextValue(x.name, x.value, x.value)) {
	                var message = x.name + (" should be " + x.value + " to use type " + node.definition().nameId());
	                if (x.name == 'location' && x.value == "ParameterLocation.FORM") {
	                    message = "file type can be only used in web forms";
	                }
	                v.accept(createIssue(hl.IssueCode.MISSED_CONTEXT_REQUIREMENT, message, node));
	            }
	        });
	        var t;
	        var isInlinedTemplate = node.definition().getAdapter(services.RAMLService).isInlinedTemplates();
	        if (isInlinedTemplate) {
	            var paramsMap = {};
	            for (var _i = 0, _a = node.lowLevel().children(); _i < _a.length; _i++) {
	                var ch = _a[_i];
	                paramsMap[ch.key()] = ch.value(true);
	            }
	            var templateKind = node.definition().isAssignableFrom(universes.Universe10.Trait.name) ? "trait" : "resource type";
	            var vt = new expander.ValueTransformer(templateKind, node.definition().nameId(), paramsMap);
	            var parent = node.parent();
	            var def = parent ? parent.definition() : node.definition();
	            while (parent != null && !universeHelpers.isResourceType(def) && !universeHelpers.isMethodType(def)) {
	                parent = parent.parent();
	            }
	            t = new expander.DefaultTransformer(parent, vt);
	        }
	        node.definition().requiredProperties().forEach(function (x) {
	            if (isInlinedTemplate) {
	                var paths = x.getAdapter(services.RAMLPropertyService).meta("templatePaths");
	                if (paths) {
	                    var parent = node.parent();
	                    var hasSufficientChild = false;
	                    for (var _i = 0, paths_1 = paths; _i < paths_1.length; _i++) {
	                        var path = paths_1[_i];
	                        path = path.map(function (x) { return t.transform(x).value; });
	                        if (_this.checkPathSufficiency(parent.lowLevel(), path, parent)) {
	                            hasSufficientChild = true;
	                            break;
	                        }
	                    }
	                    if (!hasSufficientChild) {
	                        return;
	                    }
	                }
	            }
	            var r = x.range();
	            if (r.hasArrayInHierarchy()) {
	                r = r.arrayInHierarchy().componentType();
	            }
	            if (r.hasValueTypeInHierarchy()) {
	                var nm = node.attr(x.nameId());
	                var gotValue = false;
	                if (nm != null) {
	                    if (nm.lowLevel().kind() == yaml.Kind.SCALAR
	                        || nm.lowLevel().valueKind() == yaml.Kind.SCALAR
	                        || nm.lowLevel().kind() == yaml.Kind.INCLUDE_REF
	                        || nm.lowLevel().valueKind() == yaml.Kind.INCLUDE_REF
	                        || (nm.lowLevel().valueKind() === null && !isInlinedTemplate)) {
	                        //if(nm.value()!=null){
	                        gotValue = true;
	                    }
	                    else if (nm.lowLevel().children().length != 0) {
	                        gotValue = true;
	                    }
	                }
	                if (!gotValue) {
	                    var msg = "Missing required property " + x.nameId();
	                    if (isInlinedTemplate) {
	                        msg = "value was not provided for parameter: " + x.nameId();
	                    }
	                    var i = createIssue(hl.IssueCode.MISSING_REQUIRED_PROPERTY, msg, node);
	                    v.accept(i);
	                }
	            }
	            else {
	                var el = node.elementsOfKind(x.nameId());
	                if (!el || el.length == 0) {
	                    var i = createIssue(hl.IssueCode.MISSING_REQUIRED_PROPERTY, "Missing required property " + x.nameId(), node);
	                    v.accept(i);
	                }
	            }
	        });
	    };
	    RequiredPropertiesAndContextRequirementsValidator.prototype.checkPathSufficiency = function (node, path, hlParent) {
	        if (hlParent == null || hlParent.definition() == null) {
	            return false;
	        }
	        var definition = hlParent.definition();
	        if (universeHelpers.isResourceTypeType(definition) || universeHelpers.isTraitType(definition)) {
	            return true;
	        }
	        if (path.length == 0) {
	            return false;
	        }
	        if (node == null) {
	            return false;
	        }
	        var segment = path[0];
	        if (segment == null) {
	            return false;
	        }
	        if (segment == "/") {
	            return this.checkPathSufficiency(node, path.slice(1), hlParent);
	        }
	        if (segment.length == 0) {
	            return true;
	        }
	        var children = node.children().filter(function (x) { return x.key() == segment; });
	        if (children.length == 0) {
	            path.indexOf("/") < 0;
	        }
	        var lowLevel = children[0];
	        if (lowLevel instanceof proxy.LowLevelCompositeNode) {
	            lowLevel = lowLevel.primaryNode();
	        }
	        if (lowLevel == null) {
	            return path.indexOf("/") < 0;
	        }
	        if (lowLevel.key() == "type") {
	            return true;
	        }
	        if (path.length == 1) {
	            // if(hlName==prop.nameId()&&node.definition().nameId()==prop.domain().nameId()){
	            //     return true;
	            // }
	            return lowLevel == null || lowLevel.value() == null;
	        }
	        else {
	            var path1 = path.slice(1);
	            return this.checkPathSufficiency(lowLevel, path1, hlParent);
	        }
	    };
	    return RequiredPropertiesAndContextRequirementsValidator;
	}());
	var ScalarQuoteValidator = (function () {
	    function ScalarQuoteValidator() {
	    }
	    ScalarQuoteValidator.prototype.validate = function (node, v) {
	        var r = node.lowLevel().unit();
	        node.lowLevel().visit(function (x) {
	            if (x.unit() != r) {
	                return false;
	            }
	            if (x.value() && x._node && x._node.value) {
	                if (x._node.value.doubleQuoted) {
	                    var ind = (x.value() + "").indexOf(":");
	                    var nl = (x.value() + "").indexOf("\n");
	                    if (ind != -1 && nl != -1 && (!x.includePath() || x.includePath().length == 0)) {
	                        var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "Suspicious double quoted multiline scalar, it is likely that you forgot closing \" " + x.value(), node, true);
	                        i.start = x._node.value.startPosition;
	                        i.end = x._node.value.endPosition;
	                        if (i.start == i.end) {
	                            i.end++;
	                        }
	                        v.accept(i);
	                    }
	                }
	            }
	            return true;
	        });
	    };
	    return ScalarQuoteValidator;
	}());
	var FixedFacetsValidator = (function () {
	    function FixedFacetsValidator() {
	    }
	    FixedFacetsValidator.prototype.validate = function (node, v) {
	        var nc = node.definition();
	        var dnode = nc.getAdapter(services.RAMLService).getDeclaringNode();
	        if (dnode) {
	            var rof = dnode.parsedType();
	            var dp = node.lowLevel().dumpToObject(true);
	            if (dp) {
	                dp = dp[Object.keys(dp)[0]];
	            }
	            if (!dp) {
	                if (nc.isAnnotationType()) {
	                    if (nc.isAssignableFrom(universes.Universe10.StringType.name)) {
	                        dp = "";
	                    }
	                }
	            }
	            var validateObject = rof.validate(dp, true, false);
	            if (!validateObject.isOk()) {
	                validateObject.getErrors().forEach(function (e) { return v.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, e.getMessage(), mapPath(node, e), false)); });
	            }
	        }
	    };
	    return FixedFacetsValidator;
	}());
	var TypeDeclarationValidator = (function () {
	    function TypeDeclarationValidator() {
	    }
	    TypeDeclarationValidator.prototype.validate = function (node, v) {
	        var nc = node.definition();
	        var rof = node.parsedType();
	        var validateObject = rof.validateType(node.types().getAnnotationTypeRegistry());
	        if (!validateObject.isOk()) {
	            validateObject.getErrors().forEach(function (e) { v.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, e.getMessage(), mapPath(node, e), e.isWarning())); });
	        }
	    };
	    return TypeDeclarationValidator;
	}());
	function mapPath(node, e) {
	    var src = e.getValidationPath();
	    return findElementAtPath(node, src);
	}
	function findElementAtPath(n, p) {
	    if (!p) {
	        return n;
	    }
	    var chld = n.children();
	    for (var i = 0; i < chld.length; i++) {
	        if (chld[i].name() === p.name) {
	            return findElementAtPath(chld[i], p.child);
	        }
	    }
	    if (!n.lowLevel()) {
	        return n;
	    }
	    var lchld = n.lowLevel().children();
	    for (var i = 0; i < lchld.length; i++) {
	        if (lchld[i].key() === p.name) {
	            var nn = new hlimpl.BasicASTNode(lchld[i], n);
	            return findElementAtPath(nn, p.child);
	        }
	    }
	    if (!isNaN(p.name)) {
	        if (lchld[p.name]) {
	            var node = lchld[p.name];
	            var nn = new hlimpl.BasicASTNode(node, n);
	            return findElementAtPath(nn, p.child);
	        }
	    }
	    return n;
	}
	var CompositeNodeValidator = (function () {
	    function CompositeNodeValidator() {
	    }
	    CompositeNodeValidator.prototype.validate = function (node, acceptor) {
	        if (node.definition().isAnnotationType()) {
	            return;
	        }
	        if (node.lowLevel().keyKind() == yaml.Kind.SEQ) {
	            var isPattern = node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name);
	            if (!isPattern) {
	                acceptor.accept(createIssue(hl.IssueCode.UNKNOWN_NODE, "Node key can not be sequence", node));
	            }
	        }
	        if (node.definition().key() == universes.Universe08.GlobalSchema) {
	            if (node.lowLevel().valueKind() != yaml.Kind.SCALAR && node.lowLevel().valueKind() != yaml.Kind.INCLUDE_REF) {
	                acceptor.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "schema " + node.name() + " must be a string", node));
	            }
	        }
	        if (!node.parent()) {
	            new RAMLVersionAndFragmentValidator().validate(node, acceptor);
	            //Note: overloading validator now checks for oveloading and rejects it
	            if (node.definition().key() == universes.Universe08.Api || node.definition().key() == universes.Universe10.Api) {
	                new OverloadingValidator().validateApi(node.wrapperNode(), acceptor);
	            }
	            new ScalarQuoteValidator().validate(node, acceptor);
	            lintNode(node, acceptor);
	        }
	        new OverlayNodesValidator().validate(node, acceptor);
	        var nc = node.definition();
	        if (nc.key() == universes.Universe08.BodyLike) {
	            if (node.lowLevel().children().map(function (x) { return x.key(); }).some(function (x) { return x === "formParameters"; })) {
	                if (node.parent() && node.parent().definition().key() == universes.Universe08.Response) {
	                    var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "form parameters can not be used in response", node);
	                    acceptor.accept(i);
	                }
	                else if (node.lowLevel().children().map(function (x) { return x.key(); }).some(function (x) { return x === "schema" || x === "example"; })) {
	                    var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "formParameters cannot be used together with the example or schema properties", node);
	                    acceptor.accept(i);
	                }
	            }
	        }
	        if (nc.key() == universes.Universe10.OAuth2SecuritySchemeSettings) {
	            var requireUrl = false;
	            node.attributes("authorizationGrants").forEach(function (x) {
	                var vl = x.value();
	                if (vl === "authorization_code" || vl === "implicit") {
	                    requireUrl = true;
	                }
	                else if (vl !== "password" && vl !== 'client_credentials') {
	                    if (vl && typeof vl === "string" && vl.indexOf("://") == -1) {
	                        var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "authorizationGrants should be one of authorization_code,implicit,password,client_credentials or to be an abolute URI", x);
	                        acceptor.accept(i);
	                    }
	                }
	            });
	            if (requireUrl) {
	                if (!node.attr("authorizationUri")) {
	                    var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "authorizationUri is required when `authorization_code` or `implicit` grant type are used ", node);
	                    acceptor.accept(i);
	                }
	            }
	        }
	        //validation of enum values;
	        if (node.definition().isAssignableFrom(universes.Universe08.Parameter.name)
	            || node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
	            var vls = node.attributes("enum").map(function (x) { return x.value(); });
	            if (vls.length != _.uniq(vls).length) {
	                var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "enum contains duplicated values", node);
	                acceptor.accept(i);
	            }
	            if (node.definition().isAssignableFrom(universes.Universe08.NumberTypeDeclaration.name) || node.definition().isAssignableFrom(universes.Universe10.NumberTypeDeclaration.name)) {
	                var isInteger = node.definition().isAssignableFrom(universes.Universe08.IntegerTypeDeclaration.name) || node.definition().isAssignableFrom(universes.Universe10.IntegerTypeDeclaration.name);
	                node.attributes("enum").forEach(function (attribute) {
	                    var value = isInteger ? parseInt(attribute.value()) : parseFloat(attribute.value());
	                    var isValid = isInteger ? !isNaN(value) && attribute.value().indexOf('.') === -1 : !isNaN(value);
	                    if (!isValid) {
	                        var issue = createIssue(hl.IssueCode.NODE_HAS_VALUE, (isInteger ? "Integer" : "Number") + " is expected", attribute);
	                        acceptor.accept(issue);
	                    }
	                });
	            }
	        }
	        if (universeHelpers.isResourceTypeType(node.definition())) {
	            if (node.value() == null && node.lowLevel().value(true) === "null") {
	                acceptor.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Resource type can not be null", node));
	            }
	        }
	        checkPropertyQuard(node, acceptor);
	        var nodeValue = node.value();
	        if ((typeof nodeValue == 'string'
	            || typeof nodeValue == 'number'
	            || typeof nodeValue == 'boolean')
	            && !node.definition().getAdapter(services.RAMLService).allowValue()) {
	            if (node.parent()) {
	                var i = createIssue(hl.IssueCode.NODE_HAS_VALUE, "node " + node.name() + " can not be a scalar", node);
	                acceptor.accept(i);
	            }
	        }
	        new RequiredPropertiesAndContextRequirementsValidator().validate(node, acceptor);
	        new ValidateChildrenKeys().validate(node, acceptor);
	        new NodeSpecificValidator().validate(node, acceptor);
	    };
	    return CompositeNodeValidator;
	}());
	var BaseUriParameterValidator = (function () {
	    function BaseUriParameterValidator() {
	    }
	    BaseUriParameterValidator.prototype.validate = function (node, acceptor) {
	        //we cant have "version" base uri parameter
	        var nameAttributeValue = node.attrValue(universes.Universe10.TypeDeclaration.properties.name.name);
	        if ("version" == nameAttributeValue) {
	            //I dont like the message, but its coming from JS 0.8 parser @Denis
	            acceptor.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY, "version parameter not allowed here", node));
	        }
	    };
	    return BaseUriParameterValidator;
	}());
	var NodeSpecificValidatorRegistryEntry = (function () {
	    /**
	     *
	     * @param definition - array of definitions from universes
	     * @param propertyName - name of the property. May be null, then property is not tested.
	     * @param assignableFrom - whether instead of direct definition comparison, the tested node will be checked
	     * for assignability from the specified definitions
	     */
	    function NodeSpecificValidatorRegistryEntry(definitions, propertyName, validator, assignableFrom) {
	        if (assignableFrom === void 0) { assignableFrom = false; }
	        this.definitions = definitions;
	        this.propertyName = propertyName;
	        this.assignableFrom = assignableFrom;
	        this.validator = validator;
	    }
	    /**
	     * Checks whether this entry is applicable to the node. If so, invokes its validator.
	     * @param node
	     * @param cb
	     */
	    NodeSpecificValidatorRegistryEntry.prototype.validate = function (node, acceptor) {
	        var nodeDefinition = node.definition();
	        if (nodeDefinition == null)
	            return;
	        var definitionMatched = false;
	        if (!this.assignableFrom) {
	            definitionMatched =
	                this.definitions.some(function (currentDefinition) { return currentDefinition === nodeDefinition; });
	        }
	        else {
	            definitionMatched =
	                this.definitions.some(function (currentDefinition) { return nodeDefinition.isAssignableFrom(currentDefinition.name); });
	        }
	        if (!definitionMatched)
	            return;
	        if (this.propertyName != null) {
	            if (node.property() == null)
	                return;
	            if (node.property().nameId() != this.propertyName)
	                return;
	        }
	        //definition and property matched, invoking validator
	        this.validator.validate(node, acceptor);
	    };
	    return NodeSpecificValidatorRegistryEntry;
	}());
	/**
	 * A central switch for validations specific to a particular node.
	 * In future it would be nice to migrate all node-specific validation scattered around the code here.
	 */
	var NodeSpecificValidator = (function () {
	    function NodeSpecificValidator() {
	    }
	    NodeSpecificValidator.createRegistry = function () {
	        var result = [];
	        NodeSpecificValidator.registerValidator(result, [universes.Universe10.TypeDeclaration, universes.Universe08.Parameter], universes.Universe10.Api.properties.baseUriParameters.name, new BaseUriParameterValidator(), true);
	        return result;
	    };
	    NodeSpecificValidator.registerValidator = function (listToAddTo, definitions, propertyName, validator, assignableFrom) {
	        if (assignableFrom === void 0) { assignableFrom = false; }
	        var entry = new NodeSpecificValidatorRegistryEntry(definitions, propertyName, validator, assignableFrom);
	        listToAddTo.push(entry);
	    };
	    NodeSpecificValidator.prototype.validate = function (node, acceptor) {
	        NodeSpecificValidator.entries.forEach(function (entry) { return entry.validate(node, acceptor); });
	    };
	    NodeSpecificValidator.entries = NodeSpecificValidator.createRegistry();
	    return NodeSpecificValidator;
	}());
	var OverlayNodesValidator = (function () {
	    function OverlayNodesValidator() {
	    }
	    /**
	     * Checks that this node is in white list and
	     * makes itself and all of its children allowed to exist in overlay even
	     * if there is no master counterpart
	     * @param node
	     * @param root
	     * @returns {boolean}
	     */
	    OverlayNodesValidator.prototype.allowsAnyChildren = function (node, root) {
	        var property = node.property();
	        var definition = node.definition();
	        //accepting new annotation types
	        if ((universeHelpers.isAnnotationTypeType(definition) || universeHelpers.isTypeDeclarationTypeOrDescendant(definition))
	            && universeHelpers.isAnnotationTypesProperty(property))
	            return true;
	        //accepting new top-level type declarations
	        if (node.parent() == root && universeHelpers.isTypesProperty(property)
	            && universeHelpers.isTypeDeclarationTypeOrDescendant(definition))
	            return true;
	        //as we allow types, it is logical to also allow schemas as "schemas are only aliases for types"
	        if (universeHelpers.isSchemasProperty(property)
	            && universeHelpers.isTypeDeclarationTypeOrDescendant(definition))
	            return true;
	        //accepting documentation
	        if (node.parent() == root && universeHelpers.isDocumentationProperty(property)
	            && universeHelpers.isDocumentationType(definition))
	            return true;
	        //accepting annotations
	        if (universeHelpers.isAnnotationsProperty(property))
	            return true;
	        //uses allowed
	        if (universeHelpers.isUsesProperty(property))
	            return true;
	        //examples allowed
	        if (universeHelpers.isExamplesProperty(property))
	            return true;
	        return false;
	    };
	    /**
	     * Checks that this node is allowed to exist in overlay even if there is no master counterpart
	     * due to it or its parent being in the white list.
	     * @param node
	     * @param root
	     */
	    OverlayNodesValidator.prototype.nodeAllowedDueToParent = function (node, root) {
	        var currentNode = node;
	        while (currentNode != root && currentNode != null) {
	            if (this.allowsAnyChildren(currentNode, root)) {
	                return true;
	            }
	            currentNode = currentNode.parent();
	        }
	        return false;
	    };
	    OverlayNodesValidator.prototype.validate = function (node, v) {
	        var root = node.root();
	        var property = node.property();
	        var definition = node.definition();
	        //we are only validating overlays
	        if (!universeHelpers.isOverlayType(root.definition()))
	            return;
	        //for root only validate properties
	        if (node == root) {
	            this.validateProperties(node, v);
	            return;
	        }
	        //we have a whitelist of IHighLevelNodes allowed to be added in an overlay like new types, annotation types,
	        //annotation etc. The contents of such nodes is checked here.
	        if (this.nodeAllowedDueToParent(node, root))
	            return;
	        //checking for a node, this node overrides
	        var overrides = root.knownIds();
	        if (!overrides) {
	            //should never happen
	            return;
	        }
	        var override = overrides.hasOwnProperty(node.id());
	        if (override) {
	            //overrides are allowed, but we need to check properties, this override potentially brings in or changes:
	            this.validateProperties(node, v);
	            return;
	        }
	        //otherwise reporting an illegal node:
	        v.accept(createIssue(hl.IssueCode.ONLY_OVERRIDE_ALLOWED, "This node does not override any node from master api:" + node.id(), node));
	    };
	    OverlayNodesValidator.prototype.validateProperties = function (node, acceptor) {
	        node.attrs().forEach(function (attribute) {
	            //ignoring key properties as they are not overriding anything
	            if (attribute.property().getAdapter(services.RAMLPropertyService).isKey()) {
	                return;
	            }
	            //ignoring nodes, which are not coming from this node, but from is master chain
	            if (attribute.parent() != node) {
	                return;
	            }
	            //yes, that also happens!
	            if (attribute.isElement()) {
	                return;
	            }
	            //title allowed
	            if (universeHelpers.isTitlePropertyName(attribute.name()))
	                return;
	            //description allowed
	            if (universeHelpers.isDescriptionPropertyName(attribute.name()))
	                return;
	            //displayName allowed
	            if (universeHelpers.isDisplayNamePropertyName(attribute.name()))
	                return;
	            //usage allowed
	            if (universeHelpers.isUsagePropertyName(attribute.name()))
	                return;
	            //example allowed
	            if (universeHelpers.isExampleProperty(attribute.property()))
	                return;
	            //masterRef allowed
	            if (universeHelpers.isMasterRefProperty(attribute.property()))
	                return;
	            //annotations allowed
	            if (universeHelpers.isAnnotationsProperty(attribute.property()))
	                return;
	            //uses allowed
	            if (universeHelpers.isUsesProperty(attribute.property()))
	                return;
	            //reporting the error
	            var message = "Property " + attribute.name() +
	                " is not allowed to be overriden or added in overlays";
	            acceptor.accept(createIssue(hl.IssueCode.ONLY_OVERRIDE_ALLOWED, message, attribute));
	        });
	    };
	    return OverlayNodesValidator;
	}());
	var RecurrentOverlayValidator = (function () {
	    function RecurrentOverlayValidator() {
	    }
	    RecurrentOverlayValidator.prototype.validate = function (node, v) {
	        var _this = this;
	        var z = new OverlayNodesValidator();
	        z.validate(node, v);
	        node.elements().forEach(function (x) { return _this.validate(x, v); });
	    };
	    return RecurrentOverlayValidator;
	}());
	var RecurrentValidateChildrenKeys = (function () {
	    function RecurrentValidateChildrenKeys() {
	    }
	    RecurrentValidateChildrenKeys.prototype.val = function (node, v, p) {
	        var _this = this;
	        if (node.kind() == yaml.Kind.MAP || node.kind() == yaml.Kind.MAPPING) {
	            var ms = {};
	            node.children().forEach(function (x) {
	                var c = x.key();
	                if (c) {
	                    if (ms.hasOwnProperty(c)) {
	                        var issue = createIssue(hl.IssueCode.KEY_SHOULD_BE_UNIQUE_INTHISCONTEXT, "keys should be unique", p, false);
	                        if (x.unit() == p.lowLevel().unit()) {
	                            issue.start = x.keyStart();
	                            issue.end = x.keyEnd();
	                        }
	                        v.accept(issue);
	                    }
	                    ms[c] = 1;
	                }
	            });
	        }
	        node.children().forEach(function (x) {
	            _this.val(x, v, p);
	        });
	    };
	    RecurrentValidateChildrenKeys.prototype.validate = function (node, v) {
	        this.val(node.lowLevel(), v, node);
	    };
	    return RecurrentValidateChildrenKeys;
	}());
	var ValidateChildrenKeys = (function () {
	    function ValidateChildrenKeys() {
	    }
	    ValidateChildrenKeys.prototype.validate = function (node, acceptor) {
	        //validation is being performed at high level instead of low-level
	        //to provide more meaningful and specific error messages
	        this.validateChildElements(node, acceptor);
	        var lowLevelChildren = node.lowLevel().children();
	        var keyToLowLevelChildren = _.groupBy(lowLevelChildren.filter(function (x) { return x.key() != null; }), function (x) { return x.key(); });
	        this.validateChildAttributes(node, keyToLowLevelChildren, acceptor);
	        this.validateUnrecognizedLowLevelChildren(node, keyToLowLevelChildren, acceptor);
	    };
	    ValidateChildrenKeys.prototype.validateChildElements = function (node, acceptor) {
	        //testing for child elements having equal keys
	        var keyToElements = {};
	        var childElements = node.directChildren().filter(function (x) { return x.isElement(); });
	        childElements.forEach(function (childNode) {
	            var childElement = childNode;
	            if (childElement["_computed"]) {
	                return;
	            }
	            if (!childElement.name()) {
	                return; //handling nodes with no key (documentation)
	            }
	            var elementKey = childElement.name() + childElement.property().nameId();
	            if (keyToElements.hasOwnProperty(elementKey)) {
	                if (!childElement.isNamePatch()) {
	                    keyToElements[elementKey].push(childElement);
	                }
	            }
	            else {
	                keyToElements[elementKey] = [childElement];
	            }
	        });
	        Object.keys(keyToElements).forEach(function (key) {
	            var childElements = keyToElements[key];
	            if (!childElements || childElements.length < 2)
	                return;
	            childElements.forEach(function (childElement) {
	                var message = "";
	                var humanReadableName = getHumanReadableNodeName(childElement);
	                if (humanReadableName) {
	                    var capitalized = humanReadableName.charAt(0).toUpperCase() + humanReadableName.slice(1);
	                    var message = capitalized + " '" + childElement.name() + "' already exists";
	                }
	                else {
	                    message = childElement.name() + " already exists in this context";
	                }
	                var issue = createIssue(hl.IssueCode.KEY_SHOULD_BE_UNIQUE_INTHISCONTEXT, message, childElement);
	                acceptor.accept(issue);
	            });
	        });
	    };
	    ValidateChildrenKeys.prototype.validateChildAttributes = function (node, keyToLowLevelChildren, acceptor) {
	        var highLevelAttributes = this.getHighLevelAttributes(node);
	        var nameToHighLevelAttributes = _.groupBy(highLevelAttributes, function (x) { return x.name(); });
	        var allowsAnyAndHasRequireds = this.allowsAnyAndHasRequireds(node);
	        Object.keys(nameToHighLevelAttributes).forEach(function (attributeName) {
	            if (nameToHighLevelAttributes[attributeName].length < 2) {
	                return;
	            }
	            var isUnknown = nameToHighLevelAttributes[attributeName][0].isUnknown();
	            var isMultiValue = !isUnknown && nameToHighLevelAttributes[attributeName][0].property().isMultiValue();
	            if (isMultiValue && (node.definition().isAssignableFrom(universes.Universe08.SecuritySchemeSettings.name) ||
	                node.definition().isAssignableFrom(universes.Universe10.SecuritySchemeSettings.name))) {
	                isMultiValue = keyToLowLevelChildren[attributeName] && keyToLowLevelChildren[attributeName].length === 1;
	            }
	            if ((isUnknown && allowsAnyAndHasRequireds) || !isMultiValue ||
	                (isMultiValue && keyToLowLevelChildren[attributeName] != null && keyToLowLevelChildren[attributeName].length > 1)) {
	                //we blame even multivalue properties if they have duplicated low-level keys as YAML forbids this
	                nameToHighLevelAttributes[attributeName].forEach(function (attribute) {
	                    var message = "property already used: '" + (attribute.property() ? attribute.property().nameId() : attribute.name()) + "'";
	                    var humanReadableParent = getHumanReadableNodeName(attribute.parent());
	                    if (humanReadableParent) {
	                        var capitalizedParent = humanReadableParent.charAt(0).toUpperCase() + humanReadableParent.slice(1);
	                        message = capitalizedParent + " " + message;
	                    }
	                    var issue = createIssue(hl.IssueCode.PROPERTY_EXPECT_TO_HAVE_SINGLE_VALUE, message, attribute);
	                    acceptor.accept(issue);
	                });
	            }
	        });
	    };
	    ValidateChildrenKeys.prototype.validateUnrecognizedLowLevelChildren = function (node, keyToLowLevelChildren, acceptor) {
	        var highLevelChildren = node.directChildren();
	        var nameToHighLevelChildren = _.groupBy(highLevelChildren, function (x) { return x.name(); });
	        Object.keys(keyToLowLevelChildren).forEach(function (lowLevelChildKey) {
	            if (lowLevelChildKey) {
	                if (keyToLowLevelChildren[lowLevelChildKey].length > 1 && !nameToHighLevelChildren[lowLevelChildKey]) {
	                    if (node.definition().isAssignableFrom(universes.Universe10.ObjectTypeDeclaration.name)) {
	                        return;
	                    }
	                    var message = "property already used: '" + lowLevelChildKey + "'";
	                    var humanReadableNode = getHumanReadableNodeName(node);
	                    if (humanReadableNode) {
	                        var capitalizedParent = humanReadableNode.charAt(0).toUpperCase() + humanReadableNode.slice(1);
	                        message = capitalizedParent + " " + message;
	                    }
	                    keyToLowLevelChildren[lowLevelChildKey].forEach(function (lowLevelChild) {
	                        var i = createLLIssue(hl.IssueCode.PROPERTY_EXPECT_TO_HAVE_SINGLE_VALUE, message, lowLevelChild, node);
	                        i.start = lowLevelChild.keyStart();
	                        i.end = lowLevelChild.keyEnd();
	                        acceptor.accept(i);
	                    });
	                }
	            }
	        });
	    };
	    ValidateChildrenKeys.prototype.filterMultiValueAnnotations = function (node, keyToLowLevelChildren, acceptor) {
	        var highLevelAttributes = this.getHighLevelAttributes(node);
	        var computedAnnotationsMultiplValues = false;
	        Object.keys(keyToLowLevelChildren).forEach(function (lowLevelChildKey) {
	            if (lowLevelChildKey.charAt(0) !== '(' || keyToLowLevelChildren[lowLevelChildKey].length < 2) {
	                return;
	            }
	        });
	    };
	    ValidateChildrenKeys.prototype.getHighLevelAttributes = function (node) {
	        var allowsAnyAndHasRequireds = this.allowsAnyAndHasRequireds(node);
	        return node.directChildren().filter(function (x) { return x.isAttr() || allowsAnyAndHasRequireds; });
	    };
	    ValidateChildrenKeys.prototype.allowsAnyAndHasRequireds = function (node) {
	        var requireds = node.definition().requiredProperties();
	        var hasRequireds = requireds && requireds.length > 0;
	        var ramlService = node.definition().getAdapter(services.RAMLService);
	        var isAllowAny = ramlService && ramlService.getAllowAny();
	        var anyExceptRequireds = isAllowAny && hasRequireds;
	        return anyExceptRequireds;
	    };
	    return ValidateChildrenKeys;
	}());
	/**
	 * validates examples
	 */
	var ExampleValidator = (function () {
	    function ExampleValidator() {
	    }
	    ExampleValidator.prototype.validate = function (node, cb) {
	        //check if we expect to do strict validation
	        var strictValidation = this.isStrict(node);
	        if (!strictValidation) {
	            if (!settings.validateNotStrictExamples) {
	                return;
	            }
	        }
	        var pObj = this.parseObject(node, cb, strictValidation);
	        if (!pObj) {
	            return;
	        }
	        var schema = this.aquireSchema(node);
	        if (schema) {
	            schema.validate(pObj, cb, strictValidation);
	        }
	    };
	    ExampleValidator.prototype.isExampleNode = function (node) {
	        return this.isSingleExampleNode(node) || this.isExampleNodeInMultipleDecl(node);
	    };
	    ExampleValidator.prototype.isSingleExampleNode = function (node) {
	        return node.name() == universes.Universe10.TypeDeclaration.properties.example.name;
	    };
	    ExampleValidator.prototype.isExampleNodeInMultipleDecl = function (node) {
	        var parent = node.parent();
	        if (parent) {
	            return universeHelpers.isExampleSpecType(parent.definition());
	        }
	        return false;
	    };
	    ExampleValidator.prototype.findParentSchemaOrTypeAttribute = function (node) {
	        var attribute = node.parent().attr(universes.Universe10.TypeDeclaration.properties.schema.name);
	        if (attribute) {
	            return attribute;
	        }
	        attribute = node.parent().attr(universes.Universe10.TypeDeclaration.properties.type.name);
	        if (attribute) {
	            return attribute;
	        }
	        if (!node.parent()) {
	            return null;
	        }
	        attribute = node.parent().parent().attr(universes.Universe10.TypeDeclaration.properties.schema.name);
	        if (attribute) {
	            return attribute;
	        }
	        attribute = node.parent().parent().attr(universes.Universe10.TypeDeclaration.properties.type.name);
	        if (attribute) {
	            return attribute;
	        }
	        return null;
	    };
	    ExampleValidator.prototype.aquireSchema = function (node) {
	        var sp = node.parent().definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name);
	        if (this.isExampleNode(node)) {
	            var sampleRoot = node;
	            if (this.isExampleNodeInMultipleDecl(node)) {
	                sampleRoot = node.parent();
	            }
	            if (sampleRoot.parent()) {
	                if (sampleRoot.parent().definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name) && sampleRoot.parent().parent() === null) {
	                    sp = false;
	                }
	                else if (sampleRoot.parent().property().nameId() == universes.Universe10.LibraryBase.properties.types.name) {
	                    sp = false;
	                }
	                if (sampleRoot.parent().parent()) {
	                    var ppc = sampleRoot.parent().parent().definition().key();
	                    if (ppc == universes.Universe08.Method || ppc == universes.Universe10.Method) {
	                        if (sampleRoot.parent().property().nameId() == universes.Universe10.MethodBase.properties.queryParameters.name) {
	                        }
	                        else {
	                            sp = true;
	                        }
	                    }
	                    if (ppc == universes.Universe08.Response || ppc == universes.Universe10.Response) {
	                        sp = true;
	                    }
	                }
	            }
	        }
	        if (node.parent().definition().key() == universes.Universe08.BodyLike || sp) {
	            //FIXME MULTIPLE INHERITANCE
	            var sa = this.findParentSchemaOrTypeAttribute(node);
	            if (sa) {
	                var val = sa.value();
	                if (val instanceof hlimpl.StructuredValue) {
	                    return null;
	                }
	                var strVal = ("" + val).trim();
	                var so = null;
	                if (strVal.charAt(0) == "{") {
	                    try {
	                        so = su.getJSONSchema(strVal, new contentprovider.ContentProvider(sa.lowLevel().unit()));
	                    }
	                    catch (e) {
	                        return null;
	                    }
	                }
	                if (strVal.charAt(0) == "<") {
	                    try {
	                        so = su.getXMLSchema(strVal);
	                    }
	                    catch (e) {
	                        return null;
	                    }
	                }
	                if (so) {
	                    return {
	                        validate: function (pObje, cb, strict) {
	                            try {
	                                if (pObje.__$validated) {
	                                    return;
	                                }
	                                so.validateObject(pObje);
	                            }
	                            catch (e) {
	                                var illegalRequiredMessageStart = "Cannot assign to read only property '__$validated' of ";
	                                if (e.message && e.message.indexOf(illegalRequiredMessageStart) == 0) {
	                                    var propertyName = e.message.substr(illegalRequiredMessageStart.length, e.message.length - illegalRequiredMessageStart.length);
	                                    var patchedErrorMessage = "Invalid JSON schema. Potentially, required array containing value "
	                                        + propertyName + " has invalid location";
	                                    cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, patchedErrorMessage, sa, !strict));
	                                    return;
	                                }
	                                if (e.message == "Object.keys called on non-object") {
	                                    return;
	                                }
	                                cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Example does not conform to schema:" + e.message, node, !strict));
	                                return;
	                            }
	                            //validate using classical schema;
	                        }
	                    };
	                }
	                else {
	                    if (strVal.length > 0) {
	                        var nodeParent = node.parent();
	                        var grandParent = nodeParent && nodeParent.parent();
	                        var owner = nodeParent && nodeParent.definition() && nodeParent.definition().isAssignableFrom(universes.Universe10.ObjectTypeDeclaration.name) && nodeParent;
	                        owner = owner || (grandParent && grandParent.definition() && grandParent.definition().isAssignableFrom(universes.Universe10.ObjectTypeDeclaration.name) && grandParent);
	                        if (owner) {
	                            return this.typeValidator(owner, node);
	                        }
	                    }
	                }
	            }
	        }
	        return this.getSchemaFromModel(node);
	    };
	    ExampleValidator.prototype.getSchemaFromModel = function (node) {
	        var p = node.parent();
	        // if (node.property().nameId()==universes.Universe10.ExampleSpec.properties.content.name){
	        //     p=p.parent();
	        // }
	        return this.typeValidator(p, node);
	    };
	    ExampleValidator.prototype.typeValidator = function (p, node) {
	        var newVar = {
	            validate: function (pObje, cb, strict) {
	                var pt = p.parsedType();
	                if (pt && !pt.isUnknown()) {
	                    if (typeof pObje === "number" && pt.isString()) {
	                        pObje = "" + pObje;
	                    }
	                    if (typeof pObje === "boolean" && pt.isString()) {
	                        pObje = "" + pObje;
	                    }
	                    var validateObject = pt.validate(pObje, true);
	                    if (!validateObject.isOk()) {
	                        validateObject.getErrors().forEach(function (e) { return cb.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, e.getMessage(), node, !strict)); });
	                    }
	                }
	            }
	        };
	        return newVar;
	    };
	    ;
	    ExampleValidator.prototype.toObject = function (h, v, cb) {
	        var res = v.lowLevel().dumpToObject(true);
	        this.testDublication(h, v.lowLevel(), cb);
	        if (res["example"]) {
	            return res["example"];
	        }
	        if (res["content"]) {
	            return res["content"];
	        }
	    };
	    ExampleValidator.prototype.testDublication = function (h, v, cb) {
	        var _this = this;
	        var map = {};
	        v.children().forEach(function (x) {
	            if (x.key()) {
	                if (map[x.key()]) {
	                    cb.accept(createIssue(hl.IssueCode.KEY_SHOULD_BE_UNIQUE_INTHISCONTEXT, "Keys should be unique", new hlimpl.BasicASTNode(x, h.parent())));
	                }
	                map[x.key()] = x;
	            }
	            _this.testDublication(h, x, cb);
	        });
	    };
	    ExampleValidator.prototype.parseObject = function (node, cb, strictValidation) {
	        var pObj = null;
	        var vl = node.value();
	        var mediaType = getMediaType(node);
	        if (vl instanceof hlimpl.StructuredValue) {
	            //validate in context of type/schema
	            pObj = this.toObject(node, vl, cb);
	        }
	        else {
	            if (mediaType) {
	                if (isJson(mediaType)) {
	                    try {
	                        pObj = JSON.parse(vl);
	                    }
	                    catch (e) {
	                        cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Can not parse JSON:" + e.message, node, !strictValidation));
	                        return;
	                    }
	                }
	                if (isXML(mediaType)) {
	                    try {
	                        pObj = xmlutil(vl);
	                    }
	                    catch (e) {
	                        cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Can not parse XML:" + e.message, node, !strictValidation));
	                        return;
	                    }
	                }
	            }
	            else {
	                try {
	                    if (vl && vl.length > 0 && (vl.trim().charAt(0) == '[' || vl.trim().charAt(0) == '{' || vl.trim().charAt(0) == '<')) {
	                        pObj = JSON.parse(vl);
	                    }
	                    else {
	                        if (vl == "true") {
	                            return true;
	                        }
	                        else if (vl == "false") {
	                            return false;
	                        }
	                        else {
	                            var n = parseFloat(vl);
	                            if (!isNaN(n)) {
	                                return n;
	                            }
	                        }
	                        return vl;
	                    }
	                }
	                catch (e) {
	                    if (vl.trim().indexOf("<") == 0) {
	                        try {
	                            pObj = xmlutil(vl);
	                        }
	                        catch (e) {
	                            cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Can not parse XML:" + e.message, node, !strictValidation));
	                            return;
	                        }
	                    }
	                    else {
	                        //cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA, "Can not parse JSON:" + e.message, node, !strictValidation));
	                        return vl;
	                    }
	                }
	            }
	        }
	        return pObj;
	    };
	    ExampleValidator.prototype.isStrict = function (node) {
	        var strictValidation = false;
	        var strict = node.parent().attr("strict");
	        if (strict) {
	            if (strict.value() == 'true') {
	                strictValidation = true;
	            }
	        }
	        return strictValidation;
	    };
	    return ExampleValidator;
	}());
	exports.ExampleValidator = ExampleValidator;
	var toReadableName = function (template, toLowerCase, pluralize_) {
	    var templateName = changeCase.sentence(template);
	    if (!toLowerCase) {
	        templateName = changeCase.ucFirst(templateName);
	    }
	    if (pluralize_) {
	        templateName = pluralize.plural(templateName);
	    }
	    return templateName;
	};
	var OptionalPropertiesValidator = (function () {
	    function OptionalPropertiesValidator() {
	    }
	    OptionalPropertiesValidator.prototype.validate = function (node, v) {
	        if (node.isAttr()) {
	            if (!node.optional()) {
	                return;
	            }
	            var attr = node;
	            var prop = attr.property();
	            if (prop.isMultiValue() || prop.range().isArray()) {
	                return;
	            }
	            if (!prop.isFromParentKey()) {
	                var template = typeOfContainingTemplate(attr.parent());
	                if (template) {
	                    if (prop.isValueProperty()) {
	                        var templateNamePlural = toReadableName(template, true, true);
	                        var message = "Optional scalar properties are not allowed in "
	                            + templateNamePlural + " and their descendants: " + attr.name() + "?";
	                        var issue = createIssue(hl.IssueCode.INVALID_PROPERTY, message, attr, false);
	                        v.accept(issue);
	                    }
	                }
	            }
	        }
	        else if (node.isElement()) {
	            var aNode = node;
	            var prop = aNode.property();
	            var allowsQuestion = aNode.allowsQuestion();
	            if (!allowsQuestion) {
	                var propName = prop ? toReadableName(prop.nameId(), true, true) : 'API root';
	                aNode.optionalProperties().forEach(function (x) {
	                    aNode.children().forEach(function (y) {
	                        var message = "Optional properties are not allowed in "
	                            + propName + ": " + y.lowLevel().key() + "?";
	                        var issue = createIssue(hl.IssueCode.INVALID_PROPERTY, message, node, false);
	                        v.accept(issue);
	                    });
	                });
	            }
	            var def = node.asElement().definition();
	            if (node.optional() && def.universe().version() == "RAML10") {
	                var prop = node.property();
	                var isParam = universeHelpers.isQueryParametersProperty(prop)
	                    || universeHelpers.isUriParametersProperty(prop)
	                    || universeHelpers.isHeadersProperty(prop);
	                if (!universeHelpers.isMethodType(def) && !(universeHelpers.isTypeDeclarationType(def) && isParam)) {
	                    var issue = createIssue(hl.IssueCode.MISSED_CONTEXT_REQUIREMENT, "Only method nodes can be optional", node, false);
	                    v.accept(issue);
	                }
	            }
	        }
	    };
	    return OptionalPropertiesValidator;
	}());
	var UriParametersValidator = (function () {
	    function UriParametersValidator() {
	    }
	    UriParametersValidator.prototype.validate = function (node, v) {
	        var def = node.definition();
	        var baseUriPropName = universes.Universe10.Api.properties.baseUri.name;
	        var baseUriParamsPropName = universes.Universe10.Api.properties.baseUriParameters.name;
	        var uriPropName = universes.Universe10.Resource.properties.relativeUri.name;
	        var uriParamsPropName = universes.Universe10.ResourceBase.properties.uriParameters.name;
	        if (universeHelpers.isApiSibling(def)) {
	            this.inspectParameters(node, v, baseUriPropName, baseUriParamsPropName);
	        }
	        else if (universeHelpers.isResourceType(def)) {
	            var rootNode = node.root();
	            this.inspectParameters(node, v, baseUriPropName, baseUriParamsPropName, rootNode);
	            this.inspectParameters(node, v, uriPropName, uriParamsPropName);
	        }
	        else if (universeHelpers.isResourceTypeType(def)) {
	            var rootNode = node.root();
	            this.inspectParameters(node, v, baseUriPropName, baseUriParamsPropName, rootNode);
	        }
	    };
	    UriParametersValidator.prototype.inspectParameters = function (node, v, uriPropName, paramsPropName, rootNode) {
	        rootNode = rootNode || node;
	        var uriValue = '';
	        var uriAttr = rootNode.attr(uriPropName);
	        if (uriAttr) {
	            uriValue = uriAttr.value();
	            if (!uriValue) {
	                uriValue = '';
	            }
	        }
	        var paramElements = node.elementsOfKind(paramsPropName);
	        paramElements.forEach(function (x) {
	            var nameAttr = x.attr(universes.Universe10.TypeDeclaration.properties.name.name);
	            if (nameAttr) {
	                var name = nameAttr.value();
	                if (name != null) {
	                    if (uriValue.indexOf('{' + name + '}') < 0) {
	                        if (universeHelpers.isResourceTypeType(node.definition())) {
	                            if (name.indexOf('<<') >= 0) {
	                                return;
	                            }
	                        }
	                        var propNameReadable = pluralize.singular(changeCase.sentence(paramsPropName));
	                        var message = changeCase.ucFirst(propNameReadable) + " unused";
	                        var issue = createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, message, x, false);
	                        v.accept(issue);
	                    }
	                }
	            }
	        });
	    };
	    return UriParametersValidator;
	}());
	var TemplateCyclesDetector = (function () {
	    function TemplateCyclesDetector() {
	        this.nameProperty = universes.Universe10.ResourceType.properties.name.name;
	    }
	    TemplateCyclesDetector.prototype.validate = function (node, v) {
	        var definition = node.definition();
	        if (!(universeHelpers.isLibraryBaseSibling(definition) || universeHelpers.isApiType(definition))) {
	            return;
	        }
	        var resourceTypesProp = universes.Universe10.LibraryBase.properties.resourceTypes.name;
	        var typeProp = universes.Universe10.ResourceBase.properties.type.name;
	        var traitsProp = universes.Universe10.LibraryBase.properties.traits.name;
	        var isProp = universes.Universe10.MethodBase.properties.is.name;
	        var allResourceTypes = search.globalDeclarations(node)
	            .filter(function (x) { return universeHelpers.isResourceTypeType(x.definition()); });
	        var alltraits = search.globalDeclarations(node)
	            .filter(function (x) { return universeHelpers.isTraitType(x.definition()); });
	        this.checkCycles(allResourceTypes, typeProp, v);
	        this.checkCycles(alltraits, isProp, v);
	    };
	    TemplateCyclesDetector.prototype.checkCycles = function (templates, propName, v) {
	        var _this = this;
	        var templatesMap = {};
	        templates.forEach(function (x) {
	            var name = _this.templateName(x);
	            templatesMap[name] = x;
	        });
	        var templatesWithCycle = {};
	        templates.forEach(function (template) {
	            var name = _this.templateName(template);
	            if (templatesWithCycle[name]) {
	                //skip checking templates which are already known to have cycles in definition;
	                return;
	            }
	            _this.findCyclesInDefinition(template, propName, templatesMap).forEach(function (cycle) {
	                //mark templates which have cycles in definitions
	                cycle.forEach(function (x) { return templatesWithCycle[x] = true; });
	                cycle = cycle.reverse();
	                var message = toReadableName(template.definition().nameId())
	                    + " definition contains cycle: " + cycle.join(" -> ");
	                var issue = createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE, message, template, false);
	                v.accept(issue);
	            });
	        });
	    };
	    TemplateCyclesDetector.prototype.templateName = function (node) {
	        var nameAttribute = node.attr(this.nameProperty);
	        if (!nameAttribute) {
	            return null;
	        }
	        return nameAttribute.value();
	    };
	    TemplateCyclesDetector.prototype.findCyclesInDefinition = function (node, propName, templatesMap, occuredTemplates) {
	        if (occuredTemplates === void 0) { occuredTemplates = {}; }
	        var name = this.templateName(node);
	        if (occuredTemplates[name]) {
	            return [[name]];
	        }
	        var nextOccuredTemplates = {};
	        Object.keys(occuredTemplates).forEach(function (x) { return nextOccuredTemplates[x] = occuredTemplates[x]; });
	        nextOccuredTemplates[name] = true;
	        var occuredCycles = [];
	        var templatesRefs = node.attributes(propName);
	        for (var i = 0; i < templatesRefs.length; i++) {
	            var ref = templatesRefs[i];
	            var val = ref.value();
	            if (val) {
	                var refName = typeof (val) == 'string' ? val : val.valueName();
	                var template = templatesMap[refName];
	                if (template != null) {
	                    var newCycles = this.findCyclesInDefinition(template, propName, templatesMap, nextOccuredTemplates);
	                    newCycles.forEach(function (x) { return occuredCycles.push(x); });
	                }
	            }
	        }
	        occuredCycles.forEach(function (x) { return x.push(name); });
	        return occuredCycles;
	    };
	    return TemplateCyclesDetector;
	}());
	function isJson(s) {
	    return s.indexOf("json") != -1;
	}
	exports.isJson = isJson;
	function isXML(s) {
	    return s.indexOf("xml") != -1;
	}
	exports.isXML = isXML;
	function getMediaType(node) {
	    var vl = getMediaType2(node);
	    if (vl == 'body') {
	        var rootMedia = node.root().attr("mediaType");
	        if (rootMedia) {
	            return rootMedia.value();
	        }
	        return null;
	    }
	    return vl;
	}
	exports.getMediaType = getMediaType;
	function getMediaType2(node) {
	    if (node.parent()) {
	        var pc = node.parent().definition();
	        if (pc.key() == universes.Universe08.BodyLike) {
	            return node.parent().name();
	        }
	        if (node.parent().parent()) {
	            var ppc = node.parent().parent().definition().key();
	            if (ppc == universes.Universe08.Response || ppc == universes.Universe10.Response) {
	                if (node.parent().property().nameId() == universes.Universe08.Response.properties.headers.name) {
	                    return null;
	                }
	                return node.parent().name();
	            }
	            if (ppc == universes.Universe08.Method || ppc == universes.Universe10.Method) {
	                if (node.parent().property().nameId() == universes.Universe10.MethodBase.properties.queryParameters.name
	                    || node.parent().property().nameId() == universes.Universe10.MethodBase.properties.headers.name) {
	                    return null;
	                }
	                return node.parent().name();
	            }
	        }
	    }
	    return null;
	}
	var localError = function (node, c, w, message, p, prop) {
	    var contents = node.lowLevel().unit() && node.lowLevel().unit().contents();
	    var contentLength = contents && contents.length;
	    var st = node.lowLevel().start();
	    var et = node.lowLevel().end();
	    if (contentLength && contentLength < et) {
	        et = contentLength - 1;
	    }
	    if (node.lowLevel().key() && node.lowLevel().keyStart()) {
	        var ks = node.lowLevel().keyStart();
	        if (ks > 0) {
	            st = ks;
	        }
	        var ke = node.lowLevel().keyEnd();
	        if (ke > 0) {
	            et = ke;
	        }
	    }
	    if (et < st) {
	        et = st + 1;
	        //this happens for empty APIs, when we basically have nothing to parse
	        if (node.isElement()) {
	            var definition = node.definition();
	            if (universeHelpers.isApiType(definition)) {
	                st = contentLength == 0 ? 0 : contentLength - 1;
	                et = st;
	            }
	        }
	    }
	    if (prop && !prop.getAdapter(services.RAMLPropertyService).isMerged() && node.parent() == null) {
	        var nm = _.find(node.lowLevel().children(), function (x) { return x.key() == prop.nameId(); });
	        if (nm) {
	            var ks = nm.keyStart();
	            var ke = nm.keyEnd();
	            if (ks > 0 && ke > ks) {
	                st = ks;
	                et = ke;
	            }
	        }
	    }
	    return {
	        code: c,
	        isWarning: w,
	        message: message,
	        node: node,
	        start: st,
	        end: et,
	        path: p ? (node.lowLevel().unit() ? node.lowLevel().unit().path() : "") : null,
	        extras: [],
	        unit: node ? node.lowLevel().unit() : null
	    };
	};
	var localLowLevelError = function (node, highLevelAnchor, issueCode, isWarning, message, path) {
	    var contents = node.unit() && node.unit().contents();
	    var contentLength = contents && contents.length;
	    var st = node.start();
	    var et = node.end();
	    if (contentLength && contentLength >= et) {
	        et = contentLength - 1;
	    }
	    if (node.key() && node.keyStart()) {
	        var ks = node.keyStart();
	        if (ks > 0) {
	            st = ks;
	        }
	        var ke = node.keyEnd();
	        if (ke > 0) {
	            et = ke;
	        }
	    }
	    return {
	        code: issueCode,
	        isWarning: isWarning,
	        message: message,
	        node: highLevelAnchor,
	        start: st,
	        end: et,
	        path: path ? (node.unit() ? node.unit().path() : "") : null,
	        extras: [],
	        unit: node ? node.unit() : null
	    };
	};
	function createIssue(c, message, node, w) {
	    if (w === void 0) { w = false; }
	    //console.log(node.name()+node.lowLevel().start()+":"+node.id());
	    var original = null;
	    var pr = null;
	    if (node.lowLevel() instanceof proxy.LowLevelProxyNode) {
	        var proxyNode = node.lowLevel();
	        while (!proxyNode.primaryNode()) {
	            if (!original) {
	                original = localError(node, c, w, message, true, pr);
	            }
	            node = node.parent();
	            proxyNode = node.lowLevel();
	        }
	    }
	    if (node) {
	        pr = node.property();
	        if (node.lowLevel().unit() != node.root().lowLevel().unit()) {
	            original = localError(node, c, w, message, true, pr);
	            var v = node.lowLevel().unit();
	            if (v) {
	            }
	            while (node.lowLevel().unit() != node.root().lowLevel().unit()) {
	                pr = node.property();
	                node = node.parent();
	            }
	        }
	    }
	    if (original) {
	        if (node.property() && node.property().nameId() == universes.Universe10.FragmentDeclaration.properties.uses.name && node.parent() != null) {
	            pr = node.property(); //FIXME there should be other cases
	            node = node.parent();
	        }
	    }
	    var error = localError(node, c, w, message, false, pr);
	    if (original) {
	        error.extras.push(original);
	    }
	    //console.log(error.start+":"+error.end)
	    return error;
	}
	exports.createIssue = createIssue;
	function createLLIssue(issueCode, message, node, rootCalculationAnchor, isWarning) {
	    if (isWarning === void 0) { isWarning = false; }
	    var original = null;
	    if (node) {
	        if (rootCalculationAnchor.lowLevel().unit() != rootCalculationAnchor.root().lowLevel().unit()) {
	            original = localLowLevelError(node, rootCalculationAnchor, issueCode, isWarning, message, true);
	            var v = rootCalculationAnchor.lowLevel().unit();
	            if (v) {
	                message = message + " " + v.path();
	            }
	            while (rootCalculationAnchor.lowLevel().unit() != rootCalculationAnchor.root().lowLevel().unit()) {
	                rootCalculationAnchor = rootCalculationAnchor.parent();
	            }
	        }
	    }
	    if (original) {
	        if (rootCalculationAnchor.property() && rootCalculationAnchor.property().nameId()
	            == universes.Universe10.FragmentDeclaration.properties.uses.name && rootCalculationAnchor.parent() != null) {
	            rootCalculationAnchor = rootCalculationAnchor.parent();
	        }
	    }
	    var error = localLowLevelError(node, rootCalculationAnchor, issueCode, isWarning, message, false);
	    if (original) {
	        error.extras.push(original);
	    }
	    //console.log(error.start+":"+error.end)
	    return error;
	}
	exports.createLLIssue = createLLIssue;
	function validateResponseString(v) {
	    if (v.length != 3) {
	        return new Error("Status code should be 3 digits number with optional 'x' as wildcards");
	    }
	    for (var i = 0; i < v.length; i++) {
	        var c = v[i];
	        if (!_.find(['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'x', 'X'], function (x) { return x == c; })) {
	            return new Error("Status code should be 3 digits number with optional 'x' as wildcards");
	        }
	    }
	    return null;
	}
	exports.validateResponseString = validateResponseString;
	//# sourceMappingURL=linter.js.map

/***/ },
/* 28 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../../typings/main.d.ts" />
	"use strict";
	var defs = __webpack_require__(35);
	var yaml = __webpack_require__(72);
	var _ = __webpack_require__(61);
	var def = defs;
	var hlimpl = __webpack_require__(16);
	var proxy = __webpack_require__(37);
	var search = __webpack_require__(29);
	var universes = __webpack_require__(22);
	var universeHelpers = __webpack_require__(14);
	var services = defs;
	var KeyMatcher = (function () {
	    function KeyMatcher(_props) {
	        this._props = _props;
	        this.parentValue = _.find(_props, function (x) { return x.isFromParentValue(); });
	        this.parentKey = _.find(_props, function (x) { return x.isFromParentKey(); });
	        this.canBeValue = _.find(_props, function (x) { return x.canBeValue(); });
	    }
	    KeyMatcher.prototype.add = function (p) {
	        this._props.push(p);
	    };
	    KeyMatcher.prototype.match = function (key) {
	        var _this = this;
	        var _res = null;
	        var lastPref = "";
	        this._props.forEach(function (p) {
	            if (p.getAdapter(services.RAMLPropertyService).isSystem()) {
	                return;
	            }
	            if (p != _this.parentValue && p != _this.parentKey && p.matchKey(key)) {
	                if (p.keyPrefix() != null) {
	                    if (p.keyPrefix().length >= lastPref.length) {
	                        lastPref = p.keyPrefix();
	                        _res = p;
	                    }
	                }
	                else {
	                    _res = p;
	                    lastPref = p.nameId();
	                }
	            }
	        });
	        return _res;
	    };
	    return KeyMatcher;
	}());
	function getAllOptions(c, deep) {
	    if (deep === void 0) { deep = 0; }
	    if (deep > 20) {
	        return [];
	    }
	    try {
	        var result = [];
	        var tp = c.leftType();
	        if (tp) {
	            result.push(tp);
	        }
	        var r = c.rightType();
	        if (r) {
	            if (r.hasUnionInHierarchy()) {
	                var options = getAllOptions(r.unionInHierarchy(), deep + 1);
	                result = result.concat(options);
	            }
	            else {
	                result.push(r);
	            }
	        }
	        return result;
	    }
	    finally {
	    }
	}
	var ad = 0;
	var BasicNodeBuilder = (function () {
	    function BasicNodeBuilder() {
	        this.shouldDescriminate = false;
	    }
	    BasicNodeBuilder.prototype.process = function (node, childrenToAdopt) {
	        var _this = this;
	        var nn = node.lowLevel();
	        var cha = nn._node ? nn._node : nn;
	        try {
	            if (cha['currentChildren']) {
	                return cha['currentChildren'];
	            }
	            if (!node.definition()) {
	                return;
	            }
	            if (node.parent() == null && (!this.shouldDescriminate)) {
	                this.shouldDescriminate = true;
	                try {
	                    var children = this.process(node, childrenToAdopt);
	                    var ts = node;
	                    ts._children = children;
	                    var t = doDescrimination(node);
	                    if (t) {
	                        ts.patchType(t);
	                    }
	                    var children = this.process(node, childrenToAdopt);
	                    ts._children = children;
	                }
	                finally {
	                    this.shouldDescriminate = false;
	                }
	            }
	            if (node.definition().hasUnionInHierarchy()) {
	                if (true &&
	                    (node.parent() && node.property().nameId() == universes.Universe10.LibraryBase.properties.annotations.name)) {
	                    var optins = getAllOptions(node.definition().unionInHierarchy());
	                    var actualResult = null;
	                    var bestResult = null;
	                    var bestType = null;
	                    var bestCount = 1000;
	                    var llnode = node;
	                    optins.forEach(function (x) {
	                        if (!actualResult) {
	                            //TODO ADD UNION + Descriminator
	                            if (!(x).hasUnionInHierarchy()) {
	                                var tp = llnode.patchType(x);
	                                if (ad == 0) {
	                                    ad++;
	                                    try {
	                                        var result = _this.process(node, childrenToAdopt);
	                                        var uc = 0;
	                                        for (var i = 0; i < result.length; i++) {
	                                            if (result[i].isUnknown()) {
	                                                uc++;
	                                            }
	                                        }
	                                        if (uc == 0) {
	                                            actualResult = result;
	                                        }
	                                        if (bestCount > uc) {
	                                            bestCount = uc;
	                                            bestResult = result;
	                                            bestType = x;
	                                        }
	                                    }
	                                    finally {
	                                        ad--;
	                                    }
	                                }
	                            }
	                        }
	                    });
	                    if (actualResult) {
	                        llnode.patchType(bestType);
	                        return actualResult;
	                    }
	                    if (bestResult) {
	                        llnode.patchType(bestType);
	                    }
	                }
	            }
	            var km = new KeyMatcher(node.definition().allProperties());
	            if (node.parent() == null || node.lowLevel().includePath()) {
	                var u = node.definition().universe();
	                if (u.version() == "RAML10") {
	                    if (!node.definition().property("uses")) {
	                        u.type("FragmentDeclaration").allProperties().forEach(function (x) { return km.add(x); });
	                    }
	                }
	            }
	            var aNode = node;
	            var allowsQuestion = aNode._allowQuestion || node.definition().getAdapter(services.RAMLService).getAllowQuestion();
	            var res = [];
	            //cha['currentChildren']=res;
	            if (km.parentKey) {
	                if (node.lowLevel().key()) {
	                    var keyAttr = new hlimpl.ASTPropImpl(node.lowLevel(), node, km.parentKey.range(), km.parentKey, true);
	                    res.push(keyAttr);
	                    var isDefaultMediaType = node.property()
	                        && universeHelpers.isBodyProperty(node.property())
	                        && node.lowLevel().key() == node.property().nameId();
	                    if (isDefaultMediaType) {
	                        var isInsideTraitOrResourceType = isInTtraitOrResourceType(aNode);
	                        if (!isInsideTraitOrResourceType && aNode._computedKey) {
	                            keyAttr.overrideValue(aNode._computedKey);
	                        }
	                    }
	                }
	                if (node.lowLevel().valueKind() === yaml.Kind.SEQ) {
	                    var error = new hlimpl.BasicASTNode(node.lowLevel(), aNode);
	                    error.errorMessage = "" + node.definition().nameId() + " definition should be a map";
	                    res.push(error);
	                    return res;
	                }
	            }
	            if (node.lowLevel().value()) {
	                if (km.parentValue) {
	                    res.push(new hlimpl.ASTPropImpl(node.lowLevel(), node, km.parentValue.range(), km.parentValue));
	                }
	                else if (km.canBeValue) {
	                    var s = node.lowLevel().value();
	                    if (typeof s == 'string' && s.trim().length > 0) {
	                        //if (km.canBeValue.nameId()==universes.Universe10.Resource.properties.signature.name){
	                        //      if (s.trim().charAt(0)=='('){
	                        //          //TODO BETTER DECITION current one prevents completion from working correctly
	                        //          //in few other cases
	                        //          res.push(new hlimpl.ASTPropImpl(node.lowLevel(), node, km.canBeValue.range(), km.canBeValue));
	                        //      }
	                        //}
	                        //else {
	                        res.push(new hlimpl.ASTPropImpl(node.lowLevel(), node, km.canBeValue.range(), km.canBeValue));
	                    }
	                }
	            }
	            else {
	            }
	            aNode._children = res;
	            if (!aNode.definition().getAdapter(services.RAMLService).isUserDefined()) {
	                if (aNode.definition().key() == universes.Universe08.Api || aNode.definition().key() == universes.Universe10.Api) {
	                    var uses = childrenToAdopt.filter(function (x) { return x.key() == "uses"; });
	                    res = this.processChildren(uses, aNode, res, allowsQuestion, km);
	                    var types = childrenToAdopt.filter(function (x) { return x.key() == "types"; });
	                    res = this.processChildren(types, aNode, res, allowsQuestion, km);
	                    var other = childrenToAdopt.filter(function (x) { return (x.key() != "types" && x.key() != "uses"); });
	                    res = this.processChildren(other, aNode, res, allowsQuestion, km);
	                }
	                else {
	                    res = this.processChildren(childrenToAdopt, aNode, res, allowsQuestion, km);
	                }
	            }
	            else {
	                res = this.processChildren(childrenToAdopt, aNode, res, allowsQuestion, km);
	            }
	            aNode._children = res;
	            return res;
	        }
	        finally {
	        }
	    };
	    BasicNodeBuilder.prototype.isTypeDeclarationShortcut = function (node, property) {
	        var isTypeDeclarationType = universeHelpers.isTypeProperty(property);
	        var isTypeDeclaration = node.definition() && universeHelpers.isTypeDeclarationTypeOrDescendant(node.definition());
	        if (isTypeDeclaration && isTypeDeclarationType && node.lowLevel() && node.lowLevel().valueKind() === yaml.Kind.SEQ) {
	            return true;
	        }
	        return false;
	    };
	    BasicNodeBuilder.prototype.processChildren = function (childrenToAdopt, aNode, res, allowsQuestion, km) {
	        var _this = this;
	        var typeDeclarationName = universes.Universe10.TypeDeclaration.name;
	        var typePropertyName = universes.Universe10.TypeDeclaration.properties.type.name;
	        if (aNode.definition() && aNode.definition().isAssignableFrom(typeDeclarationName) && aNode.lowLevel() && (km.canBeValue && km.canBeValue.nameId() === typePropertyName) && aNode.lowLevel()._node && aNode.lowLevel()._node.value && aNode.lowLevel()._node.value.kind === yaml.Kind.SEQ) {
	            childrenToAdopt.forEach(function (child) {
	                var property = new hlimpl.ASTPropImpl(child, aNode, km.canBeValue.range(), km.canBeValue);
	                res.push(property);
	            });
	            return res;
	        }
	        childrenToAdopt.forEach(function (x) {
	            if (km.canBeValue && _this.isTypeDeclarationShortcut(aNode, km.canBeValue)) {
	                res.push(new hlimpl.ASTPropImpl(x, aNode, km.canBeValue.range(), km.canBeValue));
	                return;
	            }
	            var key = x.key();
	            //if (x.optional()&&!allowsQuestion) {
	            //    return;
	            //}
	            var p = key != null ? km.match(key) : null;
	            if (p != null) {
	                var range = p.range();
	                if (p.isAnnotation() && key != "annotations") {
	                    var pi = new hlimpl.ASTPropImpl(x, aNode, range, p);
	                    res.push(pi);
	                    return;
	                }
	                var um = false;
	                var multyValue = p.isMultiValue();
	                if (range.isArray()) {
	                    multyValue = true;
	                    range = range.array().componentType();
	                    um = true;
	                }
	                else if (range.hasArrayInHierarchy()) {
	                    multyValue = true;
	                    um = true;
	                }
	                //TODO DESCRIMINATORS
	                if (range.hasValueTypeInHierarchy()) {
	                    var ch = x.children();
	                    var seq = (x.valueKind() == yaml.Kind.SEQ);
	                    if ((seq && ch.length > 0 || ch.length > 1) && multyValue) {
	                        var values = [];
	                        ch.forEach(function (y) {
	                            var pi = new hlimpl.ASTPropImpl(y, aNode, range, p);
	                            res.push(pi);
	                            values.push(y.value());
	                        });
	                        if (p.isInherited()) {
	                            aNode.setComputed(p.nameId(), values);
	                        }
	                    }
	                    else {
	                        if (p.isInherited()) {
	                            aNode.setComputed(p.nameId(), x.value());
	                        }
	                        var attrNode = new hlimpl.ASTPropImpl(x, aNode, range, p);
	                        if ((seq || x.valueKind() == yaml.Kind.MAP)) {
	                            var rng = p.range().nameId();
	                            if (!p.getAdapter(services.RAMLPropertyService).isExampleProperty()) {
	                                if (rng == 'StringType') {
	                                    rng = "string";
	                                }
	                                if (rng == 'NumberType') {
	                                    rng = "number";
	                                }
	                                if (rng == 'BooleanType') {
	                                    rng = "boolean";
	                                }
	                                if (rng == "string" || rng == "number" || rng == "boolean") {
	                                    if (!x.isAnnotatedScalar()) {
	                                        attrNode.errorMessage = "property '" + p.groupName() + "' must be a " + rng;
	                                        if (x.children().length == 0 && p.groupName() == "enum") {
	                                            attrNode.errorMessage = "enum is empty";
	                                            if (x.valueKind() == yaml.Kind.MAP) {
	                                                attrNode.errorMessage = "the value of enum must be an array";
	                                            }
	                                        }
	                                    }
	                                }
	                            }
	                        }
	                        res.push(attrNode);
	                    }
	                    //}
	                    return;
	                }
	                else {
	                    var rs = [];
	                    //now we need determine actual type
	                    aNode._children = res;
	                    if (x.value() != null && (typeof x.value() == 'string' || typeof x.value() == 'boolean' || typeof x.value() == 'number')) {
	                        if (("" + x.value()).trim().length > 0) {
	                            var c = p.range();
	                            if (!c.allProperties().some(function (x) {
	                                var srv = x;
	                                if (srv) {
	                                    return srv.canBeValue() && srv.isFromParentValue();
	                                }
	                                return false;
	                            })) {
	                                var bnode = new hlimpl.BasicASTNode(x, aNode);
	                                bnode.getLowLevelEnd = function () {
	                                    return -1;
	                                };
	                                bnode.getLowLevelStart = function () {
	                                    return -1;
	                                };
	                                bnode.knownProperty = p;
	                                res.push(bnode);
	                            }
	                        }
	                    }
	                    if (!p.isMerged()) {
	                        if (multyValue) {
	                            if (p.getAdapter(services.RAMLPropertyService).isEmbedMap()) {
	                                var chld = x.children();
	                                if (chld.length == 0) {
	                                    if (p.range().key() == universes.Universe08.ResourceType) {
	                                        var error = new hlimpl.BasicASTNode(x, aNode);
	                                        error.errorMessage = "property: '" + p.nameId() + "' must be a map";
	                                        res.push(error);
	                                    }
	                                    if (x.valueKind() == yaml.Kind.SCALAR) {
	                                        if (p.range().key() == universes.Universe08.AbstractSecurityScheme) {
	                                            var error = new hlimpl.BasicASTNode(x.parent(), aNode);
	                                            error.errorMessage = "property: '" + p.nameId() + "' must be a map";
	                                            res.push(error);
	                                        }
	                                    }
	                                }
	                                var exit = false;
	                                chld.forEach(function (y) {
	                                    if (exit) {
	                                        return;
	                                    }
	                                    //TODO TRACK GROUP KEY
	                                    var cld = y.children();
	                                    if (!y.key() && cld.length == 1) {
	                                        if (aNode.universe().version() == "RAML10" && !aNode.parent()) {
	                                            var bnode = new hlimpl.BasicASTNode(x, aNode);
	                                            res.push(bnode);
	                                            bnode.needMap = true;
	                                            bnode.knownProperty = p;
	                                            exit = true;
	                                        }
	                                        else {
	                                            var node = new hlimpl.ASTNodeImpl(cld[0], aNode, range, p);
	                                            node._allowQuestion = allowsQuestion;
	                                            rs.push(node);
	                                        }
	                                    }
	                                    else {
	                                        if (aNode.universe().version() == "RAML10") {
	                                            var node = new hlimpl.ASTNodeImpl(y, aNode, range, p);
	                                            node._allowQuestion = allowsQuestion;
	                                            rs.push(node);
	                                        }
	                                        else {
	                                            var bnode = new hlimpl.BasicASTNode(y, aNode);
	                                            res.push(bnode);
	                                            if (y.key()) {
	                                                bnode.needSequence = true;
	                                            }
	                                        }
	                                    }
	                                });
	                            }
	                            else {
	                                var filter = {};
	                                var inherited = [];
	                                if (range instanceof defs.NodeClass) {
	                                    var nc = range;
	                                    if (nc.getAdapter(services.RAMLService).getCanInherit().length > 0) {
	                                        nc.getAdapter(services.RAMLService).getCanInherit().forEach(function (v) {
	                                            var originalValue = aNode.computedValue(v);
	                                            var actualValue = Array.isArray(originalValue) ? originalValue : [originalValue];
	                                            for (var pos = 0; pos < actualValue.length; pos++) {
	                                                var vl = actualValue[pos];
	                                                if (vl && p.nameId() == universes.Universe10.Response.properties.body.name) {
	                                                    if (!_.find(x.children(), function (x) { return x.key() == vl; })) {
	                                                        //we can create inherited node;
	                                                        var pc = aNode.parent().definition().key();
	                                                        var node = new hlimpl.ASTNodeImpl(x, aNode, range, p);
	                                                        node._computedKey = vl;
	                                                        if (pc == universes.Universe10.MethodBase || pc == universes.Universe08.MethodBase) {
	                                                            node.setComputed("form", "true"); //FIXME
	                                                        }
	                                                        var isInsideTraitOrResourceType = isInTtraitOrResourceType(aNode);
	                                                        var t = descriminate(p, aNode, node);
	                                                        if (t) {
	                                                            if (!isInsideTraitOrResourceType) {
	                                                                t.setName(vl);
	                                                            }
	                                                            node.patchType(t);
	                                                        }
	                                                        var ch = node.children();
	                                                        //this are false unknowns actual unknowns will be reported by parent node
	                                                        node._children = ch.filter(function (x) { return !x.isUnknown(); });
	                                                        node._allowQuestion = allowsQuestion;
	                                                        inherited.push(node);
	                                                        node.children().forEach(function (x) {
	                                                            if (x.property().getAdapter(services.RAMLPropertyService).isKey()) {
	                                                                var atr = x;
	                                                                atr._computed = true;
	                                                                return;
	                                                            }
	                                                            if (x.isElement()) {
	                                                                if (!x.property().getAdapter(services.RAMLPropertyService).isMerged()) {
	                                                                    filter[x.property().nameId()] = true;
	                                                                }
	                                                            }
	                                                            if (x.property().isAnnotation()) {
	                                                                var atr = x;
	                                                                var vl = atr.value();
	                                                                var strVal = "";
	                                                                if (vl instanceof hlimpl.StructuredValue) {
	                                                                    strVal = vl.valueName();
	                                                                }
	                                                                else {
	                                                                    strVal = "" + vl;
	                                                                }
	                                                                filter["(" + strVal + ")"] = true;
	                                                            }
	                                                            else {
	                                                                filter[x.name()] = true;
	                                                            }
	                                                        });
	                                                        var ap = node.definition().allProperties();
	                                                        ap.forEach(function (p) {
	                                                            if (p.getAdapter(services.RAMLPropertyService).isKey()) {
	                                                                return;
	                                                            }
	                                                            if (p.getAdapter(services.RAMLPropertyService).isSystem()) {
	                                                                return;
	                                                            }
	                                                            if (node.lowLevel().children().some(function (x) { return x.key() == p.nameId(); })) {
	                                                                filter[p.nameId()] = true;
	                                                            }
	                                                        });
	                                                        node._computed = true;
	                                                    }
	                                                }
	                                            }
	                                        });
	                                    }
	                                }
	                                var parsed = [];
	                                if (x.children().length == 0) {
	                                    if (x.valueKind() == yaml.Kind.SEQ) {
	                                        if (p.range().key() == universes.Universe08.Parameter) {
	                                            var error = new hlimpl.BasicASTNode(x, aNode);
	                                            error.errorMessage = "property: '" + p.nameId() + "' must be a map";
	                                            res.push(error);
	                                        }
	                                    }
	                                }
	                                if (p.nameId() === "documentation" && x.valueKind() !== yaml.Kind.SEQ) {
	                                    if (!aNode.definition().getAdapter(services.RAMLService).isUserDefined()) {
	                                        var error = new hlimpl.BasicASTNode(x, aNode);
	                                        error.errorMessage = "property: '" + p.nameId() + "' should be a sequence";
	                                        res.push(error);
	                                    }
	                                }
	                                else {
	                                    x.children().forEach(function (y) {
	                                        if (filter[y.key()]) {
	                                            return;
	                                        }
	                                        if (y.valueKind() == yaml.Kind.SEQ) {
	                                            y.children().forEach(function (z) {
	                                                var node = new hlimpl.ASTNodeImpl(z, aNode, range, p);
	                                                node._allowQuestion = allowsQuestion;
	                                                node.setNamePatch(y.key());
	                                                parsed.push(node);
	                                            });
	                                            if (y.children().length == 0) {
	                                                var error = new hlimpl.BasicASTNode(y, aNode);
	                                                if (p.range().key() == universes.Universe08.Parameter) {
	                                                    error.errorMessage = "named parameter needs at least one type";
	                                                }
	                                                else {
	                                                    error.errorMessage = "node should have at least one member value";
	                                                }
	                                                res.push(error);
	                                            }
	                                        }
	                                        else {
	                                            var node = new hlimpl.ASTNodeImpl(y, aNode, range, p);
	                                            var dc = p.domain().key();
	                                            if (p.nameId() == "body" && (dc == universes.Universe08.MethodBase || dc == universes.Universe10.MethodBase)) {
	                                                node.setComputed("form", "true"); //FIXME
	                                            }
	                                            node._allowQuestion = allowsQuestion;
	                                            parsed.push(node);
	                                        }
	                                    });
	                                }
	                                if (parsed.length > 0) {
	                                    parsed.forEach(function (x) { return rs.push(x); });
	                                }
	                                else {
	                                    inherited.forEach(function (x) { return rs.push(x); });
	                                }
	                            }
	                        }
	                        else {
	                            //var y=x.children()[0];
	                            rs.push(new hlimpl.ASTNodeImpl(x, aNode, range, p));
	                        }
	                    }
	                    else {
	                        var node = new hlimpl.ASTNodeImpl(x, aNode, range, p);
	                        node._allowQuestion = allowsQuestion;
	                        rs.push(node);
	                    }
	                    aNode._children = aNode._children.concat(rs);
	                    res = res.concat(rs);
	                    rs.forEach(function (x) {
	                        var rt = descriminate(p, aNode, x);
	                        if (rt && rt != x.definition()) {
	                            x.patchType(rt);
	                        }
	                        x._associatedDef = null;
	                        p.childRestrictions().forEach(function (y) {
	                            x.setComputed(y.name, y.value);
	                        });
	                        var def = x.definition();
	                    });
	                }
	            }
	            else {
	                if (!(x instanceof proxy.LowLevelCompositeNode)
	                    || x.primaryNode() != null) {
	                    res.push(new hlimpl.BasicASTNode(x, aNode));
	                }
	            }
	        });
	        return res;
	    };
	    return BasicNodeBuilder;
	}());
	exports.BasicNodeBuilder = BasicNodeBuilder;
	function getType(node, expression) {
	    if (!expression) {
	        return node.definition().universe().type(universes.Universe10.StringTypeDeclaration.name);
	    }
	    var pt = node.parsedType();
	    if (pt.isString()) {
	        return (node.definition().universe().type(universes.Universe10.StringTypeDeclaration.name));
	    }
	    else if (pt.isNumber()) {
	        if (pt.isInteger()) {
	            return (node.definition().universe().type(universes.Universe10.IntegerTypeDeclaration.name));
	        }
	        return (node.definition().universe().type(universes.Universe10.NumberTypeDeclaration.name));
	    }
	    else if (pt.isBoolean()) {
	        return (node.definition().universe().type(universes.Universe10.BooleanTypeDeclaration.name));
	    }
	    else if (pt.isObject()) {
	        return (node.definition().universe().type(universes.Universe10.ObjectTypeDeclaration.name));
	    }
	    else if (pt.isArray()) {
	        return (node.definition().universe().type(universes.Universe10.ArrayTypeDeclaration.name));
	    }
	    else if (pt.isFile()) {
	        return (node.definition().universe().type(universes.Universe10.FileTypeDeclaration.name));
	    }
	    else if (pt.isDateTime()) {
	        return (node.definition().universe().type(universes.Universe10.DateTimeTypeDeclaration.name));
	    }
	    else if (pt.isDateTimeOnly()) {
	        return (node.definition().universe().type(universes.Universe10.DateTimeOnlyTypeDeclaration.name));
	    }
	    else if (pt.isDateOnly()) {
	        return (node.definition().universe().type(universes.Universe10.DateOnlyTypeDeclaration.name));
	    }
	    else if (pt.isTimeOnly()) {
	        return (node.definition().universe().type(universes.Universe10.DateOnlyTypeDeclaration.name));
	    }
	    return (node.definition().universe().type(universes.Universe10.TypeDeclaration.name));
	}
	function desc1(p, parent, x) {
	    var tp = x.attr("type");
	    var value = "";
	    if (tp) {
	        var mn = {};
	        var c = new def.NodeClass(x.name(), x.definition().universe(), "");
	        c.getAdapter(services.RAMLService).setDeclaringNode(x);
	        c._superTypes.push(x.definition().universe().type(universes.Universe10.TypeDeclaration.name));
	        mn[tp.value()] = c;
	        var newType = getType(x, tp.value());
	        if (newType) {
	            if (newType.superTypes().length == 0) {
	                newType._superTypes.push(x.definition().universe().type(universes.Universe10.TypeDeclaration.name));
	            }
	        }
	        return newType;
	    }
	    else {
	        var propertiesName = universes.Universe10.ObjectTypeDeclaration.properties.properties.name;
	        if (p) {
	            if (p.nameId() == "body" || _.find(x.lowLevel().children(), function (x) { return x.key() === propertiesName; })) {
	                return x.definition().universe().type(universes.Universe10.ObjectTypeDeclaration.name);
	            }
	        }
	        else {
	            if (!parent && x.lowLevel() && _.find(x.lowLevel().children(), function (x) { return x.key() === propertiesName; })) {
	                return x.definition().universe().type(universes.Universe10.ObjectTypeDeclaration.name);
	            }
	        }
	        return x.definition().universe().type(universes.Universe10.StringTypeDeclaration.name);
	    }
	}
	function doDescrimination(node) {
	    try {
	        var nodeDefenitionName = node.definition().nameId();
	        var isApi = nodeDefenitionName === universes.Universe10.Api.name || nodeDefenitionName === universes.Universe08.Api.name;
	        if (!isApi && !node.property() && !node.parent() && node.definition().nameId() === hlimpl.getFragmentDefenitionName(node)) {
	            if (node.property() && node.property().nameId() === universes.Universe10.LibraryBase.properties.annotationTypes.name) {
	                return descriminate(null, null, node);
	            }
	            var result = null;
	            var subTypes = node.definition().allSubTypes();
	            subTypes.forEach(function (subType) {
	                if (!result && match(subType, node, null)) {
	                    result = subType;
	                }
	            });
	            return result;
	        }
	    }
	    catch (exception) {
	    }
	    return descriminate(node.property(), node.parent(), node);
	}
	exports.doDescrimination = doDescrimination;
	function descriminate(p, parent, x) {
	    var n = x.lowLevel();
	    if (p) {
	        if (p.nameId() == universes.Universe10.FragmentDeclaration.properties.uses.name &&
	            p.range().nameId() == universes.Universe10.Library.name) {
	        }
	    }
	    var range = p ? p.range().nameId() : x.definition().nameId();
	    if (n._node && n._node['descriminate']) {
	        return null;
	    }
	    if (n._node) {
	        n._node['descriminate'] = 1;
	    }
	    try {
	        if (range == universes.Universe10.TypeDeclaration.name) {
	            var res = desc1(p, parent, x);
	            if (p || (!p && !parent && x.lowLevel())) {
	                if (p && res != null && ((p.nameId() == universes.Universe10.MethodBase.properties.body.name
	                    || p.nameId() == universes.Universe10.Response.properties.headers.name) ||
	                    p.nameId() == universes.Universe10.MethodBase.properties.queryParameters.name)) {
	                    var ares = new defs.UserDefinedClass(x.lowLevel().key(), res.universe(), x, x.lowLevel().unit() ? x.lowLevel().unit().path() : "", "");
	                    ares._superTypes.push(res);
	                    return ares;
	                }
	                if (res != null && universeHelpers.isAnnotationTypeType(x.definition())) {
	                    var annotationType = descriminateAnnotationType(res);
	                    var ares = new defs.UserDefinedClass(x.lowLevel().key(), res.universe(), x, x.lowLevel().unit() ? x.lowLevel().unit().path() : "", "");
	                    ares._superTypes.push(annotationType);
	                    ares._superTypes.push(res);
	                    return ares;
	                }
	            }
	            if (res) {
	                return res;
	            }
	        }
	        //generic case;
	        var rt = null;
	        if (p && parent) {
	            var types = search.findAllSubTypes(p, parent);
	            if (types.length > 0) {
	                types.forEach(function (y) {
	                    if (!rt) {
	                        if (match(y, x, rt)) {
	                            rt = y;
	                        }
	                    }
	                });
	            }
	            return rt;
	        }
	    }
	    finally {
	        if (n._node) {
	            delete n._node['descriminate'];
	        }
	    }
	}
	;
	function descriminateAnnotationType(type) {
	    var arr = [type].concat(type.allSuperTypes());
	    var candidate;
	    for (var i = 0; i < arr.length; i++) {
	        var t = arr[i];
	        if (t.isUserDefined()) {
	            continue;
	        }
	        if (t.isUnion()) {
	            var ut = t;
	            var lt = ut.leftType();
	            var rt = ut.leftType();
	            var lat = descriminateAnnotationType(lt);
	            var rat = descriminateAnnotationType(lt);
	            if (lat.isAssignableFrom(rat.nameId()) && (candidate == null || lat.isAssignableFrom(candidate.nameId()))) {
	                candidate = lat;
	                continue;
	            }
	            if (rat.isAssignableFrom(lat.nameId()) && (candidate == null || rat.isAssignableFrom(candidate.nameId()))) {
	                candidate = rat;
	                continue;
	            }
	            candidate = type.universe().type(universes.Universe10.UnionTypeDeclaration.name);
	            break;
	        }
	        if (t.isArray()) {
	            candidate = type.universe().type(universes.Universe10.ArrayTypeDeclaration.name);
	            break;
	        }
	        var subTypes = t.subTypes();
	        for (var j = 0; j < subTypes.length; j++) {
	            var st = subTypes[j];
	            if (st.isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
	                if (candidate == null) {
	                    candidate = st;
	                }
	                else if (st.isAssignableFrom(candidate.nameId())) {
	                    candidate = st;
	                }
	            }
	        }
	    }
	    return candidate != null ? candidate : type.universe().type(universes.Universe10.TypeDeclaration.name);
	}
	var isInTtraitOrResourceType = function (aNode) {
	    var isInsideTraitOrResourceType = false;
	    var parent_ = aNode;
	    while (parent_) {
	        var pDef = parent_.definition();
	        if (universeHelpers.isTraitType(pDef)
	            || universeHelpers.isResourceTypeType(pDef)) {
	            isInsideTraitOrResourceType = true;
	            break;
	        }
	        parent_ = parent_.parent();
	    }
	    return isInsideTraitOrResourceType;
	};
	function match(t, r, alreadyFound) {
	    //this.vReqInitied=true;
	    if (r.isAttr() || r.isUnknown()) {
	        return false;
	    }
	    var el = r;
	    var hasSuperType = _.find(t.superTypes(), function (x) {
	        var dp = _.find(x.allProperties(), function (x) { return (x).isDescriminator(); });
	        if (dp) {
	            var a = el.attr(dp.nameId());
	            if (a) {
	                if (a.value() == t.nameId()) {
	                    return true;
	                }
	            }
	        }
	        return false;
	    });
	    if (hasSuperType) {
	        return true;
	    }
	    if (t.valueRequirements().length == 0) {
	        return false;
	    }
	    var matches = true;
	    //descriminating constraint
	    t.valueRequirements().forEach(function (x) {
	        var a = el.attr(x.name);
	        if (a) {
	            if (a.value() == x.value) {
	            }
	            else {
	                if (t.getAdapter(services.RAMLService).getConsumesRefs()) {
	                    var vl = a.value();
	                    var allSubs = [];
	                    t.superTypes().forEach(function (x) { return x.allSubTypes().forEach(function (y) {
	                        allSubs.push(y);
	                    }); });
	                    var allSubNames = [];
	                    _.unique(allSubs).forEach(function (x) {
	                        allSubNames.push(x.nameId());
	                        x.valueRequirements().forEach(function (y) {
	                            allSubNames.push(y.value);
	                        });
	                        x.getAdapter(services.RAMLService).getAliases().forEach(function (y) { return allSubNames.push(y); });
	                    });
	                    if (_.find(allSubNames, function (x) { return x == vl; })) {
	                        matches = false;
	                    }
	                }
	                else {
	                    matches = false;
	                }
	            }
	        }
	        else {
	            var m = t.getAdapter(services.RAMLService).getDefining();
	            var ms = false;
	            m.forEach(function (x) {
	                el.lowLevel().children().forEach(function (y) {
	                    if (y.key() == x) {
	                        ms = true;
	                    }
	                });
	            });
	            if (ms) {
	                matches = true;
	                return;
	            }
	            if (!alreadyFound) {
	                var pr = t.property(x.name);
	                if (pr && pr.defaultValue() == x.value) {
	                }
	                else {
	                    matches = false;
	                }
	            }
	            else {
	                matches = false;
	            }
	        }
	    });
	    return matches;
	}
	//# sourceMappingURL=builder.js.map

/***/ },
/* 29 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../../typings/main.d.ts" />
	"use strict";
	var defs = __webpack_require__(35);
	var _ = __webpack_require__(61);
	var def = __webpack_require__(35);
	// import high=require("../highLevelAST");
	var hlimpl = __webpack_require__(16);
	var universes = __webpack_require__(22);
	var ramlServices = def;
	exports.declRoot = function (h) {
	    var declRoot = h;
	    while (true) {
	        if (declRoot.definition().key() == universes.Universe10.Library) {
	            break;
	        }
	        var np = declRoot.parent();
	        if (!np) {
	            break;
	        }
	        declRoot = np;
	    }
	    return declRoot;
	};
	function globalDeclarations(h) {
	    var result = [];
	    while (h.parent() != null) {
	        if (h.lowLevel().includePath()) {
	            result = result.concat(findDeclarations(h));
	        }
	        h = h.parent();
	    }
	    result = result.concat(findDeclarations(h));
	    return result;
	}
	exports.globalDeclarations = globalDeclarations;
	function mark(h, rs) {
	    var n = h.lowLevel();
	    n = n._node ? n._node : n;
	    if (n['mark']) {
	        return rs;
	    }
	    n['mark'] = rs;
	    return null;
	}
	function unmark(h) {
	    var n = h.lowLevel();
	    n = n._node ? n._node : n;
	    delete n['mark'];
	}
	function findDeclarations(h) {
	    var rs = [];
	    h.elements().forEach(function (x) {
	        if (x.definition().key() == universes.Universe10.UsesDeclaration) {
	            var mm = x.attr("value");
	            if (mm) {
	                var unit = h.root().lowLevel().unit().resolve(mm.value());
	                if (unit != null) {
	                    unit.highLevel().children().forEach(function (x) {
	                        if (x.isElement()) {
	                            rs.push(x.asElement());
	                        }
	                    });
	                }
	            }
	        }
	        else {
	            rs.push(x);
	        }
	    });
	    return rs;
	}
	exports.findDeclarations = findDeclarations;
	function getIndent2(offset, text) {
	    var spaces = "";
	    for (var i = offset - 1; i >= 0; i--) {
	        var c = text.charAt(i);
	        if (c == ' ' || c == '\t') {
	            if (spaces) {
	                spaces += c;
	            }
	            else {
	                spaces = c;
	            }
	        }
	        else if (c == '\r' || c == '\n') {
	            return spaces;
	        }
	    }
	}
	function deepFindNode(n, offset, end, goToOtherUnits, returnAttrs) {
	    if (goToOtherUnits === void 0) { goToOtherUnits = true; }
	    if (returnAttrs === void 0) { returnAttrs = true; }
	    if (n == null) {
	        return null;
	    }
	    if (n.lowLevel()) {
	        //var node:ASTNode=<ASTNode>n;
	        if (n.lowLevel().start() <= offset && n.lowLevel().end() >= end) {
	            if (n instanceof hlimpl.ASTNodeImpl) {
	                var hn = n;
	                var all = goToOtherUnits ? hn.children() : hn.directChildren();
	                for (var i = 0; i < all.length; i++) {
	                    if (!goToOtherUnits && all[i].lowLevel().unit() != n.lowLevel().unit()) {
	                        continue;
	                    }
	                    var node = deepFindNode(all[i], offset, end, goToOtherUnits);
	                    if (node) {
	                        if (!returnAttrs && node instanceof hlimpl.ASTPropImpl) {
	                            node = node.parent();
	                        }
	                        return node;
	                    }
	                }
	                return n;
	            }
	            else if (n instanceof hlimpl.ASTPropImpl) {
	                var attr = n;
	                if (!attr.property().isKey()) {
	                    var vl = attr.value();
	                    if (vl instanceof hlimpl.StructuredValue) {
	                        var st = vl;
	                        var hl = st.toHighLevel2();
	                        if (hl) {
	                            if (!goToOtherUnits && hl.lowLevel().unit() != n.lowLevel().unit()) {
	                                return null;
	                            }
	                        }
	                        var node = deepFindNode(hl, offset, end, goToOtherUnits);
	                        if (node) {
	                            if (!returnAttrs && node instanceof hlimpl.ASTPropImpl) {
	                                node = node.parent();
	                            }
	                            return node;
	                        }
	                    }
	                    if (returnAttrs)
	                        return attr;
	                    else
	                        return attr.parent();
	                }
	                return null;
	            }
	            return null;
	        }
	    }
	    return null;
	}
	exports.deepFindNode = deepFindNode;
	function getValueAt(text, offset) {
	    var sp = -1;
	    for (var i = offset - 1; i >= 0; i--) {
	        var c = text.charAt(i);
	        if (c == '\r' || c == '\n' || c == ' ' || c == '\t' || c == '"' || c == '\'' || c == ':') {
	            sp = i + 1;
	            break;
	        }
	    }
	    var ep = -1;
	    for (var i = offset; i < text.length; i++) {
	        var c = text.charAt(i);
	        if (c == '\r' || c == '\n' || c == ' ' || c == '\t' || c == '"' || c == '\'' || c == ':') {
	            ep = i;
	            break;
	        }
	    }
	    if (sp != -1 && ep != -1) {
	        return text.substring(sp, ep);
	    }
	    return "";
	}
	function extractName(cleaned, offset) {
	    var txt = "";
	    for (var i = offset; i >= 0; i--) {
	        var c = cleaned[i];
	        if (c == ' ' || c == '\r' || c == '\n' || c == '|' || c == '[' || c == ']' || c == ':' || c == '(' || c == ')') {
	            break;
	        }
	        txt = c + txt;
	    }
	    for (var i = offset + 1; i < cleaned.length; i++) {
	        var c = cleaned[i];
	        if (c == ' ' || c == '\r' || c == '\n' || c == '|' || c == '[' || c == ']' || c == ':' || c == '(' || c == ')') {
	            break;
	        }
	        txt = txt + c;
	    }
	    return txt;
	}
	exports.extractName = extractName;
	var searchInTheValue = function (offset, content, attr, hlnode, p) {
	    if (p === void 0) { p = attr.property(); }
	    var targets = referenceTargets(p, hlnode);
	    var txt = extractName(content, offset);
	    var t = _.find(targets, function (x) { return hlimpl.qName(x, hlnode) == txt; });
	    if (t) {
	        //TODO EXTRACT COMMON OPEN NODE FUNC
	        return t;
	    }
	    if (p instanceof defs.UserDefinedProp) {
	        var up = p;
	        return up._node;
	    }
	    return null;
	};
	function findUsages(unit, offset) {
	    var decl = findDeclaration(unit, offset);
	    if (decl) {
	        if (decl instanceof hlimpl.ASTNodeImpl) {
	            var hnode = decl;
	            return { node: hnode, results: hnode.findReferences() };
	        }
	        if (decl instanceof hlimpl.ASTPropImpl) {
	        }
	    }
	    var node = deepFindNode(hlimpl.fromUnit(unit), offset, offset, false);
	    if (node instanceof hlimpl.ASTNodeImpl) {
	        return { node: node, results: node.findReferences() };
	    }
	    if (node instanceof hlimpl.ASTPropImpl) {
	        var prop = node;
	        if (prop.property().canBeValue()) {
	            return { node: prop.parent(), results: prop.parent().findReferences() };
	        }
	    }
	    return { node: null, results: [] };
	}
	exports.findUsages = findUsages;
	function referenceTargets(p0, c) {
	    var p = p0;
	    if (p.getAdapter(ramlServices.RAMLPropertyService).isTypeExpr()) {
	        var definitionNodes = globalDeclarations(c).filter(function (node) {
	            var nc = node.definition().key();
	            if (nc === universes.Universe08.GlobalSchema) {
	                return true;
	            }
	            return node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name);
	        });
	        return definitionNodes;
	    }
	    if (p.getAdapter(ramlServices.RAMLPropertyService).isDescriminating()) {
	        var subTypes = nodesDeclaringType(p.range(), c);
	        return subTypes;
	    }
	    if (p.isReference()) {
	        var rt = p.referencesTo();
	        var subTypes = nodesDeclaringType(rt, c);
	        return subTypes;
	    }
	    if (p.range().hasValueTypeInHierarchy()) {
	        var vt = p.range().getAdapter(ramlServices.RAMLService);
	        if (vt.globallyDeclaredBy().length > 0) {
	            var definitionNodes = globalDeclarations(c).filter(function (z) { return _.find(vt.globallyDeclaredBy(), function (x) { return x == z.definition(); }) != null; });
	            return definitionNodes;
	        }
	    }
	    return [];
	}
	exports.referenceTargets = referenceTargets;
	function enumValues(p, c) {
	    if (c) {
	        var rs = [];
	        //TODO FIXME it is very very weird idea but I need to get it working right now
	        if (p.getAdapter(ramlServices.RAMLPropertyService).isTypeExpr()) {
	            var definitionNodes = globalDeclarations(c).filter(function (node) {
	                var nc = node.definition().key();
	                if (nc === universes.Universe08.GlobalSchema) {
	                    return true;
	                }
	                return (node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name));
	                //return true;
	            });
	            rs = definitionNodes.map(function (x) { return hlimpl.qName(x, c); });
	            var de = c.definition().universe().type(universes.Universe10.TypeDeclaration.name);
	            if (de) {
	                var subTypes = de.allSubTypes();
	                rs = rs.concat(subTypes.map(function (x) { return (x).getAdapter(ramlServices.RAMLService).descriminatorValue(); }));
	            }
	            return rs;
	        }
	        else {
	            var rangeKey = p.range().key();
	            if (rangeKey == universes.Universe08.SchemaString
	                || rangeKey == universes.Universe10.SchemaString) {
	                if (p.range().universe().version() == "RAML10") {
	                    if (p.range().hasValueTypeInHierarchy()) {
	                        var definitionNodes = globalDeclarations(c).filter(function (node) {
	                            return node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name);
	                        });
	                        rs = definitionNodes.map(function (x) { return hlimpl.qName(x, c); });
	                    }
	                }
	            }
	        }
	        if (p.getAdapter(ramlServices.RAMLPropertyService).isDescriminating()) {
	            var subTypes = subTypesWithLocals(p.domain(), c);
	            rs = rs.concat(subTypes.map(function (x) { return (x).getAdapter(ramlServices.RAMLService).descriminatorValue(); }));
	        }
	        else if (p.isReference()) {
	            rs = nodesDeclaringType(p.referencesTo(), c).map(function (x) { return hlimpl.qName(x, c); });
	        }
	        else if (p.range().hasValueTypeInHierarchy()) {
	            var vt = p.range().getAdapter(ramlServices.RAMLService);
	            if (vt.globallyDeclaredBy().length > 0) {
	                var definitionNodes = globalDeclarations(c).filter(function (z) {
	                    return _.find(vt.globallyDeclaredBy(), function (x) { return x == z.definition(); }) != null;
	                });
	                rs = rs.concat(definitionNodes.map(function (x) { return hlimpl.qName(x, c); }));
	            }
	        }
	        if (p.isAllowNull()) {
	            rs.push("null");
	        }
	        if (p.enumOptions()) {
	            rs = rs.concat(p.enumOptions());
	        }
	        return rs;
	    }
	    if (p.enumOptions() && typeof p.enumOptions() == 'string') {
	        return [p.enumOptions() + ""];
	    }
	    return p.enumOptions();
	}
	exports.enumValues = enumValues;
	function getLibraryName(node) {
	    if (node.isElement() && node.asElement().definition().key() != universes.Universe10.Library) {
	        return null;
	    }
	    return node.asElement().attrValue("name");
	}
	function findDeclarationByNode(node, nodePart) {
	    var unit = node.lowLevel().unit();
	    if (!unit) {
	        return null;
	    }
	    var start = node.lowLevel().start();
	    var end = node.lowLevel().end();
	    if (nodePart && nodePart == LocationKind.KEY_COMPLETION) {
	        start = node.lowLevel().keyStart();
	        end = node.lowLevel().keyEnd();
	    }
	    else if (nodePart && nodePart == LocationKind.VALUE_COMPLETION) {
	        start = node.lowLevel().valueStart();
	        end = node.lowLevel().valueEnd();
	    }
	    if (start == -1 || end == -1) {
	        return null;
	    }
	    var offset = Math.floor((start + end) / 2);
	    return findDeclaration(unit, offset, nodePart);
	}
	exports.findDeclarationByNode = findDeclarationByNode;
	function findDeclaration(unit, offset, nodePart) {
	    var node = deepFindNode(hlimpl.fromUnit(unit), offset, offset, false);
	    var result = null;
	    //TODO This should be changed after next refactoring
	    if (node.isElement()) {
	        if (node.asElement().definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
	            node.asElement().directChildren().forEach(function (x) {
	                if (x.isUnknown()) {
	                    //facets are not in hierarchy now
	                    if (x.getLowLevelStart() < offset && x.getLowLevelEnd() > offset) {
	                        var tp = node.asElement().localType();
	                        tp.allFacets().forEach(function (f) {
	                            if (f.nameId() == x.lowLevel().key()) {
	                                if (f instanceof def.UserDefinedProp) {
	                                    var up = f._node;
	                                    result = up;
	                                }
	                            }
	                        });
	                    }
	                }
	            });
	        }
	    }
	    if (!node.property()) {
	        return node;
	    }
	    if (node.property().nameId() == "example") {
	        var nm = node.parent().localType();
	        node.lowLevel().children().forEach(function (y) {
	            if (y.key() == "example") {
	                y.children().forEach(function (x) {
	                    if (x.start() < offset && x.end() > offset) {
	                        var tp = node.parent().asElement().localType();
	                        tp.allProperties().forEach(function (f) {
	                            if (f.nameId() == x.key()) {
	                                if (f instanceof def.UserDefinedProp) {
	                                    var up = f._node;
	                                    result = up;
	                                }
	                            }
	                        });
	                    }
	                });
	            }
	        });
	    }
	    if (result) {
	        return result;
	    }
	    var kind = nodePart ? nodePart : determineCompletionKind(unit.contents(), offset);
	    if (kind == LocationKind.VALUE_COMPLETION) {
	        var hlnode = node;
	        if (node instanceof hlimpl.ASTPropImpl) {
	            var attr = node;
	            if (attr) {
	                if (attr.value()) {
	                    if (attr.value() instanceof hlimpl.StructuredValue) {
	                        var sval = attr.value();
	                        var hlvalue = sval.toHighLevel();
	                        if (hlvalue) {
	                            var newAttr = _.find(hlvalue.attrs(), function (x) { return x.lowLevel().start() < offset && x.lowLevel().end() >= offset; });
	                            if (newAttr) {
	                                return searchInTheValue(offset, unit.contents(), newAttr, hlvalue, attr.property());
	                            }
	                        }
	                    }
	                    else {
	                        return searchInTheValue(offset, unit.contents(), attr, hlnode);
	                    }
	                }
	            }
	        }
	        else {
	            var nodeProperty = hlnode.property();
	            if (nodeProperty) {
	                return searchInTheValue(offset, unit.contents(), null, hlnode, nodeProperty);
	            }
	        }
	    }
	    if (kind == LocationKind.KEY_COMPLETION || kind == LocationKind.SEQUENCE_KEY_COPLETION) {
	        var hlnode = node;
	        var pp = node.property();
	        if (pp instanceof defs.UserDefinedProp) {
	            var up = pp;
	            return up.node();
	        }
	        if (node instanceof hlimpl.ASTNodeImpl) {
	            if (hlnode.definition() instanceof defs.UserDefinedClass) {
	                var uc = hlnode.definition();
	                if (uc.isAssignableFrom("TypeDeclaration")) {
	                    return node;
	                }
	                return uc.getAdapter(ramlServices.RAMLService).getDeclaringNode();
	            }
	        }
	        if (node instanceof hlimpl.ASTPropImpl) {
	            var pr = node;
	            if (isExampleNodeContent(pr)) {
	                var contentType = findExampleContentType(pr);
	                if (contentType) {
	                    var documentationRoot = parseDocumentationContent(pr, contentType);
	                    if (documentationRoot) {
	                        var node = deepFindNode(documentationRoot, offset, offset);
	                        pp = node.property();
	                        if (pp instanceof defs.UserDefinedProp) {
	                            var up = pp;
	                            return up.node();
	                        }
	                        if (node instanceof hlimpl.ASTNodeImpl) {
	                            if (hlnode.definition() instanceof defs.UserDefinedClass) {
	                                var uc = hlnode.definition();
	                                return uc.getAdapter(ramlServices.RAMLService).getDeclaringNode();
	                            }
	                        }
	                    }
	                }
	            }
	        }
	    }
	    if (kind == LocationKind.PATH_COMPLETION) {
	        var inclpath = getValueAt(unit.contents(), offset);
	        if (inclpath) {
	            var ap = unit.resolve(inclpath);
	            return ap;
	        }
	    }
	}
	exports.findDeclaration = findDeclaration;
	function findExampleContentType(node) {
	    var p = node.parent();
	    return p.localType();
	}
	exports.findExampleContentType = findExampleContentType;
	function parseDocumentationContent(attribute, type) {
	    if (!(attribute.value() instanceof hlimpl.StructuredValue)) {
	        return null;
	    }
	    return new hlimpl.ASTNodeImpl(attribute.value().lowLevel(), attribute.parent(), type, attribute.property());
	}
	exports.parseDocumentationContent = parseDocumentationContent;
	function isExampleNodeContent(node) {
	    var typeExampleName10 = universes.Universe10.TypeDeclaration.properties.example.name;
	    var objectName10 = universes.Universe10.ObjectTypeDeclaration.name;
	    if (!(node instanceof hlimpl.ASTPropImpl)) {
	        return false;
	    }
	    var property = node;
	    var parent = property.parent();
	    var parentProperty = parent && parent.property();
	    var parentPropertyName = parentProperty && parentProperty.nameId();
	    if (typeExampleName10 === property.name() && property.isString()) {
	        if (parent instanceof hlimpl.ASTNodeImpl && parent.definition().isAssignableFrom(objectName10)) {
	            return true;
	        }
	    }
	    return false;
	}
	exports.isExampleNodeContent = isExampleNodeContent;
	function determineCompletionKind(text, offset) {
	    var hasIn = false;
	    var hasSeq = false;
	    var canBeInComment = false;
	    var canBeAnnotation = false;
	    for (var i = offset - 1; i >= 0; i--) {
	        var c = text.charAt(i);
	        if (c == '(') {
	            canBeAnnotation = true;
	        }
	        else if (canBeAnnotation) {
	            if (c == '\r' || c == '\n') {
	                var hasClosing = false;
	                for (var j = offset - 1; j < text.length; j++) {
	                    var ch = text[j];
	                    if (ch == ')') {
	                        hasClosing = true;
	                    }
	                    if (ch == '\r' || ch == "\n") {
	                        break;
	                    }
	                    if (ch == ':') {
	                        canBeAnnotation = false;
	                        break;
	                    }
	                }
	                if (canBeAnnotation && hasClosing) {
	                    return LocationKind.ANNOTATION_COMPLETION;
	                }
	                else {
	                    break;
	                }
	            }
	            if (c == ' ' || c == '\t') {
	                continue;
	            }
	            else {
	                break;
	            }
	        }
	        else {
	            if (c == '\r' || c == '\n') {
	                break;
	            }
	            if (c == ':') {
	                break;
	            }
	        }
	    }
	    for (var i = offset - 1; i >= 0; i--) {
	        var c = text.charAt(i);
	        if (c == '#') {
	            if (i == 0) {
	                return LocationKind.VERSION_COMPLETION;
	            }
	            //subsequent check for include
	            for (var j = i - 1; j >= 0; j--) {
	                var currentChar = text.charAt(j);
	                if (currentChar == '\r' || currentChar == '\n') {
	                    break;
	                }
	                else if (currentChar == '!') {
	                    if (text.indexOf("!include", j) == j) {
	                        return LocationKind.PATH_COMPLETION;
	                    }
	                }
	            }
	            return LocationKind.INCOMMENT;
	        }
	        if (c == ':') {
	            if (hasIn) {
	                return LocationKind.DIRECTIVE_COMPLETION;
	            }
	            return LocationKind.VALUE_COMPLETION;
	        }
	        if (c == '\r' || c == '\n') {
	            //check for multiline literal
	            var insideOfMultiline = false;
	            var ind = getIndent2(offset, text);
	            for (var a = i; a > 0; a--) {
	                c = text.charAt(a);
	                //TODO this can be further improved
	                if (c == ':') {
	                    if (insideOfMultiline) {
	                        var ll = getIndent2(a, text);
	                        if (ll.length < ind.length) {
	                            return LocationKind.VALUE_COMPLETION;
	                        }
	                    }
	                    break;
	                }
	                if (c == '|') {
	                    insideOfMultiline = true;
	                    continue;
	                }
	                if (c == '\r' || c == '\n') {
	                    insideOfMultiline = false;
	                }
	                if (c != ' ' && c != '\t') {
	                    insideOfMultiline = false;
	                }
	            }
	            if (hasSeq) {
	                return LocationKind.SEQUENCE_KEY_COPLETION;
	            }
	            return LocationKind.KEY_COMPLETION;
	        }
	        if (c == '-') {
	            hasSeq = true;
	        }
	        if (c == '!') {
	            if (text.indexOf("!include", i) == i) {
	                return LocationKind.PATH_COMPLETION;
	            }
	            if (text.indexOf("!i", i) == i) {
	                hasIn = true;
	            }
	        }
	    }
	}
	exports.determineCompletionKind = determineCompletionKind;
	(function (LocationKind) {
	    LocationKind[LocationKind["VALUE_COMPLETION"] = 0] = "VALUE_COMPLETION";
	    LocationKind[LocationKind["KEY_COMPLETION"] = 1] = "KEY_COMPLETION";
	    LocationKind[LocationKind["PATH_COMPLETION"] = 2] = "PATH_COMPLETION";
	    LocationKind[LocationKind["DIRECTIVE_COMPLETION"] = 3] = "DIRECTIVE_COMPLETION";
	    LocationKind[LocationKind["VERSION_COMPLETION"] = 4] = "VERSION_COMPLETION";
	    LocationKind[LocationKind["ANNOTATION_COMPLETION"] = 5] = "ANNOTATION_COMPLETION";
	    LocationKind[LocationKind["SEQUENCE_KEY_COPLETION"] = 6] = "SEQUENCE_KEY_COPLETION";
	    LocationKind[LocationKind["INCOMMENT"] = 7] = "INCOMMENT";
	})(exports.LocationKind || (exports.LocationKind = {}));
	var LocationKind = exports.LocationKind;
	function resolveReference(point, path) {
	    if (!path) {
	        return null;
	    }
	    var sp = path.split("/");
	    var result = point;
	    for (var i = 0; i < sp.length; i++) {
	        if (sp[i] == '#') {
	            result = point.unit().ast();
	            continue;
	        }
	        result = _.find(result.children(), function (x) { return x.key() == sp[i]; });
	        if (!result) {
	            return null;
	        }
	    }
	    return result;
	}
	exports.resolveReference = resolveReference;
	/**
	 * return all sub types of given type visible from parent node
	 * @param range
	 * @param parentNode
	 * @returns ITypeDefinition[]
	 */
	function subTypesWithLocals(range, parentNode) {
	    if (range == null) {
	        return [];
	    }
	    var name = range.nameId();
	    parentNode = exports.declRoot(parentNode);
	    var actual = parentNode;
	    if (actual._subTypesCache) {
	        var cached = actual._subTypesCache[name];
	        if (cached) {
	            return cached;
	        }
	    }
	    else {
	        actual._subTypesCache = {};
	    }
	    var result = range.allSubTypes();
	    if (range.getAdapter(ramlServices.RAMLService).getRuntimeExtenders().length > 0 && parentNode) {
	        var decls = globalDeclarations(parentNode);
	        var extenders = range.getAdapter(ramlServices.RAMLService).getRuntimeExtenders();
	        var root = parentNode.root();
	        extenders.forEach(function (x) {
	            var definitionNodes = decls.filter(function (z) {
	                var def = z.definition().allSuperTypes();
	                def.push(z.definition());
	                var rr = (z.definition() == x) || (_.find(def, function (d) { return d == x; }) != null) || (_.find(def, function (d) { return d == range; }) != null);
	                return rr;
	            });
	            result = result.concat(definitionNodes.map(function (x) { return x.localType(); }));
	        });
	    }
	    result = _.unique(result);
	    actual._subTypesCache[name] = result;
	    return result;
	}
	exports.subTypesWithLocals = subTypesWithLocals;
	;
	function subTypesWithName(tname, parentNode, backup) {
	    parentNode = exports.declRoot(parentNode);
	    var decls = globalDeclarations(parentNode);
	    var declNode = _.find(decls, function (x) { return hlimpl.qName(x, parentNode) == tname && x.property() &&
	        (x.property().nameId() == universes.Universe10.LibraryBase.properties.types.name); });
	    return declNode.localType();
	}
	exports.subTypesWithName = subTypesWithName;
	;
	function schemasWithName(tname, parentNode, backup) {
	    parentNode = exports.declRoot(parentNode);
	    var decls = globalDeclarations(parentNode);
	    var declNode = _.find(decls, function (x) { return hlimpl.qName(x, parentNode) == tname && x.property() &&
	        (x.property().nameId() == universes.Universe10.LibraryBase.properties.schemas.name); });
	    return declNode.localType();
	}
	exports.schemasWithName = schemasWithName;
	;
	function nodesDeclaringType(range, n) {
	    var result = [];
	    var extenders = range.getAdapter(ramlServices.RAMLService).getRuntimeExtenders();
	    if (extenders.length > 0 && n) {
	        var root = n;
	        extenders.forEach(function (x) {
	            var definitionNodes = globalDeclarations(root).filter(function (z) { return z.definition().isAssignableFrom(x.nameId()); });
	            result = result.concat(definitionNodes);
	        });
	    }
	    var isElementType = !range.hasValueTypeInHierarchy();
	    if (isElementType && range.getAdapter(ramlServices.RAMLService).isInlinedTemplates() && n) {
	        var root = n;
	        //TODO I did not like it it might be written much better
	        var definitionNodes = globalDeclarations(root).filter(function (z) { return z.definition() == range; });
	        result = result.concat(definitionNodes);
	    }
	    else {
	        var root = n;
	        var q = {};
	        range.allSubTypes().forEach(function (x) { return q[x.nameId()] = true; });
	        q[range.nameId()] = true;
	        var definitionNodes = globalDeclarations(root).filter(function (z) { return q[z.definition().nameId()]; });
	        result = result.concat(definitionNodes);
	    }
	    return result;
	}
	exports.nodesDeclaringType = nodesDeclaringType;
	;
	function findAllSubTypes(p, n) {
	    var range = p.range();
	    return subTypesWithLocals(range, n);
	}
	exports.findAllSubTypes = findAllSubTypes;
	;
	function possibleNodes(p, c) {
	    if (c) {
	        if (p.isDescriminator()) {
	            var range = p.range();
	            var extenders = range.getAdapter(ramlServices.RAMLService).getRuntimeExtenders();
	            if (extenders.length > 0 && c) {
	                var result = [];
	                extenders.forEach(function (x) {
	                    var definitionNodes = globalDeclarations(c).filter(function (z) { return z.definition() == x; });
	                    result = result.concat(definitionNodes);
	                });
	                return result;
	            }
	            return [];
	        }
	        if (p.isReference()) {
	            return nodesDeclaringType(p.referencesTo(), c);
	        }
	        if (p.range().hasValueTypeInHierarchy()) {
	            var vt = p.range().getAdapter(ramlServices.RAMLService);
	            if (vt.globallyDeclaredBy && vt.globallyDeclaredBy().length > 0) {
	                var definitionNodes = globalDeclarations(c).filter(function (z) { return _.find(vt.globallyDeclaredBy(), function (x) { return x == z.definition(); }) != null; });
	                return definitionNodes;
	            }
	        }
	    }
	    return [];
	}
	function allChildren(node) {
	    var res = [];
	    gather(node, res);
	    return res;
	}
	exports.allChildren = allChildren;
	function gather(node, result) {
	    node.children().forEach(function (x) { result.push(x); gather(x, result); });
	}
	var testUsage = function (ck, x, node, result) {
	    var tp = ck.getAdapter(ramlServices.RAMLService).getDeclaringNode();
	    if (tp) {
	        if (node.isSameNode(tp)) {
	            result.push(x);
	            return;
	        }
	    }
	    if (ck.isArray()) {
	        testUsage(ck.array().componentType(), x, node, result);
	    }
	    if (ck.isUnion()) {
	        var uni = ck.union();
	        testUsage(uni.leftType(), x, node, result);
	        testUsage(uni.rightType(), x, node, result);
	    }
	    if (ck.superTypes().some(function (x) { return x.nameId() == node.name(); })) {
	        result.push(x);
	    }
	};
	function refFinder(root, node, result) {
	    root.elements().forEach(function (x) {
	        refFinder(x, node, result);
	        //console.log(x.name())
	        var ck = x.definition();
	        //testUsage(ck, x, node, result);
	    });
	    root.attrs().forEach(function (a) {
	        var pr = a.property();
	        var vl = a.value();
	        //if (pr.isTypeExpr()){
	        //    typeExpression.
	        //}
	        if (pr instanceof defs.UserDefinedProp) {
	            var up = pr.node();
	            if (up == node) {
	                result.push(a);
	            }
	            else if (up.lowLevel().start() == node.lowLevel().start()) {
	                if (up.lowLevel().unit() == node.lowLevel().unit()) {
	                    result.push(a);
	                }
	            }
	        }
	        if (isExampleNodeContent(a)) {
	            var contentType = findExampleContentType(a);
	            if (contentType) {
	                var documentationRoot = parseDocumentationContent(a, contentType);
	                if (documentationRoot) {
	                    refFinder(documentationRoot, node, result);
	                }
	            }
	        }
	        else if (pr.getAdapter(ramlServices.RAMLPropertyService).isTypeExpr() && typeof vl == "string") {
	            var tpa = root.localType();
	            testUsage(tpa, a, node, result);
	            var libraryName = getLibraryName(node);
	            if (libraryName && vl.indexOf(libraryName) != -1) {
	                var referencingLibrary = getLibraryDefiningNode(a);
	                if (referencingLibrary &&
	                    referencingLibrary.lowLevel().start() == node.lowLevel().start()) {
	                    result.push(a);
	                }
	            }
	        }
	        if (pr.isReference() || pr.isDescriminator()) {
	            if (typeof vl == 'string') {
	                var pn = possibleNodes(pr, root);
	                if (_.find(pn, function (x) { return x.name() == vl && x == node; })) {
	                    result.push(a);
	                }
	                var libraryName = getLibraryName(node);
	                if (libraryName && vl.indexOf(libraryName) != -1) {
	                    var referencingLibrary = getLibraryDefiningNode(a);
	                    if (referencingLibrary &&
	                        referencingLibrary.lowLevel().start() == node.lowLevel().start()) {
	                        result.push(a);
	                    }
	                }
	            }
	            else {
	                var st = vl;
	                if (st) {
	                    var vn = st.valueName();
	                    var pn = possibleNodes(pr, root);
	                    if (_.find(pn, function (x) { return x.name() == vn && x == node; })) {
	                        result.push(a);
	                    }
	                    var hnode = st.toHighLevel();
	                    if (hnode) {
	                        refFinder(hnode, node, result);
	                    }
	                    var libraryName = getLibraryName(node);
	                    if (libraryName && vn.indexOf(libraryName) != -1) {
	                        var referencingLibrary = getLibraryDefiningNode(hnode);
	                        if (referencingLibrary &&
	                            referencingLibrary.lowLevel().start() == node.lowLevel().start()) {
	                            result.push(a);
	                        }
	                    }
	                }
	            }
	        }
	        else {
	            var pn = possibleNodes(pr, root);
	            if (_.find(pn, function (x) { return x.name() == vl && x == node; })) {
	                result.push(a);
	            }
	        }
	    });
	}
	exports.refFinder = refFinder;
	/**
	 * Returns library node that definition of the current node is located in, or null
	 * if current node is not defined in a library.
	 */
	function getLibraryDefiningNode(nodeToCheck) {
	    if (!nodeToCheck.lowLevel) {
	        return null;
	    }
	    var lowLevelNode = nodeToCheck.lowLevel();
	    if (!lowLevelNode) {
	        return null;
	    }
	    if (lowLevelNode.key()) {
	        var offset = Math.floor((lowLevelNode.keyEnd() + lowLevelNode.keyStart()) / 2);
	        var result = getLibraryDefiningNodeByOffset(lowLevelNode.unit(), offset);
	        if (result)
	            return result;
	    }
	    if (lowLevelNode.value()) {
	        var offset = Math.floor((lowLevelNode.valueEnd() + lowLevelNode.valueStart()) / 2);
	        var result = getLibraryDefiningNodeByOffset(lowLevelNode.unit(), offset);
	        if (result)
	            return result;
	    }
	    return null;
	}
	function getLibraryDefiningNodeByOffset(unit, offset) {
	    var declaration = findDeclaration(unit, offset);
	    if (declaration && declaration.isElement && declaration.isElement()) {
	        var declarationNode = declaration.asElement();
	        var parent = declarationNode;
	        while (parent) {
	            if (parent.definition().key() == universes.Universe10.Library) {
	                return parent;
	            }
	            parent = parent.parent();
	        }
	    }
	    return null;
	}
	//# sourceMappingURL=search.js.map

/***/ },
/* 30 */
/***/ function(module, exports, __webpack_require__) {

	(function (root) {

	  // Store setTimeout reference so promise-polyfill will be unaffected by
	  // other code modifying setTimeout (like sinon.useFakeTimers())
	  var setTimeoutFunc = setTimeout;

	  function noop() {
	  }

	  // Use polyfill for setImmediate for performance gains
	  var asap = (typeof setImmediate === 'function' && setImmediate) ||
	    function (fn) {
	      setTimeoutFunc(fn, 1);
	    };

	  var onUnhandledRejection = function onUnhandledRejection(err) {
	    if (typeof console !== 'undefined' && console) {
	      console.warn('Possible Unhandled Promise Rejection:', err); // eslint-disable-line no-console
	    }
	  };

	  // Polyfill for Function.prototype.bind
	  function bind(fn, thisArg) {
	    return function () {
	      fn.apply(thisArg, arguments);
	    };
	  }

	  function Promise(fn) {
	    if (typeof this !== 'object') throw new TypeError('Promises must be constructed via new');
	    if (typeof fn !== 'function') throw new TypeError('not a function');
	    this._state = 0;
	    this._handled = false;
	    this._value = undefined;
	    this._deferreds = [];

	    doResolve(fn, this);
	  }

	  function handle(self, deferred) {
	    while (self._state === 3) {
	      self = self._value;
	    }
	    if (self._state === 0) {
	      self._deferreds.push(deferred);
	      return;
	    }
	    self._handled = true;
	    asap(function () {
	      var cb = self._state === 1 ? deferred.onFulfilled : deferred.onRejected;
	      if (cb === null) {
	        (self._state === 1 ? resolve : reject)(deferred.promise, self._value);
	        return;
	      }
	      var ret;
	      try {
	        ret = cb(self._value);
	      } catch (e) {
	        reject(deferred.promise, e);
	        return;
	      }
	      resolve(deferred.promise, ret);
	    });
	  }

	  function resolve(self, newValue) {
	    try {
	      // Promise Resolution Procedure: https://github.com/promises-aplus/promises-spec#the-promise-resolution-procedure
	      if (newValue === self) throw new TypeError('A promise cannot be resolved with itself.');
	      if (newValue && (typeof newValue === 'object' || typeof newValue === 'function')) {
	        var then = newValue.then;
	        if (newValue instanceof Promise) {
	          self._state = 3;
	          self._value = newValue;
	          finale(self);
	          return;
	        } else if (typeof then === 'function') {
	          doResolve(bind(then, newValue), self);
	          return;
	        }
	      }
	      self._state = 1;
	      self._value = newValue;
	      finale(self);
	    } catch (e) {
	      reject(self, e);
	    }
	  }

	  function reject(self, newValue) {
	    self._state = 2;
	    self._value = newValue;
	    finale(self);
	  }

	  function finale(self) {
	    if (self._state === 2 && self._deferreds.length === 0) {
	      asap(function() {
	        if (!self._handled) {
	          onUnhandledRejection(self._value);
	        }
	      }, 1);
	    }
	    
	    for (var i = 0, len = self._deferreds.length; i < len; i++) {
	      handle(self, self._deferreds[i]);
	    }
	    self._deferreds = null;
	  }

	  function Handler(onFulfilled, onRejected, promise) {
	    this.onFulfilled = typeof onFulfilled === 'function' ? onFulfilled : null;
	    this.onRejected = typeof onRejected === 'function' ? onRejected : null;
	    this.promise = promise;
	  }

	  /**
	   * Take a potentially misbehaving resolver function and make sure
	   * onFulfilled and onRejected are only called once.
	   *
	   * Makes no guarantees about asynchrony.
	   */
	  function doResolve(fn, self) {
	    var done = false;
	    try {
	      fn(function (value) {
	        if (done) return;
	        done = true;
	        resolve(self, value);
	      }, function (reason) {
	        if (done) return;
	        done = true;
	        reject(self, reason);
	      });
	    } catch (ex) {
	      if (done) return;
	      done = true;
	      reject(self, ex);
	    }
	  }

	  Promise.prototype['catch'] = function (onRejected) {
	    return this.then(null, onRejected);
	  };

	  Promise.prototype.then = function (onFulfilled, onRejected) {
	    var prom = new Promise(noop);
	    handle(this, new Handler(onFulfilled, onRejected, prom));
	    return prom;
	  };

	  Promise.all = function (arr) {
	    var args = Array.prototype.slice.call(arr);

	    return new Promise(function (resolve, reject) {
	      if (args.length === 0) return resolve([]);
	      var remaining = args.length;

	      function res(i, val) {
	        try {
	          if (val && (typeof val === 'object' || typeof val === 'function')) {
	            var then = val.then;
	            if (typeof then === 'function') {
	              then.call(val, function (val) {
	                res(i, val);
	              }, reject);
	              return;
	            }
	          }
	          args[i] = val;
	          if (--remaining === 0) {
	            resolve(args);
	          }
	        } catch (ex) {
	          reject(ex);
	        }
	      }

	      for (var i = 0; i < args.length; i++) {
	        res(i, args[i]);
	      }
	    });
	  };

	  Promise.resolve = function (value) {
	    if (value && typeof value === 'object' && value.constructor === Promise) {
	      return value;
	    }

	    return new Promise(function (resolve) {
	      resolve(value);
	    });
	  };

	  Promise.reject = function (value) {
	    return new Promise(function (resolve, reject) {
	      reject(value);
	    });
	  };

	  Promise.race = function (values) {
	    return new Promise(function (resolve, reject) {
	      for (var i = 0, len = values.length; i < len; i++) {
	        values[i].then(resolve, reject);
	      }
	    });
	  };

	  /**
	   * Set the immediate function to execute callbacks
	   * @param fn {function} Function to execute
	   * @private
	   */
	  Promise._setImmediateFn = function _setImmediateFn(fn) {
	    asap = fn;
	  };
	  
	  Promise._setUnhandledRejectionFn = function _setUnhandledRejectionFn(fn) {
	    onUnhandledRejection = fn;
	  };

	  if (typeof module !== 'undefined' && module.exports) {
	    module.exports = Promise;
	  } else if (!root.Promise) {
	    root.Promise = Promise;
	  }

	})(this);


/***/ },
/* 31 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var path = __webpack_require__(15);
	var URL = __webpack_require__(59);
	var util = __webpack_require__(43);
	var resourceRegistry = __webpack_require__(24);
	var ContentProvider = (function () {
	    function ContentProvider(unit) {
	        this.unit = unit;
	    }
	    ContentProvider.prototype.contextPath = function () {
	        if (!this.unit) {
	            return "";
	        }
	        var rootPath = this.unit.absolutePath();
	        return rootPath || "";
	    };
	    ContentProvider.prototype.normalizePath = function (url) {
	        if (!url) {
	            return url;
	        }
	        var result;
	        if (!isWebPath(url)) {
	            result = path.normalize(url);
	        }
	        else {
	            var prefix = url.toLowerCase().indexOf('https') === 0 ? 'https://' : 'http://';
	            result = prefix + path.normalize(url.substring(prefix.length));
	        }
	        return result;
	    };
	    ContentProvider.prototype.content = function (reference) {
	        var absolutePath = this.normalizePath(reference);
	        var unit = this.unit.resolve(absolutePath);
	        if (!unit) {
	            return "";
	        }
	        return unit.contents() || "";
	    };
	    ContentProvider.prototype.contentAsync = function (reference) {
	        var absolutePath = this.normalizePath(reference);
	        var unitPromise = this.unit.resolveAsync(absolutePath);
	        if (!unitPromise) {
	            return Promise.resolve("");
	        }
	        var result = unitPromise.then(function (unit) {
	            return (unit && unit.contents()) || "";
	        });
	        return result;
	    };
	    ContentProvider.prototype.hasAsyncRequests = function () {
	        return resourceRegistry.hasAsyncRequests();
	    };
	    ContentProvider.prototype.resolvePath = function (context, relativePath) {
	        if (!relativePath || !context) {
	            return relativePath;
	        }
	        var result;
	        if (!isWebPath(context)) {
	            result = path.resolve(path.dirname(context), relativePath);
	        }
	        else {
	            result = URL.resolve(context, relativePath);
	        }
	        return result;
	    };
	    ContentProvider.prototype.isAbsolutePath = function (uri) {
	        if (!uri) {
	            return false;
	        }
	        if (isWebPath(uri)) {
	            return true;
	        }
	        return path.isAbsolute(uri);
	    };
	    ContentProvider.prototype.promiseResolve = function (arg) {
	        return Promise.resolve(arg);
	    };
	    return ContentProvider;
	}());
	exports.ContentProvider = ContentProvider;
	function isWebPath(str) {
	    if (str == null)
	        return false;
	    return util.stringStartsWith(str, "http://") || util.stringStartsWith(str, "https://");
	}
	//# sourceMappingURL=contentprovider.js.map

/***/ },
/* 32 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var RamlWrapper = __webpack_require__(12);
	var RamlWrapperImpl = __webpack_require__(20);
	var factory = __webpack_require__(41);
	var core = __webpack_require__(45);
	var ramlPathMatch = __webpack_require__(51);
	var hl = __webpack_require__(10);
	var hlimpl = __webpack_require__(16);
	var linter = __webpack_require__(27);
	var stubs = __webpack_require__(17);
	var defs = __webpack_require__(35);
	var universeDef = __webpack_require__(22);
	var universes = __webpack_require__(22);
	var Opt = __webpack_require__(21);
	var util = __webpack_require__(43);
	var expander = __webpack_require__(26);
	var lowLevelProxy = __webpack_require__(37);
	var search = __webpack_require__(29);
	var ll = __webpack_require__(23);
	var json = __webpack_require__(48);
	var path = __webpack_require__(15);
	var universeHelpers = __webpack_require__(14);
	function resolveType(p) {
	    return p.highLevel().localType();
	}
	exports.resolveType = resolveType;
	//__$helperMethod__ Runtime representation of type represented by this AST node
	function runtimeType(p) {
	    return p.highLevel().localType();
	}
	exports.runtimeType = runtimeType;
	function load(pth) {
	    var m = new ll.Project(path.dirname(pth));
	    var unit = m.unit(path.basename(pth));
	    if (unit) {
	        if (unit.isRAMLUnit()) {
	            return hlimpl.fromUnit(unit).wrapperNode();
	        }
	    }
	    return null;
	}
	exports.load = load;
	//__$helperMethod__ Path relative to API root
	function completeRelativeUri(res) {
	    var uri = '';
	    var parent = res;
	    do {
	        res = parent; //(parent instanceof RamlWrapper.ResourceImpl) ? <RamlWrapper.Resource>parent : null;
	        uri = res.relativeUri().value() + uri;
	        parent = res.parent();
	    } while (parent.definition().key().name == universes.Universe10.Resource.name);
	    return uri;
	}
	exports.completeRelativeUri = completeRelativeUri;
	/**
	 * __$helperMethod__
	 * Equivalent API with traits and resource types expanded
	 * __$meta__={"name":"expand"}
	 */
	function expandTraitsAndResourceTypes(api) {
	    var lowLevelNode = api.highLevel().lowLevel();
	    if (lowLevelNode instanceof lowLevelProxy.LowLevelProxyNode) {
	        return api;
	    }
	    return expander.expandTraitsAndResourceTypes(api);
	}
	exports.expandTraitsAndResourceTypes = expandTraitsAndResourceTypes;
	//__$helperMethod__ baseUri of owning Api concatenated with completeRelativeUri
	function absoluteUri(res) {
	    var uri = '';
	    var parent = res;
	    do {
	        res = parent; //(parent instanceof RamlWrapper.ResourceImpl) ? <RamlWrapper.Resource>parent : null;
	        uri = res.relativeUri().value() + uri;
	        parent = res.parent();
	    } while (parent.definition().key().name == universes.Universe10.Resource.name);
	    uri = uri.replace(/\/\//g, '/');
	    var buri = parent.baseUri();
	    var base = buri ? buri.value() : "";
	    base = base ? base : '';
	    if (util.stringEndsWith(base, '/')) {
	        uri = uri.substring(1);
	    }
	    uri = base + uri;
	    return uri;
	}
	exports.absoluteUri = absoluteUri;
	//__$helperMethod__ validate an instance against type
	function validateInstance(res, value) {
	    return res.runtimeType().validate(value).map(function (x) { return x.getMessage(); });
	    //throw new Error("Fix me");
	}
	exports.validateInstance = validateInstance;
	//__$helperMethod__ validate an instance against type
	function validateInstanceWithDetailedStatuses(res, value) {
	    return res.runtimeType().validate(value);
	}
	exports.validateInstanceWithDetailedStatuses = validateInstanceWithDetailedStatuses;
	/**
	 * __$helperMethod__
	 * Retrieve all traits including those defined in libraries
	 * __$meta__{"name":"traits","override":true}
	 */
	function traitsPrimary(a) {
	    return allTraits(a);
	}
	exports.traitsPrimary = traitsPrimary;
	/**
	 * __$helperMethod__ Retrieve all traits including those defined in libraries
	 * __$meta__{"deprecated":true}
	 */
	function allTraits(a) {
	    return findTemplates(a, function (d) { return universeHelpers.isTraitType(d); });
	}
	exports.allTraits = allTraits;
	/**
	 * __$helperMethod__
	 * Retrieve all resource types including those defined in libraries
	 * __$meta__{"name":"resourceTypes","override":true}
	 */
	function resourceTypesPrimary(a) {
	    return allResourceTypes(a);
	}
	exports.resourceTypesPrimary = resourceTypesPrimary;
	/**
	 * __$helperMethod__ Retrieve all resource types including those defined in libraries
	 * __$meta__{"deprecated":true}
	 */
	function allResourceTypes(a) {
	    return findTemplates(a, function (d) { return universeHelpers.isResourceTypeType(d); });
	}
	exports.allResourceTypes = allResourceTypes;
	function findTemplates(a, filter) {
	    var arr = search.globalDeclarations(a.highLevel()).filter(function (x) { return filter(x.definition()); });
	    var ll = a.highLevel().lowLevel();
	    var nodePath = ll.includePath();
	    if (!nodePath) {
	        nodePath = ll.unit().path();
	    }
	    var topLevelArr = arr.map(function (x) {
	        var topLevelNode;
	        var p = x.lowLevel().unit().path();
	        if (p != nodePath) {
	            topLevelNode = factory.buildWrapperNode(x, false);
	            topLevelNode.meta().setCalculated();
	        }
	        else {
	            topLevelNode = x.wrapperNode();
	        }
	        return topLevelNode;
	    });
	    return topLevelArr;
	}
	;
	function relativeUriSegments(res) {
	    var result = [];
	    var parent = res;
	    do {
	        res = parent; //(parent instanceof RamlWrapper.ResourceImpl) ? <RamlWrapper.Resource>parent : null;
	        result.push(res.relativeUri().value());
	        parent = res.parent();
	    } while (parent.definition().key().name == universes.Universe10.Resource.name);
	    return result.reverse();
	}
	exports.relativeUriSegments = relativeUriSegments;
	//__$helperMethod__ For methods of Resources returns parent resource. For methods of ResourceTypes returns null.
	function parentResource(method) {
	    if (method.parent() instanceof RamlWrapperImpl.ResourceImpl) {
	        return method.parent();
	    }
	    return null;
	}
	exports.parentResource = parentResource;
	/**
	 * __$helperMethod__
	 * Parent resource for non top level resources
	 * __$meta__={"name":"parentResource"}
	 */
	function parent(resource) {
	    var parent = resource.parent();
	    if (parent.definition().key().name == universes.Universe10.Resource.name) {
	        return parent;
	    }
	    return null;
	}
	exports.parent = parent;
	//__$helperMethod__ Get child resource by its relative path
	function childResource(container, relPath) {
	    if (container == null) {
	        return null;
	    }
	    var resources = container.resources();
	    if (!resources) {
	        return null;
	    }
	    resources = resources.filter(function (x) { return x.relativeUri().value() == relPath; });
	    if (resources.length == 0) {
	        return null;
	    }
	    return resources[0];
	}
	exports.childResource = childResource;
	function getResource(container, path) {
	    if (!container) {
	        return null;
	    }
	    var res = null;
	    for (var i = 0; i < path.length; i++) {
	        res = childResource(container, path[i]);
	        if (!res) {
	            return null;
	        }
	        container = res;
	    }
	    return res;
	}
	exports.getResource = getResource;
	//__$helperMethod__ Get child method by its name
	function childMethod(resource, method) {
	    if (!resource) {
	        return null;
	    }
	    return resource.methods().filter(function (x) { return x.method() == method; });
	}
	exports.childMethod = childMethod;
	function getMethod(container, path, method) {
	    var resource = getResource(container, path);
	    if (!resource) {
	        return null;
	    }
	    return childMethod(resource, method);
	}
	exports.getMethod = getMethod;
	function isApi(obj) {
	    return universeHelpers.isApiSibling(obj.definition());
	}
	;
	//__$helperMethod__ Api owning the resource as a sibling
	function ownerApi(method) {
	    var obj = method;
	    while (!isApi(obj)) {
	        obj = obj.parent();
	    }
	    return obj;
	}
	exports.ownerApi = ownerApi;
	/**
	 * __$helperMethod__
	 * For methods of Resources: `{parent Resource relative path} {methodName}`.
	 * For methods of ResourceTypes: `{parent ResourceType name} {methodName}`.
	 * For other methods throws Exception.
	 */
	function methodId(method) {
	    var parent = method.parent();
	    if (parent instanceof RamlWrapperImpl.ResourceImpl) {
	        return completeRelativeUri(parent) + ' ' + method.method().toLowerCase();
	    }
	    else if (parent instanceof RamlWrapperImpl.ResourceTypeImpl) {
	        return parent.name() + ' ' + method.method().toLowerCase();
	    }
	    throw new Error("Method is supposed to be owned by Resource or ResourceType.\nHere the method is owned by " + method.definition().key().name);
	}
	exports.methodId = methodId;
	//__$helperMethod__ true for codes < 400 and false otherwise
	function isOkRange(response) {
	    var str = response.code().value();
	    var err = linter.validateResponseString(str);
	    if (err != null) {
	        return false;
	    }
	    try {
	        if (parseInt(str.charAt(0)) < 4) {
	            return true;
	        }
	    }
	    catch (e) { }
	    return false;
	}
	exports.isOkRange = isOkRange;
	//__$helperMethod__  Retrieve all resources of the Api
	function allResources(api) {
	    var resources = [];
	    var visitor = function (res) {
	        resources.push(res);
	        res.resources().forEach(function (x) { return visitor(x); });
	    };
	    api.resources().forEach(function (x) { return visitor(x); });
	    return resources;
	}
	exports.allResources = allResources;
	function matchUri(apiRootRelativeUri, resource) {
	    var allParameters = {};
	    while (resource != null) {
	        uriParameters(resource).forEach(function (x) { return allParameters[x.name()] = new ParamWrapper(x); });
	        resource = parent(resource);
	    }
	    var result = ramlPathMatch.ramlPathMatch(completeRelativeUri(resource), allParameters, {})(apiRootRelativeUri);
	    if (result) {
	        return new Opt(Object.keys(result.params)
	            .map(function (x) { return new ParamValue(x, result['params'][x]); }));
	    }
	    return Opt.empty();
	}
	exports.matchUri = matchUri;
	var schemaContentChars = ['{', '<'];
	// export function schema(body:RamlWrapper.TypeDeclaration, api:RamlWrapper.Api):Opt<SchemaDef>{
	//
	//     var schemaNode = body.schema();
	//     if(!schemaNode){
	//         return Opt.empty<SchemaDef>();
	//     }
	//     var schemaString = schemaNode;
	//     var isContent:boolean = false;
	//     schemaContentChars.forEach(x=>{try{ isContent = isContent||schemaString.indexOf(x)>=0}catch(e){}});
	//     var schDef:SchemaDef;
	//     if(isContent) {
	//         schDef = new SchemaDef(schemaString);
	//     }
	//     else{
	//         var globalSchemes = api.schemas().filter(x=>x.name()==schemaString);
	//         if(globalSchemes.length>0){
	//             schDef = new SchemaDef(globalSchemes[0].type(),globalSchemes[0].name());
	//         }
	//         else{
	//             return Opt.empty<SchemaDef>();
	//         }
	//     }
	//     return new Opt<SchemaDef>(schDef);
	// }
	/**
	 * __$helperMethod__
	 * Retrieve an ordered list of all uri parameters including those which are not described in the `uriParameters` node.
	 * Consider a fragment of RAML specification:
	 * ```yaml
	 * /resource/{objectId}/{propertyId}:
	 *   uriParameters:
	 *     objectId:
	 * ```
	 * Here `propertyId` uri parameter is not described in the `uriParameters` node,
	 * but it is among Resource.uriParameters().
	 * __$meta__={"name":"uriParameters","override": true}
	 */
	function uriParametersPrimary(resource) {
	    return uriParameters(resource);
	}
	exports.uriParametersPrimary = uriParametersPrimary;
	/**
	 * __$helperMethod__
	 * Retrieve an ordered list of all uri parameters including those which are not described in the `uriParameters` node.
	 * Consider a fragment of RAML specification:
	 * ```yaml
	 * /resource/{objectId}/{propertyId}:
	 *   uriParameters:
	 *     objectId:
	 * ```
	 * Here `propertyId` uri parameter is not described in the `uriParameters` node,
	 * but it is among Resource.allUriParameters().
	 * __$meta__={"name":"allUriParameters","deprecated":true}
	 */
	function uriParameters(resource) {
	    var params = resource.uriParameters_original();
	    if (!(resource instanceof RamlWrapperImpl.ResourceImpl)) {
	        return params;
	    }
	    var uri = resource.relativeUri().value();
	    var propName = universes.Universe10.ResourceBase.properties.uriParameters.name;
	    return extractParams(params, uri, resource, propName);
	}
	exports.uriParameters = uriParameters;
	/**
	 * __$helperMethod__
	 * Retrieve an ordered list of all base uri parameters regardless of whether they are described in `baseUriParameters` or not
	 * Consider a fragment of RAML specification:
	 * ```yaml
	 * version: v1
	 * baseUri: https://{organization}.example.com/{version}/{service}
	 * baseUriParameters:
	 *   service:
	 * ```
	 * Here `version` and `organization` are base uri parameters which are not described in the `baseUriParameters` node,
	 * but they are among `Api.baseUriParameters()`.
	 * __$meta__={"name":"baseUriParameters","override":true}
	 */
	function baseUriParametersPrimary(api) {
	    return baseUriParameters(api);
	}
	exports.baseUriParametersPrimary = baseUriParametersPrimary;
	/**
	 * __$helperMethod__
	 * Retrieve an ordered list of all base uri parameters regardless of whether they are described in `baseUriParameters` or not
	 * Consider a fragment of RAML specification:
	 * ```yaml
	 * version: v1
	 * baseUri: https://{organization}.example.com/{version}/{service}
	 * baseUriParameters:
	 *   service:
	 * ```
	 * Here `version` and `organization` are base uri parameters which are not described in the `baseUriParameters` node,
	 * but they are among `Api.allBaseUriParameters()`.
	 * __$meta__={"name":"allBaseUriParameters","deprecated":true}
	 */
	function baseUriParameters(api) {
	    var uri = api.baseUri() ? api.baseUri().value() : '';
	    var params = api.baseUriParameters_original();
	    var propName = universes.Universe10.Api.properties.baseUriParameters.name;
	    return extractParams(params, uri, api, propName);
	}
	exports.baseUriParameters = baseUriParameters;
	/**
	 * __$helperMethod__
	 * Retrieve an ordered list of all absolute uri parameters. Returns a union of `Api.baseUriParameters()`
	 * for `Api` owning the `Resource` and `Resource.uriParameters()`.
	 */
	function absoluteUriParameters(res) {
	    var params = [];
	    var parent = res;
	    do {
	        res = parent;
	        params = uriParameters(res).concat(params);
	        parent = res.parent();
	    } while (parent.definition().key().name == universes.Universe10.Resource.name);
	    var api = parent;
	    var baseUri = api.baseUri().value();
	    var baseUriParams = api.baseUriParameters();
	    params = baseUriParameters(api).concat(params);
	    return params;
	}
	exports.absoluteUriParameters = absoluteUriParameters;
	/**
	 * _//_$helperMethod__
	 * Protocols used by the API. Returns the `protocols` property value if it is specified.
	 * Otherwise, returns protocol, specified in the base URI.
	 * __$meta__={"name":"protocols","override":true}
	 */
	function protocolsPrimary(api) {
	    return allProtocols(api);
	}
	exports.protocolsPrimary = protocolsPrimary;
	/**
	 * __$helperMethod__
	 * Protocols used by the API. Returns the `protocols` property value if it is specified.
	 * Otherwise, returns protocol, specified in the base URI.
	 * __$meta__{"deprecated":true}
	 */
	function allProtocols(api) {
	    return api.protocols().map(function (x) { return x.toUpperCase(); });
	    //var attributeDefaults = (<RamlWrapper.ApiImpl>api).attributeDefaults();
	    //var result = (<RamlWrapper.ApiImpl>api).protocols_original();
	    //if(result.length!=0||!attributeDefaults){
	    //    return result;
	    //}
	    //var baseUriAttr = api.baseUri();
	    //if(baseUriAttr) {
	    //    var baseUri = baseUriAttr.value();
	    //    if (baseUri) {
	    //        var ind = baseUri.indexOf('://');
	    //        if (ind >= 0) {
	    //            result = [baseUri.substring(0, ind)];
	    //        }
	    //        if(result.length==0){
	    //            result = [ 'HTTP' ];
	    //        }
	    //    }
	    //}
	    //return result;
	}
	exports.allProtocols = allProtocols;
	/**
	 * _//_$helperMethod__
	 * Returns security schemes, resource or method is secured with. If no security schemes are set at resource or method level,
	 * returns schemes defined with `securedBy` at API level.
	 * __$meta__={"name":"securedBy","override":true}
	 */
	function securedByPrimary(resourceOrMethod) {
	    return allSecuredBy(resourceOrMethod);
	}
	exports.securedByPrimary = securedByPrimary;
	/**
	 * __$helperMethod__
	 * Returns security schemes, resource or method is secured with. If no security schemes are set at resource or method level,
	 * returns schemes defined with `securedBy` at API level.
	 * __$meta__{"deprecated":true}
	 */
	function allSecuredBy(resourceOrMethod) {
	    //var currentSecuredBy = (<RamlWrapper.ResourceBaseImpl|RamlWrapper.MethodImpl>resourceOrMethod).securedBy_original();
	    //if (currentSecuredBy && currentSecuredBy.length > 0) {
	    //    return currentSecuredBy;
	    //}
	    //
	    ////instanceof, but have to avoid direct usage of instanceof in JS.
	    //var key = resourceOrMethod.highLevel().definition().key();
	    //if (key == universes.Universe10.Method) {
	    //    var method = (<RamlWrapper.Method>resourceOrMethod);
	    //    var resource = <RamlWrapper.ResourceImpl>method.parentResource();
	    //    if (resource && resource.securedBy_original() && resource.securedBy_original().length > 0) {
	    //        return resource.securedBy_original();
	    //    }
	    //    return method.ownerApi().securedBy();
	    //}
	    //if (key == universes.Universe10.Resource) {
	    //    return (<RamlWrapper.Resource>resourceOrMethod).ownerApi().securedBy();
	    //}
	    return resourceOrMethod.securedBy(); //return currentSecuredBy;
	}
	exports.allSecuredBy = allSecuredBy;
	/**
	 * __$helperMethod__
	 * __$meta__={"primary":true}
	 */
	function securitySchemeName(schemeReference) {
	    var highLevel = schemeReference.highLevel();
	    if (!highLevel)
	        return "";
	    var attributeValue = highLevel.value();
	    if (!attributeValue)
	        return "";
	    return attributeValue.toString();
	}
	exports.securitySchemeName = securitySchemeName;
	/**
	 * __$helperMethod__
	 * __$meta__={"primary":true}
	 */
	function securityScheme(schemeReference) {
	    var highLevel = schemeReference.highLevel();
	    if (!highLevel)
	        return null;
	    var declaration = search.findDeclarationByNode(highLevel, search.LocationKind.VALUE_COMPLETION);
	    if (!declaration)
	        return null;
	    if (!declaration.getKind || declaration.getKind() != hl.NodeKind.NODE) {
	        return null;
	    }
	    var result = declaration.wrapperNode();
	    if (!(result instanceof RamlWrapperImpl.AbstractSecuritySchemeImpl)) {
	        //I do not see how to avoid instanceof here
	        return null;
	    }
	    return result;
	}
	exports.securityScheme = securityScheme;
	/**
	 * __$helperMethod__
	 * __$meta__={"primary":true}
	 */
	function RAMLVersion(api) {
	    return api.highLevel().definition().universe().version();
	}
	exports.RAMLVersion = RAMLVersion;
	/**
	 * __$helperMethod__
	 * __$meta__={"primary":true}
	 */
	function structuredValue(reference) {
	    var hl = reference.value().lowLevel();
	    return new core.TypeInstanceImpl(hl);
	}
	exports.structuredValue = structuredValue;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"name","primary":true}
	 */
	function referenceName(reference) {
	    var val = reference.highLevel().value();
	    return (typeof val == 'string') || val == null ? val : val.valueName();
	}
	exports.referenceName = referenceName;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"trait","primary":true}
	 */
	function referencedTrait(ref) {
	    return referencedObject(ref);
	}
	exports.referencedTrait = referencedTrait;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"annotation","primary":true}
	 */
	function referencedAnnotation(ref) {
	    return referencedObject(ref);
	}
	exports.referencedAnnotation = referencedAnnotation;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"resourceType","primary":true}
	 */
	function referencedResourceType(ref) {
	    return referencedObject(ref);
	}
	exports.referencedResourceType = referencedResourceType;
	function referencedObject(ref) {
	    var attr = ref.highLevel();
	    var parent = attr.parent();
	    var vn = ref.name();
	    var cands = search.referenceTargets(attr.property(), parent).filter(function (x) { return hlimpl.qName(x, parent) == vn; });
	    if (cands.length == 0) {
	        return null;
	    }
	    return cands[0].wrapperNode();
	}
	function getExpandableExamples(node, isSingle) {
	    if (isSingle === void 0) { isSingle = false; }
	    var runtimeDefinition = node.runtimeDefinition();
	    if (!runtimeDefinition) {
	        return [];
	    }
	    var hlParent = node.highLevel();
	    var property = hlParent.definition().property(isSingle ? "example" : "examples");
	    var universe = defs.getUniverse("RAML10");
	    var definition = universe.type(universeDef.Universe10.ExampleSpec.name);
	    var expandables = runtimeDefinition.examples().filter(function (x) { return !x.isEmpty() && x.isSingle() == isSingle; });
	    return expandables.map(function (x) {
	        var obj = x.asJSON();
	        var llParent = hlParent.lowLevel();
	        var key = x.isSingle() ? "example" : null;
	        var jsonNode = new json.AstNode(llParent.unit(), obj, llParent, null, key);
	        var hlNode = new hlimpl.ASTNodeImpl(jsonNode, hlParent, definition, property);
	        var wrapperAnnotations = [];
	        var annotations = x.annotations();
	        if (annotations) {
	            var aProp = universe.type("Annotable").property("annotations");
	            for (var _i = 0, _a = Object.keys(annotations); _i < _a.length; _i++) {
	                var aName = _a[_i];
	                var aObj = annotations[aName];
	                var aJson = new json.AstNode(llParent.unit(), aObj, jsonNode, null, "(" + aName + ")");
	                var aHlNode = new hlimpl.ASTPropImpl(aJson, hlNode, aProp.range(), aProp);
	                var wAnnotation = new RamlWrapperImpl.AnnotationRefImpl(aHlNode);
	                wrapperAnnotations.push(wAnnotation);
	            }
	        }
	        return new ExampleSpecImpl(hlNode, x, wrapperAnnotations);
	    });
	}
	;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"example","primary":true}
	 */
	function getTypeExample(td) {
	    var examples = getExpandableExamples(td, true);
	    if (examples.length > 0) {
	        return examples[0];
	    }
	    return null;
	}
	exports.getTypeExample = getTypeExample;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"examples","primary":true}
	 */
	function getTypeExamples(td) {
	    return getExpandableExamples(td);
	}
	exports.getTypeExamples = getTypeExamples;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"fixedFacets","primary":true}
	 */
	function typeFixedFacets(td) {
	    var obj = td.runtimeDefinition().getFixedFacets();
	    if (Object.keys(obj).length == 0) {
	        return null;
	    }
	    var node = new json.AstNode(null, obj);
	    return new core.TypeInstanceImpl(node);
	}
	exports.typeFixedFacets = typeFixedFacets;
	/**
	 * __$helperMethod__ A base type which the current type extends, or more generally a type expression.
	 * __$meta__={"name":"type","override":true}
	 */
	function typeValue(typeDeclaration) {
	    var attrs = typeDeclaration.highLevel().attributes(defs.universesInfo.Universe10.TypeDeclaration.properties.type.name);
	    var structuredAttrs = attrs.filter(function (x) { return x.value() instanceof hlimpl.StructuredValue; });
	    if (structuredAttrs.length == 0) {
	        return typeDeclaration.type_original();
	    }
	    var nullify = false;
	    var values = attrs.map(function (x) {
	        var val = x.value();
	        if (typeof (val) == "string") {
	            return val;
	        }
	        else if (val instanceof hlimpl.StructuredValue) {
	            nullify = true;
	        }
	        return val.toString();
	    });
	    if (nullify) {
	        return null;
	    }
	    return values;
	}
	exports.typeValue = typeValue;
	/**
	 * __$helperMethod__ A base type which the current type extends, or more generally a type expression.
	 * __$meta__={"name":"schema","override":true}
	 */
	function schemaValue(typeDeclaration) {
	    var nullify = false;
	    var attrs = typeDeclaration.highLevel().attributes(defs.universesInfo.Universe10.TypeDeclaration.properties.schema.name);
	    if (nullify) {
	        return null;
	    }
	    var structuredAttrs = attrs.filter(function (x) { return x.value() instanceof hlimpl.StructuredValue; });
	    if (structuredAttrs.length == 0) {
	        return typeDeclaration.schema_original();
	    }
	    var values = attrs.map(function (x) {
	        var val = x.value();
	        if (typeof (val) == "string") {
	            return val;
	        }
	        else if (val instanceof hlimpl.StructuredValue) {
	            nullify = true;
	        }
	        return val.toString();
	    });
	    if (nullify) {
	        return null;
	    }
	    return values;
	}
	exports.schemaValue = schemaValue;
	/**
	 * __$helperMethod__ Inlined supertype definition.
	 * __$meta__={"name":"structuredType","primary":true}
	 */
	function typeStructuredValue(typeDeclaration) {
	    var attrs = typeDeclaration.highLevel().attributes(defs.universesInfo.Universe10.TypeDeclaration.properties.type.name);
	    attrs = attrs.concat(typeDeclaration.highLevel().attributes(defs.universesInfo.Universe10.TypeDeclaration.properties.schema.name));
	    var values = attrs.map(function (x) { return x.value(); });
	    for (var _i = 0, values_1 = values; _i < values_1.length; _i++) {
	        var val = values_1[_i];
	        if (val instanceof hlimpl.StructuredValue) {
	            var typeInstance = new core.TypeInstanceImpl(val.lowLevel());
	            return typeInstance;
	        }
	    }
	    return null;
	}
	exports.typeStructuredValue = typeStructuredValue;
	/**
	 * __$helperMethod__
	 * Returns the root node of the AST, uses statement refers.
	 * __$meta__={"name":"ast"}
	 */
	function referencedNode(usesDecl) {
	    var ref = usesDecl.value();
	    var unit = usesDecl.highLevel().lowLevel().unit().resolve(ref);
	    var hlNode = unit.highLevel();
	    var hlElement = hlNode.asElement();
	    if (hlElement) {
	        //we know, only libraries can be referenced from uses
	        var wrapperNode = hlElement.wrapperNode();
	        if (RamlWrapper.isLibrary(wrapperNode)) {
	            wrapperNode.setAttributeDefaults(usesDecl.attributeDefaults());
	            return wrapperNode;
	        }
	        else {
	            return null;
	        }
	    }
	    return null;
	}
	exports.referencedNode = referencedNode;
	function extractParams(params, uri, owner, propName) {
	    var ownerHl = owner.highLevel();
	    var definition = ownerHl.definition();
	    var prop = definition.property(propName);
	    if (!uri) {
	        return [];
	    }
	    var describedParams = {};
	    params.forEach(function (x) {
	        var arr = describedParams[x.name()];
	        if (!arr) {
	            arr = [];
	            describedParams[x.name()] = arr;
	        }
	        arr.push(x);
	    });
	    var allParams = [];
	    var prev = 0;
	    var mentionedParams = {};
	    for (var i = uri.indexOf('{'); i >= 0; i = uri.indexOf('{', prev)) {
	        prev = uri.indexOf('}', ++i);
	        var paramName = uri.substring(i, prev);
	        mentionedParams[paramName] = true;
	        if (describedParams[paramName]) {
	            describedParams[paramName].forEach(function (x) { return allParams.push(x); });
	        }
	        else {
	            var universe = definition.universe();
	            var nc = universe.type(universeDef.Universe10.StringTypeDeclaration.name);
	            var node = stubs.createStubNode(nc, null, paramName);
	            var uriParameter = factory.buildWrapperNode(node);
	            var hlNode = uriParameter.highLevel();
	            hlNode.setParent(ownerHl);
	            uriParameter.meta().setCalculated();
	            uriParameter.setName(paramName);
	            hlNode.patchProp(prop);
	            allParams.push(uriParameter);
	        }
	    }
	    Object.keys(describedParams).filter(function (x) { return !mentionedParams[x]; })
	        .forEach(function (x) { return describedParams[x].forEach(function (y) { return allParams.push(y); }); });
	    return allParams;
	}
	;
	/**
	 * __$helperMethod__
	 * __$meta__={"name":"parametrizedProperties","primary":true}
	 */
	function getTemplateParametrizedProperties(node) {
	    var highLevelNode = node.highLevel();
	    if (highLevelNode == null) {
	        return null;
	    }
	    var lowLevelNode = highLevelNode.lowLevel();
	    if (lowLevelNode == null) {
	        return null;
	    }
	    var children = lowLevelNode.children().filter(function (x) { return x.key().indexOf("<<") >= 0; });
	    if (children.length == 0) {
	        return null;
	    }
	    var result = new core.TypeInstanceImpl(children);
	    return result;
	}
	exports.getTemplateParametrizedProperties = getTemplateParametrizedProperties;
	var SchemaDef = (function () {
	    function SchemaDef(_content, _name) {
	        this._content = _content;
	        this._name = _name;
	    }
	    SchemaDef.prototype.name = function () { return this._name; };
	    SchemaDef.prototype.content = function () { return this._content; };
	    return SchemaDef;
	}());
	exports.SchemaDef = SchemaDef;
	var ParamValue = (function () {
	    function ParamValue(key, value) {
	        this.key = key;
	        this.value = value;
	    }
	    return ParamValue;
	}());
	exports.ParamValue = ParamValue;
	var ParamWrapper = (function () {
	    function ParamWrapper(_param) {
	        this._param = _param;
	        this.description = _param.description() ? _param.description().value() : this.description;
	        this.displayName = _param.displayName();
	        //        this.enum = _param.enum();
	        this.type = _param.type().length > 0 ? _param.type()[0] : "string";
	        this.example = _param.example();
	        this.repeat = _param.repeat();
	        this.required = _param.required();
	        this.default = _param.default();
	    }
	    return ParamWrapper;
	}());
	var ExampleSpecImpl = (function (_super) {
	    __extends(ExampleSpecImpl, _super);
	    function ExampleSpecImpl(hlNode, expandable, _annotations) {
	        _super.call(this, hlNode);
	        this.expandable = expandable;
	        this._annotations = _annotations;
	    }
	    ExampleSpecImpl.prototype.value = function () {
	        if (this.expandable.isJSONString() || this.expandable.isYAML()) {
	            return this.expandable.asJSON();
	        }
	        return this.expandable.original();
	    };
	    ExampleSpecImpl.prototype.structuredValue = function () {
	        var obj = this.value();
	        var llParent = this._node.lowLevel();
	        var key = this.expandable.isSingle() ? "example" : null;
	        var jsonNode = new json.AstNode(llParent.unit(), obj, llParent, null, key);
	        return new core.TypeInstanceImpl(jsonNode);
	    };
	    ExampleSpecImpl.prototype.strict = function () {
	        return this.expandable.strict();
	    };
	    ExampleSpecImpl.prototype.description = function () {
	        var descriptionValue = this.expandable.description();
	        if (descriptionValue == null && descriptionValue !== null) {
	            return null;
	        }
	        var attr = stubs.createAttr(this._node.definition().property(universeDef.Universe10.ExampleSpec.properties.description.name), descriptionValue);
	        var result = new RamlWrapperImpl.MarkdownStringImpl(attr);
	        return result;
	    };
	    ExampleSpecImpl.prototype.name = function () {
	        return this.expandable.name();
	    };
	    ExampleSpecImpl.prototype.displayName = function () {
	        return this.expandable.displayName();
	    };
	    ExampleSpecImpl.prototype.annotations = function () {
	        return this._annotations;
	    };
	    ExampleSpecImpl.prototype.scalarsAnnotations = function () { return {}; };
	    return ExampleSpecImpl;
	}(core.BasicNodeImpl));
	exports.ExampleSpecImpl = ExampleSpecImpl;
	//# sourceMappingURL=wrapperHelper.js.map

/***/ },
/* 33 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var serv = __webpack_require__(35);
	module.exports = serv;
	//# sourceMappingURL=ramlServices.js.map

/***/ },
/* 34 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var ramlDS = __webpack_require__(35);
	module.exports = ramlDS.getUniverse;
	//# sourceMappingURL=universeProvider.js.map

/***/ },
/* 35 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	exports.rt = __webpack_require__(77);
	var typeSystem = exports.rt.nominalTypes;
	function getSchemaUtils() {
	    return exports.rt.getSchemaUtils();
	}
	exports.getSchemaUtils = getSchemaUtils;
	function registerAdapters(a) {
	    if (a["isUnion"]) {
	        a.addAdapter(new RAMLService(a));
	    }
	    else if (a["range"]) {
	        a.addAdapter(new RAMLPropertyService(a));
	    }
	}
	exports.TOP_LEVEL_EXTRA = exports.rt.TOP_LEVEL_EXTRA;
	exports.DEFINED_IN_TYPES_EXTRA = exports.rt.DEFINED_IN_TYPES_EXTRA;
	exports.USER_DEFINED_EXTRA = exports.rt.USER_DEFINED_EXTRA;
	exports.SOURCE_EXTRA = exports.rt.SOURCE_EXTRA;
	// export function instanceOfHasExtra(instance : any) : instance is rt.IHasExtra {
	//     return rt.instanceOfHasExtra(instance);
	// }
	exports.injector = {
	    inject: function (a) {
	        registerAdapters(a);
	    }
	};
	typeSystem.registerInjector(exports.injector);
	var AbstractType = (function (_super) {
	    __extends(AbstractType, _super);
	    function AbstractType() {
	        _super.apply(this, arguments);
	    }
	    return AbstractType;
	}(typeSystem.AbstractType));
	exports.AbstractType = AbstractType;
	var ValueType = (function (_super) {
	    __extends(ValueType, _super);
	    function ValueType() {
	        _super.apply(this, arguments);
	    }
	    return ValueType;
	}(typeSystem.ValueType));
	exports.ValueType = ValueType;
	var SourceProvider = (function () {
	    function SourceProvider() {
	    }
	    SourceProvider.prototype.getSource = function () {
	        return null;
	    };
	    return SourceProvider;
	}());
	exports.SourceProvider = SourceProvider;
	var EnumType = (function (_super) {
	    __extends(EnumType, _super);
	    function EnumType() {
	        _super.apply(this, arguments);
	        this.values = [];
	    }
	    return EnumType;
	}(ValueType));
	exports.EnumType = EnumType;
	var ReferenceType = (function (_super) {
	    __extends(ReferenceType, _super);
	    function ReferenceType(name, path, referenceTo, _universe) {
	        _super.call(this, name, _universe, path);
	        this.referenceTo = referenceTo;
	    }
	    ReferenceType.prototype.getReferencedType = function () {
	        return this.universe().type(this.referenceTo);
	    };
	    ReferenceType.prototype.hasStructure = function () {
	        return true;
	    };
	    return ReferenceType;
	}(ValueType));
	exports.ReferenceType = ReferenceType;
	var NodeClass = (function (_super) {
	    __extends(NodeClass, _super);
	    function NodeClass(_name, universe, path, _description) {
	        if (_description === void 0) { _description = ""; }
	        _super.call(this, _name, universe, path);
	    }
	    NodeClass.prototype.allProperties = function (v) {
	        if (v === void 0) { v = {}; }
	        return _super.prototype.allProperties.call(this, v);
	    };
	    return NodeClass;
	}(typeSystem.StructuredType));
	exports.NodeClass = NodeClass;
	var UserDefinedClass = (function (_super) {
	    __extends(UserDefinedClass, _super);
	    function UserDefinedClass(name, universe, hl, path, description) {
	        _super.call(this, name, universe, path, description);
	        this.getAdapter(RAMLService).setDeclaringNode(hl);
	    }
	    UserDefinedClass.prototype.key = function () {
	        return null;
	    };
	    UserDefinedClass.prototype.isUserDefined = function () {
	        return true;
	    };
	    UserDefinedClass.prototype.typeId = function () {
	        var rs = this.nameId();
	        var node = this.getAdapter(RAMLService).getDeclaringNode();
	        if (node) {
	            rs = rs + node.lowLevel().start();
	            var unit = node.lowLevel().unit();
	            if (unit) {
	                rs = rs + unit.absolutePath();
	            }
	        }
	        return rs;
	    };
	    UserDefinedClass.prototype.hasValueTypeInHierarchy = function () {
	        if (this._value) {
	            return true;
	        }
	        if (this.isAssignableFrom("ObjectTypeDeclaration")) {
	            return false;
	        }
	        return false;
	    };
	    /**
	     * Returns whether this type contain genuine user defined type in its hierarchy.
	     * Genuine user defined type is a type user intentionally defined and filled with
	     * properties or facets, or having user-defined name as opposed to a synthetic user-defined type.
	     */
	    UserDefinedClass.prototype.isGenuineUserDefinedType = function () {
	        return this.genuineUserDefinedType() != null;
	    };
	    /**
	     * Returns nearest genuine user-define type in the hierarchy.
	     * Genuine user defined type is a type user intentionally defined and filled with
	     * properties or facets, or having user-defined name as opposed to a synthetic user-defined type.
	     */
	    UserDefinedClass.prototype.genuineUserDefinedType = function () {
	        var declaringNode = this.getAdapter(RAMLService).getDeclaringNode();
	        if (!declaringNode) {
	            return null;
	        }
	        return this;
	    };
	    return UserDefinedClass;
	}(NodeClass));
	exports.UserDefinedClass = UserDefinedClass;
	var AnnotationType = (function (_super) {
	    __extends(AnnotationType, _super);
	    function AnnotationType() {
	        _super.apply(this, arguments);
	    }
	    AnnotationType.prototype.allProperties = function (ps) {
	        if (ps === void 0) { ps = {}; }
	        var rs = this.superTypes()[0].allProperties();
	        if (rs.length == 0) {
	            var up = new UserDefinedProp("value");
	            up.withDomain(this);
	            up._node = this.getAdapter(RAMLService).getDeclaringNode();
	            up.withCanBeValue();
	            up.withRequired(false);
	            var tp = this.superTypes()[0];
	            rs = [];
	            up.withRange(up._node.asElement().definition().universe().type("string"));
	            rs.push(up);
	        }
	        return rs;
	    };
	    AnnotationType.prototype.isAnnotationType = function () {
	        return true;
	    };
	    return AnnotationType;
	}(UserDefinedClass));
	exports.AnnotationType = AnnotationType;
	var Universe = (function (_super) {
	    __extends(Universe, _super);
	    function Universe(dec, name, _parent, v) {
	        if (name === void 0) { name = ""; }
	        if (_parent === void 0) { _parent = null; }
	        if (v === void 0) { v = "RAML08"; }
	        _super.call(this, name);
	        this._parent = _parent;
	        this._classes = [];
	        this._uversion = "RAML08";
	        this.aMap = {};
	        this.matchedObjects = dec;
	        this._uversion = v;
	    }
	    Universe.prototype.matched = function () {
	        if (this.matchedObjects) {
	            return this.matchedObjects;
	        }
	        if (this._parent) {
	            return this._parent.matched();
	        }
	    };
	    Universe.prototype.setTopLevel = function (t) {
	        this._topLevel = t;
	    };
	    Universe.prototype.getTopLevel = function () {
	        return this._topLevel;
	    };
	    Universe.prototype.setTypedVersion = function (tv) {
	        this._typedVersion = tv;
	    };
	    Universe.prototype.getTypedVersion = function () {
	        return this._typedVersion;
	    };
	    Universe.prototype.version = function () {
	        return this._uversion;
	    };
	    Universe.prototype.setUniverseVersion = function (version) {
	        this._uversion = version;
	    };
	    Universe.prototype.types = function () {
	        var result = [].concat(this._classes);
	        if (this._parent != null) {
	            result = result.concat(this._parent.types());
	        }
	        return result;
	    };
	    Universe.prototype.type = function (name) {
	        if (this.aMap[name]) {
	            return this.aMap[name];
	        }
	        var tp;
	        for (var i = 0; i < this._classes.length; i++) {
	            if (this._classes[i].nameId() == name) {
	                tp = this._classes[i];
	                break;
	            }
	        }
	        if (tp == null) {
	            if (this._parent) {
	                var tp = this._parent.type(name);
	                if (tp instanceof AbstractType) {
	                    var at = tp;
	                    at._universe = this; //FIXME
	                }
	            }
	        }
	        return tp;
	    };
	    Universe.prototype.register = function (t) {
	        this._classes.push(t);
	        if (t instanceof NodeClass) {
	            this._classes.forEach(function (x) {
	                if (x instanceof NodeClass) {
	                    var nc = x;
	                    if (nc.getAdapter(RAMLService).getExtendedType() == t) {
	                        t.getAdapter(RAMLService).getRuntimeExtenders().push(x);
	                    }
	                }
	            });
	        }
	        return this;
	    };
	    Universe.prototype.registerAlias = function (a, t) {
	        this.aMap[a] = t;
	    };
	    Universe.prototype.unregister = function (t) {
	        this._classes = this._classes.filter(function (x) { return x != t; });
	        var st = t.superTypes();
	        st.forEach(function (x) {
	            var a = x;
	            a._superTypes = a._superTypes.filter(function (x) { return x != t; });
	        });
	        st = t.subTypes();
	        st.forEach(function (x) {
	            var a = x;
	            a._subTypes = a._subTypes.filter(function (x) { return x != t; });
	        });
	        return this;
	    };
	    Universe.prototype.registerSuperClass = function (t0, t1) {
	        var a0 = t0;
	        var a1 = t1;
	        a0._superTypes.push(t1);
	        a1._subTypes.push(t0);
	    };
	    return Universe;
	}(typeSystem.Described));
	exports.Universe = Universe;
	function prop(name, desc, domain, range, custom) {
	    var prop = new Property(name, desc);
	    return prop.withDomain(domain, custom).withRange(range);
	}
	exports.prop = prop;
	var ChildValueConstraint = (function () {
	    function ChildValueConstraint(name, value) {
	        this.name = name;
	        this.value = value;
	    }
	    return ChildValueConstraint;
	}());
	exports.ChildValueConstraint = ChildValueConstraint;
	var Property = (function (_super) {
	    __extends(Property, _super);
	    function Property() {
	        _super.apply(this, arguments);
	        this._isFromParentValue = false;
	        this._isFromParentKey = false;
	        this._key = false;
	        this._describes = null;
	        this._selfNode = false;
	        this._noDirectParse = false;
	        this._contextReq = [];
	        this.determinesChildValues = [];
	    }
	    Property.prototype.isPrimitive = function () {
	        return this.range().isValueType() && !(this.range() instanceof ReferenceType);
	    };
	    Property.prototype.withNoDirectParse = function () {
	        this._noDirectParse = true;
	    };
	    Property.prototype.isNoDirectParse = function () {
	        return this._noDirectParse;
	    };
	    Property.prototype.withSelfNode = function () {
	        this._selfNode = true;
	    };
	    Property.prototype.isSelfNode = function () {
	        return this._selfNode;
	    };
	    Property.prototype.matchKey = function (k) {
	        if (this._groupName != null) {
	            if (this.isAnnotation()) {
	                if (k.charAt(0) == '(' && k.charAt(k.length - 1) == ')') {
	                    return true;
	                }
	                return false;
	            }
	            return this._groupName == k;
	        }
	        else {
	            return _super.prototype.matchKey.call(this, k);
	        }
	    };
	    Property.prototype.valueDocProvider = function () {
	        return this._vprovider;
	    };
	    Property.prototype.setValueDocProvider = function (v) {
	        this._vprovider = v;
	        return this;
	    };
	    Property.prototype.suggester = function () {
	        return this._suggester;
	    };
	    Property.prototype.setValueSuggester = function (s) {
	        this._suggester = s;
	    };
	    Property.prototype.enumOptions = function () {
	        if (this._enumOptions && typeof this._enumOptions == 'string') {
	            return [this._enumOptions + ""];
	        }
	        return this._enumOptions;
	    };
	    Property.prototype.getOftenKeys = function () {
	        return this._oftenKeys;
	    };
	    Property.prototype.withOftenKeys = function (keys) {
	        this._oftenKeys = keys;
	        return this;
	    };
	    Property.prototype.withCanBeValue = function () {
	        this._canBeValue = true;
	        return this;
	    };
	    Property.prototype.withInherited = function (w) {
	        this._isInherited = w;
	    };
	    Property.prototype.isInherited = function () {
	        return this._isInherited;
	    };
	    Property.prototype.isAllowNull = function () {
	        return this._allowsNull;
	    };
	    Property.prototype.withAllowNull = function () {
	        this._allowsNull = true;
	    };
	    Property.prototype.getCanBeDuplicator = function () {
	        return this._canBeDuplicator;
	    };
	    Property.prototype.canBeValue = function () {
	        return this._canBeValue;
	    };
	    Property.prototype.setCanBeDuplicator = function () {
	        this._canBeDuplicator = true;
	        return true;
	    };
	    Property.prototype.inheritedContextValue = function () {
	        return this._inheritsValueFromContext;
	    };
	    Property.prototype.withInheritedContextValue = function (v) {
	        this._inheritsValueFromContext = v;
	        return this;
	    };
	    Property.prototype.withContextRequirement = function (name, value) {
	        this._contextReq.push({ name: name, value: value });
	    };
	    Property.prototype.getContextRequirements = function () {
	        return this._contextReq;
	    };
	    Property.prototype.withDescribes = function (a) {
	        this._describes = a;
	        return this;
	    };
	    Property.prototype.describesAnnotation = function () {
	        return this._describes != null;
	    };
	    Property.prototype.describedAnnotation = function () {
	        return this._describes;
	    };
	    Property.prototype.isReference = function () {
	        return this.range() instanceof ReferenceType;
	    };
	    Property.prototype.referencesTo = function () {
	        return this.range().getReferencedType();
	    };
	    Property.prototype.newInstanceName = function () {
	        if (this._newInstanceName) {
	            return this._newInstanceName;
	        }
	        return this.range().nameId();
	    };
	    Property.prototype.withThisPropertyDeclaresFields = function (b) {
	        if (b === void 0) { b = true; }
	        this._declaresFields = b;
	        return this;
	    };
	    Property.prototype.isThisPropertyDeclaresTypeFields = function () {
	        return this._declaresFields;
	    };
	    Property.prototype.withNewInstanceName = function (name) {
	        this._newInstanceName = name;
	        return this;
	    };
	    Property.prototype.addChildValueConstraint = function (c) {
	        this.determinesChildValues.push(c);
	    };
	    Property.prototype.getChildValueConstraints = function () {
	        return this.determinesChildValues;
	    };
	    Property.prototype.childRestrictions = function () {
	        return this.determinesChildValues;
	    };
	    Property.prototype.id = function () {
	        if (this._id) {
	            return this._id;
	        }
	        if (!this._groupName) {
	            return null;
	        }
	        if (this.domain().getAdapter(RAMLService).getDeclaringNode()) {
	            return null;
	        }
	        this._id = this._groupName + this.domain().nameId();
	        return this._id;
	    };
	    Property.prototype.isAnnotation = function () {
	        return this._groupName == "annotations" &&
	            this.domain() && !this.domain().isUserDefined();
	    };
	    Property.prototype.withFromParentValue = function (v) {
	        if (v === void 0) { v = true; }
	        this._isFromParentValue = v;
	        return this;
	    };
	    Property.prototype.withFromParentKey = function (v) {
	        if (v === void 0) { v = true; }
	        this._isFromParentKey = v;
	        return this;
	    };
	    Property.prototype.isFromParentKey = function () {
	        return this._isFromParentKey;
	    };
	    Property.prototype.isFromParentValue = function () {
	        return this._isFromParentValue;
	    };
	    Property.prototype.withGroupName = function (gname) {
	        this._groupName = gname;
	        return this;
	    };
	    Property.prototype.unmerge = function () {
	        this._groupName = this.nameId();
	        return this;
	    };
	    Property.prototype.merge = function () {
	        this._groupName = null;
	        return this;
	    };
	    Property.prototype.withKey = function (isKey) {
	        this._key = isKey;
	        return this;
	    };
	    /**
	     * TODO THIS STUFF SHOULD BE MORE ABSTRACT (LATER...)
	     * @param keyShouldStartFrom
	     * @returns {Property}
	     */
	    Property.prototype.isKey = function () {
	        return this._key;
	    };
	    Property.prototype.isMerged = function () {
	        return this._groupName == null;
	    };
	    Property.prototype.groupName = function () {
	        return this._groupName;
	    };
	    Property.prototype.key = function () {
	        //TODO implement that
	        return null;
	    };
	    return Property;
	}(typeSystem.Property));
	exports.Property = Property;
	var UserDefinedProp = (function (_super) {
	    __extends(UserDefinedProp, _super);
	    function UserDefinedProp() {
	        _super.apply(this, arguments);
	    }
	    UserDefinedProp.prototype.withDisplayName = function (name) {
	        this._displayName = name;
	    };
	    UserDefinedProp.prototype.getDisplayName = function () {
	        return this._displayName;
	    };
	    UserDefinedProp.prototype.node = function () {
	        return this._node;
	    };
	    return UserDefinedProp;
	}(Property));
	exports.UserDefinedProp = UserDefinedProp;
	var RAMLPropertyDocumentationService = (function () {
	    function RAMLPropertyDocumentationService() {
	    }
	    RAMLPropertyDocumentationService.prototype.setDocTableName = function (val) {
	        this._documentationTableName = val;
	    };
	    RAMLPropertyDocumentationService.prototype.docTableName = function () {
	        return this._documentationTableName;
	    };
	    RAMLPropertyDocumentationService.prototype.setHidden = function (val) {
	        this._isHidden = val;
	    };
	    RAMLPropertyDocumentationService.prototype.isHidden = function () {
	        return this._isHidden;
	    };
	    RAMLPropertyDocumentationService.prototype.setMarkdownDescription = function (val) {
	        this._markdownDescription = val;
	    };
	    RAMLPropertyDocumentationService.prototype.markdownDescription = function () {
	        return this._markdownDescription;
	    };
	    RAMLPropertyDocumentationService.prototype.setValueDescription = function (val) {
	        this._valueDescription = val;
	    };
	    RAMLPropertyDocumentationService.prototype.valueDescription = function () {
	        return this._valueDescription;
	    };
	    return RAMLPropertyDocumentationService;
	}());
	exports.RAMLPropertyDocumentationService = RAMLPropertyDocumentationService;
	var RAMLPropertyParserService = (function (_super) {
	    __extends(RAMLPropertyParserService, _super);
	    function RAMLPropertyParserService() {
	        _super.apply(this, arguments);
	    }
	    RAMLPropertyParserService.prototype.isSystem = function () {
	        return this._isSystemProperty;
	    };
	    RAMLPropertyParserService.prototype.withSystem = function (s) {
	        this._isSystemProperty = s;
	        return this;
	    };
	    RAMLPropertyParserService.prototype.isEmbedMap = function () {
	        return this._isEmbededMap;
	    };
	    RAMLPropertyParserService.prototype.withEmbedMap = function () {
	        this._isEmbededMap = true;
	        return this;
	    };
	    return RAMLPropertyParserService;
	}(RAMLPropertyDocumentationService));
	exports.RAMLPropertyParserService = RAMLPropertyParserService;
	var RAMLPropertyService = (function (_super) {
	    __extends(RAMLPropertyService, _super);
	    function RAMLPropertyService(_property) {
	        _super.call(this);
	        this._property = _property;
	        this._meta = {};
	        if (!_property) {
	            throw new Error();
	        }
	        _super.call(this);
	    }
	    RAMLPropertyService.prototype.valueDocProvider = function () {
	        return this._property.valueDocProvider();
	    };
	    RAMLPropertyService.prototype.withPropertyGrammarType = function (pt) {
	        this._propertyGrammarType = pt;
	    };
	    RAMLPropertyService.prototype.getPropertyGrammarType = function () {
	        return this._propertyGrammarType;
	    };
	    RAMLPropertyService.prototype.id = function () {
	        return this._property.nameId();
	    };
	    RAMLPropertyService.prototype.range = function () {
	        return this._property.range();
	    };
	    RAMLPropertyService.prototype.domain = function () {
	        return this._property.domain();
	    };
	    RAMLPropertyService.prototype.isAllowNull = function () {
	        return this._property.isAllowNull();
	    };
	    RAMLPropertyService.prototype.referencesTo = function () {
	        return this._property.referencesTo();
	    };
	    RAMLPropertyService.prototype.isReference = function () {
	        return this._property.isReference();
	    };
	    RAMLPropertyService.prototype.nameId = function () {
	        return this._property.nameId();
	    };
	    RAMLPropertyService.prototype.priority = function () {
	        if (this.isKey())
	            return 128;
	        else if (this.isReference())
	            return 64;
	        else if (this.isTypeExpr())
	            return 32;
	        else if (this.nameId() == 'example')
	            return 0;
	        else
	            return -1024;
	    };
	    RAMLPropertyService.prototype.isKey = function () {
	        if (this._property instanceof Property) {
	            return this._property.isKey();
	        }
	        return false;
	    };
	    RAMLPropertyService.prototype.isMerged = function () {
	        if (this._property instanceof Property) {
	            return this._property.isMerged();
	        }
	        return false;
	    };
	    RAMLPropertyService.prototype.isTypeExpr = function () {
	        return this.texpr;
	    };
	    RAMLPropertyService.prototype.isExampleProperty = function () {
	        return this.example;
	    };
	    RAMLPropertyService.prototype.setExample = function (e) {
	        this.example = e;
	    };
	    RAMLPropertyService.prototype.setTypeExpression = function (e) {
	        this.texpr = e;
	    };
	    RAMLPropertyService.prototype.isDescriminating = function () {
	        return this._property.isDescriminator();
	    };
	    RAMLPropertyService.prototype.putMeta = function (key, value) {
	        this._meta[key] = value;
	    };
	    RAMLPropertyService.prototype.meta = function (key) {
	        return this._meta[key];
	    };
	    return RAMLPropertyService;
	}(RAMLPropertyParserService));
	exports.RAMLPropertyService = RAMLPropertyService;
	var RAMLService = (function () {
	    function RAMLService(d) {
	        //!!!
	        this._allowsOptionalProperties = false;
	        this._possibleInterfaces = [];
	        //!!!
	        this._canInherit = [];
	        this._declaredBy = [];
	        //!!!
	        this._aliases = [];
	        //!!!???
	        this._defining = [];
	        this._runtimeExtenders = [];
	        //!!!???
	        this._declaresType = null;
	        //!!!
	        this._isTemplate = false;
	        //!!!
	        this._contextRequirements = [];
	        this._type = d;
	    }
	    RAMLService.prototype.withAllowQuestion = function () {
	        this._allowsOptionalProperties = true;
	    };
	    RAMLService.prototype.getAllowQuestion = function () {
	        return this._allowsOptionalProperties;
	    };
	    RAMLService.prototype.withCanInherit = function (clazz) {
	        this._canInherit.push(clazz);
	    };
	    RAMLService.prototype.getReferenceIs = function () {
	        return this._referenceIs;
	    };
	    RAMLService.prototype.withReferenceIs = function (fname) {
	        this._referenceIs = fname;
	    };
	    RAMLService.prototype.descriminatorValue = function () {
	        if (this.valueRequirements().length == 0) {
	            return this.nameId();
	        }
	        return this.valueRequirements()[0].value;
	    };
	    RAMLService.prototype.getCanInherit = function () {
	        return this._canInherit;
	    };
	    RAMLService.prototype.withAllowAny = function () {
	        this._allowsAnyChildren = true;
	    };
	    RAMLService.prototype.getAllowAny = function () {
	        return this._allowsAnyChildren || this._type.isExternal();
	    };
	    RAMLService.prototype.globallyDeclaredBy = function () {
	        return this._declaredBy;
	    };
	    RAMLService.prototype.setGloballyDeclaredBy = function (c) {
	        this._declaredBy.push(c);
	    };
	    RAMLService.prototype.setDeclaringNode = function (n) {
	        this._node = n;
	    };
	    RAMLService.prototype.nameId = function () {
	        return this._type.nameId();
	    };
	    RAMLService.prototype.universe = function () {
	        return this._type.universe();
	    };
	    RAMLService.prototype.isAssignableFrom = function (name) {
	        return this._type.isAssignableFrom(name);
	    };
	    RAMLService.prototype.setConsumesRefs = function (b) {
	        this._consumesRef = b;
	    };
	    RAMLService.prototype.definingPropertyIsEnough = function (v) {
	        this._defining.push(v);
	    };
	    RAMLService.prototype.getDefining = function () {
	        return this._defining;
	    };
	    RAMLService.prototype.getConsumesRefs = function () {
	        return this._consumesRef;
	    };
	    RAMLService.prototype.addAlias = function (al) {
	        this._aliases.push(al);
	    };
	    RAMLService.prototype.getAliases = function () {
	        return this._aliases;
	    };
	    RAMLService.prototype.valueRequirements = function () {
	        return this._type.valueRequirements();
	    };
	    RAMLService.prototype.isAnnotation = function () {
	        if (this._annotationChecked) {
	            return this._isAnnotation;
	        }
	        this._annotationChecked = true;
	        this._isAnnotation = (this._type.allSuperTypes().some(function (x) { return x.key() && x.key().name == "AnnotationRef"; }));
	        return this._isAnnotation;
	    };
	    RAMLService.prototype.allowValue = function () {
	        if (this._allowsValueSet) {
	            return this._allowsValue;
	        }
	        if (this._type.allProperties().some(function (x) { return x.isFromParentValue() || x.canBeValue(); })) {
	            this._allowsValue = true;
	            this._allowsValueSet = true;
	            return true;
	        }
	        this._allowsValueSet = true;
	        return false;
	    };
	    RAMLService.prototype.key = function () {
	        return this._type.key();
	    };
	    RAMLService.prototype.getRepresentationOf = function () {
	        return this._representationOf;
	    };
	    RAMLService.prototype.getPath = function () {
	        return "";
	    };
	    RAMLService.prototype.isDeclaration = function () {
	        if (this._isTemplate) {
	            return true;
	        }
	        if (this._convertsToGlobal) {
	            return true;
	        }
	        if (this._declaresType) {
	            return true;
	        }
	        if (this.key() && this.key().name === "Library") {
	            return true;
	        }
	        return false;
	    };
	    RAMLService.prototype.isGlobalDeclaration = function () {
	        if (this._actuallyExports) {
	            return true;
	        }
	        if (this._isTemplate) {
	            return true;
	        }
	        if (this._declaresType) {
	            return true;
	        }
	        return false;
	    };
	    RAMLService.prototype.isTypeSystemMember = function () {
	        return this._declaresType != null;
	    };
	    RAMLService.prototype.getExtendedType = function () {
	        return this.universe().type(this._declaresType);
	    };
	    RAMLService.prototype.setInlinedTemplates = function (b) {
	        this._isTemplate = b;
	        return this;
	    };
	    RAMLService.prototype.getRuntimeExtenders = function () {
	        return this._runtimeExtenders;
	    };
	    RAMLService.prototype.isInlinedTemplates = function () {
	        return this._isTemplate;
	    };
	    RAMLService.prototype.setExtendedTypeName = function (name) {
	        this._declaresType = name;
	        var tp = this.universe().type(name);
	        if (tp instanceof NodeClass) {
	            var nc = tp;
	            nc.getAdapter(RAMLService)._runtimeExtenders.push(this._type);
	        }
	    };
	    RAMLService.prototype.getKeyProp = function () {
	        var c = this._type.allProperties();
	        for (var i = 0; i < c.length; i++) {
	            if (c[i].getAdapter(RAMLPropertyService).isKey()) {
	                return c[i];
	            }
	        }
	        return null;
	    };
	    RAMLService.prototype.withActuallyExports = function (pname) {
	        this._actuallyExports = pname;
	    };
	    RAMLService.prototype.withConvertsToGlobal = function (pname) {
	        this._convertsToGlobal = pname;
	    };
	    RAMLService.prototype.getConvertsToGlobal = function () {
	        return this._convertsToGlobal;
	    };
	    RAMLService.prototype.getActuallyExports = function () {
	        return this._actuallyExports;
	    };
	    RAMLService.prototype.isUserDefined = function () {
	        if (this._type instanceof NodeClass) {
	            return this._type.isUserDefined();
	        }
	        if (this._type instanceof AbstractType) {
	            return this._type.isUserDefined();
	        }
	        return false;
	    };
	    RAMLService.prototype.withContextRequirement = function (name, value) {
	        this._contextRequirements.push({ name: name, value: value });
	    };
	    RAMLService.prototype.getContextRequirements = function () {
	        return this._contextRequirements;
	    };
	    RAMLService.prototype.findMembersDeterminer = function () {
	        var c = this._type.allProperties();
	        for (var i = 0; i < c.length; i++) {
	            if (c[i].isThisPropertyDeclaresTypeFields()) {
	                return c[i];
	            }
	        }
	        return null;
	    };
	    RAMLService.prototype.getDeclaringNode = function () {
	        if (this._node)
	            return this._node;
	        else if (this._type) {
	            //if this service have no source registered,
	            //checking if the type has source contributed via extras
	            var sourceNode = this._type.getExtra(exports.SOURCE_EXTRA);
	            if (!sourceNode)
	                return null;
	            if (sourceNode instanceof SourceProvider ||
	                (sourceNode.getSource && typeof (sourceNode.getSource) == "function")) {
	                return sourceNode.getSource();
	            }
	            else {
	                return sourceNode;
	            }
	        }
	        return null;
	    };
	    RAMLService.prototype.registerSupertypes = function (classNames) {
	        var _this = this;
	        var universe = this._type.universe();
	        classNames.forEach(function (x) {
	            var supertype = universe.type(x);
	            if (!supertype) {
	                return;
	            }
	            _this._type.addSuperType(supertype);
	        });
	    };
	    RAMLService.prototype.registerPossibleInterfaces = function (classNames) {
	        var universe = this._type.universe();
	        for (var _i = 0, classNames_1 = classNames; _i < classNames_1.length; _i++) {
	            var x = classNames_1[_i];
	            var supertype = universe.type(x);
	            if (!supertype) {
	                return;
	            }
	            this._possibleInterfaces.push(supertype);
	        }
	    };
	    RAMLService.prototype.possibleInterfaces = function () {
	        return this._possibleInterfaces;
	    };
	    return RAMLService;
	}());
	exports.RAMLService = RAMLService;
	var universes = {};
	var ramlDS = __webpack_require__(52);
	var jsonDefinitions = ramlDS.universeDumps;
	var ts2def = __webpack_require__(53);
	var unDesc = __webpack_require__(54);
	exports.universesInfo = unDesc;
	exports.getUniverse = (function () {
	    var x = function (key) {
	        if (universes[key]) {
	            return universes[key];
	        }
	        var src = jsonDefinitions[key];
	        var universe = ts2def.toDefSystem(src, (key == "RAML08") ? unDesc["Universe08"] : unDesc["Universe10"]);
	        if (universe) {
	            universe.setUniverseVersion(key);
	            universes[key] = universe;
	        }
	        return universe;
	    };
	    x.availableUniverses = function () { return Object.keys(jsonDefinitions); };
	    return x;
	})();
	//# sourceMappingURL=definitionSystem.js.map

/***/ },
/* 36 */
/***/ function(module, exports, __webpack_require__) {

	module.exports = __WEBPACK_EXTERNAL_MODULE_36__;

/***/ },
/* 37 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var yaml = __webpack_require__(72);
	var json = __webpack_require__(48);
	var stringify = __webpack_require__(63);
	var impl = __webpack_require__(23);
	var universes = __webpack_require__(22);
	var def = __webpack_require__(35);
	var _ = __webpack_require__(61);
	var LowLevelProxyNode = (function () {
	    function LowLevelProxyNode(_parent, _transformer, ramlVersion) {
	        this._parent = _parent;
	        this._transformer = _transformer;
	        this.ramlVersion = ramlVersion;
	    }
	    LowLevelProxyNode.prototype.keyKind = function () {
	        return this._originalNode.keyKind();
	    };
	    LowLevelProxyNode.prototype.primaryNode = function () {
	        return null;
	    };
	    LowLevelProxyNode.prototype.isAnnotatedScalar = function () {
	        return this._originalNode.isAnnotatedScalar();
	    };
	    LowLevelProxyNode.prototype.actual = function () {
	        if (this._originalNode) {
	            return this._originalNode.actual();
	        }
	        return this;
	    };
	    LowLevelProxyNode.prototype.transformer = function () { return this._transformer; };
	    LowLevelProxyNode.prototype.originalNode = function () {
	        return this._originalNode;
	    };
	    LowLevelProxyNode.prototype.start = function () { return this._originalNode.start(); };
	    LowLevelProxyNode.prototype.end = function () { return this._originalNode.end(); };
	    LowLevelProxyNode.prototype.value = function (toString) {
	        throw new Error('The method must be overridden');
	    };
	    LowLevelProxyNode.prototype.includeErrors = function () { return this._originalNode.includeErrors(); };
	    LowLevelProxyNode.prototype.includePath = function () { return this._originalNode.includePath(); };
	    LowLevelProxyNode.prototype.includeReference = function () { return this._originalNode.includeReference(); };
	    LowLevelProxyNode.prototype.setKeyOverride = function (_key) {
	        this._keyOverride = _key;
	    };
	    LowLevelProxyNode.prototype.key = function () {
	        if (this._keyOverride) {
	            return this._keyOverride;
	        }
	        return this._originalNode.key();
	    };
	    LowLevelProxyNode.prototype.optional = function () {
	        return this.originalNode().optional();
	    };
	    LowLevelProxyNode.prototype.children = function () {
	        throw new Error('The method must be overridden');
	    };
	    LowLevelProxyNode.prototype.parent = function () { return this._parent; };
	    LowLevelProxyNode.prototype.unit = function () { return this._originalNode.unit(); };
	    LowLevelProxyNode.prototype.anchorId = function () { return this._originalNode.anchorId(); };
	    LowLevelProxyNode.prototype.errors = function () { return this._originalNode.errors(); };
	    LowLevelProxyNode.prototype.anchoredFrom = function () { return this._originalNode.anchoredFrom(); };
	    LowLevelProxyNode.prototype.includedFrom = function () { return this._originalNode.includedFrom(); };
	    LowLevelProxyNode.prototype.visit = function (v) {
	        if (v(this)) {
	            this.children().forEach(function (x) { return x.visit(v); });
	        }
	    };
	    //TODO
	    LowLevelProxyNode.prototype.addChild = function (n) { };
	    //TODO
	    LowLevelProxyNode.prototype.execute = function (cmd) { };
	    //TODO
	    LowLevelProxyNode.prototype.dump = function () { return null; };
	    LowLevelProxyNode.prototype.dumpToObject = function () {
	        var serialized = json.serialize(this);
	        if (this.kind() == yaml.Kind.MAPPING) {
	            var obj = {};
	            obj[this.key()] = serialized;
	            return obj;
	        }
	        return serialized;
	    };
	    LowLevelProxyNode.prototype.keyStart = function () { return this._originalNode.keyStart(); };
	    LowLevelProxyNode.prototype.keyEnd = function () { return this._originalNode.keyEnd(); };
	    LowLevelProxyNode.prototype.valueStart = function () { return this._originalNode.valueStart(); };
	    LowLevelProxyNode.prototype.valueEnd = function () { return this._originalNode.valueEnd(); };
	    LowLevelProxyNode.prototype.isValueLocal = function () { return this._originalNode.isValueLocal(); };
	    LowLevelProxyNode.prototype.kind = function () { return this._originalNode.kind(); };
	    LowLevelProxyNode.prototype.valueKind = function () { return this._originalNode.valueKind(); };
	    LowLevelProxyNode.prototype.show = function (msg) { this._originalNode.show(msg); };
	    LowLevelProxyNode.prototype.setHighLevelParseResult = function (highLevelParseResult) {
	        this._highLevelParseResult = highLevelParseResult;
	    };
	    LowLevelProxyNode.prototype.highLevelParseResult = function () {
	        return this._highLevelParseResult;
	    };
	    LowLevelProxyNode.prototype.setHighLevelNode = function (highLevel) {
	        this._highLevelNode = highLevel;
	    };
	    LowLevelProxyNode.prototype.highLevelNode = function () {
	        if (this._highLevelNode) {
	            return this._highLevelNode;
	        }
	        return this._originalNode.highLevelNode();
	    };
	    LowLevelProxyNode.prototype.text = function (unitText) {
	        throw new Error("not implemented");
	    };
	    LowLevelProxyNode.prototype.copy = function () {
	        throw new Error("not implemented");
	    };
	    LowLevelProxyNode.prototype.markup = function (json) {
	        throw new Error("not implemented");
	    };
	    LowLevelProxyNode.prototype.nodeDefinition = function () {
	        return impl.getDefinitionForLowLevelNode(this);
	    };
	    LowLevelProxyNode.prototype.includesContents = function () {
	        return this._originalNode.includesContents();
	    };
	    return LowLevelProxyNode;
	}());
	exports.LowLevelProxyNode = LowLevelProxyNode;
	var LowLevelCompositeNode = (function (_super) {
	    __extends(LowLevelCompositeNode, _super);
	    function LowLevelCompositeNode(node, parent, transformer, ramlVersion, isPrimary) {
	        if (isPrimary === void 0) { isPrimary = true; }
	        _super.call(this, parent, transformer, ramlVersion);
	        this.isPrimary = isPrimary;
	        //Colliding nodes of the initioal AST
	        this._adoptedNodes = [];
	        var originalParent = this.parent() ? this.parent().originalNode() : null;
	        this._originalNode = new LowLevelValueTransformingNode(node, originalParent, transformer, this.ramlVersion);
	        this._adoptedNodes.push(this._originalNode);
	    }
	    LowLevelCompositeNode.prototype.adoptedNodes = function () {
	        return this._adoptedNodes;
	    };
	    LowLevelCompositeNode.prototype.primaryNode = function () {
	        return this.isPrimary ? this._originalNode : null;
	    };
	    LowLevelCompositeNode.prototype.parent = function () { return this._parent; };
	    LowLevelCompositeNode.prototype.adopt = function (node, transformer) {
	        if (!transformer) {
	            transformer = this._transformer;
	        }
	        var originalParent = this.parent() ? this.parent().originalNode() : null;
	        var tNode = new LowLevelValueTransformingNode(node, originalParent, transformer, this.ramlVersion);
	        this._adoptedNodes.push(tNode);
	        if (this._children) {
	            this._children.forEach(function (x) { return x._parent = null; });
	        }
	        this._children = null;
	        if (this.highLevelNode()) {
	            this.highLevelNode().resetChildren();
	        }
	    };
	    LowLevelCompositeNode.prototype.value = function (toString) {
	        var valuableNodes = this._adoptedNodes.filter(function (x) { return x.value() != null; });
	        if (valuableNodes.length > 0) {
	            return valuableNodes[0].value(toString);
	        }
	        return this._originalNode.value(toString);
	    };
	    LowLevelCompositeNode.prototype.children = function () {
	        var _this = this;
	        if (this._children) {
	            return this._children;
	        }
	        var result = [];
	        var canBeMap = false;
	        var canBeSeq = false;
	        this._adoptedNodes.forEach(function (x) {
	            if (x.children() && x.children().length > 0) {
	                canBeSeq = true;
	                if (x.children()[0].key()) {
	                    canBeMap = true;
	                }
	            }
	        });
	        if (canBeMap) {
	            result = this.collectChildrenWithKeys();
	        }
	        else if (canBeSeq) {
	            result = this.collectChildrenWithKeys();
	            var map = {};
	            this._adoptedNodes.forEach(function (x) { return x.children().filter(function (y) { return !y.key(); }).forEach(function (y) {
	                var isPrimary = x == _this.primaryNode();
	                var key = _this.buildKey(y);
	                if (!isPrimary && map[key]) {
	                    //filtering away values with repeating keys
	                    //primary node is not subjected to filtration
	                    return;
	                }
	                map[key] = true;
	                var transformer = x.transformer() ? x.transformer() : _this.transformer();
	                var ch = (y instanceof LowLevelValueTransformingNode)
	                    ? y.originalNode() : y;
	                result.push(new LowLevelCompositeNode(ch, _this, transformer, _this.ramlVersion, isPrimary));
	            }); });
	        }
	        else {
	            result = [];
	        }
	        this._children = result;
	        return result;
	    };
	    LowLevelCompositeNode.prototype.buildKey = function (y) {
	        var obj = json.serialize(y);
	        var def = this.nodeDefinition();
	        if (def && (def.key() == universes.Universe08.TraitRef || def.key() == universes.Universe08.ResourceTypeRef
	            || def.key() == universes.Universe10.TraitRef || def.key() == universes.Universe10.ResourceTypeRef)) {
	            if (typeof obj == 'object') {
	                var keys = Object.keys(obj);
	                if (keys.length > 0) {
	                    obj = keys[0];
	                }
	            }
	        }
	        return stringify(obj);
	    };
	    LowLevelCompositeNode.prototype.collectChildrenWithKeys = function () {
	        var _this = this;
	        var result = [];
	        var m = {};
	        this._adoptedNodes.forEach(function (x) {
	            var isPrimary = x == _this.primaryNode();
	            x.originalNode().children().forEach(function (y) {
	                var key = y.key();
	                if (key && x.transformer()) {
	                    key = x.transformer().transform(key).value;
	                }
	                if (_this.skipKey(key, isPrimary)) {
	                    return;
	                }
	                if (!key) {
	                    return;
	                }
	                var arr = m[key];
	                if (!arr) {
	                    arr = [];
	                    m[key] = arr;
	                }
	                arr.push({ node: y, transformer: x.transformer(), isPrimary: isPrimary });
	            });
	        });
	        var ramlVersion = this.unit().highLevel().root().definition().universe().version();
	        var isResource = this.key() && this.key()[0] == "/";
	        var methodType = def.getUniverse("RAML10").type(universes.Universe10.Method.name);
	        var options = methodType.property(universes.Universe10.Method.properties.method.name).enumOptions();
	        Object.keys(m).forEach(function (key) {
	            var arr = m[key];
	            var allOptional = true;
	            var hasPrimaryChildren = false;
	            var isMethod = options.indexOf(key) >= 0;
	            arr.forEach(function (x) {
	                var isOpt = x.node.optional() &&
	                    (ramlVersion != "RAML10" ||
	                        (isResource && isMethod));
	                allOptional = allOptional && isOpt;
	                hasPrimaryChildren = hasPrimaryChildren || x.isPrimary;
	            });
	            if (hasPrimaryChildren) {
	                var primaryChildren = [];
	                arr.filter(function (x) { return x.isPrimary; }).forEach(function (x) {
	                    var tr = x.transformer ? x.transformer : _this.transformer();
	                    primaryChildren.push(new LowLevelCompositeNode(x.node, _this, tr, _this.ramlVersion, true));
	                });
	                var primaryChild = primaryChildren[0];
	                arr.filter(function (x) { return !x.isPrimary; }).forEach(function (x) {
	                    primaryChild.adopt(x.node, x.transformer);
	                });
	                primaryChildren.forEach(function (x) { return result.push(x); });
	            }
	            else if (!allOptional) {
	                var tr = arr[0].transformer ? arr[0].transformer : _this.transformer();
	                var primaryChild = new LowLevelCompositeNode(arr[0].node, _this, tr, _this.ramlVersion, false);
	                for (var i = 1; i < arr.length; i++) {
	                    primaryChild.adopt(arr[i].node, arr[i].transformer);
	                }
	                result.push(primaryChild);
	            }
	        });
	        return result;
	    };
	    LowLevelCompositeNode.prototype.skipKey = function (key, isPrimary) {
	        if (isPrimary) {
	            return false;
	        }
	        if (this.ramlVersion != 'RAML08') {
	            return false;
	        }
	        var methodDef = universes.Universe08.Method;
	        var resourceDef = universes.Universe08.Resource;
	        //if(key==hasNormalParametersDef.properties.displayName.name
	        //    &&this.highLevelNode().definition().key().name==methodDef.name){
	        //    return true;
	        //}
	        //if(key==resourceDef.properties.displayName.name
	        //    &&this.highLevelNode().definition().key().name==resourceDef.name){
	        //    return true;
	        //}
	        return false;
	    };
	    LowLevelCompositeNode.prototype.valueKind = function () {
	        if (this._originalNode.kind() != yaml.Kind.MAPPING) {
	            return null;
	        }
	        for (var i = 0; i < this._adoptedNodes.length; i++) {
	            var node = this._adoptedNodes[i];
	            var yamlNode = node.originalNode()._node;
	            if (yamlNode && yamlNode.value != null) {
	                return node.valueKind();
	            }
	        }
	        return null;
	    };
	    LowLevelCompositeNode.prototype.includePath = function () {
	        for (var i = 0; i < this._adoptedNodes.length; i++) {
	            var node = this._adoptedNodes[i];
	            var includePath = node.includePath();
	            if (includePath != null) {
	                return includePath;
	            }
	        }
	        return null;
	    };
	    LowLevelCompositeNode.prototype.includeReference = function () {
	        for (var i = 0; i < this._adoptedNodes.length; i++) {
	            var node = this._adoptedNodes[i];
	            if (node.value() != null) {
	                return node.includeReference();
	            }
	        }
	        return null;
	    };
	    LowLevelCompositeNode.prototype.optional = function () {
	        return _.all(this._adoptedNodes, function (x) { return x.optional(); });
	    };
	    return LowLevelCompositeNode;
	}(LowLevelProxyNode));
	exports.LowLevelCompositeNode = LowLevelCompositeNode;
	var LowLevelValueTransformingNode = (function (_super) {
	    __extends(LowLevelValueTransformingNode, _super);
	    function LowLevelValueTransformingNode(node, parent, transformer, ramlVersion) {
	        _super.call(this, parent, transformer, ramlVersion);
	        this._originalNode = node;
	    }
	    LowLevelValueTransformingNode.prototype.value = function (toString) {
	        var val = this.originalNode().value(toString);
	        var t = this.transformer();
	        if (t) {
	            var transformationResult = t.transform(val);
	            val = transformationResult.value;
	        }
	        return val;
	    };
	    LowLevelValueTransformingNode.prototype.children = function () {
	        var _this = this;
	        return this.originalNode().children().map(function (x) { return new LowLevelValueTransformingNode(x, _this, _this._transformer, _this.ramlVersion); });
	    };
	    LowLevelValueTransformingNode.prototype.parent = function () { return this._parent; };
	    LowLevelValueTransformingNode.prototype.key = function () {
	        var key = _super.prototype.key.call(this);
	        if (this.transformer() != null) {
	            return this.transformer().transform(key).value;
	        }
	        return key;
	    };
	    return LowLevelValueTransformingNode;
	}(LowLevelProxyNode));
	exports.LowLevelValueTransformingNode = LowLevelValueTransformingNode;
	//# sourceMappingURL=LowLevelASTProxy.js.map

/***/ },
/* 38 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../../typings/main.d.ts" />
	"use strict";
	var jsyaml = __webpack_require__(23);
	var defs = __webpack_require__(35);
	var ll = __webpack_require__(11);
	var _ = __webpack_require__(61);
	var yaml = __webpack_require__(72);
	var hlimpl = __webpack_require__(16);
	var universes = __webpack_require__(22);
	var universeHelpers = __webpack_require__(14);
	var services = defs;
	function findLastAttributeIndex(n) {
	    var last = -1;
	    var children = n.lowLevel().children();
	    for (var i = 0; i < children.length; i++) {
	        var node = children[i];
	        if (!node.isMapping())
	            continue;
	        var name = node.asMapping().key.value;
	        var property = n.definition().property(name);
	        if (!property)
	            continue;
	        if (property.isFromParentValue() || property.range().hasValueTypeInHierarchy()) {
	            last = i;
	        }
	    }
	    //console.log('last attr index1: ' + last);
	    return last;
	}
	function findInsertionPointLowLevel(node, llnode, property, attr) {
	    //console.log('LL find insertion: node is attr: ' + attr);
	    var insertionPoint = null;
	    var embed = property && property.getAdapter(services.RAMLPropertyService).isEmbedMap();
	    if (embed && _.find(node.lowLevel().children(), function (x) { return x.key() == property.nameId(); })) {
	        embed = false;
	    }
	    if (attr || embed) {
	        var last = findLastAttribute(node);
	        if (!last) {
	            //insertionPoint = new jsyaml.InsertionPoint(jsyaml.InsertionPointType.START);
	            insertionPoint = jsyaml.InsertionPoint.atStart();
	        }
	        else {
	            insertionPoint = last;
	        }
	    }
	    return insertionPoint;
	}
	function findInsertionPoint(where, node) {
	    //console.log('find insertion point for node (HL): ' + node.property().name() + '; attr: ' + node.isAttr());
	    //console.log('node1: ' + node.lowLevel().text());
	    //always insert attributes at start
	    if (!where.isStub()) {
	        where.clearChildrenCache();
	    }
	    var ch = where.children();
	    var embed = node.property() && node.property().getAdapter(services.RAMLPropertyService).isEmbedMap();
	    if (embed && _.find(where.lowLevel().children(), function (x) { return x.key() == node.property().nameId(); })) {
	        embed = false;
	    }
	    var uh = universeHelpers;
	    if ((node.isAttr() || embed) && !(universeHelpers.isMethodType(where.definition()) && node.property && universeHelpers.isIsProperty(node.property()))) {
	        var toRet = null;
	        for (var i = 0; i < ch.length; i++) {
	            if (!ch[i].isAttr()) {
	                break;
	            }
	            else {
	                toRet = ch[i].lowLevel();
	            }
	        }
	        if (toRet == null) {
	            toRet = where.lowLevel();
	        }
	        return toRet;
	    }
	    else {
	        var pname = node.property().nameId();
	        var cls = where.definition();
	        var props = cls.allProperties();
	        //console.log('class: ' + cls.name());
	        //props.forEach(x=> console.log('  prop: ' + x.name()));
	        var pindex = cls.propertyIndex(pname);
	        if (pindex < 0) {
	            return null;
	        }
	        var llchilds = where.lowLevel().children();
	        //console.log('lookup: ' + pname + ' index: ' + pindex + ' childs: ' + llchilds.length);
	        for (var i = 0; i < llchilds.length; i++) {
	            var llch = llchilds[i];
	            //console.log('  child: ' + llch.kindName());
	            if (!llch.isMapping())
	                continue;
	            var cpnme = llch.asMapping().key.value;
	            var pi = cls.propertyIndex(cpnme);
	            //console.log('  property: ' + cpnme + ' index: ' + pi + ' at pos: ' + i);
	            if (pi > pindex) {
	                //console.log('  property: ' + cpnme + ' - found');
	                var lastok = i - 1;
	                //console.log('lastok: ' + lastok);
	                if (lastok < 0) {
	                    //TODO insert at the very beginning
	                    //console.log('insert to very beginning');
	                    return null;
	                }
	                else {
	                    console.log('insert to node: ' + lastok);
	                    return llchilds[lastok];
	                }
	            }
	            else {
	            }
	        }
	        return null;
	    }
	    //console.log('HL insertion: ' + toRet);
	    //return toRet;
	}
	function removeNodeFrom(source, node) {
	    if (source.isStub()) {
	        if (node instanceof hlimpl.ASTNodeImpl) {
	            var cm = node;
	            if (cm.isInEdit) {
	                return;
	            }
	        }
	        if (!source._children) {
	            return;
	        }
	        source._children = source._children.filter(function (x) { return x != node; });
	        return;
	    }
	    var command = new ll.CompositeCommand();
	    if (node instanceof hlimpl.ASTNodeImpl) {
	        var aNode = node;
	        if (!aNode.property().getAdapter(services.RAMLPropertyService).isMerged()) {
	            if (source.elementsOfKind(aNode.property().nameId()).length == 1) {
	                command.commands.push(ll.removeNode(source.lowLevel(), aNode.lowLevel().parent().parent()));
	            }
	            else {
	                command.commands.push(ll.removeNode(source.lowLevel(), aNode.lowLevel()));
	            }
	        }
	        else {
	            command.commands.push(ll.removeNode(source.lowLevel(), aNode.lowLevel()));
	        }
	    }
	    else {
	        command.commands.push(ll.removeNode(source.lowLevel(), node.lowLevel()));
	    }
	    source.lowLevel().execute(command);
	}
	exports.removeNodeFrom = removeNodeFrom;
	function initEmptyRAMLFile(node) {
	    var llroot = node.lowLevel().root();
	    var command = new ll.CompositeCommand();
	    var newroot = jsyaml.createMap([]);
	    command.commands.push(ll.initRamlFile(node.lowLevel(), newroot));
	    node.lowLevel().execute(command);
	    var root = node.root();
	    //console.log('root: ' + root);
	    root._node = newroot;
	    root.clearChildrenCache();
	}
	exports.initEmptyRAMLFile = initEmptyRAMLFile;
	function setValue(node, value) {
	    if (value == node.value())
	        return;
	    var c = new ll.CompositeCommand();
	    if (typeof value === 'string') {
	        var val = value;
	        //FIXME actually isFromParentKey should be enough in future does not changing it now for safety reasons
	        if (node.property().isFromParentKey() || node.property().isAnnotation()) {
	            if (node.property().isAnnotation()) {
	                val = '(' + val + ')';
	            }
	            c.commands.push(ll.setKey(node.lowLevel(), val));
	        }
	        else {
	            if ((!val || val.length == 0) && !node.isEmbedded()) {
	                c.commands.push(ll.removeNode(node.lowLevel().parent(), node.lowLevel()));
	                node.parent().clearChildrenCache();
	            }
	            else {
	                if (!val)
	                    val = '';
	                c.commands.push(ll.setAttr(node.lowLevel(), val));
	            }
	        }
	    }
	    else {
	        if (node.property().isFromParentKey()) {
	            throw new Error("couldn't set structured value to a key: " + node.property().nameId());
	        }
	        var sval = value;
	        c.commands.push(ll.setAttrStructured(node.lowLevel(), sval));
	    }
	    node.lowLevel().execute(c);
	}
	exports.setValue = setValue;
	function addStringValue(attr, value) {
	    var sc = jsyaml.createScalar(value);
	    var target = attr.lowLevel();
	    //console.log('add to target: ' + target.kindName());
	    if (target.isScalar()) {
	        target = target.parent();
	    }
	    else if (target.isMapping()) {
	    }
	    var command = new ll.CompositeCommand();
	    command.commands.push(ll.insertNode(target, sc, null, true));
	    attr.lowLevel().execute(command);
	    attr.parent().clearChildrenCache();
	}
	exports.addStringValue = addStringValue;
	function addStructuredValue(attr, sv) {
	    //var sc = jsyaml.createScalar(value);
	    var target = attr.lowLevel();
	    //console.log('add to target: ' + target.kindName());
	    if (target.isScalar()) {
	        target = target.parent();
	    }
	    else if (target.isMapping()) {
	        var ln = attr.lowLevel();
	    }
	    var command = new ll.CompositeCommand();
	    command.commands.push(ll.insertNode(target, sv.lowLevel(), null, true));
	    attr.lowLevel().execute(command);
	    attr.parent().clearChildrenCache();
	}
	exports.addStructuredValue = addStructuredValue;
	function removeAttr(attr) {
	    var llparent = attr.lowLevel().parent();
	    if (!attr.property().isMultiValue() && attr.isEmbedded()) {
	        // it's embedded value, need to clean scalar instead
	        //console.log('embedded!');
	        attr.setValue('');
	    }
	    else {
	        var command = new ll.CompositeCommand();
	        command.commands.push(ll.removeNode(llparent, attr.lowLevel()));
	        attr.lowLevel().execute(command);
	        attr.parent().clearChildrenCache();
	    }
	}
	exports.removeAttr = removeAttr;
	function setValues(attr, values) {
	    if (!attr.property().isMultiValue())
	        throw new Error("setValue(string[]) only apply to multi-values properties");
	    var node = attr.parent();
	    if (attr.isEmpty()) {
	    }
	    else {
	        var llnode = node.lowLevel();
	        var attrs = node.attributes(attr.name());
	        attrs.forEach(function (attr) { return attr.remove(); });
	    }
	    if (values.length == 1 && attr.property().isDescriminator()) {
	        node.attrOrCreate(attr.name()).setValue(values[0]);
	    }
	    else {
	        values.forEach(function (val) {
	            return node.attrOrCreate(attr.name()).addValue(val);
	        });
	    }
	}
	exports.setValues = setValues;
	function setKey(node, value) {
	    if (value == node.name())
	        return;
	    var c = new ll.CompositeCommand();
	    c.commands.push(ll.setKey(node.lowLevel(), value));
	    node.lowLevel().execute(c);
	}
	exports.setKey = setKey;
	function createAttr(node, n, v) {
	    var mapping = jsyaml.createMapping(n, v);
	    var map;
	    if (node.definition() && node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
	        if (node.lowLevel() && node.lowLevel().valueKind() === yaml.Kind.SCALAR) {
	            var typePropertyName = universes.Universe10.TypeDeclaration.properties.type.name;
	            var typeMapping = jsyaml.createMapping(typePropertyName, node.lowLevel().value());
	            var command = new ll.CompositeCommand();
	            command.commands.push(ll.setAttr(node.lowLevel(), ""));
	            node.lowLevel().execute(command);
	            command = new ll.CompositeCommand();
	            var insertionPoint = findInsertionPointLowLevel(node, typeMapping, node.definition().property(typePropertyName), true);
	            command.commands.push(ll.insertNode(node.lowLevel(), typeMapping, insertionPoint));
	            node.lowLevel().execute(command);
	        }
	    }
	    //console.log('create attribute: ' + n);
	    if (node.isStub()) {
	        //console.log('create-attr: stub case');
	        var insertionIndex = findLastAttributeIndex(node);
	        //console.log('stub insertion index: ' + insertionIndex);
	        node.lowLevel().addChild(mapping, insertionIndex + 1);
	    }
	    else {
	        //console.log('root: ' + llroot.kindName());
	        if (node.isEmptyRamlFile()) {
	            node.initRamlFile();
	        }
	        //console.log('create-attr: real node case');
	        //node._node.addChild(mapping);
	        //node.clearChildrenCache();
	        var command = new ll.CompositeCommand();
	        var insertionPoint = findInsertionPointLowLevel(node, mapping, node.definition().property(n), true);
	        //command.commands.push(ll.insertNode(node.lowLevel(), mapping, null));
	        //var toseq = node.property().isMultiValue();
	        command.commands.push(ll.insertNode(node.lowLevel(), mapping, insertionPoint));
	        node.lowLevel().execute(command);
	    }
	    node.clearChildrenCache();
	}
	exports.createAttr = createAttr;
	function findLastAttribute(node) {
	    var childs = node.lowLevel().children();
	    var index = findLastAttributeIndex(node);
	    //console.log('last attr index2: ' + index);
	    return (index < 0) ? null : childs[index];
	}
	function addToNode(target, node) {
	    if (!target.isStub() && target.isEmptyRamlFile()) {
	        target.initRamlFile();
	    }
	    var llnode = node.lowLevel();
	    if (!target._children) {
	        target._children = [];
	    }
	    if (!node.property()) {
	        //now we should find correct property;
	        var an = node;
	        var allProps = target.definition().allProperties();
	        var cp = null;
	        allProps.forEach(function (x) {
	            var r = x.range();
	            if (r == an.definition()) {
	                cp = x;
	            }
	            var isOk = _.find(an.definition().allSuperTypes(), function (x) { return x == r; });
	            if (isOk) {
	                cp = x;
	            }
	        });
	        if (!cp) {
	            throw new Error("Unable to find correct child");
	        }
	        else {
	            an.patchProp(cp);
	        }
	    }
	    var insertionPoint = findInsertionPoint(target, node);
	    //console.log('high level op: ' + target.property() + '.add ' + node.property().name());
	    if (insertionPoint) {
	    }
	    //var newLowLevel:ll.ILowLevelASTNode=null;
	    var command = new ll.CompositeCommand();
	    //now we need to understand to which low level node it should go
	    //command.commands.push(ll.insertNode(target.lowLevel(), node.lowLevel()))
	    var insertionTarget = null;
	    if (node.property().getAdapter(services.RAMLPropertyService).isMerged() || node.property().range().hasValueTypeInHierarchy()) {
	        //console.log('CASE 1');
	        //newLowLevel = node.lowLevel();
	        command.commands.push(ll.insertNode(target.lowLevel(), node.lowLevel(), insertionPoint));
	        insertionTarget = target.lowLevel();
	    }
	    else {
	        //console.log('CASE 2');
	        var name = node.property().nameId();
	        var target2 = target.lowLevel();
	        //target.show('TARGET:');
	        //llnode.show('NODE:');
	        var found = target.lowLevel().find(name);
	        insertionTarget = found;
	        if (!found) {
	            //console.log('node not found');
	            var nn = null;
	            //var nn: jsyaml.ASTNode = jsyaml.createSeqNode(name);
	            //var mapping = <yaml.YAMLMapping>nn._actualNode();
	            //var seq: yaml.YAMLSequence = <yaml.YAMLSequence>mapping.value;
	            //if(!seq.items) seq.items = [];
	            //seq.items.push((<jsyaml.ASTNode>node.lowLevel())._actualNode());
	            if (node.property().getAdapter(services.RAMLPropertyService).isEmbedMap()) {
	                var v10 = target.definition().universe().version() == 'RAML10';
	                if (llnode.isValueMap() && v10)
	                    nn = jsyaml.createMapNode(name);
	                else
	                    nn = jsyaml.createSeqNode(name);
	                //console.log('NN: ' + yaml.Kind[nn._actualNode().kind]);
	                nn.addChild(node.lowLevel());
	            }
	            else {
	                nn = jsyaml.createNode(name);
	                nn.addChild(node.lowLevel());
	            }
	            //nn.show('WRAPPED NODE:');
	            //target.show('INSERT WRAPPED NODE TO:');
	            command.commands.push(ll.insertNode(target2, nn, insertionPoint));
	            insertionTarget = target2;
	        }
	        else {
	            //console.log('node found');
	            //found.show('INSERT2: ');
	            if (node.property().getAdapter(services.RAMLPropertyService).isEmbedMap()) {
	                //newLowLevel=node.lowLevel();
	                command.commands.push(ll.insertNode(found, node.lowLevel(), insertionPoint, true));
	            }
	            else {
	                //newLowLevel=node.lowLevel();
	                command.commands.push(ll.insertNode(found, node.lowLevel(), insertionPoint, false));
	            }
	        }
	    }
	    if (target.isStub()) {
	        var insertionIndex = findLastAttributeIndex(target);
	        if (insertionIndex < 0) {
	            target._children.push(node);
	        }
	        else {
	            //TODO behavior should be smarter we are ignoring insertion points now
	            target._children.splice(insertionIndex, 0, node);
	        }
	        command.commands.forEach(function (x) { return insertionTarget.addChild(x.value); });
	        return;
	    }
	    target.lowLevel().execute(command);
	    target._children.push(node);
	    //now we need to add new child to our children;
	    node.setParent(target);
	}
	exports.addToNode = addToNode;
	//# sourceMappingURL=mutators.js.map

/***/ },
/* 39 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../../typings/main.d.ts" />
	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var defs = __webpack_require__(35);
	var ramlTypes = defs.rt;
	var _ = __webpack_require__(61);
	var universes = __webpack_require__(22);
	var hlimpl = __webpack_require__(16);
	var services = defs;
	var linter = __webpack_require__(27);
	ramlTypes.setPropertyConstructor(function (x) {
	    var v = new defs.Property(x);
	    v.unmerge();
	    return v;
	});
	function templateFields(node, d) {
	    var u = node.root().definition().universe();
	    node.children().forEach(function (x) { return templateFields(x, d); });
	    if (node instanceof hlimpl.ASTPropImpl) {
	        var prop = node;
	        //TODO RECURSIVE PARAMETERS
	        var v = prop.value();
	        if (typeof v == 'string') {
	            var strV = v;
	            handleValue(strV, d, prop, false, u);
	        }
	        else {
	            node.lowLevel().visit(function (x) {
	                if (x.value()) {
	                    var strV = x.value() + "";
	                    handleValue(strV, d, prop, true, u);
	                }
	                return true;
	            });
	        }
	    }
	    else if (node instanceof hlimpl.BasicASTNode) {
	        var v = node.lowLevel().value();
	        if (typeof v == 'string') {
	            var strV = v;
	            handleValue(strV, d, null, false, u);
	        }
	        else {
	            node.lowLevel().visit(function (x) {
	                if (x.value()) {
	                    var strV = x.value() + "";
	                    handleValue(strV, d, null, true, u);
	                }
	                return true;
	            });
	        }
	    }
	}
	var extractUsedParamNames = function (strV) {
	    var parameterUsages = [];
	    var ps = 0;
	    while (true) {
	        var pos = strV.indexOf("<<", ps);
	        if (pos == -1) {
	            break;
	        }
	        var end = strV.indexOf(">>", pos);
	        var isFull = pos == 0 && end == strV.length - 2;
	        var parameterUsage = strV.substring(pos + 2, end);
	        ps = pos + 2;
	        var directiveIndex = parameterUsage.indexOf("|");
	        if (directiveIndex != -1) {
	            parameterUsage = parameterUsage.substring(0, directiveIndex);
	        }
	        parameterUsage = parameterUsage.trim();
	        parameterUsages.push(parameterUsage);
	    }
	    return { parameterUsages: parameterUsages, isFull: isFull };
	};
	var handleValue = function (strV, d, prop, allwaysString, u) {
	    var __ret = extractUsedParamNames(strV);
	    var parameterUsages = __ret.parameterUsages;
	    var isFull = __ret.isFull;
	    var r = (prop) ? prop.property().range() : null;
	    if (prop) {
	        if (prop.property().nameId() == universes.Universe10.TypeDeclaration.properties.type.name ||
	            prop.property().nameId() == universes.Universe10.TypeDeclaration.properties.schema.name) {
	            if (prop.property().domain().key() == universes.Universe10.TypeDeclaration) {
	                r = u.type(universes.Universe10.SchemaString.name);
	            }
	        }
	    }
	    for (var _i = 0, parameterUsages_1 = parameterUsages; _i < parameterUsages_1.length; _i++) {
	        var parameterUsage = parameterUsages_1[_i];
	        if (linter.RESERVED_TEMPLATE_PARAMETERS[parameterUsage] != null) {
	            //Handling reserved parameter names;
	            continue;
	        }
	        var q = d[parameterUsage];
	        if (!isFull || allwaysString) {
	            r = u.type(universes.Universe10.StringType.name);
	        }
	        //FIX ME NOT WHOLE TEMPLATES
	        if (q) {
	            q.push({
	                tp: r,
	                attr: prop
	            });
	        }
	        else {
	            d[parameterUsage] = [{
	                    tp: r,
	                    attr: prop
	                }];
	        }
	    }
	};
	function fillParamPaths(node, paramPaths, path) {
	    if (path === void 0) { path = []; }
	    if (node.optional()) {
	        path = path.concat("/");
	    }
	    var v = node.value();
	    if (typeof v == 'string') {
	        var strV = v;
	        var __ret = extractUsedParamNames(strV);
	        var parameterUsages = __ret.parameterUsages;
	        for (var _i = 0, parameterUsages_2 = parameterUsages; _i < parameterUsages_2.length; _i++) {
	            var pu = parameterUsages_2[_i];
	            var paths = paramPaths[pu];
	            if (paths == null) {
	                paths = [];
	                paramPaths[pu] = paths;
	            }
	            paths.push(path);
	        }
	    }
	    else {
	        for (var _a = 0, _b = node.children(); _a < _b.length; _a++) {
	            var ch = _b[_a];
	            fillParamPaths(ch, paramPaths, path.concat(ch.key()));
	        }
	    }
	}
	function fillTemplateType(result, node) {
	    var usages = {};
	    var paramPaths = {};
	    templateFields(node, usages);
	    fillParamPaths(node.lowLevel(), paramPaths);
	    for (var _i = 0, _a = Object.keys(paramPaths); _i < _a.length; _i++) {
	        var pu = _a[_i];
	        paramPaths[pu] = _.unique(paramPaths[pu]);
	    }
	    result.getAdapter(services.RAMLService).setInlinedTemplates(true);
	    Object.keys(usages).forEach(function (x) {
	        var prop = new defs.UserDefinedProp(x);
	        //prop._node=node;
	        prop.withDomain(result);
	        var paths = paramPaths[x];
	        prop.getAdapter(services.RAMLPropertyService).putMeta("templatePaths", paths);
	        var tp = _.unique(usages[x]).map(function (x) { return x.tp; }).filter(function (x) { return x && x.nameId() != universes.Universe08.StringType.name; });
	        prop.withRange(tp.length == 1 ? tp[0] : node.definition().universe().type(universes.Universe08.StringType.name));
	        prop.withRequired(true);
	        if (usages[x].length > 0) {
	            prop._node = usages[x][0].attr;
	        }
	        if (tp.length == 1 && node.definition().universe().version() == 'RAML10') {
	            if (tp[0].key() == universes.Universe10.SchemaString) {
	                prop.getAdapter(services.RAMLPropertyService).setTypeExpression(true);
	            }
	        }
	        prop.unmerge();
	    });
	    var keyProp = new defs.UserDefinedProp("____key");
	    //prop._node=node;
	    keyProp.withDomain(result);
	    keyProp.withKey(true);
	    keyProp._node = node;
	    keyProp.withFromParentKey(true);
	    keyProp.withRange(node.definition().universe().type(universes.Universe08.StringType.name));
	    return result;
	}
	function fillReferenceType(result, def) {
	    if (def.universe().version() == "RAML08") {
	        result.getAdapter(services.RAMLService).withAllowAny();
	    }
	    var p = def.property(def.getAdapter(services.RAMLService).getReferenceIs());
	    if (p) {
	        p.range().properties().forEach(function (x) {
	            var prop = new defs.Property(x.nameId());
	            prop.unmerge();
	            prop.withDomain(result);
	            prop.withRange(x.range());
	            prop.withMultiValue(x.isMultiValue());
	        });
	    }
	    return result;
	}
	var AnnotationType = (function (_super) {
	    __extends(AnnotationType, _super);
	    function AnnotationType() {
	        _super.apply(this, arguments);
	    }
	    AnnotationType.prototype.allProperties = function (ps) {
	        if (ps === void 0) { ps = {}; }
	        var rs = this.superTypes()[0].allProperties();
	        if (rs.length == 1 && rs[0].nameId() == "annotations") {
	            var up = new defs.UserDefinedProp("value");
	            up.withDomain(this);
	            up._node = this.getAdapter(defs.RAMLService).getDeclaringNode();
	            up.withCanBeValue();
	            up.withRequired(false);
	            var tp = this.superTypes()[0];
	            rs = [];
	            up.withRange(up._node.asElement().definition().universe().type("string"));
	            rs.push(up);
	        }
	        return rs;
	    };
	    AnnotationType.prototype.isAnnotationType = function () {
	        return true;
	    };
	    return AnnotationType;
	}(defs.UserDefinedClass));
	function typeFromNode(node) {
	    if (!node) {
	        return null;
	    }
	    if (node.associatedType()) {
	        return node.associatedType();
	    }
	    var u = node.lowLevel().unit();
	    var upath = u ? u.path() : "";
	    ramlTypes.setPropertyConstructor(function (x) {
	        var v = new defs.UserDefinedProp(x);
	        var rs = node.elementsOfKind("properties").filter(function (y) { return y.name() == x; });
	        if (rs) {
	            v._node = rs[0];
	        }
	        v.unmerge();
	        return v;
	    });
	    var def = node.definition();
	    if (node.property() && node.property().nameId() == universes.Universe10.LibraryBase.properties.annotationTypes.name) {
	        //var st=node.definition().getAdapter(services.RAMLService).toRuntime();
	        var result = new AnnotationType(node.name(), node.definition().universe(), node, upath, "");
	        var st = getSimpleType(node);
	        result._superTypes.push(st);
	        if (node.elementsOfKind(universes.Universe10.ObjectTypeDeclaration.properties.properties.name).length == 0) {
	            result.getAdapter(services.RAMLService).withAllowAny();
	        }
	        var extType = def.getAdapter(services.RAMLService).getExtendedType();
	        if (extType) {
	            result._superTypes.push(extType);
	        }
	        return result;
	    }
	    else {
	        var result = new defs.UserDefinedClass(node.name(), node.definition().universe(), node, upath, "");
	    }
	    node.setAssociatedType(result);
	    //result.setDeclaringNode(node);
	    if (def.getAdapter(services.RAMLService).isInlinedTemplates()) {
	        return fillTemplateType(result, node);
	    }
	    else if (def.getAdapter(services.RAMLService).getReferenceIs()) {
	        return fillReferenceType(result, def);
	    }
	    var rs = getSimpleType(node);
	    rs.getAdapter(services.RAMLService).setDeclaringNode(node);
	    node.setAssociatedType(rs);
	    return rs;
	}
	exports.typeFromNode = typeFromNode;
	function getSimpleType(node) {
	    return ramlTypes.toNominal(node.parsedType(), function (x) {
	        var m = node.definition().universe().type(x);
	        if (!m) {
	            var ut = new defs.UserDefinedClass("", node.definition().universe(), node, "", "");
	        }
	        return m;
	    });
	}
	function convertType(root, t) {
	    var node = _.find(root.elementsOfKind("types"), function (x) { return x.name() == t.name(); });
	    if (node) {
	        ramlTypes.setPropertyConstructor(function (x) {
	            var v = new defs.UserDefinedProp(x);
	            var rs = node.elementsOfKind("properties").filter(function (y) { return y.name() == x; });
	            if (rs && rs.length > 0) {
	                v._node = rs[0];
	            }
	            else {
	                var rs = node.elementsOfKind("facets").filter(function (y) { return y.name() == x; });
	                if (rs && rs.length > 0) {
	                    v._node = rs[0];
	                }
	            }
	            v.unmerge();
	            return v;
	        });
	    }
	    return ramlTypes.toNominal(t, function (x) {
	        var m = root.definition().universe().type(x);
	        if (!m) {
	            var ut = new defs.UserDefinedClass("", root.definition().universe(), root, "", "");
	        }
	        return m;
	    });
	}
	exports.convertType = convertType;
	//# sourceMappingURL=typeBuilder.js.map

/***/ },
/* 40 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../typings/main.d.ts" />
	"use strict";
	function isMultiLine(s) {
	    return s && s.indexOf('\n') >= 0;
	}
	exports.isMultiLine = isMultiLine;
	function isMultiLineValue(s) {
	    return isMultiLine(s) && s.length > 2 && s[0] == '|' && (s[1] == '\n' || s[1] == '\r' || s[2] == '\n');
	}
	exports.isMultiLineValue = isMultiLineValue;
	function makeMutiLine(s, lev) {
	    var xbuf = '';
	    if (isMultiLine(s)) {
	        xbuf += '|\n';
	        var lines = splitOnLines(s);
	        for (var i = 0; i < lines.length; i++) {
	            xbuf += indent(lev, lines[i]);
	        }
	    }
	    else {
	        xbuf += s;
	    }
	    return xbuf;
	}
	exports.makeMutiLine = makeMutiLine;
	function fromMutiLine(s) {
	    if (!isMultiLineValue(s))
	        return s;
	    var res = null;
	    var lines = splitOnLines(s);
	    for (var i = 1; i < lines.length; i++) {
	        var line = lines[i];
	        var str = line.substring(2);
	        if (!res)
	            res = str;
	        else
	            res += str;
	    }
	    return res;
	}
	exports.fromMutiLine = fromMutiLine;
	function trimStart(s) {
	    if (!s)
	        return s;
	    var pos = 0;
	    while (pos < s.length) {
	        var ch = s[pos];
	        if (ch != '\r' && ch != '\n' && ch != ' ' && ch != '\t')
	            break;
	        pos++;
	    }
	    return s.substring(pos, s.length);
	}
	exports.trimStart = trimStart;
	function indent(lev, str) {
	    if (str === void 0) { str = ''; }
	    var leading = '';
	    //leading += '[' + lev + ']';
	    for (var i = 0; i < lev; i++)
	        leading += '  ';
	    return leading + str;
	}
	exports.indent = indent;
	function print(lev, str) {
	    if (str === void 0) { str = ''; }
	    console.log(indent(lev, str));
	}
	exports.print = print;
	function replaceNewlines(s, rep) {
	    if (rep === void 0) { rep = null; }
	    var res = '';
	    for (var i = 0; i < s.length; i++) {
	        var ch = s[i];
	        if (ch == '\r')
	            ch = rep == null ? '\\r' : rep;
	        if (ch == '\n')
	            ch = rep == null ? '\\n' : rep;
	        res += ch;
	    }
	    return res;
	}
	exports.replaceNewlines = replaceNewlines;
	function trimEnd(s) {
	    var pos = s.length;
	    while (pos > 0) {
	        var ch = s[pos - 1];
	        if (ch != ' ' && ch != '\t' && ch != '\r' && ch != '\n')
	            break;
	        pos--;
	    }
	    return s.substring(0, pos);
	}
	exports.trimEnd = trimEnd;
	function trim(s) {
	    return trimStart(trimEnd(s));
	}
	exports.trim = trim;
	function splitOnLines(text) {
	    var lines = text.match(/^.*((\r\n|\n|\r)|$)/gm);
	    return lines;
	}
	exports.splitOnLines = splitOnLines;
	function startsWith(s, suffix) {
	    if (!s || !suffix || s.length < suffix.length)
	        return false;
	    for (var i = 0; i < suffix.length; i++) {
	        if (s[i] != suffix[i])
	            return false;
	    }
	    return true;
	}
	exports.startsWith = startsWith;
	function endsWith(s, suffix) {
	    if (!s || !suffix || s.length < suffix.length)
	        return false;
	    for (var i = 0; i < suffix.length; i++) {
	        if (s[s.length - 1 - i] != suffix[suffix.length - 1 - i])
	            return false;
	    }
	    return true;
	}
	exports.endsWith = endsWith;
	function capitalize(s) {
	    return s.charAt(0).toUpperCase() + s.slice(1);
	}
	exports.capitalize = capitalize;
	var TextRange = (function () {
	    function TextRange(contents, start, end) {
	        this.contents = contents;
	        this.start = start;
	        this.end = end;
	    }
	    TextRange.prototype.text = function () {
	        return this.contents.substring(this.start, this.end);
	    };
	    TextRange.prototype.startpos = function () { return this.start; };
	    TextRange.prototype.endpos = function () { return this.end; };
	    TextRange.prototype.len = function () { return this.end - this.start; };
	    TextRange.prototype.unitText = function () { return this.contents; };
	    TextRange.prototype.withStart = function (start) {
	        return new TextRange(this.contents, start, this.end);
	    };
	    TextRange.prototype.withEnd = function (end) {
	        return new TextRange(this.contents, this.start, end);
	    };
	    TextRange.prototype.sub = function (start, end) {
	        return this.contents.substring(start, end);
	    };
	    TextRange.prototype.trimStart = function () {
	        var pos = this.start;
	        while (pos < this.contents.length - 1) {
	            var ch = this.contents[pos];
	            if (ch != ' ' && ch != '\t')
	                break;
	            pos++;
	        }
	        return new TextRange(this.contents, pos, this.end);
	    };
	    TextRange.prototype.trimEnd = function () {
	        var pos = this.end;
	        while (pos > 0) {
	            var ch = this.contents[pos - 1];
	            if (ch != ' ' && ch != '\t')
	                break;
	            pos--;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.extendToStartOfLine = function () {
	        var pos = this.start;
	        while (pos > 0) {
	            var prevchar = this.contents[pos - 1];
	            if (prevchar == '\r' || prevchar == '\n')
	                break;
	            pos--;
	        }
	        return new TextRange(this.contents, pos, this.end);
	    };
	    TextRange.prototype.extendAnyUntilNewLines = function () {
	        var pos = this.end;
	        if (pos > 0) {
	            var last = this.contents[pos - 1];
	            if (last == '\n')
	                return this;
	        }
	        while (pos < this.contents.length - 1) {
	            var nextchar = this.contents[pos];
	            if (nextchar == '\r' || nextchar == '\n')
	                break;
	            pos++;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.extendSpacesUntilNewLines = function () {
	        var pos = this.end;
	        if (pos > 0) {
	            var last = this.contents[pos - 1];
	            if (last == '\n')
	                return this;
	        }
	        while (pos < this.contents.length - 1) {
	            var nextchar = this.contents[pos];
	            if (nextchar != ' ' || nextchar == '\r' || nextchar == '\n')
	                break;
	            pos++;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.extendSpaces = function () {
	        var pos = this.end;
	        while (pos < this.contents.length - 1) {
	            var nextchar = this.contents[pos];
	            if (nextchar != ' ')
	                break;
	            pos++;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.extendSpacesBack = function () {
	        var pos = this.start;
	        while (pos > 0) {
	            var nextchar = this.contents[pos - 1];
	            if (nextchar != ' ')
	                break;
	            pos--;
	        }
	        return new TextRange(this.contents, pos, this.end);
	    };
	    TextRange.prototype.extendCharIfAny = function (ch) {
	        var pos = this.end;
	        if (pos < this.contents.length - 1 && this.contents[pos] == ch) {
	            pos++;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.extendCharIfAnyBack = function (ch) {
	        var pos = this.start;
	        if (pos > 0 && this.contents[pos - 1] == ch) {
	            pos--;
	        }
	        return new TextRange(this.contents, pos, this.end);
	    };
	    TextRange.prototype.extendToNewlines = function () {
	        var pos = this.end;
	        if (pos > 0) {
	            var last = this.contents[pos - 1];
	            if (last == '\n')
	                return this;
	        }
	        while (pos < this.contents.length - 1) {
	            var nextchar = this.contents[pos];
	            if (nextchar != '\r' && nextchar != '\n')
	                break;
	            pos++;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.extendUntilNewlinesBack = function () {
	        var pos = this.start;
	        while (pos > 0) {
	            var nextchar = this.contents[pos - 1];
	            if (nextchar == '\r' || nextchar == '\n')
	                break;
	            pos--;
	        }
	        return new TextRange(this.contents, pos, this.end);
	    };
	    TextRange.prototype.reduceNewlinesEnd = function () {
	        var pos = this.end;
	        while (pos > this.start) {
	            var last = this.contents[pos - 1];
	            if (last != '\r' && last != '\n')
	                break;
	            pos--;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.reduceSpaces = function () {
	        var pos = this.end;
	        while (pos > this.start) {
	            var last = this.contents[pos - 1];
	            if (last != ' ')
	                break;
	            pos--;
	        }
	        return new TextRange(this.contents, this.start, pos);
	    };
	    TextRange.prototype.replace = function (text) {
	        return this.sub(0, this.start) + text + this.sub(this.end, this.unitText().length);
	    };
	    TextRange.prototype.remove = function () {
	        return this.sub(0, this.start) + this.sub(this.end, this.unitText().length);
	    };
	    return TextRange;
	}());
	exports.TextRange = TextRange;
	//# sourceMappingURL=textutil.js.map

/***/ },
/* 41 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var RamlWrapper = __webpack_require__(20);
	/**
	 * @hidden
	 * Build Wrapper node corresponding to the High Level node
	 **/
	function buildWrapperNode(node, setAsTopLevel) {
	    if (setAsTopLevel === void 0) { setAsTopLevel = true; }
	    var definition = node.definition();
	    var nodeClassName = definition.nameId();
	    var wrapperConstructor = classMap[nodeClassName];
	    if (!wrapperConstructor) {
	        var priorities = determineSuperclassesPriorities(definition);
	        var superTypes = definition.allSuperTypes().sort(function (x, y) { return priorities[x.nameId()] - priorities[y.nameId()]; });
	        var wr = null;
	        for (var i = 0; i < superTypes.length; i++) {
	            var superTypeName = superTypes[i].nameId();
	            wrapperConstructor = classMap[superTypeName];
	            if (superTypeName == "DataElement") {
	                wr = superTypeName;
	                //This is only case of nested hierarchy
	                continue;
	            }
	            if (superTypeName == "hl.BasicNode") {
	                //depth first
	                continue;
	            }
	            if (wrapperConstructor) {
	                break;
	            }
	        }
	        if (!wrapperConstructor) {
	            wr = superTypeName;
	        }
	    }
	    if (!wrapperConstructor) {
	        wrapperConstructor = classMap["hl.BasicNode"];
	    }
	    return wrapperConstructor(node, setAsTopLevel);
	}
	exports.buildWrapperNode = buildWrapperNode;
	function determineSuperclassesPriorities(td, priorities, path) {
	    if (priorities === void 0) { priorities = {}; }
	    if (path === void 0) { path = {}; }
	    var typeName = td.nameId();
	    if (path[typeName]) {
	        return;
	    }
	    path[typeName] = true;
	    var rank = (priorities[typeName] != null && priorities[typeName] + 1) || 0;
	    var superTypes = td.superTypes();
	    superTypes.forEach(function (x) {
	        var name = x.nameId();
	        var r = priorities[name];
	        if (r == null || rank > r) {
	            priorities[name] = rank;
	            determineSuperclassesPriorities(x, priorities, path);
	        }
	    });
	    delete path[typeName];
	    return priorities;
	}
	var classMap = {
	    "AbstractSecurityScheme": function (x, y) { return new RamlWrapper.AbstractSecuritySchemeImpl(x, y); },
	    "Annotable": function (x, y) { return new RamlWrapper.AnnotableImpl(x, y); },
	    "AnnotationRef": function (x) { return new RamlWrapper.AnnotationRefImpl(x); },
	    "AnnotationTarget": function (x) { return new RamlWrapper.AnnotationTargetImpl(x); },
	    "AnyType": function (x) { return new RamlWrapper.AnyTypeImpl(x); },
	    "Api": function (x, y) { return new RamlWrapper.ApiImpl(x, y); },
	    "ArrayTypeDeclaration": function (x, y) { return new RamlWrapper.ArrayTypeDeclarationImpl(x, y); },
	    "BasicSecurityScheme": function (x, y) { return new RamlWrapper.BasicSecuritySchemeImpl(x, y); },
	    "BooleanType": function (x) { return new RamlWrapper.BooleanTypeImpl(x); },
	    "BooleanTypeDeclaration": function (x, y) { return new RamlWrapper.BooleanTypeDeclarationImpl(x, y); },
	    "ContentType": function (x) { return new RamlWrapper.ContentTypeImpl(x); },
	    "CustomSecurityScheme": function (x, y) { return new RamlWrapper.CustomSecuritySchemeImpl(x, y); },
	    "DateOnlyType": function (x) { return new RamlWrapper.DateOnlyTypeImpl(x); },
	    "DateOnlyTypeDeclaration": function (x, y) { return new RamlWrapper.DateOnlyTypeDeclarationImpl(x, y); },
	    "DateTimeOnlyType": function (x) { return new RamlWrapper.DateTimeOnlyTypeImpl(x); },
	    "DateTimeOnlyTypeDeclaration": function (x, y) { return new RamlWrapper.DateTimeOnlyTypeDeclarationImpl(x, y); },
	    "DateTimeType": function (x) { return new RamlWrapper.DateTimeTypeImpl(x); },
	    "DateTimeTypeDeclaration": function (x, y) { return new RamlWrapper.DateTimeTypeDeclarationImpl(x, y); },
	    "DateTypeDeclaration": function (x, y) { return new RamlWrapper.DateTypeDeclarationImpl(x, y); },
	    "DigestSecurityScheme": function (x, y) { return new RamlWrapper.DigestSecuritySchemeImpl(x, y); },
	    "DocumentationItem": function (x, y) { return new RamlWrapper.DocumentationItemImpl(x, y); },
	    "Extension": function (x, y) { return new RamlWrapper.ExtensionImpl(x, y); },
	    "FileType": function (x) { return new RamlWrapper.FileTypeImpl(x); },
	    "FileTypeDeclaration": function (x, y) { return new RamlWrapper.FileTypeDeclarationImpl(x, y); },
	    "FixedUriString": function (x) { return new RamlWrapper.FixedUriStringImpl(x); },
	    "FragmentDeclaration": function (x, y) { return new RamlWrapper.FragmentDeclarationImpl(x, y); },
	    "FullUriTemplateString": function (x) { return new RamlWrapper.FullUriTemplateStringImpl(x); },
	    "IntegerType": function (x) { return new RamlWrapper.IntegerTypeImpl(x); },
	    "IntegerTypeDeclaration": function (x, y) { return new RamlWrapper.IntegerTypeDeclarationImpl(x, y); },
	    "Library": function (x, y) { return new RamlWrapper.LibraryImpl(x, y); },
	    "LibraryBase": function (x, y) { return new RamlWrapper.LibraryBaseImpl(x, y); },
	    "LocationKind": function (x) { return new RamlWrapper.LocationKindImpl(x); },
	    "MarkdownString": function (x) { return new RamlWrapper.MarkdownStringImpl(x); },
	    "Method": function (x, y) { return new RamlWrapper.MethodImpl(x, y); },
	    "MethodBase": function (x, y) { return new RamlWrapper.MethodBaseImpl(x, y); },
	    "MimeType": function (x) { return new RamlWrapper.MimeTypeImpl(x); },
	    "ModelLocation": function (x) { return new RamlWrapper.ModelLocationImpl(x); },
	    "NullType": function (x) { return new RamlWrapper.NullTypeImpl(x); },
	    "NumberType": function (x) { return new RamlWrapper.NumberTypeImpl(x); },
	    "NumberTypeDeclaration": function (x, y) { return new RamlWrapper.NumberTypeDeclarationImpl(x, y); },
	    "OAuth1SecurityScheme": function (x, y) { return new RamlWrapper.OAuth1SecuritySchemeImpl(x, y); },
	    "OAuth1SecuritySchemeSettings": function (x, y) { return new RamlWrapper.OAuth1SecuritySchemeSettingsImpl(x, y); },
	    "OAuth2SecurityScheme": function (x, y) { return new RamlWrapper.OAuth2SecuritySchemeImpl(x, y); },
	    "OAuth2SecuritySchemeSettings": function (x, y) { return new RamlWrapper.OAuth2SecuritySchemeSettingsImpl(x, y); },
	    "ObjectTypeDeclaration": function (x, y) { return new RamlWrapper.ObjectTypeDeclarationImpl(x, y); },
	    "Operation": function (x, y) { return new RamlWrapper.OperationImpl(x, y); },
	    "Overlay": function (x, y) { return new RamlWrapper.OverlayImpl(x, y); },
	    "PassThroughSecurityScheme": function (x, y) { return new RamlWrapper.PassThroughSecuritySchemeImpl(x, y); },
	    "Reference": function (x) { return new RamlWrapper.ReferenceImpl(x); },
	    "RelativeUriString": function (x) { return new RamlWrapper.RelativeUriStringImpl(x); },
	    "Resource": function (x, y) { return new RamlWrapper.ResourceImpl(x, y); },
	    "ResourceBase": function (x, y) { return new RamlWrapper.ResourceBaseImpl(x, y); },
	    "ResourceType": function (x, y) { return new RamlWrapper.ResourceTypeImpl(x, y); },
	    "ResourceTypeRef": function (x) { return new RamlWrapper.ResourceTypeRefImpl(x); },
	    "Response": function (x, y) { return new RamlWrapper.ResponseImpl(x, y); },
	    "SchemaString": function (x) { return new RamlWrapper.SchemaStringImpl(x); },
	    "SecuritySchemePart": function (x, y) { return new RamlWrapper.SecuritySchemePartImpl(x, y); },
	    "SecuritySchemeRef": function (x) { return new RamlWrapper.SecuritySchemeRefImpl(x); },
	    "SecuritySchemeSettings": function (x, y) { return new RamlWrapper.SecuritySchemeSettingsImpl(x, y); },
	    "StatusCodeString": function (x) { return new RamlWrapper.StatusCodeStringImpl(x); },
	    "StringType": function (x) { return new RamlWrapper.StringTypeImpl(x); },
	    "StringTypeDeclaration": function (x, y) { return new RamlWrapper.StringTypeDeclarationImpl(x, y); },
	    "TimeOnlyType": function (x) { return new RamlWrapper.TimeOnlyTypeImpl(x); },
	    "TimeOnlyTypeDeclaration": function (x, y) { return new RamlWrapper.TimeOnlyTypeDeclarationImpl(x, y); },
	    "Trait": function (x, y) { return new RamlWrapper.TraitImpl(x, y); },
	    "TraitRef": function (x) { return new RamlWrapper.TraitRefImpl(x); },
	    "TypeDeclaration": function (x, y) { return new RamlWrapper.TypeDeclarationImpl(x, y); },
	    "UnionTypeDeclaration": function (x, y) { return new RamlWrapper.UnionTypeDeclarationImpl(x, y); },
	    "UriTemplate": function (x) { return new RamlWrapper.UriTemplateImpl(x); },
	    "UsesDeclaration": function (x, y) { return new RamlWrapper.UsesDeclarationImpl(x, y); },
	    "ValueType": function (x) { return new RamlWrapper.ValueTypeImpl(x); },
	    "XMLFacetInfo": function (x, y) { return new RamlWrapper.XMLFacetInfoImpl(x, y); }
	};
	//# sourceMappingURL=raml10factory.js.map

/***/ },
/* 42 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var RamlWrapper = __webpack_require__(46);
	/**
	 * @hidden
	 * Build Wrapper node corresponding to the High Level node
	 **/
	function buildWrapperNode(node, setAsTopLevel) {
	    if (setAsTopLevel === void 0) { setAsTopLevel = true; }
	    var definition = node.definition();
	    var nodeClassName = definition.nameId();
	    var wrapperConstructor = classMap[nodeClassName];
	    if (!wrapperConstructor) {
	        var priorities = determineSuperclassesPriorities(definition);
	        var superTypes = definition.allSuperTypes().sort(function (x, y) { return priorities[x.nameId()] - priorities[y.nameId()]; });
	        var wr = null;
	        for (var i = 0; i < superTypes.length; i++) {
	            var superTypeName = superTypes[i].nameId();
	            wrapperConstructor = classMap[superTypeName];
	            if (superTypeName == "DataElement") {
	                wr = superTypeName;
	                //This is only case of nested hierarchy
	                continue;
	            }
	            if (superTypeName == "hl.BasicNode") {
	                //depth first
	                continue;
	            }
	            if (wrapperConstructor) {
	                break;
	            }
	        }
	        if (!wrapperConstructor) {
	            wr = superTypeName;
	        }
	    }
	    if (!wrapperConstructor) {
	        wrapperConstructor = classMap["hl.BasicNode"];
	    }
	    return wrapperConstructor(node, setAsTopLevel);
	}
	exports.buildWrapperNode = buildWrapperNode;
	function determineSuperclassesPriorities(td, priorities, path) {
	    if (priorities === void 0) { priorities = {}; }
	    if (path === void 0) { path = {}; }
	    var typeName = td.nameId();
	    if (path[typeName]) {
	        return;
	    }
	    path[typeName] = true;
	    var rank = (priorities[typeName] != null && priorities[typeName] + 1) || 0;
	    var superTypes = td.superTypes();
	    superTypes.forEach(function (x) {
	        var name = x.nameId();
	        var r = priorities[name];
	        if (r == null || rank > r) {
	            priorities[name] = rank;
	            determineSuperclassesPriorities(x, priorities, path);
	        }
	    });
	    delete path[typeName];
	    return priorities;
	}
	var classMap = {
	    "AbstractSecurityScheme": function (x, y) { return new RamlWrapper.AbstractSecuritySchemeImpl(x, y); },
	    "AnyType": function (x) { return new RamlWrapper.AnyTypeImpl(x); },
	    "Api": function (x, y) { return new RamlWrapper.ApiImpl(x, y); },
	    "BasicSecurityScheme": function (x, y) { return new RamlWrapper.BasicSecuritySchemeImpl(x, y); },
	    "BodyLike": function (x, y) { return new RamlWrapper.BodyLikeImpl(x, y); },
	    "BooleanType": function (x) { return new RamlWrapper.BooleanTypeImpl(x); },
	    "BooleanTypeDeclaration": function (x, y) { return new RamlWrapper.BooleanTypeDeclarationImpl(x, y); },
	    "CustomSecurityScheme": function (x, y) { return new RamlWrapper.CustomSecuritySchemeImpl(x, y); },
	    "DateTypeDeclaration": function (x, y) { return new RamlWrapper.DateTypeDeclarationImpl(x, y); },
	    "DigestSecurityScheme": function (x, y) { return new RamlWrapper.DigestSecuritySchemeImpl(x, y); },
	    "DocumentationItem": function (x, y) { return new RamlWrapper.DocumentationItemImpl(x, y); },
	    "ExampleString": function (x) { return new RamlWrapper.ExampleStringImpl(x); },
	    "FileTypeDeclaration": function (x, y) { return new RamlWrapper.FileTypeDeclarationImpl(x, y); },
	    "FixedUri": function (x) { return new RamlWrapper.FixedUriImpl(x); },
	    "FullUriTemplateString": function (x) { return new RamlWrapper.FullUriTemplateStringImpl(x); },
	    "GlobalSchema": function (x, y) { return new RamlWrapper.GlobalSchemaImpl(x, y); },
	    "IntegerTypeDeclaration": function (x, y) { return new RamlWrapper.IntegerTypeDeclarationImpl(x, y); },
	    "JSONBody": function (x, y) { return new RamlWrapper.JSONBodyImpl(x, y); },
	    "JSONExample": function (x) { return new RamlWrapper.JSONExampleImpl(x); },
	    "JSonSchemaString": function (x) { return new RamlWrapper.JSonSchemaStringImpl(x); },
	    "MarkdownString": function (x) { return new RamlWrapper.MarkdownStringImpl(x); },
	    "Method": function (x, y) { return new RamlWrapper.MethodImpl(x, y); },
	    "MethodBase": function (x, y) { return new RamlWrapper.MethodBaseImpl(x, y); },
	    "MimeType": function (x) { return new RamlWrapper.MimeTypeImpl(x); },
	    "NumberType": function (x) { return new RamlWrapper.NumberTypeImpl(x); },
	    "NumberTypeDeclaration": function (x, y) { return new RamlWrapper.NumberTypeDeclarationImpl(x, y); },
	    "OAuth1SecurityScheme": function (x, y) { return new RamlWrapper.OAuth1SecuritySchemeImpl(x, y); },
	    "OAuth1SecuritySchemeSettings": function (x, y) { return new RamlWrapper.OAuth1SecuritySchemeSettingsImpl(x, y); },
	    "OAuth2SecurityScheme": function (x, y) { return new RamlWrapper.OAuth2SecuritySchemeImpl(x, y); },
	    "OAuth2SecuritySchemeSettings": function (x, y) { return new RamlWrapper.OAuth2SecuritySchemeSettingsImpl(x, y); },
	    "Parameter": function (x, y) { return new RamlWrapper.ParameterImpl(x, y); },
	    "ParameterLocation": function (x) { return new RamlWrapper.ParameterLocationImpl(x); },
	    "RAMLSimpleElement": function (x, y) { return new RamlWrapper.RAMLSimpleElementImpl(x, y); },
	    "Reference": function (x) { return new RamlWrapper.ReferenceImpl(x); },
	    "RelativeUriString": function (x) { return new RamlWrapper.RelativeUriStringImpl(x); },
	    "Resource": function (x, y) { return new RamlWrapper.ResourceImpl(x, y); },
	    "ResourceType": function (x, y) { return new RamlWrapper.ResourceTypeImpl(x, y); },
	    "ResourceTypeRef": function (x) { return new RamlWrapper.ResourceTypeRefImpl(x); },
	    "Response": function (x, y) { return new RamlWrapper.ResponseImpl(x, y); },
	    "SchemaString": function (x) { return new RamlWrapper.SchemaStringImpl(x); },
	    "SecuritySchemePart": function (x, y) { return new RamlWrapper.SecuritySchemePartImpl(x, y); },
	    "SecuritySchemeRef": function (x) { return new RamlWrapper.SecuritySchemeRefImpl(x); },
	    "SecuritySchemeSettings": function (x, y) { return new RamlWrapper.SecuritySchemeSettingsImpl(x, y); },
	    "StatusCodeString": function (x) { return new RamlWrapper.StatusCodeStringImpl(x); },
	    "StringType": function (x) { return new RamlWrapper.StringTypeImpl(x); },
	    "StringTypeDeclaration": function (x, y) { return new RamlWrapper.StringTypeDeclarationImpl(x, y); },
	    "Trait": function (x, y) { return new RamlWrapper.TraitImpl(x, y); },
	    "TraitRef": function (x) { return new RamlWrapper.TraitRefImpl(x); },
	    "UriTemplate": function (x) { return new RamlWrapper.UriTemplateImpl(x); },
	    "ValueType": function (x) { return new RamlWrapper.ValueTypeImpl(x); },
	    "XMLBody": function (x, y) { return new RamlWrapper.XMLBodyImpl(x, y); },
	    "XMLExample": function (x) { return new RamlWrapper.XMLExampleImpl(x); },
	    "XMLSchemaString": function (x) { return new RamlWrapper.XMLSchemaStringImpl(x); }
	};
	//# sourceMappingURL=raml08factory.js.map

/***/ },
/* 43 */
/***/ function(module, exports, __webpack_require__) {

	/// <reference path="../../typings/main.d.ts" />
	"use strict";
	var _ = __webpack_require__(61);
	var Opt = __webpack_require__(21);
	exports.defined = function (x) { return (x !== null) && (x !== undefined); };
	/**
	 * Arrays of Objects are common in RAML08.
	 * @param x
	 * @returns {{}}
	 */
	function flattenArrayOfObjects(x) {
	    var res = {};
	    x.forEach(function (v) { return Object.keys(v).forEach(function (k) { return res[k] = v[k]; }); });
	    return res;
	}
	exports.flattenArrayOfObjects = flattenArrayOfObjects;
	function find(xs, f) {
	    return new Opt(_.find(xs || [], f));
	}
	exports.find = find;
	exports.isInstance = function (v, C) { return (v instanceof C) ? [v] : []; };
	exports.ifInstanceOf = function (v, C, f) { return (v instanceof C) ? f(v) : null; };
	function toTuples(map) {
	    return Object.keys(map).map(function (k) { return [k, map[k]]; });
	}
	exports.toTuples = toTuples;
	function fromTuples(tuples) {
	    var obj = {};
	    tuples.forEach(function (x) { return obj[x[0]] = x[1]; });
	    return obj;
	}
	exports.fromTuples = fromTuples;
	exports.collectInstancesOf = function (xs, C) { return tap([], function (res) { return xs.forEach(function (v) { return exports.ifInstanceOf(v, C, function (x) { return res.push(x); }); }); }); };
	exports.collectInstancesOfInMap = function (map, C) {
	    return Object.keys(map).map(function (k) { return [k, map[k]]; }).filter(function (x) { return x[1] instanceof C; }).map(function (x) { return x; });
	};
	exports.asArray = function (v) { return exports.defined(v) ? ((v instanceof Array) ? v : [v]) : []; };
	exports.shallowCopy = function (obj) { return tap({}, function (copy) { return Object.keys(obj).forEach(function (k) { return copy[k] = obj[k]; }); }); };
	exports.flatMap = function (xs, f) { return exports.flatten(xs.map(f)); };
	exports.flatten = function (xss) { return Array.prototype.concat.apply([], xss); };
	exports.takeWhile = function (xs, f) { return tap([], function (res) {
	    for (var i = 0; i < xs.length; i++) {
	        if (!f(xs[i]))
	            break;
	        res.push(xs[i]);
	    }
	}); };
	function tap(v, f) {
	    f(v);
	    return v;
	}
	exports.tap = tap;
	function kv(obj, iter) {
	    if (typeof obj === 'object')
	        Object.keys(obj).forEach(function (k) { return iter(k, obj[k]); });
	}
	exports.kv = kv;
	function indexed(objects, key, delKey) {
	    if (delKey === void 0) { delKey = false; }
	    var obj = {};
	    objects.forEach(function (original) {
	        var copy = exports.shallowCopy(original);
	        if (delKey)
	            delete copy[key];
	        obj[original[key]] = copy;
	    });
	    return obj;
	}
	exports.indexed = indexed;
	function stringEndsWith(str, search) {
	    var dif = str.length - search.length;
	    return dif >= 0 && str.lastIndexOf(search) === dif;
	}
	exports.stringEndsWith = stringEndsWith;
	function stringStartsWith(str, search) {
	    return str.length - search.length >= 0 && str.substring(0, search.length) === search;
	}
	exports.stringStartsWith = stringStartsWith;
	function lazypropkeyfilter(k) {
	    return k[k.length - 1] == "_"; // ends with underscore
	}
	exports.lazypropkeyfilter = lazypropkeyfilter;
	function lazyprop(obj, key, func) {
	    var result, ready = false;
	    obj[key] = function () {
	        if (!ready) {
	            ready = true;
	            result = func.apply(obj);
	        }
	        return result;
	    };
	}
	function lazyprops(obj, keyfilter) {
	    if (keyfilter === void 0) { keyfilter = lazypropkeyfilter; }
	    for (var k in obj) {
	        if (keyfilter(k)) {
	            exports.ifInstanceOf(obj[k], Function, function (vf) { return (vf.length === 0) ? lazyprop(obj, k, vf) : null; });
	        }
	    }
	}
	exports.lazyprops = lazyprops;
	function iff(v, f) {
	    if (v !== undefined)
	        f(v);
	}
	exports.iff = iff;
	function isRAMLUrl(str) {
	    if (typeof str !== 'string' || str == '')
	        return false;
	    return stringEndsWith(str, ".raml");
	}
	exports.isRAMLUrl = isRAMLUrl;
	function getAllRequiredExternalModulesFromCode(code) {
	    var match;
	    var mods = [];
	    // both quoting styles
	    var r1 = new RegExp("require\\('([^']+)'\\)", "gi");
	    while (match = r1.exec(code)) {
	        mods.push(match[1]);
	    }
	    var r2 = new RegExp('require\\("([^"]+)"\\)', "gi");
	    while (match = r2.exec(code)) {
	        mods.push(match[1]);
	    }
	    mods = _.unique(mods).filter(function (x) { return x != ""; });
	    mods.sort();
	    return mods;
	}
	exports.getAllRequiredExternalModulesFromCode = getAllRequiredExternalModulesFromCode;
	exports.serial = (function () { var i = 0; return function () { return i++; }; })();
	function isEssential(arg) {
	    return typeof arg !== 'undefined' && arg != null;
	}
	exports.isEssential = isEssential;
	function firstToUpper(q) {
	    if (q.length == 0) {
	        return q;
	    }
	    return q.charAt(0).toUpperCase() + q.substr(1);
	}
	exports.firstToUpper = firstToUpper;
	function updateObject(source, target, addNewFields) {
	    if (addNewFields === void 0) { addNewFields = false; }
	    var keySet = Object.keys(target);
	    if (addNewFields) {
	        var map = {};
	        keySet.forEach(function (x) { return map[x] = true; });
	        Object.keys(source).forEach(function (x) { return map[x] = true; });
	        keySet = Object.keys(map);
	    }
	    keySet.forEach(function (x) {
	        var value = source[x];
	        if (value instanceof Object) {
	            if (!target[x]) {
	                target[x] = {};
	            }
	            updateObject(value, target[x], true);
	        }
	        else if (value != undefined) {
	            target[x] = source[x];
	        }
	    });
	}
	exports.updateObject = updateObject;
	;
	/**
	 * In 'str' replace all occurences of 'map' keys to their values.
	 */
	function replaceMap(str, map) {
	    Object.keys(map).forEach(function (x) { return str = replace(str, x, map[x]); });
	    return str;
	}
	exports.replaceMap = replaceMap;
	/**
	 * Replace all occurences of 'x' in 'str' to 'r' without thinking if 'x' can be passed without
	 * escaping as argument to RegExp constructor
	 */
	function replace(str, x, r) {
	    var result = '';
	    var prev = 0;
	    for (var i = str.indexOf(x); i < str.length && i >= 0; i = str.indexOf(x, prev)) {
	        result += str.substring(prev, i);
	        result += r;
	        prev = i + x.length;
	    }
	    result += str.substring(prev, str.length);
	    return result;
	}
	exports.replace = replace;
	//# sourceMappingURL=index.js.map

/***/ },
/* 44 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var contentprovider = __webpack_require__(31);
	var def = __webpack_require__(35);
	var su = def.getSchemaUtils();
	function isScheme(content) {
	    try {
	        var schemeObject = JSON.parse(content);
	        return schemeObject['$schema'];
	    }
	    catch (exception) {
	        return false;
	    }
	}
	exports.isScheme = isScheme;
	function startDownloadingReferencesAsync(schemaContent, unit) {
	    var schemaObject = su.getJSONSchema(schemaContent, new contentprovider.ContentProvider(unit));
	    var missedReferences = schemaObject.getMissingReferences([]).map(function (reference) { return schemaObject.contentAsync(reference); });
	    if (missedReferences.length === 0) {
	        return Promise.resolve(unit);
	    }
	    var allReferences = Promise.all(missedReferences);
	    var result = getRefs(allReferences, schemaObject);
	    return result.then(function () { return unit; });
	}
	exports.startDownloadingReferencesAsync = startDownloadingReferencesAsync;
	function getReferences(schemaContent, unit) {
	    var schemaObject = su.getJSONSchema(schemaContent, new contentprovider.ContentProvider(unit));
	    return schemaObject.getMissingReferences([], true);
	}
	exports.getReferences = getReferences;
	function getRefs(promise, schemaObject) {
	    return promise.then(function (references) {
	        if (references.length > 0) {
	            var missedRefs = schemaObject.getMissingReferences(references);
	            if (missedRefs.length === 0) {
	                return [];
	            }
	            var promises = [];
	            missedRefs.forEach(function (ref) {
	                promises.push(schemaObject.contentAsync(ref));
	            });
	            return getRefs(Promise.all(promises.concat(references)), schemaObject);
	        }
	        return Promise.resolve([]);
	    });
	}
	//# sourceMappingURL=schemaAsync.js.map

/***/ },
/* 45 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var hlImpl = __webpack_require__(16);
	var jsyaml = __webpack_require__(23);
	var def = __webpack_require__(35);
	var ramlService = def;
	var defaultCalculator = __webpack_require__(68);
	var search = __webpack_require__(29);
	var universeHelpers = __webpack_require__(14);
	var tckDumper = __webpack_require__(69);
	var yaml = __webpack_require__(72);
	var BasicNodeImpl = (function () {
	    /**
	     * @hidden
	     **/
	    function BasicNodeImpl(_node, setAsWrapper) {
	        if (setAsWrapper === void 0) { setAsWrapper = true; }
	        this._node = _node;
	        this._meta = new NodeMetadataImpl(false, false, universeHelpers.isMethodType(this.highLevel().definition()) && this.optional());
	        if (setAsWrapper) {
	            _node.setWrapperNode(this);
	        }
	    }
	    /**
	     * @hidden
	     **/
	    BasicNodeImpl.prototype.wrapperClassName = function () {
	        return 'BasicNodeImpl';
	    };
	    BasicNodeImpl.prototype.kind = function () {
	        return 'BasicNode';
	    };
	    /**
	     * @return Direct ancestor in RAML hierarchy
	     **/
	    BasicNodeImpl.prototype.parent = function () {
	        var parent = this._node.parent();
	        return parent ? parent.wrapperNode() : null;
	    };
	    /**
	     * @hidden
	     * @return Underlying node of the High Level model
	     **/
	    BasicNodeImpl.prototype.highLevel = function () {
	        return this._node;
	    };
	    /**
	     * @hidden
	     **/
	    BasicNodeImpl.prototype.attributes = function (name, constr) {
	        var attrs = this._node.attributes(name);
	        if (!attrs || attrs.length == 0) {
	            var defaultValue = this.getDefaultsCalculator().
	                attributeDefaultIfEnabled(this._node, this._node.definition().property(name));
	            if (defaultValue == null)
	                return [];
	            return Array.isArray(defaultValue) ? defaultValue : [defaultValue];
	        }
	        //TODO not sure if we want to artificially create missing attributes having
	        //default values
	        return attributesToValues(attrs, constr);
	    };
	    /**
	     * @hidden
	     **/
	    BasicNodeImpl.prototype.attribute = function (name, constr) {
	        var attr = this._node.attr(name);
	        if (constr && !(constr == this.toString ||
	            constr == this.toBoolean ||
	            constr == this.toNumber ||
	            constr == this.toAny)) {
	            //we're not putting values directly inside anything, besides the default
	            //convertors for default types we support
	            if (attr == null)
	                return null;
	            return constr(attr);
	        }
	        var attributeValue = attr ? attr.value() : null;
	        if (attributeValue == null) {
	            attributeValue = this.getDefaultsCalculator().
	                attributeDefaultIfEnabled(this._node, this._node.definition().property(name));
	        }
	        if (attributeValue == null)
	            return null;
	        if (constr) {
	            return constr(attributeValue);
	        }
	        else {
	            return attributeValue;
	        }
	    };
	    /**
	     * @hidden
	     **/
	    BasicNodeImpl.prototype.elements = function (name) {
	        var elements = this._node.elementsOfKind(name);
	        if (!elements) {
	            return null;
	        }
	        return elements.map(function (x) { return x.wrapperNode(); });
	    };
	    /**
	     * @hidden
	     **/
	    BasicNodeImpl.prototype.element = function (name) {
	        var element = this._node.element(name);
	        if (!element) {
	            return null;
	        }
	        return element.wrapperNode();
	    };
	    /**
	     * Append node as child
	     * @param node node to be appended
	     **/
	    BasicNodeImpl.prototype.add = function (node) {
	        this.highLevel().add(node.highLevel());
	    };
	    /**
	     * Append node as property value
	     * @param node node to be set as property value
	     * @param prop name of property to set value for
	     **/
	    BasicNodeImpl.prototype.addToProp = function (node, prop) {
	        var hl = node.highLevel();
	        var pr = this.highLevel().definition().property(prop);
	        hl._prop = pr;
	        this.highLevel().add(hl);
	    };
	    /**
	     * Remove node from children set
	     * @param node node to be removed
	     **/
	    BasicNodeImpl.prototype.remove = function (node) {
	        this.highLevel().remove(node.highLevel());
	    };
	    /**
	     * @return YAML string representing the node
	     **/
	    BasicNodeImpl.prototype.dump = function () {
	        return this.highLevel().dump("yaml");
	    };
	    BasicNodeImpl.prototype.toString = function (arg) {
	        var obj;
	        //kind of instanceof for hl.IAttribute without actually calling instanceof
	        if (arg.lowLevel && arg.property) {
	            obj = arg.value();
	        }
	        else {
	            obj = arg;
	        }
	        return obj != null ? obj.toString() : obj;
	    };
	    BasicNodeImpl.prototype.toAny = function (arg) {
	        return arg;
	    };
	    BasicNodeImpl.prototype.toBoolean = function (arg) {
	        var obj;
	        //kind of instanceof for hl.IAttribute without actually calling instanceof
	        if (arg.lowLevel && arg.property) {
	            obj = arg.value();
	        }
	        else {
	            obj = arg;
	        }
	        return obj != null ? obj.toString() == 'true' : obj;
	    };
	    BasicNodeImpl.prototype.toNumber = function (arg) {
	        var obj;
	        //kind of instanceof for hl.IAttribute without actually calling instanceof
	        if (arg.lowLevel && arg.property) {
	            obj = arg.value();
	        }
	        else {
	            obj = arg;
	        }
	        if (!obj) {
	            return obj;
	        }
	        try {
	            var nValue = parseFloat(obj.toString());
	            return nValue;
	        }
	        catch (e) { }
	        return Number.MAX_VALUE;
	    };
	    /**
	     * @return Array of errors
	     **/
	    BasicNodeImpl.prototype.errors = function () {
	        var _this = this;
	        var issues = [];
	        var highLevelErrors = this._node.errors();
	        if (highLevelErrors != null) {
	            issues = issues.concat(highLevelErrors);
	        }
	        var lineMapper = this._node.lowLevel().unit().lineMapper();
	        var result = issues.map(function (x) {
	            var startPoint = null;
	            try {
	                startPoint = lineMapper.position(x.start);
	            }
	            catch (e) {
	                console.warn(e);
	            }
	            var endPoint = null;
	            try {
	                endPoint = lineMapper.position(x.end);
	            }
	            catch (e) {
	                console.warn(e);
	            }
	            var path;
	            if (x.path) {
	                path = x.path;
	            }
	            else if (x.node) {
	                path = x.node.lowLevel().unit().path();
	            }
	            else {
	                path = search.declRoot(_this.highLevel()).lowLevel().unit().path();
	            }
	            return {
	                code: x.code,
	                message: x.message,
	                path: path,
	                start: x.start,
	                end: x.end,
	                line: startPoint.errorMessage ? null : startPoint.line,
	                column: startPoint.errorMessage ? null : startPoint.column,
	                range: [startPoint, endPoint],
	                isWarning: x.isWarning
	            };
	        });
	        return result;
	    };
	    /**
	     * @return object representing class of the node
	     **/
	    BasicNodeImpl.prototype.definition = function () {
	        return this.highLevel().definition();
	    };
	    /**
	     * @return for user class instances returns object representing actual user class
	     **/
	    BasicNodeImpl.prototype.runtimeDefinition = function () {
	        if (universeHelpers.isTypeDeclarationSibling(this.highLevel().definition())) {
	            return this.highLevel().localType();
	        }
	        return null;
	    };
	    BasicNodeImpl.prototype.toJSON = function (serializeOptions) {
	        return new tckDumper.TCKDumper(serializeOptions).dump(this);
	    };
	    /**
	     * @return Whether the element is an optional sibling of trait or resource type
	     **/
	    BasicNodeImpl.prototype.optional = function () {
	        var highLevel = this.highLevel();
	        return highLevel != null ? highLevel.optional() : false;
	    };
	    /**
	     * @return For siblings of traits or resource types returns an array of optional properties names.
	     **/
	    BasicNodeImpl.prototype.optionalProperties = function () {
	        if (!this.highLevel()) {
	            return [];
	        }
	        return this.highLevel().optionalProperties();
	    };
	    /**
	     * @hidden
	     **/
	    BasicNodeImpl.prototype.getDefaultsCalculator = function () {
	        if (this.defaultsCalculator) {
	            return this.defaultsCalculator;
	        }
	        if (this.parent()) {
	            this.defaultsCalculator = this.parent().getDefaultsCalculator();
	        }
	        if (!this.defaultsCalculator) {
	            this.defaultsCalculator = new defaultCalculator.AttributeDefaultsCalculator(false);
	        }
	        return this.defaultsCalculator;
	    };
	    /**
	     * @hidden
	     **/
	    BasicNodeImpl.prototype.setAttributeDefaults = function (attributeDefaults) {
	        this.defaultsCalculator = new defaultCalculator.AttributeDefaultsCalculator(attributeDefaults);
	    };
	    BasicNodeImpl.prototype.attributeDefaults = function () {
	        return this.getDefaultsCalculator() && this.getDefaultsCalculator().isEnabled();
	    };
	    BasicNodeImpl.prototype.meta = function () {
	        return fillElementMeta(this);
	    };
	    BasicNodeImpl.prototype.RAMLVersion = function () {
	        return this.highLevel().definition().universe().version();
	    };
	    return BasicNodeImpl;
	}());
	exports.BasicNodeImpl = BasicNodeImpl;
	var AttributeNodeImpl = (function () {
	    function AttributeNodeImpl(attr) {
	        this._meta = new ValueMetadataImpl();
	        this.attr = attr;
	    }
	    /**
	     * @return Underlying High Level attribute node
	     **/
	    AttributeNodeImpl.prototype.highLevel = function () { return this.attr; };
	    /**
	     * @hidden
	     **/
	    AttributeNodeImpl.prototype.wrapperClassName = function () {
	        return 'AttributeNodeImpl';
	    };
	    AttributeNodeImpl.prototype.kind = function () {
	        return 'AttributeNode';
	    };
	    /**
	     * @return Whether the element is an optional sibling of trait or resource type
	     **/
	    AttributeNodeImpl.prototype.optional = function () {
	        var highLevel = this.highLevel();
	        return highLevel != null ? highLevel.optional() : false;
	    };
	    AttributeNodeImpl.prototype.meta = function () {
	        return this._meta;
	    };
	    AttributeNodeImpl.prototype.RAMLVersion = function () {
	        return this.highLevel().definition().universe().version();
	    };
	    AttributeNodeImpl.prototype.parent = function () {
	        var parent = this.attr.parent();
	        return parent ? parent.wrapperNode() : null;
	    };
	    AttributeNodeImpl.prototype.toJSON = function (serializeOptions) {
	        return new tckDumper.TCKDumper(serializeOptions).dump(this);
	    };
	    return AttributeNodeImpl;
	}());
	exports.AttributeNodeImpl = AttributeNodeImpl;
	/**
	 * @hidden
	 **/
	function toStructuredValue(node) {
	    var value = node.value();
	    if (typeof value === 'string' || value == null) {
	        var mockNode = jsyaml.createNode(value);
	        mockNode._actualNode().startPosition = node.lowLevel().valueStart();
	        mockNode._actualNode().endPosition = node.lowLevel().valueEnd();
	        var stv = new hlImpl.StructuredValue(mockNode, node.parent(), node.property());
	        return stv;
	    }
	    else {
	        return value;
	    }
	}
	exports.toStructuredValue = toStructuredValue;
	var TypeInstanceImpl = (function () {
	    function TypeInstanceImpl(nodes) {
	        if (!Array.isArray(nodes)) {
	            this.node = nodes;
	        }
	        else {
	            this.children = nodes;
	        }
	    }
	    TypeInstanceImpl.prototype.properties = function () {
	        return this.getChildren().map(function (x) { return new TypeInstancePropertyImpl(x); });
	    };
	    TypeInstanceImpl.prototype.getChildren = function () {
	        return (this.node && this.node.children()) || this.children;
	    };
	    TypeInstanceImpl.prototype.value = function () {
	        return this.node && this.node.value();
	    };
	    TypeInstanceImpl.prototype.isScalar = function () {
	        if (!this.node) {
	            return false;
	        }
	        if (this.node.children().length != 0) {
	            return false;
	        }
	        var hl = this.node.highLevelNode();
	        if (hl) {
	            var prop = hl.property();
	            var range = prop.range();
	            if (range) {
	                return range.isValueType();
	            }
	        }
	        return true;
	    };
	    TypeInstanceImpl.prototype.toJSON = function () {
	        return new tckDumper.TCKDumper().serializeTypeInstance(this);
	    };
	    return TypeInstanceImpl;
	}());
	exports.TypeInstanceImpl = TypeInstanceImpl;
	var TypeInstancePropertyImpl = (function () {
	    function TypeInstancePropertyImpl(node) {
	        this.node = node;
	    }
	    TypeInstancePropertyImpl.prototype.name = function () {
	        return this.node.key();
	    };
	    TypeInstancePropertyImpl.prototype.value = function () {
	        if (this.isArray()) {
	            var children = this.node.children();
	            return children.length > 0 ? new TypeInstanceImpl(children[0]) : null;
	        }
	        else {
	            return new TypeInstanceImpl(this.node);
	        }
	    };
	    TypeInstancePropertyImpl.prototype.values = function () {
	        return this.isArray()
	            ? this.node.children().map(function (x) { return new TypeInstanceImpl(x); })
	            : [new TypeInstanceImpl(this.node)];
	    };
	    TypeInstancePropertyImpl.prototype.isArray = function () {
	        var children = this.node.children();
	        if (children.length > 0 && children[0].key() == null) {
	            return true;
	        }
	        var hl = this.node.highLevelNode();
	        if (hl) {
	            var prop = hl.property();
	            if (prop) {
	                var range = prop.range();
	                if (range) {
	                    return range.isArray();
	                }
	            }
	        }
	        return this.node.valueKind() == yaml.Kind.SEQ;
	    };
	    return TypeInstancePropertyImpl;
	}());
	exports.TypeInstancePropertyImpl = TypeInstancePropertyImpl;
	var ValueMetadataImpl = (function () {
	    function ValueMetadataImpl(_insertedAsDefault, _calculated, _optional) {
	        if (_insertedAsDefault === void 0) { _insertedAsDefault = false; }
	        if (_calculated === void 0) { _calculated = false; }
	        if (_optional === void 0) { _optional = false; }
	        this._insertedAsDefault = _insertedAsDefault;
	        this._calculated = _calculated;
	        this._optional = _optional;
	    }
	    ValueMetadataImpl.prototype.calculated = function () { return this._calculated; };
	    ValueMetadataImpl.prototype.insertedAsDefault = function () { return this._insertedAsDefault; };
	    ValueMetadataImpl.prototype.setCalculated = function () {
	        this._calculated = true;
	    };
	    ValueMetadataImpl.prototype.setInsertedAsDefault = function () {
	        this._insertedAsDefault = true;
	    };
	    ValueMetadataImpl.prototype.setOptional = function () {
	        this._optional = true;
	    };
	    ValueMetadataImpl.prototype.optional = function () {
	        return this._optional;
	    };
	    ValueMetadataImpl.prototype.isDefault = function () {
	        return !(this._insertedAsDefault || this._calculated || this._optional);
	    };
	    ValueMetadataImpl.prototype.toJSON = function () {
	        var obj = {};
	        if (this._calculated) {
	            obj['calculated'] = true;
	        }
	        if (this._insertedAsDefault) {
	            obj['insertedAsDefault'] = true;
	        }
	        if (this._optional) {
	            obj['optional'] = true;
	        }
	        return obj;
	    };
	    return ValueMetadataImpl;
	}());
	exports.ValueMetadataImpl = ValueMetadataImpl;
	var NodeMetadataImpl = (function (_super) {
	    __extends(NodeMetadataImpl, _super);
	    function NodeMetadataImpl() {
	        _super.apply(this, arguments);
	        this.valuesMeta = {};
	    }
	    NodeMetadataImpl.prototype.primitiveValuesMeta = function () { return this.valuesMeta; };
	    NodeMetadataImpl.prototype.registerInsertedAsDefaultValue = function (propName) {
	        var pMeta = this.valuesMeta[propName];
	        if (pMeta == null) {
	            this.valuesMeta[propName] = new ValueMetadataImpl(true);
	        }
	        else {
	            pMeta.setInsertedAsDefault();
	        }
	    };
	    NodeMetadataImpl.prototype.registerCalculatedValue = function (propName) {
	        var pMeta = this.valuesMeta[propName];
	        if (pMeta == null) {
	            this.valuesMeta[propName] = new ValueMetadataImpl(false, true);
	        }
	        else {
	            pMeta.setCalculated();
	        }
	    };
	    NodeMetadataImpl.prototype.registerOptionalValue = function (propName) {
	        var pMeta = this.valuesMeta[propName];
	        if (pMeta == null) {
	            this.valuesMeta[propName] = new ValueMetadataImpl(false, false, true);
	        }
	        else {
	            pMeta.setOptional();
	        }
	    };
	    NodeMetadataImpl.prototype.resetPrimitiveValuesMeta = function () {
	        this.valuesMeta = {};
	    };
	    NodeMetadataImpl.prototype.isDefault = function () {
	        if (!_super.prototype.isDefault.call(this)) {
	            return false;
	        }
	        return Object.keys(this.valuesMeta).length == 0;
	    };
	    NodeMetadataImpl.prototype.toJSON = function () {
	        var _this = this;
	        var obj = _super.prototype.toJSON.call(this);
	        var valuesObj = {};
	        var propKeys = Object.keys(this.valuesMeta);
	        if (propKeys.length > 0) {
	            propKeys.forEach(function (x) {
	                var childMeta = _this.valuesMeta[x].toJSON();
	                if (Object.keys(childMeta).length > 0) {
	                    valuesObj[x] = childMeta;
	                }
	            });
	            obj['primitiveValuesMeta'] = valuesObj;
	        }
	        return obj;
	    };
	    return NodeMetadataImpl;
	}(ValueMetadataImpl));
	exports.NodeMetadataImpl = NodeMetadataImpl;
	function fillElementMeta(node) {
	    var meta = node._meta;
	    meta.resetPrimitiveValuesMeta();
	    var highLevelNode = node.highLevel();
	    highLevelNode.definition().allProperties().forEach(function (p) {
	        var name = p.nameId();
	        var attrs = highLevelNode.attributes(name);
	        var gotValue = false;
	        var optional = false;
	        attrs.forEach(function (a) {
	            gotValue = gotValue || a.value() != null;
	            optional = optional || a.optional();
	        });
	        if (!gotValue) {
	            var calculator = node.getDefaultsCalculator();
	            var defVal = calculator.attributeDefaultIfEnabled(highLevelNode, p);
	            if (defVal != null) {
	                var insertionKind = calculator.insertionKind(highLevelNode, p);
	                if (insertionKind == defaultCalculator.InsertionKind.CALCULATED) {
	                    meta.registerCalculatedValue(name);
	                }
	                else if (insertionKind == defaultCalculator.InsertionKind.BY_DEFAULT) {
	                    meta.registerInsertedAsDefaultValue(name);
	                }
	            }
	        }
	    });
	    return meta;
	}
	exports.fillElementMeta = fillElementMeta;
	function attributesToValues(attrs, constr) {
	    if (constr) {
	        return attrs.map(function (x) { return constr(x); });
	    }
	    else {
	        return attrs.map(function (x) { return x.value(); });
	    }
	}
	exports.attributesToValues = attributesToValues;
	//# sourceMappingURL=parserCore.js.map

/***/ },
/* 46 */
/***/ function(module, exports, __webpack_require__) {

	"use strict";
	var __extends = (this && this.__extends) || function (d, b) {
	    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
	    function __() { this.constructor = d; }
	    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
	};
	var stubs = __webpack_require__(17);
	var def = __webpack_require__(35);
	var core = __webpack_require__(45);
	var apiLoader = __webpack_require__(9);
	var helper = __webpack_require__(47);
	var ApiImpl = (function (_super) {
	    __extends(ApiImpl, _super);
	    function ApiImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * The title property is a short plain text description of the RESTful API. The value SHOULD be suitable for use as a title for the contained user documentation.
	     **/
	    ApiImpl.prototype.title = function () {
	        return _super.prototype.attribute.call(this, 'title', this.toString);
	    };
	    /**
	     * @hidden
	     * Set title value
	     **/
	    ApiImpl.prototype.setTitle = function (param) {
	        this.highLevel().attrOrCreate("title").setValue("" + param);
	        return this;
	    };
	    /**
	     * If the RAML API definition is targeted to a specific API version, the API definition MUST contain a version property. The version property is OPTIONAL and should not be used if: The API itself is not versioned. The API definition does not change between versions. The API architect can decide whether a change to user documentation elements, but no change to the API's resources, constitutes a version change. The API architect MAY use any versioning scheme so long as version numbers retain the same format. For example, 'v3', 'v3.0', and 'V3' are all allowed, but are not considered to be equal.
	     **/
	    ApiImpl.prototype.version = function () {
	        return _super.prototype.attribute.call(this, 'version', this.toString);
	    };
	    /**
	     * @hidden
	     * Set version value
	     **/
	    ApiImpl.prototype.setVersion = function (param) {
	        this.highLevel().attrOrCreate("version").setValue("" + param);
	        return this;
	    };
	    /**
	     * (Optional during development; Required after implementation) A RESTful API's resources are defined relative to the API's base URI. The use of the baseUri field is OPTIONAL to allow describing APIs that have not yet been implemented. After the API is implemented (even a mock implementation) and can be accessed at a service endpoint, the API definition MUST contain a baseUri property. The baseUri property's value MUST conform to the URI specification RFC2396 or a Level 1 Template URI as defined in RFC6570. The baseUri property SHOULD only be used as a reference value.
	     **/
	    ApiImpl.prototype.baseUri = function () {
	        return _super.prototype.attribute.call(this, 'baseUri', function (attr) { return new FullUriTemplateStringImpl(attr); });
	    };
	    /**
	     * Base uri parameters are named parameters which described template parameters in the base uri
	     * @hidden
	     **/
	    ApiImpl.prototype.baseUriParameters_original = function () {
	        return _super.prototype.elements.call(this, 'baseUriParameters');
	    };
	    /**
	     * URI parameters can be further defined by using the uriParameters property. The use of uriParameters is OPTIONAL. The uriParameters property MUST be a map in which each key MUST be the name of the URI parameter as defined in the baseUri property. The uriParameters CANNOT contain a key named version because it is a reserved URI parameter name. The value of the uriParameters property is itself a map that specifies  the property's attributes as named parameters
	     **/
	    ApiImpl.prototype.uriParameters = function () {
	        return _super.prototype.elements.call(this, 'uriParameters');
	    };
	    /**
	     * A RESTful API can be reached HTTP, HTTPS, or both. The protocols property MAY be used to specify the protocols that an API supports. If the protocols property is not specified, the protocol specified at the baseUri property is used. The protocols property MUST be an array of strings, of values `HTTP` and/or `HTTPS`.
	     **/
	    ApiImpl.prototype.protocols = function () {
	        return _super.prototype.attributes.call(this, 'protocols', this.toString);
	    };
	    /**
	     * @hidden
	     * Set protocols value
	     **/
	    ApiImpl.prototype.setProtocols = function (param) {
	        this.highLevel().attrOrCreate("protocols").setValue("" + param);
	        return this;
	    };
	    /**
	     * (Optional) The media types returned by API responses, and expected from API requests that accept a body, MAY be defaulted by specifying the mediaType property. This property is specified at the root level of the API definition. The property's value MAY be a single string with a valid media type described in the specification.
	     **/
	    ApiImpl.prototype.mediaType = function () {
	        return _super.prototype.attribute.call(this, 'mediaType', function (attr) { return new MimeTypeImpl(attr); });
	    };
	    /**
	     * To better achieve consistency and simplicity, the API definition SHOULD include an OPTIONAL schemas property in the root section. The schemas property specifies collections of schemas that could be used anywhere in the API definition. The value of the schemas property is an array of maps; in each map, the keys are the schema name, and the values are schema definitions. The schema definitions MAY be included inline or by using the RAML !include user-defined data type.
	     **/
	    ApiImpl.prototype.schemas = function () {
	        return _super.prototype.elements.call(this, 'schemas');
	    };
	    /**
	     * Declarations of traits used in this API
	     * @hidden
	     **/
	    ApiImpl.prototype.traits_original = function () {
	        return _super.prototype.elements.call(this, 'traits');
	    };
	    /**
	     * A list of the security schemes to apply to all methods, these must be defined in the securitySchemes declaration.
	     **/
	    ApiImpl.prototype.securedBy = function () {
	        return _super.prototype.attributes.call(this, 'securedBy', function (attr) { return new SecuritySchemeRefImpl(attr); });
	    };
	    /**
	     * Security schemes that can be applied using securedBy
	     **/
	    ApiImpl.prototype.securitySchemes = function () {
	        return _super.prototype.elements.call(this, 'securitySchemes');
	    };
	    /**
	     * Declaration of resource types used in this API
	     * @hidden
	     **/
	    ApiImpl.prototype.resourceTypes_original = function () {
	        return _super.prototype.elements.call(this, 'resourceTypes');
	    };
	    /**
	     * Resources are identified by their relative URI, which MUST begin with a slash (/). A resource defined as a root-level property is called a top-level resource. Its property's key is the resource's URI relative to the baseUri. A resource defined as a child property of another resource is called a nested resource, and its property's key is its URI relative to its parent resource's URI. Every property whose key begins with a slash (/), and is either at the root of the API definition or is the child property of a resource property, is a resource property. The key of a resource, i.e. its relative URI, MAY consist of multiple URI path fragments separated by slashes; e.g. `/bom/items` may indicate the collection of items in a bill of materials as a single resource. However, if the individual URI path fragments are themselves resources, the API definition SHOULD use nested resources to describe this structure; e.g. if `/bom` is itself a resource then `/items` should be a nested resource of `/bom`, while `/bom/items` should not be used.
	     **/
	    ApiImpl.prototype.resources = function () {
	        return _super.prototype.elements.call(this, 'resources');
	    };
	    /**
	     * The API definition can include a variety of documents that serve as a user guides and reference documentation for the API. Such documents can clarify how the API works or provide business context. Documentation-generators MUST include all the sections in an API definition's documentation property in the documentation output, and they MUST preserve the order in which the documentation is declared. To add user documentation to the API, include the documentation property at the root of the API definition. The documentation property MUST be an array of documents. Each document MUST contain title and content attributes, both of which are REQUIRED. If the documentation property is specified, it MUST include at least one document. Documentation-generators MUST process the content field as if it was defined using Markdown.
	     **/
	    ApiImpl.prototype.documentation = function () {
	        return _super.prototype.elements.call(this, 'documentation');
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ApiImpl.prototype.wrapperClassName = function () { return "ApiImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ApiImpl.prototype.kind = function () { return "Api"; };
	    /**
	     * @return RAML version of the node
	     * @hidden
	     **/
	    ApiImpl.prototype.RAMLVersion_original = function () { return "RAML08"; };
	    /**
	     * Equivalent API with traits and resource types expanded
	     **/
	    ApiImpl.prototype.expand = function () {
	        return helper.expandTraitsAndResourceTypes(this);
	    };
	    /**
	     * Declarations of traits used in this API
	     **/
	    ApiImpl.prototype.traits = function () {
	        return helper.traitsPrimary(this);
	    };
	    /**
	     * Retrieve all traits including those defined in libraries *
	     * @deprecated
	     **/
	    ApiImpl.prototype.allTraits = function () {
	        return helper.allTraits(this);
	    };
	    /**
	     * Declaration of resource types used in this API
	     **/
	    ApiImpl.prototype.resourceTypes = function () {
	        return helper.resourceTypesPrimary(this);
	    };
	    /**
	     * Retrieve all resource types including those defined in libraries
	     * @deprecated
	     **/
	    ApiImpl.prototype.allResourceTypes = function () {
	        return helper.allResourceTypes(this);
	    };
	    /**
	     * Get child resource by its relative path
	     **/
	    ApiImpl.prototype.childResource = function (relPath) {
	        return helper.childResource(this, relPath);
	    };
	    /**
	     * Retrieve all resources of the Api
	     **/
	    ApiImpl.prototype.allResources = function () {
	        return helper.allResources(this);
	    };
	    /**
	     * Base uri parameters are named parameters which described template parameters in the base uri
	     **/
	    ApiImpl.prototype.baseUriParameters = function () {
	        return helper.baseUriParametersPrimary(this);
	    };
	    /**
	     * Retrieve an ordered list of all base uri parameters regardless of whether they are described in `baseUriParameters` or not
	     * Consider a fragment of RAML specification:
	     * ```yaml
	     * version: v1
	     * baseUri: https://{organization}.example.com/{version}/{service}
	     * baseUriParameters:
	     * service:
	     * ```
	     * Here `version` and `organization` are base uri parameters which are not described in the `baseUriParameters` node,
	     * Thus, they are not among `Api.baseUriParameters()`, but they are among `Api.allBaseUriParameters()`.
	     * @deprecated
	     **/
	    ApiImpl.prototype.allBaseUriParameters = function () {
	        return helper.baseUriParameters(this);
	    };
	    /**
	     * Protocols used by the API. Returns the `protocols` property value if it is specified.
	     * Otherwise, returns protocol, specified in the base URI.
	     * @deprecated
	     **/
	    ApiImpl.prototype.allProtocols = function () {
	        return helper.allProtocols(this);
	    };
	    /**
	     * @return RAML version of the node
	     **/
	    ApiImpl.prototype.RAMLVersion = function () {
	        return helper.RAMLVersion(this);
	    };
	    return ApiImpl;
	}(core.BasicNodeImpl));
	exports.ApiImpl = ApiImpl;
	var ValueTypeImpl = (function (_super) {
	    __extends(ValueTypeImpl, _super);
	    function ValueTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ValueTypeImpl.prototype.wrapperClassName = function () { return "ValueTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ValueTypeImpl.prototype.kind = function () { return "ValueType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ValueTypeImpl.prototype.RAMLVersion = function () { return "undefined"; };
	    /**
	     * @return JS representation of the node value
	     **/
	    ValueTypeImpl.prototype.value = function () { return this.attr.value(); };
	    return ValueTypeImpl;
	}(core.AttributeNodeImpl));
	exports.ValueTypeImpl = ValueTypeImpl;
	var AnyTypeImpl = (function (_super) {
	    __extends(AnyTypeImpl, _super);
	    function AnyTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    AnyTypeImpl.prototype.wrapperClassName = function () { return "AnyTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    AnyTypeImpl.prototype.kind = function () { return "AnyType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    AnyTypeImpl.prototype.RAMLVersion = function () { return "RAML08"; };
	    return AnyTypeImpl;
	}(ValueTypeImpl));
	exports.AnyTypeImpl = AnyTypeImpl;
	var NumberTypeImpl = (function (_super) {
	    __extends(NumberTypeImpl, _super);
	    function NumberTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    NumberTypeImpl.prototype.wrapperClassName = function () { return "NumberTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    NumberTypeImpl.prototype.kind = function () { return "NumberType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    NumberTypeImpl.prototype.RAMLVersion = function () { return "RAML08"; };
	    /**
	     * @return Number representation of the node value
	     **/
	    NumberTypeImpl.prototype.value = function () { return this.attr.value(); };
	    return NumberTypeImpl;
	}(ValueTypeImpl));
	exports.NumberTypeImpl = NumberTypeImpl;
	var BooleanTypeImpl = (function (_super) {
	    __extends(BooleanTypeImpl, _super);
	    function BooleanTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    BooleanTypeImpl.prototype.wrapperClassName = function () { return "BooleanTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    BooleanTypeImpl.prototype.kind = function () { return "BooleanType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    BooleanTypeImpl.prototype.RAMLVersion = function () { return "RAML08"; };
	    /**
	     * @return Boolean representation of the node value
	     **/
	    BooleanTypeImpl.prototype.value = function () { return this.attr.value(); };
	    return BooleanTypeImpl;
	}(ValueTypeImpl));
	exports.BooleanTypeImpl = BooleanTypeImpl;
	var ReferenceImpl = (function (_super) {
	    __extends(ReferenceImpl, _super);
	    function ReferenceImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ReferenceImpl.prototype.wrapperClassName = function () { return "ReferenceImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ReferenceImpl.prototype.kind = function () { return "Reference"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ReferenceImpl.prototype.RAMLVersion = function () { return "RAML08"; };
	    /**
	     * @return StructuredValue object representing the node value
	     **/
	    ReferenceImpl.prototype.value = function () { return core.toStructuredValue(this.attr); };
	    ReferenceImpl.prototype.structuredValue = function () {
	        return helper.structuredValue(this);
	    };
	    ReferenceImpl.prototype.name = function () {
	        return helper.referenceName(this);
	    };
	    return ReferenceImpl;
	}(core.AttributeNodeImpl));
	exports.ReferenceImpl = ReferenceImpl;
	var ResourceTypeRefImpl = (function (_super) {
	    __extends(ResourceTypeRefImpl, _super);
	    function ResourceTypeRefImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ResourceTypeRefImpl.prototype.wrapperClassName = function () { return "ResourceTypeRefImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ResourceTypeRefImpl.prototype.kind = function () { return "ResourceTypeRef"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ResourceTypeRefImpl.prototype.RAMLVersion = function () { return "RAML08"; };
	    ResourceTypeRefImpl.prototype.resourceType = function () {
	        return helper.referencedResourceType(this);
	    };
	    return ResourceTypeRefImpl;
	}(ReferenceImpl));
	exports.ResourceTypeRefImpl = ResourceTypeRefImpl;
	var ResourceTypeImpl = (function (_super) {
	    __extends(ResourceTypeImpl, _super);
	    function ResourceTypeImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Name of the resource type
	     **/
	    ResourceTypeImpl.prototype.name = function () {
	        return _super.prototype.attribute.call(this, 'name', this.toString);
	    };
	    /**
	     * @hidden
	     * Set name value
	     **/
	    ResourceTypeImpl.prototype.setName = function (param) {
	        this.highLevel().attrOrCreate("name").setValue("" + param);
	        return this;
	    };
	    /**
	     * Instructions on how and when the resource type should be used.
	     **/
	    ResourceTypeImpl.prototype.usage = function () {
	        return _super.prototype.attribute.call(this, 'usage', this.toString);
	    };
	    /**
	     * @hidden
	     * Set usage value
	     **/
	    ResourceTypeImpl.prototype.setUsage = function (param) {
	        this.highLevel().attrOrCreate("usage").setValue("" + param);
	        return this;
	    };
	    /**
	     * Methods that are part of this resource type definition
	     **/
	    ResourceTypeImpl.prototype.methods = function () {
	        return _super.prototype.elements.call(this, 'methods');
	    };
	    /**
	     * Instantiation of applyed traits
	     **/
	    ResourceTypeImpl.prototype.is = function () {
	        return _super.prototype.attributes.call(this, 'is', function (attr) { return new TraitRefImpl(attr); });
	    };
	    /**
	     * Instantiation of applyed resource type
	     **/
	    ResourceTypeImpl.prototype["type"] = function () {
	        return _super.prototype.attribute.call(this, 'type', function (attr) { return new ResourceTypeRefImpl(attr); });
	    };
	    /**
	     * securityScheme may also be applied to a resource by using the securedBy key, which is equivalent to applying the securityScheme to all methods that may be declared, explicitly or implicitly, by defining the resourceTypes or traits property for that resource. To indicate that the method may be called without applying any securityScheme, the method may be annotated with the null securityScheme.
	     **/
	    ResourceTypeImpl.prototype.securedBy = function () {
	        return _super.prototype.attributes.call(this, 'securedBy', function (attr) { return new SecuritySchemeRefImpl(attr); });
	    };
	    /**
	     * Uri parameters of this resource
	     **/
	    ResourceTypeImpl.prototype.uriParameters = function () {
	        return _super.prototype.elements.call(this, 'uriParameters');
	    };
	    /**
	     * An alternate, human-friendly name for the resource type
	     **/
	    ResourceTypeImpl.prototype.displayName = function () {
	        return _super.prototype.attribute.call(this, 'displayName', this.toString);
	    };
	    /**
	     * @hidden
	     * Set displayName value
	     **/
	    ResourceTypeImpl.prototype.setDisplayName = function (param) {
	        this.highLevel().attrOrCreate("displayName").setValue("" + param);
	        return this;
	    };
	    /**
	     * A resource or a method can override a base URI template's values. This is useful to restrict or change the default or parameter selection in the base URI. The baseUriParameters property MAY be used to override any or all parameters defined at the root level baseUriParameters property, as well as base URI parameters not specified at the root level.
	     **/
	    ResourceTypeImpl.prototype.baseUriParameters = function () {
	        return _super.prototype.elements.call(this, 'baseUriParameters');
	    };
	    /**
	     * The description attribute describes the intended use or meaning of the $self. This value MAY be formatted using Markdown.
	     **/
	    ResourceTypeImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    ResourceTypeImpl.prototype.wrapperClassName = function () { return "ResourceTypeImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    ResourceTypeImpl.prototype.kind = function () { return "ResourceType"; };
	    /**
	     * @return RAML version of the node
	     **/
	    ResourceTypeImpl.prototype.RAMLVersion = function () { return "RAML08"; };
	    ResourceTypeImpl.prototype.parametrizedProperties = function () {
	        return helper.getTemplateParametrizedProperties(this);
	    };
	    return ResourceTypeImpl;
	}(core.BasicNodeImpl));
	exports.ResourceTypeImpl = ResourceTypeImpl;
	/**
	 * Method object allows description of http methods
	 **/
	var MethodBaseImpl = (function (_super) {
	    __extends(MethodBaseImpl, _super);
	    function MethodBaseImpl() {
	        _super.apply(this, arguments);
	    }
	    /**
	     * Resource methods MAY have one or more responses. Responses MAY be described using the description property, and MAY include example attributes or schema properties.
	     **/
	    MethodBaseImpl.prototype.responses = function () {
	        return _super.prototype.elements.call(this, 'responses');
	    };
	    /**
	     * Some method verbs expect the resource to be sent as a request body. For example, to create a resource, the request must include the details of the resource to create. Resources CAN have alternate representations. For example, an API might support both JSON and XML representations. A method's body is defined in the body property as a hashmap, in which the key MUST be a valid media type.
	     **/
	    MethodBaseImpl.prototype.body = function () {
	        return _super.prototype.elements.call(this, 'body');
	    };
	    /**
	     * A method can override an API's protocols value for that single method by setting a different value for the fields.
	     **/
	    MethodBaseImpl.prototype.protocols = function () {
	        return _super.prototype.attributes.call(this, 'protocols', this.toString);
	    };
	    /**
	     * @hidden
	     * Set protocols value
	     **/
	    MethodBaseImpl.prototype.setProtocols = function (param) {
	        this.highLevel().attrOrCreate("protocols").setValue("" + param);
	        return this;
	    };
	    /**
	     * A list of the security schemas to apply, these must be defined in the securitySchemes declaration. To indicate that the method may be called without applying any securityScheme, the method may be annotated with the null securityScheme. Security schemas may also be applied to a resource with securedBy, which is equivalent to applying the security schemas to all methods that may be declared, explicitly or implicitly, by defining the resourceTypes or traits property for that resource.
	     **/
	    MethodBaseImpl.prototype.securedBy = function () {
	        return _super.prototype.attributes.call(this, 'securedBy', function (attr) { return new SecuritySchemeRefImpl(attr); });
	    };
	    /**
	     * A resource or a method can override a base URI template's values. This is useful to restrict or change the default or parameter selection in the base URI. The baseUriParameters property MAY be used to override any or all parameters defined at the root level baseUriParameters property, as well as base URI parameters not specified at the root level.
	     **/
	    MethodBaseImpl.prototype.baseUriParameters = function () {
	        return _super.prototype.elements.call(this, 'baseUriParameters');
	    };
	    /**
	     * An APIs resources MAY be filtered (to return a subset of results) or altered (such as transforming a response body from JSON to XML format) by the use of query strings. If the resource or its method supports a query string, the query string MUST be defined by the queryParameters property
	     **/
	    MethodBaseImpl.prototype.queryParameters = function () {
	        return _super.prototype.elements.call(this, 'queryParameters');
	    };
	    /**
	     * Headers that allowed at this position
	     **/
	    MethodBaseImpl.prototype.headers = function () {
	        return _super.prototype.elements.call(this, 'headers');
	    };
	    MethodBaseImpl.prototype.description = function () {
	        return _super.prototype.attribute.call(this, 'description', function (attr) { return new MarkdownStringImpl(attr); });
	    };
	    /**
	     * @hidden
	     * @return Actual name of instance class
	     **/
	    MethodBaseImpl.prototype.wrapperClassName = function () { return "MethodBaseImpl"; };
	    /**
	     * @return Actual name of instance interface
	     **/
	    MethodBaseImpl.prototype.kind = function () { return "MethodBase"; };
	    /**
	     * @return RAML version of the node
	     **/
	    MethodBaseImpl.prototype.RAMLVersion = function () { return "RAML08"; };
	    return MethodBaseImpl;
	}(core.BasicNodeImpl));
	exports.MethodBaseImpl = MethodBase